#include "../Commands/OneWire.h"

#if FEATURE_DALLAS_HELPER && FEATURE_COMMAND_OWSCAN

# include "../Commands/Common.h"

# include "../Helpers/Dallas1WireHelper.h"
# include "../Helpers/Hardware_GPIO.h"
# include "../Helpers/StringConverter.h"

String Command_OneWire_Owscan(struct EventStruct *event,
                              const char         *Line) {
  int pinnr; bool input; bool output1; bool output; bool warning;

  if (getGpioInfo(event->Par1, pinnr, input, output1, warning) && input) {      // Input pin required
    if (!parseString(Line, 3).isEmpty()) {
      if (getGpioInfo(event->Par2, pinnr, input, output, warning) && !output) { // Output required if specified
        return return_command_failed();                                         // Argument error
      }
    } else {
      event->Par2 = event->Par1;                                                // RX and TX on same pin

      if (!output1) {                                                           // Single pin must be input & output capable
        return return_command_failed();                                         // Argument error
      }
    }
    pinMode(event->Par1, INPUT);

    if (event->Par2 != event->Par1) {
      pinMode(event->Par2, OUTPUT);
    }

    uint8_t addr[8]{};

    Dallas_reset(event->Par1, event->Par2);
    String res;
    res.reserve(80);

    while (Dallas_search(addr, event->Par1, event->Par2)) { // Scan the 1-wire
      res += Dallas_format_address(addr);
      res += '\n';
    }
    return res;
  }

  return return_command_failed();
}

#endif // if FEATURE_DALLAS_HELPER && FEATURE_COMMAND_OWSCAN

#include "../Commands/ExecuteCommand.h"

#include "../../_Plugin_Helper.h"

#include "../Commands/Common.h"
#include "../Commands/InternalCommands.h"

#include "../DataStructs/ESPEasy_EventStruct.h"

#include "../ESPEasyCore/Controller.h"

#include "../Helpers/StringConverter.h"
#include "../Helpers/StringParser.h"

ExecuteCommandArgs::ExecuteCommandArgs(EventValueSource::Enum source,
                                       const char            *Line) :
  _taskIndex(INVALID_TASK_INDEX),
  _source(source),
  _Line(Line),
  _tryPlugin(false),
  _tryInternal(false),
  _tryRemoteConfig(false) {}

ExecuteCommandArgs::ExecuteCommandArgs(EventValueSource::Enum source,
                                       const String         & Line) :
  _taskIndex(INVALID_TASK_INDEX),
  _source(source),
  _Line(Line),
  _tryPlugin(false),
  _tryInternal(false),
  _tryRemoteConfig(false) {}

ExecuteCommandArgs::ExecuteCommandArgs(EventValueSource::Enum source,
                                       String              && Line) :
  _taskIndex(INVALID_TASK_INDEX),
  _source(source),
  _Line(Line),
  _tryPlugin(false),
  _tryInternal(false),
  _tryRemoteConfig(false) {}


ExecuteCommandArgs::ExecuteCommandArgs(
  taskIndex_t            taskIndex,
  EventValueSource::Enum source,
  const char            *Line,
  bool                   tryPlugin,
  bool                   tryInternal,
  bool                   tryRemoteConfig) :
  _taskIndex(taskIndex),
  _source(source),
  _Line(Line),
  _tryPlugin(tryPlugin),
  _tryInternal(tryInternal),
  _tryRemoteConfig(tryRemoteConfig) {}

ExecuteCommandArgs::ExecuteCommandArgs(
  taskIndex_t            taskIndex,
  EventValueSource::Enum source,
  const String         & Line,
  bool                   tryPlugin,
  bool                   tryInternal,
  bool                   tryRemoteConfig) :
  _taskIndex(taskIndex),
  _source(source),
  _Line(Line),
  _tryPlugin(tryPlugin),
  _tryInternal(tryInternal),
  _tryRemoteConfig(tryRemoteConfig) {}

ExecuteCommandArgs::ExecuteCommandArgs(
  taskIndex_t            taskIndex,
  EventValueSource::Enum source,
  String              && Line,
  bool                   tryPlugin,
  bool                   tryInternal,
  bool                   tryRemoteConfig) :
  _taskIndex(taskIndex),
  _source(source),
  _Line(std::move(Line)),
  _tryPlugin(tryPlugin),
  _tryInternal(tryInternal),
  _tryRemoteConfig(tryRemoteConfig) {}


std::list<ExecuteCommandArgs> ExecuteCommand_queue;

bool processExecuteCommandQueue()
{
  bool res = false;

  if (!ExecuteCommand_queue.empty()) {
    auto it = ExecuteCommand_queue.front();

    res = ExecuteCommand(std::move(it), false);
    ExecuteCommand_queue.pop_front();
  }
  return res;
}

// Execute command which may be plugin or internal commands
bool ExecuteCommand_all(ExecuteCommandArgs&& args,
                        bool                      addToQueue)
{
  args._tryPlugin = true;
  args._tryInternal = true;
  args._tryRemoteConfig = false;
  return ExecuteCommand(std::move(args), addToQueue);
}

bool ExecuteCommand_all_config(ExecuteCommandArgs&& args,
                               bool                      addToQueue)
{
  args._tryPlugin = true;
  args._tryInternal = true;
  args._tryRemoteConfig = true;
  return ExecuteCommand(std::move(args), addToQueue);
}

bool ExecuteCommand_plugin_config(ExecuteCommandArgs&& args,
                                  bool                      addToQueue)
{
  args._tryPlugin = true;
  args._tryInternal = false;
  args._tryRemoteConfig = true;
  return ExecuteCommand(std::move(args), addToQueue);
}

bool ExecuteCommand_internal(ExecuteCommandArgs&& args,
                             bool                      addToQueue)
{
  args._tryPlugin = false;
  args._tryInternal = true;
  args._tryRemoteConfig = false;
  return ExecuteCommand(std::move(args), addToQueue);
}

bool ExecuteCommand(taskIndex_t            taskIndex,
                    EventValueSource::Enum source,
                    const char            *Line,
                    bool                   tryPlugin,
                    bool                   tryInternal,
                    bool                   tryRemoteConfig,
                    bool                   addToQueue)
{
  ExecuteCommandArgs args(taskIndex,
                          source,
                          Line,
                          tryPlugin,
                          tryInternal,
                          tryRemoteConfig);
  return ExecuteCommand(std::move(args), addToQueue);
}

bool ExecuteCommand(ExecuteCommandArgs&& args, bool addToQueue)
{
  if (addToQueue) {
    ExecuteCommand_queue.emplace_back(std::move(args));
    return false; // What should be returned here?
  }
  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("ExecuteCommand"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
  String cmd;

  // We first try internal commands, which should not have a taskIndex set.
  struct EventStruct TempEvent;

  if (!GetArgv(args._Line.c_str(), cmd, 1)) {
    SendStatus(&TempEvent, return_command_failed());
    return false;
  }

  if (args._tryInternal) {
    // Small optimization for events, which happen frequently
    // FIXME TD-er: Make quick check to see if a command is an internal command, so we don't need to try all
    if (cmd.equalsIgnoreCase(F("event"))) {
      args._tryPlugin       = false;
      args._tryRemoteConfig = false;
    }
  }

  TempEvent.Source = args._source;

  #ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    addLogMove(LOG_LEVEL_DEBUG, concat(F("Command: "), cmd));
    addLog(LOG_LEVEL_DEBUG, args._Line); // for debug purposes add the whole line.
  }
  #endif

  args._Line = parseTemplate(args._Line); // parseTemplate before executing the command

  // Split the arguments into Par1...5 of the event.
  // Do not split it in executeInternalCommand, since that one will be called from the scheduler with pre-set events.
  // FIXME TD-er: Why call this for all commands? The CalculateParam function is quite heavy.
  parseCommandString(&TempEvent, args._Line);

  // FIXME TD-er: This part seems a bit strange.
  // It can't schedule a call to PLUGIN_WRITE.
  // Maybe ExecuteCommand can be scheduled?
  delay(0);

#ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    addLogMove(LOG_LEVEL_DEBUG, strformat(
                 F("Par1: %d Par2: %d Par3: %d Par4: %d Par5: %d"),
                 TempEvent.Par1,
                 TempEvent.Par2,
                 TempEvent.Par3,
                 TempEvent.Par4,
                 TempEvent.Par5));
  }
#endif // ifndef BUILD_NO_DEBUG


  if (args._tryInternal) {
    InternalCommands internalCommands(cmd.c_str(), &TempEvent, args._Line.c_str());
    bool handled = internalCommands.executeInternalCommand();

    const command_case_data& data = internalCommands.getData();

    if (data.status.length() > 0) {
      delay(0);
      SendStatus(&TempEvent, data.status);
      delay(0);
    }

    if (handled) {
      //      addLog(LOG_LEVEL_INFO, F("executeInternalCommand accepted"));
      return true;
    }
  }

  // When trying a task command, set the task index, even if it is not a valid task index.
  // For example commands from elsewhere may not have a proper task index.
  TempEvent.setTaskIndex(args._taskIndex);
  checkDeviceVTypeForTask(&TempEvent);

  if (args._tryPlugin) {
    // Use a tmp string to call PLUGIN_WRITE, since PluginCall may inadvertenly
    // alter the string.
    String tmpAction(args._Line);
    bool   handled = PluginCall(PLUGIN_WRITE, &TempEvent, tmpAction);

    //    if (handled) addLog(LOG_LEVEL_INFO, F("PLUGIN_WRITE accepted"));

    #ifndef BUILD_NO_DEBUG

    if (!tmpAction.equals(args._Line)) {
      if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
        String log = F("PLUGIN_WRITE altered the string: ");
        log += args._Line;
        log += F(" to: ");
        log += tmpAction;
        addLogMove(LOG_LEVEL_ERROR, log);
      }
    }
    #endif // ifndef BUILD_NO_DEBUG

    if (!handled) {
      // Try a controller
      handled = CPluginCall(CPlugin::Function::CPLUGIN_WRITE, &TempEvent, tmpAction);

      //      if (handled) addLog(LOG_LEVEL_INFO, F("CPLUGIN_WRITE accepted"));
    }

    if (handled) {
      SendStatus(&TempEvent, return_command_success());
      return true;
    }
  }

  if (args._tryRemoteConfig) {
    if (remoteConfig(&TempEvent, args._Line)) {
      SendStatus(&TempEvent, return_command_success());

      //      addLog(LOG_LEVEL_INFO, F("remoteConfig accepted"));

      return true;
    }
  }
  String errorUnknown = concat(F("Command unknown: "), args._Line);
  SendStatus(&TempEvent, errorUnknown);
  addLogMove(LOG_LEVEL_INFO, errorUnknown);
  delay(0);
  return false;
}

#include "../Commands/UPD.h"


#include "../../ESPEasy_common.h"

#include "../Commands/Common.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../Globals/NetworkState.h"
#include "../Globals/Settings.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Network.h"
#include "../Helpers/Networking.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringParser.h"

String Command_UDP_Port(struct EventStruct *event, const char *Line)
{
  return Command_GetORSetBool(event, F("UDPPort:"),
                              Line,
                              (bool *)&Settings.UDPPort,
                              1);
}

#if FEATURE_ESPEASY_P2P

const __FlashStringHelper* Command_UDP_Test(struct EventStruct *event, const char *Line)
{
  for (uint8_t x = 0; x < event->Par2; x++)
  {
    String eventName = "Test ";
    eventName += x;
    SendUDPCommand(event->Par1, eventName.c_str(), eventName.length());
  }
  return return_command_success_flashstr();
}

const __FlashStringHelper* Command_UPD_SendTo(struct EventStruct *event, const char *Line)
{
  int destUnit = parseCommandArgumentInt(Line, 1);

  if ((destUnit > 0) && (destUnit < 255))
  {
    String eventName = tolerantParseStringKeepCase(Line, 3);
    stripEscapeCharacters(eventName);
    SendUDPCommand(destUnit, eventName.c_str(), eventName.length());
  }
  return return_command_success_flashstr();
}

#endif // FEATURE_ESPEASY_P2P

const __FlashStringHelper* Command_UDP_SendToUPD(struct EventStruct *event, const char *Line)
{
  return Command_UDP_SendToUPD(event, Line, false);
}

const __FlashStringHelper* Command_UDP_SendToUPDMix(struct EventStruct *event, const char *Line)
{
  return Command_UDP_SendToUPD(event, Line, true);
}

const __FlashStringHelper* Command_UDP_SendToUPD(struct EventStruct *event, const char *Line, const bool handleMix)
{
  if (NetworkConnected()) {
    const String ip   = parseString(Line, 2);
    const int    port = parseCommandArgumentInt(Line, 2, -1);

    if ((port < 0) || (port > 65535)) {
      return return_command_failed_flashstr();
    }

    // FIXME TD-er: This command is not using the tolerance setting
    // tolerantParseStringKeepCase(Line, 4);
    String message;
    std::vector<uint8_t> argument;

    if (handleMix) {
      argument = parseHexTextData(Line, 4);
    } else {
      message = parseStringToEndKeepCase(Line, 4);
    }
    IPAddress UDP_IP;

    if (UDP_IP.fromString(ip)) {
      FeedSW_watchdog();
      portUDP.beginPacket(UDP_IP, port);

      if (handleMix) {
        portUDP.write(&argument[0], argument.size());
      } else {
        #if defined(ESP8266)
        portUDP.write(message.c_str(),                                    message.length());
        #endif // if defined(ESP8266)
        #if defined(ESP32)
        portUDP.write(reinterpret_cast<const uint8_t *>(message.c_str()), message.length());
        #endif // if defined(ESP32)
      }
      portUDP.endPacket();
      FeedSW_watchdog();
      delay(0);
    }
    return return_command_success_flashstr();
  }
  return return_not_connected();
}

#include "../Commands/System.h"

#include "../../ESPEasy_common.h"


#include "../Commands/Common.h"

#include "../Globals/Settings.h"

#include "../Helpers/DeepSleep.h"
#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Scheduler.h"

const __FlashStringHelper * Command_System_NoSleep(struct EventStruct *event, const char* Line)
{
	if (event->Par1 > 0)
		Settings.deepSleep_wakeTime = event->Par1; // set deep Sleep awake time
	else Settings.deepSleep_wakeTime = 0;
	return return_command_success_flashstr();
}

const __FlashStringHelper * Command_System_deepSleep(struct EventStruct *event, const char* Line)
{
	if (event->Par1 >= 0) {
		deepSleepStart(event->Par1); // call the second part of the function to avoid check and enable one-shot operation
	}
	return return_command_success_flashstr();
}

const __FlashStringHelper * Command_System_Reboot(struct EventStruct *event, const char* Line)
{
	pinMode(0, INPUT);
	pinMode(2, INPUT);
	pinMode(15, INPUT);
	reboot(IntendedRebootReason_e::CommandReboot);
	return return_command_success_flashstr();
}

#ifdef ESP8266
// Erase the RF calibration partition (4k)
const __FlashStringHelper * Command_System_Erase_RFcal(struct EventStruct *event, const char* Line)
{
	if (clearRFcalPartition()) {
		return F("Cleared RFcal partition. Please reboot!");
	}
	return return_command_failed_flashstr();
}

// Erase the SDK WiFI partition (12k)
const __FlashStringHelper * Command_System_Erase_SDK_WiFiconfig(struct EventStruct *event, const char* Line)
{
	if (clearWiFiSDKpartition()) {
		return F("Cleared SDK WiFi partition. Please reboot!");
	}
	return return_command_failed_flashstr();
}

#endif

#include "../../ESPEasy_common.h"
#include "../Globals/MQTT.h"

#if FEATURE_MQTT



#include "../Commands/Common.h"
#include "../Commands/MQTT.h"

#include "../ESPEasyCore/Controller.h"
#include "../ESPEasyCore/ESPEasy_Log.h"

#include "../Globals/CPlugins.h"
#include "../Globals/ESPEasy_Scheduler.h"
#include "../Globals/Settings.h"

#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/PeriodicalActions.h"
#include "../Helpers/Scheduler.h"
#include "../Helpers/StringConverter.h"


const __FlashStringHelper* Command_MQTT_Publish(struct EventStruct *event, const char *Line) {
  return Command_MQTT_Publish_handler(event, Line, false);
}

const __FlashStringHelper* Command_MQTT_PublishR(struct EventStruct *event, const char *Line) {
  return Command_MQTT_Publish_handler(event, Line, true);
}

const __FlashStringHelper* Command_MQTT_Publish_handler(struct EventStruct *event, const char *Line, const bool forceRetain)
{
  // ToDo TD-er: Not sure about this function, but at least it sends to an existing MQTTclient
  controllerIndex_t enabledMqttController = firstEnabledMQTT_ControllerIndex();

  if (!validControllerIndex(enabledMqttController)) {
    return F("No MQTT controller enabled");
  }

  // Command structure:  Publish,<topic>,<value>
  const String topic = parseStringKeepCase(Line, 2);
  const String value = tolerantParseStringKeepCase(Line, 3);
  # ifndef BUILD_NO_DEBUG
  addLog(LOG_LEVEL_DEBUG, strformat(F("Publish%c: %s:%s"), forceRetain ? 'R' : ' ', topic.c_str(), value.c_str()));
  # endif

  if (!topic.isEmpty()) {

    bool mqtt_retainFlag = forceRetain;
    if (!forceRetain) {
      // Place the ControllerSettings in a scope to free the memory as soon as we got all relevant information.
      MakeControllerSettings(ControllerSettings); //-V522
      if (!AllocatedControllerSettings()) {
        addLog(LOG_LEVEL_ERROR, F("MQTT : Cannot publish, out of RAM"));
        return F("MQTT : Cannot publish, out of RAM");
      }

      LoadControllerSettings(enabledMqttController, *ControllerSettings);
      mqtt_retainFlag = ControllerSettings->mqtt_retainFlag();
    }


    // @giig1967g: if payload starts with '=' then treat it as a Formula and evaluate accordingly
    // The evaluated value is already present in event->Par2
    // FIXME TD-er: Is the evaluated value always present in event->Par2 ?
    // Should it already be evaluated, or should we evaluate it now?

    bool success = false;
    if (value[0] != '=') {
      success = MQTTpublish(enabledMqttController, INVALID_TASK_INDEX, topic.c_str(), value.c_str(), mqtt_retainFlag);
    }
    else {
      success = MQTTpublish(enabledMqttController, INVALID_TASK_INDEX,  topic.c_str(), String(event->Par2).c_str(), mqtt_retainFlag);
    }
    if (success) {
      return return_command_success_flashstr();
    }
  }
  return return_command_failed_flashstr();
}


boolean MQTTsubscribe(controllerIndex_t controller_idx, const char* topic, boolean retained)
{
  if (MQTTclient.subscribe(topic)) {
    Scheduler.setIntervalTimerOverride(SchedulerIntervalTimer_e::TIMER_MQTT, 10); // Make sure the MQTT is being processed as soon as possible.
    scheduleNextMQTTdelayQueue();
    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      String log = F("Subscribed to: ");  
      log += topic;
      addLogMove(LOG_LEVEL_INFO, log);
    }
    return true;
  }
  addLog(LOG_LEVEL_ERROR, F("MQTT : subscribe failed"));
  return false;
}

const __FlashStringHelper * Command_MQTT_Subscribe(struct EventStruct *event, const char* Line)
{
  if (MQTTclient.connected() ) {
    // ToDo TD-er: Not sure about this function, but at least it sends to an existing MQTTclient
    controllerIndex_t enabledMqttController = firstEnabledMQTT_ControllerIndex();
    if (validControllerIndex(enabledMqttController)) {
      bool mqtt_retainFlag;
      {
        // Place the ControllerSettings in a scope to free the memory as soon as we got all relevant information.
        MakeControllerSettings(ControllerSettings); //-V522
        if (!AllocatedControllerSettings()) {
          addLog(LOG_LEVEL_ERROR, F("MQTT : Cannot subscribe, out of RAM"));
          return F("MQTT : Cannot subscribe, out of RAM");
        }
        LoadControllerSettings(event->ControllerIndex, *ControllerSettings);
        mqtt_retainFlag = ControllerSettings->mqtt_retainFlag();
      }

      String eventName = Line;
      String topic = eventName.substring(10);
      return return_command_boolean_result_flashstr(
        MQTTsubscribe(enabledMqttController, topic.c_str(), mqtt_retainFlag));
    }
    return F("No MQTT controller enabled");
  }
  return return_not_connected();
}


#endif // if FEATURE_MQTT

#include "../Commands/Timer.h"




#include "../../ESPEasy_common.h"


#include "../Commands/Common.h"

#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/ESPEasyRules.h"

#include "../Globals/ESPEasy_Scheduler.h"

#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Scheduler.h"

const __FlashStringHelper * command_setRulesTimer(int msecFromNow, int timerIndex, int recurringCount, bool startImmediately = false) {
  if (msecFromNow < 0)
  {
    addLog(LOG_LEVEL_ERROR, F("TIMER: time must be positive"));
  } else {
    // start new timer when msecFromNow > 0
    // Clear timer when msecFromNow == 0
    if (Scheduler.setRulesTimer(msecFromNow, timerIndex, recurringCount, startImmediately))
    { 
      return return_command_success_flashstr();
    }
  }
  return return_command_failed_flashstr();
}

const __FlashStringHelper * Command_Timer_Set(struct EventStruct *event, const char *Line)
{
  return command_setRulesTimer(
    event->Par2 * 1000, // msec from now
    event->Par1,        // timer index
    0                   // recurringCount
    );
}

const __FlashStringHelper * Command_Timer_Set_ms (struct EventStruct *event, const char* Line)
{
  return command_setRulesTimer(
    event->Par2, // interval
    event->Par1, // timer index
    0            // recurringCount
    );
}

const __FlashStringHelper * Command_Loop_Timer_Set (struct EventStruct *event, const char* Line)
{
  int recurringCount = event->Par3;
  if (recurringCount == 0) {
    // if the optional 3rd parameter is not given, set it to "run always"
    recurringCount = -1;
  }
  return command_setRulesTimer(
    event->Par2 * 1000, // msec from now
    event->Par1,        // timer index
    recurringCount
    );
}

const __FlashStringHelper * Command_Loop_Timer_Set_ms (struct EventStruct *event, const char* Line)
{
  int recurringCount = event->Par3;
  if (recurringCount == 0) {
    // if the optional 3rd parameter is not given, set it to "run always"
    recurringCount = -1;
  }
  return command_setRulesTimer(
    event->Par2, // interval
    event->Par1, // timer index
    recurringCount
    );
}

const __FlashStringHelper * Command_Loop_Timer_SetAndRun (struct EventStruct *event, const char* Line)
{
  int recurringCount = event->Par3;
  if (recurringCount == 0) {
    // if the optional 3rd parameter is not given, set it to "run always"
    recurringCount = -1;
  }
  return command_setRulesTimer(
    event->Par2 * 1000, // msec from now
    event->Par1,        // timer index
    recurringCount,
    true);
}

const __FlashStringHelper * Command_Loop_Timer_SetAndRun_ms (struct EventStruct *event, const char* Line)
{
  int recurringCount = event->Par3;
  if (recurringCount == 0) {
    // if the optional 3rd parameter is not given, set it to "run always"
    recurringCount = -1;
  }
  return command_setRulesTimer(
    event->Par2, // interval
    event->Par1, // timer index
    recurringCount,
    true);
}

const __FlashStringHelper * Command_Timer_Pause(struct EventStruct *event, const char *Line)
{
  if (Scheduler.pause_rules_timer(event->Par1)) {
    String eventName = F("Rules#TimerPause=");
    eventName += event->Par1;
    rulesProcessing(eventName); // TD-er: Process right now
    return return_command_success_flashstr();
  }
  return return_command_failed_flashstr();
}

const __FlashStringHelper * Command_Timer_Resume(struct EventStruct *event, const char *Line)
{
  if (Scheduler.resume_rules_timer(event->Par1)) {
    String eventName = F("Rules#TimerResume=");
    eventName += event->Par1;
    rulesProcessing(eventName); // TD-er: Process right now
    return return_command_success_flashstr();
  }
  return return_command_failed_flashstr();
}

const __FlashStringHelper * Command_Delay(struct EventStruct *event, const char *Line)
{
  delayBackground(event->Par1);
  return return_command_success_flashstr();
}

#include "../Commands/Networks.h"

#include "../../ESPEasy_common.h"
#include "../Commands/Common.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/ESPEasyEth.h"
#include "../Globals/NetworkState.h"
#include "../Globals/Settings.h"
#include "../Helpers/StringConverter.h"
#include "../WebServer/AccessControl.h"


#if FEATURE_ETHERNET
#include <ETH.h>
#endif

String Command_AccessInfo_Ls(struct EventStruct *event, const char* Line)
{
  return return_result(event, concat(F("Allowed IP range : "), describeAllowedIPrange()));
}

String Command_AccessInfo_Clear (struct EventStruct *event, const char* Line)
{
  clearAccessBlock();
  return Command_AccessInfo_Ls(event, Line);
}

String Command_DNS (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetIP(event, F("DNS:"), Line, Settings.DNS, NetworkDnsIP(0), 1);
}

String Command_Gateway (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetIP(event, F("Gateway:"), Line, Settings.Gateway, NetworkGatewayIP(),1);
}

String Command_IP (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetIP(event, F("IP:"), Line, Settings.IP, NetworkLocalIP(),1);
}

#if FEATURE_USE_IPV6
String Command_show_all_IP6 (struct EventStruct *event, const char* Line)
{
  // Only get all IPv6 addresses
  IP6Addresses_t addresses = NetworkAllIPv6();
  String res;
  res += '[';
  bool first = true;
  for (auto it = addresses.begin(); it != addresses.end(); ++it)
  {
    if (first) {
      first = false;
    } else {
      res += ',';
    }
    res += wrap_String(it->toString(true), '"');
  }
  res += ']';
  return res;
}
#endif


String Command_Subnet (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetIP(event, F("Subnet:"), Line, Settings.Subnet, NetworkSubnetMask(), 1);
}

#if FEATURE_ETHERNET
String Command_ETH_Phy_Addr (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetInt8_t(event, F("ETH_Phy_Addr:"), Line, reinterpret_cast<int8_t*>(&Settings.ETH_Phy_Addr),1);
}

String Command_ETH_Pin_mdc (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetInt8_t(event, F("ETH_Pin_mdc_cs:"), Line, reinterpret_cast<int8_t*>(&Settings.ETH_Pin_mdc_cs),1);
}

String Command_ETH_Pin_mdio (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetInt8_t(event, F("ETH_Pin_mdio_irq:"), Line, reinterpret_cast<int8_t*>(&Settings.ETH_Pin_mdio_irq),1);
}

String Command_ETH_Pin_power (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetInt8_t(event, F("ETH_Pin_power_rst:"), Line, reinterpret_cast<int8_t*>(&Settings.ETH_Pin_power_rst),1);
}

String Command_ETH_Phy_Type (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetInt8_t(event, F("ETH_Phy_Type:"), Line, reinterpret_cast<int8_t*>(&Settings.ETH_Phy_Type),1);
}

String Command_ETH_Clock_Mode (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetETH(event, 
                             F("ETH_Clock_Mode:"), 
                             toString(Settings.ETH_Clock_Mode),
                             Line, 
                             reinterpret_cast<uint8_t*>(&Settings.ETH_Clock_Mode),
                             1);
}

String Command_ETH_IP (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetIP(event, F("ETH_IP:"), Line, Settings.ETH_IP,ETH.localIP(),1);
}

String Command_ETH_Gateway (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetIP(event, F("ETH_Gateway:"), Line, Settings.ETH_Gateway,ETH.gatewayIP(),1);
}

String Command_ETH_Subnet (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetIP(event, F("ETH_Subnet:"), Line, Settings.ETH_Subnet,ETH.subnetMask(),1);
}

String Command_ETH_DNS (struct EventStruct *event, const char* Line)
{
  return Command_GetORSetIP(event, F("ETH_DNS:"), Line, Settings.ETH_DNS,ETH.dnsIP(),1);
}

String Command_ETH_Wifi_Mode (struct EventStruct *event, const char* Line)
{
  const NetworkMedium_t orig_medium = Settings.NetworkMedium;
  const String result = Command_GetORSetETH(event, 
                             F("NetworkMedium:"), 
                             toString(active_network_medium),
                             Line, 
                             reinterpret_cast<uint8_t*>(&Settings.NetworkMedium), 
                             1);
  if (orig_medium != Settings.NetworkMedium) {
    if (!isValid(Settings.NetworkMedium)) {
      Settings.NetworkMedium = orig_medium;
      return return_command_failed();
    }
    setNetworkMedium(Settings.NetworkMedium);
  }
  
  return result;
}

String Command_ETH_Disconnect (struct EventStruct *event, const char* Line)
{

  ethPower(0);
  delay(400);
//  ethPower(1);
  setNetworkMedium(NetworkMedium_t::Ethernet);
  ETHConnectRelaxed();

  return return_command_success();
}

#endif // if FEATURE_ETHERNET

#include "../Commands/GPIO.h"

#include "../../ESPEasy_common.h"


#include "../../ESPEasy-Globals.h"

#include "../Commands/Common.h"
#include "../DataStructs/PinMode.h"
#include "../ESPEasyCore/Controller.h"
#include "../ESPEasyCore/ESPEasyGPIO.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../Globals/ESPEasy_Scheduler.h"
#include "../Globals/GlobalMapPortStatus.h"
#include "../Helpers/Audio.h"
#include "../Helpers/Hardware_PWM.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/PortStatus.h"
#include "../Helpers/Numerical.h"

#if FEATURE_I2C_MULTIPLE
#include "../Globals/Settings.h"
#include "../Helpers/Hardware_device_info.h"
#include "../Helpers/I2C_access.h"
#endif // if FEATURE_I2C_MULTIPLE

#if FEATURE_GPIO_USE_ESP8266_WAVEFORM
# include <core_esp8266_waveform.h>
#endif 

// Forward declarations of functions used in this module
// Normally those would be declared in the .h file as private members
// But since these are not part of a class, forward declare them in the .cpp
//void createAndSetPortStatus_Mode_State(uint32_t key, uint8_t newMode, int8_t newState);
const __FlashStringHelper * getPluginIDAndPrefix(char selection, pluginID_t& pluginID, bool& success);
void logErrorGpioOffline(const __FlashStringHelper * prefix, int port);
void logErrorGpioOutOfRange(const __FlashStringHelper * prefix, int port, const char* Line = nullptr);
void logErrorGpioNotOutput(const __FlashStringHelper * prefix, int port);
void logErrorModeOutOfRange(const __FlashStringHelper * prefix, int port);
bool gpio_monitor_helper(int port, struct EventStruct *event, const char* Line);
bool gpio_unmonitor_helper(int port, struct EventStruct *event, const char* Line);
#ifdef USES_P009
bool mcpgpio_range_pattern_helper(struct EventStruct *event, const char* Line, bool isWritePattern);
#endif
#ifdef USES_P019
bool pcfgpio_range_pattern_helper(struct EventStruct *event, const char* Line, bool isWritePattern);
#endif
bool gpio_mode_range_helper(uint8_t pin, uint8_t pinMode, struct EventStruct *event, const char* Line);
#ifdef USES_P019
uint8_t getPcfAddress(uint8_t pin);
#endif
bool setGPIOMode(uint8_t pin, uint8_t mode);
#ifdef USES_P019
bool setPCFMode(uint8_t pin, uint8_t mode);
#endif
#ifdef USES_P009
bool setMCPMode(uint8_t pin, uint8_t mode);
bool mcpgpio_plugin_range_helper(uint8_t pin1, uint8_t pin2, uint16_t &result);
#endif
#ifdef USES_P019
bool pcfgpio_plugin_range_helper(uint8_t pin1, uint8_t pin2, uint16_t &result);
#endif


/*************************************************************************/

const __FlashStringHelper * Command_GPIO_Monitor(struct EventStruct *event, const char *Line)
{
  return return_command_boolean_result_flashstr(gpio_monitor_helper(event->Par2, event, Line));
}

const __FlashStringHelper * Command_GPIO_MonitorRange(struct EventStruct *event, const char *Line)
{
  bool success = true;

  for (uint8_t i = event->Par2; i <= event->Par3; i++) {
    success &= gpio_monitor_helper(i, event, Line);
  }
  return return_command_boolean_result_flashstr(success);
}

bool gpio_monitor_helper(int port, struct EventStruct *event, const char *Line)
{
  pluginID_t pluginID = INVALID_PLUGIN_ID;
  bool success = false;

  // parseString(Line, 2).charAt(0)='g':gpio; ='p':pcf; ='m':mcp
  const __FlashStringHelper * logPrefix = getPluginIDAndPrefix(parseString(Line, 2).charAt(0), pluginID, success);

  if (success && checkValidPortRange(pluginID, port))
  {
    const uint32_t key = createKey(pluginID, port); // WARNING: 'monitor' uses Par2 instead of Par1
    // if (!existPortStatus(key)) globalMapPortStatus[key].mode=PIN_MODE_OUTPUT;
    addMonitorToPort(key);

    int8_t state;

    // giig1967g: Comment next 3 lines to receive an EVENT just after calling the monitor command
    GPIO_Read(pluginID, port, state);
    globalMapPortStatus[key].state = state;

    if (state == -1) { globalMapPortStatus[key].mode = PIN_MODE_OFFLINE; }
    #ifndef BUILD_MINIMAL_OTA
    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLog(LOG_LEVEL_INFO, concat(
        logPrefix,
        strformat(F(" port #%d: added to monitor list."), port))); 
    }
    #endif
    String dummy;
    SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, dummy, 0);

    return true;
  } else {
    logErrorGpioOutOfRange(logPrefix, port, Line);
    return false;
  }
}

const __FlashStringHelper * Command_GPIO_UnMonitor(struct EventStruct *event, const char *Line)
{
  return return_command_boolean_result_flashstr(gpio_unmonitor_helper(event->Par2, event, Line));
}

const __FlashStringHelper * Command_GPIO_UnMonitorRange(struct EventStruct *event, const char *Line)
{
  bool success = true;

  for (uint8_t i = event->Par2; i <= event->Par3; i++) {
    success &= gpio_unmonitor_helper(i, event, Line);
  }
  return return_command_boolean_result_flashstr(success);
}

bool gpio_unmonitor_helper(int port, struct EventStruct *event, const char *Line)
{

  pluginID_t pluginID = INVALID_PLUGIN_ID;
  bool success = false;

  // parseString(Line, 2).charAt(0)='g':gpio; ='p':pcf; ='m':mcp
  const __FlashStringHelper * logPrefix = getPluginIDAndPrefix(parseString(Line, 2).charAt(0), pluginID, success);

  if (success && checkValidPortRange(pluginID, port))
  {
    const uint32_t key = createKey(pluginID, port); // WARNING: 'monitor' uses Par2 instead of Par1
    String dummy;
    SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, dummy, 0);

    removeMonitorFromPort(key);
    #ifndef BUILD_MINIMAL_OTA
    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLog(LOG_LEVEL_INFO, concat(
        logPrefix,
        strformat(F(" port #%d: removed from monitor list."), port)));
    }
    #endif

    return true;
  } else {
    logErrorGpioOutOfRange(logPrefix, port, Line);
    return false;
  }
}

const __FlashStringHelper * Command_GPIO_LongPulse(struct EventStruct *event, const char *Line)
{
  event->Par3 *= 1000;
  event->Par4 *= 1000;
  event->Par5 *= 1000;
  return Command_GPIO_LongPulse_Ms(event, Line);
}

const __FlashStringHelper * Command_GPIO_LongPulse_Ms(struct EventStruct *event, const char *Line)
{
  pluginID_t pluginID = INVALID_PLUGIN_ID;
  bool success = false;

  // Line[0]='l':longpulse; ='p':pcflongpulse; ='m':mcplongpulse
  const __FlashStringHelper * logPrefix = getPluginIDAndPrefix(Line[0], pluginID, success);

  if (success && checkValidPortRange(pluginID, event->Par1))
  {
    // Event parameters
    // Par1: pin nr
    // Par2: state
    // Par3: duration of  state
    // Par4: duration of !state
    // Par5: repeat count (only when Par4 > 0)
    //       -1 = repeat indefinately
    //        0 = only once
    //        N = Repeat N times
    Scheduler.clearGPIOTimer(pluginID, event->Par1);
    const uint32_t key = createKey(pluginID, event->Par1);
    createAndSetPortStatus_Mode_State(key, PIN_MODE_OUTPUT, event->Par2);

    #if FEATURE_GPIO_USE_ESP8266_WAVEFORM
    bool usingWaveForm = 
        event->Par3 > 0 && event->Par3 < 15000 &&
        event->Par4 > 0 && event->Par4 < 15000 &&
        event->Par5 != 0;
        
    if (usingWaveForm) {
      // Preferred is to use the ESP8266 waveform function.
      // Max time high or low is roughly 53 sec @ 80 MHz or half @160 MHz.
      // This is not available on ESP32.

      const uint8_t pin = event->Par1;
      uint32_t timeHighUS = event->Par3 * 1000;
      uint32_t timeLowUS  = event->Par4 * 1000;

      if (event->Par2 == 0) {
        std::swap(timeHighUS, timeLowUS);
      }
      uint32_t runTimeUS = 0;
      if (event->Par5 > 0) {
        runTimeUS = event->Par5 * (timeHighUS + timeLowUS);
        if (event->Par2 == 0) {
          // When having an inverted state repeat-cycle, add some overshoot to return to the original state
          runTimeUS += timeHighUS / 2;
        } else {
          // Must set slightly lower than expected duration as it will be rounded up.
          runTimeUS -= ((timeHighUS + timeLowUS) / 2);
        }
      }

      pinMode(event->Par1, OUTPUT);
      usingWaveForm = startWaveform(
        pin, timeHighUS, timeLowUS, runTimeUS);

      if (event->Par5 > 0 && event->Par2 == 0) {
        // Schedule switching pin back to original state
        Scheduler.setGPIOTimer(
          (runTimeUS / 1000) + 1, // msecFromNow, rounded up
          pluginID,    
          event->Par1,            // Pin/port nr
          !event->Par2,           // pin state
          0,                      // repeatInterval
          0);                     // repeatCount
      }
    }
    #else
    // waveform function not available on ESP32
    const bool usingWaveForm = false;
    #endif

    if (!usingWaveForm) {
      // Par1 = pinnr
      // Par2 = pin state
      // Par3 = timeHigh in msec
      // Par4 = timeLow in msec
      // Par5 = repeat count
      GPIO_Write(pluginID, event->Par1, event->Par2);
      if (event->Par4 > 0 && event->Par5 != 0) {
        // Compute repeat interval

        // Schedule switching pin to given state for repeat
        Scheduler.setGPIOTimer(
          event->Par3,   // msecFromNow
          pluginID,    
          event->Par1,   // Pin/port nr
          event->Par2,   // pin state
          event->Par3,   // repeat interval (high)
          event->Par5,   // repeat count
          event->Par4);  // alternate interval (low)
      } else {
        // Schedule switching pin back to original state
        Scheduler.setGPIOTimer(
          event->Par3,   // msecFromNow
          pluginID,    
          event->Par1,   // Pin/port nr
          !event->Par2,  // pin state
          0, // repeatInterva
          0); // repeatCount
      }
    }


    String log = concat(
      logPrefix, 
      strformat(F(" : port %d. Pulse H:%d"), event->Par1, event->Par3));
    if (event->Par4 > 0 && event->Par5 != 0) {
      log += strformat(F(" L:%d #:%d"), event->Par4, event->Par5);
    }
    log += F(" ms");
    addLog(LOG_LEVEL_INFO, log);
    SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, log, 0);

    return return_command_success_flashstr();
  } else {
    logErrorGpioOutOfRange(logPrefix, event->Par1, Line);
    return return_command_failed_flashstr();
  }
}

const __FlashStringHelper * Command_GPIO_Status(struct EventStruct *event, const char *Line)
{
  bool sendStatusFlag;
  pluginID_t pluginID;
  int8_t value = -1;

  switch (tolower(parseString(Line, 2).charAt(0)))
  {
    case 'g': // gpio
      pluginID       = PLUGIN_GPIO;
      sendStatusFlag = true;
      break;
#ifdef USES_P009
    case 'm': // mcp
      pluginID       = PLUGIN_MCP;
      value          = GPIO_MCP_Read(event->Par2);
      sendStatusFlag = value == -1;
      break;
#endif
#ifdef USES_P019
    case 'p': // pcf
      pluginID       = PLUGIN_PCF;
      value          = GPIO_PCF_Read(event->Par2);
      sendStatusFlag = value == -1;
      break;
#endif
    default:
      addLog(LOG_LEVEL_ERROR, F("Plugin not included in build"));
      return return_command_failed_flashstr();
  }

  if (!checkValidPortRange(pluginID, event->Par2))
  {
    return return_command_failed_flashstr();
  }
  const uint32_t key = createKey(pluginID, event->Par2); // WARNING: 'status' uses Par2 instead of Par1
  String dummy;
  SendStatusOnlyIfNeeded(event, sendStatusFlag, key, dummy, value);
  return return_command_success_flashstr();
}

const __FlashStringHelper * Command_GPIO_PWM(struct EventStruct *event, const char *Line)
{
  // Par1: GPIO
  // Par2: Duty Cycle
  // Par3: Fade duration
  // Par4: Frequency

  // For now, we only support the internal GPIO pins.
  const __FlashStringHelper * logPrefix = F("GPIO");
  uint32_t frequency = event->Par4;
  uint32_t key       = 0;

  if (set_Gpio_PWM(event->Par1, event->Par2, event->Par3, frequency, key)) {
    String log = strformat(F("PWM  : GPIO: %d duty: %d"), event->Par1, event->Par2);

    if (event->Par3 != 0) {
      log += strformat(F(" Fade: %d ms"), event->Par3);
    }

    if (event->Par4 != 0) {
      log += strformat(F(" f: %d Hz"), frequency);
    }
    addLog(LOG_LEVEL_INFO, log);
    SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, log, 0);

    // SendStatus(event, getPinStateJSON(SEARCH_PIN_STATE, pluginID, event->Par1, log, 0));

    return return_command_success_flashstr();
  }
  logErrorGpioOutOfRange(logPrefix, event->Par1, Line);
  return return_command_failed_flashstr();
}

const __FlashStringHelper * Command_GPIO_Tone(struct EventStruct *event, const char *Line)
{
  // play a tone on pin par1, with frequency par2 and duration in msec par3.
  unsigned long duration   = event->Par3;
  bool mustScheduleToneOff = false;

  if (duration > 50) {
    duration            = 0;
    mustScheduleToneOff = true;
  }

  if (tone_espEasy(event->Par1, event->Par2, duration)) {
    if (mustScheduleToneOff) {
      // For now, we only support the internal GPIO pins.
      const pluginID_t pluginID = PLUGIN_GPIO;
      Scheduler.setGPIOTimer(event->Par3, pluginID, event->Par1, 0);
    }
    return return_command_success_flashstr();
  }
  return return_command_failed_flashstr();
}

const __FlashStringHelper * Command_GPIO_RTTTL(struct EventStruct *event, const char *Line)
{
  #if FEATURE_RTTTL

  // FIXME: Absolutely no error checking in play_rtttl, until then keep it only in testing
  // play a tune via a RTTTL string, look at https://www.letscontrolit.com/forum/viewtopic.php?f=4&t=343&hilit=speaker&start=10 for
  // more info.

  // First assume 'new' syntax: rtttl,<gpio>,<rtttl string>
  // Difference between 'old' and 'new':
  // Comma between the GPIO argument and the melody
  String melody = parseStringToEndKeepCase(Line, 2);
  melody = melody.substring(melody.indexOf(':'), melody.length());

  melody.replace('-', '#');
  melody.replace('_', '#');

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLog(LOG_LEVEL_INFO, strformat(F("RTTTL: pin: %d melody: %s"), event->Par1, melody.c_str()));
  }
  #if FEATURE_ANYRTTTL_LIB && FEATURE_ANYRTTTL_ASYNC
  set_rtttl_melody(melody);
  #endif // if FEATURE_ANYRTTTL_LIB && FEATURE_ANYRTTTL_ASYNC

  if (play_rtttl(event->Par1, melody.c_str())) {
    return return_command_success_flashstr();
  }
  #else // if FEATURE_RTTTL
  #ifndef BUILD_MINIMAL_OTA
  addLog(LOG_LEVEL_ERROR, F("RTTTL: command not included in build"));
  #endif
  #endif // if FEATURE_RTTTL
  return return_command_failed_flashstr();
}

const __FlashStringHelper * Command_GPIO_Pulse(struct EventStruct *event, const char *Line)
{
  const __FlashStringHelper * logPrefix = F("");
  bool   success  = false;
  pluginID_t   pluginID;

  switch (tolower(Line[0]))
  {
    case 'p':                        // pulse or pcfpulse

      if (tolower(Line[1]) == 'u') { // pulse
        pluginID  = PLUGIN_GPIO;
        logPrefix = F("GPIO");
        success   = true;
      } else if (tolower(Line[1]) == 'c') { // pcfpulse
        pluginID  = PLUGIN_PCF;
        logPrefix = F("PCF");
        success   = true;
      }
      break;
    case 'm': // mcp
      pluginID  = PLUGIN_MCP;
      logPrefix = F("MCP");
      success   = true;
      break;
  }

  if (success && checkValidPortRange(pluginID, event->Par1))
  {
    const uint32_t key = createKey(pluginID, event->Par1);

    createAndSetPortStatus_Mode_State(key, PIN_MODE_OUTPUT, event->Par2);
    GPIO_Write(pluginID, event->Par1, event->Par2);

    delay(event->Par3);

    createAndSetPortStatus_Mode_State(key, PIN_MODE_OUTPUT, !event->Par2);
    GPIO_Write(pluginID, event->Par1, !event->Par2);

    String log = logPrefix;
    log += strformat(F(" : port %d. Pulse set for %d ms"), event->Par1, event->Par3);
    addLog(LOG_LEVEL_INFO, log);
    SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, log, 0);

    return return_command_success_flashstr();
  } else {
    logErrorGpioOutOfRange(logPrefix, event->Par1, Line);
    return return_command_failed_flashstr();
  }
}

const __FlashStringHelper * Command_GPIO_Toggle(struct EventStruct *event, const char *Line)
{
  pluginID_t pluginID = INVALID_PLUGIN_ID;
  bool success = false;

  // Line[0]='g':gpiotoggle; ='p':pcfgpiotoggle; ='m':mcpgpiotoggle
  const __FlashStringHelper * logPrefix = getPluginIDAndPrefix(Line[0], pluginID, success);

  if (success && checkValidPortRange(pluginID, event->Par1))
  {
    const uint32_t key = createKey(pluginID, event->Par1);

    // WARNING: operator [] creates an entry in the map if key does not exist
    // So the next command should be part of each command:
    uint8_t   mode;
    int8_t state;

    auto it = globalMapPortStatus.find(key);

    if (it != globalMapPortStatus.end()) {
      mode  = it->second.mode;
      state = it->second.state;
    } else {
      GPIO_Read(pluginID, event->Par1, state);
      mode = (state == -1) ? PIN_MODE_OFFLINE : PIN_MODE_OUTPUT;
    }

    switch (mode) {
      case PIN_MODE_OUTPUT:
      case PIN_MODE_UNDEFINED:
      {
        createAndSetPortStatus_Mode_State(key, PIN_MODE_OUTPUT, !state);
        GPIO_Write(pluginID, event->Par1, !state);

        String log = logPrefix;
        log += concat(
          F(" toggle"),
          strformat(F(": port#%d: set to %d"), event->Par1, static_cast<int>(!state)));
        addLog(LOG_LEVEL_ERROR, log);
        SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, log, 0);

        return return_command_success_flashstr();
      }
      case PIN_MODE_OFFLINE:
        logErrorGpioOffline(logPrefix, event->Par1);
        return return_command_failed_flashstr();
      default:
        logErrorGpioNotOutput(logPrefix, event->Par1);
        return return_command_failed_flashstr();
    }
  } else {
    logErrorGpioOutOfRange(logPrefix, event->Par1, Line);
    return return_command_failed_flashstr();
  }
}

const __FlashStringHelper * Command_GPIO(struct EventStruct *event, const char *Line)
{
  pluginID_t pluginID;
  bool success = false;

  // Line[0]='g':gpio; ='p':pcfgpio; ='m':mcpgpio
  const __FlashStringHelper * logPrefix = getPluginIDAndPrefix(Line[0], pluginID, success);

  if (success && checkValidPortRange(pluginID, event->Par1))
  {
    int8_t state = 0;
    uint8_t   mode;

    if (event->Par2 == 2) { // INPUT
      mode = PIN_MODE_INPUT_PULLUP;

      switch (pluginID.value) {
        case PLUGIN_GPIO_INT:
          setInternalGPIOPullupMode(event->Par1);
          state = GPIO_Read_Switch_State(event->Par1, PIN_MODE_INPUT_PULLUP);
          break;
#ifdef USES_P009
        case PLUGIN_MCP_INT:
          setMCPInputAndPullupMode(event->Par1, true);
          GPIO_Read(pluginID, event->Par1, state);
          break;
#endif
#ifdef USES_P019
        case PLUGIN_PCF_INT:
          // PCF8574 specific: only can read 0/low state, so we must send 1
          state = 1;
          break;
#endif
        default:
          addLog(LOG_LEVEL_ERROR, F("Plugin not included in build"));
          return return_command_failed_flashstr();
      }
    } else { // OUTPUT
      mode  = PIN_MODE_OUTPUT;
      state = (event->Par2 == 0) ? 0 : 1;
    }

    const uint32_t key = createKey(pluginID, event->Par1);

    if (globalMapPortStatus[key].mode != PIN_MODE_OFFLINE)
    {
      int8_t currentState;
      GPIO_Read(pluginID, event->Par1, currentState);

      if (currentState == -1) {
        mode  = PIN_MODE_OFFLINE;
        state = -1;
      }

      createAndSetPortStatus_Mode_State(key, mode, state);

      if ((mode == PIN_MODE_OUTPUT) || (pluginID == PLUGIN_PCF)) { GPIO_Write(pluginID, event->Par1, state, mode); }

      const String log = concat(
        logPrefix,
        strformat(F(": port#%d: set to %d"), event->Par1, state));
      addLog(LOG_LEVEL_INFO, log);
      SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, log, 0);
      return return_command_success_flashstr();
    } else {
      logErrorGpioOffline(logPrefix, event->Par1);
      return return_command_failed_flashstr();
    }
  } else {
    logErrorGpioOutOfRange(logPrefix, event->Par1, Line);
    return return_command_failed_flashstr();
  }
}

void logErrorGpio(const __FlashStringHelper * prefix, int port, const __FlashStringHelper * description)
{
  if (port >= 0) {
    String log = prefix;
    log += concat(F(" : port#"), port);
    log += description;
    addLogMove(LOG_LEVEL_ERROR, log);
  }
}

void logErrorModeOutOfRange(const __FlashStringHelper * prefix, int port)
{
  logErrorGpio(prefix, port, F(" mode selection is incorrect. Valid values are: 0, 1 or 2."));
}

void logErrorGpioOffline(const __FlashStringHelper * prefix, int port)
{
  logErrorGpio(prefix, port, F(" is offline."));
}

void logErrorGpioOutOfRange(const __FlashStringHelper * prefix, int port, const char *Line)
{
  logErrorGpio(prefix, port, F(" is out of range"));
  # ifndef BUILD_NO_DEBUG
  if (port >= 0) {
    if (Line != nullptr) {
      addLog(LOG_LEVEL_DEBUG, Line);
    }
  }
  #endif
}

void logErrorGpioNotOutput(const __FlashStringHelper * prefix, int port)
{
  logErrorGpio(prefix, port, F(" is not an output port"));
}

void createAndSetPortStatus_Mode_State(uint32_t key, uint8_t newMode, int8_t newState)
{
  // WARNING: operator [] creates an entry in the map if key does not exist

  #ifdef ESP32
  switch (newMode) {
    case PIN_MODE_PWM:
    case PIN_MODE_SERVO:
      break;
    default:
      checkAndClearPWM(key);
      break;
  }
  #endif

  auto it = globalMapPortStatus.find(key);
  if (it == globalMapPortStatus.end()) {
#ifdef ESP32
    if (getPluginFromKey(key).value == PLUGIN_GPIO_INT) {
      gpio_reset_pin((gpio_num_t)getPortFromKey(key));
    }
#endif
  }
  // If it doesn't exist, it is now created.
  globalMapPortStatus[key].mode = newMode;
  if (it == globalMapPortStatus.end()) {
    it = globalMapPortStatus.find(key);
  }

  if (it != globalMapPortStatus.end()) {
    // Should always be true, as it would be created if it didn't exist.
    it->second.command = 1; // set to 1 in order to display the status in the PinStatus page

    // only force events if state has changed
    if (it->second.state != newState) {
      it->second.state        = newState;
      it->second.output       = newState;
      it->second.forceEvent   = 1;
      it->second.forceMonitor = 1;
    }
  }
}

const __FlashStringHelper * getPluginIDAndPrefix(char selection, pluginID_t& pluginID, bool& success)
{
  success = true;
  switch (tolower(selection))
  {
    case 'g': // gpio
    case 'l': // longpulse (gpio)
      pluginID  = PLUGIN_GPIO;
      return F("GPIO");
#ifdef USES_P009
    case 'm': // mcp & mcplongpulse
      pluginID  = PLUGIN_MCP;
      return F("MCP");
#endif
#ifdef USES_P019
    case 'p': // pcf & pcflongpulse
      pluginID  = PLUGIN_PCF;
      return F("PCF");
#endif
    default:
      break;
  }
  success = false;
  return F("Plugin not included in build");
}

struct range_pattern_helper_data {
  range_pattern_helper_data() {
    // Make sure the pointer is always set.
    logPrefix = F("GPIO");
  }


  const __FlashStringHelper *  logPrefix;
  uint32_t write = 0;
  uint32_t mask  = 0;

  uint8_t firstPin     = 0;
  uint8_t lastPin      = 0;
  uint8_t numBytes     = 0;
  uint8_t deltaStart   = 0;
  uint8_t numBits      = 0;
  uint8_t firstAddress = 0;
  uint8_t firstBank    = 0;
  uint8_t initVal      = 0;
  bool isMask       = false;
  bool valid        = false;
};


range_pattern_helper_data range_helper_shared(pluginID_t plugin, uint8_t pin1, uint8_t pin2)
{
  range_pattern_helper_data data;

  switch (plugin.value) {
    case PLUGIN_PCF_INT:
      data.logPrefix = F("PCF");
      break;
    case PLUGIN_MCP_INT:
      data.logPrefix = F("MCP");
      break;
  }

  if ((pin2 < pin1) ||
      !checkValidPortRange(plugin, pin1) ||
      !checkValidPortRange(plugin, pin2) ||
      ((pin2 - pin1 + 1) > 16)) {
    if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
      addLog(LOG_LEVEL_ERROR, concat(data.logPrefix, F(": pin numbers out of range.")));
    }
    return data;
  }

  data.firstPin   = ((pin1 - 1) & 0xF8) + 1;
  data.lastPin    = ((pin2 - 1) & 0xF8) + 8;
  data.numBytes   = (data.lastPin - data.firstPin + 1) / 8;
  data.deltaStart = pin1 - data.firstPin;
  data.numBits    = pin2 - pin1 + 1;

  if (plugin == PLUGIN_MCP) {
    data.firstAddress = ((pin1 - 1) / 16) + 0x20;
    data.firstBank    = (((data.firstPin - 1) / 8) + 2) % 2;
    data.initVal      = 2 * data.firstAddress + data.firstBank;
  }

  data.valid = true;
  return data;
}

range_pattern_helper_data range_pattern_helper_shared(pluginID_t plugin, struct EventStruct *event, const char *Line, bool isWritePattern)
{
  range_pattern_helper_data data = range_helper_shared(plugin, event->Par1, event->Par2);

  if (!data.valid) {
    return data;
  }

  if (isWritePattern) {
    data.logPrefix = F("GPIOPattern");
  } else {
    data.logPrefix = F("GPIORange");
  }
  data.valid  = false;
  data.isMask = !parseString(Line, 5).isEmpty();

  if (data.isMask) {
    data.mask  = event->Par4 & ((1 << (data.numBytes * 8)) - 1);
    data.mask &= ((1 << data.numBits) - 1);
    data.mask  = data.mask << data.deltaStart;
  } else {
    data.mask = (1 << data.numBits) - 1;
    data.mask = data.mask << (data.deltaStart);
  }

  if (isWritePattern) {                                         // write pattern is present
    data.write  = event->Par3 & ((1 << (data.numBytes * 8)) - 1); // limit number of bytes
    data.write &= ((1 << data.numBits) - 1);                    // limit to number of bits
    data.write  = data.write << data.deltaStart;                // shift to start from starting pin
  } else {                                                      // write pattern not present
    if (event->Par3 == 0) {
      data.write = 0;
    } else if (event->Par3 == 1) {
      data.write = (1 << data.numBits) - 1;
      data.write = data.write << data.deltaStart;
    } else {
      if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
        addLog(LOG_LEVEL_ERROR, concat(data.logPrefix,  F(": write value must be 0 or 1.")));
      }
      return data;
    }
  }


  data.valid = true;
  return data;
}

/******************************************************************************
** Par1=starting pin
** Par2=ending pin (must be higher of starting pin; and maximum 16 pin per command)
** Par3=write pattern: it's a write pattern. Write 0 or 1.
**                     Example: use decimal number 15 (in binary is 00001111) to set to 1 pin 1,2,3 and 4 and to set to 0 pins 5,6,7,8
**                     if number of bit lower than number of pins, then padded with 0;
**                     if number of bit higher than number of pins, then it's truncated.
** Par4=mask (optional): if not present assume to operate in all pins; if present is used as a mask (1=update, 0=do not update).
**            if number of bit lower than number of pins, then padded with 0;
**            if number of bit higher than number of pins, then it's truncated.
**
**  examples:
**  mcpgpioPattern,1,8,255
**     write pattern = '1101' that will be padded as: '0000001101'
**     mask not present, assume mask = '1111111111'
**  mcpgpioPattern,3,12,13
**     write pattern = '1101' that will be padded as: '0000001101'
**     mask not present, assume mask = '1111111111'
**  mcpgpioPattern,3,12,525
**     write pattern = 525 = '100001101'
**     mask not present, assume mask = '1111111111'
**  mcpgpioPattern,3,12,525,973
**     write pattern = 525 = '100001101'
**     mask = 973 = '1111001101'
**     write pattern after mask = '1000xx11x1' where x indicates that the pin will not be changed
******************************************************************************/
#ifdef USES_P009
const __FlashStringHelper * Command_GPIO_McpGPIOPattern(struct EventStruct *event, const char *Line)
{
  return return_command_boolean_result_flashstr(mcpgpio_range_pattern_helper(event, Line, true));
}
#endif

#ifdef USES_P019
const __FlashStringHelper * Command_GPIO_PcfGPIOPattern(struct EventStruct *event, const char *Line)
{
  return return_command_boolean_result_flashstr(pcfgpio_range_pattern_helper(event, Line, true));
}
#endif

/******************************************************************************
** Par1=starting pin
** Par2=ending pin (must be higher of starting pin; and maximum 16 pin per command)
** Par3=write value: if 0 (or 1) then assume 0 (or 1) for all the pins in the range;
** Par4=mask (optional): if not present assume to operate in all pins; if present is used as a mask (1=update, 0=do not update).
**            if number of bit lower than number of pins, then padded with 0;
**            if number of bit higher than number of pins, then it's truncated.
**
**  examples:
**  mcpgpioRange,1,8,1: set pins 1 to 8 to 1
**  mcpgpioRange,3,12,1: set pins 3 to 12 to 1
**  mcpgpioRange,5,17,0: set pins 5 to 17 to 0
**  mcpgpioRange,3,12,1,525
**     mask = '0100001101'
**     write pattern after mask = 'x1xxxx11x1' where x indicates that the pin will not be changed
**  mcpgpioRange,3,12,1,973
**     mask = 973 = '1111001101'
**     write pattern after mask = '1111xx11x1' where x indicates that the pin will not be changed
******************************************************************************/
#ifdef USES_P009
const __FlashStringHelper * Command_GPIO_McpGPIORange(struct EventStruct *event, const char *Line)
{
  return return_command_boolean_result_flashstr(mcpgpio_range_pattern_helper(event, Line, false));
}
#endif

#ifdef USES_P019
const __FlashStringHelper * Command_GPIO_PcfGPIORange(struct EventStruct *event, const char *Line)
{
  return return_command_boolean_result_flashstr(pcfgpio_range_pattern_helper(event, Line, false));
}
#endif

#ifdef USES_P009
// FIXME TD-er: Function is nearly identical to pcfgpio_range_pattern_helper
bool mcpgpio_range_pattern_helper(struct EventStruct *event, const char *Line, bool isWritePattern)
{
  range_pattern_helper_data data = range_pattern_helper_shared(PLUGIN_MCP, event, Line, isWritePattern);

  if (!data.valid) {
    return false;
  }

  bool   onLine = false;
  for (uint8_t i = 0; i < data.numBytes; i++) {
    uint8_t readValue;
    const uint8_t    currentVal            = data.initVal + i;
    const uint8_t    currentAddress        = static_cast<int>(currentVal / 2);
    uint8_t          currentMask           = (data.mask  >> (8 * i)) & 0xFF;
    const uint8_t    currentInvertedMask   = 0xFF - currentMask;
    const uint8_t    currentWrite          = (data.write >> (8 * i)) & 0xFF;
    const uint8_t    currentGPIORegister   = ((currentVal % 2) == 0) ? MCP23017_GPIOA : MCP23017_GPIOB;
    const uint8_t    currentIOModeRegister = ((currentVal % 2) == 0) ? MCP23017_IODIRA : MCP23017_IODIRB;
    uint8_t    writeGPIOValue        = 0;

    if (GPIO_MCP_ReadRegister(currentAddress, currentIOModeRegister, &readValue)) {
      // set type to output only for the pins of the mask
      uint8_t writeModeValue = (readValue & currentInvertedMask);
      GPIO_MCP_WriteRegister(currentAddress, currentIOModeRegister, writeModeValue);
      GPIO_MCP_ReadRegister(currentAddress, currentGPIORegister, &readValue);

      // write to port
      writeGPIOValue = (readValue & currentInvertedMask) | (currentWrite & data.mask);
      GPIO_MCP_WriteRegister(currentAddress, currentGPIORegister, writeGPIOValue);

      onLine = true;
    } else {
      onLine = false;
    }

    const uint8_t   mode = (onLine) ? PIN_MODE_OUTPUT : PIN_MODE_OFFLINE;
    for (uint8_t j = 0; currentMask != 0 && j < 8; j++) {
      if (currentMask & 1) {  // only for the pins in the mask
        uint8_t currentPin    = data.firstPin + j + 8 * i;
        const uint32_t key = createKey(PLUGIN_MCP, currentPin);

        // state = onLine ? ((writeGPIOValue & uint8_t(pow(2,j))) >> j) : -1;
        const int8_t state = onLine ? ((writeGPIOValue & (1 << j)) >> j) : -1;

        createAndSetPortStatus_Mode_State(key, mode, state);
        const String log = concat(
          data.logPrefix,
          strformat(F(": port#%d: set to %d"), currentPin, state));
        addLog(LOG_LEVEL_INFO, log);
        SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, log, 0);
      }
      currentMask >>= 1; // Shift the mask 1 position
    }
  }
  return onLine;
}
#endif

#ifdef USES_P019
uint8_t getPcfAddress(uint8_t pin)
{
  uint8_t retValue = static_cast<int>((pin - 1) / 8) + 0x20;

  if (retValue > 0x27) { retValue += 0x10; }
  return retValue;
}

// FIXME TD-er: Function is nearly identical to mcpgpio_range_pattern_helper
bool pcfgpio_range_pattern_helper(struct EventStruct *event, const char *Line, bool isWritePattern)
{
  range_pattern_helper_data data = range_pattern_helper_shared(PLUGIN_PCF, event, Line, isWritePattern);

  if (!data.valid) {
    return false;
  }

  bool   onLine = false;
  for (uint8_t i = 0; i < data.numBytes; i++) {
    uint8_t readValue;
    uint8_t    currentAddress = getPcfAddress(event->Par1 + 8 * i);

    uint8_t currentMask         = (data.mask  >> (8 * i)) & 0xFF;
    uint8_t currentInvertedMask = 0xFF - currentMask;
    uint8_t currentWrite        = (data.write >> (8 * i)) & 0xFF;
    uint8_t writeGPIOValue      = 255;

    onLine = GPIO_PCF_ReadAllPins(currentAddress, &readValue);

    if (onLine) { writeGPIOValue = (readValue & currentInvertedMask) | (currentWrite & data.mask); }

    uint8_t   mode = (onLine) ? PIN_MODE_OUTPUT : PIN_MODE_OFFLINE;
    int8_t state;

    for (uint8_t j = 0; j < 8; j++) {
      uint8_t currentPin    = data.firstPin + j + 8 * i;
      const uint32_t key = createKey(PLUGIN_PCF, currentPin);

      if (currentMask & 1) {  // only for the pins in the mask
        state = onLine ? ((writeGPIOValue & (1 << j)) >> j) : -1;

        createAndSetPortStatus_Mode_State(key, mode, state);
        const String log = concat(
          data.logPrefix,
          strformat(F(": port#%d: set to %d"), currentPin, state));
        addLog(LOG_LEVEL_INFO, log);
        SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, log, 0);
      } else {
        // set to 1 the INPUT pins and the PIN that have not been initialized yet.
        if (!existPortStatus(key) ||
            (existPortStatus(key) &&
             ((globalMapPortStatus[key].mode == PIN_MODE_INPUT) || (globalMapPortStatus[key].mode == PIN_MODE_INPUT_PULLUP)))) {
          readValue |= (1 << j); // set port j = 1
        }
      }
      currentMask >>= 1; // Shift the mask 1 position
    }

    if (onLine) {
      writeGPIOValue = (readValue & currentInvertedMask) | (currentWrite & data.mask);

      // write to port
      GPIO_PCF_WriteAllPins(currentAddress, writeGPIOValue);
    }
  }
  return onLine;
}
#endif

bool setGPIOMode(uint8_t pin, uint8_t mode)
{
  if (!checkValidPortRange(PLUGIN_GPIO, pin)) {
    return false;
  }
  switch (mode) {
    case PIN_MODE_OUTPUT:
      pinMode(pin, OUTPUT);
      break;
    case PIN_MODE_INPUT_PULLUP:
      setInternalGPIOPullupMode(pin);
      break;
    case PIN_MODE_INPUT:
      pinMode(pin, INPUT);
      break;
  }
  return true;
}

#ifdef USES_P009
bool setMCPMode(uint8_t pin, uint8_t mode)
{
  if (checkValidPortRange(PLUGIN_MCP, pin)) {
    switch (mode) {
      case PIN_MODE_OUTPUT:
        setMCPOutputMode(pin);
        break;
      case PIN_MODE_INPUT_PULLUP:
        setMCPInputAndPullupMode(pin, true);
        break;
      case PIN_MODE_INPUT:
        setMCPInputAndPullupMode(pin, false);
        break;
    }
    return true;
  } else {
    return false;
  }
}
#endif

#ifdef USES_P019
bool setPCFMode(uint8_t pin, uint8_t mode)
{
  if (checkValidPortRange(PLUGIN_PCF, pin)) {
    switch (mode) {
      case PIN_MODE_OUTPUT:
        // do nothing
        break;
      case PIN_MODE_INPUT_PULLUP:
      case PIN_MODE_INPUT:
        setPCFInputMode(pin);
        break;
    }
    return true;
  } else {
    return false;
  }
}
#endif

/***********************************************
 * event->Par1: PIN to be set
 * event->Par2: MODE to be set:
 *             0 = OUTPUT
 *             1 = INPUT PULLUP or INPUT PULLDOWN (only for GPIO16)
 *             2 = INPUT
 **********************************************/
const __FlashStringHelper * Command_GPIO_Mode(struct EventStruct *event, const char *Line)
{
  return return_command_boolean_result_flashstr(gpio_mode_range_helper(event->Par1, event->Par2, event, Line));
}

const __FlashStringHelper * Command_GPIO_ModeRange(struct EventStruct *event, const char *Line)
{
  bool success = true;

  for (uint8_t i = event->Par1; i <= event->Par2; i++) {
    success &= gpio_mode_range_helper(i, event->Par3, event, Line);
  }
  return return_command_boolean_result_flashstr(success);
}

bool gpio_mode_range_helper(uint8_t pin, uint8_t pinMode, struct EventStruct *event, const char *Line)
{
  pluginID_t pluginID = INVALID_PLUGIN_ID;
  bool success = false;

  // Line[0]='g':gpio; ='p':pcfgpio; ='m':mcpgpio
  const __FlashStringHelper * logPrefix = getPluginIDAndPrefix(Line[0], pluginID, success);
  const __FlashStringHelper * logPostfix = F(""); // = new char;

  if (success && checkValidPortRange(pluginID, pin))
  {
    // int8_t state=0;
    uint8_t mode = 255;

    // bool setSuccess=false;

    switch (pinMode) {
      case 0:
        mode       = PIN_MODE_OUTPUT;
        logPostfix = F("OUTPUT");
        break;
      case 1:
        mode       = PIN_MODE_INPUT_PULLUP;
        logPostfix = F("INPUT PULLUP");
        break;
      case 2:
        mode       = PIN_MODE_INPUT;
        logPostfix = F("INPUT");
        break;
    }

    if (mode < 255) {
      switch (pluginID.value) {
        case PLUGIN_GPIO_INT:
          /* setSuccess = */ setGPIOMode(pin, mode);
          break;
#ifdef USES_P019
        case PLUGIN_PCF_INT:
          // set pin = 1 when INPUT
          /* setSuccess = */ setPCFMode(pin, mode);
          break;
#endif
#ifdef USES_P009
        case PLUGIN_MCP_INT:
          /* setSuccess = */ setMCPMode(pin, mode);
          break;
#endif
        default:
          addLog(LOG_LEVEL_ERROR, F("Plugin not included in build"));
          return false;
      }

      const uint32_t key = createKey(pluginID, pin);

      if (globalMapPortStatus[key].mode != PIN_MODE_OFFLINE)
      {
        int8_t currentState;
        GPIO_Read(pluginID, pin, currentState);

        // state = currentState;

        if (currentState == -1) {
          mode = PIN_MODE_OFFLINE;

          // state = -1;
        }

        createAndSetPortStatus_Mode_State(key, mode, currentState);

        String log = logPrefix;
        log += strformat(F(" : port#%d: MODE set to "), pin);
        log += logPostfix;
        log += concat(F(". Value = "), currentState);
        addLog(LOG_LEVEL_INFO, log);
        SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, log, 0);
        return true;
      } else {
        logErrorGpioOffline(logPrefix, pin);
        return false;
      }
    }
    logErrorModeOutOfRange(logPrefix, pin);
    return false;
  }
  logErrorGpioOutOfRange(logPrefix, pin, Line);
  return false;
}

bool getGPIOPinStateValues(String& str) {
  // parseString(string, 1) = device (gpio,mcpgpio,pcfgpio) that can be shortened to g, m or p
  // parseString(string, 2) = command (pinstate,pinrange)
  // parseString(string, 3) = gpio 1st number or a range separated by '-'
  bool   success = false;
  const String device     = parseString(str, 1);
  const String command    = parseString(str, 2);
  const String gpio_descr = parseString(str, 3);

  if ((command.length() >= 8) && equals(command, F("pinstate")) && (device.length() > 0)) {
    #ifndef BUILD_NO_DEBUG
    String logPrefix;
    #endif
    // returns pin value using syntax: [plugin#xxxxxxx#pinstate#x]
    int32_t par1{};
    const bool validArgument = validIntFromString(gpio_descr, par1);
    #if FEATURE_PINSTATE_EXTENDED
    pluginID_t pluginID = INVALID_PLUGIN_ID;
    #endif // if FEATURE_PINSTATE_EXTENDED

    if (validArgument) {
      switch (device[0]) {
        case 'g':
        {
          #if FEATURE_PINSTATE_EXTENDED
          pluginID  = PLUGIN_GPIO;
          #endif // if FEATURE_PINSTATE_EXTENDED
          str       = digitalRead(par1);
          #ifndef BUILD_NO_DEBUG
          logPrefix = F("GPIO");
          #endif
          success   = true;
          break;
        }

#ifdef USES_P009
        case 'm':
          #if FEATURE_PINSTATE_EXTENDED
          pluginID  = PLUGIN_MCP;
          #endif // if FEATURE_PINSTATE_EXTENDED
          #if FEATURE_I2C_MULTIPLE
          if (getI2CBusCount() > 1) {
            I2CSelectHighClockSpeed(Settings.getI2CInterfacePCFMCP());
          }
          #endif // if FEATURE_I2C_MULTIPLE
          str       = GPIO_MCP_Read(par1);
          #ifndef BUILD_NO_DEBUG
          logPrefix = F("MCP");
          #endif
          success   = true;
          break;
#endif

#ifdef USES_P019
        case 'p':
          #if FEATURE_PINSTATE_EXTENDED
          pluginID  = PLUGIN_PCF;
          #endif // if FEATURE_PINSTATE_EXTENDED
          #if FEATURE_I2C_MULTIPLE
          if (getI2CBusCount() > 1) {
            I2CSelectHighClockSpeed(Settings.getI2CInterfacePCFMCP());
          }
          #endif // if FEATURE_I2C_MULTIPLE
          str       = GPIO_PCF_Read(par1);
          #ifndef BUILD_NO_DEBUG
          logPrefix = F("PCF");
          #endif
          success   = true;
          break;
#endif
        default:
        {
          #if FEATURE_PINSTATE_EXTENDED
          uint32_t plugin = INVALID_PLUGIN_ID.value;
          if (validUIntFromString(device, plugin) && (plugin != INVALID_PLUGIN_ID.value)) { // Valid plugin ID?
            pluginID.value  = plugin;
            #ifndef BUILD_NO_DEBUG
            logPrefix = get_formatted_Plugin_number(pluginID);
            #endif
          } else 
          #endif // if FEATURE_PINSTATE_EXTENDED
          {
            addLog(LOG_LEVEL_ERROR, F("Plugin not included in build"));
            return false;
          }
        }
      }
      #if FEATURE_PINSTATE_EXTENDED
      if (pluginID != INVALID_PLUGIN_ID) {
        const uint32_t key       = createKey(pluginID, par1);
        const auto it            = globalMapPortStatus.find(key);
        const bool notGpioMcpPcf = ((pluginID != PLUGIN_GPIO) && (pluginID != PLUGIN_MCP) && (pluginID != PLUGIN_PCF));

        if (it != globalMapPortStatus.end() && ((it->second.mode == PIN_MODE_PWM) || (it->second.mode == PIN_MODE_SERVO) || notGpioMcpPcf)) {
          // For GPIO/MCP/PCF PWM or SERVO mode get the last set duty cycle or for other plugins get the PWM/SERVO or pinstate
          str     = it->second.getValue();
          success = true;
        }
      }
      #endif // if FEATURE_PINSTATE_EXTENDED
    }

    if (success) {
      #ifndef BUILD_NO_DEBUG
      String log = logPrefix;
      log += strformat(F(" PLUGIN PINSTATE pin =%d; value=%s"), par1, str.c_str());
      addLog(LOG_LEVEL_DEBUG, log);
      #endif // ifndef BUILD_NO_DEBUG
    } else {
      addLog(LOG_LEVEL_ERROR, F(" PLUGIN PINSTATE. Syntax error. Pin parameter is not numeric"));
    }
  } else if ((command.length() >= 8) && equals(command, F("pinrange"))) {
    // returns pin value using syntax: [plugin#xxxxxxx#pinrange#x-y]
    int32_t  par1, par2;
    bool successPar = false;
    int  dashpos    = gpio_descr.indexOf('-');

    if (dashpos != -1) {
      // Found an extra '-' in the 4th param, will split.
      successPar  = validIntFromString(gpio_descr.substring(dashpos + 1), par2);
      successPar &= validIntFromString(gpio_descr.substring(0, dashpos), par1);
    }

    if (successPar) {
      const __FlashStringHelper * logPrefix = F("");

      switch (device[0]) {
#ifdef USES_P009
        case 'm':
        {
          uint16_t tempValue = 0;
          logPrefix = F("MCP");
          success   = mcpgpio_plugin_range_helper(par1, par2, tempValue);
          str       = String(tempValue);
          break;
        }
#endif

#ifdef USES_P019
        case 'p':
        {
          uint16_t tempValue = 0;
          logPrefix = F("PCF");
          success   = pcfgpio_plugin_range_helper(par1, par2, tempValue);
          str       = String(tempValue);
          break;
        }
#endif
        default:
          addLog(LOG_LEVEL_ERROR, F("PLUGIN PINSTATE. Plugin not included in build"));
          return false;

      }

      if (success) {
        #ifndef BUILD_NO_DEBUG
        addLog(LOG_LEVEL_DEBUG, concat(
          logPrefix,
          strformat(F(" PLUGIN RANGE pin start=%d; pin end=%d; value=%s"), par1, par2, str.c_str())));
        #endif // ifndef BUILD_NO_DEBUG
      } else {
        addLog(LOG_LEVEL_ERROR, concat(
          logPrefix,
          strformat(F(" IS OFFLINE. PLUGIN RANGE pin start=%d; pin end=%d; value=%s"), par1, par2, str.c_str())));
      }
    } else {
      addLog(LOG_LEVEL_ERROR, F(" PLUGIN PINRANGE. Syntax error. Pin parameters are not numeric."));
    }
  } else {
    addLog(LOG_LEVEL_ERROR, F("Syntax error. Invalid command. Valid commands are 'pinstate' and 'pinrange'."));
  }

  if (!success) {
    str = '0';
  }
  return success;
}

#ifdef USES_P009
bool mcpgpio_plugin_range_helper(uint8_t pin1, uint8_t pin2, uint16_t& result)
{
  const range_pattern_helper_data data = range_helper_shared(PLUGIN_MCP, pin1, pin2);

  if (!data.valid) {
    return false;
  }

  //  data.logPrefix += F("PluginRead");

  String log;
  bool success = false;
  uint32_t tempResult = 0;

  for (uint8_t i = 0; i < data.numBytes; i++) {
    uint8_t readValue                 = 0;
    const uint8_t currentVal          = data.initVal + i;
    const uint8_t currentAddress      = static_cast<int>(currentVal / 2);
    const uint8_t currentGPIORegister = ((currentVal % 2) == 0) ? MCP23017_GPIOA : MCP23017_GPIOB;

    const bool onLine = GPIO_MCP_ReadRegister(currentAddress, currentGPIORegister, &readValue);

    if (onLine) {
      success = true; // One valid address
      tempResult += (static_cast<uint32_t>(readValue) << (8 * i)); 
    }
  }

  tempResult  = tempResult >> data.deltaStart;
  tempResult &= ((1 << data.numBits) - 1);
  result      = uint16_t(tempResult);

  return success;
}
#endif

#ifdef USES_P019
bool pcfgpio_plugin_range_helper(uint8_t pin1, uint8_t pin2, uint16_t& result)
{
  const range_pattern_helper_data data = range_helper_shared(PLUGIN_PCF, pin1, pin2);

  if (!data.valid) {
    return false;
  }

  //  data.logPrefix += F("PluginRead");

  String log;

  bool success = false;
  uint32_t tempResult = 0;

  for (uint8_t i = 0; i < data.numBytes; i++) {
    uint8_t readValue         = 0;
    const uint8_t currentAddress = getPcfAddress(pin1 + 8 * i);

    const bool onLine = GPIO_PCF_ReadAllPins(currentAddress, &readValue);

    if (onLine) { 
      success = true; // One valid address
      tempResult += (static_cast<uint32_t>(readValue) << (8 * i)); 
    }
  }

  tempResult  = tempResult >> data.deltaStart;
  tempResult &= ((1 << data.numBits) - 1);
  result      = uint16_t(tempResult);

  return success;
}
#endif
#include "../Commands/Provisioning.h"

#if FEATURE_CUSTOM_PROVISIONING

# include "../Commands/Common.h"
# include "../DataTypes/ESPEasyFileType.h"
# include "../DataStructs/ESPEasy_EventStruct.h"
# include "../Helpers/ESPEasy_Storage.h"
# include "../Helpers/Networking.h"
# include "../Helpers/StringConverter.h"

String Command_Provisioning_Dispatcher(struct EventStruct *event,
                                       const char         *Line)
{
  const String cmd = parseString(Line, 2);

  if (equals(cmd, F("config"))) {
    return Command_Provisioning_Config();
  } else
  if (equals(cmd, F("firmware"))) {
    return Command_Provisioning_Firmware(event, Line);
  } else
  # if FEATURE_NOTIFIER
  if (equals(cmd, F("notification"))) {
    return Command_Provisioning_Notification();
  } else
  # endif // if FEATURE_NOTIFIER
  if (equals(cmd, F("provision"))) {
    return Command_Provisioning_Provision();
  } else
  if (equals(cmd, F("rules"))) {
    return Command_Provisioning_Rules(event);
  } else
  if (equals(cmd, F("security"))) {
    return Command_Provisioning_Security();
  }
  return return_command_failed_flashstr();
}

String Command_Provisioning_Config()
{
  return downloadFileType(FileType::CONFIG_DAT);
}

String Command_Provisioning_Security()
{
  return downloadFileType(FileType::SECURITY_DAT);
}

# if FEATURE_NOTIFIER
String Command_Provisioning_Notification()
{
  return downloadFileType(FileType::NOTIFICATION_DAT);
}

# endif // if FEATURE_NOTIFIER

String Command_Provisioning_Provision()
{
  return downloadFileType(FileType::PROVISIONING_DAT);
}

String Command_Provisioning_Rules(struct EventStruct *event)
{
  if ((event->Par2 <= 0) || (event->Par2 > 4)) {
    return F("Provision,Rules: rules index out of range");
  }
  return downloadFileType(FileType::RULES_TXT, event->Par2 - 1);
}

String Command_Provisioning_Firmware(struct EventStruct *event, const char *Line)
{
  // FIXME TD-er: Must only allow to use set prefix in the provisioning settings
  const String url = parseStringToEndKeepCase(Line, 3);
  String error;

  downloadFirmware(url, error); // Events are sent from download handler
  return error;
}

# ifdef PLUGIN_BUILD_MAX_ESP32
void Command_Provisioning_DeprecatedMessage(const String& param) {
  addLog(LOG_LEVEL_ERROR, strformat(F("WARNING: 'Provision%s' is deprecated, change to 'Provision,%s'"), param.c_str(), param.c_str()));
}

String Command_Provisioning_ConfigFallback(struct EventStruct *event, const char *Line)
{
  Command_Provisioning_DeprecatedMessage(F("Config"));
  return Command_Provisioning_Config();
}

String Command_Provisioning_SecurityFallback(struct EventStruct *event, const char *Line)
{
  Command_Provisioning_DeprecatedMessage(F("Security"));
  return Command_Provisioning_Security();
}

#  if FEATURE_NOTIFIER
String Command_Provisioning_NotificationFallback(struct EventStruct *event, const char *Line)
{
  Command_Provisioning_DeprecatedMessage(F("Notification"));
  return Command_Provisioning_Notification();
}

String Command_Provisioning_ProvisionFallback(struct EventStruct *event, const char *Line)
{
  Command_Provisioning_DeprecatedMessage(F("Provision"));
  return Command_Provisioning_Provision();
}

String Command_Provisioning_RulesFallback(struct EventStruct *event, const char *Line)
{
  Command_Provisioning_DeprecatedMessage(F("Rules,<n>"));

  if ((event->Par1 <= 0) || (event->Par1 > 4)) {
    return F("ProvisionRules: rules index out of range");
  }
  return downloadFileType(FileType::RULES_TXT, event->Par1 - 1);
}

String Command_Provisioning_FirmwareFallback(struct EventStruct *event, const char *Line)
{
  Command_Provisioning_DeprecatedMessage(F("Firmware,<Firmware.bin>"));

  // FIXME TD-er: Must only allow to use set prefix in the provisioning settings
  const String url = parseStringToEndKeepCase(Line, 2);
  String error;

  downloadFirmware(url, error);
  return error;
}

#  endif // if FEATURE_NOTIFIER

# endif // ifdef PLUGIN_BUILD_MAX_ESP32

#endif // if FEATURE_CUSTOM_PROVISIONING

#include "../Commands/wd.h"

#ifndef LIMIT_BUILD_SIZE

#include "../Commands/Common.h"

#include "../DataStructs/ESPEasy_EventStruct.h"

#include "../ESPEasyCore/Serial.h"

#include "../Helpers/StringConverter.h"


#include <Wire.h>

const __FlashStringHelper * Command_WD_Config(EventStruct *event, const char* Line)
{
  Wire.beginTransmission(event->Par1);  // address
  Wire.write(event->Par2);              // command
  Wire.write(event->Par3);              // data
  Wire.endTransmission();
  return return_command_success_flashstr();
}

String Command_WD_Read(EventStruct *event, const char* Line)
{
  Wire.beginTransmission(event->Par1);  // address
  Wire.write(0x83);                     // command to set pointer
  Wire.write(event->Par2);              // pointer value
  Wire.endTransmission();
  if ( Wire.requestFrom(static_cast<uint8_t>(event->Par1), static_cast<uint8_t>(1)) == 1 )
  {
    uint8_t value = Wire.read();
    return return_result(
      event, 
      concat(F("I2C Read address "),  formatToHex(event->Par1)) 
    + concat(F(" Value "), formatToHex(value)));
  }
  return return_command_success();
}

#endif
#include "../Commands/InternalCommands_decoder.h"

#include "../DataStructs/TimingStats.h"
#include "../Helpers/StringConverter.h"

// Keep the order of elements in ESPEasy_cmd_e enum
// the same as in the PROGMEM strings below
//
// The first item in the PROGMEM strings below should be an enum
// which is always included in each build as it is used to set an offset
// to compute the final enum value.
//
// Keep the offset used in match_ESPEasy_internal_command in sync
// when adding new commands


const char Internal_commands_ab[] PROGMEM =
  "accessinfo|"
  "asyncevent|"
  "build|"
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
  "background|"
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
#ifdef USES_C012
  "blynkget|"
#endif // #ifdef USES_C012
#ifdef USES_C015
  "blynkset|"
#endif // #ifdef USES_C015
;

#define Int_cmd_c_offset ESPEasy_cmd_e::clearaccessblock
const char Internal_commands_c[] PROGMEM =
  "clearaccessblock|"
  "clearpassword|"
  "clearrtcram|"
#ifdef ESP8266
  "clearsdkwifi|"
  "clearwifirfcal|"
#endif // #ifdef ESP8266
  "config|"
  "controllerdisable|"
  "controllerenable|"
;

#define Int_cmd_d_offset ESPEasy_cmd_e::datetime
const char Internal_commands_d[] PROGMEM =
  "datetime|"
  "debug|"
  "dec|"
  "deepsleep|"
  "delay|"
#if FEATURE_PLUGIN_PRIORITY
  "disableprioritytask|"
#endif // #if FEATURE_PLUGIN_PRIORITY
  "dns|"
  "dst|"
;

#define Int_cmd_e_offset ESPEasy_cmd_e::erasesdkwifi
const char Internal_commands_e[] PROGMEM =
  "erasesdkwifi|"
  "event|"
  "executerules|"
#if FEATURE_ETHERNET
  "ethphyadr|"
  "ethpinmdc|"
  "ethpinmdio|"
  "ethpinpower|"
  "ethphytype|"
  "ethclockmode|"
  "ethip|"
  "ethgateway|"
  "ethsubnet|"
  "ethdns|"
  "ethdisconnect|"
  "ethwifimode|"
#endif // FEATURE_ETHERNET
;

#define Int_cmd_fghij_offset ESPEasy_cmd_e::factoryreset
const char Internal_commands_fghij[] PROGMEM =
  "factoryreset|"
  "gateway|"
  "gpio|"
  "gpiotoggle|"
  "hiddenssid|"
  "i2cscanner|"
  "inc|"
  "ip|"
#if FEATURE_USE_IPV6
  "ip6|"
#endif
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
  "jsonportstatus|"
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
;

#if FEATURE_LAT_LONG_VAR_CMD
#define Int_cmd_l_offset ESPEasy_cmd_e::latitude
#else // if FEATURE_LAT_LONG_VAR_CMD
#define Int_cmd_l_offset ESPEasy_cmd_e::let
#endif // if FEATURE_LAT_LONG_VAR_CMD
const char Internal_commands_l[] PROGMEM =
  #if FEATURE_LAT_LONG_VAR_CMD
  "latitude|"
  #endif // if FEATURE_LAT_LONG_VAR_CMD
  "let|"
  #if FEATURE_STRING_VARIABLES
  "letstr|"
  #endif
  "load|"
  "logentry|"
  #if FEATURE_LAT_LONG_VAR_CMD
  "longitude|"
  #endif // if FEATURE_LAT_LONG_VAR_CMD
  "looptimerset|"
  "looptimerset_ms|"
  "looptimersetandrun|"
  "looptimersetandrun_ms|"
  "longpulse|"
  "longpulse_ms|"
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
  "logportstatus|"
  "lowmem|"
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
;

#define Int_cmd_m_offset ESPEasy_cmd_e::monitor
const char Internal_commands_m[] PROGMEM =
  "monitor|"
  "monitorrange|"
#ifdef USES_P009
  "mcpgpio|"
  "mcpgpiorange|"
  "mcpgpiopattern|"
  "mcpgpiotoggle|"
  "mcplongpulse|"
  "mcplongpulse_ms|"
  "mcpmode|"
  "mcpmoderange|"
  "mcppulse|"
#endif // #ifdef USES_P009
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
  "malloc|"
  "meminfo|"
  "meminfodetail|"
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
;

#define Int_cmd_no_offset ESPEasy_cmd_e::name
const char Internal_commands_no[] PROGMEM =
  "name|"
  "nosleep|"
#if FEATURE_NOTIFIER
  "notify|"
#endif // #if FEATURE_NOTIFIER
  "ntphost|"
#if FEATURE_DALLAS_HELPER && FEATURE_COMMAND_OWSCAN
  "owscan|"
#endif // if FEATURE_DALLAS_HELPER && FEATURE_COMMAND_OWSCAN
;

#define Int_cmd_p_offset ESPEasy_cmd_e::password
const char Internal_commands_p[] PROGMEM =
  "password|"
#ifdef USES_P019
  "pcfgpio|"
  "pcfgpiorange|"
  "pcfgpiopattern|"
  "pcfgpiotoggle|"
  "pcflongpulse|"
  "pcflongpulse_ms|"
  "pcfmode|"
  "pcfmoderange|"
  "pcfpulse|"
#endif // #ifdef USES_P019
#if FEATURE_POST_TO_HTTP
  "posttohttp|"
#if FEATURE_HTTP_TLS
  "posttohttps|"
#endif // if FEATURE_HTTP_TLS
#endif // #if FEATURE_POST_TO_HTTP
#if FEATURE_CUSTOM_PROVISIONING
  "provision|"
 # ifdef PLUGIN_BUILD_MAX_ESP32 // FIXME DEPRECATED: Fallback for temporary backward compatibility
  "provisionconfig|"
  "provisionsecurity|"
  #  if FEATURE_NOTIFIER
  "provisionnotification|"
  #  endif // #if FEATURE_NOTIFIER
  "provisionprovision|"
  "provisionrules|"
  "provisionfirmware|"
 # endif // #ifdef PLUGIN_BUILD_MAX_ESP32
#endif   // #if FEATURE_CUSTOM_PROVISIONING
  "pulse|"
#if FEATURE_MQTT
  "publish|"
  "publishr|"
#endif // #if FEATURE_MQTT
#if FEATURE_PUT_TO_HTTP
  "puttohttp|"
#if FEATURE_HTTP_TLS
  "puttohttps|"
#endif // if FEATURE_HTTP_TLS
#endif // #if FEATURE_PUT_TO_HTTP
  "pwm|"
;

#define Int_cmd_r_offset ESPEasy_cmd_e::reboot
const char Internal_commands_r[] PROGMEM =
  "reboot|"
  "resetflashwritecounter|"
  "restart|"
  "rtttl|"
  "rules|"
;

#define Int_cmd_s_offset ESPEasy_cmd_e::save
const char Internal_commands_s[] PROGMEM =
  "save|"
  "scheduletaskrun|"
#if FEATURE_SD
  "sdcard|"
  "sdremove|"
#endif // #if FEATURE_SD
#if FEATURE_ESPEASY_P2P
  "sendto|"
#endif // #if FEATURE_ESPEASY_P2P
#if FEATURE_SEND_TO_HTTP
  "sendtohttp|"
#if FEATURE_HTTP_TLS
  "sendtohttps|"
#endif // if FEATURE_HTTP_TLS
#endif // FEATURE_SEND_TO_HTTP
  "sendtoudp|"
  "sendtoudpmix|"
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
  "serialfloat|"
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
  "settings|"
#if FEATURE_SERVO
  "servo|"
#endif // #if FEATURE_SERVO
  "status|"
  "subnet|"
#if FEATURE_MQTT
  "subscribe|"
#endif // #if FEATURE_MQTT
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
  "sysload|"
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
;

#define Int_cmd_t_offset ESPEasy_cmd_e::taskclear
const char Internal_commands_t[] PROGMEM =
  "taskclear|"
  "taskclearall|"
  "taskdisable|"
  "taskenable|"
  "taskrun|"
  "taskrunat|"
  "taskvalueset|"
  #if FEATURE_STRING_VARIABLES
  "taskvaluesetderived|"
  "taskvaluesetpresentation|"
  #endif // if FEATURE_STRING_VARIABLES
  "taskvaluetoggle|"
  "taskvaluesetandrun|"
  "timerpause|"
  "timerresume|"
  "timerset|"
  "timerset_ms|"
  "timezone|"
  "tone|"
;

#define Int_cmd_u_offset ESPEasy_cmd_e::udpport
const char Internal_commands_u[] PROGMEM =
  "udpport|"
#if FEATURE_ESPEASY_P2P
  "udptest|"
#endif // #if FEATURE_ESPEASY_P2P
  "unit|"
  "unmonitor|"
  "unmonitorrange|"
  "usentp|"
;

#define Int_cmd_w_offset ESPEasy_cmd_e::wifiallowap
const char Internal_commands_w[] PROGMEM =
  "wifiallowap|"
  "wifiapmode|"
  "wificonnect|"
  "wifidisconnect|"
  "wifikey|"
  "wifikey2|"
  "wifimode|"
  "wifiscan|"
  "wifissid|"
  "wifissid2|"
  "wifistamode|"
#ifndef LIMIT_BUILD_SIZE
  "wdconfig|"
  "wdread|"
#endif // ifndef LIMIT_BUILD_SIZE
;

const char* getInternalCommand_Haystack_Offset(const char firstLetter, int& offset)
{
  const char *haystack = nullptr;

  offset = static_cast<int>(ESPEasy_cmd_e::NotMatched);

  // Keep the offset in sync when adding new commands
  switch (firstLetter)
  {
    case 'a':
    case 'b':
      offset   = 0;
      haystack = Internal_commands_ab;
      break;
    case 'c':
      offset   = static_cast<int>(Int_cmd_c_offset);
      haystack = Internal_commands_c;
      break;
    case 'd':
      offset   = static_cast<int>(Int_cmd_d_offset);
      haystack = Internal_commands_d;
      break;
    case 'e':
      offset   = static_cast<int>(Int_cmd_e_offset);
      haystack = Internal_commands_e;
      break;
    case 'f':
    case 'g':
    case 'h':
    case 'i':
    case 'j':
      offset   = static_cast<int>(Int_cmd_fghij_offset);
      haystack = Internal_commands_fghij;
      break;
    case 'l':
      offset   = static_cast<int>(Int_cmd_l_offset);
      haystack = Internal_commands_l;
      break;
    case 'm':
      offset   = static_cast<int>(Int_cmd_m_offset);
      haystack = Internal_commands_m;
      break;
    case 'n':
    case 'o':
      offset   = static_cast<int>(Int_cmd_no_offset);
      haystack = Internal_commands_no;
      break;
    case 'p':
      offset   = static_cast<int>(Int_cmd_p_offset);
      haystack = Internal_commands_p;
      break;
    case 'r':
      offset   = static_cast<int>(Int_cmd_r_offset);
      haystack = Internal_commands_r;
      break;
    case 's':
      offset   = static_cast<int>(Int_cmd_s_offset);
      haystack = Internal_commands_s;
      break;
    case 't':
      offset   = static_cast<int>(Int_cmd_t_offset);
      haystack = Internal_commands_t;
      break;
    case 'u':
      offset   = static_cast<int>(Int_cmd_u_offset);
      haystack = Internal_commands_u;
      break;
    case 'w':
      offset   = static_cast<int>(Int_cmd_w_offset);
      haystack = Internal_commands_w;
      break;

    default:
      return nullptr;
  }
  return haystack;
}

ESPEasy_cmd_e match_ESPEasy_internal_command(const String& cmd)
{
  START_TIMER;
  ESPEasy_cmd_e res = ESPEasy_cmd_e::NotMatched;

  if (cmd.length() < 2) {
    // No commands less than 2 characters
    return res;
  }

  int offset           = 0;
  const char *haystack = getInternalCommand_Haystack_Offset(cmd[0], offset);

  if (haystack == nullptr) {
/*
    addLog(LOG_LEVEL_ERROR, strformat(
             F("Internal command: No Haystack/offset '%s', offset: %d"),
             cmd.c_str(),
             offset));
*/
    return res;
  }


  if (haystack != nullptr) {
    const int command_i = GetCommandCode(cmd.c_str(), haystack);

    if (command_i != -1) {
      res = static_cast<ESPEasy_cmd_e>(command_i + offset);
    }
/*
    else {
      addLog(LOG_LEVEL_ERROR, strformat(
               F("Internal command: Not found '%s', haystack: %s"),
               cmd.c_str(),
               String(haystack).c_str()));
    }
*/
  }
  STOP_TIMER(COMMAND_DECODE_INTERNAL);
  return res;
}

#ifndef BUILD_NO_DEBUG
bool toString(ESPEasy_cmd_e cmd, String& str)
{
  if (cmd == ESPEasy_cmd_e::NotMatched) {
    return false;
  }
  char c     = 'z';
  bool found = false;
  int  offset;
  const char *haystack = nullptr;

  while (!found && c >= 'a') {
    haystack = getInternalCommand_Haystack_Offset(c, offset);

    if (haystack != nullptr) {
      if (offset <= static_cast<int>(cmd)) {
        found = true;
      }
    }
    --c;
  }

  if (found) {
    const int index = static_cast<int>(cmd) - offset;
/*
    addLog(LOG_LEVEL_INFO, strformat(
             F("Internal command: cmd=%d offset=%d index=%d"),
             static_cast<int>(cmd),
             offset,
             index));
*/

    if ((index >= 0) && (haystack != nullptr)) {
      // Likely long enough to parse any command
      char temp[32]{};
      str = GetTextIndexed(temp, sizeof(temp), index, haystack);
      return !str.isEmpty();
    }
  }
  return false;
}

bool checkAll_internalCommands()
{
  constexpr int last = static_cast<int>(ESPEasy_cmd_e::NotMatched);
  bool no_error      = true;

  for (int i = 0; i < last; ++i) {
    const ESPEasy_cmd_e cmd = static_cast<ESPEasy_cmd_e>(i);
    String cmd_str;

    if (!toString(cmd, cmd_str)) {
      no_error = false;
//      addLog(LOG_LEVEL_ERROR, concat(F("Internal command: no matching string for "), i));
    } else {
      const ESPEasy_cmd_e cmd_found = match_ESPEasy_internal_command(cmd_str);

      if (cmd_found != cmd) {
        if (cmd_str.isEmpty()) {
          addLog(LOG_LEVEL_ERROR, strformat(
                   F("Internal command: mismatch (%d)"), i));
        }
        else {
          addLog(LOG_LEVEL_ERROR, strformat(
                   F("Internal command: mismatch '%s' (%d)"),
                   cmd_str.c_str(),
                   i));
        }
        no_error = false;
      }
    }
  }

  if (no_error) {
    addLog(LOG_LEVEL_INFO, F("Internal command: All checked OK"));
  } 
/*
  else {
    const int index = static_cast<int>(match_ESPEasy_internal_command(F("build")));
    addLog(LOG_LEVEL_ERROR, concat(F("Internal command: index 'build'="), index));
  }
*/
  return no_error;
}

#endif // ifndef BUILD_NO_DEBUG

#include "../Commands/Settings.h"

#include "../../ESPEasy_common.h"

#include "../Commands/Common.h"

#include "../CustomBuild/CompiletimeDefines.h"

#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/Serial.h"

#include "../Globals/SecuritySettings.h"
#include "../Globals/Settings.h"

#include "../Helpers/ESPEasy_FactoryDefault.h"
#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/Memory.h"
#include "../Helpers/MDNS_Helper.h"
#include "../Helpers/Misc.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringGenerator_System.h"


String Command_Settings_Build(struct EventStruct *event, const char* Line)
{
	if (HasArgv(Line, 2)) {
	  Settings.Build = event->Par1;
	} else {
      return return_result(event, concat(F("Build:"), static_cast<int>(Settings.Build)));
	}
	return return_command_success();
}

String Command_Settings_Unit(struct EventStruct *event, const char* Line)
{
	if (HasArgv(Line, 2)) {
	  Settings.Unit = event->Par1;
	  update_mDNS();
	} else {
      return return_result(event, concat(F("Unit:"), static_cast<int>(Settings.Unit)));
	}
	return return_command_success();
}

String Command_Settings_Name(struct EventStruct *event, const char* Line)
{
	return Command_GetORSetString(event, F("Name:"),
							Line,
							Settings.Name,
							sizeof(Settings.Name),
							1);
}

String Command_Settings_Password(struct EventStruct *event, const char* Line)
{
	return Command_GetORSetString(event, F("Password:"),
				      Line,
				      SecuritySettings.Password,
				      sizeof(SecuritySettings.Password),
				      1
				      );
}

const __FlashStringHelper *  Command_Settings_Password_Clear(struct EventStruct *event, const char* Line)
{
	const String storedPassword = SecuritySettings.getPassword();
	if (storedPassword.length() > 0) {
		// There is a password set, so we must check it.
		const String password = parseStringKeepCase(Line, 2);
		if (!storedPassword.equals(password)) {
			return return_command_failed_flashstr();
		}
        ZERO_FILL(SecuritySettings.Password);
	}
	return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Settings_Save(struct EventStruct *event, const char* Line)
{
	SaveSettings();
	return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Settings_Load(struct EventStruct *event, const char* Line)
{
	LoadSettings();
	return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Settings_Print(struct EventStruct *event, const char* Line)
{
	serialPrintln();

	serialPrintln(F("System Info"));
	serialPrint(F("  IP Address    : ")); serialPrintln(formatIP(NetworkLocalIP()));
	serialPrint(F("  Build         : ")); serialPrintln(String(get_build_nr()) + '/' + getSystemBuildString());
	serialPrint(F("  Name          : ")); serialPrintln(Settings.getName());
	serialPrint(F("  Unit          : ")); serialPrintln(String(static_cast<int>(Settings.Unit)));
	serialPrint(F("  WifiSSID      : ")); serialPrintln(SecuritySettings.WifiSSID);
	serialPrint(F("  WifiKey       : ")); serialPrintln(SecuritySettings.WifiKey);
	serialPrint(F("  WifiSSID2     : ")); serialPrintln(SecuritySettings.WifiSSID2);
	serialPrint(F("  WifiKey2      : ")); serialPrintln(SecuritySettings.WifiKey2);
	serialPrint(F("  Free mem      : ")); serialPrintln(String(FreeMem()));
	return return_see_serial(event);
}

const __FlashStringHelper * Command_Settings_FactoryReset(struct EventStruct *event, const char* Line)
{
	ResetFactory();
	reboot(IntendedRebootReason_e::ResetFactoryCommand);
	return return_command_success_flashstr();
}

#include "../Commands/Common.h"
#include "../Commands/LatitudeLongitude.h"
#include "../Globals/Settings.h"

String Command_Latitude(struct EventStruct *event,
                        const char         *Line) {
  return Command_GetORSetFloatMinMax(event, F("Latitude:"), Line, &Settings.Latitude, 1, -90.0001f, 90.0001f);
}

String Command_Longitude(struct EventStruct *event,
                         const char         *Line) {
  return Command_GetORSetFloatMinMax(event, F("Longitude:"), Line, &Settings.Longitude, 1, -180.0001f, 180.0001f);
}

#include "../Commands/Servo.h"

#include "../Commands/Common.h"
#if FEATURE_SERVO

#include "../DataStructs/EventStructCommandWrapper.h"
#include "../DataStructs/PinMode.h"
#include "../DataStructs/PortStatusStruct.h"
#include "../ESPEasyCore/Controller.h"
#include "../ESPEasyCore/ESPEasyGPIO.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../Globals/GlobalMapPortStatus.h"
#include "../Helpers/Hardware_PWM.h"
#include "../Helpers/PortStatus.h"
#include "../Helpers/StringConverter.h"

// Needed also here for PlatformIO's library finder as the .h file 
// is in a directory which is excluded in the src_filter
# include <Servo.h>
ServoPinMap_t ServoPinMap;
#endif // if FEATURE_SERVO

const __FlashStringHelper * Command_Servo(struct EventStruct *event, const char *Line)
{
  #if FEATURE_SERVO

  // GPIO number is stored inside event->Par2 instead of event->Par1 as in all the other commands
  // So needs to reload the tempPortStruct.

  // FIXME TD-er: For now only fixed to "P001" even when it is for internal GPIO pins
  pluginID_t pluginID = PLUGIN_GPIO;

  // Par1: Servo ID (obsolete/unused since 2020/11/22)
  // Par2: GPIO pin
  // Par3: angle 0...180 degree
  if (checkValidPortRange(pluginID, event->Par2)) {
    portStatusStruct tempStatus;
    const uint32_t   key = createKey(pluginID, event->Par2); // WARNING: 'servo' uses Par2 instead of Par1
    // WARNING: operator [] creates an entry in the map if key does not exist
    // So the next command should be part of each command:
    tempStatus = globalMapPortStatus[key];

    String log = concat(F("Servo : GPIO "), event->Par2);

    // SPECIAL CASE TO ALLOW SERVO TO BE DETATTCHED AND SAVE POWER.
    if (event->Par3 >= 9000) {
      auto it = ServoPinMap.find(event->Par2);

      if (it != ServoPinMap.end()) {
        it->second.detach();
        # ifdef ESP32
          detachLedChannel(event->Par2);
        # endif // ifdef ESP32
        ServoPinMap.erase(it);
      }

      // Set parameters to make sure the port status will be removed.
      tempStatus.task    = 0;
      tempStatus.monitor = 0;
      tempStatus.command = 0;
      savePortStatus(key, tempStatus);
      addLog(LOG_LEVEL_INFO, concat(log, F(" Servo detached")));
      return return_command_success_flashstr();

    }
    # ifdef ESP32
      // Must keep track of used channels or else cause conflicts with PWM
      int8_t ledChannel = attachLedChannel(event->Par2);
      ServoPinMap[event->Par2].attach(event->Par2, ledChannel);
    # else // ifdef ESP32
      ServoPinMap[event->Par2].attach(event->Par2);
    # endif // ifdef ESP32
    ServoPinMap[event->Par2].write(event->Par3);

    tempStatus.command   = 1; // set to 1 in order to display the status in the PinStatus page
    tempStatus.state     = 1;
    tempStatus.output    = 1;
    tempStatus.dutyCycle = event->Par3;

    // setPinState(PLUGIN_ID_001, event->Par2, PIN_MODE_SERVO, event->Par3);
    tempStatus.mode = PIN_MODE_SERVO;
    savePortStatus(key, tempStatus);
    log += F(" Servo set to ");
    log += event->Par3;
    addLog(LOG_LEVEL_INFO, log);
    SendStatusOnlyIfNeeded(event, SEARCH_PIN_STATE, key, log, 0);

    // SendStatus(event, getPinStateJSON(SEARCH_PIN_STATE, PLUGIN_ID_001, event->Par2, log, 0));
    return return_command_success_flashstr();
  }
  #else // if FEATURE_SERVO
  #ifndef BUILD_MINIMAL_OTA
  addLog(LOG_LEVEL_ERROR, F("SERVO : command not included in build"));
  #endif
  #endif // FEATURE_SERVO
  return return_command_failed_flashstr();
}

#include "../Commands/Common.h"

#include <ctype.h>
#include <IPAddress.h>

#include "../../ESPEasy_common.h"


#include "../ESPEasyCore/ESPEasyWifi.h"
#include "../ESPEasyCore/Serial.h"

#include "../Helpers/Networking.h"
#include "../Helpers/Numerical.h"
#include "../Helpers/StringConverter.h"


// Simple function to return "Ok", to avoid flash string duplication in the firmware.
const __FlashStringHelper * return_command_success_flashstr() { return F("\nOK"); }
const __FlashStringHelper * return_command_failed_flashstr() { return F("\nFailed"); }

const __FlashStringHelper * return_command_boolean_result_flashstr(bool success) 
{
    return success ? return_command_success_flashstr() : return_command_failed_flashstr();
}


String return_command_success()
{
  return return_command_success_flashstr();
}

String return_command_failed()
{
  return return_command_failed_flashstr();
}

const __FlashStringHelper * return_incorrect_nr_arguments() { return F("Too many arguments, try using quotes!"); }
const __FlashStringHelper * return_incorrect_source() { return F("Command not allowed from this source!"); }
const __FlashStringHelper * return_not_connected() { return F("Not connected to WiFi"); }


String return_result(struct EventStruct *event, const String& result)
{
  serialPrintln();
  serialPrintln(result);

  if (event->Source == EventValueSource::Enum::VALUE_SOURCE_SERIAL) {
    return return_command_success();
  }
  return result;
}


const __FlashStringHelper * return_see_serial(struct EventStruct *event)
{
  return (event->Source == EventValueSource::Enum::VALUE_SOURCE_SERIAL)
    ? return_command_success_flashstr() 
    : F("Output sent to serial");
}


String Command_GetORSetIP(struct EventStruct *event,
                          const __FlashStringHelper * targetDescription,
                          const char         *Line,
                          uint8_t               *IP,
                          const IPAddress   & dhcpIP,
                          int                 arg)
{
  bool hasArgument = false;
  {
    // Check if command is valid. Leave in separate scope to delete the TmpStr1
    String TmpStr1;

    if (GetArgv(Line, TmpStr1, arg + 1)) {
      hasArgument = true;

      if (!str2ip(TmpStr1, IP)) {
        return return_result(event, concat(F("Invalid parameter: "), TmpStr1));
      }
    }
  }

  if (!hasArgument) {
    String result = targetDescription;

    if (useStaticIP()) {
      result += formatIP(IP);
    } else {
      result += formatIP(dhcpIP);
      result += F("(DHCP)");
    }
    return return_result(event, result);
  }
  return return_command_success();
}

String Command_GetORSetString(struct EventStruct *event,
                              const __FlashStringHelper * targetDescription,
                              const char         *Line,
                              char               *target,
                              size_t              len,
                              int                 arg
                              )
{
  bool hasArgument = false;
  {
    // Check if command is valid. Leave in separate scope to delete the TmpStr1
    String TmpStr1;

    if (GetArgv(Line, TmpStr1, arg + 1)) {
      hasArgument = true;

      if (TmpStr1.length() > len) {
        String result = concat(targetDescription, F(" is too large. max size is "));
        result += len;
        return return_result(event, result);
      }
      safe_strncpy(target, TmpStr1, len);
    }
  }

  if (hasArgument) {
    String result = targetDescription;
    result += target;
    return return_result(event, result);
  }
  return return_command_success();
}

String Command_GetORSetBool(struct EventStruct *event,
                            const __FlashStringHelper * targetDescription,
                            const char         *Line,
                            bool               *value,
                            int                 arg)
{
  bool hasArgument = false;
  {
    // Check if command is valid. Leave in separate scope to delete the TmpStr1
    String TmpStr1;

    if (GetArgv(Line, TmpStr1, arg + 1)) {
      hasArgument = true;
      TmpStr1.toLowerCase();

      int32_t tmp_int = 0;
      if (validIntFromString(TmpStr1, tmp_int)) {
        *value = tmp_int > 0;
      }
      else if (equals(TmpStr1, F("on"))) { *value = true; }
      else if (equals(TmpStr1, F("true"))) { *value = true; }
      else if (equals(TmpStr1, F("off"))) { *value = false; }
      else if (equals(TmpStr1, F("false"))) { *value = false; }
    }
  }

  if (hasArgument) {
    return return_result(event, concat(targetDescription, boolToString(*value)));
  }
  return return_command_success();
}

#if FEATURE_ETHERNET
String Command_GetORSetETH(struct EventStruct *event,
                            const __FlashStringHelper * targetDescription,
                            const __FlashStringHelper * valueToString,
                            const char         *Line,
                            uint8_t            *value,
                            int                 arg)
{
  bool hasArgument = false;
  {
    // Check if command is valid. Leave in separate scope to delete the TmpStr1
    String TmpStr1;

    if (GetArgv(Line, TmpStr1, arg + 1)) {
      hasArgument = true;
      TmpStr1.toLowerCase();

      int32_t tmp_int = 0;
      if (validIntFromString(TmpStr1, tmp_int)) {
        *value = static_cast<uint8_t>(tmp_int);
      }

      // FIXME TD-er: This should not be in a generic function, but rather pre-processed in the command itself


      // WiFi/Eth mode
      else if (equals(TmpStr1, F("wifi"))) { *value = 0; }
      else if (equals(TmpStr1, F("ethernet"))) { *value = 1; }

      // ETH clockMode
      else if (TmpStr1.startsWith(F("ext"))) { *value = 0; }
      else if (TmpStr1.indexOf(F("gpio0"))  != -1) { *value = 1; }
      else if (TmpStr1.indexOf(F("gpio16")) != -1) { *value = 2; }
      else if (TmpStr1.indexOf(F("gpio17")) != -1) { *value = 3; }
    }
  }

  String result = targetDescription;
  if (hasArgument) {
    result += *value;
  } else {
    result += valueToString;
  }
  return return_result(event, result);
}
#endif

String Command_GetORSetInt8_t(struct EventStruct *event,
                            const __FlashStringHelper * targetDescription,
                            const char         *Line,
                            int8_t             *value,
                            int                 arg)
{
  bool hasArgument = false;
  {
    // Check if command is valid. Leave in separate scope to delete the TmpStr1
    String TmpStr1;

    if (GetArgv(Line, TmpStr1, arg + 1)) {
      hasArgument = true;
      TmpStr1.toLowerCase();

      int32_t tmp_int = 0;
      if (validIntFromString(TmpStr1, tmp_int)) {
        *value = static_cast<int8_t>(tmp_int);
      }
    }
  }

  if (hasArgument) {
    String result = targetDescription;
    result += *value;
    return return_result(event, result);
  }
  return return_command_success();
}

String Command_GetORSetFloatMinMax(struct EventStruct        *event,
                                   const __FlashStringHelper *targetDescription,
                                   const char                *Line,
                                   float                     *value,
                                   int                        arg,
                                   float                      _min,
                                   float                      _max)
{
  {
    // Check if command is valid. Leave in separate scope to delete the TmpStr1
    String TmpStr1;

    if (GetArgv(Line, TmpStr1, arg + 1)) {
      TmpStr1.toLowerCase();

      float tmp_float{};

      if (validFloatFromString(TmpStr1, tmp_float) &&
          definitelyGreaterThan(tmp_float, _min) &&
          definitelyLessThan(tmp_float, _max)) {
        *value = tmp_float;
      } else {
        return return_command_failed();
      }
    }
  }

  return return_result(event, concat(targetDescription, toString(*value)));
}

#include "../Commands/Rules.h"


#include "../../ESPEasy_common.h"


#include "../Commands/Common.h"

#include "../DataTypes/EventValueSource.h"

#include "../ESPEasyCore/Controller.h"
#include "../ESPEasyCore/ESPEasyRules.h"

#include "../Globals/EventQueue.h"
#include "../Globals/ExtraTaskSettings.h"
#include "../Globals/RulesCalculate.h"
#include "../Globals/RuntimeData.h"
#include "../Globals/Settings.h"

#include "../Helpers/Misc.h"
#include "../Helpers/StringConverter.h"

const __FlashStringHelper * Command_Rules_Execute(struct EventStruct *event, const char *Line)
{
  String filename;

  if (GetArgv(Line, filename, 2)) {
    String event;
    rulesProcessingFile(filename, event);
  }
  return return_command_success_flashstr();
}

String Command_Rules_UseRules(struct EventStruct *event, const char *Line)
{
  return Command_GetORSetBool(event, F("Rules:"),
                              Line,
                              (bool *)&Settings.UseRules,
                              1);
}

const __FlashStringHelper * Command_Rules_Async_Events(struct EventStruct *event, const char *Line)
{
  if (Settings.UseRules) {
    String eventName = parseStringToEndKeepCase(Line, 2);

    eventName.replace('$', '#');
    eventQueue.addMove(std::move(eventName));
  }
  return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Rules_Events(struct EventStruct *event, const char *Line)
{
  if (Settings.UseRules) {
    const bool executeImmediately =
      SourceNeedsStatusUpdate(event->Source) ||
      event->Source == EventValueSource::Enum::VALUE_SOURCE_RULES ||
      event->Source == EventValueSource::Enum::VALUE_SOURCE_RULES_RESTRICTED;

    String eventName = parseStringToEndKeepCase(Line, 2);

    eventName.replace('$', '#');
    if (executeImmediately) {
      rulesProcessing(eventName); // TD-er: Process right now
    } else {
      eventQueue.addMove(std::move(eventName));
    }
  }
  return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Rules_Let(struct EventStruct *event, const char *Line)
{
  String varName;
  String TmpStr1;
  GetArgv(Line, varName, 2);

  if (!varName.isEmpty() &&
      ExtraTaskSettings.checkInvalidCharInNames(varName.c_str()) && GetArgv(Line, TmpStr1, 3)) {
    ESPEASY_RULES_FLOAT_TYPE result{};

    if (!isError(Calculate(TmpStr1, result))) {
      setCustomFloatVar(varName, result);
      return return_command_success_flashstr();
    }
  }
  return return_command_failed_flashstr();
}

#if FEATURE_STRING_VARIABLES
const __FlashStringHelper * Command_Rules_LetStr(struct EventStruct *event, const char *Line)
{
  String varName;
  String TmpStr1;
  GetArgv(Line, varName, 2);

  if (!varName.isEmpty() &&
      ExtraTaskSettings.checkInvalidCharInNames(varName.c_str()) && GetArgv(Line, TmpStr1, 3)) {
    String result = parseTemplate(TmpStr1);

    setCustomStringVar(varName, result);
    return return_command_success_flashstr();
  }
  return return_command_failed_flashstr();
}
#endif // if FEATURE_STRING_VARIABLES

const __FlashStringHelper * Command_Rules_IncDec(struct EventStruct *event, const char *Line, const ESPEASY_RULES_FLOAT_TYPE factor)
{
  String varName;
  String TmpStr1;
  ESPEASY_RULES_FLOAT_TYPE result = 1;
  GetArgv(Line, varName, 2);

  if (!varName.isEmpty()) {
    if (GetArgv(Line, TmpStr1, 3) && isError(Calculate(TmpStr1, result))) {
      return return_command_failed_flashstr();
    }
    setCustomFloatVar(varName, getCustomFloatVar(varName) + (result * factor));
    return return_command_success_flashstr();
  }
  return return_command_failed_flashstr();
}

const __FlashStringHelper * Command_Rules_Inc(struct EventStruct *event, const char *Line)
{
  return Command_Rules_IncDec(event, Line, 1.0);
}

const __FlashStringHelper * Command_Rules_Dec(struct EventStruct *event, const char *Line)
{
  return Command_Rules_IncDec(event, Line, -1.0);
}

#include "../Commands/InternalCommands.h"

#include "../../ESPEasy_common.h"

#include "../../_Plugin_Helper.h"
#include "../Globals/Settings.h"

#if FEATURE_BLYNK
# include "../Commands/Blynk.h"
# include "../Commands/Blynk_c015.h"
#endif // if FEATURE_BLYNK

#include "../Commands/Common.h"
#include "../Commands/Controller.h"
#include "../Commands/Diagnostic.h"
#include "../Commands/GPIO.h"
#include "../Commands/HTTP.h"
#include "../Commands/InternalCommands_decoder.h"
#include "../Commands/i2c.h"

#if FEATURE_LAT_LONG_VAR_CMD
#include "../Commands/LatitudeLongitude.h"
#endif // if FEATURE_LAT_LONG_VAR_CMD

#if FEATURE_MQTT
# include "../Commands/MQTT.h"
#endif // if FEATURE_MQTT

#include "../Commands/Networks.h"
#if FEATURE_NOTIFIER
# include "../Commands/Notifications.h"
#endif // if FEATURE_NOTIFIER
#if FEATURE_DALLAS_HELPER && FEATURE_COMMAND_OWSCAN
#include "../Commands/OneWire.h"
#endif // if FEATURE_DALLAS_HELPER && FEATURE_COMMAND_OWSCAN
#include "../Commands/Provisioning.h"
#include "../Commands/RTC.h"
#include "../Commands/Rules.h"
#include "../Commands/SDCARD.h"
#include "../Commands/Settings.h"
#if FEATURE_SERVO
# include "../Commands/Servo.h"
#endif // if FEATURE_SERVO
#include "../Commands/System.h"
#include "../Commands/Tasks.h"
#include "../Commands/Time.h"
#include "../Commands/Timer.h"
#include "../Commands/UPD.h"
#include "../Commands/wd.h"
#include "../Commands/WiFi.h"

#include "../DataStructs/TimingStats.h"

#include "../ESPEasyCore/ESPEasy_Log.h"

#include "../Helpers/Misc.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringParser.h"


bool checkNrArguments(const char *cmd, const String& Line, int nrArguments) {
  if (nrArguments < 0) { return true; }

  // 0 arguments means argument on pos1 is valid (the command) and argpos 2 should not be there.
  if (HasArgv(Line.c_str(), nrArguments + 2)) {
    #ifndef BUILD_NO_DEBUG

    if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
      String log;

      if (reserve_special(log, 128)) {
        log += F("Too many arguments: cmd=");
        log += cmd;

        if (nrArguments < 1) {
          log += Line;
        } else {
          // Check for one more argument than allowed, since we apparently have one.
          bool done = false;
          int  i    = 1;

          while (!done) {
            String parameter;

            if (i == nrArguments) {
              parameter = tolerantParseStringKeepCase(Line, i + 1);
            } else {
              parameter = parseStringKeepCase(Line, i + 1);
            }
            done = parameter.isEmpty();

            if (!done) {
              if (i <= nrArguments) {
                if (Settings.TolerantLastArgParse() && (i == nrArguments)) {
                  log += F(" (fixed)");
                }
                log += F(" Arg");
              } else {
                log += F(" ExtraArg");
              }
              log += i;
              log += '=';
              log += parameter;
            }
            ++i;
          }
        }
        log += F(" lineLength=");
        log += Line.length();
        addLogMove(LOG_LEVEL_ERROR, log);
      }
      addLogMove(LOG_LEVEL_ERROR, strformat(F("Line: _%s_"), Line.c_str()));

      addLogMove(LOG_LEVEL_ERROR, concat(Settings.TolerantLastArgParse() ?
                                         F("Command executed, but may fail.") : F("Command not executed!"),
                                         F(" See: https://github.com/letscontrolit/ESPEasy/issues/2724")));
    }
    #endif // ifndef BUILD_NO_DEBUG

    if (Settings.TolerantLastArgParse()) {
      return true;
    }
    return false;
  }
  return true;
}

bool checkSourceFlags(EventValueSource::Enum source, EventValueSourceGroup::Enum group) {
  if (EventValueSource::partOfGroup(source, group)) {
    return true;
  }
  addLog(LOG_LEVEL_ERROR, return_incorrect_source());
  return false;
}

command_case_data::command_case_data(const char *cmd, struct EventStruct *event, const char *line) :
  cmd(cmd), event(event), line(line)
{
  cmd_lc = cmd;
  cmd_lc.toLowerCase();
}

InternalCommands::InternalCommands(const char *cmd, struct EventStruct *event, const char *line)
  : _data(cmd, event, line) {}


// Wrapper to reduce generated code by macro
bool InternalCommands::do_command_case_all(command_function_fs pFunc,
                                           int                 nrArguments)
{
  return do_command_case(_data, pFunc, nrArguments, EventValueSourceGroup::Enum::ALL);
}

bool InternalCommands::do_command_case_all(command_function pFunc,
                                           int              nrArguments)
{
  return do_command_case(_data, pFunc, nrArguments, EventValueSourceGroup::Enum::ALL);
}

// Wrapper to reduce generated code by macro
bool InternalCommands::do_command_case_all_restricted(command_function_fs pFunc,
                                                      int                 nrArguments)
{
  return do_command_case(_data,  pFunc, nrArguments, EventValueSourceGroup::Enum::RESTRICTED);
}

bool InternalCommands::do_command_case_all_restricted(command_function pFunc,
                                                      int              nrArguments)
{
  return do_command_case(_data,  pFunc, nrArguments, EventValueSourceGroup::Enum::RESTRICTED);
}

bool do_command_case_check(command_case_data         & data,
                           int                         nrArguments,
                           EventValueSourceGroup::Enum group)
{
  // The data struct is re-used on each attempt to process an internal command.
  // Re-initialize the only two members that may have been altered by a previous call.
  data.retval = false;
  free_string(data.status);

  if (!checkSourceFlags(data.event->Source, group)) {
    data.status = return_incorrect_source();
    return false;
  }

  // FIXME TD-er: Do not check nr arguments from MQTT source.
  // See https://github.com/letscontrolit/ESPEasy/issues/3344
  // C005 does recreate command partly from topic and published message
  // e.g. ESP_Easy/Bathroom_pir_env/GPIO/14 with data 0 or 1
  // This only allows for 2 parameters, but some commands need more arguments (default to "0")
  const bool mustCheckNrArguments = data.event->Source != EventValueSource::Enum::VALUE_SOURCE_MQTT;

  if (mustCheckNrArguments) {
    if (!checkNrArguments(data.cmd, data.line, nrArguments)) {
      data.status = return_incorrect_nr_arguments();

      // data.retval = false;
      return true;    // Command is handled
    }
  }
  data.retval = true; // Mark the command should be executed.
  return true;        // Command is handled
}

bool InternalCommands::do_command_case(command_case_data         & data,
                                       command_function_fs         pFunc,
                                       int                         nrArguments,
                                       EventValueSourceGroup::Enum group)
{
  if (do_command_case_check(data, nrArguments, group)) {
    // It has been handled, check if we need to execute it.
    // FIXME TD-er: Must change command function signature to use const String&
    START_TIMER;
    data.status = pFunc(data.event, data.line.c_str());
    STOP_TIMER(COMMAND_EXEC_INTERNAL);
    return true;
  }
  return false;
}

bool InternalCommands::do_command_case(command_case_data         & data,
                                       command_function            pFunc,
                                       int                         nrArguments,
                                       EventValueSourceGroup::Enum group)
{
  if (do_command_case_check(data, nrArguments, group)) {
    // It has been handled, check if we need to execute it.
    // FIXME TD-er: Must change command function signature to use const String&
    START_TIMER;
    data.status = pFunc(data.event, data.line.c_str());
    STOP_TIMER(COMMAND_EXEC_INTERNAL);
    return true;
  }
  return false;
}

bool InternalCommands::executeInternalCommand()
{
  // Simple macro to match command to function call.

  // EventValueSourceGroup::Enum::ALL
  #define COMMAND_CASE_A(C, NARGS) \
  do_command_case_all(&C, NARGS); break;

  // EventValueSourceGroup::Enum::RESTRICTED
  #define COMMAND_CASE_R(C, NARGS) \
   do_command_case_all_restricted(&C, NARGS); break;


  const ESPEasy_cmd_e cmd = match_ESPEasy_internal_command(_data.cmd_lc);

  _data.retval = false;

  if (cmd == ESPEasy_cmd_e::NotMatched) {
    return false;
  }

  // FIXME TD-er: Should we execute command when number of arguments is wrong?

  // FIXME TD-er: must determine nr arguments where NARGS is set to -1
  switch (cmd) {
    case ESPEasy_cmd_e::accessinfo:                 COMMAND_CASE_A(Command_AccessInfo_Ls,       0); // Network Command
    case ESPEasy_cmd_e::asyncevent:                 COMMAND_CASE_A(Command_Rules_Async_Events, -1); // Rule.h
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    case ESPEasy_cmd_e::background:                 COMMAND_CASE_R(Command_Background, 1);          // Diagnostic.h
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
#ifdef USES_C012
    case ESPEasy_cmd_e::blynkget:                   COMMAND_CASE_A(Command_Blynk_Get, -1);
#endif // ifdef USES_C012
#ifdef USES_C015
    case ESPEasy_cmd_e::blynkset:                   COMMAND_CASE_R(Command_Blynk_Set, -1);
#endif // ifdef USES_C015
    case ESPEasy_cmd_e::build:                      COMMAND_CASE_A(Command_Settings_Build, 1);      // Settings.h
    case ESPEasy_cmd_e::clearaccessblock:           COMMAND_CASE_R(Command_AccessInfo_Clear,   0);  // Network Command
    case ESPEasy_cmd_e::clearpassword:              COMMAND_CASE_R(Command_Settings_Password_Clear, 1);      // Settings.h
    case ESPEasy_cmd_e::clearrtcram:                COMMAND_CASE_R(Command_RTC_Clear,          0);           // RTC.h
#ifdef ESP8266
    case ESPEasy_cmd_e::clearsdkwifi:               COMMAND_CASE_R(Command_System_Erase_SDK_WiFiconfig,  0); // System.h
    case ESPEasy_cmd_e::clearwifirfcal:             COMMAND_CASE_R(Command_System_Erase_RFcal,  0);          // System.h
#endif // ifdef ESP8266
    case ESPEasy_cmd_e::config:                     COMMAND_CASE_R(Command_Task_RemoteConfig, -1);           // Tasks.h
    case ESPEasy_cmd_e::controllerdisable:          COMMAND_CASE_R(Command_Controller_Disable, 1);           // Controller.h
    case ESPEasy_cmd_e::controllerenable:           COMMAND_CASE_R(Command_Controller_Enable,  1);           // Controller.h
    case ESPEasy_cmd_e::datetime:                   COMMAND_CASE_R(Command_DateTime,             2);         // Time.h
    case ESPEasy_cmd_e::debug:                      COMMAND_CASE_R(Command_Debug,                1);         // Diagnostic.h
    case ESPEasy_cmd_e::dec:                        COMMAND_CASE_A(Command_Rules_Dec,           -1);         // Rules.h
    case ESPEasy_cmd_e::deepsleep:                  COMMAND_CASE_R(Command_System_deepSleep,     1);         // System.h
    case ESPEasy_cmd_e::delay:                      COMMAND_CASE_R(Command_Delay,                1);         // Timers.h
#if FEATURE_PLUGIN_PRIORITY
    case ESPEasy_cmd_e::disableprioritytask:        COMMAND_CASE_R(Command_PriorityTask_Disable, 1);         // Tasks.h
#endif // if FEATURE_PLUGIN_PRIORITY
    case ESPEasy_cmd_e::dns:                        COMMAND_CASE_R(Command_DNS,                  1);         // Network Command
    case ESPEasy_cmd_e::dst:                        COMMAND_CASE_R(Command_DST,                  1);         // Time.h
#if FEATURE_ETHERNET
    case ESPEasy_cmd_e::ethphyadr:                  COMMAND_CASE_R(Command_ETH_Phy_Addr,   1);               // Network Command
    case ESPEasy_cmd_e::ethpinmdc:                  COMMAND_CASE_R(Command_ETH_Pin_mdc,    1);               // Network Command
    case ESPEasy_cmd_e::ethpinmdio:                 COMMAND_CASE_R(Command_ETH_Pin_mdio,   1);               // Network Command
    case ESPEasy_cmd_e::ethpinpower:                COMMAND_CASE_R(Command_ETH_Pin_power,  1);               // Network Command
    case ESPEasy_cmd_e::ethphytype:                 COMMAND_CASE_R(Command_ETH_Phy_Type,   1);               // Network Command
    case ESPEasy_cmd_e::ethclockmode:               COMMAND_CASE_R(Command_ETH_Clock_Mode, 1);               // Network Command
    case ESPEasy_cmd_e::ethip:                      COMMAND_CASE_R(Command_ETH_IP,         1);               // Network Command
    case ESPEasy_cmd_e::ethgateway:                 COMMAND_CASE_R(Command_ETH_Gateway,    1);               // Network Command
    case ESPEasy_cmd_e::ethsubnet:                  COMMAND_CASE_R(Command_ETH_Subnet,     1);               // Network Command
    case ESPEasy_cmd_e::ethdns:                     COMMAND_CASE_R(Command_ETH_DNS,        1);               // Network Command
    case ESPEasy_cmd_e::ethdisconnect:              COMMAND_CASE_A(Command_ETH_Disconnect, 0);               // Network Command
    case ESPEasy_cmd_e::ethwifimode:                COMMAND_CASE_R(Command_ETH_Wifi_Mode,  1);               // Network Command
#endif // FEATURE_ETHERNET
    case ESPEasy_cmd_e::erasesdkwifi:               COMMAND_CASE_R(Command_WiFi_Erase,     0);               // WiFi.h
    case ESPEasy_cmd_e::event:                      COMMAND_CASE_A(Command_Rules_Events,  -1);               // Rule.h
    case ESPEasy_cmd_e::executerules:               COMMAND_CASE_A(Command_Rules_Execute, -1);               // Rule.h
    case ESPEasy_cmd_e::factoryreset:               COMMAND_CASE_R(Command_Settings_FactoryReset, 0);        // Settings.h
    case ESPEasy_cmd_e::gateway:                    COMMAND_CASE_R(Command_Gateway,     1);                  // Network Command
    case ESPEasy_cmd_e::gpio:                       COMMAND_CASE_A(Command_GPIO,        2);                  // Gpio.h
    case ESPEasy_cmd_e::gpiotoggle:                 COMMAND_CASE_A(Command_GPIO_Toggle, 1);                  // Gpio.h
    case ESPEasy_cmd_e::hiddenssid:                 COMMAND_CASE_R(Command_Wifi_HiddenSSID, 1);              // wifi.h
    case ESPEasy_cmd_e::i2cscanner:                 COMMAND_CASE_R(Command_i2c_Scanner, -1);                 // i2c.h
    case ESPEasy_cmd_e::inc:                        COMMAND_CASE_A(Command_Rules_Inc,   -1);                 // Rules.h
    case ESPEasy_cmd_e::ip:                         COMMAND_CASE_R(Command_IP,           1);                 // Network Command
#if FEATURE_USE_IPV6
    case ESPEasy_cmd_e::ip6:                        COMMAND_CASE_A(Command_show_all_IP6,           0);                 // Network Command
#endif
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    case ESPEasy_cmd_e::jsonportstatus:             COMMAND_CASE_A(Command_JSONPortStatus, -1);              // Diagnostic.h
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    #if FEATURE_LAT_LONG_VAR_CMD
    case ESPEasy_cmd_e::latitude:                   COMMAND_CASE_A(Command_Latitude,               -1);      // LatitudeLongitude.h
    #endif // if FEATURE_LAT_LONG_VAR_CMD
    case ESPEasy_cmd_e::let:                        COMMAND_CASE_A(Command_Rules_Let,               2);      // Rules.h
    #if FEATURE_STRING_VARIABLES
    case ESPEasy_cmd_e::letstr:                     COMMAND_CASE_A(Command_Rules_LetStr,            2);      // Rules.h
    #endif // if FEATURE_STRING_VARIABLES
    case ESPEasy_cmd_e::load:                       COMMAND_CASE_A(Command_Settings_Load,           0);      // Settings.h
    case ESPEasy_cmd_e::logentry:                   COMMAND_CASE_A(Command_logentry,               -1);      // Diagnostic.h
    #if FEATURE_LAT_LONG_VAR_CMD
    case ESPEasy_cmd_e::longitude:                  COMMAND_CASE_A(Command_Longitude,              -1);      // LatitudeLongitude.h
    #endif // if FEATURE_LAT_LONG_VAR_CMD
    case ESPEasy_cmd_e::looptimerset:               COMMAND_CASE_A(Command_Loop_Timer_Set,          3);      // Timers.h
    case ESPEasy_cmd_e::looptimerset_ms:            COMMAND_CASE_A(Command_Loop_Timer_Set_ms,       3);      // Timers.h
    case ESPEasy_cmd_e::looptimersetandrun:         COMMAND_CASE_A(Command_Loop_Timer_SetAndRun,    3);      // Timers.h
    case ESPEasy_cmd_e::looptimersetandrun_ms:      COMMAND_CASE_A(Command_Loop_Timer_SetAndRun_ms, 3);      // Timers.h
    case ESPEasy_cmd_e::longpulse:                  COMMAND_CASE_A(Command_GPIO_LongPulse,          5);      // GPIO.h
    case ESPEasy_cmd_e::longpulse_ms:               COMMAND_CASE_A(Command_GPIO_LongPulse_Ms,       5);      // GPIO.h
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    case ESPEasy_cmd_e::logportstatus:              COMMAND_CASE_A(Command_logPortStatus,     0);            // Diagnostic.h
    case ESPEasy_cmd_e::lowmem:                     COMMAND_CASE_A(Command_Lowmem,            0);            // Diagnostic.h
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
#ifdef USES_P009
    case ESPEasy_cmd_e::mcpgpio:                    COMMAND_CASE_A(Command_GPIO,              2);            // Gpio.h
    case ESPEasy_cmd_e::mcpgpiorange:               COMMAND_CASE_A(Command_GPIO_McpGPIORange, -1);           // Gpio.h
    case ESPEasy_cmd_e::mcpgpiopattern:             COMMAND_CASE_A(Command_GPIO_McpGPIOPattern, -1);         // Gpio.h
    case ESPEasy_cmd_e::mcpgpiotoggle:              COMMAND_CASE_A(Command_GPIO_Toggle,       1);            // Gpio.h
    case ESPEasy_cmd_e::mcplongpulse:               COMMAND_CASE_A(Command_GPIO_LongPulse,    3);            // GPIO.h
    case ESPEasy_cmd_e::mcplongpulse_ms:            COMMAND_CASE_A(Command_GPIO_LongPulse_Ms, 3);            // GPIO.h
    case ESPEasy_cmd_e::mcpmode:                    COMMAND_CASE_A(Command_GPIO_Mode,         2);            // Gpio.h
    case ESPEasy_cmd_e::mcpmoderange:               COMMAND_CASE_A(Command_GPIO_ModeRange,    3);            // Gpio.h
    case ESPEasy_cmd_e::mcppulse:                   COMMAND_CASE_A(Command_GPIO_Pulse,        3);            // GPIO.h
#endif // ifdef USES_P009
    case ESPEasy_cmd_e::monitor:                    COMMAND_CASE_A(Command_GPIO_Monitor,      2);            // GPIO.h
    case ESPEasy_cmd_e::monitorrange:               COMMAND_CASE_A(Command_GPIO_MonitorRange, 3);            // GPIO.h
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    case ESPEasy_cmd_e::malloc:                     COMMAND_CASE_A(Command_Malloc,         1);               // Diagnostic.h
    case ESPEasy_cmd_e::meminfo:                    COMMAND_CASE_A(Command_MemInfo,        0);               // Diagnostic.h
    case ESPEasy_cmd_e::meminfodetail:              COMMAND_CASE_A(Command_MemInfo_detail, 0);               // Diagnostic.h
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    case ESPEasy_cmd_e::name:                       COMMAND_CASE_R(Command_Settings_Name,        1);         // Settings.h
    case ESPEasy_cmd_e::nosleep:                    COMMAND_CASE_R(Command_System_NoSleep,       1);         // System.h
#if FEATURE_NOTIFIER
    case ESPEasy_cmd_e::notify:                     COMMAND_CASE_R(Command_Notifications_Notify, -1);        // Notifications.h
#endif // if FEATURE_NOTIFIER
    case ESPEasy_cmd_e::ntphost:                    COMMAND_CASE_R(Command_NTPHost,              1);         // Time.h
#if FEATURE_DALLAS_HELPER && FEATURE_COMMAND_OWSCAN
    case ESPEasy_cmd_e::owscan:                     COMMAND_CASE_R(Command_OneWire_Owscan,       -1);         // OneWire.h
#endif // if FEATURE_DALLAS_HELPER && FEATURE_COMMAND_OWSCAN
#ifdef USES_P019
    case ESPEasy_cmd_e::pcfgpio:                    COMMAND_CASE_A(Command_GPIO,                 2);         // Gpio.h
    case ESPEasy_cmd_e::pcfgpiorange:               COMMAND_CASE_A(Command_GPIO_PcfGPIORange,   -1);         // Gpio.h
    case ESPEasy_cmd_e::pcfgpiopattern:             COMMAND_CASE_A(Command_GPIO_PcfGPIOPattern, -1);         // Gpio.h
    case ESPEasy_cmd_e::pcfgpiotoggle:              COMMAND_CASE_A(Command_GPIO_Toggle,          1);         // Gpio.h
    case ESPEasy_cmd_e::pcflongpulse:               COMMAND_CASE_A(Command_GPIO_LongPulse,       3);         // GPIO.h
    case ESPEasy_cmd_e::pcflongpulse_ms:            COMMAND_CASE_A(Command_GPIO_LongPulse_Ms,    3);         // GPIO.h
    case ESPEasy_cmd_e::pcfmode:                    COMMAND_CASE_A(Command_GPIO_Mode,            2);         // Gpio.h
    case ESPEasy_cmd_e::pcfmoderange:               COMMAND_CASE_A(Command_GPIO_ModeRange,       3);         // Gpio.h   ************
    case ESPEasy_cmd_e::pcfpulse:                   COMMAND_CASE_A(Command_GPIO_Pulse,           3);         // GPIO.h
#endif // ifdef USES_P019
    case ESPEasy_cmd_e::password:                   COMMAND_CASE_R(Command_Settings_Password, 1);            // Settings.h
#if FEATURE_POST_TO_HTTP
    case ESPEasy_cmd_e::posttohttp:                 COMMAND_CASE_A(Command_HTTP_PostToHTTP,  -1);            // HTTP.h
#if FEATURE_HTTP_TLS
    case ESPEasy_cmd_e::posttohttps:                COMMAND_CASE_A(Command_HTTP_PostToHTTPS, -1);            // HTTP.h
#endif // if FEATURE_HTTP_TLS
#endif // if FEATURE_POST_TO_HTTP
#if FEATURE_CUSTOM_PROVISIONING
    case ESPEasy_cmd_e::provision:                  COMMAND_CASE_A(Command_Provisioning_Dispatcher, -1);     // Provisioning.h
# ifdef PLUGIN_BUILD_MAX_ESP32

    // FIXME DEPRECATED: Fallback for temporary backward compatibility
    case ESPEasy_cmd_e::provisionconfig:            COMMAND_CASE_A(Command_Provisioning_ConfigFallback,       0); // Provisioning.h
    case ESPEasy_cmd_e::provisionsecurity:          COMMAND_CASE_A(Command_Provisioning_SecurityFallback,     0); // Provisioning.h
#  if FEATURE_NOTIFIER
    case ESPEasy_cmd_e::provisionnotification:      COMMAND_CASE_A(Command_Provisioning_NotificationFallback, 0); // Provisioning.h
#  endif // if FEATURE_NOTIFIER
    case ESPEasy_cmd_e::provisionprovision:         COMMAND_CASE_A(Command_Provisioning_ProvisionFallback,    0); // Provisioning.h
    case ESPEasy_cmd_e::provisionrules:             COMMAND_CASE_A(Command_Provisioning_RulesFallback,        1); // Provisioning.h
    case ESPEasy_cmd_e::provisionfirmware:          COMMAND_CASE_A(Command_Provisioning_FirmwareFallback,     1); // Provisioning.h
# endif // ifdef PLUGIN_BUILD_MAX_ESP32
#endif // if FEATURE_CUSTOM_PROVISIONING
    case ESPEasy_cmd_e::pulse:                      COMMAND_CASE_A(Command_GPIO_Pulse,        3);                 // GPIO.h
#if FEATURE_MQTT
    case ESPEasy_cmd_e::publish:                    COMMAND_CASE_A(Command_MQTT_Publish,     -1);                 // MQTT.h
    case ESPEasy_cmd_e::publishr:                   COMMAND_CASE_A(Command_MQTT_PublishR,    -1);                 // MQTT.h
#endif // if FEATURE_MQTT
#if FEATURE_PUT_TO_HTTP
    case ESPEasy_cmd_e::puttohttp:                  COMMAND_CASE_A(Command_HTTP_PutToHTTP,  -1);                  // HTTP.h
#if FEATURE_HTTP_TLS
    case ESPEasy_cmd_e::puttohttps:                 COMMAND_CASE_A(Command_HTTP_PutToHTTPS, -1);                  // HTTP.h
#endif // if FEATURE_HTTP_TLS
#endif // if FEATURE_PUT_TO_HTTP
    case ESPEasy_cmd_e::pwm:                        COMMAND_CASE_A(Command_GPIO_PWM,          4);                 // GPIO.h
    case ESPEasy_cmd_e::reboot:                     COMMAND_CASE_A(Command_System_Reboot,              0);        // System.h
    case ESPEasy_cmd_e::resetflashwritecounter:     COMMAND_CASE_A(Command_RTC_resetFlashWriteCounter, 0);        // RTC.h
    case ESPEasy_cmd_e::restart:                    COMMAND_CASE_A(Command_System_Reboot,              0);        // System.h
    case ESPEasy_cmd_e::rtttl:                      COMMAND_CASE_A(Command_GPIO_RTTTL,                -1);        // GPIO.h
    case ESPEasy_cmd_e::rules:                      COMMAND_CASE_A(Command_Rules_UseRules,             1);        // Rule.h
    case ESPEasy_cmd_e::save:                       COMMAND_CASE_R(Command_Settings_Save, 0);                     // Settings.h
    case ESPEasy_cmd_e::scheduletaskrun:            COMMAND_CASE_A(Command_ScheduleTask_Run, 2);                  // Tasks.h

#if FEATURE_SD
    case ESPEasy_cmd_e::sdcard:                     COMMAND_CASE_R(Command_SD_LS,         0);                     // SDCARDS.h
    case ESPEasy_cmd_e::sdremove:                   COMMAND_CASE_R(Command_SD_Remove,     1);                     // SDCARDS.h
#endif // if FEATURE_SD

#if FEATURE_ESPEASY_P2P

    // FIXME TD-er: These send commands, can we determine the nr of arguments?
    case ESPEasy_cmd_e::sendto:                     COMMAND_CASE_A(Command_UPD_SendTo,      2);      // UDP.h
#endif // if FEATURE_ESPEASY_P2P
#if FEATURE_SEND_TO_HTTP
    case ESPEasy_cmd_e::sendtohttp:                 COMMAND_CASE_A(Command_HTTP_SendToHTTP, 3);      // HTTP.h
#if FEATURE_HTTP_TLS
    case ESPEasy_cmd_e::sendtohttps:                COMMAND_CASE_A(Command_HTTP_SendToHTTPS, 3);     // HTTP.h
#endif // if FEATURE_HTTP_TLS
#endif // FEATURE_SEND_TO_HTTP
    case ESPEasy_cmd_e::sendtoudp:                  COMMAND_CASE_A(Command_UDP_SendToUPD,   3);      // UDP.h
    case ESPEasy_cmd_e::sendtoudpmix:               COMMAND_CASE_A(Command_UDP_SendToUPDMix, 3);     // UDP.h
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    case ESPEasy_cmd_e::serialfloat:                COMMAND_CASE_R(Command_SerialFloat,    0);       // Diagnostic.h
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    case ESPEasy_cmd_e::settings:                   COMMAND_CASE_R(Command_Settings_Print, 0);       // Settings.h
#if FEATURE_SERVO
    case ESPEasy_cmd_e::servo:                      COMMAND_CASE_A(Command_Servo,          3);       // Servo.h
#endif // if FEATURE_SERVO

    case ESPEasy_cmd_e::status:                     COMMAND_CASE_A(Command_GPIO_Status,          2); // GPIO.h
    case ESPEasy_cmd_e::subnet:                     COMMAND_CASE_R(Command_Subnet, 1);               // Network Command
#if FEATURE_MQTT
    case ESPEasy_cmd_e::subscribe:                  COMMAND_CASE_A(Command_MQTT_Subscribe, 1);       // MQTT.h
#endif // if FEATURE_MQTT
#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    case ESPEasy_cmd_e::sysload:                    COMMAND_CASE_A(Command_SysLoad,        0);       // Diagnostic.h
#endif // ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
    case ESPEasy_cmd_e::taskclear:                  COMMAND_CASE_R(Command_Task_Clear,    1);        // Tasks.h
    case ESPEasy_cmd_e::taskclearall:               COMMAND_CASE_R(Command_Task_ClearAll, 0);        // Tasks.h
    case ESPEasy_cmd_e::taskdisable:                COMMAND_CASE_R(Command_Task_Disable,  1);        // Tasks.h
    case ESPEasy_cmd_e::taskenable:                 COMMAND_CASE_R(Command_Task_Enable,   1);        // Tasks.h
    case ESPEasy_cmd_e::taskrun:                    COMMAND_CASE_A(Command_Task_Run,            1);  // Tasks.h
    case ESPEasy_cmd_e::taskrunat:                  COMMAND_CASE_A(Command_Task_Run,            2);  // Tasks.h
    case ESPEasy_cmd_e::taskvalueset:               COMMAND_CASE_A(Command_Task_ValueSet,       3);  // Tasks.h
    #if FEATURE_STRING_VARIABLES
    case ESPEasy_cmd_e::taskvaluesetderived:        COMMAND_CASE_A(Command_Task_ValueSetDerived,      5); // Tasks.h
    case ESPEasy_cmd_e::taskvaluesetpresentation:   COMMAND_CASE_A(Command_Task_ValueSetPresentation, 3); // Tasks.h
    #endif // if FEATURE_STRING_VARIABLES
    case ESPEasy_cmd_e::taskvaluetoggle:            COMMAND_CASE_A(Command_Task_ValueToggle,    2);  // Tasks.h
    case ESPEasy_cmd_e::taskvaluesetandrun:         COMMAND_CASE_A(Command_Task_ValueSetAndRun, 3);  // Tasks.h
    case ESPEasy_cmd_e::timerpause:                 COMMAND_CASE_A(Command_Timer_Pause,  1);         // Timers.h
    case ESPEasy_cmd_e::timerresume:                COMMAND_CASE_A(Command_Timer_Resume, 1);         // Timers.h
    case ESPEasy_cmd_e::timerset:                   COMMAND_CASE_A(Command_Timer_Set,    2);         // Timers.h
    case ESPEasy_cmd_e::timerset_ms:                COMMAND_CASE_A(Command_Timer_Set_ms, 2);         // Timers.h
    case ESPEasy_cmd_e::timezone:                   COMMAND_CASE_R(Command_TimeZone, 1);             // Time.h
    case ESPEasy_cmd_e::tone:                       COMMAND_CASE_A(Command_GPIO_Tone, 3);            // GPIO.h
    case ESPEasy_cmd_e::udpport:                    COMMAND_CASE_R(Command_UDP_Port,      1);        // UDP.h
#if FEATURE_ESPEASY_P2P
    case ESPEasy_cmd_e::udptest:                    COMMAND_CASE_R(Command_UDP_Test,      2);        // UDP.h
#endif // if FEATURE_ESPEASY_P2P
    case ESPEasy_cmd_e::unit:                       COMMAND_CASE_R(Command_Settings_Unit, 1);        // Settings.h
    case ESPEasy_cmd_e::unmonitor:                  COMMAND_CASE_A(Command_GPIO_UnMonitor, 2);       // GPIO.h
    case ESPEasy_cmd_e::unmonitorrange:             COMMAND_CASE_A(Command_GPIO_UnMonitorRange, 3);  // GPIO.h
    case ESPEasy_cmd_e::usentp:                     COMMAND_CASE_R(Command_useNTP, 1);               // Time.h
#ifndef LIMIT_BUILD_SIZE
    case ESPEasy_cmd_e::wdconfig:                   COMMAND_CASE_R(Command_WD_Config, 3);            // WD.h
    case ESPEasy_cmd_e::wdread:                     COMMAND_CASE_R(Command_WD_Read,   2);            // WD.h
#endif // ifndef LIMIT_BUILD_SIZE

    case ESPEasy_cmd_e::wifiallowap:                COMMAND_CASE_R(Command_Wifi_AllowAP,    0);      // WiFi.h
    case ESPEasy_cmd_e::wifiapmode:                 COMMAND_CASE_R(Command_Wifi_APMode,     0);      // WiFi.h
    case ESPEasy_cmd_e::wificonnect:                COMMAND_CASE_A(Command_Wifi_Connect,    0);      // WiFi.h
    case ESPEasy_cmd_e::wifidisconnect:             COMMAND_CASE_A(Command_Wifi_Disconnect, 0);      // WiFi.h
    case ESPEasy_cmd_e::wifikey:                    COMMAND_CASE_R(Command_Wifi_Key,        1);      // WiFi.h
    case ESPEasy_cmd_e::wifikey2:                   COMMAND_CASE_R(Command_Wifi_Key2,       1);      // WiFi.h
    case ESPEasy_cmd_e::wifimode:                   COMMAND_CASE_R(Command_Wifi_Mode,       1);      // WiFi.h
    case ESPEasy_cmd_e::wifiscan:                   COMMAND_CASE_R(Command_Wifi_Scan,       0);      // WiFi.h
    case ESPEasy_cmd_e::wifissid:                   COMMAND_CASE_R(Command_Wifi_SSID,       1);      // WiFi.h
    case ESPEasy_cmd_e::wifissid2:                  COMMAND_CASE_R(Command_Wifi_SSID2,      1);      // WiFi.h
    case ESPEasy_cmd_e::wifistamode:                COMMAND_CASE_R(Command_Wifi_STAMode,    0);      // WiFi.h


    case ESPEasy_cmd_e::NotMatched:
      return false;

      // Do not add default: here
      // The compiler will then warn when a command is not included
  }

  #undef COMMAND_CASE_R
  #undef COMMAND_CASE_A
  return _data.retval;
}

#include "../Commands/Tasks.h"


#include "../../ESPEasy_common.h"
#include "../../_Plugin_Helper.h"

#include "../Commands/Common.h"

#include "../DataStructs/TimingStats.h"

#include "../ESPEasyCore/Controller.h"
#include "../ESPEasyCore/Serial.h"

#include "../Globals/ESPEasy_Scheduler.h"
#include "../Globals/RulesCalculate.h"
#include "../Globals/RuntimeData.h"
#include "../Globals/Settings.h"

#include "../Helpers/ESPEasy_UnitOfMeasure.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Numerical.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringParser.h"

// taskIndex = (event->Par1 - 1);   Par1 is here for 1 ... TASKS_MAX
bool validTaskIndexVar(struct EventStruct *event, taskIndex_t& taskIndex)
{
  if (event == nullptr) { return false; }

  if (event->Par1 <= 0) { return false; }
  const taskIndex_t tmp_taskIndex = static_cast<taskIndex_t>(event->Par1 - 1);

  if (!validTaskIndex(tmp_taskIndex)) { return false; }

  taskIndex = tmp_taskIndex;

  return true;
}

// taskIndex = (event->Par1 - 1);   Par1 is here for 1 ... TASKS_MAX
// varNr = event->Par2 - 1;
bool validTaskVars(struct EventStruct *event, taskIndex_t& taskIndex, unsigned int& varNr)
{
  if (event == nullptr) { return false; }

  taskIndex_t tmp_taskIndex;
  if (!validTaskIndexVar(event, tmp_taskIndex)) { return false; }

  varNr = 0;

  if (event->Par2 > 0 && event->Par2 <= VARS_PER_TASK) {
    varNr = event->Par2 - 1;
    taskIndex = tmp_taskIndex;
    return true;
  }

  return false;
}

bool validateAndParseTaskIndexArguments(struct EventStruct * event, const char *Line, taskIndex_t &taskIndex)
{
  if (!validTaskIndexVar(event, taskIndex)) {
    String taskName;
    taskIndex_t tmpTaskIndex = taskIndex;
    if ((event->Par1 <= 0 || event->Par1 >= INVALID_TASK_INDEX) && GetArgv(Line, taskName, 2)) {
      tmpTaskIndex = findTaskIndexByName(taskName, true);
      if (tmpTaskIndex != INVALID_TASK_INDEX) {
        event->Par1 = tmpTaskIndex + 1;
      }
    }
    return validTaskIndexVar(event, taskIndex);
  }
  return true;
}

/**
 * parse TaskName/TaskValue when not numeric for task name and value name and validate values
 */
bool validateAndParseTaskValueArguments(struct EventStruct * event, const char *Line, taskIndex_t &taskIndex, unsigned int &varNr)
{
  if (!validTaskVars(event, taskIndex, varNr) || (event->Par2 <= 0 || event->Par2 >= VARS_PER_TASK))  // Extra check required because of shortcutting in validTaskVars()
  { 
    String taskName;
    taskIndex_t tmpTaskIndex = taskIndex;
    if ((event->Par1 <= 0 || event->Par1 >= INVALID_TASK_INDEX) && GetArgv(Line, taskName, 2)) {
      tmpTaskIndex = findTaskIndexByName(taskName, true);
      if (tmpTaskIndex != INVALID_TASK_INDEX) {
        event->Par1 = tmpTaskIndex + 1;
      }
    }
    String valueName;
    if ((event->Par2 <= 0 || event->Par2 >= VARS_PER_TASK) && event->Par1 - 1 != INVALID_TASK_INDEX && GetArgv(Line, valueName, 3))
    {
      uint8_t tmpVarNr = findDeviceValueIndexByName(valueName, event->Par1 - 1);
      if (tmpVarNr != VARS_PER_TASK) {
        event->Par2 = tmpVarNr + 1;
      }
    }
    if (!validTaskVars(event, taskIndex, varNr)) return false; 
  }

  return true;
}

const __FlashStringHelper * taskValueSet(struct EventStruct *event, const char *Line, taskIndex_t& taskIndex, bool& success)
{
  String TmpStr1;
  unsigned int varNr;

  if (!validateAndParseTaskValueArguments(event, Line, taskIndex, varNr)) {
    success = false;
    return F("INVALID_PARAMETERS");
  }
  if (!Settings.AllowTaskValueSetAllPlugins() && 
      getPluginID_from_TaskIndex(taskIndex).value != 33) { // PluginID 33 = Dummy Device
    success = false;
    return F("NOT_A_DUMMY_TASK");
  }
  if (!Settings.TaskDeviceEnabled[taskIndex]) {
    success = false;
    return F("TASK_NOT_ENABLED");
  }
  const uint8_t valueCount = getValueCountForTask(taskIndex);
  if (valueCount <= varNr) {
    success = false;
    return F("INVALID_VAR_INDEX");
  }

  EventStruct tmpEvent(taskIndex);
  if (GetArgv(Line, TmpStr1, 4)) {
    const Sensor_VType sensorType = tmpEvent.getSensorType();

    // FIXME TD-er: Must check if the value has to be computed and not convert to double when sensor type is 64 bit int.

    // Perform calculation with float result.
    ESPEASY_RULES_FLOAT_TYPE result{};

    if (isError(Calculate(TmpStr1, result))) {
      success = false;
      return F("CALCULATION_ERROR");
    }
    #ifndef BUILD_NO_DEBUG
    addLog(LOG_LEVEL_INFO, strformat(
      F("taskValueSet: %s  taskindex: %d varNr: %d result: %f type: %d"),
      Line,
      taskIndex,
      varNr, result, sensorType));
    #endif

    UserVar.set(taskIndex, varNr, result, sensorType);
  } else  {
    // TODO: Get Task description and var name
    serialPrintln(formatUserVarNoCheck(&tmpEvent, varNr));
  }
  success = true;
  return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Task_Clear(struct EventStruct *event, const char *Line)
{
  taskIndex_t  taskIndex;

  if (!validateAndParseTaskIndexArguments(event, Line, taskIndex)) {
    return F("INVALID_PARAMETERS"); 
  }

  taskClear(taskIndex, true);
  return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Task_ClearAll(struct EventStruct *event, const char *Line)
{
  for (taskIndex_t t = 0; t < TASKS_MAX; t++) {
    taskClear(t, false);
  }
  return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Task_EnableDisable(struct EventStruct *event, bool enable, const char *Line)
{
  taskIndex_t  taskIndex;
  if (validateAndParseTaskIndexArguments(event, Line, taskIndex)) {
    // This is a command so no guarantee the taskIndex is correct in the event
    event->setTaskIndex(taskIndex);

    #if FEATURE_PLUGIN_PRIORITY
    if (Settings.isPriorityTask(event->TaskIndex)) {
      return return_command_failed_flashstr();
    }
    #endif // if FEATURE_PLUGIN_PRIORITY
    return return_command_boolean_result_flashstr(setTaskEnableStatus(event, enable));
  }
  return F("INVALID_PARAMETERS");
}

#if FEATURE_PLUGIN_PRIORITY
const __FlashStringHelper * Command_PriorityTask_DisableTask(struct EventStruct *event, const char *Line)
{
  taskIndex_t  taskIndex;

  if (validateAndParseTaskIndexArguments(event, Line, taskIndex)) {
    // This is a command so no guarantee the taskIndex is correct in the event
    event->setTaskIndex(taskIndex);

    if (Settings.isPowerManagerTask(event->TaskIndex))
    {
      Settings.setPowerManagerTask(event->TaskIndex, false);
      return return_command_success_flashstr();
    }
    // Handle other Priotiry task options
    Settings.setTaskEnableReadonly(event->TaskIndex, false);
    return return_command_failed_flashstr();
  }
  return F("INVALID_PARAMETERS");
}
#endif // if FEATURE_PLUGIN_PRIORITY

const __FlashStringHelper * Command_Task_Disable(struct EventStruct *event, const char *Line)
{
  return Command_Task_EnableDisable(event, false, Line);
}

const __FlashStringHelper * Command_Task_Enable(struct EventStruct *event, const char *Line)
{
  return Command_Task_EnableDisable(event, true, Line);
}

#if FEATURE_PLUGIN_PRIORITY
const __FlashStringHelper * Command_PriorityTask_Disable(struct EventStruct *event, const char *Line)
{
  return Command_PriorityTask_DisableTask(event, Line);
}
#endif

const __FlashStringHelper * Command_Task_ValueSet(struct EventStruct *event, const char *Line)
{
  taskIndex_t taskIndex;
  bool success;
  return taskValueSet(event, Line, taskIndex, success);
}

#if FEATURE_STRING_VARIABLES
const __FlashStringHelper * Command_Task_ValueSetDerived(struct EventStruct *event, const char *Line)
{
  return taskValueSetString(event, Line, F(TASK_VALUE_DERIVED_PREFIX_TEMPLATE), F(TASK_VALUE_UOM_PREFIX_TEMPLATE), F(TASK_VALUE_VTYPE_PREFIX_TEMPLATE));
}

const __FlashStringHelper * Command_Task_ValueSetPresentation(struct EventStruct *event, const char *Line)
{
  return taskValueSetString(event, Line, F(TASK_VALUE_PRESENTATION_PREFIX_TEMPLATE));
}

const __FlashStringHelper * taskValueSetString(struct EventStruct *event, const char *Line, const __FlashStringHelper * storageTemplate, const __FlashStringHelper * uomTemplate, const __FlashStringHelper * vTypeTemplate)
{
  String taskName;
  taskIndex_t tmpTaskIndex = INVALID_TASK_INDEX;
  if ((event->Par1 <= 0 || event->Par1 >= INVALID_TASK_INDEX) && GetArgv(Line, taskName, 2)) {
    tmpTaskIndex = findTaskIndexByName(taskName, true);
    if (tmpTaskIndex != INVALID_TASK_INDEX) {
      event->Par1 = tmpTaskIndex + 1;
    }
  }
  String valueName;
  const bool hasValueName = GetArgv(Line, valueName, 3);
  if ((event->Par2 <= 0 || event->Par2 >= VARS_PER_TASK) && event->Par1 - 1 != INVALID_TASK_INDEX && hasValueName)
  {
    uint8_t tmpVarNr = findDeviceValueIndexByName(valueName, event->Par1 - 1);
    if (tmpVarNr != VARS_PER_TASK) {
      event->Par2 = tmpVarNr + 1;
    }
  }

  if (event->Par1 > 0 && validTaskIndex(event->Par1 - 1)) {
    taskName = getTaskDeviceName(event->Par1 - 1); // Taskname must be valid
  }

  if (!hasValueName || (event->Par1 > 0 && validTaskIndex(event->Par1 - 1) && event->Par2 > 0 && validTaskVarIndex(event->Par2 - 1))) {
    valueName = getTaskValueName(static_cast<taskIndex_t>(event->Par1 - 1), event->Par2 - 1); // Convert numeric var index into name
  }
  // addLog(LOG_LEVEL_INFO, strformat(F("TaskValueSetStorage: task: %s (%d), value: %s (%d), Line: %s"), taskName.c_str(), event->Par1, valueName.c_str(), event->Par2, Line)); // FIXME

  String argument;
  if (!taskName.isEmpty() && !valueName.isEmpty() && GetArgv(Line, argument, 4)) {
    taskName.toLowerCase();
    String orgValueName(valueName);
    valueName.toLowerCase();
    String key = strformat(storageTemplate, taskName.c_str(), valueName.c_str());
    const String key2 = strformat(F(TASK_VALUE_NAME_PREFIX_TEMPLATE), taskName.c_str(), valueName.c_str());
    setCustomStringVar(key, argument);
    if (getCustomStringVar(key2).isEmpty() || argument.isEmpty()) {
      setCustomStringVar(key2, argument.isEmpty() ? EMPTY_STRING : orgValueName);
    }
    if (uomTemplate != nullptr) { // We have a Unit of Measure template
      if (GetArgv(Line, argument, 5)) { // check for extra argument holding UoM
        key = strformat(uomTemplate, taskName.c_str(), valueName.c_str());
        if (!argument.isEmpty()) {
          #if FEATURE_TASKVALUE_UNIT_OF_MEASURE

          uint8_t i = 1; // Index 0 is empty/None
          String uom = toUnitOfMeasureName(i);
          while (i < 255 && !uom.isEmpty()) {
            if (argument.equalsIgnoreCase(uom)) {
              setCustomStringVar(key, uom);
              argument.clear();
              break;
            }
            ++i;
            uom = toUnitOfMeasureName(i);
          }
          if (!argument.isEmpty()) 
          #endif // if FEATURE_TASKVALUE_UNIT_OF_MEASURE
          {
            setCustomStringVar(key, argument);
          }
        } else {
          clearCustomStringVar(key);
        }
      }
    }
    if (vTypeTemplate != nullptr) { // We have a ValueType template
      if (GetArgv(Line, argument, 6)) { // check for extra argument holding VType
        key = strformat(vTypeTemplate, taskName.c_str(), valueName.c_str());
        if (!argument.isEmpty()) {
          constexpr uint8_t maxVType = static_cast<uint8_t>(Sensor_VType::SENSOR_TYPE_NOT_SET);

          for (uint8_t i = 0; i < maxVType; ++i) {
            const Sensor_VType vt = static_cast<Sensor_VType>(i);

            if ((getValueCountFromSensorType(vt, false) == 1) &&
                argument.equalsIgnoreCase(getSensorTypeLabel(vt))) {
              setCustomStringVar(key, getSensorTypeLabel(vt)); // Set 'official' VType label
              break;
            }
          }
        } else {
          clearCustomStringVar(key);
        }
      }
    }
    // addLog(LOG_LEVEL_INFO, strformat(F("TaskValueSetStorage: key: %s, argument: %s"), key.c_str(), argument.c_str())); // FIXME
    return return_command_success_flashstr();
  }
  return return_command_failed_flashstr(); // taskValueSet(event, Line, taskIndex, success);
}
#endif

const __FlashStringHelper * Command_Task_ValueToggle(struct EventStruct *event, const char *Line)
{
  taskIndex_t  taskIndex;
  unsigned int varNr;

  if (!validateAndParseTaskValueArguments(event, Line, taskIndex, varNr)) {
    return F("INVALID_PARAMETERS");
  }
  if (!Settings.TaskDeviceEnabled[taskIndex]) {
    return F("TASK_NOT_ENABLED");
  }

  EventStruct tmpEvent(taskIndex);
  const Sensor_VType sensorType = tmpEvent.getSensorType();
  const int    result       = lround(UserVar.getAsDouble(taskIndex, varNr, sensorType));
  if ((result == 0) || (result == 1)) {
    UserVar.set(taskIndex, varNr, (result == 0) ? 1.0 : 0.0, sensorType);
  }
  return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Task_ValueSetAndRun(struct EventStruct *event, const char *Line)
{
  taskIndex_t taskIndex;
  bool success;
  const __FlashStringHelper * returnvalue = taskValueSet(event, Line, taskIndex, success);
  if (success)
  {
    struct EventStruct TempEvent(taskIndex);
    TempEvent.Source = event->Source;
    SensorSendTask(&TempEvent);

    return return_command_success_flashstr();
  }
  return returnvalue;
}

const __FlashStringHelper * Command_ScheduleTask_Run(struct EventStruct *event, const char* Line)
{
  taskIndex_t  taskIndex;

  if (!validateAndParseTaskIndexArguments(event, Line, taskIndex) || event->Par2 < 0) {
    return F("INVALID_PARAMETERS");
  }
  if (!Settings.TaskDeviceEnabled[taskIndex]) {
    return F("TASK_NOT_ENABLED");
  }

  uint32_t msecFromNow = 0;
  String par3;
  if (GetArgv(Line, par3, 3)) {
    if (validUIntFromString(par3, msecFromNow)) {
      Scheduler.schedule_task_device_timer(taskIndex, millis() + msecFromNow);
      return return_command_success_flashstr();
    }
  }
  return F("INVALID_PARAMETERS");  
}

const __FlashStringHelper * Command_Task_Run(struct EventStruct *event, const char *Line)
{
  taskIndex_t  taskIndex;

  if (!validateAndParseTaskIndexArguments(event, Line, taskIndex) || event->Par2 < 0) {
    return F("INVALID_PARAMETERS");
  }
  if (!Settings.TaskDeviceEnabled[taskIndex]) {
    return F("TASK_NOT_ENABLED");
  }
  uint32_t unixTime = 0;
  String par3;
  if (GetArgv(Line, par3, 3)) {
    validUIntFromString(par3, unixTime);
  }

  struct EventStruct TempEvent(taskIndex);
  TempEvent.Source = event->Source;
  SensorSendTask(&TempEvent, unixTime);

  return return_command_success_flashstr();
}

const __FlashStringHelper * Command_Task_RemoteConfig(struct EventStruct *event, const char *Line)
{
  struct EventStruct TempEvent(event->TaskIndex);
  String request = Line;

  // FIXME TD-er: Should we call ExecuteCommand here? The command is not parsed like any other call.
  remoteConfig(&TempEvent, request);
  return return_command_success_flashstr();
}

#include "../Commands/SDCARD.h"

#include "../../ESPEasy_common.h"
#include "../Commands/Common.h"


#if FEATURE_SD

#include "../ESPEasyCore/Serial.h"
#include "../Globals/Settings.h"
#include "../Helpers/StringConverter.h"

#include <SD.h>


void printDirectory(fs::File dir, int numTabs)
{
  while (true) {
    fs::File entry = dir.openNextFile();

    if (!entry) {
      // no more files
      break;
    }

    for (uint8_t i = 0; i < numTabs; i++) {
      serialPrint("\t");
    }
    serialPrint(entry.name());

    if (entry.isDirectory()) {
      serialPrintln("/");
      printDirectory(entry, numTabs + 1);
    } else {
      // files have sizes, directories do not
      serialPrint("\t\t");
      serialPrintln(String(entry.size(), DEC));
    }
    entry.close();
  }
}


const __FlashStringHelper * Command_SD_LS(struct EventStruct *event, const char* Line)
{
  fs::File root = SD.open("/");
  root.rewindDirectory();
  printDirectory(root, 0);
  root.close();
  return return_see_serial(event);
}

String Command_SD_Remove(struct EventStruct *event, const char* Line)
{
  // FIXME TD-er: This one is not using parseString* function
  String fname = Line;
  fname = fname.substring(9);
  SD.remove((char*)fname.c_str());
  return return_result(event, concat(F("Removing:"), fname));
}
#endif

#include "../Commands/RTC.h"

#include "../../ESPEasy_common.h"


#include "../Commands/Common.h"

#include "../DataStructs/RTCStruct.h"

#include "../Globals/RTC.h"

#include "../Helpers/ESPEasyRTC.h"


const __FlashStringHelper * Command_RTC_Clear(struct EventStruct *event, const char* Line)
{
	initRTC();
	return return_command_success_flashstr();
}

const __FlashStringHelper * Command_RTC_resetFlashWriteCounter(struct EventStruct *event, const char* Line)
{
	RTC.flashDayCounter = 0;
	return return_command_success_flashstr();
}

#include "../Commands/Time.h"


#include "../../ESPEasy_common.h"


#include "../Commands/Common.h"

#include "../DataStructs/ESPEasy_EventStruct.h"

#include "../ESPEasyCore/Serial.h"

#include "../Globals/ESPEasy_time.h"
#include "../Globals/Settings.h"

#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/StringConverter.h"


String Command_NTPHost(struct EventStruct *event, const char *Line)
{
  return Command_GetORSetString(event, F("NTPHost:"),
                                Line,
                                Settings.NTPHost,
                                sizeof(Settings.NTPHost),
                                1);
}

String Command_useNTP(struct EventStruct *event, const char *Line)
{
  if (HasArgv(Line, 2)) {
    Settings.UseNTP(event->Par1);
  } else {
    return return_result(event, concat(F("UseNTP:"), boolToString(Settings.UseNTP())));
  }
  return return_command_success();
}

String Command_TimeZone(struct EventStruct *event, const char *Line)
{
  if (HasArgv(Line, 2)) {
    Settings.TimeZone = event->Par1;
  } else {
    return return_result(event, concat(F("TimeZone:"), static_cast<int>(Settings.TimeZone)));
  }
  return return_command_success();
}

String Command_DST(struct EventStruct *event, const char *Line)
{
  if (HasArgv(Line, 2)) {
    Settings.DST = event->Par1;
  } else  {
    return return_result(event, concat(F("DST:"),  boolToString(Settings.DST)));
  }
  return return_command_success();
}

String Command_DateTime(struct EventStruct *event, const char *Line)
{
  String TmpStr1;

  if (GetArgv(Line, TmpStr1, 2)) {
    struct tm newtime;
    int yr = 1970;
    int mnth = 1;
    int d = 1;
    sscanf(TmpStr1.c_str(), "%4d-%2d-%2d", &yr, &mnth, &d);
    newtime.tm_year = yr - 1900;
    newtime.tm_mon  = mnth - 1; // tm_mon starts at 0
    newtime.tm_mday = d;

    int h = 0;
    int m = 0;
    int s = 0;

    if (GetArgv(Line, TmpStr1, 3)) {
      sscanf(TmpStr1.c_str(), "%2d:%2d:%2d", &h, &m, &s);
    }
    newtime.tm_hour = h;
    newtime.tm_min  = m;
    newtime.tm_sec  = s;

    // Please note the time set in this command is in UTC time, not local time.
    node_time.setExternalTimeSource(makeTime(newtime), timeSource_t::Manual_set);
  } else  {
    // serialPrintln();
    return return_result(event, concat(F("Datetime:"), node_time.getDateTimeString('-', ':', ' ')));
  }
  return return_command_success();
}
#include "../Commands/Controller.h"


#include "../../ESPEasy_common.h"


#include "../Commands/Common.h"

#include "../DataStructs/ESPEasy_EventStruct.h"

#include "../DataTypes/ControllerIndex.h"

#include "../Globals/CPlugins.h"

#include "../Helpers/Misc.h"

//      controllerIndex = (event->Par1 - 1);   Par1 is here for 1 ... CONTROLLER_MAX
bool validControllerVar(struct EventStruct *event, controllerIndex_t& controllerIndex)
{
  if (event->Par1 <= 0) { return false; }
  controllerIndex = static_cast<controllerIndex_t>(event->Par1 - 1);
  return validControllerIndex(controllerIndex);
}

const __FlashStringHelper * Command_Controller_Disable(struct EventStruct *event, const char *Line)
{
  controllerIndex_t controllerIndex;
  return return_command_boolean_result_flashstr(validControllerVar(event, controllerIndex) && setControllerEnableStatus(controllerIndex, false));
}

const __FlashStringHelper * Command_Controller_Enable(struct EventStruct *event, const char *Line)
{
  controllerIndex_t controllerIndex;
  return return_command_boolean_result_flashstr(validControllerVar(event, controllerIndex) && setControllerEnableStatus(controllerIndex, true));
}

#include "../Commands/Notifications.h"

#if FEATURE_NOTIFIER

#include "../Commands/Common.h"

#include "../../ESPEasy_common.h"
#include "../DataTypes/ESPEasy_plugin_functions.h"
#include "../Globals/ESPEasy_Scheduler.h"
#include "../Globals/Settings.h"
#include "../Globals/NPlugins.h"
#include "../Helpers/StringConverter.h"


const __FlashStringHelper * Command_Notifications_Notify(struct EventStruct *event, const char* Line)
{
	String message;
	String subject;
	GetArgv(Line, message, 3);
	GetArgv(Line, subject, 4);

	if (event->Par1 > 0) {
		int index = event->Par1 - 1;
		if (Settings.NotificationEnabled[index] && Settings.Notification[index] != INVALID_N_PLUGIN_ID.value) {
			nprotocolIndex_t NotificationProtocolIndex = 
			  getNProtocolIndex(npluginID_t::toPluginID(Settings.Notification[index]));
			if (validNProtocolIndex(NotificationProtocolIndex )) {
				struct EventStruct TempEvent(event->TaskIndex);
				// TempEvent.NotificationProtocolIndex = NotificationProtocolIndex;
				TempEvent.NotificationIndex = index;
				TempEvent.String1 = message;
				TempEvent.String2 = subject;
				Scheduler.schedule_notification_event_timer(NotificationProtocolIndex, NPlugin::Function::NPLUGIN_NOTIFY, std::move(TempEvent));
			}
		}
	}
	return return_command_success_flashstr();
}

#endif
#include "../Commands/Diagnostic.h"


#include "../Commands/Common.h"

#include "../DataStructs/PortStatusStruct.h"

#include "../DataTypes/SettingsType.h"

#include "../ESPEasyCore/ESPEasy_backgroundtasks.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/Serial.h"

#include "../Globals/Device.h"
#include "../Globals/ExtraTaskSettings.h"
#include "../Globals/GlobalMapPortStatus.h"
#include "../Globals/SecuritySettings.h"
#include "../Globals/Settings.h"
#include "../Globals/Statistics.h"

#include "../Helpers/Convert.h"
#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/Misc.h"
#include "../Helpers/_Plugin_init.h"
#include "../Helpers/PortStatus.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringParser.h"

#include <map>
#include <stdint.h>


#ifndef BUILD_MINIMAL_OTA
bool showSettingsFileLayout = false;
#endif // ifndef BUILD_MINIMAL_OTA

#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
String Command_Lowmem(struct EventStruct *event, const char *Line)
{
  String result;

  result += lowestRAM;
  result += F(" : ");
  result += lowestRAMfunction;
  return return_result(event, result);
}

const __FlashStringHelper * Command_Malloc(struct EventStruct *event, const char *Line)
{
  char *ramtest;
  int size = parseCommandArgumentInt(Line, 1);

  ramtest = (char *)malloc(size);

  if (ramtest == nullptr) { return return_command_failed_flashstr(); }
  free(ramtest);
  return return_command_success_flashstr();
}

String Command_SysLoad(struct EventStruct *event, const char *Line)
{
  String result = toString(getCPUload(), 2);

  result += F("% (LC=");
  result += getLoopCountPerSec();
  result += ')';
  return return_result(event, result);
}

const __FlashStringHelper * Command_SerialFloat(struct EventStruct *event, const char *Line)
{
  pinMode(1, INPUT);
  pinMode(3, INPUT);
  delay(60000);
  return return_command_success_flashstr();
}

const __FlashStringHelper * Command_MemInfo(struct EventStruct *event, const char *Line)
{
  serialPrint(F("SecurityStruct         | "));
  serialPrintln(String(sizeof(SecuritySettings)));
  serialPrint(F("SettingsStruct         | "));
  serialPrintln(String(sizeof(Settings)));
  serialPrint(F("ExtraTaskSettingsStruct| "));
  serialPrintln(String(sizeof(ExtraTaskSettings)));
  serialPrint(F("DeviceStruct           | "));
  serialPrintln(String(getNrBuiltInDeviceIndex()));
  return return_see_serial(event);
}

const __FlashStringHelper * Command_MemInfo_detail(struct EventStruct *event, const char *Line)
{
#ifndef BUILD_MINIMAL_OTA
  showSettingsFileLayout = true;
  Command_MemInfo(event, Line);

  for (int st = 0; st < static_cast<int>(SettingsType::Enum::SettingsType_MAX); ++st) {
    SettingsType::SettingsType::Enum settingsType = static_cast<SettingsType::Enum>(st);
    int max_index, offset, max_size;
    int struct_size = 0;
    serialPrintln();
    if (SettingsType::getSettingsParameters(settingsType, 0, max_index, offset, max_size, struct_size))
    {
      serialPrint(SettingsType::getSettingsTypeString(settingsType));
      serialPrintln(F(" | start | end | max_size | struct_size"));
      serialPrintln(F("--- | --- | --- | --- | ---"));

      for (int i = 0; i < max_index; ++i) {
        if (SettingsType::getSettingsParameters(settingsType, i, offset, max_size))
        {
          serialPrintln(strformat(
            F("%d|%d|%d|%d|%d"),
            i,
            offset,
            offset + max_size - 1,
            max_size,
            struct_size
          ));
        }
      }
    }
  }
  return return_see_serial(event);
  #else
  return return_command_failed_flashstr();
  #endif // ifndef BUILD_MINIMAL_OTA
}

const __FlashStringHelper * Command_Background(struct EventStruct *event, const char *Line)
{
  unsigned long timer = millis() + parseCommandArgumentInt(Line, 1);

  serialPrintln(F("start"));

  while (!timeOutReached(timer)) {
    backgroundtasks();
  }
  serialPrintln(F("end"));
  return return_see_serial(event);
}
#endif // BUILD_NO_DIAGNOSTIC_COMMANDS

const __FlashStringHelper * Command_Debug(struct EventStruct *event, const char *Line)
{
  if (HasArgv(Line, 2)) {
    setLogLevelFor(LOG_TO_SERIAL, parseCommandArgumentInt(Line, 1));
  } else  {
    serialPrintln();
    serialPrint(F("Serial debug level: "));
    serialPrintln(String(Settings.SerialLogLevel));
  }
  return return_see_serial(event);
}

String Command_logentry(struct EventStruct *event, const char *Line)
{
  uint8_t level = LOG_LEVEL_INFO;
  // An extra optional parameter to set log level.
  if (event->Par2 > LOG_LEVEL_NONE && event->Par2 <= 
  # ifndef BUILD_NO_DEBUG
    LOG_LEVEL_DEBUG_MORE
  #else
    LOG_LEVEL_INFO
  #endif
    ) { level = event->Par2; }
  String res = tolerantParseStringKeepCase(Line, 2);
  addLog(level, res);
  return res;
}

#ifndef BUILD_NO_DIAGNOSTIC_COMMANDS
const __FlashStringHelper * Command_JSONPortStatus(struct EventStruct *event, const char *Line)
{
  addLog(LOG_LEVEL_INFO, F("JSON Port Status: Command not implemented yet."));
  return return_command_success_flashstr();
}

void createLogPortStatus(std::map<uint32_t, portStatusStruct>::iterator it)
{  
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLogMove(LOG_LEVEL_INFO, strformat(
      F("PortStatus detail: Port=%u State=%d Output=%d Mode=%u Task=%u Monitor=%u Command=%d Init=%d PreviousTask=%d"),
      getPortFromKey(it->first),
      it->second.getValue(),
      it->second.output,
      it->second.mode,
      it->second.task,
      it->second.monitor,
      it->second.command,
      it->second.init,
      it->second.previousTask));
  }
}

void debugPortStatus(std::map<uint32_t, portStatusStruct>::iterator it)
{
  createLogPortStatus(it);
}


const __FlashStringHelper * Command_logPortStatus(struct EventStruct *event, const char *Line)
{
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLogMove(LOG_LEVEL_INFO, concat(F("PortStatus structure: Called from=Rules Count="), static_cast<int>(globalMapPortStatus.size())));
  }

  for (auto it = globalMapPortStatus.begin(); it != globalMapPortStatus.end(); ++it) {
    debugPortStatus(it);
  }

  return return_command_success_flashstr();
}
#endif // BUILD_NO_DIAGNOSTIC_COMMANDS

#include "../Commands/Blynk.h"

#ifdef USES_C012

#include "../Commands/Common.h"
#include "../DataStructs/ESPEasy_EventStruct.h"
#include "../ESPEasyCore/ESPEasy_backgroundtasks.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../Globals/Settings.h"
#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/_CPlugin_Helper.h"

#include "../../ESPEasy_fdwdecl.h"


controllerIndex_t firstEnabledBlynk_ControllerIndex() {
  for (controllerIndex_t i = 0; i < CONTROLLER_MAX; ++i) {
    protocolIndex_t ProtocolIndex = getProtocolIndex_from_ControllerIndex(i);

    if (validProtocolIndex(ProtocolIndex)) {
      const cpluginID_t number = getCPluginID_from_ProtocolIndex(ProtocolIndex);

      if ((number == 12) && Settings.ControllerEnabled[i]) {
        return i;
      }
    }
  }
  return INVALID_CONTROLLER_INDEX;
}

const __FlashStringHelper * Command_Blynk_Get(struct EventStruct *event, const char *Line)
{
  controllerIndex_t first_enabled_blynk_controller = firstEnabledBlynk_ControllerIndex();

  if (!validControllerIndex(first_enabled_blynk_controller)) {
    return F("Controller not enabled");
  } else {
    // FIXME TD-er: This one is not using parseString* function
    String strLine = Line;
    strLine = strLine.substring(9);
    int index = strLine.indexOf(',');

    if (index > 0)
    {
      int index           = strLine.lastIndexOf(',');
      String blynkcommand = strLine.substring(index + 1);
      float  value        = 0;

      if (Blynk_get(blynkcommand, first_enabled_blynk_controller, &value))
      {
        UserVar.setFloat((event->Par1 - 1), event->Par2 - 1, value);
      }
      else {
        return F("Error getting data");
      }
    }
    else
    {
      if (!Blynk_get(strLine, first_enabled_blynk_controller, nullptr))
      {
        return F("Error getting data");
      }
    }
  }
  return return_command_success_flashstr();
}

bool Blynk_get(const String& command, controllerIndex_t controllerIndex, float *data)
{
  bool MustCheckReply = false;
  String hostname, pass;
  unsigned int ClientTimeout = 0;
  WiFiClient client;

  {
    // Place ControllerSettings in its own scope, as it is quite big.
    MakeControllerSettings(ControllerSettings); //-V522
    if (!AllocatedControllerSettings()) {
      addLog(LOG_LEVEL_ERROR, F("Blynk : Cannot run GET, out of RAM"));
      return false;
    }

    LoadControllerSettings(controllerIndex, *ControllerSettings);
    MustCheckReply = ControllerSettings->MustCheckReply;
    hostname = ControllerSettings->getHost();
    pass = getControllerPass(controllerIndex, *ControllerSettings);
    ClientTimeout = ControllerSettings->ClientTimeout;

    if (pass.isEmpty()) {
      addLog(LOG_LEVEL_ERROR, F("Blynk : No password set"));
      return false;
    }

    if (!try_connect_host(/* CPLUGIN_ID_012 */ 12, client, *ControllerSettings)) {
      return false;
    }
  }

  // We now create a URI for the request
  {
    // Place this stack allocated array in its own scope, as it is quite big.
    char request[300] = { 0 };
    sprintf_P(request,
              PSTR("GET /%s/%s HTTP/1.1\r\n Host: %s \r\n Connection: close\r\n\r\n"),
              pass.c_str(),
              command.c_str(),
              hostname.c_str());
#ifndef BUILD_NO_DEBUG
    addLog(LOG_LEVEL_DEBUG, request);
#endif
    client.print(request);
  }
  bool success = !MustCheckReply;

  if (MustCheckReply || data) {
    unsigned long timer = millis() + ClientTimeout;

    while (!client_available(client) && !timeOutReached(timer)) {
      delay(1);
    }

    #ifndef BUILD_NO_DEBUG
    char log[80] = { 0 };
    #endif
    timer = millis() + 1500;

    // Read all the lines of the reply from server and log them
    while (client_available(client) && !success && !timeOutReached(timer)) {
      String line;
      safeReadStringUntil(client, line, '\n');
      #ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_DEBUG_MORE, line);
      #endif

      // success ?
      if (equals(line.substring(0, 15), F("HTTP/1.1 200 OK"))) {
        #ifndef BUILD_NO_DEBUG
        strcpy_P(log, PSTR("HTTP : Success"));
        #endif

        if (!data) { success = true; }
      }
      #ifndef BUILD_NO_DEBUG
      else if (equals(line.substring(0, 24), F("HTTP/1.1 400 Bad Request"))) {
        strcpy_P(log, PSTR("HTTP : Unauthorized"));
      }
      else if (equals(line.substring(0, 25), F("HTTP/1.1 401 Unauthorized"))) {
        strcpy_P(log, PSTR("HTTP : Unauthorized"));
      }
      addLog(LOG_LEVEL_DEBUG, log);
      #endif

      // data only
      if (data && line.startsWith("["))
      {
        String strValue = line;
        uint8_t   pos      = strValue.indexOf('"', 2);
        strValue = strValue.substring(2, pos);
        strValue.trim();
        *data   = 0.0f;
        validFloatFromString(strValue, *data);
        success = true;

        char value_char[5] = { 0 };
        strValue.toCharArray(value_char, 5);
        #ifndef BUILD_NO_DEBUG
        sprintf_P(log, PSTR("Blynk get - %s => %s"), command.c_str(), value_char);
        addLog(LOG_LEVEL_DEBUG, log);
        #endif
      }
      delay(0);
    }
  }
  #ifndef BUILD_NO_DEBUG
  addLog(LOG_LEVEL_DEBUG, F("HTTP : closing connection (012)"));
  #endif

  client.PR_9453_FLUSH_TO_CLEAR();
  client.stop();

  // important - backgroundtasks - free mem
  unsigned long timer = millis() + 10;

  while (!timeOutReached(timer)) {
    backgroundtasks();
  }

  return success;
}

#endif // ifdef USES_C012

#include "../Commands/HTTP.h"

#include "../../ESPEasy_common.h"

#include "../Commands/Common.h"

#include "../DataStructs/ControllerSettingsStruct.h"
#include "../DataStructs/SettingsStruct.h"

#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"

#include "../Globals/Settings.h"

#include "../Helpers/_CPlugin_Helper.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Networking.h"
#include "../Helpers/StringParser.h"

#if FEATURE_SEND_TO_HTTP || FEATURE_POST_TO_HTTP || FEATURE_PUT_TO_HTTP
const __FlashStringHelper* httpEmitToHTTP(struct EventStruct        *event,
                                          const __FlashStringHelper *logIdentifier,
                                          const __FlashStringHelper *HttpMethod,
                                          const char                *Line,
                                          const int                  timeout,
                                          const bool                 waitForAck,
                                          const bool                 useHeader,
                                          const bool                 useBody,
                                          const bool                 useHttps)
{
  if (NetworkConnected()) {
    String   user, pass, host, file, path, header, postBody;
    uint16_t port;
    uint8_t  idx;

    const String arg1 = parseStringKeepCase(Line, 2);

    if (arg1.indexOf(F("://")) != -1) {
      // Full url given
      path = splitURL(arg1, user, pass, host, port, file);
      idx  = 3;

      if (useHeader || useBody) {
        path = parseStringKeepCase(path, 1); // Only first part is path when having header or body
      }
    } else {
      // Command arguments are split into: host, port, url
      if (!splitUserPass_HostPortString(
            arg1,
            user,
            pass,
            host,
            port))
      {
        return return_command_failed_flashstr();
      }

      const int port_arg = event->Par2;

      if ((port_arg > 0) && (port_arg < 65536)) {
        port = port_arg;
      } else {
        if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
          addLogMove(LOG_LEVEL_ERROR, strformat(F("%s: Invalid port argument: %d will use: %d"),
                                                FsP(logIdentifier), port_arg, port));
        }
      }

      if (useHeader || useBody) {
        path = parseStringKeepCase(Line, 4);
      } else {
        path = parseStringToEndKeepCase(Line, 4);
      }
      idx = 5;
    }

    if (useHeader) {
      header = parseStringKeepCase(Line, idx);
      idx++;
    }

    if (useBody) {
      postBody = parseStringKeepCase(Line, idx);
    }
    # ifndef BUILD_NO_DEBUG

    if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
      addLogMove(LOG_LEVEL_DEBUG, strformat(F("%s: Host: %s port: %d path: %s"),
                                            FsP(logIdentifier), host.c_str(), port, path.c_str()));
    }
    # endif // ifndef BUILD_NO_DEBUG

    #if FEATURE_HTTP_TLS
    TLS_types tlsType = TLS_types::NoTLS;
    if (useHttps) {
      tlsType = TLS_types::TLS_insecure;
      // TODO: ?Parse Line for type of TLS to use?
    }
    #endif // if FEATURE_HTTP_TLS

    int httpCode = -1;
    send_via_http(
      logIdentifier,
      timeout,
      user,
      pass,
      host,
      port,
      path,
      HttpMethod,
      header,
      postBody,
      httpCode,
      waitForAck
      #if FEATURE_HTTP_TLS
      , tlsType
      #endif // if FEATURE_HTTP_TLS
     );

    if ((httpCode >= 100) && (httpCode < 300)) {
      return return_command_success_flashstr();
    }
  } else {
    addLog(LOG_LEVEL_ERROR, concat(logIdentifier, F(": Not connected to network")));
  }
  return return_command_failed_flashstr();
}

#endif // if FEATURE_SEND_TO_HTTP || FEATURE_POST_TO_HTTP || FEATURE_PUT_TO_HTTP

#if FEATURE_SEND_TO_HTTP

// syntax 1: SendToHttp,<[<user>:<password>@]<host>,<port>,<path>
// syntax 2: SendToHttp,http://<[<user>:<password>@]<host>[:<port>]/<path>
const __FlashStringHelper* Command_HTTP_SendToHTTP(struct EventStruct *event, const char *Line)
{
  // Some servers don't give an ack.
  // For these it is adviced to uncheck to wait for an acknowledgement.
  // However the default timeout of 4000 msec is then way too long
  // FIXME TD-er: Make sendToHttp timeout a setting.
  const int timeout = Settings.SendToHttp_ack()
         ? CONTROLLER_CLIENTTIMEOUT_MAX : 1000;

  return httpEmitToHTTP(event, F("SendToHTTP"), F("GET"), Line, timeout, Settings.SendToHttp_ack(), false, false, false);
}

#if FEATURE_HTTP_TLS

// syntax 1: SendToHttpS,<[<user>:<password>@]<host>,<port>,<path>
// syntax 2: SendToHttpS,http://<[<user>:<password>@]<host>[:<port>]/<path>
const __FlashStringHelper* Command_HTTP_SendToHTTPS(struct EventStruct *event, const char *Line)
{
  // Some servers don't give an ack.
  // For these it is adviced to uncheck to wait for an acknowledgement.
  // However the default timeout of 4000 msec is then way too long
  // FIXME TD-er: Make sendToHttp timeout a setting.
  const int timeout = Settings.SendToHttp_ack()
         ? CONTROLLER_CLIENTTIMEOUT_MAX : 1000;

  return httpEmitToHTTP(event, F("SendToHTTPS"), F("GET"), Line, timeout, Settings.SendToHttp_ack(), false, false, true);
}
#endif // if FEATURE_HTTP_TLS

#endif // FEATURE_SEND_TO_HTTP

#if FEATURE_POST_TO_HTTP

// syntax 1: PostToHttp,<[<user>:<password>@]<host>,<port>,<path>,<header>,<body>
// syntax 2: PostToHttp,http://<[<user>:<password>@]<host>[:<port>]/<path>,<header>,<body>
const __FlashStringHelper* Command_HTTP_PostToHTTP(struct EventStruct *event, const char *Line)
{
  // FIXME tonhuisman: Make postToHttp timeout a setting, now using a somewhat sensible default
  const int timeout = CONTROLLER_CLIENTTIMEOUT_MAX;

  // FIXME tonhuisman: make PostToHttp_ack a setting, using SendToHttp_ack for now...

  return httpEmitToHTTP(event, F("PostToHTTP"), F("POST"), Line, timeout, Settings.SendToHttp_ack(), true, true, false);
}

#if FEATURE_HTTP_TLS

// syntax 1: PostToHttpS,<[<user>:<password>@]<host>,<port>,<path>,<header>,<body>
// syntax 2: PostToHttpS,http://<[<user>:<password>@]<host>[:<port>]/<path>,<header>,<body>
const __FlashStringHelper* Command_HTTP_PostToHTTPS(struct EventStruct *event, const char *Line)
{
  // FIXME tonhuisman: Make postToHttp timeout a setting, now using a somewhat sensible default
  const int timeout = CONTROLLER_CLIENTTIMEOUT_MAX;

  // FIXME tonhuisman: make PostToHttp_ack a setting, using SendToHttp_ack for now...

  return httpEmitToHTTP(event, F("PostToHTTPS"), F("POST"), Line, timeout, Settings.SendToHttp_ack(), true, true, true);
}
#endif // if FEATURE_HTTP_TLS

#endif // if FEATURE_POST_TO_HTTP

#if FEATURE_PUT_TO_HTTP

// syntax 1: PutToHttp,<[<user>:<password>@]<host>,<port>,<path>,<header>,<body>
// syntax 2: PutToHttp,http://<[<user>:<password>@]<host>[:<port>]/<path>,<header>,<body>
const __FlashStringHelper* Command_HTTP_PutToHTTP(struct EventStruct *event, const char *Line)
{
  // FIXME tonhuisman: Make putToHttp timeout a setting, now using a somewhat sensible default
  const int timeout = CONTROLLER_CLIENTTIMEOUT_MAX;

  // FIXME tonhuisman: make PutToHttp_ack a setting, using SendToHttp_ack for now...

  return httpEmitToHTTP(event, F("PutToHTTP"), F("PUT"), Line, timeout, Settings.SendToHttp_ack(), true, true, false);
}

#if FEATURE_HTTP_TLS

// syntax 1: PutToHttpS,<[<user>:<password>@]<host>,<port>,<path>,<header>,<body>
// syntax 2: PutToHttpS,http://<[<user>:<password>@]<host>[:<port>]/<path>,<header>,<body>
const __FlashStringHelper* Command_HTTP_PutToHTTPS(struct EventStruct *event, const char *Line)
{
  // FIXME tonhuisman: Make putToHttp timeout a setting, now using a somewhat sensible default
  const int timeout = CONTROLLER_CLIENTTIMEOUT_MAX;

  // FIXME tonhuisman: make PutToHttp_ack a setting, using SendToHttp_ack for now...

  return httpEmitToHTTP(event, F("PutToHTTPS"), F("PUT"), Line, timeout, Settings.SendToHttp_ack(), true, true, true);
}
#endif // if FEATURE_HTTP_TLS

#endif // if FEATURE_PUT_TO_HTTP

#include "../Commands/WiFi.h"

#include "../../ESPEasy_common.h"

#include "../Commands/Common.h"

#include "../DataStructs/ESPEasy_EventStruct.h"

#include "../ESPEasyCore/ESPEasyWifi.h"
#include "../ESPEasyCore/Serial.h"

#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/RTC.h"
#include "../Globals/SecuritySettings.h"
#include "../Globals/Settings.h"

#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/StringConverter.h"


#define WIFI_MODE_MAX (WiFiMode_t)4



String Command_Wifi_SSID(struct EventStruct *event, const char *Line)
{
  return Command_GetORSetString(event, F("Wifi SSID:"),
                                Line,
                                SecuritySettings.WifiSSID,
                                sizeof(SecuritySettings.WifiSSID),
                                1);
}

String Command_Wifi_Key(struct EventStruct *event, const char *Line)
{
  return Command_GetORSetString(event, F("Wifi Key:"),
                                Line,
                                SecuritySettings.WifiKey,
                                sizeof(SecuritySettings.WifiKey),
                                1);
}

String Command_Wifi_SSID2(struct EventStruct *event, const char *Line)
{
  return Command_GetORSetString(event, F("Wifi2 SSID:"),
                                Line,
                                SecuritySettings.WifiSSID2,
                                sizeof(SecuritySettings.WifiSSID2),
                                1);
}

String Command_Wifi_Key2(struct EventStruct *event, const char *Line)
{
  return Command_GetORSetString(event, F("Wifi2 Key:"),
                                Line,
                                SecuritySettings.WifiKey2,
                                sizeof(SecuritySettings.WifiKey2),
                                1);
}

String Command_Wifi_HiddenSSID(struct EventStruct *event, const char *Line)
{
  bool   includeHiddenSSID = Settings.IncludeHiddenSSID();
  String result            = Command_GetORSetBool(event, F("Include Hidden SSID:"),
                                                  Line,
                                                  (bool *)&includeHiddenSSID,
                                                  1);

  if (Settings.IncludeHiddenSSID() != includeHiddenSSID) { // Update if changed
    Settings.IncludeHiddenSSID(includeHiddenSSID);
  }
  return result;
}

const __FlashStringHelper* Command_Wifi_Scan(struct EventStruct *event, const char *Line)
{
  WiFiScan_log_to_serial();
  return return_command_success_flashstr();
}

const __FlashStringHelper* Command_Wifi_Connect(struct EventStruct *event, const char *Line)
{
  WiFiEventData.wifiConnectAttemptNeeded = true;
  return return_command_success_flashstr();
}

const __FlashStringHelper* Command_Wifi_Disconnect(struct EventStruct *event, const char *Line)
{
  RTC.clearLastWiFi(); // Force a WiFi scan
  WifiDisconnect();

  return return_command_success_flashstr();
}

const __FlashStringHelper* Command_Wifi_APMode(struct EventStruct *event, const char *Line)
{
  setAP(true);
  return return_command_success_flashstr();
}

const __FlashStringHelper* Command_Wifi_STAMode(struct EventStruct *event, const char *Line)
{
  setSTA(true);
  return return_command_success_flashstr();
}

String Command_Wifi_Mode(struct EventStruct *event, const char *Line)
{
  String TmpStr1;

  if (GetArgv(Line, TmpStr1, 2)) {
    WiFiMode_t mode = WIFI_MODE_MAX;

    if ((event->Par1 > 0) && (event->Par1 < WIFI_MODE_MAX)) {
      mode = static_cast<WiFiMode_t>(event->Par1 - 1);
    } else {
      TmpStr1.toLowerCase();

      if (strcmp_P(TmpStr1.c_str(), PSTR("off")) == 0) { mode = WIFI_OFF; }
      else if (strcmp_P(TmpStr1.c_str(), PSTR("sta")) == 0) { mode = WIFI_STA; }
      else if (strcmp_P(TmpStr1.c_str(), PSTR("ap")) == 0) { mode = WIFI_AP; }
      else if (strcmp_P(TmpStr1.c_str(), PSTR("ap+sta")) == 0) { mode = WIFI_AP_STA; }
    }

    if ((mode >= WIFI_OFF) && (mode < WIFI_MODE_MAX)) {
      setWifiMode(mode);
    } else {
      return return_result(event, F("Wifi Mode: invalid arguments"));
    }
  } else {
    return return_result(event, concat(F("WiFi Mode:"),  getWifiModeString(WiFi.getMode())));
  }
  return return_command_success();
}

const __FlashStringHelper* Command_Wifi_AllowAP(struct EventStruct *event, const char *Line)
{
  Settings.DoNotStartAP(false);
  return return_command_success_flashstr();
}

// FIXME: TD-er This is not an erase, but actually storing the current settings
// in the wifi settings of the core library
const __FlashStringHelper* Command_WiFi_Erase(struct EventStruct *event, const char *Line)
{
  return Erase_WiFi_Calibration() 
    ? return_command_success_flashstr()
    : return_command_failed_flashstr();
}

#include "../Commands/i2c.h"

#include "../Commands/Common.h"
#include "../ESPEasyCore/Serial.h"

#include "../Globals/I2Cdev.h"
#include "../Globals/Settings.h"

#include "../Helpers/Hardware_I2C.h"
#include "../Helpers/StringConverter.h"

#include "../../ESPEasy_common.h"

void i2c_scanI2Cbus(bool dbg, int8_t channel, uint8_t i2cBus) {
  uint8_t error, address;

  #if FEATURE_I2CMULTIPLEXER

  if (-1 == channel) {
    serialPrintln(concat(F("Standard I2C bus "), i2cBus));
  } else {
    serialPrintln(concat(F("Multiplexer channel "), channel));
  }
  #endif // if FEATURE_I2CMULTIPLEXER

  for (address = 1; address <= 127; address++) {
    Wire.beginTransmission(address);
    error = Wire.endTransmission();

    if (error == 0) {
      serialPrintln(strformat(F("I2C  : Found 0x%02x"), address));
    } else if ((error == 4) || dbg) {
      serialPrintln(strformat(F("I2C  : Error %d at 0x%02x"), error, address));
    }
  }
}

const __FlashStringHelper* Command_i2c_Scanner(struct EventStruct *event, const char *Line)
{
  const bool dbg = equals(parseString(Line, 2), F("1"));

  #if !FEATURE_I2C_MULTIPLE
  const uint8_t i2cBus = 0;
  #else // if !FEATURE_I2C_MULTIPLE

  for (uint8_t i2cBus = 0; i2cBus < getI2CBusCount(); ++i2cBus)
  #endif // if !FEATURE_I2C_MULTIPLE
  {
    if (Settings.isI2CEnabled(i2cBus)) {
      I2CSelect_Max100kHz_ClockSpeed(i2cBus); // Scan bus using low speed

      i2c_scanI2Cbus(dbg, -1, i2cBus);        // Base I2C bus

      #if FEATURE_I2CMULTIPLEXER

      if (isI2CMultiplexerEnabled(i2cBus)) {
        uint8_t mux_max = I2CMultiplexerMaxChannels(i2cBus);

        for (int8_t channel = 0; channel < mux_max; ++channel) {
          I2CMultiplexerSelect(i2cBus, channel);
          i2c_scanI2Cbus(dbg, channel, i2cBus); // Multiplexer I2C bus
        }
        I2CMultiplexerOff(0);
      }
      #endif // if FEATURE_I2CMULTIPLEXER
    } else {
      serialPrintln(strformat(F("I2C %d: Not enabled."), i2cBus));
    }
  }
  I2CSelectHighClockSpeed(0); // By default the bus is in standard speed
  return return_see_serial(event);
}

#include "../ControllerQueue/C016_queue_element.h"

#ifdef USES_C016

# include "../DataStructs/ESPEasy_EventStruct.h"
# include "../Globals/Plugins.h"
# include "../Globals/RuntimeData.h"
# include "../Helpers/_Plugin_SensorTypeHelper.h"
# include "../Helpers/ESPEasy_math.h"

C016_queue_element::C016_queue_element() :  sensorType(
    Sensor_VType::SENSOR_TYPE_NONE) {
  _timestamp      = 0;
  _controller_idx = 0;
  _taskIndex      = INVALID_TASK_INDEX;
  values.clear();
}

C016_queue_element::C016_queue_element(C016_queue_element&& other)
  : sensorType(other.sensorType)
  , valueCount(other.valueCount)
{
  _timestamp      = other._timestamp;
  _controller_idx = other._controller_idx;
  _taskIndex      = other._taskIndex;
  values          = other.values;
}

C016_queue_element::C016_queue_element(const struct EventStruct *event, uint8_t value_count) :
  unixTime(event->timestamp_sec),
  sensorType(event->sensorType),
  valueCount(value_count)
{
  _controller_idx = event->ControllerIndex;
  _taskIndex      = event->TaskIndex;
  values.clear();
  const TaskValues_Data_t* data = UserVar.getRawTaskValues_Data(event->TaskIndex);

  if (data != nullptr) {
    for (uint8_t i = 0; i < value_count; ++i) {
      values.copyValue(*data, i, sensorType);
    }
  }
}

C016_queue_element& C016_queue_element::operator=(C016_queue_element&& other) {
  _timestamp      = other._timestamp;
  _taskIndex      = other._taskIndex;
  _controller_idx = other._controller_idx;
  sensorType      = other.sensorType;
  valueCount      = other.valueCount;
  unixTime        = other.unixTime;
  values          = other.values;

  return *this;
}

size_t C016_queue_element::getSize() const {
  return sizeof(*this);
}

bool C016_queue_element::isDuplicate(const Queue_element_base& other) const {
  const C016_queue_element& oth = static_cast<const C016_queue_element&>(other);

  if ((oth._controller_idx != _controller_idx) ||
      (oth._taskIndex != _taskIndex) ||
      (oth.sensorType != sensorType) ||
      (oth.valueCount != valueCount)) {
    return false;
  }

  for (uint8_t i = 0; i < valueCount; ++i) {
    if (isFloatOutputDataType(sensorType)) {
      if (!essentiallyEqual(oth.values.getFloat(i), values.getFloat(i))) {
        return false;
      }
    } else {
      if (oth.values.getUint32(i) != values.getUint32(i)) {
        return false;
      }
    }
  }
  return true;
}

C016_binary_element C016_queue_element::getBinary() const {
  C016_binary_element element;

  element.unixTime   = unixTime;
  element.TaskIndex  = _taskIndex;
  element.sensorType = sensorType;
  element.valueCount = valueCount;
  element.values     = values;

  // It makes no sense to keep the controller index when storing it.
  // re-purpose it to store the pluginID
  element.pluginID = getPluginID_from_TaskIndex(_taskIndex);

  return element;
}

#endif // ifdef USES_C016

#include "../ControllerQueue/SimpleQueueElement_formatted_Strings.h"

#include "../DataStructs/ESPEasy_EventStruct.h"
#include "../Helpers/StringConverter.h"

#include "../../_Plugin_Helper.h"


SimpleQueueElement_formatted_Strings::SimpleQueueElement_formatted_Strings(struct EventStruct *event) :
  idx(event->idx),
  sensorType(event->sensorType),
  valuesSent(0)
{
  _controller_idx = event->ControllerIndex;
  _taskIndex = event->TaskIndex;

  valueCount = getValueCountForTask(_taskIndex);

  for (uint8_t i = 0; i < valueCount; ++i) {
    move_special(txt[i], formatUserVarNoCheck(event, i));
  }
}

SimpleQueueElement_formatted_Strings::SimpleQueueElement_formatted_Strings(const struct EventStruct *event, uint8_t value_count) :
  idx(event->idx),
  sensorType(event->sensorType),
  valuesSent(0),
  valueCount(value_count) {
  _controller_idx = event->ControllerIndex;
  _taskIndex      = event->TaskIndex;
}

SimpleQueueElement_formatted_Strings::SimpleQueueElement_formatted_Strings(SimpleQueueElement_formatted_Strings&& rval)
  : idx(rval.idx),
  sensorType(rval.sensorType),
  valuesSent(rval.valuesSent), valueCount(rval.valueCount)
{
  _timestamp      = rval._timestamp;
  _controller_idx = rval._controller_idx;
  _taskIndex      = rval._taskIndex;

  for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
    move_special(txt[i], std::move(rval.txt[i]));
  }
}

SimpleQueueElement_formatted_Strings& SimpleQueueElement_formatted_Strings::operator=(SimpleQueueElement_formatted_Strings&& rval) {
  idx             = rval.idx;
  _timestamp      = rval._timestamp;
  _taskIndex      = rval._taskIndex;
  _controller_idx = rval._controller_idx;
  sensorType      = rval.sensorType;
  valuesSent      = rval.valuesSent;
  valueCount      = rval.valueCount;

  for (size_t i = 0; i < VARS_PER_TASK; ++i) {
    move_special(txt[i], std::move(rval.txt[i]));
  }
  return *this;
}

bool SimpleQueueElement_formatted_Strings::checkDone(bool succesfull) const {
  if (succesfull) { ++valuesSent; }
  return valuesSent >= valueCount || valuesSent >= VARS_PER_TASK;
}

size_t SimpleQueueElement_formatted_Strings::getSize() const {
  size_t total = sizeof(*this);

  for (int i = 0; i < VARS_PER_TASK; ++i) {
    total += txt[i].length();
  }
  return total;
}

bool SimpleQueueElement_formatted_Strings::isDuplicate(const Queue_element_base& rval) const {
  const SimpleQueueElement_formatted_Strings& oth = static_cast<const SimpleQueueElement_formatted_Strings&>(rval);

  if ((oth._controller_idx != _controller_idx) ||
      (oth._taskIndex != _taskIndex) ||
      (oth.sensorType != sensorType) ||
      (oth.valueCount != valueCount) ||
      (oth.idx != idx)) {
    return false;
  }

  for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
    if (oth.txt[i] != txt[i]) {
      return false;
    }
  }
  return true;
}

#include "../ControllerQueue/ControllerDelayHandlerStruct.h"


ControllerDelayHandlerStruct::ControllerDelayHandlerStruct() :
  lastSend(0),
  minTimeBetweenMessages(CONTROLLER_DELAY_QUEUE_DELAY_DFLT),
  expire_timeout(0),
  max_queue_depth(CONTROLLER_DELAY_QUEUE_DEPTH_DFLT),
  attempt(0),
  max_retries(CONTROLLER_DELAY_QUEUE_RETRY_DFLT),
  delete_oldest(false),
  must_check_reply(false),
  deduplicate(false),
  useLocalSystemTime(false) {}

bool ControllerDelayHandlerStruct::cacheControllerSettings(controllerIndex_t ControllerIndex)
{
  MakeControllerSettings(ControllerSettings);

  if (!AllocatedControllerSettings()) {
    return false;
  }
  LoadControllerSettings(ControllerIndex, *ControllerSettings);
  cacheControllerSettings(*ControllerSettings);
  return true;
}

void ControllerDelayHandlerStruct::cacheControllerSettings(const ControllerSettingsStruct& settings) {
  minTimeBetweenMessages = settings.MinimalTimeBetweenMessages;
  max_queue_depth        = settings.MaxQueueDepth;
  max_retries            = settings.MaxRetry;
  delete_oldest          = settings.DeleteOldest;
  must_check_reply       = settings.MustCheckReply;
  deduplicate            = settings.deduplicate();
  useLocalSystemTime     = settings.useLocalSystemTime();

  if (settings.allowExpire()) {
    expire_timeout = max_queue_depth * max_retries * (minTimeBetweenMessages + settings.ClientTimeout);

    if (expire_timeout < CONTROLLER_QUEUE_MINIMAL_EXPIRE_TIME) {
      expire_timeout = CONTROLLER_QUEUE_MINIMAL_EXPIRE_TIME;
    }
  } else {
    expire_timeout = 0;
  }

  // Set some sound limits when not configured
  if (max_queue_depth == 0) { max_queue_depth = CONTROLLER_DELAY_QUEUE_DEPTH_DFLT; }

  if (max_retries == 0) { max_retries = CONTROLLER_DELAY_QUEUE_RETRY_DFLT; }

  if (minTimeBetweenMessages == 0) { minTimeBetweenMessages = CONTROLLER_DELAY_QUEUE_DELAY_DFLT; }

  // No less than 10 msec between messages.
  if (minTimeBetweenMessages < 10) { minTimeBetweenMessages = 10; }
}

bool ControllerDelayHandlerStruct::readyToProcess(const Queue_element_base& element) const {
  const protocolIndex_t protocolIndex = getProtocolIndex_from_ControllerIndex(element._controller_idx);

  if (protocolIndex == INVALID_PROTOCOL_INDEX) {
    return false;
  }

  if (getProtocolStruct(protocolIndex).needsNetwork) {
    return NetworkConnected(10);
  }
  return true;
}

bool ControllerDelayHandlerStruct::queueFull(controllerIndex_t controller_idx) const {
  if (sendQueue.size() >= max_queue_depth) { return true; }

  // Number of elements is not exceeding the limit, check memory
  int freeHeap = FreeMem();
  {
    /*
      #ifdef USE_SECOND_HEAP
    const int freeHeap2 = FreeMem2ndHeap();

    if (freeHeap2 < freeHeap) {
      freeHeap = freeHeap2;
    }
      #endif // ifdef USE_SECOND_HEAP
      */
  }

#ifdef ESP32
  if (freeHeap > 50000) 
#else
  if (freeHeap > 5000) 
#endif
  {
    return false; // Memory is not an issue.
  }
#ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    String log = F("Controller-");
    log += controller_idx + 1;
    log += F(" : Memory used: ");
    log += getQueueMemorySize();
    log += F(" bytes ");
    log += sendQueue.size();
    log += F(" items ");
    log += freeHeap;
    log += F(" free");
    addLogMove(LOG_LEVEL_DEBUG, log);
  }
#endif // ifndef BUILD_NO_DEBUG
  return true;
}

// Return true if message is already present in the queue
bool ControllerDelayHandlerStruct::isDuplicate(const Queue_element_base& element) const {
  // Some controllers may receive duplicate messages, due to lost acknowledgement
  // This is actually the same message, so this should not be processed.
  if (!unitLastMessageCount.isNew(element.getUnitMessageCount())) {
    return true;
  }

  // The unit message count is still stored to make sure a new one with the same count
  // is considered a duplicate, even when the queue is empty.
  unitLastMessageCount.add(element.getUnitMessageCount());

  // the setting 'deduplicate' does look at the content of the message and only compares it to messages in the queue.
  if (deduplicate && !sendQueue.empty()) {
    // Use reverse iterator here, as it is more likely a duplicate is added shortly after another.
    auto it = sendQueue.rbegin(); // Same as back()

    for (; it != sendQueue.rend(); ++it) {
      if (element.isDuplicate(*(it->get()))) {
#ifndef BUILD_NO_DEBUG

        if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
          const cpluginID_t cpluginID = getCPluginID_from_ControllerIndex(it->get()->_controller_idx);
          addLogMove(LOG_LEVEL_DEBUG, concat(get_formatted_Controller_number(cpluginID), F(" : Remove duplicate")));
        }
#endif // ifndef BUILD_NO_DEBUG
        return true;
      }
    }
  }
  return false;
}

// Try to add to the queue, if permitted by "delete_oldest"
// Return true when item was added, or skipped as it was considered a duplicate
bool ControllerDelayHandlerStruct::addToQueue(std::unique_ptr<Queue_element_base>element) {
  if (!element) { 
    return false;
  }
  if (isDuplicate(*element)) {
    return true;
  }

  if (delete_oldest) {
    // Force add to the queue.
    // If max buffer is reached, the oldest in the queue (first to be served) will be removed.
    while (queueFull(element->_controller_idx)) {
      sendQueue.pop_front();
      attempt = 0;
    }
  }

  if (!queueFull(element->_controller_idx)) {
    #ifdef USE_SECOND_HEAP
    // Do not store in 2nd heap, std::list cannot handle 2nd heap well
    HeapSelectDram ephemeral;
    #endif // ifdef USE_SECOND_HEAP

    sendQueue.push_back(std::move(element));

    return true;
  }
#ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    const cpluginID_t cpluginID = getCPluginID_from_ControllerIndex((*element)._controller_idx);
    addLogMove(LOG_LEVEL_DEBUG, concat(get_formatted_Controller_number(cpluginID), F(" : queue full")));
  }
#endif // ifndef BUILD_NO_DEBUG
  return false;
}

// Get the next element.
// Remove front element when max_retries is reached.
Queue_element_base * ControllerDelayHandlerStruct::getNext() {
  if (sendQueue.empty()) { return nullptr; }

  if (attempt > max_retries) {
    sendQueue.pop_front();
    attempt = 0;
  }

  if (expire_timeout != 0) {
    bool done = false;

    while (!done && !sendQueue.empty()) {
      if ((sendQueue.front().get() != nullptr) && (timePassedSince(sendQueue.front()->_timestamp) < static_cast<long>(expire_timeout))) {
        done = true;
      } else {
        sendQueue.pop_front();
        attempt = 0;
      }
    }
  }

  if (sendQueue.empty()) { return nullptr; }
  return sendQueue.front().get();
}

// Mark as processed and return time to schedule for next process.
// Return 0 when nothing to process.
// @param remove_from_queue indicates whether the elements should be removed from the queue.
unsigned long ControllerDelayHandlerStruct::markProcessed(bool remove_from_queue) {
  if (sendQueue.empty()) { return 0; }

  if (remove_from_queue) {
    sendQueue.pop_front();
    attempt  = 0;
    lastSend = millis();
  } else {
    ++attempt;
  }
  return getNextScheduleTime();
}

unsigned long ControllerDelayHandlerStruct::getNextScheduleTime() const {
  if (sendQueue.empty()) { return 0; }
  unsigned long nextTime = lastSend + minTimeBetweenMessages;

  if (timePassedSince(nextTime) > 0) {
    nextTime = millis();
  }

  if (nextTime == 0) { nextTime = 1; // Just to make sure it will be executed
  }
  return nextTime;
}

// Set the "lastSend" to "now" + some additional delay.
// This will cause the next schedule time to be delayed to
// msecFromNow + minTimeBetweenMessages
void ControllerDelayHandlerStruct::setAdditionalDelay(unsigned long msecFromNow) {
  lastSend = millis() + msecFromNow;
}

size_t ControllerDelayHandlerStruct::getQueueMemorySize() const {
  size_t totalSize = 0;

  for (auto it = sendQueue.begin(); it != sendQueue.end(); ++it) {
    if (it->get() != nullptr) {
      totalSize += it->get()->getSize();
    }
  }
  return totalSize;
}

void ControllerDelayHandlerStruct::process(
  cpluginID_t                        cpluginID,
  do_process_function                func,
  TimingStatsElements                timerstats_id,
  SchedulerIntervalTimer_e timerID) 
{
  Queue_element_base *element(static_cast<Queue_element_base *>(getNext()));

  if (element == nullptr) { return; }

  if (readyToProcess(*element)) {
    MakeControllerSettings(ControllerSettings);

    if (AllocatedControllerSettings()) {
      LoadControllerSettings(element->_controller_idx, *ControllerSettings);
      cacheControllerSettings(*ControllerSettings);
      START_TIMER;
      markProcessed(func(cpluginID, *element, *ControllerSettings));
      #if FEATURE_TIMING_STATS
      STOP_TIMER_VAR(timerstats_id);
      #endif
    }
  }
  Scheduler.scheduleNextDelayQueue(timerID, getNextScheduleTime());
}

#include "../ControllerQueue/DelayQueueElements.h"

#include "../DataStructs/ControllerSettingsStruct.h"
#include "../DataStructs/TimingStats.h"
#include "../Globals/ESPEasy_Scheduler.h"
#include "../Helpers/PeriodicalActions.h"

#if FEATURE_MQTT
ControllerDelayHandlerStruct *MQTTDelayHandler = nullptr;

bool init_mqtt_delay_queue(controllerIndex_t ControllerIndex, String& pubname, bool& retainFlag) {
  // Make sure the controller is re-connecting with the current settings.
  MQTTDisconnect();

  MakeControllerSettings(ControllerSettings); // -V522

  if (!AllocatedControllerSettings()) {
    return false;
  }
  LoadControllerSettings(ControllerIndex, *ControllerSettings);

  if (MQTTDelayHandler == nullptr) {
    # ifdef USE_SECOND_HEAP
    HeapSelectDram ephemeral;
    # endif // ifdef USE_SECOND_HEAP

    MQTTDelayHandler = new (std::nothrow) ControllerDelayHandlerStruct;
  }

  if (MQTTDelayHandler == nullptr) {
    return false;
  }
  MQTTDelayHandler->cacheControllerSettings(*ControllerSettings);
  pubname    = ControllerSettings->Publish;
  retainFlag = ControllerSettings->mqtt_retainFlag();
  Scheduler.setIntervalTimerOverride(SchedulerIntervalTimer_e::TIMER_MQTT, 10); // Make sure the MQTT is being processed as soon
                                                                                          // as possible.
  scheduleNextMQTTdelayQueue();
  return true;
}

void exit_mqtt_delay_queue() {
  if (MQTTDelayHandler != nullptr) {
    MQTTDisconnect();
    delete MQTTDelayHandler;
    MQTTDelayHandler = nullptr;
  }
}

#endif // if FEATURE_MQTT


/*********************************************************************************************\
* C001_queue_element for queueing requests for C001.
\*********************************************************************************************/
#ifdef USES_C001
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(00, 1) // -V522
#endif // ifdef USES_C001

/*********************************************************************************************\
* C003_queue_element for queueing requests for C003 Nodo Telnet.
\*********************************************************************************************/
#ifdef USES_C003
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(00, 3) // -V522
#endif // ifdef USES_C003

#ifdef USES_C004
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(00, 4) // -V522
#endif // ifdef USES_C004

#ifdef USES_C007
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(00, 7) // -V522
#endif // ifdef USES_C007


/*********************************************************************************************\
* C008_queue_element for queueing requests for 008: Generic HTTP
* Using SimpleQueueElement_formatted_Strings
\*********************************************************************************************/
#ifdef USES_C008
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(00, 8) // -V522
#endif // ifdef USES_C008

#ifdef USES_C009
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(00, 9) // -V522
#endif // ifdef USES_C009


/*********************************************************************************************\
* C010_queue_element for queueing requests for 010: Generic UDP
* Using SimpleQueueElement_formatted_Strings
\*********************************************************************************************/
#ifdef USES_C010
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 10) // -V522
#endif // ifdef USES_C010


/*********************************************************************************************\
* C011_queue_element for queueing requests for 011: Generic HTTP Advanced
\*********************************************************************************************/
#ifdef USES_C011
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 11) // -V522
#endif // ifdef USES_C011


/*********************************************************************************************\
* C012_queue_element for queueing requests for 012: Blynk
* Using SimpleQueueElement_formatted_Strings
\*********************************************************************************************/
#ifdef USES_C012
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 12) // -V522
#endif // ifdef USES_C012

/*
 #ifdef USES_C013
   DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 13)  // -V522
 #endif
 */

/*
 #ifdef USES_C014
   DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 14)  // -V522
 #endif
 */


#ifdef USES_C015
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 15) // -V522
#endif // ifdef USES_C015


#ifdef USES_C016
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 16) // -V522
#endif // ifdef USES_C016


#ifdef USES_C017
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 17) // -V522
#endif // ifdef USES_C017

#ifdef USES_C018
DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 18) // -V522
#endif // ifdef USES_C018


/*
 #ifdef USES_C019
   DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 19)  // -V522
 #endif
 */

/*
 #ifdef USES_C020
   DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 20)  // -V522
 #endif
 */

/*
 #ifdef USES_C021
   DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 21)  // -V522
 #endif
 */

/*
 #ifdef USES_C022
   DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 22)  // -V522
 #endif
 */

/*
 #ifdef USES_C023
   DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 23)  // -V522
 #endif
 */

/*
 #ifdef USES_C024
   DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 24)  // -V522
 #endif
 */

/*
 #ifdef USES_C025
   DEFINE_Cxxx_DELAY_QUEUE_MACRO_CPP(0, 25)  // -V522
 #endif
 */


// When extending this, search for EXTEND_CONTROLLER_IDS
// in the code to find all places that need to be updated too.

#include "../ControllerQueue/Queue_element_base.h"

Queue_element_base::Queue_element_base() :
  _controller_idx(INVALID_CONTROLLER_INDEX),
  _taskIndex(INVALID_TASK_INDEX),
  _call_PLUGIN_PROCESS_CONTROLLER_DATA(false),
  _processByController(false)
{
  _timestamp = millis();
}

Queue_element_base::~Queue_element_base() {}

#include "../ControllerQueue/C011_queue_element.h"

#include "../DataStructs/ESPEasy_EventStruct.h"

#ifdef USES_C011


C011_queue_element::C011_queue_element(const struct EventStruct *event) :
  idx(event->idx),
  sensorType(event->sensorType)
{
  _controller_idx = event->ControllerIndex;
  _taskIndex = event->TaskIndex;
}

size_t C011_queue_element::getSize() const {
  size_t total = sizeof(*this);

  total += uri.length();
  total += HttpMethod.length();
  total += header.length();
  total += postStr.length();
  return total;
}

bool C011_queue_element::isDuplicate(const Queue_element_base& other) const {
  const C011_queue_element& oth = static_cast<const C011_queue_element&>(other);

  if ((oth._controller_idx != _controller_idx) ||
      (oth._taskIndex != _taskIndex) ||
      (oth.sensorType != sensorType) ||
      (oth.idx != idx)) {
    return false;
  }
  return oth.uri.equals(uri) &&
         oth.HttpMethod.equals(HttpMethod) &&
         oth.header.equals(header) &&
         oth.postStr.equals(postStr);
}

#endif // ifdef USES_C011

#include "../ControllerQueue/C018_queue_element.h"

#ifdef USES_C018

# include "../DataStructs/ESPEasy_EventStruct.h"
# include "../DataStructs/UnitMessageCount.h"

# include "../ESPEasyCore/ESPEasy_Log.h"

# include "../Helpers/_CPlugin_LoRa_TTN_helper.h"
# include "../Helpers/StringConverter.h"

C018_queue_element::C018_queue_element(struct EventStruct *event, uint8_t sampleSetCount)
{
  _controller_idx = event->ControllerIndex;
  _taskIndex      = event->TaskIndex;
  # if FEATURE_PACKED_RAW_DATA
  move_special(packed, getPackedFromPlugin(event, sampleSetCount));

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLogMove(LOG_LEVEL_INFO, concat(F("C018 queue element: "), packed));
  }
  # endif // if FEATURE_PACKED_RAW_DATA
}

size_t C018_queue_element::getSize() const {
  return sizeof(*this) + packed.length();
}

bool C018_queue_element::isDuplicate(const Queue_element_base& other) const {
  const C018_queue_element& oth = static_cast<const C018_queue_element&>(other);

  if ((oth._controller_idx != _controller_idx) ||
      (oth._taskIndex != _taskIndex) ||
      (oth.packed != packed)) {
    return false;
  }
  return true;
}

#endif // ifdef USES_C018

#include "../ControllerQueue/MQTT_queue_element.h"

#if FEATURE_MQTT

#include "../Helpers/StringConverter.h"

MQTT_queue_element::MQTT_queue_element(int ctrl_idx,
                                       taskIndex_t TaskIndex,
                                       const String& topic, const String& payload,
                                       bool retained, bool callbackTask) :
  _retained(retained)
{
  _controller_idx                      = ctrl_idx;
  _taskIndex                           = TaskIndex;
  _call_PLUGIN_PROCESS_CONTROLLER_DATA = callbackTask;

  // Copy in the scope of the constructor, so we might store it in the 2nd heap
  move_special(_topic, String(topic));
  move_special(_payload, String(payload));

  removeEmptyTopics();
}

MQTT_queue_element::MQTT_queue_element(int         ctrl_idx,
                                       taskIndex_t TaskIndex,
                                       String   && topic,
                                       String   && payload,
                                       bool        retained,
                                       bool        callbackTask)
  : _retained(retained)
{
  _controller_idx                      = ctrl_idx;
  _taskIndex                           = TaskIndex;
  _call_PLUGIN_PROCESS_CONTROLLER_DATA = callbackTask;

  // Copy in the scope of the constructor, so we might store it in the 2nd heap
  move_special(_topic, std::move(topic));
  move_special(_payload, std::move(payload));
  removeEmptyTopics();
}

size_t MQTT_queue_element::getSize() const {
  return sizeof(*this) + _topic.length() + _payload.length();
}

bool MQTT_queue_element::isDuplicate(const Queue_element_base& other) const {
  if (_call_PLUGIN_PROCESS_CONTROLLER_DATA || other._call_PLUGIN_PROCESS_CONTROLLER_DATA) {
    return false;
  }
  const MQTT_queue_element& oth = static_cast<const MQTT_queue_element&>(other);

  // TD-er: We do not compare the taskindex.
  // If it were to make a difference, the topic would be different.
  if ((oth._controller_idx != _controller_idx) ||
      (oth._retained != _retained) ||
      (oth._topic != _topic) ||
      (oth._payload != _payload)) {
    return false;
  }
  return true;
}

void MQTT_queue_element::removeEmptyTopics() {
  // some parts of the topic may have been replaced by empty strings,
  // or "/status" may have been appended to a topic ending with a "/"
  // Get rid of "//"
  while (_topic.indexOf(F("//")) != -1) {
    _topic.replace(F("//"), F("/"));
  }
}

#endif // if FEATURE_MQTT

#include "../ControllerQueue/C015_queue_element.h"

#include "../DataStructs/ESPEasy_EventStruct.h"

#ifdef USES_C015

#include "../Helpers/StringConverter.h"

C015_queue_element::C015_queue_element(C015_queue_element&& other)
  : idx(other.idx)
  , valuesSent(other.valuesSent)
  , valueCount(other.valueCount)
{
  _timestamp      = other._timestamp;
  _controller_idx = other._controller_idx;
  _taskIndex      = other._taskIndex;

  for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
    move_special(txt[i], std::move(other.txt[i]));
    vPin[i] = other.vPin[i];
  }
}

C015_queue_element::C015_queue_element(const struct EventStruct *event, uint8_t value_count) :
  idx(event->idx),
  valuesSent(0),
  valueCount(value_count) {
  _controller_idx = event->ControllerIndex;
  _taskIndex      = event->TaskIndex;
}

C015_queue_element& C015_queue_element::operator=(C015_queue_element&& other) {
  idx             = other.idx;
  _timestamp      = other._timestamp;
  _taskIndex      = other._taskIndex;
  _controller_idx = other._controller_idx;
  valuesSent      = other.valuesSent;
  valueCount      = other.valueCount;

  for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
    move_special(txt[i], std::move(other.txt[i]));
    vPin[i] = other.vPin[i];
  }
  return *this;
}

bool C015_queue_element::checkDone(bool succesfull) const {
  if (succesfull) { ++valuesSent; }
  return valuesSent >= valueCount || valuesSent >= VARS_PER_TASK;
}

size_t C015_queue_element::getSize() const {
  size_t total = sizeof(*this);

  for (int i = 0; i < VARS_PER_TASK; ++i) {
    total += txt[i].length();
  }
  return total;
}

bool C015_queue_element::isDuplicate(const Queue_element_base& other) const {
  const C015_queue_element& oth = static_cast<const C015_queue_element&>(other);

  if ((oth._controller_idx != _controller_idx) ||
      (oth._taskIndex != _taskIndex) ||
      (oth.valueCount != valueCount) ||
      (oth.idx != idx)) {
    return false;
  }

  for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
    if (oth.txt[i] != txt[i]) {
      return false;
    }

    if (oth.vPin[i] != vPin[i]) {
      return false;
    }
  }
  return true;
}

#endif // ifdef USES_C015

#include "../ControllerQueue/SimpleQueueElement_string_only.h"

#include "../Helpers/StringConverter.h"


simple_queue_element_string_only::simple_queue_element_string_only(int ctrl_idx, taskIndex_t TaskIndex,  String&& req)
{
  _controller_idx = ctrl_idx;
  _taskIndex      = TaskIndex;
  move_special(txt, std::move(req));
}

size_t simple_queue_element_string_only::getSize() const {
  return sizeof(*this) + txt.length();
}

bool simple_queue_element_string_only::isDuplicate(const Queue_element_base& other) const {
  const simple_queue_element_string_only& oth = static_cast<const simple_queue_element_string_only&>(other);

  if ((oth._controller_idx != _controller_idx) ||
      (oth._taskIndex != _taskIndex) ||
      (oth.txt != txt)) {
    return false;
  }
  return true;
}

#include "../DataStructs/SchedulerTimerID.h"

#include "../Helpers/Misc.h"

SchedulerTimerID::SchedulerTimerID(SchedulerTimerType_e timerType)
{
  set4BitToUL(mixed_id, 0, static_cast<uint32_t>(timerType));
}

void SchedulerTimerID::setTimerType(SchedulerTimerType_e timerType)
{
  set4BitToUL(mixed_id, 0, static_cast<uint32_t>(timerType));
}

SchedulerTimerType_e SchedulerTimerID::getTimerType() const
{
  return static_cast<SchedulerTimerType_e>(get4BitFromUL(mixed_id, 0));
}

uint32_t SchedulerTimerID::getId() const
{
  return mixed_id >> 4;
}

#include "../DataStructs/RulesEventCache.h"

#include "../DataStructs/TimingStats.h"
#include "../Helpers/RulesMatcher.h"
#include "../Helpers/StringConverter.h"


RulesEventCache_element::RulesEventCache_element(
  const String& filename, size_t pos, const String& event, const String& action)
    : _filename(filename), _posInFile(pos), _event(event), _action(action)
  {}


RulesEventCache_element::RulesEventCache_element(
  const String& filename, size_t pos, String&& event, String&& action) :
  _posInFile(pos)
{
  move_special(_filename, String(filename));
  move_special(_event, std::move(event));
  move_special(_action, std::move(action));  
}


void RulesEventCache::clear()
{
  _eventCache.clear();
  _initialized = false;
}

void RulesEventCache::initialize()
{
  _initialized = true;
}

bool RulesEventCache::addLine(const String& line, const String& filename, size_t pos)
{
  String event, action;

  if (getEventFromRulesLine(line, event, action)) {
    // Do not emplace on the 2nd heap
    # ifdef USE_SECOND_HEAP
    HeapSelectDram ephemeral;
    # endif // ifdef USE_SECOND_HEAP
    #ifdef ESP32
    reserve_special(event, event.length());
    reserve_special(action, action.length());
    #endif

    _eventCache.emplace_back(filename, pos, std::move(event), std::move(action));
    return true;
  }
  return false;
}

RulesEventCache_vector::const_iterator RulesEventCache::findMatchingRule(const String& event, bool optimize)
{
  RulesEventCache_vector::iterator it   = _eventCache.begin();
//  RulesEventCache_vector::iterator prev = _eventCache.end();

  // FIXME TD-er: Disable optimize as it has some side effects.
  // For example, matching a specific event first and then a more generic one is perfectly normal to do.
  // But this optimization will then put the generic one in front as it will be matched more often.
  // Thus it will never match the more specific one anymore.


  for (; it != _eventCache.end(); ++it)
  {
    START_TIMER
    const bool match = ruleMatch(event, it->_event);
    STOP_TIMER(RULES_MATCH);

    if (match) {
      /*
      if (optimize) {
        it->_nrTimesMatched++;

        if (prev != _eventCache.end()) {
          // Check to see if we need to place this one more to the front of the vector
          // to speed up parsing.
          if (prev->_nrTimesMatched < it->_nrTimesMatched) {
            std::swap(*prev, *it);
            return prev;
          }
        }
      }
      */
      return it;
    }
/*
    if (optimize) {
      if (prev == _eventCache.end()) {
        prev = it;
      }
      else if (prev->_nrTimesMatched > it->_nrTimesMatched) {
        // Found one that's having a lower match rate
        prev = it;
      }
    }
    */
  }
  return it;
}

#include "../DataStructs/FactoryDefault_WiFi_NVS.h"

#ifdef ESP32

# include "../Globals/Settings.h"
# include "../Globals/SecuritySettings.h"
# include "../Helpers/StringConverter.h"

// Max. 15 char keys for ESPEasy Factory Default marked keys
# define FACTORY_DEFAULT_NVS_SSID1_KEY      "WIFI_SSID1"
# define FACTORY_DEFAULT_NVS_WPA_PASS1_KEY  "WIFI_PASS1"
# define FACTORY_DEFAULT_NVS_SSID2_KEY      "WIFI_SSID2"
# define FACTORY_DEFAULT_NVS_WPA_PASS2_KEY  "WIFI_PASS2"
# define FACTORY_DEFAULT_NVS_AP_PASS_KEY    "WIFI_AP_PASS"
# define FACTORY_DEFAULT_NVS_WIFI_FLAGS_KEY "WIFI_Flags"


void FactoryDefault_WiFi_NVS::fromSettings() {
  bits.IncludeHiddenSSID              = Settings.IncludeHiddenSSID();
  bits.ApDontForceSetup               = Settings.ApDontForceSetup();
  bits.DoNotStartAP                   = Settings.DoNotStartAP();
  bits.ForceWiFi_bg_mode              = Settings.ForceWiFi_bg_mode();
  bits.WiFiRestart_connection_lost    = Settings.WiFiRestart_connection_lost();
  bits.WifiNoneSleep                  = Settings.WifiNoneSleep();
  bits.gratuitousARP                  = Settings.gratuitousARP();
  bits.UseMaxTXpowerForSending        = Settings.UseMaxTXpowerForSending();
  bits.UseLastWiFiFromRTC             = Settings.UseLastWiFiFromRTC();
  bits.WaitWiFiConnect                = Settings.WaitWiFiConnect();
  bits.SDK_WiFi_autoreconnect         = Settings.SDK_WiFi_autoreconnect();
  bits.HiddenSSID_SlowConnectPerBSSID = Settings.HiddenSSID_SlowConnectPerBSSID();
  bits.EnableIPv6                     = Settings.EnableIPv6();
  bits.PassiveWiFiScan                = Settings.PassiveWiFiScan();
}

void FactoryDefault_WiFi_NVS::applyToSettings() const {
  Settings.IncludeHiddenSSID(bits.IncludeHiddenSSID);
  Settings.ApDontForceSetup(bits.ApDontForceSetup);
  Settings.DoNotStartAP(bits.DoNotStartAP);
  Settings.ForceWiFi_bg_mode(bits.ForceWiFi_bg_mode);
  Settings.WiFiRestart_connection_lost(bits.WiFiRestart_connection_lost);
  Settings.WifiNoneSleep(bits.WifiNoneSleep);
  Settings.gratuitousARP(bits.gratuitousARP);
  Settings.UseMaxTXpowerForSending(bits.UseMaxTXpowerForSending);
  Settings.UseLastWiFiFromRTC(bits.UseLastWiFiFromRTC);
  Settings.WaitWiFiConnect(bits.WaitWiFiConnect);
  Settings.SDK_WiFi_autoreconnect(bits.SDK_WiFi_autoreconnect);
  Settings.HiddenSSID_SlowConnectPerBSSID(bits.HiddenSSID_SlowConnectPerBSSID);
  Settings.EnableIPv6(bits.EnableIPv6);
  Settings.PassiveWiFiScan(bits.PassiveWiFiScan);
}

struct FactoryDefault_WiFi_NVS_securityPrefs {
  FactoryDefault_WiFi_NVS_securityPrefs(const __FlashStringHelper *pref,
                                        char                      *dest,
                                        size_t                     size)
    : _pref(pref), _dest(dest), _size(size) {}

  const __FlashStringHelper *_pref;
  char                      *_dest;
  size_t                     _size;
};

const FactoryDefault_WiFi_NVS_securityPrefs _WiFi_NVS_securityPrefs_values[] = {
  { F(FACTORY_DEFAULT_NVS_SSID1_KEY),     SecuritySettings.WifiSSID,  sizeof(SecuritySettings.WifiSSID)  },
  { F(FACTORY_DEFAULT_NVS_WPA_PASS1_KEY), SecuritySettings.WifiKey,   sizeof(SecuritySettings.WifiKey)   },
  { F(FACTORY_DEFAULT_NVS_SSID2_KEY),     SecuritySettings.WifiSSID2, sizeof(SecuritySettings.WifiSSID2) },
  { F(FACTORY_DEFAULT_NVS_WPA_PASS2_KEY), SecuritySettings.WifiKey2,  sizeof(SecuritySettings.WifiKey2)  },
  { F(FACTORY_DEFAULT_NVS_AP_PASS_KEY),   SecuritySettings.WifiAPKey, sizeof(SecuritySettings.WifiAPKey) }
};


bool FactoryDefault_WiFi_NVS::applyToSettings_from_NVS(ESPEasy_NVS_Helper& preferences) {
  String tmp;
  constexpr unsigned nr__WiFi_NVS_securityPrefs_values = NR_ELEMENTS(_WiFi_NVS_securityPrefs_values);

  for (unsigned i = 0; i < nr__WiFi_NVS_securityPrefs_values; ++i) {
    if (preferences.getPreference(_WiFi_NVS_securityPrefs_values[i]._pref, tmp)) {
      safe_strncpy(_WiFi_NVS_securityPrefs_values[i]._dest, tmp, _WiFi_NVS_securityPrefs_values[i]._size);
    }
  }

  if (!preferences.getPreference(F(FACTORY_DEFAULT_NVS_WIFI_FLAGS_KEY), data)) {
    return false;
  }

  applyToSettings();
  return true;
}

void FactoryDefault_WiFi_NVS::fromSettings_to_NVS(ESPEasy_NVS_Helper& preferences) {
  fromSettings();
  preferences.setPreference(F(FACTORY_DEFAULT_NVS_WIFI_FLAGS_KEY), data);

  // Store WiFi credentials
  constexpr unsigned nr__WiFi_NVS_securityPrefs_values = NR_ELEMENTS(_WiFi_NVS_securityPrefs_values);

  for (unsigned i = 0; i < nr__WiFi_NVS_securityPrefs_values; ++i) {
    preferences.setPreference(_WiFi_NVS_securityPrefs_values[i]._pref, String(_WiFi_NVS_securityPrefs_values[i]._dest));
  }
}

void FactoryDefault_WiFi_NVS::clear_from_NVS(ESPEasy_NVS_Helper& preferences) {
  constexpr unsigned nr__WiFi_NVS_securityPrefs_values = NR_ELEMENTS(_WiFi_NVS_securityPrefs_values);

  for (unsigned i = 0; i < nr__WiFi_NVS_securityPrefs_values; ++i) {
    preferences.remove(_WiFi_NVS_securityPrefs_values[i]._pref);
  }
  preferences.remove(F(FACTORY_DEFAULT_NVS_WIFI_FLAGS_KEY));
}

#endif // ifdef ESP32

#include "../DataStructs/Scheduler_GPIOTimerID.h"

#include "../DataStructs/PinMode.h"

GPIOTimerID::GPIOTimerID(uint8_t GPIOType, uint8_t pinNumber, int Par1) :
  SchedulerTimerID(SchedulerTimerType_e::GPIO_timer)
{
  setId((Par1 << 16) + (pinNumber << 8) + GPIOType);
}


#ifndef BUILD_NO_DEBUG
String GPIOTimerID::decode() const
{
  String result;

  switch (getGPIO_type())
  {
    case GPIO_TYPE_INTERNAL:
      result += F("int");
      break;
    case GPIO_TYPE_MCP:
      result += F("MCP");
      break;
    case GPIO_TYPE_PCF:
      result += F("PCF");
      break;
    default:
      result += '?';
      break;
  }
  result += F(" pin: ");
  result += getPinNumber();
  result += F(" state: ");
  result += getPinStateValue();
  return result;
}

#endif // ifndef BUILD_NO_DEBUG

#include "../DataStructs/FactoryDefaultPref.h"

#include "../../ESPEasy_common.h"

ResetFactoryDefaultPreference_struct::ResetFactoryDefaultPreference_struct() : _preference(0)
{}

ResetFactoryDefaultPreference_struct::ResetFactoryDefaultPreference_struct(uint32_t preference) : _preference(preference)
{}

void ResetFactoryDefaultPreference_struct::set(uint32_t preference)
{
  _preference = preference;
}

#ifdef ESP32

// Max. 15 char keys for ESPEasy Factory Default marked keys
# define FACTORY_DEFAULT_NVS_PREF_KEY       "FacDefPref"

bool ResetFactoryDefaultPreference_struct::init(ESPEasy_NVS_Helper& preferences)
{
  if (preferences.begin(F(FACTORY_DEFAULT_NVS_NAMESPACE))) {
    return from_NVS(preferences);
  }
  return false;
}

bool ResetFactoryDefaultPreference_struct::from_NVS(ESPEasy_NVS_Helper& preferences)
{
  return preferences.getPreference(F(FACTORY_DEFAULT_NVS_PREF_KEY), _preference);
}

void ResetFactoryDefaultPreference_struct::to_NVS(ESPEasy_NVS_Helper& preferences) const
{
  preferences.setPreference(F(FACTORY_DEFAULT_NVS_PREF_KEY), _preference);
}

#endif // ifdef ESP32


/*
   DeviceModel ResetFactoryDefaultPreference_struct::getDeviceModel() const {
   return static_cast<DeviceModel>(_preference & 0xFF);
   }

   void ResetFactoryDefaultPreference_struct::setDeviceModel(DeviceModel model) {
   _preference &= ~(0xFF); // set DeviceModel bits to 0
   _preference |= static_cast<uint32_t>(model);
   }

   bool ResetFactoryDefaultPreference_struct::keepWiFi() const {
   return bitRead(_preference, 9);
   }

   void ResetFactoryDefaultPreference_struct::keepWiFi(bool keep) {
   bitWrite(_preference, 9, keep);
   }

   bool ResetFactoryDefaultPreference_struct::keepNTP() const {
   return bitRead(_preference, 10);
   }

   void ResetFactoryDefaultPreference_struct::keepNTP(bool keep) {
   bitWrite(_preference, 10, keep);
   }

   bool ResetFactoryDefaultPreference_struct::keepNetwork() const {
   return bitRead(_preference, 11);
   }

   void ResetFactoryDefaultPreference_struct::keepNetwork(bool keep) {
   bitWrite(_preference, 11, keep);
   }

   bool ResetFactoryDefaultPreference_struct::keepLogConsoleSettings() const {
   return bitRead(_preference, 12);
   }

   void ResetFactoryDefaultPreference_struct::keepLogConsoleSettings(bool keep) {
   bitWrite(_preference, 12, keep);
   }

   bool ResetFactoryDefaultPreference_struct::keepUnitName() const {
   return bitRead(_preference, 13);
   }

   void ResetFactoryDefaultPreference_struct::keepUnitName(bool keep) {
   bitWrite(_preference, 13, keep);
   }

   // filenr = 0...3 for files rules1.txt ... rules4.txt
   bool ResetFactoryDefaultPreference_struct::fetchRulesTXT(int filenr) const {
   return bitRead(_preference, 14 + filenr);
   }

   void ResetFactoryDefaultPreference_struct::fetchRulesTXT(int filenr, bool fetch) {
   bitWrite(_preference, 14 + filenr, fetch);
   }

   bool ResetFactoryDefaultPreference_struct::fetchNotificationDat() const {
   return bitRead(_preference, 18);
   }

   void ResetFactoryDefaultPreference_struct::fetchNotificationDat(bool fetch) {
   bitWrite(_preference, 18, fetch);
   }

   bool ResetFactoryDefaultPreference_struct::fetchSecurityDat() const {
   return bitRead(_preference, 19);
   }

   void ResetFactoryDefaultPreference_struct::fetchSecurityDat(bool fetch) {
   bitWrite(_preference, 19, fetch);
   }

   bool ResetFactoryDefaultPreference_struct::fetchConfigDat() const {
   return bitRead(_preference, 20);
   }

   void ResetFactoryDefaultPreference_struct::fetchConfigDat(bool fetch) {
   bitWrite(_preference, 20, fetch);
   }

   bool ResetFactoryDefaultPreference_struct::deleteFirst() const {
   return bitRead(_preference, 21);
   }

   void ResetFactoryDefaultPreference_struct::deleteFirst(bool checked) {
   bitWrite(_preference, 21, checked);
   }

   bool ResetFactoryDefaultPreference_struct::delete_Bak_Files() const {
   return bitRead(_preference, 23);
   }

   void ResetFactoryDefaultPreference_struct::delete_Bak_Files(bool checked) {
   bitWrite(_preference, 23, checked);
   }

   bool ResetFactoryDefaultPreference_struct::saveURL() const {
   return bitRead(_preference, 22);
   }

   void ResetFactoryDefaultPreference_struct::saveURL(bool checked) {
   bitWrite(_preference, 22, checked);
   }

   bool ResetFactoryDefaultPreference_struct::storeCredentials() const {
   return bitRead(_preference, 24);
   }

   void ResetFactoryDefaultPreference_struct::storeCredentials(bool checked) {
   bitWrite(_preference, 24, checked);
   }

   bool ResetFactoryDefaultPreference_struct::fetchProvisioningDat() const {
   return bitRead(_preference, 25);
   }

   void ResetFactoryDefaultPreference_struct::fetchProvisioningDat(bool checked) {
   bitWrite(_preference, 25, checked);
   }


   uint32_t ResetFactoryDefaultPreference_struct::getPreference() {
   return _preference;
   }
 */

#include "../DataStructs/ESPEasy_EventStruct.h"

#include "../../ESPEasy_common.h"

#include "../CustomBuild/ESPEasyLimits.h"
#include "../DataTypes/EventValueSource.h"
#include "../Globals/Plugins.h"
#include "../Globals/CPlugins.h"
#include "../Globals/NPlugins.h"

#include "../../_Plugin_Helper.h"

EventStruct::EventStruct(taskIndex_t taskIndex) :
  TaskIndex(taskIndex), BaseVarIndex(taskIndex * VARS_PER_TASK)
{
  if (taskIndex >= INVALID_TASK_INDEX) {
    BaseVarIndex = 0;
  }
}

void EventStruct::deep_copy(const struct EventStruct& other) {
  this->operator=(other);
}

void EventStruct::deep_copy(const struct EventStruct *other) {
  if (other != nullptr) {
    deep_copy(*other);
  }
}

void EventStruct::setTaskIndex(taskIndex_t taskIndex) {
  TaskIndex = taskIndex;

  if (TaskIndex < INVALID_TASK_INDEX) {
    BaseVarIndex = taskIndex * VARS_PER_TASK;
  }
  sensorType = Sensor_VType::SENSOR_TYPE_NOT_SET;
}

void EventStruct::clear() {
  *this = EventStruct();
}

Sensor_VType EventStruct::getSensorType() {
  const int tmp_idx = idx;

  checkDeviceVTypeForTask(this);
  idx = tmp_idx;
  return sensorType;
}

int64_t EventStruct::getTimestamp_as_systemMicros() const
{
  if (timestamp_sec == 0) 
  return  getMicros64();

  // FIXME TD-er: What to do when system time has not been set?
  int64_t res = node_time.Unixtime_to_systemMicros(timestamp_sec, timestamp_frac);
  if (res < 0) {
    // Unix time was from before we booted
    // FIXME TD-er: What to do now?
    return  getMicros64();
  }
  return res;
}

void EventStruct::setUnixTimeTimestamp()
{
  timestamp_sec = node_time.getUnixTime(timestamp_frac);
}

void EventStruct::setLocalTimeTimestamp()
{
  timestamp_sec = node_time.getLocalUnixTime(timestamp_frac);
}
#include "../DataStructs/PinMode.h"

constexpr pluginID_t P001_PLUGIN_ID{PLUGIN_GPIO_INT};
constexpr pluginID_t P009_PLUGIN_ID{PLUGIN_MCP_INT};
constexpr pluginID_t P019_PLUGIN_ID{PLUGIN_PCF_INT};


const pluginID_t PLUGIN_GPIO = P001_PLUGIN_ID;
const pluginID_t PLUGIN_MCP  = P009_PLUGIN_ID;
const pluginID_t PLUGIN_PCF  = P019_PLUGIN_ID;

#include "../DataStructs/Modbus.h"

#if FEATURE_MODBUS

#include "../DataStructs/ControllerSettingsStruct.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../Helpers/StringConverter.h"


Modbus::Modbus() : ModbusClient(nullptr), errcnt(0), timeout(0),
  TXRXstate(MODBUS_IDLE), RXavailable(0), payLoad(0) {}

Modbus::~Modbus() {
  if (ModbusClient) {
    ModbusClient->PR_9453_FLUSH_TO_CLEAR();
    ModbusClient->stop();
    delete (ModbusClient);
    delay(1);
    ModbusClient = nullptr;
  }
}

bool Modbus::begin(uint8_t function, uint8_t ModbusID, uint16_t ModbusRegister,  MODBUS_registerTypes_t type, char *IPaddress)
{
  currentRegister = ModbusRegister;
  currentFunction = function;
  incomingValue   = type;
  resultReceived  = false;
  if (ModbusClient) {
    ModbusClient->PR_9453_FLUSH_TO_CLEAR();
    ModbusClient->stop();
    delete (ModbusClient);
    delay(1);
    ModbusClient = nullptr;
  }
  ModbusClient    = new (std::nothrow) WiFiClient();
  if (ModbusClient == nullptr) {
    return false;
  }
  ModbusClient->setNoDelay(true);
  ModbusClient->setTimeout(CONTROLLER_CLIENTTIMEOUT_DFLT);
  timeout = millis();
  ModbusClient->PR_9453_FLUSH_TO_CLEAR();

  if (ModbusClient->connected()) {
    #ifndef BUILD_NO_DEBUG
    LogString += F(" already connected. ");
    #endif
  } else {
    #ifndef BUILD_NO_DEBUG
    LogString += F("connect: ");
    LogString += IPaddress;
    #endif

    if (ModbusClient->connect(IPaddress, 502) != 1) {
      #ifndef BUILD_NO_DEBUG
      LogString += F(" fail. ");
      #endif
      TXRXstate  = MODBUS_IDLE;
      errcnt++;

      #ifndef BUILD_NO_DEBUG
      if (LogString.length() > 1) { addLog(LOG_LEVEL_DEBUG, LogString); }
      #endif
      return false;
    } else {
      // Make sure no stale connection is left
      ModbusClient->stop();
    }
  }
  #ifndef BUILD_NO_DEBUG
  LogString += F(" OK, sending read request: ");
  #endif

  sendBuffer[6] = ModbusID;
  sendBuffer[7] = function;
  sendBuffer[8] = (ModbusRegister >> 8);
  sendBuffer[9] = (ModbusRegister & 0x00ff);

  if ((incomingValue == signed64) || (incomingValue == unsigned64)) {
    sendBuffer[11] = 4;
  }

  if ((incomingValue == signed32) || (incomingValue == unsigned32)) {
    sendBuffer[11] = 2;
  }

  if ((incomingValue == signed16) || (incomingValue == unsigned16)) {
    sendBuffer[11] = 1;
  }
  ModbusClient->PR_9453_FLUSH_TO_CLEAR();
  ModbusClient->write(&sendBuffer[0], sizeof(sendBuffer));

  #ifndef BUILD_NO_DEBUG
  for (unsigned int i = 0; i < sizeof(sendBuffer); i++) {
    LogString += ((unsigned int)(sendBuffer[i]));
    LogString += ' ';
  }
  #endif
  TXRXstate = MODBUS_RECEIVE;
  
  #ifndef BUILD_NO_DEBUG
  if (LogString.length() > 1) { addLog(LOG_LEVEL_DEBUG, LogString); }
  #endif
  return true;
}

bool Modbus::handle() {
  unsigned int RXavailable = 0;

  #ifndef BUILD_NO_DEBUG
  free_string(LogString);
  #endif
  int64_t rxValue = 0;

  switch (TXRXstate) {
    case MODBUS_IDLE:

      // clean up;
      if (ModbusClient) {
        ModbusClient->PR_9453_FLUSH_TO_CLEAR();
        ModbusClient->stop();
        delete (ModbusClient);
        delay(1);
        ModbusClient = nullptr;
      }
      break;

    case MODBUS_RECEIVE:

      if  (hasTimeout()) { break; }

      if  (ModbusClient->available() < 9) { break; }

      #ifndef BUILD_NO_DEBUG
      LogString += F("reading bytes: ");
      #endif

      for (int a = 0; a < 9; a++) {
        payLoad    = ModbusClient->read();
        #ifndef BUILD_NO_DEBUG
        LogString += (payLoad);
        LogString += ' ';
        #endif
      }
      #ifndef BUILD_NO_DEBUG
      LogString += F("> ");
      #endif

      if (payLoad > 8) {
        #ifndef BUILD_NO_DEBUG
        LogString += F("Payload too large !? ");
        #endif
        errcnt++;
        TXRXstate = MODBUS_IDLE;
      }
      // FIXME TD-er: Missing break?

    case MODBUS_RECEIVE_PAYLOAD:

      if  (hasTimeout()) { break; }
      RXavailable = ModbusClient->available();

      if (payLoad != RXavailable) {
        TXRXstate = MODBUS_RECEIVE_PAYLOAD;
        break;
      }

      for (unsigned int i = 0; i < RXavailable; i++) {
        rxValue = rxValue << 8;
        char a = ModbusClient->read();
        rxValue    = rxValue | a;
        #ifndef BUILD_NO_DEBUG
        LogString += static_cast<int>(a);  
        LogString += ' ';
        #endif
      }

      switch (incomingValue) {
        case signed16:
          result = (int16_t)rxValue;
          break;
        case unsigned16:
          result = (uint16_t)rxValue;
          break;
        case signed32:
          result = (int32_t)rxValue;
          break;
        case unsigned32:
          result = (uint32_t)rxValue;
          break;
        case signed64:
          result = (int64_t)rxValue;
          break;
        case unsigned64:
          result = (uint64_t)rxValue;
          break;
      }

      #ifndef BUILD_NO_DEBUG
      LogString += F("value: "); 
      LogString += result;
      #endif

      // if ((systemTimePresent()) && (hour() == 0)) errcnt = 0;

      TXRXstate = MODBUS_IDLE;

      resultReceived = true;
      break;

    default:
      #ifndef BUILD_NO_DEBUG
      LogString += F("default. ");
      #endif
      TXRXstate  = MODBUS_IDLE;
      break;
  }
  #ifndef BUILD_NO_DEBUG
  if (LogString.length() > 1) { addLog(LOG_LEVEL_DEBUG, LogString); }
  #endif
  return true;
}

bool Modbus::hasTimeout()
{
  if   ((millis() - timeout) > 10000) { // too many bytes or timeout
    #ifndef BUILD_NO_DEBUG
    LogString += F("Modbus RX timeout. "); 
    LogString += String(ModbusClient->available());
    #endif
    errcnt++;
    TXRXstate = MODBUS_IDLE;
    return true;
  }
  return false;
}

// tryread can be called in a round robin fashion. It will initiate a read if Modbus is idle and update the result once it is available.
// subsequent calls (if Modbus is busy etc. ) will return false and not update the result.
// Use to read multiple values non blocking in an re-entrant function. Not tested yet.
bool Modbus::tryRead(uint8_t ModbusID, uint16_t M_register,  MODBUS_registerTypes_t type, char *IPaddress, ESPEASY_RULES_FLOAT_TYPE& result) {
  if (isBusy()) { return false; // not done yet
  }

  if (available()) {
    if ((currentFunction == MODBUS_FUNCTION_READ) && (currentRegister == M_register)) {
      result = read(); // result belongs to this request.
      return true;
    }
  } else {
    begin(MODBUS_FUNCTION_READ, ModbusID, M_register, type, IPaddress); // idle and no result -> begin read request
  }
  return false;
}

#endif // FEATURE_MODBUS

#include "../DataStructs/tcp_cleanup.h"



#if defined(ESP8266)

struct tcp_pcb;
extern struct tcp_pcb* tcp_tw_pcbs;
extern "C" void tcp_abort (struct tcp_pcb* pcb);

void tcpCleanup()
{

     while(tcp_tw_pcbs!=nullptr)
    {
      tcp_abort(tcp_tw_pcbs);
    }

 }
#endif
#include "../DataStructs/Scheduler_PluginTaskTimerID.h"

#include "../Globals/Plugins.h"
#include "../Helpers/Misc.h"

PluginTaskTimerID::PluginTaskTimerID(taskIndex_t       taskIndex,
                                     int               Par1,
                                     PluginFunctions_e function) :
  SchedulerTimerID(SchedulerTimerType_e::PLUGIN_TASKTIMER_IN_e)
{
  constexpr unsigned nrBitsTaskIndex      = NR_BITS(TASKS_MAX);
  constexpr unsigned mask_taskIndex       = MASK_BITS(nrBitsTaskIndex);
  constexpr unsigned nrBitsPluginFunction = NrBitsPluginFunctions;
  constexpr unsigned mask_function        = MASK_BITS(nrBitsPluginFunction);

  if (validTaskIndex(taskIndex)) {
    setId((taskIndex & mask_taskIndex) |
         ((function & mask_function) << nrBitsTaskIndex) |
         (Par1 << (nrBitsTaskIndex + nrBitsPluginFunction)));
  }
}

taskIndex_t PluginTaskTimerID::getTaskIndex() const
{
  constexpr unsigned nrBitsTaskIndex = NR_BITS(TASKS_MAX);
  constexpr unsigned mask_taskIndex  = MASK_BITS(nrBitsTaskIndex);

  return static_cast<taskIndex_t>(getId() & mask_taskIndex);
}

PluginFunctions_e PluginTaskTimerID::getFunction() const
{
  constexpr unsigned nrBitsTaskIndex      = NR_BITS(TASKS_MAX);
  constexpr unsigned nrBitsPluginFunction = NrBitsPluginFunctions;
  constexpr unsigned mask_function        = MASK_BITS(nrBitsPluginFunction);

  return static_cast<PluginFunctions_e>((getId() >> nrBitsTaskIndex) & mask_function);
}

#ifndef BUILD_NO_DEBUG
String PluginTaskTimerID::decode() const
{
  const taskIndex_t taskIndex = getTaskIndex();

  if (validTaskIndex(taskIndex)) {
    return getTaskDeviceName(taskIndex);
  }
  return String(getId());
}

#endif // ifndef BUILD_NO_DEBUG

#include "../DataStructs/ProvisioningStruct.h"

#if FEATURE_CUSTOM_PROVISIONING

# include "../Helpers/StringConverter.h"
# include "../Helpers/Hardware.h"

ProvisioningStruct::ProvisioningStruct() {
  _allBits = 0u;
}

void ProvisioningStruct::validate() {
  ZERO_TERMINATE(user);
  ZERO_TERMINATE(pass);
  ZERO_TERMINATE(url);
}

bool ProvisioningStruct::matchingFlashSize() const
{
  return modelMatchingFlashSize(ResetFactoryDefaultPreference.getDeviceModel());
}

bool ProvisioningStruct::setUser(const String& username)
{
  return safe_strncpy(user, username, sizeof(user));
}

bool ProvisioningStruct::setPass(const String& password)
{
  return safe_strncpy(pass, password, sizeof(pass));
}

bool ProvisioningStruct::setUrl(const String& url_str)
{
  return safe_strncpy(url, url_str, sizeof(url));
}

bool ProvisioningStruct::fetchFileTypeAllowed(FileType::Enum filetype, unsigned int filenr) const
{
  switch (filetype) {
    case FileType::CONFIG_DAT:       return allowedFlags.allowFetchConfigDat;
    case FileType::SECURITY_DAT:     return allowedFlags.allowFetchSecurityDat;
    case FileType::NOTIFICATION_DAT: return allowedFlags.allowFetchNotificationDat;
    case FileType::RULES_TXT:        return (filenr < RULESETS_MAX) && bitRead(allowedFlags.allowFetchRules, filenr);
    case FileType::PROVISIONING_DAT: return allowedFlags.allowFetchProvisioningDat;

    case FileType::MAX_FILETYPE:
      break;
  }
  return false;
}

void ProvisioningStruct::setFetchFileTypeAllowed(FileType::Enum filetype, unsigned int filenr, bool checked)
{
  switch (filetype) {
    case FileType::CONFIG_DAT:       allowedFlags.allowFetchConfigDat       = checked; break;
    case FileType::SECURITY_DAT:     allowedFlags.allowFetchSecurityDat     = checked; break;
    case FileType::NOTIFICATION_DAT: allowedFlags.allowFetchNotificationDat = checked; break;
    case FileType::PROVISIONING_DAT: allowedFlags.allowFetchProvisioningDat = checked; break;
    case FileType::RULES_TXT:

      if (filenr < RULESETS_MAX) {
        bitWrite(allowedFlags.allowFetchRules, filenr, checked);
      }
      break;

    case FileType::MAX_FILETYPE:
      break;
  }
}

#endif // if FEATURE_CUSTOM_PROVISIONING

#include "../DataStructs/PortStatusStruct.h"

#include "../DataStructs/PinMode.h"


portStatusStruct::portStatusStruct() : state(-1), output(-1), command(0), init(0), not_used(0), mode(0), task(0), monitor(0), forceMonitor(0),
  forceEvent(0), previousTask(-1), x(INVALID_DEVICE_INDEX) {}

uint16_t portStatusStruct::getDutyCycle() const
{
  if (mode == PIN_MODE_PWM) {
    return dutyCycle;
  }
  return 0;
}

int16_t portStatusStruct::getValue() const
{
  if (mode == PIN_MODE_PWM || mode == PIN_MODE_SERVO)
    return dutyCycle;
  return state;
}

#include "../DataStructs/PluginStats_Config.h"


#if FEATURE_PLUGIN_STATS

PluginStats_Config_t::PluginStats_Config_t(const PluginStats_Config_t& other)
{
  setStored(other.getStored());
}

PluginStats_Config_t & PluginStats_Config_t::operator=(const PluginStats_Config_t& other)
{
  setStored(other.getStored());
  return *this;
}

#endif // if FEATURE_PLUGIN_STATS

#include "../DataStructs/FactoryDefault_Network_NVS.h"

#ifdef ESP32

# include "../Globals/Settings.h"
# include "../Helpers/StringConverter.h"

// Max. 15 char keys for ESPEasy Factory Default marked keys
# define FACTORY_DEFAULT_NVS_WIFI_IP_KEY       "WiFI_IP"
# ifdef FEATURE_ETHERNET
#  define FACTORY_DEFAULT_NVS_ETH_IP_KEY       "ETH_IP"
#  define FACTORY_DEFAULT_NVS_ETH_HW_CONF_KEY  "ETH_HW_CONF"
# endif // ifdef FEATURE_ETHERNET

bool FactoryDefault_Network_NVS::applyToSettings_from_NVS(ESPEasy_NVS_Helper& preferences)
{
  bool res = false;

  uint8_t *write = reinterpret_cast<uint8_t *>(&Settings);

  if (preferences.getPreference(F(FACTORY_DEFAULT_NVS_WIFI_IP_KEY), IP_data, sizeof(IP_data))) {
    // TD-er: This data is stored in sequence, so we can do a single memcpy call
    constexpr unsigned int offset = offsetof(SettingsStruct, IP);
    memcpy(write + offset, IP_data, sizeof(IP_data));
    res = true;
  }
# ifdef FEATURE_ETHERNET

  if (preferences.getPreference(F(FACTORY_DEFAULT_NVS_ETH_IP_KEY), IP_data, sizeof(IP_data))) {
    constexpr unsigned int offset = offsetof(SettingsStruct, ETH_IP);

    memcpy(write + offset, IP_data, sizeof(IP_data));
    res = true;
  }

  if (preferences.getPreference(F(FACTORY_DEFAULT_NVS_ETH_HW_CONF_KEY), ETH_HW_conf)) {
    Settings.ETH_Phy_Addr   = bits.ETH_Phy_Addr;
    Settings.ETH_Pin_mdc_cs    = bits.ETH_Pin_mdc_cs;
    Settings.ETH_Pin_mdio_irq   = bits.ETH_Pin_mdio_irq;
    Settings.ETH_Pin_power_rst  = bits.ETH_Pin_power_rst;
    Settings.ETH_Phy_Type   = static_cast<EthPhyType_t>(bits.ETH_Phy_Type);
    Settings.ETH_Clock_Mode = static_cast<EthClockMode_t>(bits.ETH_Clock_Mode);
    Settings.NetworkMedium  = static_cast<NetworkMedium_t>(bits.NetworkMedium);
    res                     = true;
  }
# endif // ifdef FEATURE_ETHERNET
  return res;
}

void FactoryDefault_Network_NVS::fromSettings_to_NVS(ESPEasy_NVS_Helper& preferences)
{
  const uint8_t *read = reinterpret_cast<const uint8_t *>(&Settings);

  {
    constexpr unsigned int offset = offsetof(SettingsStruct, IP);
    memcpy(IP_data, read + offset, sizeof(IP_data));
    preferences.setPreference(F(FACTORY_DEFAULT_NVS_WIFI_IP_KEY), IP_data, sizeof(IP_data));
  }
# ifdef FEATURE_ETHERNET
  {
    constexpr unsigned int offset = offsetof(SettingsStruct, ETH_IP);
    memcpy(IP_data, read + offset, sizeof(IP_data));
    preferences.setPreference(F(FACTORY_DEFAULT_NVS_ETH_IP_KEY), IP_data, sizeof(IP_data));
  }
  {
    bits.ETH_Phy_Addr   = Settings.ETH_Phy_Addr;
    bits.ETH_Pin_mdc_cs    = Settings.ETH_Pin_mdc_cs;
    bits.ETH_Pin_mdio_irq   = Settings.ETH_Pin_mdio_irq;
    bits.ETH_Pin_power_rst  = Settings.ETH_Pin_power_rst;
    bits.ETH_Phy_Type   = static_cast<uint8_t>(Settings.ETH_Phy_Type);
    bits.ETH_Clock_Mode = static_cast<uint8_t>(Settings.ETH_Clock_Mode);
    bits.NetworkMedium  = static_cast<uint8_t>(Settings.NetworkMedium);
    preferences.setPreference(F(FACTORY_DEFAULT_NVS_ETH_HW_CONF_KEY), ETH_HW_conf);
  }
# endif // ifdef FEATURE_ETHERNET
}

void FactoryDefault_Network_NVS::clear_from_NVS(ESPEasy_NVS_Helper& preferences)
{
  preferences.remove(F(FACTORY_DEFAULT_NVS_WIFI_IP_KEY));
# ifdef FEATURE_ETHERNET
  preferences.remove(F(FACTORY_DEFAULT_NVS_ETH_IP_KEY));
  preferences.remove(F(FACTORY_DEFAULT_NVS_ETH_IP_KEY));
# endif // ifdef FEATURE_ETHERNET
}

#endif  // ifdef ESP32

#include "../DataStructs/SystemTimerStruct.h"

#include "../DataStructs/ESPEasy_EventStruct.h"
#include "../ESPEasyCore/ESPEasy_Log.h"


// Rules Timer use
// ***************
systemTimerStruct::systemTimerStruct(int recurringCount, unsigned long msecFromNow, unsigned int timerIndex, int alternateInterval) :
  _recurringCount(recurringCount), _interval(msecFromNow), _timerIndex(timerIndex), _remainder(0), _loopCount(1), _alternateInterval(
    alternateInterval)
{
  if ((recurringCount > 0) && !hasAlternateInterval()) {
    // Will run with _recurringCount == 0, so must subtract one when setting the value.
    _recurringCount--;
  }

  if (msecFromNow == 0) {
    // Create a new timer which should be "scheduled" now to clear up any data
    _recurringCount = 0; // Do not reschedule
    _loopCount      = 0; // Do not execute
    #ifndef BUILD_MINIMAL_OTA
    addLog(LOG_LEVEL_INFO, F("TIMER: disable timer"));
    #endif
  }

  if (hasAlternateInterval()) {
    // Need to double the recurring count, or at least set it to 1 to make sure the alternate interval is also ran at least once.
    if (_recurringCount > 0) {
      _recurringCount *= 2;
    } else if (_recurringCount == 0) {
      _recurringCount = 1;
    }
  }
}

struct EventStruct systemTimerStruct::toEvent() const {
  struct EventStruct TempEvent(TaskIndex);

  TempEvent.Par1 = _recurringCount;
  TempEvent.Par2 = _interval;
  TempEvent.Par3 = _timerIndex;
  TempEvent.Par4 = _remainder;
  TempEvent.Par5 = _loopCount;
  return TempEvent;
}

void systemTimerStruct::fromEvent(taskIndex_t taskIndex,
                                  int         Par1,
                                  int         Par2,
                                  int         Par3,
                                  int         Par4,
                                  int         Par5)
{
  TaskIndex       = taskIndex;
  _recurringCount = Par1;
  _interval       = Par2;
  _timerIndex     = Par3;
  _remainder      = Par4;
  _loopCount      = Par5;
}

void systemTimerStruct::markNextRecurring() {
  toggleAlternateState(); // Will only toggle if it has an alternate state

  if (_recurringCount > 0) {
    // This is a timer with a limited number of runs, so decrease its value.
    _recurringCount--;
  }

  if (_loopCount > 0) {
    // This one should be executed, so increase the count.
    _loopCount++;
  }
}

void systemTimerStruct::toggleAlternateState() {
  if (hasAlternateInterval()) {
    _alternateState = !_alternateState;
  } else {
    _alternateState = false;
  }
}

#include "../DataStructs/UserVarStruct.h"

#include "../DataStructs/ESPEasy_EventStruct.h"
#include "../DataStructs/TimingStats.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../Globals/Cache.h"
#include "../Globals/Plugins.h"
#include "../Globals/RulesCalculate.h"
#include "../Helpers/_Plugin_SensorTypeHelper.h"
#include "../Helpers/CRC_functions.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringParser.h"



void UserVarStruct::clear()
{
  for (size_t i = 0; i < TASKS_MAX; ++i) {
    _rawData[i].clear();
  }
  _computed.clear();
#ifndef LIMIT_BUILD_SIZE
  _preprocessedFormula.clear();
#endif // ifndef LIMIT_BUILD_SIZE
  _prevValue.clear();
}

float UserVarStruct::operator[](unsigned int index) const
{
  const unsigned int taskIndex = index / VARS_PER_TASK;
  const unsigned int varNr     = index % VARS_PER_TASK;

  constexpr bool raw = false;

  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, Sensor_VType::SENSOR_TYPE_QUAD, raw);

  if (data != nullptr) {
    return data->getFloat(varNr);
  } else {
    static float errorvalue = NAN;
#ifndef LIMIT_BUILD_SIZE
    addLog(LOG_LEVEL_ERROR, F("UserVar index out of range"));
#endif
    return errorvalue;
  }
}

unsigned long UserVarStruct::getSensorTypeLong(taskIndex_t taskIndex, bool raw) const
{
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, 0, Sensor_VType::SENSOR_TYPE_ULONG, raw);

  if (data != nullptr) {
    return data->getSensorTypeLong();
  }
  return 0u;
}

void UserVarStruct::setSensorTypeLong(taskIndex_t taskIndex, unsigned long value)
{
  if (validTaskIndex(taskIndex)) {
    if (Cache.hasFormula(taskIndex, 0)) {
      const ESPEASY_RULES_FLOAT_TYPE tmp = value;
      applyFormulaAndSet(taskIndex, 0, tmp, Sensor_VType::SENSOR_TYPE_ULONG);
    } else {
      _rawData[taskIndex].setSensorTypeLong(value);
    }
  }
}

#if FEATURE_EXTENDED_TASK_VALUE_TYPES

int32_t UserVarStruct::getInt32(taskIndex_t    taskIndex,
                                taskVarIndex_t varNr,
                                bool           raw) const
{
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, Sensor_VType::SENSOR_TYPE_INT32_QUAD, raw);

  if (data != nullptr) {
    return data->getInt32(varNr);
  }
  return 0;
}

void UserVarStruct::setInt32(taskIndex_t    taskIndex,
                             taskVarIndex_t varNr,
                             int32_t        value)
{
  if (validTaskIndex(taskIndex)) {
    if (Cache.hasFormula(taskIndex, varNr)) {
      const ESPEASY_RULES_FLOAT_TYPE tmp = value;
      applyFormulaAndSet(taskIndex, varNr, tmp, Sensor_VType::SENSOR_TYPE_INT32_QUAD);
    } else {
      _rawData[taskIndex].setInt32(varNr, value);
    }
  }
}

#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES

uint32_t UserVarStruct::getUint32(taskIndex_t taskIndex, taskVarIndex_t varNr, bool raw) const
{
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, Sensor_VType::SENSOR_TYPE_UINT32_QUAD, raw);
#else // if FEATURE_EXTENDED_TASK_VALUE_TYPES
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, Sensor_VType::SENSOR_TYPE_NOT_SET, true);
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES

  if (data != nullptr) {
    return data->getUint32(varNr);
  }
  return 0u;
}

void UserVarStruct::setUint32(taskIndex_t taskIndex, taskVarIndex_t varNr, uint32_t value)
{
  if (validTaskIndex(taskIndex)) {
    // setUInt32 is used to read taskvalues back from RTC
    // If FEATURE_EXTENDED_TASK_VALUE_TYPES is not enabled, this function will never be used for anything else
#if FEATURE_EXTENDED_TASK_VALUE_TYPES

    if (Cache.hasFormula(taskIndex, varNr)) {
      const ESPEASY_RULES_FLOAT_TYPE tmp = value;
      applyFormulaAndSet(taskIndex, varNr, tmp, Sensor_VType::SENSOR_TYPE_UINT32_QUAD);
    } else
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
    {
      _rawData[taskIndex].setUint32(varNr, value);
    }
  }
}

#if FEATURE_EXTENDED_TASK_VALUE_TYPES

int64_t UserVarStruct::getInt64(taskIndex_t    taskIndex,
                                taskVarIndex_t varNr,
                                bool           raw) const
{
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, Sensor_VType::SENSOR_TYPE_INT64_DUAL, raw);

  if (data != nullptr) {
    return data->getInt64(varNr);
  }
  return 0;
}

void UserVarStruct::setInt64(taskIndex_t    taskIndex,
                             taskVarIndex_t varNr,
                             int64_t        value)
{
  if (validTaskIndex(taskIndex)) {
    if (Cache.hasFormula(taskIndex, varNr)) {
      const ESPEASY_RULES_FLOAT_TYPE tmp = value;

      if (applyFormulaAndSet(taskIndex, varNr, tmp, Sensor_VType::SENSOR_TYPE_INT64_DUAL)) {
        // Apply anyway so we don't loose resolution in the raw value
        _rawData[taskIndex].setInt64(varNr, value);
      }
    } else {
      _rawData[taskIndex].setInt64(varNr, value);
    }
  }
}

uint64_t UserVarStruct::getUint64(taskIndex_t    taskIndex,
                                  taskVarIndex_t varNr,
                                  bool           raw) const
{
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, Sensor_VType::SENSOR_TYPE_UINT64_DUAL, raw);

  if (data != nullptr) {
    return data->getUint64(varNr);
  }
  return 0u;
}

void UserVarStruct::setUint64(taskIndex_t    taskIndex,
                              taskVarIndex_t varNr,
                              uint64_t       value)
{
  if (validTaskIndex(taskIndex)) {
    if (Cache.hasFormula(taskIndex, varNr)) {
      const ESPEASY_RULES_FLOAT_TYPE tmp = value;

      if (applyFormulaAndSet(taskIndex, varNr, tmp, Sensor_VType::SENSOR_TYPE_UINT64_DUAL)) {
        // Apply anyway so we don't loose resolution in the raw value
        _rawData[taskIndex].setUint64(varNr, value);
      }
    } else {
      _rawData[taskIndex].setUint64(varNr, value);
    }
  }
}

#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES

float UserVarStruct::getFloat(taskIndex_t    taskIndex,
                              taskVarIndex_t varNr,
                              bool           raw) const
{
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, Sensor_VType::SENSOR_TYPE_QUAD, raw);

  if (data != nullptr) {
    return data->getFloat(varNr);
  }
  return 0.0f;
}

void UserVarStruct::setFloat(taskIndex_t    taskIndex,
                             taskVarIndex_t varNr,
                             float          value)
{
  if (validTaskIndex(taskIndex)) {
    if (Cache.hasFormula(taskIndex, varNr)) {
      const ESPEASY_RULES_FLOAT_TYPE tmp = value;
      applyFormulaAndSet(taskIndex, varNr, tmp, Sensor_VType::SENSOR_TYPE_QUAD);
    } else {
      _rawData[taskIndex].setFloat(varNr, value);
    }
  }
}

#if FEATURE_EXTENDED_TASK_VALUE_TYPES
# if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
double UserVarStruct::getDouble(taskIndex_t taskIndex,
                                taskVarIndex_t varNr, bool raw) const
{
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, Sensor_VType::SENSOR_TYPE_DOUBLE_DUAL, raw);

  if (data != nullptr) {
    return data->getDouble(varNr);
  }
  return 0.0;
}

void UserVarStruct::setDouble(taskIndex_t    taskIndex,
                              taskVarIndex_t varNr,
                              double         value)
{
  if (validTaskIndex(taskIndex)) {
    if (Cache.hasFormula(taskIndex, varNr)) {
      applyFormulaAndSet(taskIndex, varNr, value, Sensor_VType::SENSOR_TYPE_DOUBLE_DUAL);
    } else {
      _rawData[taskIndex].setDouble(varNr, value);
    }
  }
}

# endif // if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
#endif  // if FEATURE_EXTENDED_TASK_VALUE_TYPES

ESPEASY_RULES_FLOAT_TYPE UserVarStruct::getAsDouble(taskIndex_t    taskIndex,
                                                    taskVarIndex_t varNr,
                                                    Sensor_VType   sensorType,
                                                    bool           raw) const
{
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, sensorType, raw);

  if (data != nullptr) {
    return data->getAsDouble(varNr, sensorType);
  }
  return 0.0;
}

String UserVarStruct::getAsString(taskIndex_t taskIndex, taskVarIndex_t varNr, Sensor_VType  sensorType, uint8_t nrDecimals, bool raw) const
{
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, sensorType, raw);

  if (data != nullptr) {
    if (nrDecimals == 255) {
      // TD-er: Should we use the set nr of decimals here, or not round at all?
      // See: https://github.com/letscontrolit/ESPEasy/issues/3721#issuecomment-889649437
      nrDecimals = Cache.getTaskDeviceValueDecimals(taskIndex, varNr);
    }

    return data->getAsString(varNr, sensorType, nrDecimals);
  }
  return EMPTY_STRING;
}

void UserVarStruct::set(taskIndex_t taskIndex, taskVarIndex_t varNr, const ESPEASY_RULES_FLOAT_TYPE& value, Sensor_VType sensorType)
{
  applyFormulaAndSet(taskIndex, varNr, value, sensorType);
}

bool UserVarStruct::isValid(taskIndex_t    taskIndex,
                            taskVarIndex_t varNr,
                            Sensor_VType   sensorType,
                            bool           raw) const
{
  const TaskValues_Data_t *data = getRawOrComputed(taskIndex, varNr, sensorType, raw);

  if (data != nullptr) {
    return data->isValid(varNr, sensorType);
  }
  return false;
}

uint8_t * UserVarStruct::get(size_t& sizeInBytes)
{
  constexpr size_t size_rawData = TASKS_MAX * sizeof(TaskValues_Data_t);

  sizeInBytes = size_rawData;
  return reinterpret_cast<uint8_t *>(&_rawData[0]);
}

const TaskValues_Data_t * UserVarStruct::getRawTaskValues_Data(taskIndex_t taskIndex) const
{
  if (validTaskIndex(taskIndex)) {
    return &_rawData[taskIndex];
  }
  return nullptr;
}

TaskValues_Data_t * UserVarStruct::getRawTaskValues_Data(taskIndex_t taskIndex)
{
  if (validTaskIndex(taskIndex)) {
    return &_rawData[taskIndex];
  }
  return nullptr;
}

uint32_t UserVarStruct::compute_CRC32() const
{
  const uint8_t   *buffer       = reinterpret_cast<const uint8_t *>(&_rawData[0]);
  constexpr size_t size_rawData = TASKS_MAX * sizeof(TaskValues_Data_t);

  return calc_CRC32(buffer, size_rawData);
}

void UserVarStruct::clear_computed(taskIndex_t taskIndex)
{
  auto it = _computed.find(taskIndex);

  if (it != _computed.end()) {
    _computed.erase(it);
  }

  for (taskVarIndex_t varNr = 0; validTaskVarIndex(varNr); ++varNr) {
    const uint16_t key = makeWord(taskIndex, varNr);
#ifndef LIMIT_BUILD_SIZE
    {
      auto it            = _preprocessedFormula.find(key);

      if (it != _preprocessedFormula.end()) {
        _preprocessedFormula.erase(it);
      }
    }
#endif // ifndef LIMIT_BUILD_SIZE
    {
      auto it            = _prevValue.find(key);

      if (it != _prevValue.end()) {
        _prevValue.erase(it);
      }
    }
  }
}

void UserVarStruct::markPluginRead(taskIndex_t taskIndex)
{
  struct EventStruct TempEvent(taskIndex);
  for (taskVarIndex_t varNr = 0; validTaskVarIndex(varNr); ++varNr) {
    if (Cache.hasFormula_with_prevValue(taskIndex, varNr)) {
      const uint16_t key = makeWord(taskIndex, varNr);
      _prevValue[key] = formatUserVarNoCheck(&TempEvent, varNr);
    }
  }
}

const TaskValues_Data_t * UserVarStruct::getRawOrComputed(
  taskIndex_t    taskIndex,
  taskVarIndex_t varNr,
  Sensor_VType   sensorType,
  bool           raw) const
{
  if (!raw && Cache.hasFormula(taskIndex, varNr)) {
    auto it = _computed.find(taskIndex);

    if ((it == _computed.end()) || !it->second.isSet(varNr)) {
      // Try to compute values which do have a formula but not yet a 'computed' value cached.
      // FIXME TD-er: This may yield unexpected results when formula contains references to %pvalue%


      // Should not apply set nr. of decimals when calculating a formula
      const uint8_t nrDecimals = 254;
      const String value   = getAsString(taskIndex, varNr, sensorType, nrDecimals, true);

      constexpr bool applyNow = true;

      if (applyFormula(taskIndex, varNr, value, sensorType, applyNow)) {
        it = _computed.find(taskIndex);
      }
    }

    if (it != _computed.end()) {
      if (it->second.isSet(varNr)) {
        return &(it->second.values);
      }
    }
  }
  return getRawTaskValues_Data(taskIndex);
}

bool UserVarStruct::applyFormula(taskIndex_t    taskIndex,
                                 taskVarIndex_t varNr,
                                 const String & value,
                                 Sensor_VType   sensorType,
                                 bool           applyNow) const
{
  if (!validTaskIndex(taskIndex) ||
      !validTaskVarIndex(varNr) ||
      (sensorType == Sensor_VType::SENSOR_TYPE_NOT_SET))
  {
    return false;
  }

  const bool formula_has_prevvalue = Cache.hasFormula_with_prevValue(taskIndex, varNr);

  if (!applyNow && !formula_has_prevvalue) {
    // Must check whether we can delay calculations until it is read for the first time.
    auto it = _computed.find(taskIndex);

    if (it != _computed.end()) {
      // Make sure it will apply formula when the value is actually read
      it->second.clear(varNr);
    }
    return true;
  }


  String formula = getPreprocessedFormula(taskIndex, varNr);
  bool   res     = true;

  if (!formula.isEmpty()
      #if FEATURE_STRING_VARIABLES
      && formula[1] != TASK_VALUE_PRESENTATION_PREFIX_CHAR
      #endif // FEATURE_STRING_VARIABLES
     ) // Ignore display-formula
  {
    START_TIMER;

    formula.replace(F("%value%"), value);

    // TD-er: Should we use the set nr of decimals here, or not round at all?
    // See: https://github.com/letscontrolit/ESPEasy/issues/3721#issuecomment-889649437
    if (formula_has_prevvalue) {
      const String prev_str = getPreviousValue(taskIndex, varNr, sensorType);
      formula.replace(F("%pvalue%"), prev_str.isEmpty() ? value : prev_str);
      /*
      addLog(LOG_LEVEL_INFO, 
        strformat(
          F("pvalue: %s, value: %s, formula: %s"), 
          prev_str.c_str(), 
          value.c_str(),
          formula.c_str()));
      */
    }

    ESPEASY_RULES_FLOAT_TYPE result{};

    if (!isError(Calculate_preProcessed(parseTemplate(formula), result))) {
      _computed[taskIndex].set(varNr, result, sensorType);
    } else {
      // FIXME TD-er: What to do now? Just copy the raw value, set error value or don't update?
      res = false;
    }

    STOP_TIMER(COMPUTE_FORMULA_STATS);
  }
  return res;
}

bool UserVarStruct::applyFormulaAndSet(taskIndex_t                     taskIndex,
                                       taskVarIndex_t                  varNr,
                                       const ESPEASY_RULES_FLOAT_TYPE& value,
                                       Sensor_VType                    sensorType)
{
  if (!Cache.hasFormula(taskIndex, varNr)
      #if FEATURE_STRING_VARIABLES
      || Cache.getTaskDeviceFormula(taskIndex, varNr)[1] == TASK_VALUE_PRESENTATION_PREFIX_CHAR
      #endif // if FEATURE_STRING_VARIABLES
     ) {
    _rawData[taskIndex].set(varNr, value, sensorType);
    return true;
  }

  // Use a temporary TaskValues_Data_t object to have uniform formatting
  TaskValues_Data_t tmp;

  tmp.set(varNr, value, sensorType);

  // Should not apply set nr. of decimals when calculating a formula
  const uint8_t nrDecimals = 254;
  const String  value_str  = tmp.getAsString(varNr, sensorType, nrDecimals);

  constexpr bool applyNow = false;

  if (applyFormula(taskIndex, varNr, value_str, sensorType, applyNow)) {
    _rawData[taskIndex].set(varNr, value, sensorType);
    return true;
  }
  return false;
}

String UserVarStruct::getPreprocessedFormula(taskIndex_t taskIndex, taskVarIndex_t varNr) const
{
  if (!Cache.hasFormula(taskIndex, varNr)) {
    return EMPTY_STRING;
  }

#ifndef LIMIT_BUILD_SIZE
  const uint16_t key = makeWord(taskIndex, varNr);
  auto it            = _preprocessedFormula.find(key);

  if (it == _preprocessedFormula.end()) {
    _preprocessedFormula.emplace(
      std::make_pair(
        key, 
        RulesCalculate_t::preProces(Cache.getTaskDeviceFormula(taskIndex, varNr))
        ));
  }
  return _preprocessedFormula[key];
#else // ifndef LIMIT_BUILD_SIZE
  return RulesCalculate_t::preProces(Cache.getTaskDeviceFormula(taskIndex, varNr));
#endif // ifndef LIMIT_BUILD_SIZE
}

String UserVarStruct::getPreviousValue(taskIndex_t taskIndex, taskVarIndex_t varNr, Sensor_VType sensorType) const
{
  /*
     if (!Cache.hasFormula_with_prevValue(taskIndex, varNr)) {
     // Should not happen.

     }
   */

  const uint16_t key = makeWord(taskIndex, varNr);
  auto it            = _prevValue.find(key);

  if (it != _prevValue.end()) {
    return it->second;
  }

  // Probably the first run, so just return the current value

  // Do not call getAsString here as this will result in stack overflow.
  return EMPTY_STRING;
}

#include "../DataStructs/Scheduler_SystemEventQueueTimerID.h"

#include "../Globals/CPlugins.h"
#include "../Globals/Plugins.h"


SystemEventQueueTimerID::SystemEventQueueTimerID(SchedulerPluginPtrType_e ptr_type, uint8_t Index, uint8_t Function) :
  SchedulerTimerID(SchedulerTimerType_e::SystemEventQueue)
{
  setId((static_cast<uint32_t>(ptr_type) << 16) +
       (Index << 8) +
       Function);
}


#ifndef BUILD_NO_DEBUG
String SystemEventQueueTimerID::decode() const
{
  const SchedulerPluginPtrType_e ptr_type = getPtrType();
  const uint8_t index                     = getIndex();
  String result;

  result += toString(ptr_type);
  result += ',';

  if (ptr_type == SchedulerPluginPtrType_e::ControllerPlugin) {
    result += getCPluginNameFromProtocolIndex(index);
  } else if (ptr_type == SchedulerPluginPtrType_e::TaskPlugin) {
    const deviceIndex_t dev_index = deviceIndex_t::toDeviceIndex(index);
    result += getPluginNameFromDeviceIndex(dev_index);
  } else {
    result += (index + 1);
  }
  result += ',';
  result += getFunction();
  return result;
}

#endif // ifndef BUILD_NO_DEBUG

#include "../DataStructs/RTCStruct.h"

  void RTCStruct::init() {
    ID1 = 0xAA;
    ID2 = 0x55;
    clearLastWiFi();
    factoryResetCounter = 0;
    deepSleepState = 0;
    bootFailedCount = 0;
    flashDayCounter = 0;
    lastWiFiSettingsIndex = 0;
    flashCounter = 0;
    bootCounter = 0;
    lastMixedSchedulerId = 0;
    unused1 = 0;
    unused2 = 0;
    lastSysTime = 0;
  }

  void RTCStruct::clearLastWiFi() {
    for (uint8_t i = 0; i < 6; ++i) {
      lastBSSID[i] = 0;
    }
    lastWiFiChannel = 0;
    lastWiFiSettingsIndex = 0;
  }

  bool RTCStruct::lastWiFi_set() const {
    return lastBSSID[0] != 0 && lastWiFiChannel != 0 && lastWiFiSettingsIndex != 0;
  }

#include "../DataStructs/ExtraTaskSettingsStruct.h"

#include "../../ESPEasy_common.h"

#include "../DataStructs/PluginStats_Config.h"

#include "../Helpers/Misc.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringGenerator_Plugin.h"

#define EXTRA_TASK_SETTINGS_VERSION 1


ExtraTaskSettingsStruct::ExtraTaskSettingsStruct()
{
  memset(this, 0, sizeof(ExtraTaskSettingsStruct));
  TaskIndex = INVALID_TASK_INDEX;
  version = EXTRA_TASK_SETTINGS_VERSION;
  for (int i = 0; i < VARS_PER_TASK; ++i) {
    TaskDeviceValueDecimals[i] = 2;
    TaskDeviceErrorValue[i] = NAN;
  }
}

void ExtraTaskSettingsStruct::clear() {
  // Need to make sure every byte between the members is also zero
  // Otherwise the checksum will fail and settings will be saved too often.
  memset(this, 0, sizeof(ExtraTaskSettingsStruct));
  TaskIndex = INVALID_TASK_INDEX;
  //dummy1 = 0;
  version = EXTRA_TASK_SETTINGS_VERSION;
  for (int i = 0; i < VARS_PER_TASK; ++i) {
    TaskDeviceValueDecimals[i] = 2;
    TaskDeviceErrorValue[i] = NAN;
  }
}

void ExtraTaskSettingsStruct::validate() {
  ZERO_TERMINATE(TaskDeviceName);

  for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
    ZERO_TERMINATE(TaskDeviceFormula[i]);
    ZERO_TERMINATE(TaskDeviceValueNames[i]);
  }

  if (dummy1 != 0) {
    // FIXME TD-er: This check was added to add the version for allowing to make transitions on the data.
    // If we've been using this for a while, we no longer need to check for the value of this dummy and we can re-use it for something else.
    dummy1  = 0;
    version = 0;
  }

  if (version != EXTRA_TASK_SETTINGS_VERSION) {
    if (version < 1) {
      // Need to initialize the newly added fields
      for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
        setIgnoreRangeCheck(i);
        TaskDeviceErrorValue[i] = NAN;
        VariousBits[i]          = 0u;
      }
    }
    version = EXTRA_TASK_SETTINGS_VERSION;
  }
}

ChecksumType ExtraTaskSettingsStruct::computeChecksum() const {
  return ChecksumType(reinterpret_cast<const uint8_t *>(this), sizeof(ExtraTaskSettingsStruct));
}

bool ExtraTaskSettingsStruct::checkUniqueValueNames() const {
  for (int i = 0; i < (VARS_PER_TASK - 1); ++i) {
    for (int j = i; j < VARS_PER_TASK; ++j) {
      if ((i != j) && (TaskDeviceValueNames[i][0] != 0)) {
        if (strcasecmp(TaskDeviceValueNames[i], TaskDeviceValueNames[j]) == 0) {
          return false;
        }
      }
    }
  }
  return true;
}

void ExtraTaskSettingsStruct::clearUnusedValueNames(uint8_t usedVars) {
  for (uint8_t i = usedVars; i < VARS_PER_TASK; ++i) {
    ZERO_FILL(TaskDeviceFormula[i]);
    ZERO_FILL(TaskDeviceValueNames[i]);
    TaskDeviceValueDecimals[i] = 2;
    setIgnoreRangeCheck(i);
    TaskDeviceErrorValue[i] = NAN;
    VariousBits[i]          = 0;
  }
}

bool ExtraTaskSettingsStruct::checkInvalidCharInNames(const char *name) const {
  int pos = 0;

  while (*(name + pos) != 0) {
    if (!validCharForNames(*(name + pos))) { return false; }
    ++pos;
  }
  return true;
}

bool ExtraTaskSettingsStruct::checkInvalidCharInNames() const {
  if (!checkInvalidCharInNames(&TaskDeviceName[0])) { return false; }

  for (int i = 0; i < VARS_PER_TASK; ++i) {
    if (!checkInvalidCharInNames(&TaskDeviceValueNames[i][0])) { return false; }
  }
  return true;
}

String ExtraTaskSettingsStruct::getInvalidCharsForNames() {
  return F(",-+/*=^%!#[]{}()");
}

bool ExtraTaskSettingsStruct::validCharForNames(char c) {
  return c != ' ' && getInvalidCharsForNames().indexOf(c) == -1;
}

void ExtraTaskSettingsStruct::setTaskDeviceValueName(taskVarIndex_t taskVarIndex, const String& str)
{
  if (validTaskVarIndex(taskVarIndex)) {
    safe_strncpy(
      TaskDeviceValueNames[taskVarIndex],
      str,
      sizeof(TaskDeviceValueNames[taskVarIndex]));
  }
}

void ExtraTaskSettingsStruct::setTaskDeviceValueName(taskVarIndex_t taskVarIndex, const __FlashStringHelper * str)
{
  setTaskDeviceValueName(taskVarIndex, String(str));
}

void ExtraTaskSettingsStruct::clearTaskDeviceValueName(taskVarIndex_t taskVarIndex)
{
  if (validTaskVarIndex(taskVarIndex)) {
    ZERO_FILL(TaskDeviceValueNames[taskVarIndex]);
  }
}

void ExtraTaskSettingsStruct::clearDefaultTaskDeviceValueNames()
{
  for (int i = 0; i < VARS_PER_TASK; ++i) {
    if (isDefaultTaskVarName(i)) {
      clearTaskDeviceValueName(i);
    }
  }
}

void ExtraTaskSettingsStruct::setAllowedRange(taskVarIndex_t taskVarIndex, const float& minValue, const float& maxValue)
{
  if (validTaskVarIndex(taskVarIndex)) {
    if (minValue > maxValue) {
      TaskDeviceMinValue[taskVarIndex] = maxValue;
      TaskDeviceMaxValue[taskVarIndex] = minValue;
    } else {
      TaskDeviceMinValue[taskVarIndex] = minValue;
      TaskDeviceMaxValue[taskVarIndex] = maxValue;
    }
  }
}

void ExtraTaskSettingsStruct::setIgnoreRangeCheck(taskVarIndex_t taskVarIndex)
{
  if (validTaskVarIndex(taskVarIndex)) {
    // Clear range to indicate no range check should be done.
    TaskDeviceMinValue[taskVarIndex] = 0.0f;
    TaskDeviceMaxValue[taskVarIndex] = 0.0f;
  }
}

bool ExtraTaskSettingsStruct::ignoreRangeCheck(taskVarIndex_t taskVarIndex) const
{
  if (validTaskVarIndex(taskVarIndex)) {
    return essentiallyEqual(TaskDeviceMinValue[taskVarIndex], TaskDeviceMaxValue[taskVarIndex]);
  }
  return true;
}

bool ExtraTaskSettingsStruct::valueInAllowedRange(taskVarIndex_t taskVarIndex, const float& value) const
{
  if (ignoreRangeCheck(taskVarIndex)) { return true; }

  if (validTaskVarIndex(taskVarIndex)) {
    return definitelyLessThan(value, TaskDeviceMaxValue[taskVarIndex]) ||
           definitelyGreaterThan(value, TaskDeviceMinValue[taskVarIndex]);
  }
  #ifndef BUILD_NO_DEBUG
  addLog(LOG_LEVEL_ERROR, F("Programming error: invalid taskVarIndex"));
  #endif // ifndef BUILD_NO_DEBUG
  return false;
}

float ExtraTaskSettingsStruct::checkAllowedRange(taskVarIndex_t taskVarIndex, const float& value) const
{
  if (!valueInAllowedRange(taskVarIndex, value)) {
    if (validTaskVarIndex(taskVarIndex)) {
      return TaskDeviceErrorValue[taskVarIndex];
    }
  }
  return value;
}

#if FEATURE_PLUGIN_STATS

// Plugin Stats is now only a single bit, but this may later changed into a combobox with some options.
// Thus leave 8 bits for the plugin stats options.

bool ExtraTaskSettingsStruct::enabledPluginStats(taskVarIndex_t taskVarIndex) const
{
  if (!validTaskVarIndex(taskVarIndex)) { return false; }
  return bitRead(VariousBits[taskVarIndex], 0);
}

void ExtraTaskSettingsStruct::enablePluginStats(taskVarIndex_t taskVarIndex, bool enabled)
{
  if (validTaskVarIndex(taskVarIndex)) {
    bitWrite(VariousBits[taskVarIndex], 0, enabled);
  }
}

bool ExtraTaskSettingsStruct::anyEnabledPluginStats() const
{
  for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
    if (enabledPluginStats(i)) { return true; }
  }
  return false;
}

PluginStats_Config_t ExtraTaskSettingsStruct::getPluginStatsConfig(taskVarIndex_t taskVarIndex) const
{
  if (!validTaskVarIndex(taskVarIndex)) { return PluginStats_Config_t(); }

  PluginStats_Config_t res(get8BitFromUL(VariousBits[taskVarIndex], 0));
  return res;
}

void ExtraTaskSettingsStruct::setPluginStatsConfig(taskVarIndex_t taskVarIndex, PluginStats_Config_t config)
{
  if (validTaskVarIndex(taskVarIndex)) {
    uint8_t value = config.getStoredBits();
    bitWrite(value, 1, bitRead(VariousBits[taskVarIndex], 1));
    set8BitToUL(VariousBits[taskVarIndex], 0, value);
  }
}


#endif // if FEATURE_PLUGIN_STATS

bool ExtraTaskSettingsStruct::isDefaultTaskVarName(taskVarIndex_t taskVarIndex) const
{
  if (!validTaskVarIndex(taskVarIndex)) { return false; }
  return bitRead(VariousBits[taskVarIndex], 1);
}

void ExtraTaskSettingsStruct::isDefaultTaskVarName(taskVarIndex_t taskVarIndex, bool isDefault)
{
  if (validTaskVarIndex(taskVarIndex)) {
    bitWrite(VariousBits[taskVarIndex], 1, isDefault);
  }
}

#if FEATURE_TASKVALUE_UNIT_OF_MEASURE
uint8_t ExtraTaskSettingsStruct::getTaskVarUnitOfMeasure(taskVarIndex_t taskVarIndex) const {
  if (!validTaskVarIndex(taskVarIndex)) { return 0u; }
  return get8BitFromUL(VariousBits[taskVarIndex], 8);
}

void ExtraTaskSettingsStruct::setTaskVarUnitOfMeasure(taskVarIndex_t taskVarIndex,
                                                      uint8_t        unitOfMeasure) {
  if (validTaskVarIndex(taskVarIndex)) {
    set8BitToUL(VariousBits[taskVarIndex], 8, unitOfMeasure);
  }
}
#endif // if FEATURE_TASKVALUE_UNIT_OF_MEASURE

#if FEATURE_CUSTOM_TASKVAR_VTYPE
uint8_t ExtraTaskSettingsStruct::getTaskVarCustomVType(taskVarIndex_t taskVarIndex) const {
  if (!validTaskVarIndex(taskVarIndex)) { return 0u; }
  return get8BitFromUL(VariousBits[taskVarIndex], 16);
}

void ExtraTaskSettingsStruct::setTaskVarCustomVType(taskVarIndex_t taskVarIndex,
                                                    uint8_t        customVType) {
  if (validTaskVarIndex(taskVarIndex)) {
    set8BitToUL(VariousBits[taskVarIndex], 16, customVType);
  }
}
#endif // if FEATURE_CUSTOM_TASKVAR_VTYPE

#if FEATURE_MQTT_STATE_CLASS
uint8_t ExtraTaskSettingsStruct::getTaskVarStateClass(taskVarIndex_t taskVarIndex) const {
  if (!validTaskVarIndex(taskVarIndex)) { return 0u; }
  return get3BitFromUL(VariousBits[taskVarIndex], 24); // 3 Bits 24, 25, 26
}

void ExtraTaskSettingsStruct::setTaskVarStateClass(taskVarIndex_t taskVarIndex,
                                                   uint8_t        stateClass) {
  if (validTaskVarIndex(taskVarIndex)) {
    set3BitToUL(VariousBits[taskVarIndex], 24, stateClass); // 3 Bits 24, 25, 26
  }
}
#endif // if FEATURE_MQTT_STATE_CLASS


void ExtraTaskSettingsStruct::populateDeviceValueNamesSeq(
  const __FlashStringHelper *valuename,
  size_t                     nrValues,
  uint8_t                    defaultDecimals,
  bool                       displayString)
{
  for (byte i = 0; i < VARS_PER_TASK; ++i) {
    if (i < nrValues) {
      safe_strncpy(
        TaskDeviceValueNames[i],
        Plugin_valuename(valuename, i, displayString),
        sizeof(TaskDeviceValueNames[i]));
      TaskDeviceValueDecimals[i] = defaultDecimals;
    } else {
      ZERO_FILL(TaskDeviceValueNames[i]);
    }
  }
}

#include "../DataStructs/ProtocolStruct.h"

ProtocolStruct::ProtocolStruct() :
    defaultPort(0), usesMQTT(false), usesAccount(false), usesPassword(false),
    usesTemplate(false), usesID(false), Custom(false), usesHost(true), usesPort(true),
    usesQueue(true), usesCheckReply(true), usesTimeout(true), usesSampleSets(false), 
    usesExtCreds(false), needsNetwork(true), allowsExpire(true), allowLocalSystemTime(false)
  #if FEATURE_MQTT_TLS
  , usesTLS(false)
  #endif
    {}

#include "../DataStructs/PluginStats_array.h"

#if FEATURE_PLUGIN_STATS

# include "../../_Plugin_Helper.h"

# include "../Globals/TimeZone.h"

# include "../Helpers/ESPEasy_math.h"
# include "../Helpers/Memory.h"

# include "../WebServer/Chart_JS.h"
# if FEATURE_TASKVALUE_UNIT_OF_MEASURE
#  include "../Helpers/ESPEasy_UnitOfMeasure.h"
# endif // if FEATURE_TASKVALUE_UNIT_OF_MEASURE

PluginStats_array::~PluginStats_array()
{
  for (size_t i = 0; i < VARS_PER_TASK; ++i) {
    if (_plugin_stats[i] != nullptr) {
      delete _plugin_stats[i];
      _plugin_stats[i] = nullptr;
    }
  }

  if (_plugin_stats_timestamps != nullptr) {
    free(_plugin_stats_timestamps);
    _plugin_stats_timestamps = nullptr;
  }
}

void PluginStats_array::initPluginStats(taskIndex_t taskIndex, taskVarIndex_t taskVarIndex)
{
  if (taskVarIndex < VARS_PER_TASK) {
    delete _plugin_stats[taskVarIndex];
    _plugin_stats[taskVarIndex] = nullptr;

    if (!hasStats()) {
      if (_plugin_stats_timestamps != nullptr) {
        free(_plugin_stats_timestamps);
        _plugin_stats_timestamps = nullptr;
      }
    }

    if (ExtraTaskSettings.enabledPluginStats(taskVarIndex)) {
      // Try to allocate in PSRAM or 2nd heap if possible
      constexpr unsigned size = sizeof(PluginStats);
      void *ptr               = special_calloc(1, size);

      if (ptr == nullptr) { _plugin_stats[taskVarIndex] = nullptr; }
      else {
        _plugin_stats[taskVarIndex] = new (ptr) PluginStats(
          ExtraTaskSettings.TaskDeviceValueDecimals[taskVarIndex],
          ExtraTaskSettings.TaskDeviceErrorValue[taskVarIndex]);
      }


      if (_plugin_stats[taskVarIndex] != nullptr) {
        # if FEATURE_TASKVALUE_UNIT_OF_MEASURE
        const uint8_t uomIndex = ExtraTaskSettings.getTaskVarUnitOfMeasure(taskVarIndex);
        String label(ExtraTaskSettings.TaskDeviceValueNames[taskVarIndex]);

        if (uomIndex != 0) {
          label = strformat(F("%s (%s)"), ExtraTaskSettings.TaskDeviceValueNames[taskVarIndex], toUnitOfMeasureName(uomIndex).c_str());
        }
        _plugin_stats[taskVarIndex]->setLabel(label);
        # else // if FEATURE_TASKVALUE_UNIT_OF_MEASURE
        _plugin_stats[taskVarIndex]->setLabel(ExtraTaskSettings.TaskDeviceValueNames[taskVarIndex]);
        # endif // if FEATURE_TASKVALUE_UNIT_OF_MEASURE
        # if FEATURE_CHART_JS
        const __FlashStringHelper *colors[] = { F("#A52422"), F("#BEA57D"), F("#0F4C5C"), F("#A4BAB7") };
        _plugin_stats[taskVarIndex]->_ChartJS_dataset_config.color         = colors[taskVarIndex];
        _plugin_stats[taskVarIndex]->_ChartJS_dataset_config.displayConfig = ExtraTaskSettings.getPluginStatsConfig(taskVarIndex);
        # endif // if FEATURE_CHART_JS

        if (_plugin_stats_timestamps != nullptr) {
          _plugin_stats[taskVarIndex]->setPluginStats_timestamp(_plugin_stats_timestamps);
        }
      }
    }
  }

  if (hasStats()) {
    if (_plugin_stats_timestamps == nullptr) {
      // Try to allocate in PSRAM if possible
      constexpr unsigned size = sizeof(PluginStats_timestamp);
      void *ptr               = special_calloc(1, size);

      if (ptr != nullptr) {
        // TODO TD-er: Let the task decide whether we need 1/50 sec resolution or 1/10
        // 1/10 sec resolution allows for ~13.6 years without overflow
        _plugin_stats_timestamps = new (ptr) PluginStats_timestamp(true);
      }

      for (size_t i = 0; i < VARS_PER_TASK; ++i) {
        _plugin_stats[taskVarIndex]->setPluginStats_timestamp(_plugin_stats_timestamps);
      }
    }
  }
}

void PluginStats_array::clearPluginStats(taskVarIndex_t taskVarIndex)
{
  if (taskVarIndex < VARS_PER_TASK) {
    if (_plugin_stats[taskVarIndex] != nullptr) {
      delete _plugin_stats[taskVarIndex];
      _plugin_stats[taskVarIndex] = nullptr;
    }
  }

  if (!hasStats()) {
    if (_plugin_stats_timestamps != nullptr) {
      free(_plugin_stats_timestamps);
      _plugin_stats_timestamps = nullptr;
    }
  }
}

void PluginStats_array::processTimeSet(const double& time_offset)
{
  if (_plugin_stats_timestamps != nullptr) {
    _plugin_stats_timestamps->processTimeSet(time_offset);
  }

  // Also update timestamps of peaks
  for (taskVarIndex_t taskVarIndex = 0; taskVarIndex < VARS_PER_TASK; ++taskVarIndex) {
    PluginStats *stats = getPluginStats(taskVarIndex);

    if (stats != nullptr) {
      stats->processTimeSet(time_offset);
    }
  }
}

bool PluginStats_array::hasStats() const
{
  for (size_t i = 0; i < VARS_PER_TASK; ++i) {
    if (_plugin_stats[i] != nullptr) { return true; }
  }
  return false;
}

bool PluginStats_array::hasPeaks() const
{
  for (size_t i = 0; i < VARS_PER_TASK; ++i) {
    if ((_plugin_stats[i] != nullptr) && _plugin_stats[i]->hasPeaks()) {
      return true;
    }
  }
  return false;
}

size_t PluginStats_array::nrSamplesPresent() const
{
  for (size_t i = 0; i < VARS_PER_TASK; ++i) {
    if (_plugin_stats[i] != nullptr) {
      return _plugin_stats[i]->getNrSamples();
    }
  }
  return 0;
}

size_t PluginStats_array::nrPluginStats() const
{
  size_t res{};

  for (size_t i = 0; i < VARS_PER_TASK; ++i) {
    if (_plugin_stats[i] != nullptr) {
      ++res;
    }
  }
  return res;
}

uint32_t PluginStats_array::getFullPeriodInSec(uint32_t& time_frac) const
{
  if (_plugin_stats_timestamps == nullptr) {
    time_frac = 0u;
    return 0u;
  }
  return _plugin_stats_timestamps->getFullPeriodInSec(time_frac);
}

void PluginStats_array::pushPluginStatsValues(
  struct EventStruct *event,
  bool                trackPeaks,
  bool                onlyUpdateTimestampWhenSame)
{
  if (validTaskIndex(event->TaskIndex)) {
    const uint8_t valueCount = getValueCountForTask(event->TaskIndex);

    if (valueCount > 0) {
      const Sensor_VType sensorType = event->getSensorType();

      const int64_t timestamp_sysmicros = event->getTimestamp_as_systemMicros();

      if (onlyUpdateTimestampWhenSame && (_plugin_stats_timestamps != nullptr)) {
        // When only updating the timestamp of the last entry,
        // we should look at the last 2 entries to see if they are the same.
        bool   isSame = true;
        size_t i      = 0;

        while (isSame && i < valueCount) {
          if (_plugin_stats[i] != nullptr) {
            const float value = UserVar.getAsDouble(event->TaskIndex, i, sensorType);

            if (!_plugin_stats[i]->matchesLastTwoEntries(value)) {
              isSame = false;
            }
          }
          ++i;
        }

        if (isSame) {
          _plugin_stats_timestamps->updateLast(timestamp_sysmicros);
          return;
        }
      }

      if (_plugin_stats_timestamps != nullptr) {
        _plugin_stats_timestamps->push(timestamp_sysmicros);
      }

      for (size_t i = 0; i < valueCount; ++i) {
        if (_plugin_stats[i] != nullptr) {
          const float value = UserVar.getAsDouble(event->TaskIndex, i, sensorType);
          _plugin_stats[i]->push(value);

          if (trackPeaks) {
            _plugin_stats[i]->trackPeak(value, timestamp_sysmicros);
          }
        }
      }
    }
  }
}

bool PluginStats_array::plugin_get_config_value_base(struct EventStruct *event,
                                                     String            & string) const
{
  // Full value name is something like "taskvaluename.avg"
  const String fullValueName = parseString(string, 1);
  const String valueName     = parseString(fullValueName, 1, '.');

  const uint8_t valueCount = getValueCountForTask(event->TaskIndex);

  for (taskVarIndex_t i = 0; i < valueCount; i++)
  {
    if (_plugin_stats[i] != nullptr) {
      // Check case insensitive, since the user entered value name can have any case.
      if (valueName.equalsIgnoreCase(Cache.getTaskDeviceValueName(event->TaskIndex, i)))
      {
        return _plugin_stats[i]->plugin_get_config_value_base(event, string);
      }
    }
  }
  return false;
}

bool PluginStats_array::plugin_write_base(struct EventStruct *event, const String& string)
{
  bool success     = false;
  const String cmd = parseString(string, 1);                // command

  const bool resetPeaks   = equals(cmd, F("resetpeaks"));   // Command: "taskname.resetPeaks"
  const bool clearSamples = equals(cmd, F("clearsamples")); // Command: "taskname.clearSamples"

  if (resetPeaks || clearSamples) {
    for (size_t i = 0; i < VARS_PER_TASK; ++i) {
      if (_plugin_stats[i] != nullptr) {
        if (resetPeaks) {
          success = true;
          _plugin_stats[i]->resetPeaks();
        }

        if (clearSamples) {
          success = true;
          _plugin_stats[i]->clearSamples();
        }
      }
    }
  }
  return success;
}

bool PluginStats_array::webformLoad_show_stats(struct EventStruct *event, bool showTaskValues) const
{
  bool somethingAdded = false;

  uint32_t time_frac{};
  const uint32_t duration  = getFullPeriodInSec(time_frac);
  const uint32_t nrSamples = nrSamplesPresent();

  if ((duration > 0) && (nrSamples > 1)) {
    const uint32_t duration_millis = unix_time_frac_to_millis(time_frac);
    addRowLabel(F("Total Duration"));
    addHtml(strformat(
              F("%s.%03u (%u.%03u sec)"),
              secondsToDayHourMinuteSecond(duration).c_str(),
              duration_millis,
              duration,
              duration_millis));
    const float duration_f = static_cast<float>(duration) + (duration_millis / 1000.0f);
    addRowLabel(F("Total Nr Samples"));
    addHtmlInt(nrSamples);
    addRowLabel(F("Avg Rate"));
    addHtmlFloat(duration_f / static_cast<float>(nrSamples - 1), 2);
    addUnit(F("sec/sample"));
    addFormSeparator(4);
    somethingAdded = true;
  }

  if (showTaskValues) {
    for (size_t i = 0; i < VARS_PER_TASK; ++i) {
      if (_plugin_stats[i] != nullptr) {
        if (_plugin_stats[i]->webformLoad_show_stats(event)) {
          somethingAdded = true;
        }
      }
    }
  }
  return somethingAdded;
}

# if FEATURE_CHART_JS
void PluginStats_array::plot_ChartJS(bool onlyJSON) const
{
  const size_t nrSamples = nrSamplesPresent();

  if (nrSamples == 0) { return; }

  // Chart Header
  {
    ChartJS_options_scales scales;
    {
      ChartJS_options_scale scaleOption(F("x"));

      if (_plugin_stats_timestamps != nullptr) {
        scaleOption.scaleType = F("time");
      }
      scales.add(scaleOption);
    }

    for (size_t i = 0; i < VARS_PER_TASK; ++i) {
      if (_plugin_stats[i] != nullptr) {
        ChartJS_options_scale scaleOption(
          _plugin_stats[i]->_ChartJS_dataset_config.displayConfig,
          _plugin_stats[i]->getLabel());
        scaleOption.axisTitle.color = _plugin_stats[i]->_ChartJS_dataset_config.color;
        scales.add(scaleOption);

        _plugin_stats[i]->_ChartJS_dataset_config.axisID = scaleOption.axisID;
      }
    }

    scales.update_Yaxis_TickCount();

    const bool enableZoom = true;

    add_ChartJS_chart_header(
      F("line"),
      F("TaskStatsChart"),
      {},
      500 + (70 * (scales.nr_Y_scales() - 1)),
      500,
      scales.toString(),
      enableZoom,
      nrSamples,
      onlyJSON);
  }


  // Add labels
  addHtml(F("\"labels\":["));

  for (size_t i = 0; i < nrSamples; ++i) {
    if (i != 0) {
      addHtml(',');
    }

    if (_plugin_stats_timestamps != nullptr) {
      struct tm ts;
      uint32_t  unix_time_frac{};
      const uint32_t unixtime_sec    = node_time.systemMicros_to_Unixtime((*_plugin_stats_timestamps)[i], unix_time_frac);
      const uint32_t local_timestamp = time_zone.toLocal(unixtime_sec);
      breakTime(local_timestamp, ts);
      addHtml('"');
      addHtml(formatDateTimeString(ts));
      addHtml(strformat(F(".%03u"), unix_time_frac_to_millis(unix_time_frac)));
      addHtml('"');
    } else {
      addHtmlInt(i);
    }
  }
  addHtml(F("],\n\"datasets\":["));


  // Data sets
  bool first = true;

  for (size_t i = 0; i < VARS_PER_TASK; ++i) {
    if (_plugin_stats[i] != nullptr) {
      if (!first) {
        addHtml(',');
      }
      first = false;
      _plugin_stats[i]->plot_ChartJS_dataset();
    }
  }
  add_ChartJS_chart_footer(onlyJSON);
}

void PluginStats_array::plot_ChartJS_scatter(
  taskVarIndex_t                values_X_axis_index,
  taskVarIndex_t                values_Y_axis_index,
  const __FlashStringHelper    *id,
  const ChartJS_title         & chartTitle,
  const ChartJS_dataset_config& datasetConfig,
  int                           width,
  int                           height,
  bool                          showAverage,
  const String                & options,
  bool                          onlyJSON) const
{
  const PluginStats *stats_X = getPluginStats(values_X_axis_index);
  const PluginStats *stats_Y = getPluginStats(values_Y_axis_index);

  if ((stats_X == nullptr) || (stats_Y == nullptr)) {
    return;
  }

  if ((stats_X->getNrSamples() < 2) || (stats_Y->getNrSamples() < 2)) {
    return;
  }

  String axisOptions;

  {
    ChartJS_options_scales scales;
    scales.add({ F("x"), stats_X->getLabel() });
    scales.add({ F("y"), stats_Y->getLabel() });
    axisOptions = scales.toString();
  }


  const size_t nrSamples  = stats_X->getNrSamples();
  const bool   enableZoom = false;

  add_ChartJS_chart_header(
    F("scatter"),
    id,
    chartTitle,
    width,
    height,
    axisOptions,
    enableZoom,
    nrSamples,
    onlyJSON);

  // Add labels, which will be shown in a tooltip when hovering with the mouse over a point.
  addHtml(F("\"labels\":["));

  for (size_t i = 0; i < nrSamples; ++i) {
    if (i != 0) {
      addHtml(',');
    }
    addHtmlInt(i);
  }
  addHtml(F("],\n\"datasets\":["));

  // Long/Lat Coordinates
  add_ChartJS_dataset_header(datasetConfig);

  // Add scatter data
  for (size_t i = 0; i < nrSamples; ++i) {
    const float valX = (*stats_X)[i];
    const float valY = (*stats_Y)[i];
    add_ChartJS_scatter_data_point(valX, valY, 6);
  }

  add_ChartJS_dataset_footer(F("\"showLine\":true"));

  if (showAverage) {
    // Add single point showing the average
    addHtml(',');
    add_ChartJS_dataset_header(
    {
      F("Average"),
      F("#0F4C5C") });

    {
      const float valX = stats_X->getSampleAvg();
      const float valY = stats_Y->getSampleAvg();
      add_ChartJS_scatter_data_point(valX, valY, 6);
    }
    add_ChartJS_dataset_footer(F("\"pointRadius\":6,\"pointHoverRadius\":10"));
  }
  add_ChartJS_chart_footer(onlyJSON);
}

# endif // if FEATURE_CHART_JS


PluginStats * PluginStats_array::getPluginStats(taskVarIndex_t taskVarIndex) const
{
  if ((taskVarIndex < VARS_PER_TASK)) {
    return _plugin_stats[taskVarIndex];
  }
  return nullptr;
}

PluginStats * PluginStats_array::getPluginStats(taskVarIndex_t taskVarIndex)
{
  if ((taskVarIndex < VARS_PER_TASK)) {
    return _plugin_stats[taskVarIndex];
  }
  return nullptr;
}

#endif // if FEATURE_PLUGIN_STATS

#include "../DataStructs/ESPEasyControllerCache_CSV_dumper.h"

#if FEATURE_RTC_CACHE_STORAGE


# include "../Globals/C016_ControllerCache.h"
# include "../Globals/MQTT.h"

# include "../../_Plugin_Helper.h"

void ESPEasyControllerCache_CSV_element::markBegin()
{
  line.clear();

  if ((endFileNr != 0) && (endPos != 0)) {
    startFileNr = endFileNr;
    startPos    = endPos;
  } else {
    startPos = ControllerCache.getPeekFilePos(startFileNr);
  }
}

void ESPEasyControllerCache_CSV_element::markEnd()
{
  endPos = ControllerCache.getPeekFilePos(endFileNr);
}

ESPEasyControllerCache_CSV_dumper::ESPEasyControllerCache_CSV_dumper(bool joinTimestamp, bool onlySetTasks, char separator, Target target)
  : _joinTimestamp(joinTimestamp), _onlySetTasks(onlySetTasks), _separator(separator), _target(target)
{
  // Initialize arrays
  const String sep_zero = String(separator);

  constexpr size_t nrTaskValues = VARS_PER_TASK * TASKS_MAX;

  for (size_t i = 0; i < nrTaskValues; ++i) {
    _csv_values[i] = sep_zero;
    _nrDecimals[i] = Cache.getTaskDeviceValueDecimals(i / VARS_PER_TASK, i % VARS_PER_TASK);
  }

  for (size_t task = 0; validTaskIndex(task); ++task) {
    _includeTask[task] = _onlySetTasks ? validPluginID(Settings.getPluginID_for_task(task)) : true;
  }

  if (_target == Target::CSV_file) {
    // First backup the peek file positions.
    _backup_peekFilePos = ControllerCache.getPeekFilePos(_backup_peekFileNr);

    // Set peek file position to first entry:
    ControllerCache.setPeekFilePos(0, 0);
  }
  C016_flush();
}

ESPEasyControllerCache_CSV_dumper::~ESPEasyControllerCache_CSV_dumper() {
  if ((_backup_peekFileNr != 0) && (_backup_peekFilePos != 0)) {
    // Restore peek file positions.
    ControllerCache.setPeekFilePos(_backup_peekFileNr, _backup_peekFilePos);
  }
}

uint32_t ESPEasyControllerCache_CSV_dumper::writeToTarget(const String& str, bool send) const {
  if (send) {
    if (_target == Target::CSV_file) {
      addHtml(str);
    } else {
      MQTTclient.write((const uint8_t*)str.c_str(), str.length());
    }
  }
  return str.length();
}

uint32_t ESPEasyControllerCache_CSV_dumper::writeToTarget(const char& c, bool send) const {
  if (send) {
    if (_target == Target::CSV_file) {
      addHtml(c);
    } else {
      MQTTclient.write(static_cast<uint8_t>(c));
    }
  }
  return 1;
}

size_t ESPEasyControllerCache_CSV_dumper::generateCSVHeader(bool send) const
{
  size_t count = 0;

  // CSV header
  String header(F("UNIX timestamp;UTC timestamp"));

  if (_joinTimestamp) {
    // Add column with nr of joined samples
    header += F(";nrJoinedSamples");
  }

  // TaskIndex and Plugin ID will be a list of numbers when lines are joined.
  header += F(";taskindex;plugin ID");

  if (_separator != ';') { header.replace(';', _separator); }
  count += writeToTarget(header, send);

  for (taskIndex_t i = 0; i < TASKS_MAX; ++i) {
    if (_includeTask[i]) {
      for (int j = 0; j < VARS_PER_TASK; ++j) {
        count += writeToTarget(_separator, send);
        count += writeToTarget(getTaskDeviceName(i), send);
        count += writeToTarget('#', send);
        count += writeToTarget(getTaskValueName(i, j), send);
      }
    }
  }

  if (_target == Target::CSV_file) {
    count += writeToTarget('\r', send);
    count += writeToTarget('\n', send);
  }

  return count;
}

bool ESPEasyControllerCache_CSV_dumper::createCSVLine()
{
  _outputLine.markBegin();


  uint32_t lastTimestamp   = 0;
  uint32_t csv_values_left = 0;

  _taskIndex_str.clear();
  _pluginID_str.clear();


  // Fetch samples from Cache Controller bin files.
  if (_element_processed) {
    if (!C016_getTaskSample(_element)) {
      return !_outputLine.line.isEmpty();
    }
    _outputLine.markEnd();
    _element_processed = false;
  }

  while (!_element_processed) {
    if (!_joinTimestamp || (lastTimestamp != static_cast<uint32_t>(_element.unixTime))) {
      // Flush the collected CSV values
      if (csv_values_left > 0) {
        flushValuesLeft(csv_values_left);
        return !_outputLine.line.isEmpty();
      }

      // Start writing a new line in the CSV file
      // Begin with the non taskvalues
      _outputLine.line += _element.unixTime;
      _outputLine.line += _separator;
      struct tm ts;
      breakTime(_element.unixTime, ts);
      _outputLine.line += formatDateTimeString(ts);

      if (!_joinTimestamp) {
        _outputLine.line += _separator;
        _outputLine.line += _element.TaskIndex;
        _outputLine.line += _separator;
        _outputLine.line += _element.pluginID.value;
      }

      lastTimestamp = static_cast<uint32_t>(_element.unixTime);
    }

    if (validTaskIndex(_element.TaskIndex)) {
      // Collect the task values for this row in the CSV
      size_t valindex = _element.TaskIndex * VARS_PER_TASK;

      for (size_t i = 0; i < VARS_PER_TASK; ++i) {
        _csv_values[valindex] = _separator;

        if (isFloatOutputDataType(_element.sensorType)) {
          const float value = _element.values.getFloat(i);

          if (essentiallyZero(value)) {
            _csv_values[valindex] += '0';
          } else {
            _csv_values[valindex] += toString(value, static_cast<unsigned int>(_nrDecimals[valindex]));
          }
        } else {
          const String formatted = _element.values.getAsString(i, _element.sensorType, _nrDecimals[valindex]);

          if (formatted.isEmpty()) {
            _csv_values[valindex] += '0';
          }
          else {
            _csv_values[valindex] += formatted;
          }
        }
        ++valindex;
      }

      if (_joinTimestamp) {
        if (!_taskIndex_str.isEmpty()) { _taskIndex_str += '/'; }

        if (!_pluginID_str.isEmpty()) { _pluginID_str += '/'; }

        _taskIndex_str += _element.TaskIndex;
        _pluginID_str  += _element.pluginID.value;
      }
      ++csv_values_left;
    }
    _outputLine.markEnd();
    _element_processed = !C016_getTaskSample(_element);
  }

  if (csv_values_left > 0) {
    flushValuesLeft(csv_values_left);
  }
  return !_outputLine.line.isEmpty();
}

void ESPEasyControllerCache_CSV_dumper::setPeekFilePos(int peekFileNr, int peekReadPos)
{
  ControllerCache.setPeekFilePos(peekFileNr, peekReadPos);
  _element_processed = true;
}

void ESPEasyControllerCache_CSV_dumper::flushValuesLeft(uint32_t csv_values_left)
{
  if (_joinTimestamp) {
    _outputLine.line += _separator;
    _outputLine.line += csv_values_left; // Add column with nr of joined samples
    _outputLine.line += _separator;
    _outputLine.line += _taskIndex_str;
    _outputLine.line += _separator;
    _outputLine.line += _pluginID_str;
  }

  constexpr size_t nrTaskValues = VARS_PER_TASK * TASKS_MAX;

  for (size_t i = 0; i < nrTaskValues; ++i) {
    if (_includeTask[i / VARS_PER_TASK]) {
      _outputLine.line += _csv_values[i];
    }
  }

  if (_target == Target::CSV_file) {
    _outputLine.line += '\r';
    _outputLine.line += '\n';
  }
}

#endif // if FEATURE_RTC_CACHE_STORAGE

#include "../DataStructs/WiFiEventData.h"

#include "../ESPEasyCore/ESPEasy_Log.h"

#include "../Globals/RTC.h"
#include "../Globals/Settings.h"
#include "../Globals/WiFi_AP_Candidates.h"

#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/Networking.h"


#define WIFI_RECONNECT_WAIT                  30000  // in milliSeconds

#define CONNECT_TIMEOUT_MAX                  4000   // in milliSeconds


#if FEATURE_USE_IPV6
#include <esp_netif.h>

// -----------------------------------------------------------------------------------------------------------------------
// ---------------------------------------------------- Private functions ------------------------------------------------
// -----------------------------------------------------------------------------------------------------------------------

esp_netif_t* get_esp_interface_netif(esp_interface_t interface);
#endif



bool WiFiEventData_t::WiFiConnectAllowed() const {
  if (WiFi.status() == WL_IDLE_STATUS) {
    // FIXME TD-er: What to do now? Set a timer?
    //return false;
    if (last_wifi_connect_attempt_moment.isSet() && 
       !last_wifi_connect_attempt_moment.timeoutReached(WIFI_PROCESS_EVENTS_TIMEOUT)) {
      return false;
    }
  }
  if (!wifiConnectAttemptNeeded) return false;
  if (intent_to_reboot) return false;
  if (wifiSetupConnect) return true;
  if (wifiConnectInProgress) {
    if (last_wifi_connect_attempt_moment.isSet() && 
       !last_wifi_connect_attempt_moment.timeoutReached(WIFI_PROCESS_EVENTS_TIMEOUT)) {
      return false;
    }
  } 
  if (lastDisconnectMoment.isSet()) {
    // TODO TD-er: Make this time more dynamic.
    if (!lastDisconnectMoment.timeoutReached(1000)) {
      return false;
    }
  }
  return true;
}

bool WiFiEventData_t::unprocessedWifiEvents() const {
  if (processedConnect && processedDisconnect && processedGotIP && processedDHCPTimeout
#if FEATURE_USE_IPV6
      && processedGotIP6
#endif
  )
  {
    return false;
  }
  if (!processedConnect) {
    if (lastConnectMoment.isSet() && lastConnectMoment.timeoutReached(WIFI_PROCESS_EVENTS_TIMEOUT)) {
      return false;
    }
  }
  if (!processedGotIP) {
    if (lastGetIPmoment.isSet() && lastGetIPmoment.timeoutReached(WIFI_PROCESS_EVENTS_TIMEOUT)) {
      return false;
    }
  }
  if (!processedDisconnect) {
    if (lastDisconnectMoment.isSet() && lastDisconnectMoment.timeoutReached(WIFI_PROCESS_EVENTS_TIMEOUT)) {
      return false;
    }
  }
  if (!processedDHCPTimeout) {
    return false;
  }
  return true;
}

void WiFiEventData_t::clearAll() {
  markWiFiTurnOn();
  lastGetScanMoment.clear();
  last_wifi_connect_attempt_moment.clear();
  timerAPstart.clear();

  lastWiFiResetMoment.setNow();
  wifi_TX_pwr = 0;
  usedChannel = 0;
}

void WiFiEventData_t::markWiFiTurnOn() {
  setWiFiDisconnected();
//  lastDisconnectMoment.clear();
  lastConnectMoment.clear();
  lastGetIPmoment.clear();
  wifi_considered_stable    = false;
  
  clear_processed_flags();
}

void WiFiEventData_t::clear_processed_flags() {
  // Mark all flags to default to prevent handling old events.
  WiFi.scanDelete();
  processedConnect          = true;
  processedDisconnect       = true;
  processedGotIP            = true;
  #if FEATURE_USE_IPV6
  processedGotIP6           = true;
  #endif
  processedDHCPTimeout      = true;
  processedConnectAPmode    = true;
  processedDisconnectAPmode = true;
  processedScanDone         = true;
  wifiConnectAttemptNeeded  = true;
  wifiConnectInProgress     = false;
  processingDisconnect.clear();
  dns0_cache = IPAddress();
  dns1_cache = IPAddress();
}

void WiFiEventData_t::markWiFiBegin() {
  markWiFiTurnOn();
  last_wifi_connect_attempt_moment.setNow();
  wifiConnectInProgress  = true;
  usedChannel = 0;
  ++wifi_connect_attempt;
  if (!timerAPstart.isSet()) {
    timerAPstart.setMillisFromNow(3 * WIFI_RECONNECT_WAIT);
  }
}


void WiFiEventData_t::setWiFiDisconnected() {
  wifiStatus            = ESPEASY_WIFI_DISCONNECTED;
  last_wifi_connect_attempt_moment.clear();
  wifiConnectInProgress = false;
#if FEATURE_ESPEASY_P2P
  updateUDPport(true);
#endif  

}

void WiFiEventData_t::setWiFiGotIP() {
  bitSet(wifiStatus, ESPEASY_WIFI_GOT_IP);
  processedGotIP = true;
  if (valid_DNS_address(WiFi.dnsIP(0))) {
    dns0_cache = WiFi.dnsIP(0);
  }
  if (valid_DNS_address(WiFi.dnsIP(1))) {
    dns1_cache = WiFi.dnsIP(1);
  }
}

void WiFiEventData_t::setWiFiConnected() {
  bitSet(wifiStatus, ESPEASY_WIFI_CONNECTED);
  processedConnect = true;
}

void WiFiEventData_t::setWiFiServicesInitialized() {
  if (!unprocessedWifiEvents() && WiFiConnected() && WiFiGotIP()) {
    # ifndef BUILD_NO_DEBUG
    addLog(LOG_LEVEL_DEBUG, F("WiFi : WiFi services initialized"));
    #endif
    bitSet(wifiStatus, ESPEASY_WIFI_SERVICES_INITIALIZED);
    wifiConnectInProgress = false;
    wifiConnectAttemptNeeded = false;
    dns0_cache = WiFi.dnsIP(0);
    dns1_cache = WiFi.dnsIP(1);

#if FEATURE_ESPEASY_P2P
    updateUDPport(false);
#endif  
  }
}

void WiFiEventData_t::markGotIP() {
  lastGetIPmoment.setNow();

  // Create the 'got IP event' so mark the wifiStatus to not have the got IP flag set
  // This also implies the services are not fully initialized.
  bitClear(wifiStatus, ESPEASY_WIFI_GOT_IP);
  bitClear(wifiStatus, ESPEASY_WIFI_SERVICES_INITIALIZED);
  processedGotIP = false;
}

#if FEATURE_USE_IPV6
  void WiFiEventData_t::markGotIPv6(const IPAddress& ip6) {
    processedGotIP6 = false;
    unprocessed_IP6 = ip6;
  }
#endif


void WiFiEventData_t::markLostIP() {
  bitClear(wifiStatus, ESPEASY_WIFI_GOT_IP);
  bitClear(wifiStatus, ESPEASY_WIFI_SERVICES_INITIALIZED);
}

void WiFiEventData_t::markDisconnect(WiFiDisconnectReason reason) {
/*
  #if defined(ESP32)
  if ((WiFi.getMode() & WIFI_MODE_STA) == 0) return;
  #else // if defined(ESP32)
  if ((WiFi.getMode() & WIFI_STA) == 0) return;
  #endif // if defined(ESP32)
*/
  lastDisconnectMoment.setNow();
  usedChannel = 0;

  if (last_wifi_connect_attempt_moment.isSet() && !lastConnectMoment.isSet()) {
    // There was an unsuccessful connection attempt
    lastConnectedDuration_us = last_wifi_connect_attempt_moment.timeDiff(lastDisconnectMoment);
  } else {
    if (last_wifi_connect_attempt_moment.isSet())
      lastConnectedDuration_us = lastConnectMoment.timeDiff(lastDisconnectMoment);
    else 
      lastConnectedDuration_us = 0;
  }
  lastDisconnectReason = reason;
  processedDisconnect  = false;
  wifiConnectInProgress = false;
}

void WiFiEventData_t::markConnected(const String& ssid, const uint8_t bssid[6], uint8_t channel) {
  usedChannel = channel;
  lastConnectMoment.setNow();
  processedConnect    = false;
  channel_changed     = RTC.lastWiFiChannel != channel;
  last_ssid           = ssid;
  bssid_changed       = false;
  auth_mode           = WiFi_AP_Candidates.getCurrent().enc_type;

  RTC.lastWiFiChannel = channel;
  for (uint8_t i = 0; i < 6; ++i) {
    if (RTC.lastBSSID[i] != bssid[i]) {
      bssid_changed    = true;
      RTC.lastBSSID[i] = bssid[i];
    }
  }
#if FEATURE_USE_IPV6
  if (Settings.EnableIPv6()) {
    WiFi.enableIPv6(true);
  }
#endif
}

void WiFiEventData_t::markConnectedAPmode(const uint8_t mac[6]) {
  lastMacConnectedAPmode = mac;
  processedConnectAPmode = false;
}

void WiFiEventData_t::markDisconnectedAPmode(const uint8_t mac[6]) {
  lastMacDisconnectedAPmode = mac;
  processedDisconnectAPmode = false;
}



String WiFiEventData_t::ESPeasyWifiStatusToString() const {
  String log;
  if (WiFiDisconnected()) {
    log = F("DISCONNECTED");
  } else {
    if (WiFiConnected()) {
      log += F("Conn. ");
    }
    if (WiFiGotIP()) {
      log += F("IP ");
    }
    if (WiFiServicesInitialized()) {
      log += F("Init");
    }
  }
  return log;
}


uint32_t WiFiEventData_t::getSuggestedTimeout(int index, uint32_t minimum_timeout) const {
  auto it = connectDurations.find(index);
  if (it == connectDurations.end()) {
    return 3 * minimum_timeout;
  }
  const uint32_t res = 3 * it->second;
  return constrain(res, minimum_timeout, CONNECT_TIMEOUT_MAX);
}
#include "../DataStructs/ControllerSettingsStruct.h"

#include "../../ESPEasy_common.h"

#include "../CustomBuild/ESPEasyLimits.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Networking.h"
#include "../Helpers/StringConverter.h"


#include <IPAddress.h>
#include <WString.h>
#include <WiFiClient.h>
#include <WiFiUdp.h>


ControllerSettingsStruct::ControllerSettingsStruct()
{
  memset(this, 0, sizeof(ControllerSettingsStruct));
  safe_strncpy(ClientID, F(CONTROLLER_DEFAULT_CLIENTID), sizeof(ClientID));
}

void ControllerSettingsStruct::reset() {
  // Need to make sure every byte between the members is also zero
  // Otherwise the checksum will fail and settings will be saved too often.
  memset(this, 0, sizeof(ControllerSettingsStruct));

  UseDNS                                        = DEFAULT_SERVER_USEDNS;
  Port                                          = DEFAULT_PORT;
  MinimalTimeBetweenMessages                    = CONTROLLER_DELAY_QUEUE_DELAY_DFLT;
  MaxQueueDepth                                 = CONTROLLER_DELAY_QUEUE_DEPTH_DFLT;
  MaxRetry                                      = CONTROLLER_DELAY_QUEUE_RETRY_DFLT;
  DeleteOldest                                  = DEFAULT_CONTROLLER_DELETE_OLDEST;
  ClientTimeout                                 = CONTROLLER_CLIENTTIMEOUT_DFLT;
  MustCheckReply                                = DEFAULT_CONTROLLER_MUST_CHECK_REPLY;
  SampleSetInitiator                            = INVALID_TASK_INDEX;
  KeepAliveTime                                 = CONTROLLER_KEEP_ALIVE_TIME_DFLT;
  VariousBits1.mqtt_cleanSession                = 0;
  VariousBits1.mqtt_not_sendLWT                 = 0;
  VariousBits1.mqtt_not_willRetain              = 0;
  VariousBits1.mqtt_uniqueMQTTclientIdReconnect = 0;
  VariousBits1.mqtt_retainFlag                  = 0;
  VariousBits1.useExtendedCredentials           = 0;
  VariousBits1.sendBinary                       = 0;
  VariousBits1.allowExpire                      = 0;
  VariousBits1.deduplicate                      = 0;
  VariousBits1.useLocalSystemTime               = 0;
  VariousBits1.TLStype                          = 0;
  VariousBits1.mqttAutoDiscovery                = 0;

  safe_strncpy(ClientID, F(CONTROLLER_DEFAULT_CLIENTID), sizeof(ClientID));
}

bool ControllerSettingsStruct::isSet() const {
  if (UseDNS) {
    return HostName[0] != 0;
  }
  return ipSet();
}

void ControllerSettingsStruct::validate() {
  if (Port > 65535) { Port = 0; }

  if ((MinimalTimeBetweenMessages < 1) ||  (MinimalTimeBetweenMessages > CONTROLLER_DELAY_QUEUE_DELAY_MAX)) {
    MinimalTimeBetweenMessages = CONTROLLER_DELAY_QUEUE_DELAY_DFLT;
  }

  if (MaxQueueDepth > CONTROLLER_DELAY_QUEUE_DEPTH_MAX) { MaxQueueDepth = CONTROLLER_DELAY_QUEUE_DEPTH_DFLT; }

  if (MaxRetry > CONTROLLER_DELAY_QUEUE_RETRY_MAX) { MaxRetry = CONTROLLER_DELAY_QUEUE_RETRY_MAX; }

  if (MaxQueueDepth == 0) { MaxQueueDepth = CONTROLLER_DELAY_QUEUE_DEPTH_DFLT; }

  if (MaxRetry == 0) { MaxRetry = CONTROLLER_DELAY_QUEUE_RETRY_DFLT; }

  if ((ClientTimeout < 10) || (ClientTimeout > CONTROLLER_CLIENTTIMEOUT_MAX)) {
    ClientTimeout = CONTROLLER_CLIENTTIMEOUT_DFLT;
  }
  ZERO_TERMINATE(HostName);
  ZERO_TERMINATE(Publish);
  ZERO_TERMINATE(Subscribe);
  ZERO_TERMINATE(MQTTLwtTopic);
  ZERO_TERMINATE(LWTMessageConnect);
  ZERO_TERMINATE(LWTMessageDisconnect);
  ZERO_TERMINATE(ClientID);
  ZERO_TERMINATE(MqttAutoDiscoveryTopic);
  ZERO_TERMINATE(MqttAutoDiscoveryConfig);
  ZERO_TERMINATE(MqttAutoDiscoveryTrigger);


  #if FEATURE_MQTT
    #if FEATURE_MQTT_TLS
    if (TLStype() == TLS_types::NoTLS) {
      if (Port == 8883) {
        Port = 1883;
        addLog(LOG_LEVEL_ERROR, F("Not using TLS, but port set to secure 8883. Use port 1883 instead"));
      }
    } else {
      if (Port == 1883) {
        Port = 8883;
        addLog(LOG_LEVEL_ERROR, F("Using TLS, but port set to insecure port 1883. Use port 8883 instead"));
      }
    }
    #else
    if (Port == 8883) {
      // No TLS support, so when switching builds, make sure it can still work.
      Port = 1883;
      addLog(LOG_LEVEL_ERROR, F("Not using TLS, but port set to secure 8883. Use port 1883 instead"));
    }
    #endif
  #endif

}

String ControllerSettingsStruct::getHost() const {
  if (UseDNS) {
    return HostName;
  }
  return formatIP(getIP());
}

void ControllerSettingsStruct::setHostname(const String& controllerhostname) {
  safe_strncpy(HostName, controllerhostname.c_str(), sizeof(HostName));
  updateIPcache();
}

bool ControllerSettingsStruct::checkHostReachable(bool quick) {
  if (!isSet()) {
    // No IP/hostname set
    return false;
  }

  if (!NetworkConnected(10)) {
    return false; // Not connected, so no use in wasting time to connect to a host.
  }
  delay(0);       // Make sure the Watchdog will not trigger a reset.

  if (quick && ipSet()) { return true; }

  if (UseDNS) {
    if (!updateIPcache()) {
      return false;
    }
  }
  return hostReachable(getIP());
}

#if FEATURE_HTTP_CLIENT
bool ControllerSettingsStruct::connectToHost(WiFiClient& client) {
  if (!checkHostReachable(true)) {
    return false; // Host not reachable
  }
  uint8_t retry     = 2;
  bool    connected = false;

  while (retry > 0 && !connected) {
    --retry;
    connected = connectClient(client, getIP(), Port, ClientTimeout);

    if (connected) { return true; }

    if (!checkHostReachable(false)) {
      return false;
    }
  }
  return false;
}

#endif // FEATURE_HTTP_CLIENT

bool ControllerSettingsStruct::beginPacket(WiFiUDP& client) {
  if (!checkHostReachable(true)) {
    return false; // Host not reachable
  }
  uint8_t retry = 2;

  while (retry > 0) {
    --retry;
    FeedSW_watchdog();

    if (client.beginPacket(getIP(), Port) == 1) {
      return true;
    }

    if (!checkHostReachable(false)) {
      return false;
    }
    delay(10);
  }
  return false;
}

String ControllerSettingsStruct::getHostPortString() const {
  String result = getHost();

  result += ':';
  result += Port;
  return result;
}

bool ControllerSettingsStruct::ipSet() const {
  for (uint8_t i = 0; i < 4; ++i) {
    if (IP[i] != 0) { return true; }
  }
  return false;
}

bool ControllerSettingsStruct::updateIPcache() {
  if (!UseDNS) {
    return true;
  }

  if (!NetworkConnected()) { return false; }
  IPAddress tmpIP;

  if (resolveHostByName(HostName, tmpIP, ClientTimeout)) {
    for (uint8_t x = 0; x < 4; x++) {
      IP[x] = tmpIP[x];
    }
    return true;
  }
  return false;
}

/*
bool ControllerSettingsStruct::mqtt_cleanSession() const
{
   return bitRead(VariousFlags, 1);
   }

   void ControllerSettingsStruct::mqtt_cleanSession(bool value)
   {
   bitWrite(VariousFlags, 1, value);
   }

   bool ControllerSettingsStruct::mqtt_sendLWT() const
   {
   return !bitRead(VariousFlags, 2);
   }

   void ControllerSettingsStruct::mqtt_sendLWT(bool value)
   {
   bitWrite(VariousFlags, 2, !value);
   }

   bool ControllerSettingsStruct::mqtt_willRetain() const
   {
   return !bitRead(VariousFlags, 3);
   }

   void ControllerSettingsStruct::mqtt_willRetain(bool value)
   {
   bitWrite(VariousFlags, 3, !value);
   }

   bool ControllerSettingsStruct::mqtt_uniqueMQTTclientIdReconnect() const
   {
   return bitRead(VariousFlags, 4);
   }

   void ControllerSettingsStruct::mqtt_uniqueMQTTclientIdReconnect(bool value)
   {
   bitWrite(VariousFlags, 4, value);
   }

   bool ControllerSettingsStruct::mqtt_retainFlag() const
   {
   return bitRead(VariousFlags, 5);
   }

   void ControllerSettingsStruct::mqtt_retainFlag(bool value)
   {
   bitWrite(VariousFlags, 5, value);
   }

   bool ControllerSettingsStruct::useExtendedCredentials() const
   {
   return bitRead(VariousFlags, 6);
   }

   void ControllerSettingsStruct::useExtendedCredentials(bool value)
   {
   bitWrite(VariousFlags, 6, value);
   }

   bool ControllerSettingsStruct::sendBinary() const
   {
   return bitRead(VariousFlags, 7);
   }

   void ControllerSettingsStruct::sendBinary(bool value)
   {
   bitWrite(VariousFlags, 7, value);
   }

   bool ControllerSettingsStruct::allowExpire() const
   {
   return bitRead(VariousFlags, 9);
   }

   void ControllerSettingsStruct::allowExpire(bool value)
   {
   bitWrite(VariousFlags, 9, value);
   }

   bool ControllerSettingsStruct::deduplicate() const
   {
   return bitRead(VariousFlags, 10);
   }

   void ControllerSettingsStruct::deduplicate(bool value)
   {
   bitWrite(VariousFlags, 10, value);
   }

   bool ControllerSettingsStruct::useLocalSystemTime() const
   {
   return bitRead(VariousFlags, 11);
   }

   void ControllerSettingsStruct::useLocalSystemTime(bool value)
   {
  bitWrite(VariousFlags, 11, value);
}
*/

#if FEATURE_MQTT_TLS || FEATURE_HTTP_TLS
String ControllerSettingsStruct::getCertificateFilename() const
{
  return getCertificateFilename(TLStype());
}

String ControllerSettingsStruct::getCertificateFilename(TLS_types tls_type) const
{
  String certFile = HostName;
  if (certFile.isEmpty()) {
    certFile = F("<HostName>");
  }

  switch (tls_type) {
    case TLS_types::NoTLS:
    case TLS_types::TLS_insecure:
      return EMPTY_STRING;
/*
    case TLS_types::TLS_PSK:
      certFile += F(".psk");
      break;
*/
/*
    case TLS_types::TLS_CA_CLI_CERT:
      certFile += F(".caclicert");
      break;
*/
    case TLS_types::TLS_CA_CERT:
      certFile += F(".cacert");
      break;
    case TLS_types::TLS_FINGERPRINT:
      certFile += F(".fp");
      break;
  }

  // Only use the last 29 bytes of the filename
  if (certFile.length() > 28) {
    certFile = certFile.substring(certFile.length() - 28);
  }
  
  return certFile;
}
#endif // #if FEATURE_MQTT_TLS || FEATURE_HTTP_TLS

#include "../DataStructs/PluginTaskData_base.h"

#include "../DataStructs/ESPEasy_EventStruct.h"

#include "../Globals/RuntimeData.h"

#include "../Helpers/StringConverter.h"

#include "../WebServer/Chart_JS.h"
#include "../WebServer/HTML_wrappers.h"


PluginTaskData_base::PluginTaskData_base()
  : _taskdata_pluginID(INVALID_PLUGIN_ID)
#if FEATURE_PLUGIN_STATS
  , _plugin_stats_array(nullptr)
#endif // if FEATURE_PLUGIN_STATS
{}


PluginTaskData_base::~PluginTaskData_base()  {
#if FEATURE_PLUGIN_STATS
  delete _plugin_stats_array;
  _plugin_stats_array = nullptr;
#endif // if FEATURE_PLUGIN_STATS
}

bool PluginTaskData_base::hasPluginStats() const {
#if FEATURE_PLUGIN_STATS

  if (_plugin_stats_array != nullptr) {
    return _plugin_stats_array->hasStats();
  }
#endif // if FEATURE_PLUGIN_STATS
  return false;
}

bool PluginTaskData_base::hasPeaks() const {
#if FEATURE_PLUGIN_STATS

  if (_plugin_stats_array != nullptr) {
    return _plugin_stats_array->hasPeaks();
  }
#endif // if FEATURE_PLUGIN_STATS
  return false;
}

size_t PluginTaskData_base::nrSamplesPresent() const {
#if FEATURE_PLUGIN_STATS

  if (_plugin_stats_array != nullptr) {
    return _plugin_stats_array->nrSamplesPresent();
  }
#endif // if FEATURE_PLUGIN_STATS
  return 0;
}

#if FEATURE_PLUGIN_STATS
void PluginTaskData_base::initPluginStats(taskIndex_t taskIndex, taskVarIndex_t taskVarIndex)
{
  if (taskVarIndex < VARS_PER_TASK) {
    if (_plugin_stats_array == nullptr) {
      constexpr unsigned size = sizeof(PluginStats_array);
      void *ptr               = special_calloc(1, size);

      if (ptr != nullptr) {
        _plugin_stats_array = new (ptr) PluginStats_array();
      }
    }

    if (_plugin_stats_array != nullptr) {
      _plugin_stats_array->initPluginStats(taskIndex, taskVarIndex);
    }
  }
}

void PluginTaskData_base::clearPluginStats(taskVarIndex_t taskVarIndex)
{
  if ((taskVarIndex < VARS_PER_TASK) && _plugin_stats_array) {
    _plugin_stats_array->clearPluginStats(taskVarIndex);

    if (!_plugin_stats_array->hasStats()) {
      delete _plugin_stats_array;
      _plugin_stats_array = nullptr;
    }
  }
}

void PluginTaskData_base::processTimeSet(const double& time_offset)
{
  if (_plugin_stats_array != nullptr) {
    _plugin_stats_array->processTimeSet(time_offset);
  }
}

#endif // if FEATURE_PLUGIN_STATS

void PluginTaskData_base::pushPluginStatsValues(struct EventStruct *event,
                                                bool                trackPeaks,
                                                bool                onlyUpdateTimestampWhenSame)
{
#if FEATURE_PLUGIN_STATS

  if (_plugin_stats_array != nullptr) {
    _plugin_stats_array->pushPluginStatsValues(event, trackPeaks, onlyUpdateTimestampWhenSame);
  }
#endif // if FEATURE_PLUGIN_STATS
}

bool PluginTaskData_base::plugin_get_config_value_base(struct EventStruct *event,
                                                       String            & string) const
{
#if FEATURE_PLUGIN_STATS

  if (_plugin_stats_array != nullptr) {
    return _plugin_stats_array->plugin_get_config_value_base(event, string);
  }
#endif // if FEATURE_PLUGIN_STATS
  return false;
}

bool PluginTaskData_base::plugin_write_base(struct EventStruct *event,
                                            const String      & string)
{
#if FEATURE_PLUGIN_STATS

  if (_plugin_stats_array != nullptr) {
    return _plugin_stats_array->plugin_write_base(event, string);
  }
#endif // if FEATURE_PLUGIN_STATS
  return false;
}

#if FEATURE_PLUGIN_STATS
bool PluginTaskData_base::webformLoad_show_stats(struct EventStruct *event) const
{
  if (_plugin_stats_array != nullptr) {
    return _plugin_stats_array->webformLoad_show_stats(event);
  }
  return false;
}

# if FEATURE_CHART_JS
void PluginTaskData_base::plot_ChartJS(bool onlyJSON) const
{
  if (_plugin_stats_array != nullptr) {
    _plugin_stats_array->plot_ChartJS(onlyJSON);
  }
}

void PluginTaskData_base::plot_ChartJS_scatter(
  taskVarIndex_t                values_X_axis_index,
  taskVarIndex_t                values_Y_axis_index,
  const __FlashStringHelper    *id,
  const ChartJS_title         & chartTitle,
  const ChartJS_dataset_config& datasetConfig,
  int                           width,
  int                           height,
  bool                          showAverage,
  const String                & options,
  bool                          onlyJSON) const
{
  if (_plugin_stats_array != nullptr) {
    _plugin_stats_array->plot_ChartJS_scatter(
      values_X_axis_index,
      values_Y_axis_index,
      id,
      chartTitle,
      datasetConfig,
      width,
      height,
      showAverage,
      options,
      onlyJSON);
  }
}

# endif // if FEATURE_CHART_JS


PluginStats * PluginTaskData_base::getPluginStats(taskVarIndex_t taskVarIndex) const
{
  if (_plugin_stats_array != nullptr) {
    return _plugin_stats_array->getPluginStats(taskVarIndex);
  }
  return nullptr;
}

PluginStats * PluginTaskData_base::getPluginStats(taskVarIndex_t taskVarIndex)
{
  if (_plugin_stats_array != nullptr) {
    return _plugin_stats_array->getPluginStats(taskVarIndex);
  }
  return nullptr;
}

#endif // if FEATURE_PLUGIN_STATS

#include "../DataStructs/Caches.h"

#include "../Globals/Device.h"
#include "../Globals/ExtraTaskSettings.h"
#include "../Globals/Settings.h"
#include "../Globals/WiFi_AP_Candidates.h"

#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/StringConverter.h"

#ifdef PLUGIN_USES_SERIAL
# include <ESPeasySerial.h>
#endif // ifdef PLUGIN_USES_SERIAL

void Caches::clearAllCaches()
{
  clearAllButTaskCaches();
  clearAllTaskCaches();
}

void Caches::clearAllButTaskCaches() {
  clearFileCaches();
  WiFi_AP_Candidates.clearCache();
  rulesHelper.closeAllFiles();
}

void Caches::clearAllTaskCaches() {
  taskIndexName.clear();
  taskIndexValueName.clear();
  extraTaskSettings_cache.clear();
  updateActiveTaskUseSerial0();
}

void Caches::clearTaskCache(taskIndex_t TaskIndex) {
  clearTaskIndexFromMaps(TaskIndex);

  auto it = extraTaskSettings_cache.find(TaskIndex);

  if (it != extraTaskSettings_cache.end()) {
    extraTaskSettings_cache.erase(it);
  }
  updateActiveTaskUseSerial0();
}

void Caches::clearFileCaches()
{
  fileExistsMap.clear();
  fileCacheClearMoment = 0;
}

bool Caches::matchChecksumExtraTaskSettings(taskIndex_t TaskIndex, const ChecksumType& checksum) const
{
  if (validTaskIndex(TaskIndex)) {
    auto it = extraTaskSettings_cache.find(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return checksum == it->second.md5checksum;
    }
  }
  return false;
}

void Caches::updateActiveTaskUseSerial0() {
  activeTaskUseSerial0 = false;
#ifdef PLUGIN_USES_SERIAL

  if (getDeviceCount() <= 0) {
    return;
  }

  // Check to see if a task is enabled and using the pins we also use for receiving commands.
  // We're now receiving only from Serial0, so check if an enabled task is also using it.
  for (taskIndex_t task = 0; task < TASKS_MAX; ++task)
  {
    const deviceIndex_t DeviceIndex = getDeviceIndex_from_TaskIndex(task);

    if (Settings.TaskDeviceEnabled[task] && validDeviceIndex(DeviceIndex)) {
      if ((Device[DeviceIndex].Type == DEVICE_TYPE_SERIAL) ||
          (Device[DeviceIndex].Type == DEVICE_TYPE_SERIAL_PLUS1)) {
        const ESPEasySerialPort port = ESPeasySerialType::getSerialType(
          static_cast<ESPEasySerialPort>(Settings.TaskDevicePort[task]),
          Settings.TaskDevicePin1[task],
          Settings.TaskDevicePin2[task]);

        // FIXME TD-er: Must not check for conflict with serial0, but for conflict with ESPEasy_Console.
        # ifdef ESP32

        if (port == ESPEasySerialPort::serial0)
        {
          activeTaskUseSerial0 = true;
        }
        # endif // ifdef ESP32
        # ifdef ESP8266

        if ((port == ESPEasySerialPort::serial0_swap) ||
            (port == ESPEasySerialPort::serial0))
        {
          activeTaskUseSerial0 = true;
        }
        # endif // ifdef ESP8266
      }
    }
  }
#endif // ifdef PLUGIN_USES_SERIAL
}

uint8_t Caches::getTaskDeviceValueDecimals(taskIndex_t TaskIndex, uint8_t rel_index)
{
  if (validTaskIndex(TaskIndex) && (rel_index < VARS_PER_TASK)) {
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.decimals[rel_index];
    }
  }
  return 0;
}

String Caches::getTaskDeviceName(taskIndex_t TaskIndex)
{
  if (validTaskIndex(TaskIndex)) {
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.TaskDeviceName;
    }
  }
  return EMPTY_STRING;
}

String Caches::getTaskDeviceValueName(taskIndex_t TaskIndex, uint8_t rel_index)
{
  if (validTaskIndex(TaskIndex) && (rel_index < VARS_PER_TASK)) {
  #ifdef ESP8266

    // ESP8266 uses SPIFFS, which is fast enough to read the task settings
    // Also RAM is limited, so don't waste it on caching where it is not needed.
    LoadTaskSettings(TaskIndex);
    return ExtraTaskSettings.TaskDeviceValueNames[rel_index];
  #endif // ifdef ESP8266
  #ifdef ESP32

    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.TaskDeviceValueNames[rel_index];
    }
    #endif // ifdef ESP32
  }

  return EMPTY_STRING;
}

bool Caches::hasFormula(taskIndex_t TaskIndex, uint8_t rel_index)
{
  if (validTaskIndex(TaskIndex) && (rel_index < VARS_PER_TASK)) {
    // Just a quick test to see if we do have a formula present.
    // Task Formula are not used very often, so we will probably almost always have to return an empty string.
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return bitRead(it->second.hasFormula, 2 * rel_index);
    }
  }
  return false;
}

bool Caches::hasFormula(taskIndex_t TaskIndex)
{
  if (validTaskIndex(TaskIndex)) {
    // Just a quick test to see if we do have a formula present.
    // Task Formula are not used very often, so we will probably almost always have to return an empty string.
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.hasFormula != 0;
    }
  }
  return false;
}

bool Caches::hasFormula_with_prevValue(taskIndex_t TaskIndex, uint8_t rel_index)
{
  if (validTaskIndex(TaskIndex) && (rel_index < VARS_PER_TASK)) {
    // Just a quick test to see if we do have a formula present which requires %pvalue%.
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return bitRead(it->second.hasFormula, 2 * rel_index + 1);
    }
  }
  return false;
}

String Caches::getTaskDeviceFormula(taskIndex_t TaskIndex, uint8_t rel_index)
{
  if ((rel_index < VARS_PER_TASK) && hasFormula(TaskIndex, rel_index)) {
    #ifdef ESP32
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.TaskDeviceFormula[rel_index];
    }
    #else // ifdef ESP32
    LoadTaskSettings(TaskIndex);
    return ExtraTaskSettings.TaskDeviceFormula[rel_index];
    #endif // ifdef ESP32
  }
  return EMPTY_STRING;
}

long Caches::getTaskDevicePluginConfigLong(taskIndex_t TaskIndex, uint8_t rel_index)
{
  if (validTaskIndex(TaskIndex) && (rel_index < PLUGIN_EXTRACONFIGVAR_MAX)) {
    #ifdef ESP8266
    LoadTaskSettings(TaskIndex);
    return ExtraTaskSettings.TaskDevicePluginConfigLong[rel_index];
    #endif // ifdef ESP8266
    #ifdef ESP32
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      if (bitRead(it->second.TaskDevicePluginConfigLong_index_used, rel_index) == 0)
      {
        return 0;
      }
      auto it_long = it->second.long_values.find(ExtraTaskSettings_cache_t::make_Long_ValuesIndex(rel_index));

      if (it_long != it->second.long_values.end()) {
        return it_long->second;
      }

      // Should not get here, since the long_values map should be in sync with the index_used bitmap.
    }
    #endif // ifdef ESP32
  }

  return 0;
}

int16_t Caches::getTaskDevicePluginConfig(taskIndex_t TaskIndex, uint8_t rel_index)
{
  if (validTaskIndex(TaskIndex) && (rel_index < PLUGIN_EXTRACONFIGVAR_MAX)) {
    #ifdef ESP8266
    LoadTaskSettings(TaskIndex);
    return ExtraTaskSettings.TaskDevicePluginConfig[rel_index];
    #endif // ifdef ESP8266
    #ifdef ESP32
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      if (bitRead(it->second.TaskDevicePluginConfig_index_used, rel_index) == 0)
      {
        return 0;
      }
      auto it_long = it->second.long_values.find(ExtraTaskSettings_cache_t::make_Uint16_ValuesIndex(rel_index));

      if (it_long != it->second.long_values.end()) {
        return it_long->second;
      }

      // Should not get here, since the long_values map should be in sync with the index_used bitmap.
    }
    #endif // ifdef ESP32
  }

  return 0;
}

#if FEATURE_PLUGIN_STATS
bool Caches::enabledPluginStats(taskIndex_t TaskIndex, uint8_t rel_index)
{
  if (validTaskIndex(TaskIndex) && (rel_index < VARS_PER_TASK)) {
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.pluginStatsConfig[rel_index].isEnabled();
    }
  }
  return false;
}

PluginStats_Config_t Caches::getPluginStatsConfig(taskIndex_t TaskIndex, taskVarIndex_t taskVarIndex)
{
  if (validTaskIndex(TaskIndex) && (taskVarIndex < VARS_PER_TASK)) {
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.pluginStatsConfig[taskVarIndex];
    }
  }
  return PluginStats_Config_t(0);
}

#endif // if FEATURE_PLUGIN_STATS

#if FEATURE_TASKVALUE_UNIT_OF_MEASURE
uint8_t Caches::getTaskVarUnitOfMeasure(taskIndex_t    TaskIndex,
                                        taskVarIndex_t taskVarIndex) {
  if (validTaskIndex(TaskIndex) && (validTaskVarIndex(taskVarIndex))) {
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.unitOfMeasure[taskVarIndex];
    }
  }
  return 0;
}

#endif // if FEATURE_TASKVALUE_UNIT_OF_MEASURE

#if FEATURE_CUSTOM_TASKVAR_VTYPE
uint8_t Caches::getTaskVarCustomVType(taskIndex_t    TaskIndex,
                                      taskVarIndex_t taskVarIndex) {
  if (validTaskIndex(TaskIndex) && (validTaskVarIndex(taskVarIndex))) {
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.customVType[taskVarIndex];
    }
  }
  return 0;
}

#endif // if FEATURE_CUSTOM_TASKVAR_VTYPE

#if FEATURE_MQTT_STATE_CLASS
uint8_t Caches::getTaskVarStateClass(taskIndex_t    TaskIndex,
                                     taskVarIndex_t taskVarIndex) {
  if (validTaskIndex(TaskIndex) && (validTaskVarIndex(taskVarIndex))) {
    auto it = getExtraTaskSettings(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      return it->second.mqttStateClass[taskVarIndex];
    }
  }
  return 0;
}

#endif // if FEATURE_MQTT_STATE_CLASS

void Caches::updateExtraTaskSettingsCache()
{
  const taskIndex_t TaskIndex = ExtraTaskSettings.TaskIndex;

  if (validTaskIndex(TaskIndex)) {
    ExtraTaskSettings_cache_t tmp;

    auto it = extraTaskSettings_cache.find(TaskIndex);

    if (it != extraTaskSettings_cache.end()) {
      // We need to keep the original checksum, from when loaded from storage
      tmp.md5checksum                = it->second.md5checksum;
      tmp.defaultTaskDeviceValueName = it->second.defaultTaskDeviceValueName;

      // Now clear it so we can create a fresh copy.
      extraTaskSettings_cache.erase(it);
      clearTaskIndexFromMaps(TaskIndex);
    }
    move_special(tmp.TaskDeviceName, String(ExtraTaskSettings.TaskDeviceName));

    for (size_t i = 0; i < VARS_PER_TASK; ++i) {
      #ifdef ESP32
      move_special(tmp.TaskDeviceValueNames[i], String(ExtraTaskSettings.TaskDeviceValueNames[i]));
      #endif // ifdef ESP32

      if (ExtraTaskSettings.TaskDeviceFormula[i][0] != 0) {
        bitSet(tmp.hasFormula, 2 * i);
        String formula(ExtraTaskSettings.TaskDeviceFormula[i]);

        if (formula.indexOf(F("%pvalue%")) != -1) {
          bitSet(tmp.hasFormula, 2 * i + 1);
        }


        #ifdef ESP32
        tmp.TaskDeviceFormula[i] = std::move(formula);
        #endif // ifdef ESP32
      }
      tmp.decimals[i] = ExtraTaskSettings.TaskDeviceValueDecimals[i];
      #if FEATURE_PLUGIN_STATS

      tmp.pluginStatsConfig[i] = ExtraTaskSettings.getPluginStatsConfig(i);
      #endif // if FEATURE_PLUGIN_STATS

      #if FEATURE_TASKVALUE_UNIT_OF_MEASURE
      tmp.unitOfMeasure[i] = ExtraTaskSettings.getTaskVarUnitOfMeasure(i);
      #endif // if FEATURE_TASKVALUE_UNIT_OF_MEASURE
      #if FEATURE_CUSTOM_TASKVAR_VTYPE
      tmp.customVType[i] = ExtraTaskSettings.getTaskVarCustomVType(i);
      #endif // if FEATURE_CUSTOM_TASKVAR_VTYPE
      #if FEATURE_MQTT_STATE_CLASS
      tmp.mqttStateClass[i] = ExtraTaskSettings.getTaskVarStateClass(i);
      #endif // if FEATURE_MQTT_STATE_CLASS
    }
    #ifdef ESP32
    tmp.TaskDevicePluginConfigLong_index_used = 0;
    tmp.TaskDevicePluginConfig_index_used     = 0;
    tmp.long_values.clear();

    for (size_t i = 0; i < PLUGIN_EXTRACONFIGVAR_MAX; ++i) {
      if (ExtraTaskSettings.TaskDevicePluginConfigLong[i] != 0) {
        bitSet(tmp.TaskDevicePluginConfigLong_index_used, i);
        tmp.long_values[ExtraTaskSettings_cache_t::make_Long_ValuesIndex(i)] = ExtraTaskSettings.TaskDevicePluginConfigLong[i];
      }

      if (ExtraTaskSettings.TaskDevicePluginConfig[i] != 0) {
        bitSet(tmp.TaskDevicePluginConfig_index_used, i);
        tmp.long_values[ExtraTaskSettings_cache_t::make_Uint16_ValuesIndex(i)] = ExtraTaskSettings.TaskDevicePluginConfig[i];
      }
    }
    #endif // ifdef ESP32

    extraTaskSettings_cache.emplace(std::make_pair(TaskIndex, std::move(tmp)));
  }
}

void Caches::updateExtraTaskSettingsCache_afterLoad_Save()
{
  if (!validTaskIndex(ExtraTaskSettings.TaskIndex)) {
    return;
  }

  // First update all other values
  updateExtraTaskSettingsCache();

  // Iterator has changed
  auto it = extraTaskSettings_cache.find(ExtraTaskSettings.TaskIndex);

  if (it != extraTaskSettings_cache.end()) {
    for (size_t i = 0; i < VARS_PER_TASK; ++i) {
      bitWrite(it->second.defaultTaskDeviceValueName, i, ExtraTaskSettings.isDefaultTaskVarName(i));
    }
    it->second.md5checksum = ExtraTaskSettings.computeChecksum();
  }
}

ExtraTaskSettingsMap::const_iterator Caches::getExtraTaskSettings(taskIndex_t TaskIndex)
{
  if (validTaskIndex(TaskIndex)) {
    auto it = extraTaskSettings_cache.find(TaskIndex);

    if (it == extraTaskSettings_cache.end()) {
      ExtraTaskSettings.clear(); // Force reload so the cache is filled
      LoadTaskSettings(TaskIndex);
      it = extraTaskSettings_cache.find(TaskIndex);
    }
    return it;
  }
  return extraTaskSettings_cache.end();
}

void Caches::clearTaskIndexFromMaps(taskIndex_t TaskIndex)
{
  {
    auto it = taskIndexName.begin();

    for (; it != taskIndexName.end();) {
      if (it->second == TaskIndex) {
        it = taskIndexName.erase(it);
      } else {
        ++it;
      }
    }
  }
  {
    const String searchstr = String('#') + TaskIndex;
    auto it                = taskIndexValueName.begin();

    for (; it != taskIndexValueName.end();) {
      if (it->first.endsWith(searchstr)) {
        it = taskIndexValueName.erase(it);
      } else {
        ++it;
      }
    }
  }
}

  #ifdef ESP32
bool Caches::getControllerSettings(controllerIndex_t index,  ControllerSettingsStruct& ControllerSettings) const
{
  if (!validControllerIndex(index)) { return false; }

  auto it = controllerSetings_cache.find(index);

  if (it == controllerSetings_cache.end()) {
    return false;
  }

  ControllerSettings = it->second;
  return true;
}

void Caches::setControllerSettings(controllerIndex_t index, const ControllerSettingsStruct& ControllerSettings)
{
  auto it = controllerSetings_cache.find(index);

  if (it != controllerSetings_cache.end()) {
    if (it->second.computeChecksum() == ControllerSettings.computeChecksum()) {
      return;
    }
    controllerSetings_cache.erase(it);
  }
  controllerSetings_cache.emplace(index, ControllerSettings);
}

void Caches::clearControllerSettings(controllerIndex_t index)
{
  auto it = controllerSetings_cache.find(index);

  if (it != controllerSetings_cache.end()) {
    controllerSetings_cache.erase(it);
  }
}

  #endif // ifdef ESP32

#include "../DataStructs/Scheduler_PluginDeviceTimerID.h"

#include "../Globals/Plugins.h"
#include "../Helpers/_Plugin_init.h"

PluginDeviceTimerID::PluginDeviceTimerID(pluginID_t pluginID, int Par1) :
  SchedulerTimerID(SchedulerTimerType_e::PLUGIN_DEVICETIMER_IN_e)
{
  // plugin number and par1 form a unique key that can be used to restart a timer
  // Use deviceIndex instead of pluginID, since the deviceIndex uses less bits.
  const deviceIndex_t deviceIndex = getDeviceIndex(pluginID);

  if (validDeviceIndex(deviceIndex)) {
    // FIXME TD-er: Must add a constexpr function with nr of included plugins.
    const unsigned nrBits = getNrBitsDeviceIndex();
    const unsigned mask   = MASK_BITS(nrBits);
    setId((deviceIndex.value & mask) | (Par1 << nrBits));
  }
}

deviceIndex_t PluginDeviceTimerID::get_deviceIndex() const {
  const unsigned nrBits = getNrBitsDeviceIndex();
  const unsigned mask   = MASK_BITS(nrBits);

  return deviceIndex_t::toDeviceIndex(getId() & mask);
}

#ifndef BUILD_NO_DEBUG
String PluginDeviceTimerID::decode() const
{
  const deviceIndex_t deviceIndex = get_deviceIndex();

  if (validDeviceIndex(deviceIndex)) {
    return getPluginNameFromDeviceIndex(deviceIndex);
  }
  return String(getId());
}

#endif // ifndef BUILD_NO_DEBUG

#include "../DataStructs/ESPEasyControllerCache.h"

#if FEATURE_RTC_CACHE_STORAGE

#include "../Helpers/Memory.h"

ControllerCache_struct::~ControllerCache_struct() {
  if (_RTC_cache_handler != nullptr) {
    delete _RTC_cache_handler;
    _RTC_cache_handler = nullptr;
  }
}

// Write a single sample set to the buffer
bool ControllerCache_struct::write(const uint8_t *data, unsigned int size) {
  if (_RTC_cache_handler == nullptr) {
    return false;
  }
  return _RTC_cache_handler->write(data, size);
}

// Read a single sample set, either from file or buffer.
// May delete a file if it is all read and not written to.
bool ControllerCache_struct::read(uint8_t *data, unsigned int size) {
  return true;
}

// Dump whatever is in the buffer to the filesystem
bool ControllerCache_struct::flush() {
  if (_RTC_cache_handler == nullptr) {
    return false;
  }
  return _RTC_cache_handler->flush();
}

void ControllerCache_struct::init() {
  if (_RTC_cache_handler == nullptr) {
    constexpr unsigned size = sizeof(RTC_cache_handler_struct);
    void *ptr               = special_calloc(1, size);

    if (ptr != nullptr) {
      _RTC_cache_handler = new (ptr) RTC_cache_handler_struct;
    }
    if (_RTC_cache_handler != nullptr) {
      _RTC_cache_handler->init();
    }
  }
}

bool ControllerCache_struct::isInitialized() const {
  return _RTC_cache_handler != nullptr;
}

// Clear all caches
void ControllerCache_struct::clearCache() {}

bool ControllerCache_struct::deleteOldestCacheBlock() {
  if (_RTC_cache_handler != nullptr) {
    return _RTC_cache_handler->deleteOldestCacheBlock();
  }
  return false;
}

void ControllerCache_struct::closeOpenFiles() {
  if (_RTC_cache_handler != nullptr) {
    _RTC_cache_handler->closeOpenFiles();
  }
}

bool ControllerCache_struct::deleteAllCacheBlocks() {
  if (_RTC_cache_handler != nullptr) {
    return _RTC_cache_handler->deleteAllCacheBlocks();
  }
  return false;
}

bool ControllerCache_struct::deleteCacheBlock(int fileNr) {
  if (_RTC_cache_handler != nullptr) {
    return _RTC_cache_handler->deleteCacheBlock(fileNr);
  }
  return false;
}

void ControllerCache_struct::resetpeek() {
  if (_RTC_cache_handler != nullptr) {
    _RTC_cache_handler->resetpeek();
  }
}

bool ControllerCache_struct::peekDataAvailable() const {
  if (_RTC_cache_handler == nullptr) {
    return false;
  }
  return _RTC_cache_handler->peekDataAvailable();
}

int  ControllerCache_struct::getPeekFilePos(int& peekFileNr) const {
  if (_RTC_cache_handler != nullptr) {
    return _RTC_cache_handler->getPeekFilePos(peekFileNr);
  }
  return -1;
}

int  ControllerCache_struct::getPeekFileSize(int peekFileNr) const {
  if (_RTC_cache_handler != nullptr) {
    return _RTC_cache_handler->getPeekFileSize(peekFileNr);
  }
  return -1;
}

void ControllerCache_struct::setPeekFilePos(int peekFileNr, int peekReadPos) {
  if (_RTC_cache_handler != nullptr) {
    _RTC_cache_handler->setPeekFilePos(peekFileNr, peekReadPos);
  }
}

// Read data without marking it as being read.
bool ControllerCache_struct::peek(uint8_t *data, unsigned int size) const {
  if (_RTC_cache_handler == nullptr) {
    return false;
  }
  return _RTC_cache_handler->peek(data, size);
}

String ControllerCache_struct::getNextCacheFileName(int& fileNr, bool& islast) {
  if (_RTC_cache_handler == nullptr) {
    fileNr = -1;
    islast = true;
    return EMPTY_STRING;
  }
  return _RTC_cache_handler->getNextCacheFileName(fileNr, islast);
}

#endif
#include "../DataStructs/RTC_cache_handler_struct.h"

#if FEATURE_RTC_CACHE_STORAGE

#include "../../ESPEasy_common.h"
#include "../DataStructs/RTCStruct.h"
#include "../Helpers/CRC_functions.h"
#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/StringConverter.h"

#include "../ESPEasyCore/ESPEasy_backgroundtasks.h"
#include "../ESPEasyCore/ESPEasy_Log.h"

#ifdef ESP8266
# include <user_interface.h>
#endif // ifdef ESP8266

#ifdef ESP32
  # include <soc/rtc.h>

// For ESP32 the RTC mapped structure may not be a member of an object,
// but must be declared 'static'
// This also means we can only have a single instance of this
// RTC_cache_handler_struct.
RTC_NOINIT_ATTR RTC_cache_struct RTC_cache;
RTC_NOINIT_ATTR uint8_t RTC_cache_data[RTC_CACHE_DATA_SIZE];
#endif // ifdef ESP32


/********************************************************************************************\
   RTC located cache
 \*********************************************************************************************/
RTC_cache_handler_struct::RTC_cache_handler_struct() {}

bool RTC_cache_handler_struct::init() {
  bool success = loadMetaData() && loadData();

  if (!success) {
      #ifdef RTC_STRUCT_DEBUG
    addLog(LOG_LEVEL_INFO, F("RTC  : Error reading cache data"));
      #endif // ifdef RTC_STRUCT_DEBUG
    RTC_cache.init();
    flush();
  } else {
      #ifdef RTC_STRUCT_DEBUG
    rtc_debug_log(F("Read from RTC cache"), RTC_cache.writePos);
      #endif // ifdef RTC_STRUCT_DEBUG
  }
  updateRTC_filenameCounters();
  return success;
}

unsigned int RTC_cache_handler_struct::getFreeSpace() {
  if (RTC_cache.writePos >= RTC_CACHE_DATA_SIZE) {
    return 0;
  }
  return RTC_CACHE_DATA_SIZE - RTC_cache.writePos;
}

void RTC_cache_handler_struct::resetpeek() {
  if (fp) {
    fp.close();
  }
  _peekfilenr  = 0;
  _peekreadpos = 0;
}

bool RTC_cache_handler_struct::peekDataAvailable() const {
  if (fp) {
    if ((_peekreadpos + 1) < fp.size()) { return true; }
  }
  if (_peekfilenr < RTC_cache.writeFileNr) {
    return true;
  }

  if (_peekfilenr == RTC_cache.writeFileNr) {
    if (fw) {
      constexpr size_t errorcode = (size_t)-1;
      size_t pos = fw.position();
      if (pos == errorcode) {
        pos = 0;
      }
      return ((_peekreadpos + 1) < pos);
    }
//    return true;
  }
  return false;
}

int RTC_cache_handler_struct::getPeekFilePos(int& peekFileNr) {
  peekFileNr = _peekfilenr;
  constexpr size_t errorcode = (size_t)-1;
  if (fp) {
    size_t pos = fp.position();
    if (pos == errorcode) {
      _peekreadpos = 0;
      return -1;
    }      
    _peekreadpos = pos;
  }

  if (_peekreadpos == errorcode)
    return -1;
  return _peekreadpos;
}

int RTC_cache_handler_struct::getPeekFileSize(int peekFileNr) const {
  if (fp) {
    return fp.size();
  }
  return -1;
}

void RTC_cache_handler_struct::setPeekFilePos(int newPeekFileNr, int newPeekReadPos) {
  validateFilePos(newPeekFileNr, newPeekReadPos);

  if (fp) {
    constexpr size_t errorcode = (size_t)-1;
    size_t pos = fp.position();
    if (pos == errorcode) {
      pos = 0;
    }

    if (newPeekReadPos < static_cast<int>(pos)) {
      _peekfilenr = newPeekFileNr;
      _peekreadpos = newPeekReadPos;
      fp.close();
    } else
    if (newPeekReadPos >= static_cast<int>(fp.size()) && static_cast<int>(_peekfilenr) == newPeekFileNr) {
      newPeekFileNr++;
      newPeekReadPos = 0;
      validateFilePos(newPeekFileNr, newPeekReadPos);
    }
    if (static_cast<int>(_peekfilenr) != newPeekFileNr) {
      // Not the same file
      fp.close();
      _peekfilenr = newPeekFileNr;
    }
  }


  if (!fp) {
    String fname = createCacheFilename(newPeekFileNr);

    if (fname.isEmpty()) { return; }

    fp = tryOpenFile(fname, "r");
  }

  if (fp) {
    _peekfilenr = newPeekFileNr;

    if (newPeekReadPos > 0) {
      const int fileSize = fp.size();

      if (fileSize <= newPeekReadPos) {
        fp.seek(0, fs::SeekEnd);

        constexpr size_t errorcode = (size_t)-1;
        size_t pos = fp.position();
        if (pos == errorcode) {
          pos = 0;
        }

        _peekreadpos = pos;
        return;
      }

      if (fp.seek(newPeekReadPos)) {
        _peekreadpos = newPeekReadPos;
      }
    } else {
      _peekreadpos = 0;
    }
    return;
  }

  _peekreadpos = 0;
  _peekfilenr  = 0;
}

bool RTC_cache_handler_struct::peek(uint8_t *data, unsigned int size) {
  if (!fp) {
    if (_peekfilenr == 0) {
      setPeekFilePos(0, 0);
    } else {
      if (!peekDataAvailable()) {
        return false;
      }
      setPeekFilePos(_peekfilenr, _peekreadpos);
    }
  }

  if (!peekDataAvailable()) { return false; }

  if (!fp) { return false; }

  const size_t bytesRead = fp.read(data, size);

  constexpr size_t errorcode = (size_t)-1;
  size_t pos = fp.position();
  if (pos == errorcode) {
    pos = 0;
  }

  _peekreadpos = pos;

  if (_peekreadpos >= fp.size()) {
    if (_peekfilenr < RTC_cache.writeFileNr) {
      fp.close();
      _peekreadpos = 0;
      ++_peekfilenr;
    }
  }

  return bytesRead == size;
}

// Write a single sample set to the buffer
bool RTC_cache_handler_struct::write(const uint8_t *data, unsigned int size) {
    #ifdef RTC_STRUCT_DEBUG
  rtc_debug_log(F("write RTC cache data"), size);
    #endif // ifdef RTC_STRUCT_DEBUG

  if (getFreeSpace() < size) {
    if (!flush()) {
      return false;
    }
  }

  // First store it in the buffer
  for (unsigned int i = 0; i < size; ++i) {
    RTC_cache_data[RTC_cache.writePos] = data[i];
    ++RTC_cache.writePos;
  }

  // Now store the updated part of the buffer to the RTC memory.
  // Pad some extra bytes around it to allow sample sizes not multiple of 4 bytes.
  int startOffset = RTC_cache.writePos - size;
  startOffset -= startOffset % 4;

  if (startOffset < 0) {
    startOffset = 0;
  }
  int nrBytes = RTC_cache.writePos - startOffset;

  if (nrBytes % 4 != 0) {
    nrBytes -= nrBytes % 4;
    nrBytes += 4;
  }

  if ((nrBytes + startOffset) >  RTC_CACHE_DATA_SIZE) {
    // Can this happen?
    nrBytes = RTC_CACHE_DATA_SIZE - startOffset;
  }
  return saveRTCcache(startOffset, nrBytes);
}

// Mark all content as being processed and empty buffer.
bool RTC_cache_handler_struct::flush() {
  if (RTC_cache.writePos > 0) {
    if (prepareFileForWrite()) {
      #ifdef RTC_STRUCT_DEBUG
      size_t filesize = fw.size();
      #endif // ifdef RTC_STRUCT_DEBUG

      // Make sure the read and peek file handles cannot be used on possibly deleted files.
      if (fr) {
        fr.close();
      }

      if (fp) {
        fp.close();
      }

      int bytesWritten = fw.write(&RTC_cache_data[0], RTC_cache.writePos);

      delay(0);
      fw.flush();
        #ifdef RTC_STRUCT_DEBUG
      addLog(LOG_LEVEL_INFO, F("RTC  : flush RTC cache"));
        #endif // ifdef RTC_STRUCT_DEBUG


      if ((bytesWritten < RTC_cache.writePos) /*|| (fw.size() == filesize)*/) {
          #ifdef RTC_STRUCT_DEBUG

        if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
          String log = F("RTC  : error writing file. Size before: ");
          log += filesize;
          log += F(" after: ");
          log += fw.size();
          log += F(" written: ");
          log += bytesWritten;
          addLogMove(LOG_LEVEL_ERROR, log);
        }
          #endif // ifdef RTC_STRUCT_DEBUG
        fw.close();

        if (!GarbageCollection()) {
          // Garbage collection was not able to remove anything
          writeError = true;
        }
        return false;
      }
      initRTCcache_data();
      clearRTCcacheData();
      saveRTCcache();
      return true;
    }
  }
  return false;
}

// Return usable filename for reading.
// Will be empty if there is no file to process.
String RTC_cache_handler_struct::getReadCacheFileName(int& readPos) {
  initRTCcache_data();

  for (int i = 0; i < 2; ++i) {
    String fname = createCacheFilename(RTC_cache.readFileNr);

    if (fileExists(fname)) {
      if (i != 0) {
        // First attempt failed, so stored read position is not valid
        RTC_cache.readPos = 0;
      }
      readPos = RTC_cache.readPos;
      return fname;
    }

    if (i == 0) {
      updateRTC_filenameCounters();
    }
  }

  // No file found
  RTC_cache.readPos = 0;
  readPos           = RTC_cache.readPos;
  return EMPTY_STRING;
}

String RTC_cache_handler_struct::getNextCacheFileName(int& fileNr, bool& islast)  {
  int filepos = 0;
  validateFilePos(fileNr, filepos);
  islast = fileNr >= RTC_cache.writeFileNr;
  const String fname = createCacheFilename(fileNr);

  if (fileExists(fname)) {
    return fname;
  }
  islast = true;
  return EMPTY_STRING;
}

bool RTC_cache_handler_struct::deleteOldestCacheBlock() {
  if (updateRTC_filenameCounters()) {
    return deleteCacheBlock(RTC_cache.readFileNr);
  }
#ifdef RTC_STRUCT_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLog(LOG_LEVEL_INFO, F("RTC  : No Cache files found"));
  }
#endif // ifdef RTC_STRUCT_DEBUG
  return false;
}

void RTC_cache_handler_struct::closeOpenFiles()
{
  if (fr) {
    fr.close();
  }

  if (fp) {
    fp.close();
  }
  if (fw) {
    fw.close();
  }
}

bool RTC_cache_handler_struct::deleteAllCacheBlocks()
{
  if (updateRTC_filenameCounters()) {

    if (RTC_cache.readFileNr < RTC_cache.writeFileNr) {
      bool fileDeleted = false;
      int  count       = 0;
      closeOpenFiles();

      for (int fileNr = RTC_cache.readFileNr; count < 25 && fileNr < RTC_cache.writeFileNr; ++fileNr)
      {
        String fname = createCacheFilename(fileNr);

        if (tryDeleteFile(fname)) {
          ++count;
          fileDeleted = true;
          #ifdef RTC_STRUCT_DEBUG

          if (loglevelActiveFor(LOG_LEVEL_INFO)) {
            addLogMove(LOG_LEVEL_INFO, concat(F("RTC  : Removed file from FS: "), fname));
          }
          #endif // ifdef RTC_STRUCT_DEBUG
          backgroundtasks();
        }
      }

      if (fileDeleted) {
        writeError = false;
        updateRTC_filenameCounters();
        return true;
      }
    }
  }
  return false;
}

bool RTC_cache_handler_struct::deleteCacheBlock(int fileNr)
{
  bool fileDeleted = false;
  if (updateRTC_filenameCounters()) {
    while ((RTC_cache.readFileNr < RTC_cache.writeFileNr) && (RTC_cache.readFileNr <= fileNr)) {
      // read and write file nr are not the same file, remove the read file nr.
      String fname = createCacheFilename(RTC_cache.readFileNr);

      writeError = false;

      // Make sure the read and peek file handles cannot be used on possibly deleted files.
      if (fr) {
        fr.close();
      }

      if (fp) {
        fp.close();
      }

      if (tryDeleteFile(fname)) {
        fileDeleted = true;
        #ifdef RTC_STRUCT_DEBUG
        if (loglevelActiveFor(LOG_LEVEL_INFO)) {
          addLogMove(LOG_LEVEL_INFO, concat(F("RTC  : Removed file from FS: "), fname));
        }
        #endif // ifdef RTC_STRUCT_DEBUG

        updateRTC_filenameCounters();
        backgroundtasks();
      }
    }
  }
  return fileDeleted;
}

bool RTC_cache_handler_struct::loadMetaData()
{
  // No need to load on ESP32, as the data is already allocated to the RTC memory by the compiler

  #ifdef ESP8266

  if (!system_rtc_mem_read(RTC_BASE_CACHE, reinterpret_cast<uint8_t *>(&RTC_cache), sizeof(RTC_cache))) {
    return false;
  }
  #endif // ifdef ESP8266

  return RTC_cache.checksumMetadata == calc_CRC32(reinterpret_cast<const uint8_t *>(&RTC_cache), sizeof(RTC_cache) - sizeof(uint32_t));
}

bool RTC_cache_handler_struct::loadData()
{
  initRTCcache_data();

  // No need to load on ESP32, as the data is already allocated to the RTC memory by the compiler
  #ifdef ESP8266

  if (!system_rtc_mem_read(RTC_BASE_CACHE + (sizeof(RTC_cache) / 4), reinterpret_cast<uint8_t *>(&RTC_cache_data[0]), RTC_CACHE_DATA_SIZE)) {
    return false;
  }
  #endif // ifdef ESP8266

  if (RTC_cache.checksumData != getDataChecksum()) {
        #ifdef RTC_STRUCT_DEBUG
    addLog(LOG_LEVEL_ERROR, F("RTC  : Checksum error reading RTC cache data"));
        #endif // ifdef RTC_STRUCT_DEBUG
    return false;
  }
  return RTC_cache.checksumData == getDataChecksum();
}

bool RTC_cache_handler_struct::saveRTCcache() {
  return saveRTCcache(0, RTC_CACHE_DATA_SIZE);
}

bool RTC_cache_handler_struct::saveRTCcache(unsigned int startOffset, size_t nrBytes)
{
  RTC_cache.checksumData     = getDataChecksum();
  RTC_cache.checksumMetadata = calc_CRC32(reinterpret_cast<const uint8_t *>(&RTC_cache), sizeof(RTC_cache) - sizeof(uint32_t));
  #ifdef ESP32
  return true;
  #endif // ifdef ESP32

  #ifdef ESP8266

  if (!system_rtc_mem_write(RTC_BASE_CACHE, reinterpret_cast<const uint8_t *>(&RTC_cache), sizeof(RTC_cache)) || !loadMetaData())
  {
        # ifdef RTC_STRUCT_DEBUG
    addLog(LOG_LEVEL_ERROR, F("RTC  : Error while writing cache metadata to RTC"));
        # endif // ifdef RTC_STRUCT_DEBUG
    return false;
  }
  delay(0);

  if (nrBytes > 0) { // Check needed?
    const size_t address = RTC_BASE_CACHE + ((sizeof(RTC_cache) + startOffset) / 4);

    if (!system_rtc_mem_write(address, reinterpret_cast<const uint8_t *>(&RTC_cache_data[startOffset]), nrBytes))
    {
          # ifdef RTC_STRUCT_DEBUG
      addLog(LOG_LEVEL_ERROR, F("RTC  : Error while writing cache data to RTC"));
          # endif // ifdef RTC_STRUCT_DEBUG
      return false;
    }
        # ifdef RTC_STRUCT_DEBUG
    rtc_debug_log(F("Write cache data to RTC"), nrBytes);
        # endif // ifdef RTC_STRUCT_DEBUG
  }
  return true;
  #endif        // ifdef ESP8266
}

uint32_t RTC_cache_handler_struct::getDataChecksum() {
  initRTCcache_data();

  /*
     size_t dataLength = RTC_cache.writePos;

     if (dataLength > RTC_CACHE_DATA_SIZE) {
     // Is this allowed to happen?
     dataLength = RTC_CACHE_DATA_SIZE;
     }
   */

  // Only compute the checksum over the number of samples stored.
  return calc_CRC32(reinterpret_cast<const uint8_t *>(&RTC_cache_data[0]), /*dataLength*/ RTC_CACHE_DATA_SIZE);
}

void RTC_cache_handler_struct::initRTCcache_data() {
  #ifdef ESP8266

  if (RTC_cache_data.size() != RTC_CACHE_DATA_SIZE) {
    RTC_cache_data.resize(RTC_CACHE_DATA_SIZE);
  }
  #endif // ifdef ESP8266

  if (RTC_cache.writeFileNr == 0) {
    // RTC value not reliable
    updateRTC_filenameCounters();
  }
}

void RTC_cache_handler_struct::clearRTCcacheData() {
  for (size_t i = 0; i < RTC_CACHE_DATA_SIZE; ++i) {
    RTC_cache_data[i] = 0;
  }
  RTC_cache.writePos = 0;
}

// Return true if any cache file found
bool RTC_cache_handler_struct::updateRTC_filenameCounters() {
  size_t filesizeHighest;

  if (getCacheFileCounters(RTC_cache.readFileNr, RTC_cache.writeFileNr, filesizeHighest)) {
    if (filesizeHighest >= CACHE_FILE_MAX_SIZE) {
      // Start new file
      ++RTC_cache.writeFileNr;
    }
    return true;
  } else {
    // Do not use 0, since that will be the cleared content of the struct, indicating invalid RTC data.
    RTC_cache.writeFileNr = 1;
  }
  return false;
}

bool RTC_cache_handler_struct::prepareFileForWrite() {
  //    if (storageLocation != CACHE_STORAGE_SPIFFS) {
  //      return false;
  //    }
  if (SpiffsFull()) {
    writeError = true;
      #ifdef RTC_STRUCT_DEBUG
    addLog(LOG_LEVEL_ERROR, F("RTC  : FS full"));
      #endif // ifdef RTC_STRUCT_DEBUG
  }

  // Make sure the read and peek file handles cannot be used on possibly deleted files.
  if (fr) {
    fr.close();
  }

  if (fp) {
    fp.close();
  }

  unsigned int retries = 3;

  while (retries > 0) {
    --retries;

    if (fw && (fw.size() >= CACHE_FILE_MAX_SIZE)) {
      fw.close();
      GarbageCollection();
    }

    if (!fw) {
      // Open file to write
      initRTCcache_data();

      if (updateRTC_filenameCounters()) {
        if (writeError || (SpiffsFreeSpace() < ((2 * CACHE_FILE_MAX_SIZE) + SpiffsBlocksize()))) {
          // Not enough room for another file, remove the oldest one.
          deleteOldestCacheBlock();
        }
      }

      String fname = createCacheFilename(RTC_cache.writeFileNr);
      fw = tryOpenFile(fname, "a+");

      if (!fw) {
          #ifdef RTC_STRUCT_DEBUG
        addLog(LOG_LEVEL_ERROR, F("RTC  : error opening file"));
          #endif // ifdef RTC_STRUCT_DEBUG
      } else {
          #ifdef RTC_STRUCT_DEBUG

        if (loglevelActiveFor(LOG_LEVEL_INFO)) {
          String log = F("Write to ");
          log += fname;
          log += F(" size");
          rtc_debug_log(log, fw.size());
        }
          #endif // ifdef RTC_STRUCT_DEBUG
      }
    }
    delay(0);

    if (fw && (fw.size() < CACHE_FILE_MAX_SIZE)) {
      return true;
    }
  }
    #ifdef RTC_STRUCT_DEBUG
  addLog(LOG_LEVEL_ERROR, F("RTC  : prepareFileForWrite failed"));
    #endif // ifdef RTC_STRUCT_DEBUG
  return false;
}

void RTC_cache_handler_struct::validateFilePos(int& fileNr, int& readPos) {
  {
    // Check to see if we try to set it to a no longer existing file
    if (fileNr < RTC_cache.readFileNr) {
      fileNr  = RTC_cache.readFileNr;
      readPos = 0;
    }
  }

  if (fileNr > RTC_cache.writeFileNr) {
    // We're trying to set it to a not yet existing file
    fileNr = RTC_cache.writeFileNr;
  }
}

#ifdef RTC_STRUCT_DEBUG
void RTC_cache_handler_struct::rtc_debug_log(const String& description, size_t nrBytes) {
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log;

    if (log.reserve(18 + description.length())) {
      log  = F("RTC  : ");
      log += description;
      log += ' ';
      log += nrBytes;
      log += F(" bytes");
      addLogMove(LOG_LEVEL_INFO, log);
    }
  }
}

#endif // ifdef RTC_STRUCT_DEBUG


#endif
#include "../DataStructs/FactoryDefault_CDN_customurl_NVS.h"

#ifdef ESP32

# if FEATURE_ALTERNATIVE_CDN_URL
#  include "../Helpers/ESPEasy_Storage.h"

#  define FACTORY_DEFAULT_NVS_CDN_CUSTOMURL_KEY "CDN_customurl"

bool FactoryDefault_CDN_customurl_NVS::applyToSettings_from_NVS(ESPEasy_NVS_Helper& preferences)
{
  String _url;

  if (preferences.getPreference(F(FACTORY_DEFAULT_NVS_CDN_CUSTOMURL_KEY), _url)) {
    set_CDN_url_custom(_url);
    return true;
  }
  return false;
}

void FactoryDefault_CDN_customurl_NVS::fromSettings_to_NVS(ESPEasy_NVS_Helper& preferences)
{
  preferences.setPreference(F(FACTORY_DEFAULT_NVS_CDN_CUSTOMURL_KEY), get_CDN_url_custom());
}

void FactoryDefault_CDN_customurl_NVS::clear_from_NVS(ESPEasy_NVS_Helper& preferences)
{
  preferences.remove(F(FACTORY_DEFAULT_NVS_CDN_CUSTOMURL_KEY));
}

# endif // if FEATURE_ALTERNATIVE_CDN_URL
#endif  // ifdef ESP32

#include "../DataStructs/FactoryDefault_LogConsoleSettings_NVS.h"


#ifdef ESP32

# include "../Globals/Settings.h"
# include "../Helpers/StringConverter.h"

// Max. 15 char keys for ESPEasy Factory Default marked keys
# define FACTORY_DEFAULT_NVS_LOG_SETTINGS_KEY  "Log"
# define FACTORY_DEFAULT_NVS_CONSOLE_SETTINGS_KEY  "Console"


bool FactoryDefault_LogConsoleSettings_NVS::applyToSettings_from_NVS(ESPEasy_NVS_Helper& preferences)
{
  bool updated = false;

  if (preferences.getPreference(F(FACTORY_DEFAULT_NVS_LOG_SETTINGS_KEY), LogSettings)) {
    updated                 = true;
    Settings.SyslogLevel    = LogSettings_bits.SyslogLevel;
    Settings.SerialLogLevel = LogSettings_bits.SerialLogLevel;
    Settings.WebLogLevel    = LogSettings_bits.WebLogLevel;
    Settings.SDLogLevel     = LogSettings_bits.SDLogLevel;
    Settings.SyslogFacility = LogSettings_bits.SyslogFacility;
    Settings.SyslogPort     = LogSettings_bits.SyslogPort;

    for (size_t i = 0; i < 4; ++i) {
      Settings.Syslog_IP[i] = LogSettings_bits.Syslog_IP[i];
    }
  }

  if (preferences.getPreference(F(FACTORY_DEFAULT_NVS_CONSOLE_SETTINGS_KEY), ConsoleSettings)) {
    updated                           = true;
    Settings.console_serial_port      = ConsoleSettings_bits.console_serial_port;
    Settings.UseSerial                = ConsoleSettings_bits.UseSerial;
    Settings.console_serial_rxpin     = ConsoleSettings_bits.console_serial_rxpin;
    Settings.console_serial_txpin     = ConsoleSettings_bits.console_serial_txpin;
    Settings.console_serial0_fallback = ConsoleSettings_bits.console_serial0_fallback;
    Settings.BaudRate                 = ConsoleSettings_bits.BaudRate;
  }
  return updated;
}

void FactoryDefault_LogConsoleSettings_NVS::fromSettings_to_NVS(ESPEasy_NVS_Helper& preferences)
{
  {
    LogSettings_bits.SyslogLevel    = Settings.SyslogLevel;
    LogSettings_bits.SerialLogLevel = Settings.SerialLogLevel;
    LogSettings_bits.WebLogLevel    = Settings.WebLogLevel;
    LogSettings_bits.SDLogLevel     = Settings.SDLogLevel;
    LogSettings_bits.SyslogFacility = Settings.SyslogFacility;
    LogSettings_bits.SyslogPort     = Settings.SyslogPort;

    for (size_t i = 0; i < 4; ++i) {
      LogSettings_bits.Syslog_IP[i] = Settings.Syslog_IP[i];
    }
    preferences.setPreference(F(FACTORY_DEFAULT_NVS_LOG_SETTINGS_KEY), LogSettings);
  }
  {
    ConsoleSettings_bits.console_serial_port      = Settings.console_serial_port;
    ConsoleSettings_bits.UseSerial                = Settings.UseSerial;
    ConsoleSettings_bits.console_serial_rxpin     = Settings.console_serial_rxpin;
    ConsoleSettings_bits.console_serial_txpin     = Settings.console_serial_txpin;
    ConsoleSettings_bits.console_serial0_fallback = Settings.console_serial0_fallback;
    ConsoleSettings_bits.BaudRate                 = Settings.BaudRate;

    preferences.setPreference(F(FACTORY_DEFAULT_NVS_CONSOLE_SETTINGS_KEY), ConsoleSettings);
  }
}

void FactoryDefault_LogConsoleSettings_NVS::clear_from_NVS(ESPEasy_NVS_Helper& preferences)
{
  preferences.remove(F(FACTORY_DEFAULT_NVS_LOG_SETTINGS_KEY));
  preferences.remove(F(FACTORY_DEFAULT_NVS_CONSOLE_SETTINGS_KEY));
}

#endif // ifdef ESP32

#include "../DataStructs/NotificationSettingsStruct.h"

#if FEATURE_NOTIFIER

NotificationSettingsStruct::NotificationSettingsStruct() {
  memset(this, 0, sizeof(NotificationSettingsStruct));
  Pin1 = -1;
  Pin2 = -1;
}

  void NotificationSettingsStruct::validate() {
    ZERO_TERMINATE(Server);
    ZERO_TERMINATE(Domain);
    ZERO_TERMINATE(Sender);
    ZERO_TERMINATE(Receiver);
    ZERO_TERMINATE(Subject);
    ZERO_TERMINATE(Body);
    ZERO_TERMINATE(User);
    ZERO_TERMINATE(Pass);
  }

#endif
#include "../DataStructs/DeviceStruct.h"

//constexpr size_t size = sizeof(DeviceStruct);

DeviceStruct::DeviceStruct() :
  _all(0u),
// PullUpOption       (false),
// InverseLogicOption (false),
// FormulaOption      (false),
// Custom             (false),
// SendDataOption     (false),
// GlobalSyncOption   (false),
// TimerOption        (false),
// TimerOptional      (false),
// DecimalsOnly       (false),
// DuplicateDetection (false),
// ExitTaskBeforeSave (true),
// ErrorStateValues   (false),
// PluginStats        (false),
// PluginLogsPeaks    (false),
// PowerManager       (false),
// TaskLogsOwnPeaks   (false),
// I2CNoDeviceCheck   (false),
// I2CMax100kHz       (false),
// HasFormatUserVar   (false),
// I2CNoBusSelection  (false),
// CustomVTypeVar     (false),

  Number(0), Type(0), VType(Sensor_VType::SENSOR_TYPE_NONE), Ports(0), ValueCount(0),
  OutputDataType(Output_Data_type_t::Default),
  Pin1Direction(static_cast<uint8_t>(gpio_direction::gpio_direction_MAX)),
  Pin2Direction(static_cast<uint8_t>(gpio_direction::gpio_direction_MAX)), 
  Pin3Direction(static_cast<uint8_t>(gpio_direction::gpio_direction_MAX))
{
  // Only initialize union members in the constructor body
  // TODO: Rename this so it can be default initialized to 'false'
  ExitTaskBeforeSave = true;
}

void DeviceStruct::clear() {
  _all = 0;

  ExitTaskBeforeSave = true;

  Number= 0;
  Type = 0;
  VType = Sensor_VType::SENSOR_TYPE_NONE;
  Ports = 0;
  ValueCount = 0;
  OutputDataType = Output_Data_type_t::Default;
  Pin1Direction = static_cast<uint8_t>(gpio_direction::gpio_direction_MAX);
  Pin2Direction = static_cast<uint8_t>(gpio_direction::gpio_direction_MAX);
  Pin3Direction = static_cast<uint8_t>(gpio_direction::gpio_direction_MAX);

}

bool DeviceStruct::connectedToGPIOpins() const {
  switch(Type) {
    case DEVICE_TYPE_SINGLE:  // Single GPIO
    case DEVICE_TYPE_SPI:
    case DEVICE_TYPE_CUSTOM1:

    case DEVICE_TYPE_DUAL:    // Dual GPIOs
    case DEVICE_TYPE_SERIAL:
    case DEVICE_TYPE_SPI2:
    case DEVICE_TYPE_CUSTOM2:

    case DEVICE_TYPE_TRIPLE:  // Triple GPIOs
    case DEVICE_TYPE_SERIAL_PLUS1:
    case DEVICE_TYPE_SPI3:
    case DEVICE_TYPE_CUSTOM3:    
      return true;
    default:
      return false;
  }
}

bool DeviceStruct::usesTaskDevicePin(int pin) const {
  if (pin == 1)
      return connectedToGPIOpins();
  if (pin == 2)
      return connectedToGPIOpins() && 
            !(Type == DEVICE_TYPE_SINGLE  ||
              Type == DEVICE_TYPE_SPI ||
              Type == DEVICE_TYPE_CUSTOM1);
  if (pin == 3)
      return Type == DEVICE_TYPE_TRIPLE || 
             Type == DEVICE_TYPE_SERIAL_PLUS1 || 
             Type == DEVICE_TYPE_SPI3 ||
             Type == DEVICE_TYPE_CUSTOM3;
  return false;
}

bool DeviceStruct::isSerial() const {
  return (Type == DEVICE_TYPE_SERIAL) || 
         (Type == DEVICE_TYPE_SERIAL_PLUS1);
}

bool DeviceStruct::isSPI() const {
  return (Type == DEVICE_TYPE_SPI) || 
         (Type == DEVICE_TYPE_SPI2) || 
         (Type == DEVICE_TYPE_SPI3);
}

bool DeviceStruct::isCustom() const {
  return (Type == DEVICE_TYPE_CUSTOM0) || 
         (Type == DEVICE_TYPE_CUSTOM1) || 
         (Type == DEVICE_TYPE_CUSTOM2) || 
         (Type == DEVICE_TYPE_CUSTOM3);
}

void DeviceStruct::setPinDirection(int pin, gpio_direction direction)
{
  const uint8_t val = static_cast<uint8_t>(direction) & ((1 << GPIO_DIRECTION_NR_BITS) - 1);
  switch (pin) {
    case 1: Pin1Direction = val; break;
    case 2: Pin2Direction = val; break;
    case 3: Pin3Direction = val; break;
  }
}

gpio_direction DeviceStruct::getPinDirection(int pin) const {
  switch (pin) {
    case 1:
      return static_cast<gpio_direction>(Pin1Direction);
    case 2:
      return static_cast<gpio_direction>(Pin2Direction);
    case 3:
      return static_cast<gpio_direction>(Pin3Direction);
  }
  return gpio_direction::gpio_direction_MAX;
}

PinSelectPurpose DeviceStruct::pinDirectionToPurpose(gpio_direction direction) const {
  switch (direction) {
  case gpio_direction::gpio_input:
    return PinSelectPurpose::Generic_input;
  case gpio_direction::gpio_output:
    return PinSelectPurpose::Generic_output;
  case gpio_direction::gpio_bidirectional:
    return PinSelectPurpose::Generic_bidir;
  case gpio_direction::gpio_direction_MAX:
    break;
  }
  return PinSelectPurpose::Generic;
}


PinSelectPurpose DeviceStruct::getPinSelectPurpose(int pin) const {
  return pinDirectionToPurpose(getPinDirection(pin));
}

#include "../DataStructs/PluginStats.h"

#if FEATURE_PLUGIN_STATS
# include "../../_Plugin_Helper.h"

# include "../Globals/TimeZone.h"

# include "../Helpers/ESPEasy_math.h"
# include "../Helpers/Memory.h"

# include "../WebServer/Chart_JS.h"


PluginStats::PluginStats(uint8_t nrDecimals, float errorValue) :
  _errorValue(errorValue),
  _nrDecimals(nrDecimals),
  _plugin_stats_timestamps(nullptr)

{
  // Try to allocate in PSRAM if possible
  void *ptr = special_calloc(1, sizeof(PluginStatsBuffer_t));

  if (ptr == nullptr) { _samples = nullptr; }
  else {
    _samples = new (ptr) PluginStatsBuffer_t();
  }
  _errorValueIsNaN   = isnan(_errorValue);
  _minValue          = std::numeric_limits<float>::max();
  _maxValue          = std::numeric_limits<float>::lowest();
  _minValueTimestamp = 0;
  _maxValueTimestamp = 0;
}

PluginStats::~PluginStats()
{
  if (_samples != nullptr) {
    free(_samples);

    //    delete _samples;
  }
  _samples                 = nullptr;
  _plugin_stats_timestamps = nullptr;
}

void PluginStats::processTimeSet(const double& time_offset)
{
  // Check to see if there was a unix time set before the system time was set
  // For example when receiving data from a p2p node
  const int64_t cur_micros    = getMicros64();
  const int64_t offset_micros = time_offset * 1000000ull;

  if ((_maxValueTimestamp > cur_micros) && (_maxValueTimestamp > offset_micros)) {
    _maxValueTimestamp -= offset_micros;
  }

  if ((_minValueTimestamp > cur_micros) && (_minValueTimestamp > offset_micros)) {
    _minValueTimestamp -= offset_micros;
  }
}

bool PluginStats::push(float value)
{
  if (_samples == nullptr) { return false; }
  return _samples->push(value);
}

bool PluginStats::matchesLastTwoEntries(float value) const
{
  const size_t nrSamples = getNrSamples();

  if (nrSamples < 2) { return false; }

  const float last       = (*_samples)[nrSamples - 1];
  const float beforeLast = (*_samples)[nrSamples - 2];

  const String value_str = toString(value, _nrDecimals);

  return
    toString(last,       _nrDecimals).equals(value_str) &&
    toString(beforeLast, _nrDecimals).equals(value_str);


  /*
     const bool  value_valid = isValidFloat(value);
     const bool  last_valid  = isValidFloat(last);

     if (value_valid != last_valid) {
      return false;
     }
     const bool beforeLast_valid = isValidFloat(beforeLast);

     if (value_valid != beforeLast_valid) {
      return false;
     }

     if (value_valid) {
      return
        approximatelyEqual(value, last) &&
        approximatelyEqual(value, beforeLast);
     }
     return true;
   */
}

void PluginStats::trackPeak(float value, int64_t timestamp)
{
  if ((value > _maxValue) || (value < _minValue)) {
    if (timestamp == 0) {
      // Make sure both extremes are flagged with the same timestamp.
      timestamp = getMicros64();
    }

    if (value > _maxValue) {
      _maxValueTimestamp = timestamp;
      _maxValue          = value;
    }

    if (value < _minValue) {
      _minValueTimestamp = timestamp;
      _minValue          = value;
    }
  }
}

void PluginStats::resetPeaks()
{
  _minValue          = std::numeric_limits<float>::max();
  _maxValue          = std::numeric_limits<float>::lowest();
  _minValueTimestamp = 0;
  _maxValueTimestamp = 0;
}

void PluginStats::clearSamples() {
  if (_samples != nullptr) {
    _samples->clear();
  }
}

size_t PluginStats::getNrSamples() const {
  if (_samples == nullptr) { return 0u; }
  return _samples->size();
}

float PluginStats::getSampleAvg() const {
  return getSampleAvg(getNrSamples());
}

float PluginStats::getSampleAvg(PluginStatsBuffer_t::index_t lastNrSamples) const
{
  const size_t nrSamples = getNrSamples();

  if (nrSamples == 0) { return _errorValue; }
  float sum = 0.0f;

  PluginStatsBuffer_t::index_t i = 0;

  if (lastNrSamples < nrSamples) {
    i = nrSamples - lastNrSamples;
  }
  PluginStatsBuffer_t::index_t samplesUsed = 0;

  for (; i < nrSamples; ++i) {
    const float sample((*_samples)[i]);

    if (usableValue(sample)) {
      ++samplesUsed;
      sum += sample;
    }
  }

  if (samplesUsed == 0) { return _errorValue; }
  return sum / samplesUsed;
}

float PluginStats::getSampleAvg_time(PluginStatsBuffer_t::index_t lastNrSamples, uint64_t& totalDuration_usec) const
{
  const size_t nrSamples = getNrSamples();

  totalDuration_usec = 0u;

  if ((nrSamples == 0) || (_plugin_stats_timestamps == nullptr)) {
    return _errorValue;
  }

  PluginStatsBuffer_t::index_t i = 0;

  if (lastNrSamples < nrSamples) {
    i = nrSamples - lastNrSamples;
  }

  int64_t lastTimestamp   = 0;
  float   lastValue       = 0.0f;
  bool    lastValueUsable = false;
  float   sum             = 0.0f;

  for (; i < nrSamples; ++i) {
    const float   sample((*_samples)[i]);
    const int64_t curTimestamp   = (*_plugin_stats_timestamps)[i];
    const bool    curValueUsable = usableValue(sample);

    if ((lastTimestamp != 0) && lastValueUsable) {
      const int64_t duration_usec = abs(timeDiff64(lastTimestamp, curTimestamp));

      if (curValueUsable) {
        // Old and new value usable, take average of this period.
        sum += ((lastValue + sample) / 2.0f) * duration_usec;
      } else {
        // New value is not usable, so just add the last value for the duration.
        sum += lastValue * duration_usec;
      }
      totalDuration_usec += duration_usec;
    }

    lastValueUsable = curValueUsable;
    lastTimestamp   = curTimestamp;
    lastValue       = sample;
  }

  if (totalDuration_usec == 0) { return _errorValue; }
  return sum / totalDuration_usec;
}

float PluginStats::getSampleStdDev(PluginStatsBuffer_t::index_t lastNrSamples) const
{
  const size_t nrSamples = getNrSamples();
  float variance         = 0.0f;
  const float average    = getSampleAvg(lastNrSamples);

  if (!usableValue(average)) { return 0.0f; }

  PluginStatsBuffer_t::index_t i = 0;

  if (lastNrSamples < nrSamples) {
    i = nrSamples - lastNrSamples;
  }
  PluginStatsBuffer_t::index_t samplesUsed = 0;

  for (; i < nrSamples; ++i) {
    const float sample((*_samples)[i]);

    if (usableValue(sample)) {
      ++samplesUsed;
      const float diff = sample - average;
      variance += diff * diff;
    }
  }

  if (samplesUsed < 2) { return 0.0f; }

  variance /= samplesUsed;
  return sqrtf(variance);
}

float PluginStats::getSampleExtreme(PluginStatsBuffer_t::index_t lastNrSamples, bool getMax) const
{
  const size_t nrSamples = getNrSamples();

  if (nrSamples == 0) { return _errorValue; }

  PluginStatsBuffer_t::index_t i = 0;

  if (lastNrSamples < nrSamples) {
    i = nrSamples - lastNrSamples;
  }

  bool changed = false;

  float res = getMax ? INT_MIN : INT_MAX;

  for (; i < nrSamples; ++i) {
    const float sample((*_samples)[i]);

    if (usableValue(sample)) {
      if ((getMax && (sample > res)) ||
          (!getMax && (sample < res))) {
        changed = true;
        res     = sample;
      }
    }
  }

  if (!changed) { return _errorValue; }

  return res;
}

float PluginStats::getSample(int lastNrSamples) const
{
  const size_t nrSamples = getNrSamples();

  if ((nrSamples == 0) || (nrSamples < abs(lastNrSamples))) { return _errorValue; }

  PluginStatsBuffer_t::index_t i = 0;

  if (lastNrSamples > 0) {
    i = nrSamples - lastNrSamples;
  } else if (lastNrSamples < 0) {
    i = abs(lastNrSamples) - 1;
  }

  if (i < nrSamples) {
    return (*_samples)[i];
  }
  return _errorValue;
}

float PluginStats::operator[](PluginStatsBuffer_t::index_t index) const
{
  const size_t nrSamples = getNrSamples();

  if (index < nrSamples) { return (*_samples)[index]; }
  return _errorValue;
}

bool PluginStats::matchedCommand(const String& command, const __FlashStringHelper *cmd_match, int& nrSamples)
{
  const String cmd_match_str(cmd_match);

  if (command.equals(cmd_match_str)) {
    nrSamples = INT_MIN;
    return true;
  }

  if (command.startsWith(cmd_match_str)) {
    nrSamples = 0;

    // FIXME TD-er: ESP_IDF 5.x needs strict matching thus int32_t != int
    int32_t tmp{};

    if (validIntFromString(command.substring(cmd_match_str.length()), tmp)) {
      nrSamples = tmp;
      return true;
    }
  }
  return false;
}

bool PluginStats::plugin_get_config_value_base(struct EventStruct *event, String& string) const
{
  // Full value name is something like "taskvaluename.avg"
  const String fullValueName = parseString(string, 1);
  const String command       = parseString(fullValueName, 2, '.');

  if (command.isEmpty()) {
    return false;
  }

  float value{};
  int   nrSamples = 0;
  bool  success   = false;

  switch (command[0])
  {
    case 'a':

      if (matchedCommand(command, F("avg"), nrSamples)) {
        success = nrSamples != 0;

        if (nrSamples < 0) { // [taskname#valuename.avg] Average value of the last N kept samples
          value = getSampleAvg();
        } else {
          // Check for "avgN", where N is the number of most recent samples to use.
          if (nrSamples > 0) {
            // [taskname#valuename.avgN] Average over N most recent samples
            value = getSampleAvg(nrSamples);
          }
        }
      }
      break;
    case 'm':

      if (equals(command, F("minp"))) { // [taskname#valuename.minp] Lowest value seen since value reset
        value   = getPeakLow();
        success = true;
      } else if (matchedCommand(command, F("min"), nrSamples)) {
        success = nrSamples != 0;

        if (nrSamples < 0) { // [taskname#valuename.min] Lowest value of all recent samples
          value = getSampleExtreme(getNrSamples(), false);
        } else {             // Check for "minN", where N is the number of most recent samples to use.
          if (nrSamples > 0) {
            value = getSampleExtreme(nrSamples, false);
          }
        }
      } else if (equals(command, F("maxp"))) { // [taskname#valuename.maxp] Highest value seen since value reset
        value   = getPeakHigh();
        success = true;
      } else if (matchedCommand(command, F("max"), nrSamples)) {
        success = nrSamples != 0;

        if (nrSamples < 0) { // [taskname#valuename.max] Highest value of all recent samples
          value = getSampleExtreme(getNrSamples(), true);
        } else {             // Check for "maxN", where N is the number of most recent samples to use.
          if (nrSamples > 0) {
            value = getSampleExtreme(nrSamples, true);
          }
        }
      }
      break;
    case 's':

      if (matchedCommand(command, F("stddev"), nrSamples)) {
        success = nrSamples != 0;

        if (nrSamples < 0) { // [taskname#valuename.stddev] Std deviation of the last N kept samples
          value = getSampleStdDev();
        } else {
          // Check for "stddevN", where N is the number of most recent samples to use.
          if (nrSamples > 0) {
            // [taskname#valuename.stddevN] Std. deviation over N most recent samples
            value = getSampleStdDev(nrSamples);
          }
        }
      } else if (matchedCommand(command, F("size"), nrSamples)) {
        // [taskname#valuename.size] Number of samples in memory
        value   = getNrSamples();
        success = true;
      } else if (matchedCommand(command, F("sample"), nrSamples)) {
        success = nrSamples != 0;

        if (nrSamples == INT_MIN) {
          // [taskname#valuename.sample] Number of samples in memory.
          value   = getNrSamples();
          success = true;
        } else {
          if (nrSamples != 0) {
            // [taskname#valuename.sampleN]
            // With sample N:
            //   N > 0: Return N'th most recent sample
            //   N < 0: Return abs(N)'th sample in memory, starting at the oldest one.
            //   abs(N) > [number of samples]: return error value
            value = getSample(nrSamples);
          }
        }
      }
      break;
    default:
      return false;
  }


  if (success) {
    string = toString(value, _nrDecimals);
  }
  return success;
}

bool PluginStats::webformLoad_show_stats(struct EventStruct *event) const
{
  bool somethingAdded = false;

  if (webformLoad_show_avg(event)) { somethingAdded = true; }

  if (webformLoad_show_stdev(event)) { somethingAdded = true; }

  if (webformLoad_show_peaks(event)) { somethingAdded = true; }

  if (somethingAdded) {
    addFormSeparator(4);
  }

  return somethingAdded;
}

bool PluginStats::webformLoad_show_avg(struct EventStruct *event) const
{
  if (getNrSamples() > 0) {
    addRowLabel(concat(getLabel(),  F(" Average / sample")));
    addHtmlFloat(getSampleAvg(), (_nrDecimals == 0) ? 1 : _nrDecimals);
    addHtml(strformat(F(" (%u samples)"), getNrSamples()));

    if (_plugin_stats_timestamps != nullptr) {
      uint64_t totalDuration_usec = 0u;
      const float avg_per_sec     = getSampleAvg_time(totalDuration_usec);

      if (totalDuration_usec > 0) {
        addRowLabel(concat(getLabel(),  F(" Average / sec")));
        addHtmlFloat(avg_per_sec, (_nrDecimals == 0) ? 1 : _nrDecimals);
        addHtml(strformat(F(" (%s duration)"), secondsToDayHourMinuteSecond_ms(totalDuration_usec).c_str()));
      }
    }
    return true;
  }
  return false;
}

bool PluginStats::webformLoad_show_stdev(struct EventStruct *event) const
{
  const float stdDev = getSampleStdDev();

  if (usableValue(stdDev) && (getNrSamples() > 1)) {
    addRowLabel(concat(getLabel(),  F(" std. dev")));
    addHtmlFloat(stdDev, (_nrDecimals == 0) ? 1 : _nrDecimals);
    addHtml(strformat(F(" (%u samples)"), getNrSamples()));
    return true;
  }
  return false;
}

bool PluginStats::webformLoad_show_peaks(struct EventStruct *event, bool include_peak_to_peak) const
{
  if (hasPeaks() && (getNrSamples() > 1)) {
    return webformLoad_show_peaks(
      event,
      getLabel(),
      toString(getPeakLow(),  _nrDecimals),
      toString(getPeakHigh(), _nrDecimals),
      include_peak_to_peak);
  }
  return false;
}

bool PluginStats::webformLoad_show_peaks(struct EventStruct *event,
                                         const String      & label,
                                         const String      & lowValue,
                                         const String      & highValue,
                                         bool                include_peak_to_peak) const
{
  if (hasPeaks() && (getNrSamples() > 1)) {
    uint32_t peakLow_frac{};
    uint32_t peakHigh_frac{};
    const uint32_t peakLow     = node_time.systemMicros_to_Unixtime(getPeakLowTimestamp(), peakLow_frac);
    const uint32_t peakHigh    = node_time.systemMicros_to_Unixtime(getPeakHighTimestamp(), peakHigh_frac);
    const uint32_t current     = node_time.getUnixTime();
    const bool     useTimeOnly = (current - peakLow) < 86400 && (current - peakHigh) < 86400;
    struct tm ts;
    breakTime(time_zone.toLocal(peakLow), ts);


    addRowLabel(concat(label,  F(" Peak Low")));
    addHtml(strformat(
              F("%s @ %s.%03u"),
              lowValue.c_str(),
              useTimeOnly
      ? formatTimeString(ts).c_str()
      : formatDateTimeString(ts).c_str(),
              unix_time_frac_to_millis(peakLow_frac)));


    breakTime(time_zone.toLocal(peakHigh), ts);

    addRowLabel(concat(label,  F(" Peak High")));
    addHtml(strformat(
              F("%s @ %s.%03u"),
              highValue.c_str(),
              useTimeOnly
      ? formatTimeString(ts).c_str()
      : formatDateTimeString(ts).c_str(),
              unix_time_frac_to_millis(peakHigh_frac)));

    if (include_peak_to_peak) {
      addRowLabel(concat(getLabel(),  F(" Peak-to-peak")));
      addHtmlFloat(getPeakHigh() - getPeakLow(), _nrDecimals);
    }
    return true;
  }
  return false;
}

void PluginStats::webformLoad_show_val(
  struct EventStruct      *event,
  const String           & label,
  ESPEASY_RULES_FLOAT_TYPE value,
  const String           & unit) const
{
  addRowLabel(concat(getLabel(), label));
  addHtmlFloat(value, _nrDecimals);

  if (!unit.isEmpty()) {
    addUnit(unit);
  }
}

# if FEATURE_CHART_JS
void PluginStats::plot_ChartJS_dataset() const
{
  add_ChartJS_dataset_header(_ChartJS_dataset_config);

  PluginStatsBuffer_t::index_t i = 0;
  const size_t nrSamples         = getNrSamples();

  for (; i < nrSamples; ++i) {
    if (i != 0) {
      addHtml(',');
    }

    addHtmlFloat_NaN_toNull((*_samples)[i], _nrDecimals);
  }
  add_ChartJS_dataset_footer();
}

# endif // if FEATURE_CHART_JS

bool PluginStats::usableValue(float value) const
{
  if (!isnan(value)) {
    if (_errorValueIsNaN || !essentiallyEqual(_errorValue, value)) {
      return true;
    }
  }
  return false;
}

#endif // if FEATURE_PLUGIN_STATS

#include "../DataStructs/EthernetEventData.h"

#if FEATURE_ETHERNET

#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../Globals/Settings.h"
#include "../Helpers/Networking.h"

#include <ETH.h>

// Bit numbers for Eth status
#define ESPEASY_ETH_CONNECTED               0
#define ESPEASY_ETH_GOT_IP                  1
#define ESPEASY_ETH_SERVICES_INITIALIZED    2


#if FEATURE_USE_IPV6
#include <esp_netif.h>

// -----------------------------------------------------------------------------------------------------------------------
// ---------------------------------------------------- Private functions ------------------------------------------------
// -----------------------------------------------------------------------------------------------------------------------

esp_netif_t* get_esp_interface_netif(esp_interface_t interface);
#endif



bool EthernetEventData_t::EthConnectAllowed() const {
  if (!ethConnectAttemptNeeded) return false;
  if (last_eth_connect_attempt_moment.isSet()) {
    // TODO TD-er: Make this time more dynamic.
    if (!last_eth_connect_attempt_moment.timeoutReached(10000)) {
      return false;
    }
  }
  return true;
}

bool EthernetEventData_t::unprocessedEthEvents() const {
  if (processedConnect && processedDisconnect && processedGotIP && processedDHCPTimeout
#if FEATURE_USE_IPV6
      && processedGotIP6
#endif
  )
  {
    return false;
  }
  return true;
}

void EthernetEventData_t::clearAll() {
  lastDisconnectMoment.clear();
  lastConnectMoment.clear();
  lastGetIPmoment.clear();
  last_eth_connect_attempt_moment.clear();

  lastEthResetMoment.setNow();
  eth_considered_stable = false;

  // Mark all flags to default to prevent handling old events.
  processedConnect          = true;
  processedDisconnect       = true;
  processedGotIP            = true;
  #if FEATURE_USE_IPV6
  processedGotIP6           = true;
  #endif
  processedDHCPTimeout      = true;
  ethConnectAttemptNeeded  = true;
  dns0_cache = IPAddress();
  dns1_cache = IPAddress();
}

void EthernetEventData_t::markEthBegin() {
  lastDisconnectMoment.clear();
  lastConnectMoment.clear();
  lastGetIPmoment.clear();
  last_eth_connect_attempt_moment.setNow();
  eth_considered_stable = false;
  ethConnectInProgress  = true;
  ++eth_connect_attempt;
}

bool EthernetEventData_t::EthDisconnected() const {
  return ethStatus == ESPEASY_ETH_DISCONNECTED;
}

bool EthernetEventData_t::EthGotIP() const {

  return bitRead(ethStatus, ESPEASY_ETH_GOT_IP);
}

bool EthernetEventData_t::EthConnected() const {
  return bitRead(ethStatus, ESPEASY_ETH_CONNECTED);
}

bool EthernetEventData_t::EthServicesInitialized() const {
  return bitRead(ethStatus, ESPEASY_ETH_SERVICES_INITIALIZED);
}

void EthernetEventData_t::setEthDisconnected() {
  processedConnect          = true;
  processedDisconnect       = true;
  processedGotIP            = true;
  #if FEATURE_USE_IPV6
  processedGotIP6           = true;
  #endif
  processedDHCPTimeout      = true;

  ethStatus = ESPEASY_ETH_DISCONNECTED;
}

void EthernetEventData_t::setEthGotIP() {
  bitSet(ethStatus, ESPEASY_ETH_GOT_IP);
  setEthServicesInitialized();
}

void EthernetEventData_t::setEthConnected() {
  bitSet(ethStatus, ESPEASY_ETH_CONNECTED);
  setEthServicesInitialized();
}

bool EthernetEventData_t::setEthServicesInitialized() {
  if (!unprocessedEthEvents() && !EthServicesInitialized()) {
    if (EthGotIP() && EthConnected()) {
      if (valid_DNS_address(ETH.dnsIP(0))) {
        dns0_cache = ETH.dnsIP(0);
      }
      if (valid_DNS_address(ETH.dnsIP(1))) {
        dns1_cache = ETH.dnsIP(1);
      }

      #ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_DEBUG, F("Eth : Eth services initialized"));
      #endif
      bitSet(ethStatus, ESPEASY_ETH_SERVICES_INITIALIZED);
      ethConnectInProgress = false;
      return true;
    }
  }
  return false;
}

void EthernetEventData_t::markGotIP() {
  lastGetIPmoment.setNow();

  // Create the 'got IP event' so mark the ethStatus to not have the got IP flag set
  // This also implies the services are not fully initialized.
  bitClear(ethStatus, ESPEASY_ETH_GOT_IP);
  bitClear(ethStatus, ESPEASY_ETH_SERVICES_INITIALIZED);
  processedGotIP = false;
}

#if FEATURE_USE_IPV6
void EthernetEventData_t::markGotIPv6(const IPAddress& ip6) {
  processedGotIP6 = false;
  unprocessed_IP6 = ip6;
}
#endif


void EthernetEventData_t::markLostIP() {
  bitClear(ethStatus, ESPEASY_ETH_GOT_IP);
  bitClear(ethStatus, ESPEASY_ETH_SERVICES_INITIALIZED);
  lastGetIPmoment.clear();
  processedGotIP = false;
}

void EthernetEventData_t::markDisconnect() {
  lastDisconnectMoment.setNow();

  if (last_eth_connect_attempt_moment.isSet() && !lastConnectMoment.isSet()) {
    // There was an unsuccessful connection attempt
    lastConnectedDuration_us = last_eth_connect_attempt_moment.timeDiff(lastDisconnectMoment);
  } else {
    lastConnectedDuration_us = lastConnectMoment.timeDiff(lastDisconnectMoment);
  }
  lastConnectMoment.clear();
  processedDisconnect  = false;
#if ESP_IDF_VERSION_MAJOR >= 5
  WiFi.STA.setDefault();
#endif
}

void EthernetEventData_t::markConnected() {
  lastConnectMoment.setNow();
  processedConnect    = false;
#if ESP_IDF_VERSION_MAJOR >= 5
  ETH.setDefault();
#endif

#if FEATURE_USE_IPV6
  if (Settings.EnableIPv6()) {
    ETH.enableIPv6(true);
  }
#endif
}

String EthernetEventData_t::ESPEasyEthStatusToString() const {
  String log;
  if (EthDisconnected()) {
    log = F("DISCONNECTED");
  } else {
    if (EthConnected()) {
      log += F("Conn. ");
    }
    if (EthGotIP()) {
      log += F("IP ");
    }
    if (EthServicesInitialized()) {
      log += F("Init");
    }
  }
  return log;
}

#endif

#include "../DataStructs/FactoryDefault_UnitName_NVS.h"

#ifdef ESP32

# include "../Globals/Settings.h"
# include "../Helpers/StringConverter.h"

// Max. 15 char keys for ESPEasy Factory Default marked keys
# define FACTORY_DEFAULT_NVS_UNIT_NAME_KEY  "UnitName"


void FactoryDefault_UnitName_NVS::fromSettings() {
  bitWrite(data[1], 0, Settings.appendUnitToHostname());
  data[0] = Settings.Unit;
  memcpy((char *)(data + 2), Settings.Name, sizeof(Settings.Name));
  data[2 + sizeof(Settings.Name)] = Settings.UDPPort >> 8;
  data[3 + sizeof(Settings.Name)] = Settings.UDPPort & 0xFF;
}

void FactoryDefault_UnitName_NVS::applyToSettings() const {
  Settings.appendUnitToHostname(bitRead(data[1], 0));
  Settings.Unit = data[0];
  memcpy(Settings.Name, (char *)(data + 2), sizeof(Settings.Name));

  Settings.UDPPort = data[2 + sizeof(Settings.Name)] << 8 | data[3 + sizeof(Settings.Name)];
}

bool FactoryDefault_UnitName_NVS::applyToSettings_from_NVS(ESPEasy_NVS_Helper& preferences) {
  if (preferences.getPreference(F(FACTORY_DEFAULT_NVS_UNIT_NAME_KEY), data, sizeof(data))) {
    applyToSettings();
    return true;
  }
  return false;
}

void FactoryDefault_UnitName_NVS::fromSettings_to_NVS(ESPEasy_NVS_Helper& preferences) {
  fromSettings();
  preferences.setPreference(F(FACTORY_DEFAULT_NVS_UNIT_NAME_KEY), data, sizeof(data));
}

void FactoryDefault_UnitName_NVS::clear_from_NVS(ESPEasy_NVS_Helper& preferences) {
  preferences.remove(F(FACTORY_DEFAULT_NVS_UNIT_NAME_KEY));
}

#endif // ifdef ESP32

#include "../DataStructs/Web_StreamingBuffer.h"

#include "../DataStructs/tcp_cleanup.h"
#include "../DataTypes/ESPEasyTimeSource.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"

// FIXME TD-er: Should keep a pointer to the webserver as a member, not use the global defined one.
#include "../Globals/Services.h"

#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/Convert.h"
#include "../Helpers/StringConverter.h"

#include "../../ESPEasy_common.h"

#ifdef ESP8266
#define CHUNKED_BUFFER_SIZE         512
#else 
#define CHUNKED_BUFFER_SIZE         1200
#endif

Web_StreamingBuffer::Web_StreamingBuffer(void) : lowMemorySkip(false),
  initialRam(0), beforeTXRam(0), duringTXRam(0), finalRam(0), maxCoreUsage(0),
  maxServerUsage(0), sentBytes(0), flashStringCalls(0), flashStringData(0)
{
  // Make sure this is allocated on the DRAM since access to primary heap is faster
  # ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  # endif // ifdef USE_SECOND_HEAP

  buf.reserve(CHUNKED_BUFFER_SIZE + 50);
  buf.clear();
}

Web_StreamingBuffer& Web_StreamingBuffer::operator+=(char a)                   {
  if (this->buf.length() >= CHUNKED_BUFFER_SIZE) {
    flush();
  }
  this->buf += a;
  return *this;
}

Web_StreamingBuffer& Web_StreamingBuffer::operator+=(uint64_t a) {
  return addString(ull2String(a));
}

Web_StreamingBuffer& Web_StreamingBuffer::operator+=(int64_t a) {
  return addString(ll2String(a));
}

Web_StreamingBuffer& Web_StreamingBuffer::operator+=(const float& a)           {
  return addString(toString(a, 2));
}

#if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
Web_StreamingBuffer& Web_StreamingBuffer::operator+=(const double& a)          {
  return addString(doubleToString(a));
}
#endif

Web_StreamingBuffer& Web_StreamingBuffer::operator+=(const String& a)          {
  return addString(a);
}

Web_StreamingBuffer& Web_StreamingBuffer::operator+=(PGM_P str) {
  return addFlashString(str);
}

Web_StreamingBuffer& Web_StreamingBuffer::operator+=(const __FlashStringHelper* str) {
  return addFlashString((PGM_P)str);
}

Web_StreamingBuffer& Web_StreamingBuffer::addFlashString(PGM_P str, int length) {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif


  if (!str) { 
    return *this; // return if the pointer is void
  }

  #ifdef USE_SECOND_HEAP
  if (mmu_is_iram(str)) {
    // Have to copy the string using mmu_get functions
    // This is not a flash string.
    bool done = false;
    const char* cur_char = str;
    while (!done) {
      const uint8_t ch = mmu_get_uint8(cur_char++);
      if (ch == 0) return *this;
      if (this->buf.length() >= CHUNKED_BUFFER_SIZE) {
        flush();
      }
      this->buf += (char)ch;
    }
  }
  #endif

  ++flashStringCalls;

  if (lowMemorySkip) { return *this; }

  checkFull();

  int flush_step = CHUNKED_BUFFER_SIZE - this->buf.length();
  if (flush_step < 1) { flush_step = 0; }

  {
    // Copy to internal buffer and send in chunks
    PGM_P pos = str;
    while (length != 0) {
      if (flush_step == 0) {
        flush();
        flush_step = CHUNKED_BUFFER_SIZE;
      }
      const char c = (char)pgm_read_byte(pos);
      if (c == '\0' && length < 0) {
        // Only check for \0 when length was given (e.g. binary data)
        return *this;
      }
      this->buf += c;
      ++flashStringData;
      ++pos;
      --length;
      --flush_step;
    }
  }
  return *this;
}

Web_StreamingBuffer& Web_StreamingBuffer::addString(const String& a) {
  # ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  # endif // ifdef USE_SECOND_HEAP

  if (lowMemorySkip) { return *this; }
  const unsigned int length = a.length();
  if (length == 0) { return *this; }

  checkFull();
  int flush_step = CHUNKED_BUFFER_SIZE - this->buf.length();

  if (flush_step < 1) { flush_step = 0; }

  if (length < static_cast<unsigned int>(flush_step)) {
    // Just use the faster String operator to copy flash strings.
    this->buf += a;
    return *this;
  }

  unsigned int pos = 0;
  while (pos < length) {
    if (flush_step <= 0) {
      flush();
      flush_step = CHUNKED_BUFFER_SIZE;
    } else {
      const int remaining = length - pos;
      const int fetchLength = flush_step >= remaining ? remaining : flush_step;
      const char* ch = a.begin() + pos;
      this->buf.concat(ch, static_cast<unsigned int>(fetchLength));
      pos += fetchLength;
      flush_step -= fetchLength;
    }
  }
  return *this;
}

void Web_StreamingBuffer::flush() {
  if (lowMemorySkip) {
    this->buf.clear();
  } else {
    if (this->buf.length() > 0) {
      sendContentBlocking(this->buf);
    }
  }
}

void Web_StreamingBuffer::checkFull() {
  if (lowMemorySkip) { this->buf.clear(); }

  if (this->buf.length() >= CHUNKED_BUFFER_SIZE) {
    trackTotalMem();
    flush();
  }
}

void Web_StreamingBuffer::startStream(int httpCode) {
  startStream(false, F("text/html"), F(""), httpCode);
}

void Web_StreamingBuffer::startStream(const __FlashStringHelper * origin, int httpCode) {
  startStream(false, F("text/html"), origin, httpCode);
}

void Web_StreamingBuffer::startStream(const __FlashStringHelper * content_type,
                                      const __FlashStringHelper * origin, 
                                      int httpCode, 
                                      bool cacheable) {
  startStream(false, content_type, origin, httpCode, cacheable);
}


void Web_StreamingBuffer::startJsonStream() {
  startStream(true, F("application/json"), F("*"));
}

void Web_StreamingBuffer::startStream(bool allowOriginAll, 
                                      const __FlashStringHelper * content_type, 
                                      const __FlashStringHelper * origin,
                                      int httpCode,
                                      bool cacheable) {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif

  maxCoreUsage = maxServerUsage = 0;
  initialRam   = ESP.getFreeHeap();
  beforeTXRam  = initialRam;
  sentBytes    = 0;
  buf.clear();
  buf.reserve(CHUNKED_BUFFER_SIZE);
  web_server.client().setNoDelay(true);
#ifdef ESP32
  web_server.client().setSSE(false);
#endif
  
  if (beforeTXRam < 3000) {
    lowMemorySkip = true;
    web_server.send_P(200, (PGM_P)F("text/plain"), (PGM_P)F("Low memory. Cannot display webpage :-("));
      #if defined(ESP8266)
    tcpCleanup();
      #endif // if defined(ESP8266)
    return;
  } else {
    sendHeaderBlocking(allowOriginAll, content_type, origin, httpCode, cacheable);
  }
}

void Web_StreamingBuffer::trackTotalMem() {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif

  beforeTXRam = ESP.getFreeHeap();

  if ((initialRam - beforeTXRam) > maxServerUsage) {
    maxServerUsage = initialRam - beforeTXRam;
  }
}

void Web_StreamingBuffer::trackCoreMem() {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif

  duringTXRam = ESP.getFreeHeap();

  if ((initialRam - duringTXRam) > maxCoreUsage) {
    maxCoreUsage = (initialRam - duringTXRam);
  }
}

void Web_StreamingBuffer::endStream() {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif

  if (!lowMemorySkip) {
    if (buf.length() > 0) { sendContentBlocking(buf); }
    buf.clear();
    sendContentBlocking(buf);

    web_server.client().PR_9453_FLUSH_TO_CLEAR();

    finalRam = ESP.getFreeHeap();

/*
#ifndef BUILD_NO_DEBUG
        if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
        String log = String("Ram usage: Webserver only: ") + maxServerUsage +
                    " including Core: " + maxCoreUsage +
                    " flashStringCalls: " + flashStringCalls +
                    " flashStringData: " + flashStringData;
        addLog(LOG_LEVEL_DEBUG, log);
        }
#endif // ifndef BUILD_NO_DEBUG
*/

  } else {
    if (loglevelActiveFor(LOG_LEVEL_ERROR))
      addLog(LOG_LEVEL_ERROR, concat("Webpage skipped: low memory: ", finalRam));
    lowMemorySkip = false;
  }

  delay(5);
  web_server.client().stop();
  #ifdef ESP8266
  tcpCleanup();
  #endif
}


void Web_StreamingBuffer::sendContentBlocking(String& data) {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif

  delay(0); // Try to prevent WDT reboots

  const uint32_t length   = data.length();
#ifndef BUILD_NO_DEBUG
  if (loglevelActiveFor(LOG_LEVEL_DEBUG_DEV)) {
    addLogMove(LOG_LEVEL_DEBUG_DEV, strformat(
      F("sendcontent free: %u  chunk size: %u"), 
      ESP.getFreeHeap(), 
      length));
  }
#endif // ifndef BUILD_NO_DEBUG
  const uint32_t freeBeforeSend = ESP.getFreeHeap();
  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("sendContentBlocking"));
  #endif

  if (beforeTXRam > freeBeforeSend) {
    beforeTXRam = freeBeforeSend;
  }
  duringTXRam = freeBeforeSend;
  
#if defined(ESP8266) && defined(ARDUINO_ESP8266_RELEASE_2_3_0)
  String size = formatToHex(length) + "\r\n";

  // do chunked transfer encoding ourselves (WebServer doesn't support it)
  web_server.sendContent(size);

  if (length > 0) { web_server.sendContent(data); }
  web_server.sendContent("\r\n");
#else // ESP8266 2.4.0rc2 and higher and the ESP32 webserver supports chunked http transfer
  #if defined(ESP8266) && defined(USE_SECOND_HEAP)
  {
    HeapSelectIram ephemeral;
    web_server.sendContent(data);
  }
  #else
  web_server.sendContent(data);
  #endif

  if (data.length() > (CHUNKED_BUFFER_SIZE + 1)) {
    free_string(data); // Clear also allocated memory
  } else {
    data.clear();
  }

  const uint32_t timeout = millis() + 100;
  while ((!data.reserve(CHUNKED_BUFFER_SIZE) || (ESP.getFreeHeap() < 4000 /*freeBeforeSend*/ )) &&
         !timeOutReached(timeout)) {
    if (ESP.getFreeHeap() < duringTXRam) {
      duringTXRam = ESP.getFreeHeap();
    }
    trackCoreMem();
    #ifndef BUILD_NO_RAM_TRACKER
    checkRAM(F("duringDataTX"));
    #endif

    delay(1);
  }
#endif // if defined(ESP8266) && defined(ARDUINO_ESP8266_RELEASE_2_3_0)

  sentBytes += length;
  delay(1);
}

void Web_StreamingBuffer::sendHeaderBlocking(bool allowOriginAll, 
                                             const String& content_type, 
                                             const String& origin,
                                             int httpCode,
                                             bool cacheable) {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif

  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("sendHeaderBlocking"));
  #endif
  
  web_server.client().PR_9453_FLUSH_TO_CLEAR();

#if defined(ESP8266) && defined(ARDUINO_ESP8266_RELEASE_2_3_0)
  web_server.setContentLength(CONTENT_LENGTH_UNKNOWN);
  sendHeader(F("Accept-Ranges"),     F("none"));
  sendHeader(F("Cache-Control"),     F("no-cache"));
  sendHeader(F("Transfer-Encoding"), F("chunked"));

  if (allowOriginAll) {
    sendHeader(F("Access-Control-Allow-Origin"), "*");
  }
  web_server.send(httpCode, content_type, EMPTY_STRING);
#else // if defined(ESP8266) && defined(ARDUINO_ESP8266_RELEASE_2_3_0)
  unsigned int timeout          = 100;
  const uint32_t freeBeforeSend = ESP.getFreeHeap();

  const uint32_t beginWait = millis();

  web_server.setContentLength(CONTENT_LENGTH_UNKNOWN);
  if (!cacheable)
    web_server.sendHeader(F("Cache-Control"), F("no-cache"));

#if ESP_IDF_VERSION_MAJOR>4
  if (origin.equals("*")) {
    web_server.enableCORS(true);
  } else
#endif
  if (origin.length() > 0) {
#if ESP_IDF_VERSION_MAJOR>4
    web_server.enableCORS(false);
#endif
    web_server.sendHeader(F("Access-Control-Allow-Origin"), origin);
  }
  web_server.send(httpCode, content_type, EMPTY_STRING);
#ifdef ESP8266
  // dont wait on 2.3.0. Memory returns just too slow.
  while ((ESP.getFreeHeap() < freeBeforeSend) &&
         !timeOutReached(beginWait + timeout)) {
    #ifndef BUILD_NO_RAM_TRACKER
    checkRAM(F("duringHeaderTX"));
    #endif
    delay(1);
  }
#endif
#endif // if defined(ESP8266) && defined(ARDUINO_ESP8266_RELEASE_2_3_0)
  delay(0);
}

#include "../DataStructs/PluginStats_timestamp.h"

#if FEATURE_PLUGIN_STATS

# include "../Globals/ESPEasy_time.h"
# include "../Helpers/ESPEasy_time_calc.h"

PluginStats_timestamp::PluginStats_timestamp(bool useHighRes)
  : _internal_to_micros_ratio(useHighRes
? 20000ull   // 1/50 sec resolution
: 100000ull) // 1/10 sec resolution
{}

PluginStats_timestamp::~PluginStats_timestamp()
{}

bool PluginStats_timestamp::push(const int64_t& timestamp_sysmicros)
{
  return _timestamps.push(systemMicros_to_internalTimestamp(timestamp_sysmicros));
}

bool PluginStats_timestamp::updateLast(const int64_t& timestamp_sysmicros)
{
  const size_t nrElements = _timestamps.size();

  if (nrElements == 0) { return false; }
  return _timestamps.set(nrElements - 1, systemMicros_to_internalTimestamp(timestamp_sysmicros));
}

void PluginStats_timestamp::clear()
{
  _timestamps.clear();
}

void PluginStats_timestamp::processTimeSet(const double& time_offset)
{
  // Check to see if there was a unix time set before the system time was set
  // For example when receiving data from a p2p node

  /*
     const uint64_t cur_micros    = getMicros64();
     const uint64_t offset_micros = time_offset * 1000000ull;
     const size_t   nrSamples     = _timestamps.size();

     // GMT  Wed Jan 01 2020 00:00:00 GMT+0000
     const int64_t unixTime_20200101 = 1577836800ll * _internal_to_micros_ratio;

     for (PluginStatsTimestamps_t::index_t i = 0; i < nrSamples; ++i) {
       if (_timestamps[i] < unixTime_20200101) {
         _timestamps.set(i, _timestamps[i] + time_offset);
       }
     }
   */
}

int64_t PluginStats_timestamp::getTimestamp(int lastNrSamples) const
{
  if ((_timestamps.size() == 0) || (_timestamps.size() < abs(lastNrSamples))) { return 0u; }

  PluginStatsTimestamps_t::index_t i = 0;

  if (lastNrSamples > 0) {
    i = _timestamps.size() - lastNrSamples;
  } else if (lastNrSamples < 0) {
    i = abs(lastNrSamples) - 1;
  }

  if (i < _timestamps.size()) {
    return internalTimestamp_to_systemMicros(_timestamps[i]);
  }
  return 0u;
}

uint32_t PluginStats_timestamp::getFullPeriodInSec(uint32_t& time_frac) const
{
  const size_t nrSamples = _timestamps.size();

  time_frac = 0u;

  if (nrSamples <= 1) {
    return 0u;
  }

  const int64_t start = internalTimestamp_to_systemMicros(_timestamps[0]);
  const int64_t end   = internalTimestamp_to_systemMicros(_timestamps[nrSamples - 1]);

  const int64_t period_usec = (end < start) ? (start - end) : (end - start);

  return micros_to_sec_time_frac(period_usec, time_frac);
}

int64_t PluginStats_timestamp::operator[](PluginStatsTimestamps_t::index_t index) const
{
  if (index < _timestamps.size()) {
    return internalTimestamp_to_systemMicros(_timestamps[index]);
  }
  return 0u;
}

uint32_t PluginStats_timestamp::systemMicros_to_internalTimestamp(const int64_t& timestamp_sysmicros) const
{
  return static_cast<uint32_t>(timestamp_sysmicros / _internal_to_micros_ratio);
}

int64_t PluginStats_timestamp::internalTimestamp_to_systemMicros(const uint32_t& internalTimestamp) const
{
  const uint64_t cur_micros    = getMicros64();
  const uint64_t overflow_step = 4294967296ull * _internal_to_micros_ratio;

  uint64_t sysMicros = static_cast<uint64_t>(internalTimestamp) * _internal_to_micros_ratio;

  // Try to get in the range of the current system micros
  // This only does play a role in high res mode, when uptime is over 994 days.
  while ((sysMicros + overflow_step) < cur_micros) {
    sysMicros += overflow_step;
  }
  return sysMicros;
}

#endif // if FEATURE_PLUGIN_STATS

#include "../DataStructs/C013_p2p_SensorDataStruct.h"

#ifdef USES_C013

# include "../DataStructs/NodeStruct.h"
# include "../Globals/Nodes.h"
# include "../Globals/ESPEasy_time.h"
# include "../Globals/Plugins.h"

# include "../CustomBuild/CompiletimeDefines.h"

bool C013_SensorDataStruct::prepareForSend()
{
  sourceNodeBuild = get_build_nr();
  checksum.clear();

  if (sourceNodeBuild >= 20871) {
    if (node_time.systemTimePresent()) {
      uint32_t unix_time_frac{};
      timestamp_sec  = node_time.getUnixTime(unix_time_frac);
      timestamp_frac = unix_time_frac >> 16;
    }


    // Make sure to add checksum as last step
    constexpr unsigned len_upto_checksum = offsetof(C013_SensorDataStruct, checksum);

    const ShortChecksumType tmpChecksum(
      reinterpret_cast<const uint8_t *>(this),
      sizeof(C013_SensorDataStruct),
      len_upto_checksum);

    checksum = tmpChecksum;
  }

  return validTaskIndex(sourceTaskIndex) &&
         validTaskIndex(destTaskIndex);
}

bool C013_SensorDataStruct::setData(const uint8_t *data, size_t size)
{
  // First clear entire struct
  memset(this, 0, sizeof(C013_SensorDataStruct));

  if (size < 6) {
    return false;
  }

  if ((data[0] != 255) || // header
      (data[1] != 5)) {   // ID
    return false;
  }

  constexpr unsigned len_upto_checksum = offsetof(C013_SensorDataStruct, checksum);
  const ShortChecksumType tmpChecksum(
    data,
    size,
    len_upto_checksum);


  // Need to keep track of different possible versions of data which still need to be supported.
  // Really old versions of ESPEasy might send upto 80 bytes of uninitialized data
  // meaning for sizes > 24 bytes we may need to check the version of ESPEasy running on the node.
  if (size > sizeof(C013_SensorDataStruct)) {
    size = sizeof(C013_SensorDataStruct);
  }
  NodeStruct *sourceNode = Nodes.getNode(data[2]); // sourceUnit

  if (sourceNode != nullptr) {
    if (sourceNode->build < 20871) {
      if (size > 24) {
        size = 24;
      }
    }
  }

  if (size <= 24) {
    deviceNumber = INVALID_PLUGIN_ID;
    sensorType   = Sensor_VType::SENSOR_TYPE_NONE;

    if (sourceNode != nullptr) {
      sourceNodeBuild = sourceNode->build;
    }
  }

  memcpy(this, data, size);

  if (checksum.isSet()) {
    if (!(tmpChecksum == checksum)) {
      return false;
    }
  }

  return validTaskIndex(sourceTaskIndex) &&
         validTaskIndex(destTaskIndex);
}

bool C013_SensorDataStruct::matchesPluginID(pluginID_t pluginID) const
{
  if ((deviceNumber.value == 255) || !validPluginID(deviceNumber) || !validPluginID(pluginID)) {
    // Was never set, so probably received data from older node.
    return true;
  }
  return pluginID == deviceNumber;
}

bool C013_SensorDataStruct::matchesSensorType(Sensor_VType sensor_type) const
{
  if ((deviceNumber.value == 255) || (sensorType == Sensor_VType::SENSOR_TYPE_NONE)) {
    // Was never set, so probably received data from older node.
    return true;
  }
  return sensorType == sensor_type;
}

#endif // ifdef USES_C013

#include "../DataStructs/TimingStats.h"

#if FEATURE_TIMING_STATS

# include "../DataTypes/ESPEasy_plugin_functions.h"
# include "../Globals/CPlugins.h"
# include "../Helpers/_CPlugin_Helper.h"
# include "../Helpers/StringConverter.h"

std::map<int, TimingStats> pluginStats;
std::map<int, TimingStats> controllerStats;
std::map<TimingStatsElements, TimingStats> miscStats;
unsigned long timingstats_last_reset(0);


void TimingStats::add(int32_t duration_usec) {
  // Max duration in usec is roughly 35 minutes.
  // For timing stats more than enough
  if (duration_usec < 0) return;
  _timeTotal += static_cast<uint64_t>(duration_usec);
  ++_count;

  if (static_cast<uint32_t>(duration_usec) > _maxVal) { _maxVal = duration_usec; }

  if (static_cast<uint32_t>(duration_usec) < _minVal) { _minVal = duration_usec; }
}

void TimingStats::reset() {
  _timeTotal = 0u;
  _count     = 0u;
  _maxVal    = 0u;
  _minVal    = 4294967295u;
}

bool TimingStats::isEmpty() const {
  return _count == 0u;
}

float TimingStats::getAvg() const {
  if (_count == 0) { return 0.0f; }
  return static_cast<float>(_timeTotal) / static_cast<float>(_count);
}

uint32_t TimingStats::getMinMax(uint32_t& minVal, uint32_t& maxVal) const {
  minVal = _minVal;
  maxVal = _maxVal;
  return _count;
}

bool TimingStats::thresholdExceeded(const uint32_t& threshold) const {
  if (_count == 0) {
    return false;
  }
  return _maxVal > threshold;
}

/********************************************************************************************\
   Functions used for displaying timing stats
 \*********************************************************************************************/
const __FlashStringHelper* getPluginFunctionName(int function) {
  switch (function) {
    case PLUGIN_INIT_ALL:              return F("INIT_ALL");
    case PLUGIN_INIT:                  return F("INIT");
    case PLUGIN_READ:                  return F("READ");
    case PLUGIN_ONCE_A_SECOND:         return F("ONCE_A_SECOND");
    case PLUGIN_TEN_PER_SECOND:        return F("TEN_PER_SECOND");
    case PLUGIN_DEVICE_ADD:            return F("DEVICE_ADD");
    case PLUGIN_EVENTLIST_ADD:         return F("EVENTLIST_ADD");
    case PLUGIN_WEBFORM_SAVE:          return F("WEBFORM_SAVE");
    case PLUGIN_WEBFORM_LOAD:          return F("WEBFORM_LOAD");
    case PLUGIN_WEBFORM_SHOW_VALUES:   return F("WEBFORM_SHOW_VALUES");
    case PLUGIN_FORMAT_USERVAR:        return F("FORMAT_USERVAR");
    case PLUGIN_GET_DEVICENAME:        return F("GET_DEVICENAME");
    case PLUGIN_GET_DEVICEVALUENAMES:  return F("GET_DEVICEVALUENAMES");
    case PLUGIN_GET_DEVICEVALUECOUNT:  return F("GET_DEVICEVALUECOUNT");
    case PLUGIN_GET_DEVICEVTYPE:       return F("GET_DEVICEVTYPE");
    case PLUGIN_WRITE:                 return F("WRITE");
    case PLUGIN_WEBFORM_SHOW_CONFIG:   return F("WEBFORM_SHOW_CONFIG");
    #if FEATURE_PLUGIN_STATS
    case PLUGIN_WEBFORM_LOAD_SHOW_STATS: return F("WEBFORM_LOAD_SHOW_STATS");
    #endif
    case PLUGIN_SERIAL_IN:             return F("SERIAL_IN");
    case PLUGIN_UDP_IN:                return F("UDP_IN");
    case PLUGIN_CLOCK_IN:              return F("CLOCK_IN");
    case PLUGIN_TASKTIMER_IN:          return F("TASKTIMER_IN");
    case PLUGIN_FIFTY_PER_SECOND:      return F("FIFTY_PER_SECOND");
    case PLUGIN_SET_CONFIG:            return F("SET_CONFIG");
    case PLUGIN_GET_DEVICEGPIONAMES:   return F("GET_DEVICEGPIONAMES");
    case PLUGIN_EXIT:                  return F("EXIT");
    case PLUGIN_GET_CONFIG_VALUE:      return F("GET_CONFIG");
//    case PLUGIN_UNCONDITIONAL_POLL:    return F("UNCONDITIONAL_POLL");
    case PLUGIN_REQUEST:               return F("REQUEST");
    case PLUGIN_PROCESS_CONTROLLER_DATA: return F("PROCESS_CONTROLLER_DATA");
    case PLUGIN_I2C_GET_ADDRESS:       return F("I2C_CHECK_DEVICE");
    case PLUGIN_READ_ERROR_OCCURED:    return F("PLUGIN_READ_ERROR_OCCURED");
  }
  return F("Unknown");
}

bool mustLogFunction(int function) {
  if (!Settings.EnableTimingStats()) { return false; }

  switch (function) {
//    case PLUGIN_INIT_ALL:              return false;
//    case PLUGIN_INIT:                  return false;
    case PLUGIN_READ:                  return true;
    case PLUGIN_ONCE_A_SECOND:         return true;
    case PLUGIN_TEN_PER_SECOND:        return true;
//    case PLUGIN_DEVICE_ADD:            return false;
//    case PLUGIN_EVENTLIST_ADD:         return false;
//    case PLUGIN_WEBFORM_SAVE:          return false;
//    case PLUGIN_WEBFORM_LOAD:          return false;
//    case PLUGIN_WEBFORM_SHOW_VALUES:   return false;
    case PLUGIN_FORMAT_USERVAR:        return true;
    case PLUGIN_GET_DEVICENAME:        return true;
//    case PLUGIN_GET_DEVICEVALUENAMES:  return false;
//    case PLUGIN_GET_DEVICEVALUECOUNT:  return true;
//    case PLUGIN_GET_DEVICEVTYPE:       return true;
    case PLUGIN_WRITE:                 return true;
//    case PLUGIN_WEBFORM_SHOW_CONFIG:   return false;
    case PLUGIN_SERIAL_IN:             return true;
//    case PLUGIN_UDP_IN:                return false;
//    case PLUGIN_CLOCK_IN:              return false;
    case PLUGIN_TASKTIMER_IN:          return true;
    case PLUGIN_FIFTY_PER_SECOND:      return true;
//    case PLUGIN_SET_CONFIG:            return false;
//    case PLUGIN_GET_DEVICEGPIONAMES:   return false;
//    case PLUGIN_EXIT:                  return false;
//    case PLUGIN_GET_CONFIG_VALUE:      return false;
//    case PLUGIN_UNCONDITIONAL_POLL:    return false;
    case PLUGIN_REQUEST:               return true;
    case PLUGIN_I2C_GET_ADDRESS:       return true;
    case PLUGIN_PROCESS_CONTROLLER_DATA: return true;
    case PLUGIN_READ_ERROR_OCCURED:    return true;
  }
  return false;
}

const __FlashStringHelper* getCPluginCFunctionName(CPlugin::Function function) {
  switch (function) {
    case CPlugin::Function::CPLUGIN_PROTOCOL_ADD:              return F("CPLUGIN_PROTOCOL_ADD");
    case CPlugin::Function::CPLUGIN_CONNECT_SUCCESS:           return F("CPLUGIN_CONNECT_SUCCESS");
    case CPlugin::Function::CPLUGIN_CONNECT_FAIL:              return F("CPLUGIN_CONNECT_FAIL");
    case CPlugin::Function::CPLUGIN_PROTOCOL_TEMPLATE:         return F("CPLUGIN_PROTOCOL_TEMPLATE");
    case CPlugin::Function::CPLUGIN_PROTOCOL_SEND:             return F("CPLUGIN_PROTOCOL_SEND");
    case CPlugin::Function::CPLUGIN_PROTOCOL_RECV:             return F("CPLUGIN_PROTOCOL_RECV");
    case CPlugin::Function::CPLUGIN_GET_DEVICENAME:            return F("CPLUGIN_GET_DEVICENAME");
    case CPlugin::Function::CPLUGIN_WEBFORM_SAVE:              return F("CPLUGIN_WEBFORM_SAVE");
    case CPlugin::Function::CPLUGIN_WEBFORM_LOAD:              return F("CPLUGIN_WEBFORM_LOAD");
    case CPlugin::Function::CPLUGIN_GET_PROTOCOL_DISPLAY_NAME: return F("CPLUGIN_GET_PROTOCOL_DISPLAY_NAME");
    case CPlugin::Function::CPLUGIN_TASK_CHANGE_NOTIFICATION:  return F("CPLUGIN_TASK_CHANGE_NOTIFICATION");
    case CPlugin::Function::CPLUGIN_INIT:                      return F("CPLUGIN_INIT");
    case CPlugin::Function::CPLUGIN_UDP_IN:                    return F("CPLUGIN_UDP_IN");
    case CPlugin::Function::CPLUGIN_FLUSH:                     return F("CPLUGIN_FLUSH");
    case CPlugin::Function::CPLUGIN_TEN_PER_SECOND:            return F("CPLUGIN_TEN_PER_SECOND");
    case CPlugin::Function::CPLUGIN_FIFTY_PER_SECOND:          return F("CPLUGIN_FIFTY_PER_SECOND");
    case CPlugin::Function::CPLUGIN_INIT_ALL:                  return F("CPLUGIN_INIT_ALL");
    case CPlugin::Function::CPLUGIN_EXIT:                      return F("CPLUGIN_EXIT");
    case CPlugin::Function::CPLUGIN_WRITE:                     return F("CPLUGIN_WRITE");

    case CPlugin::Function::CPLUGIN_GOT_CONNECTED:
    case CPlugin::Function::CPLUGIN_GOT_INVALID:
    case CPlugin::Function::CPLUGIN_INTERVAL:
    case CPlugin::Function::CPLUGIN_ACKNOWLEDGE:
    case CPlugin::Function::CPLUGIN_WEBFORM_SHOW_HOST_CONFIG:
      break;
  }
  return F("Unknown");
}

bool mustLogCFunction(CPlugin::Function function) {
  if (!Settings.EnableTimingStats()) { return false; }

  switch (function) {
    case CPlugin::Function::CPLUGIN_PROTOCOL_ADD:              return false;
    case CPlugin::Function::CPLUGIN_CONNECT_SUCCESS:           return true;
    case CPlugin::Function::CPLUGIN_CONNECT_FAIL:              return true;
    case CPlugin::Function::CPLUGIN_PROTOCOL_TEMPLATE:         return false;
    case CPlugin::Function::CPLUGIN_PROTOCOL_SEND:             return true;
    case CPlugin::Function::CPLUGIN_PROTOCOL_RECV:             return true;
    case CPlugin::Function::CPLUGIN_GET_DEVICENAME:            return false;
    case CPlugin::Function::CPLUGIN_WEBFORM_SAVE:              return false;
    case CPlugin::Function::CPLUGIN_WEBFORM_LOAD:              return false;
    case CPlugin::Function::CPLUGIN_GET_PROTOCOL_DISPLAY_NAME: return false;
    case CPlugin::Function::CPLUGIN_TASK_CHANGE_NOTIFICATION:  return false;
    case CPlugin::Function::CPLUGIN_INIT:                      return false;
    case CPlugin::Function::CPLUGIN_UDP_IN:                    return true;
    case CPlugin::Function::CPLUGIN_FLUSH:                     return false;
    case CPlugin::Function::CPLUGIN_TEN_PER_SECOND:            return true;
    case CPlugin::Function::CPLUGIN_FIFTY_PER_SECOND:          return true;
    case CPlugin::Function::CPLUGIN_INIT_ALL:                  return false;
    case CPlugin::Function::CPLUGIN_EXIT:                      return false;
    case CPlugin::Function::CPLUGIN_WRITE:                     return true;

    case CPlugin::Function::CPLUGIN_GOT_CONNECTED:
    case CPlugin::Function::CPLUGIN_GOT_INVALID:
    case CPlugin::Function::CPLUGIN_INTERVAL:
    case CPlugin::Function::CPLUGIN_ACKNOWLEDGE:
    case CPlugin::Function::CPLUGIN_WEBFORM_SHOW_HOST_CONFIG:
      break;
  }
  return false;
}

// Return flash string type to reduce bin size
const __FlashStringHelper* getMiscStatsName_F(TimingStatsElements stat) {
  switch (stat) {
    case TimingStatsElements::LOADFILE_STATS:             return F("Load File");
    case TimingStatsElements::SAVEFILE_STATS:             return F("Save File");
    case TimingStatsElements::LOOP_STATS:                 return F("Loop");
    case TimingStatsElements::PLUGIN_CALL_50PS:           return F("Plugin call 50 p/s");
    case TimingStatsElements::PLUGIN_CALL_10PS:           return F("Plugin call 10 p/s");
    case TimingStatsElements::PLUGIN_CALL_10PSU:          return F("Plugin call 10 p/s U");
    case TimingStatsElements::PLUGIN_CALL_1PS:            return F("Plugin call  1 p/s");
    case TimingStatsElements::CPLUGIN_CALL_50PS:          return F("CPlugin call 50 p/s");
    case TimingStatsElements::CPLUGIN_CALL_10PS:          return F("CPlugin call 10 p/s");
    case TimingStatsElements::SENSOR_SEND_TASK:           return F("SensorSendTask()");
    case TimingStatsElements::COMMAND_EXEC_INTERNAL:      return F("Exec Internal Command");
    case TimingStatsElements::COMMAND_DECODE_INTERNAL:    return F("Decode Internal Command");
    case TimingStatsElements::CONSOLE_LOOP:               return F("Console loop()");
    case TimingStatsElements::CONSOLE_WRITE_SERIAL:       return F("Console out");
    case TimingStatsElements::SEND_DATA_STATS:            return F("sendData()");
    case TimingStatsElements::COMPUTE_FORMULA_STATS:      return F("Compute formula");
    case TimingStatsElements::COMPUTE_STATS:              return F("Compute()");
    case TimingStatsElements::PLUGIN_CALL_DEVICETIMER_IN: return F("PLUGIN_DEVICETIMER_IN");
    case TimingStatsElements::SET_NEW_TIMER:              return F("setNewTimerAt()");
    case TimingStatsElements::MQTT_DELAY_QUEUE:           return F("Delay queue MQTT");
    case TimingStatsElements::TRY_CONNECT_HOST_TCP:       return F("try_connect_host() (TCP)");
    case TimingStatsElements::TRY_CONNECT_HOST_UDP:       return F("try_connect_host() (UDP)");
    case TimingStatsElements::HOST_BY_NAME_STATS:         return F("hostByName()");
    case TimingStatsElements::CONNECT_CLIENT_STATS:       return F("connectClient()");
    case TimingStatsElements::LOAD_CUSTOM_TASK_STATS:     return F("LoadCustomTaskSettings()");
    case TimingStatsElements::WIFI_ISCONNECTED_STATS:     return F("WiFi.isConnected()");
    case TimingStatsElements::WIFI_NOTCONNECTED_STATS:    return F("WiFi.isConnected() (fail)");
    case TimingStatsElements::LOAD_TASK_SETTINGS:         return F("LoadTaskSettings()");
    case TimingStatsElements::SAVE_TASK_SETTINGS:         return F("SaveTaskSettings()");
    case TimingStatsElements::LOAD_CONTROLLER_SETTINGS:   return F("LoadControllerSettings()");
    #ifdef ESP32
    case TimingStatsElements::LOAD_CONTROLLER_SETTINGS_C: return F("LoadControllerSettings() (cached)");
    #endif
    case TimingStatsElements::SAVE_CONTROLLER_SETTINGS:   return F("SaveControllerSettings()");
    case TimingStatsElements::TRY_OPEN_FILE:              return F("TryOpenFile()");
    case TimingStatsElements::FS_GC_SUCCESS:              return F("ESPEASY_FS GC success");
    case TimingStatsElements::FS_GC_FAIL:                 return F("ESPEASY_FS GC fail");
    case TimingStatsElements::RULES_PROCESSING:           return F("rulesProcessing()");
    case TimingStatsElements::RULES_PARSE_LINE:           return F("parseCompleteNonCommentLine()");
    case TimingStatsElements::RULES_PROCESS_MATCHED:      return F("processMatchedRule()");
    case TimingStatsElements::RULES_MATCH:                return F("rulesMatch()");
    case TimingStatsElements::GRAT_ARP_STATS:             return F("sendGratuitousARP()");
    case TimingStatsElements::SAVE_TO_RTC:                return F("saveToRTC()");
    case TimingStatsElements::BACKGROUND_TASKS:           return F("backgroundtasks()");
    case TimingStatsElements::UPDATE_RTTTL:               return F("update_rtttl()");
    case TimingStatsElements::CHECK_UDP:                  return F("checkUDP()");
    case TimingStatsElements::C013_SEND_UDP:              return F("C013_sendUDP() SUCCESS");
    case TimingStatsElements::C013_SEND_UDP_FAIL:         return F("C013_sendUDP() FAIL");
    case TimingStatsElements::C013_RECEIVE_SENSOR_DATA:   return F("C013 Receive sensor data");
    case TimingStatsElements::WEBSERVER_HANDLE_CLIENT:    return F("web_server.handleClient()");
    case TimingStatsElements::PROCESS_SYSTEM_EVENT_QUEUE: return F("process_system_event_queue()");
    case TimingStatsElements::FORMAT_USER_VAR:            return F("doFormatUserVar()");
    case TimingStatsElements::IS_NUMERICAL:               return F("isNumerical()");
    case TimingStatsElements::HANDLE_SCHEDULER_IDLE:      return F("handle_schedule() idle");
    case TimingStatsElements::HANDLE_SCHEDULER_TASK:      return F("handle_schedule() task");
    case TimingStatsElements::PARSE_TEMPLATE_PADDED:      return F("parseTemplate_padded()");
    case TimingStatsElements::PARSE_SYSVAR:               return F("parseSystemVariables()");
    case TimingStatsElements::PARSE_SYSVAR_NOCHANGE:      return F("parseSystemVariables() No change");
    case TimingStatsElements::HANDLE_SERVING_WEBPAGE:     return F("handle webpage");
    case TimingStatsElements::HANDLE_SERVING_WEBPAGE_JSON: return F("handle webpage JSON");
    case TimingStatsElements::WIFI_SCAN_ASYNC:            return F("WiFi Scan Async");
    case TimingStatsElements::WIFI_SCAN_SYNC:             return F("WiFi Scan Sync (blocking)");
    case TimingStatsElements::NTP_SUCCESS:                return F("NTP Success");
    case TimingStatsElements::NTP_FAIL:                   return F("NTP Fail");
    case TimingStatsElements::SYSTIME_UPDATED:            return F("Systime Set");
    case TimingStatsElements::C018_AIR_TIME:              return F("C018 LoRa TTN - Air Time");
#ifdef LIMIT_BUILD_SIZE
    default: break;
#else
    // Include all elements of the enum, to allow the compiler to check if we missed some
    case TimingStatsElements::C001_DELAY_QUEUE:
    case TimingStatsElements::C002_DELAY_QUEUE:
    case TimingStatsElements::C003_DELAY_QUEUE:
    case TimingStatsElements::C004_DELAY_QUEUE:
    case TimingStatsElements::C005_DELAY_QUEUE:
    case TimingStatsElements::C006_DELAY_QUEUE:
    case TimingStatsElements::C007_DELAY_QUEUE:
    case TimingStatsElements::C008_DELAY_QUEUE:
    case TimingStatsElements::C009_DELAY_QUEUE:
    case TimingStatsElements::C010_DELAY_QUEUE:
    case TimingStatsElements::C011_DELAY_QUEUE:
    case TimingStatsElements::C012_DELAY_QUEUE:
    case TimingStatsElements::C013_DELAY_QUEUE:
    case TimingStatsElements::C014_DELAY_QUEUE:
    case TimingStatsElements::C015_DELAY_QUEUE:
    case TimingStatsElements::C016_DELAY_QUEUE:
    case TimingStatsElements::C017_DELAY_QUEUE:
    case TimingStatsElements::C018_DELAY_QUEUE:
    case TimingStatsElements::C019_DELAY_QUEUE:
    case TimingStatsElements::C020_DELAY_QUEUE:
    case TimingStatsElements::C021_DELAY_QUEUE:
    case TimingStatsElements::C022_DELAY_QUEUE:
    case TimingStatsElements::C023_DELAY_QUEUE:
    case TimingStatsElements::C024_DELAY_QUEUE:
    case TimingStatsElements::C025_DELAY_QUEUE:
      break;

#endif
  }
  return F("Unknown");
}

String getMiscStatsName(TimingStatsElements stat) {
  if ((stat >= TimingStatsElements::C001_DELAY_QUEUE) && 
      (stat <= TimingStatsElements::C025_DELAY_QUEUE)) {
    return concat(
      F("Delay queue "),
      get_formatted_Controller_number(static_cast<cpluginID_t>(static_cast<int>(stat) - static_cast<int>(TimingStatsElements::C001_DELAY_QUEUE) + 1)));
  }
  return getMiscStatsName_F(static_cast<TimingStatsElements>(stat));
}

void stopTimerTask(deviceIndex_t T, int F, uint32_t statisticsTimerStart)
{
  if (mustLogFunction(F)) { pluginStats[static_cast<int>(T.value) * 256 + (F)].add(usecPassedSince_fast(statisticsTimerStart)); }
}

void stopTimerController(protocolIndex_t T, CPlugin::Function F, uint32_t statisticsTimerStart)
{
  if (mustLogCFunction(F)) { controllerStats[static_cast<int>(T) * 256 + static_cast<int>(F)].add(usecPassedSince_fast(statisticsTimerStart)); }
}

void stopTimer(TimingStatsElements L, uint32_t statisticsTimerStart)
{
  if (Settings.EnableTimingStats()) { miscStats[L].add(usecPassedSince_fast(statisticsTimerStart)); }
}

void addMiscTimerStat(TimingStatsElements L, int32_t T)
{
  if (Settings.EnableTimingStats()) { miscStats[L].add(T); }
}

#endif // if FEATURE_TIMING_STATS

#include "../DataStructs/LogStruct.h"

#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/StringConverter.h"

void LogStruct::add_end() {
  if (isFull()) {
    read_idx = nextIndex(read_idx);
  }
  write_idx = nextIndex(write_idx);
  is_full   = (write_idx == read_idx);
}

void LogStruct::add(const uint8_t loglevel, const String& line) {
  if (Message[write_idx].add(loglevel, line)) {
    add_end();
  }
}

void LogStruct::add(const uint8_t loglevel, String&& line) {
  if (Message[write_idx].add(loglevel, std::move(line))) {
    add_end();
  }
}

bool LogStruct::getNext(bool& logLinesAvailable, unsigned long& timestamp, String& message, uint8_t& loglevel) {
  lastReadTimeStamp = millis();
  logLinesAvailable = false;

  if (isEmpty()) {
    return false;
  }
  timestamp = Message[read_idx]._timestamp;
  message   = std::move(Message[read_idx]._message);
  loglevel  = Message[read_idx]._loglevel;
  clearOldest();

  if (!isEmpty()) {
    logLinesAvailable = true;
  }
  return true;
}


bool LogStruct::logActiveRead() {
  clearExpiredEntries();
  return timePassedSince(lastReadTimeStamp) < LOG_BUFFER_ACTIVE_READ_TIMEOUT;
}

void LogStruct::clearExpiredEntries() {
  unsigned int maxLoops = LOG_STRUCT_MESSAGE_LINES;

  while (maxLoops > 0) {
    --maxLoops;

    if (isEmpty() || // Nothing left
        !Message[read_idx].isExpired())
    {
      return;
    }
    clearOldest();
  }
}

void LogStruct::clearOldest() {
  if (!isEmpty()) {
    is_full = false;
    Message[read_idx].clear();
    read_idx = nextIndex(read_idx);
  }
}

#include "../DataStructs/Scheduler_ConstIntervalTimerID.h"

/*

ConstIntervalTimerID::ConstIntervalTimerID(SchedulerIntervalTimer_e timer) :
  SchedulerTimerID(SchedulerTimerType_e::ConstIntervalTimer)
{
  id = static_cast<uint32_t>(timer);
}

SchedulerIntervalTimer_e ConstIntervalTimerID::getIntervalTimer() const
{
  return static_cast<SchedulerIntervalTimer_e>(id);
}

#ifndef BUILD_NO_DEBUG
String ConstIntervalTimerID::decode() const
{
  return toString(getIntervalTimer());
}

#endif // ifndef BUILD_NO_DEBUG
*/
#include "../DataStructs/Scheduler_RulesTimerID.h"

#include "../Helpers/StringConverter.h"

RulesTimerID::RulesTimerID(unsigned int timerIndex) :
  SchedulerTimerID(SchedulerTimerType_e::RulesTimer)
{
  setId(timerIndex);
}

#ifndef BUILD_NO_DEBUG
String RulesTimerID::decode() const
{
  return concat(F("Rules#Timer="), getId());
}

#endif // ifndef BUILD_NO_DEBUG

#include "../DataStructs/NTP_candidate.h"

#if FEATURE_ESPEASY_P2P

# include "../CustomBuild/CompiletimeDefines.h"
# include "../DataTypes/ESPEasyTimeSource.h"
# include "../Globals/Settings.h"
# include "../Helpers/ESPEasy_time_calc.h"
# include "../Helpers/StringConverter.h"


bool NTP_candidate_struct::set(const NodeStruct& node)
{
  if (node.unit == Settings.Unit) { return false; }

  if (node.unix_time_sec < get_build_unixtime()) { return false; }
  const timeSource_t timeSource = static_cast<timeSource_t>(node.timeSource);

  if (timeSource == timeSource_t::No_time_source) { return false; }

  // Only allow time from p2p nodes who only got it via p2p themselves as "last resource"
  const unsigned long p2p_source_penalty =
    isExternalTimeSource(timeSource)  ? 0 : 10000;
  const unsigned long time_wander_other =
    p2p_source_penalty + 
    computeExpectedWander(timeSource, node.lastUpdated); // node.lastUpdated is already set to "time passed since" when sent


  if (timePassedSince(_received_moment) > EXT_TIME_SOURCE_MIN_UPDATE_INTERVAL_MSEC
      && timeSource < timeSource_t::ESP_now_peer) 
  {
    // Every now and then refresh the collected time source, but only on reliable sources
    clear(); 
  }

  if ((_time_wander < 0) || (time_wander_other < static_cast<unsigned long>(_time_wander))) {
    _time_wander     = time_wander_other;
    _unix_time_sec   = node.unix_time_sec;
    _unix_time_frac  = node.unix_time_frac;
    _received_moment = millis();
    _unit            = node.unit;
    _timeSource      = timeSource;

    if (_first_received_moment == 0) {
      _first_received_moment = _received_moment;
    }
    # ifndef BUILD_NO_DEBUG

    if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
      String log = F("NTP  : Time candidate: ");
      log += node.getSummary();

      log += concat(F(" time: "), _unix_time_sec);
      log += ' ';
      log += toString(timeSource);
      log += concat(F(" est. wander: "), _time_wander);
      addLogMove(LOG_LEVEL_DEBUG, log);
    }
    # endif // ifndef BUILD_NO_DEBUG
    return true;
  }
  return false;
}

void NTP_candidate_struct::clear()
{
  _unix_time_sec         = 0;
  _unix_time_frac        = 0;
  _time_wander           = -1;
  _received_moment       = 0;
  _first_received_moment = 0;
  _timeSource            = timeSource_t::No_time_source;
}

timeSource_t NTP_candidate_struct::getUnixTime(
  double & unix_time_d,
  int32_t& wander,
  uint8_t& unit) const
{
  if ((_unix_time_sec == 0) ||
      (_time_wander < 0) ||
      (_received_moment == 0) ||
      (_timeSource == timeSource_t::No_time_source)) {
    return timeSource_t::No_time_source;
  }

  if (timePassedSince(_first_received_moment) < 30000) {
    // Make sure to allow for enough time to collect the "best" option.
    return timeSource_t::No_time_source;
  }

  unit = _unit;

  const int32_t timePassed = timePassedSince(_received_moment);

  const int64_t unix_time_usec =
    sec_time_frac_to_Micros(_unix_time_sec, _unix_time_frac) +
    (static_cast<int64_t>(timePassed) * 1000ll); // Add time since it was received

  unix_time_d = static_cast<double>(unix_time_usec) / 1000000.0;

  wander = updateExpectedWander(_time_wander, timePassed);

  // FIXME TD-er: Must somehow know whether the p2p node was seen via UDP or ESPEasy-NOW
  return timeSource_t::ESPEASY_p2p_UDP;
}

#endif // if FEATURE_ESPEASY_P2P

#include "../DataStructs/Scheduler_IntendedRebootTimerID.h"


IntendedRebootTimerID::IntendedRebootTimerID(IntendedRebootReason_e reason) :
  SchedulerTimerID(SchedulerTimerType_e::IntendedReboot)
{
  setId(static_cast<uint32_t>(reason));
}


#include "../DataStructs/ShortChecksumType.h"

#include "../Helpers/StringConverter.h"

#include <MD5Builder.h>

void ShortChecksumType::md5sumToShortChecksum(const uint8_t md5[16], uint8_t shortChecksum[4])
{
  memset(shortChecksum, 0, 4);

  for (uint8_t i = 0; i < 16; ++i) {
    // shortChecksum is XOR per 32 bit
    shortChecksum[i % 4] ^= md5[i];
  }
}

ShortChecksumType::ShortChecksumType(const ShortChecksumType& rhs)
{
  memcpy(_checksum, rhs._checksum, 4);
}

ShortChecksumType::ShortChecksumType(uint8_t checksum[4])
{
  memcpy(_checksum, checksum, 4);
}

ShortChecksumType::ShortChecksumType(const uint8_t *data,
                                     size_t         data_length)
{
  computeChecksum(_checksum, data, data_length, data_length, true);
}

ShortChecksumType::ShortChecksumType(const uint8_t *data,
                                     size_t         data_length,
                                     size_t         len_upto_checksum)
{
  computeChecksum(_checksum, data, data_length, len_upto_checksum, true);
}

ShortChecksumType::ShortChecksumType(const String strings[], size_t nrStrings)
{
  MD5Builder md5;

  md5.begin();

  for (size_t i = 0; i < nrStrings; ++i) {
    md5.add(strings[i].c_str());
  }
  md5.calculate();
  uint8_t tmp_md5[16] = { 0 };

  md5.getBytes(tmp_md5);
  md5sumToShortChecksum(tmp_md5, _checksum);
}

bool ShortChecksumType::computeChecksum(
  uint8_t        checksum[4],
  const uint8_t *data,
  size_t         data_length,
  size_t         len_upto_checksum,
  bool           updateChecksum)
{
  if (len_upto_checksum > data_length) { len_upto_checksum = data_length; }
  MD5Builder md5;

  md5.begin();

  if (len_upto_checksum > 0) {
    // MD5Builder::add has non-const argument
    md5.add(const_cast<uint8_t *>(data), len_upto_checksum);
  }

  if ((len_upto_checksum + 4) < data_length) {
    data += len_upto_checksum + 4;
    const int len_after_checksum = data_length - 4 - len_upto_checksum;

    if (len_after_checksum > 0) {
      // MD5Builder::add has non-const argument
      md5.add(const_cast<uint8_t *>(data), len_after_checksum);
    }
  }
  md5.calculate();
  uint8_t tmp_checksum[4] = { 0 };

  {
    uint8_t tmp_md5[16] = { 0 };
    md5.getBytes(tmp_md5);
    md5sumToShortChecksum(tmp_md5, tmp_checksum);
  }

  if (memcmp(tmp_checksum, checksum, 4) != 0) {
    // Data has changed, copy computed checksum
    if (updateChecksum) {
      memcpy(checksum, tmp_checksum, 4);
    }
    return false;
  }
  return true;
}

void ShortChecksumType::getChecksum(uint8_t checksum[4]) const {
  memcpy(checksum, _checksum, 4);
}

void ShortChecksumType::setChecksum(const uint8_t checksum[4]) {
  memcpy(_checksum, checksum, 4);
}

bool ShortChecksumType::matchChecksum(const uint8_t checksum[4]) const {
  return memcmp(_checksum, checksum, 4) == 0;
}

bool ShortChecksumType::operator==(const ShortChecksumType& rhs) const {
  return memcmp(_checksum, rhs._checksum, 4) == 0;
}

ShortChecksumType& ShortChecksumType::operator=(const ShortChecksumType& rhs) {
  memcpy(_checksum, rhs._checksum, 4);
  return *this;
}

String ShortChecksumType::toString() const {
  return formatToHex_array(_checksum, 4);
}

bool ShortChecksumType::isSet() const {
  return
    _checksum[0] != 0 ||
    _checksum[1] != 0 ||
    _checksum[2] != 0 ||
    _checksum[3] != 0;
}

void ShortChecksumType::clear()
{
  memset(_checksum, 0, 4);
}

#include "../DataStructs/WiFi_AP_Candidates_NVS.h"

#ifdef ESP32
# include "../Helpers/ESPEasy_NVS_Helper.h"

# define WIFI_AP_CANDIDATE_NVS_KEY "WIFICANDIDATE"

struct WiFi_AP_Candidates_NVS_data_t {
  union {
    struct {
      uint8_t BSSID[6];
      uint8_t lastWiFiChannel;
      uint8_t lastWiFiSettingsIndex;
    } APdata;

    uint64_t rawdata{};
  };
};

bool WiFi_AP_Candidates_NVS::loadCandidate_from_NVS(WiFi_AP_Candidate& candidate)
{
  WiFi_AP_Candidates_NVS_data_t fromNVS;

  {
    ESPEasy_NVS_Helper preferences;

    if (!preferences.begin(F(WIFI_CONNECTION_NVS_NAMESPACE))) {
      return false;
    }

    if (!preferences.getPreference(F(WIFI_AP_CANDIDATE_NVS_KEY), fromNVS.rawdata)) {
      return false;
    }
  }
  candidate.bssid.set(fromNVS.APdata.BSSID);
  candidate.channel = fromNVS.APdata.lastWiFiChannel;
  candidate.index   = fromNVS.APdata.lastWiFiSettingsIndex;
  return true;
}

void WiFi_AP_Candidates_NVS::currentConnection_to_NVS(const WiFi_AP_Candidate& candidate)
{
  ESPEasy_NVS_Helper preferences;

  preferences.begin(F(WIFI_CONNECTION_NVS_NAMESPACE));
  WiFi_AP_Candidates_NVS_data_t toNVS;

  candidate.bssid.get(toNVS.APdata.BSSID);
  toNVS.APdata.lastWiFiChannel       = candidate.channel;
  toNVS.APdata.lastWiFiSettingsIndex = candidate.index;

  preferences.setPreference(F(WIFI_AP_CANDIDATE_NVS_KEY), toNVS.rawdata);
}

void WiFi_AP_Candidates_NVS::clear_from_NVS()
{
  ESPEasy_NVS_Helper preferences;

  if (preferences.begin(F(WIFI_CONNECTION_NVS_NAMESPACE))) {
    preferences.remove(F(WIFI_AP_CANDIDATE_NVS_KEY));
  }
}

#endif // ifdef ESP32

#include "../DataStructs/NodesHandler.h"

#include "../../ESPEasy_common.h"

#if FEATURE_ESPEASY_P2P
#include "../../ESPEasy-Globals.h"

#ifdef USES_ESPEASY_NOW
#include "../Globals/ESPEasy_now_peermanager.h"
#include "../Globals/ESPEasy_now_state.h"
#endif

#include "../Globals/EventQueue.h"

#include "../DataTypes/NodeTypeID.h"

#if FEATURE_MQTT
#include "../ESPEasyCore/Controller.h"
#endif

#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/ESPEasyWifi.h"
#include "../Globals/ESPEasy_time.h"
#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/MQTT.h"
#include "../Globals/NetworkState.h"
#include "../Globals/RTC.h"
#include "../Globals/Settings.h"
#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/Misc.h"
#include "../Helpers/PeriodicalActions.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringGenerator_System.h"

#define ESPEASY_NOW_ALLOWED_AGE_NO_TRACEROUTE  35000

bool NodesHandler::addNode(const NodeStruct& node)
{
  int8_t rssi = 0;
  MAC_address match_sta;
  MAC_address match_ap;
  MAC_address ESPEasy_NOW_MAC;

  bool isNewNode = true;

  // Erase any existing node with matching MAC address
  for (auto it = _nodes.begin(); it != _nodes.end(); )
  {
    const MAC_address sta = it->second.sta_mac;
    const MAC_address ap  = it->second.ap_mac;
    if ((!sta.all_zero() && node.match(sta)) || (!ap.all_zero() && node.match(ap))) {
      rssi = it->second.getRSSI();
      if (!sta.all_zero())
        match_sta = sta;
      if (!ap.all_zero())
        match_ap = ap;
      ESPEasy_NOW_MAC = it->second.ESPEasy_Now_MAC();

      isNewNode = false;
      {
        _nodes_mutex.lock();
        it = _nodes.erase(it);
        _nodes_mutex.unlock();
      }
    } else {
      ++it;
    }
  }
  bool ntp_candidate_updated = false;
  {
    _nodes_mutex.lock();
    {
      #ifdef USE_SECOND_HEAP
      // FIXME TD-er: Must check whether this is working well as the NodesMap is a std::map
      HeapSelectIram ephemeral;
      #endif
      _nodes[node.unit] = node;
    }
    // Make sure to set first as NTP candidate, as it was set to time since
    // last time sync by the sender node before sending.
    ntp_candidate_updated = _ntp_candidate.set(node);
    // Now set lastUpdated so we can keep track of its age.
    _nodes[node.unit].lastUpdated = millis();
    if (node.getRSSI() >= 0 && rssi < 0) {
      _nodes[node.unit].setRSSI(rssi);
    }
    const MAC_address node_ap(node.ap_mac);
    if (node_ap.all_zero()) {
      _nodes[node.unit].setAP_MAC(node_ap);
    }
    if (node.ESPEasy_Now_MAC().all_zero()) {
      _nodes[node.unit].setESPEasyNow_mac(ESPEasy_NOW_MAC);
    }
    _nodes_mutex.unlock();
  }

  // Check whether the current time source is considered "worse" than received from p2p node.
  if (!node_time.systemTimePresent() ||
      (node_time.getTimeSource() > timeSource_t::ESPEASY_p2p_UDP) ||
      (timePassedSince(node_time.lastSyncTime_ms) > EXT_TIME_SOURCE_MIN_UPDATE_INTERVAL_MSEC)) {
    double  unixTime{};
    uint8_t unit                  = 0;
    int32_t wander                = -1;
    const timeSource_t timeSource = _ntp_candidate.getUnixTime(unixTime, wander, unit);

    if (timeSource != timeSource_t::No_time_source) {
      bool shouldUpdate = false;

      if (!node_time.systemTimePresent()) {
        if (timeSource < timeSource_t::ESP_now_peer || getUptimeMinutes() > 0) {
          // After 1 minute we should have had at least 2 loops of p2p nodes announcing their time source.
          // Though no need to wait longer if we already got a good source
          shouldUpdate = true;
        }
      } else if (ntp_candidate_updated) {
        // Got a better time source
        shouldUpdate = true;
      }

      if (shouldUpdate) {
        node_time.setExternalTimeSource_withTimeWander(unixTime, timeSource, wander, unit);
      }
    }
  }

  if (isNewNode) {
    if (Settings.UseRules && (node.unit != 0))
    {
      // Generate event announcing new p2p node
      // TODO TD-er: Maybe also add other info like ESP type, IP-address, etc?
      eventQueue.addMove(strformat(
                           F("p2pNode#Connected=%d,'%s','%s'"),
                           node.unit,
                           node.getNodeName().c_str(),
                           formatSystemBuildNr(node.build).c_str()
                           ));
    }
  }

  return isNewNode;
}

#ifdef USES_ESPEASY_NOW
bool NodesHandler::addNode(const NodeStruct& node, const ESPEasy_now_traceroute_struct& traceRoute)
{
  const bool isNewNode = addNode(node);
  {
    _nodeStats_mutex.lock();
    _nodeStats[node.unit].setDiscoveryRoute(node.unit, traceRoute);
    _nodeStats_mutex.unlock();
  }

  ESPEasy_now_peermanager.addPeer(node.ESPEasy_Now_MAC(), node.channel);  

  if (!node.isThisNode()) {
    if (traceRoute.getDistance() != 255) {
      if (loglevelActiveFor(LOG_LEVEL_INFO)) {
        String log;
        if (reserve_special(log, 80)) {
          log  = F(ESPEASY_NOW_NAME);
          log += F(": Node: ");
          log += String(node.unit);
          log += F(" DiscoveryRoute received: ");
          log += traceRoute.toString();
          addLog(LOG_LEVEL_INFO, log);
        }
      }
    } else {}
  }
  return isNewNode;
}
#endif

bool NodesHandler::hasNode(uint8_t unit_nr) const
{
  return _nodes.find(unit_nr) != _nodes.end();
}

bool NodesHandler::hasNode(const uint8_t *mac) const
{
  return getNodeByMac(mac) != nullptr;
}

NodeStruct * NodesHandler::getNode(uint8_t unit_nr)
{
  auto it = _nodes.find(unit_nr);

  if (it == _nodes.end()) {
    return nullptr;
  }
  return &(it->second);
}

const NodeStruct * NodesHandler::getNode(uint8_t unit_nr) const
{
  auto it = _nodes.find(unit_nr);

  if (it == _nodes.end()) {
    return nullptr;
  }
  return &(it->second);
}

NodeStruct * NodesHandler::getNodeByMac(const MAC_address& mac)
{
  if (mac.all_zero()) {
    return nullptr;
  }
  delay(0);

  for (auto it = _nodes.begin(); it != _nodes.end(); ++it)
  {
    if (mac == it->second.sta_mac) {
      return &(it->second);
    }

    if (mac == it->second.ap_mac) {
      return &(it->second);
    }
  }
  return nullptr;
}

const NodeStruct * NodesHandler::getNodeByMac(const MAC_address& mac) const
{
  bool match_STA;

  return getNodeByMac(mac, match_STA);
}

const NodeStruct * NodesHandler::getNodeByMac(const MAC_address& mac, bool& match_STA) const
{
  if (mac.all_zero()) {
    return nullptr;
  }
  delay(0);

  for (auto it = _nodes.begin(); it != _nodes.end(); ++it)
  {
    if (mac == it->second.sta_mac) {
      match_STA = true;
      return &(it->second);
    }

    if (mac == it->second.ap_mac) {
      match_STA = false;
      return &(it->second);
    }
  }
  return nullptr;
}

const NodeStruct * NodesHandler::getPreferredNode() const {
  MAC_address dummy;

  return getPreferredNode_notMatching(dummy);
}

const NodeStruct* NodesHandler::getPreferredNode_notMatching(uint8_t unit_nr) const {
  MAC_address not_matching;
  if (unit_nr != 0 && unit_nr != 255) {
    const NodeStruct* node = getNode(unit_nr);
    if (node != nullptr) {
      not_matching = node->ESPEasy_Now_MAC();
    }
  }
  return getPreferredNode_notMatching(not_matching);
}

const NodeStruct * NodesHandler::getPreferredNode_notMatching(const MAC_address& not_matching) const {
  MAC_address this_mac = NetworkMacAddress();
  const NodeStruct *thisNode = getNodeByMac(this_mac);
  const NodeStruct *reject   = getNodeByMac(not_matching);

  const NodeStruct *res = nullptr;

  for (auto it = _nodes.begin(); it != _nodes.end(); ++it)
  {
    if ((&(it->second) != reject) && (&(it->second) != thisNode)) {
      bool mustSet = false;
      if (res == nullptr) {
        mustSet = true;
      } else {
        #ifdef USES_ESPEASY_NOW

        uint8_t distance_new, distance_res = 255;

        const int successRate_new = getRouteSuccessRate(it->second.unit, distance_new);
        const int successRate_res = getRouteSuccessRate(res->unit, distance_res);

        if (successRate_new == 0 || successRate_res == 0) {
          // One of the nodes does not (yet) have a route.
          if (successRate_new == 0 && successRate_res == 0) {
            distance_new = it->second.distance;
            distance_res = res->distance;
          } else if (successRate_res == 0) {
            // The new one has a route, so must set the new one.
            distance_res = res->distance;
            if (distance_new < 255) {
              mustSet = true;
            }
          }
        }

        if (distance_new == distance_res) {
          if (successRate_new > successRate_res && distance_new < 255) {
            mustSet = true;
          }
        } else if (distance_new < distance_res) {
          if (it->second.getAge() < ESPEASY_NOW_ALLOWED_AGE_NO_TRACEROUTE) {
            // Only allow this new one if it was seen recently 
            // as it does not (yet) have a traceroute.
            mustSet = true;
          }
        }
        #else
        if (it->second < *res) {
            mustSet = true;
        }
        #endif
      }
      if (mustSet) {
        #ifdef USES_ESPEASY_NOW
        if (it->second.ESPEasyNowPeer && it->second.distance < 255) {
          res = &(it->second);
        }
        #else
        res = &(it->second);
        #endif
      }
    }
  }

/*
  #ifdef USES_ESPEASY_NOW
  if (res != nullptr)
  {
    uint8_t distance_res = 255;
    const int successRate_res = getRouteSuccessRate(res->unit, distance_res);
    if (distance_res == 255) {
      return nullptr;
    }
  }
  #endif
*/

  return res;
}

#ifdef USES_ESPEASY_NOW
const ESPEasy_now_traceroute_struct* NodesHandler::getTraceRoute(uint8_t unit) const
{
  auto trace_it = _nodeStats.find(unit);
  if (trace_it == _nodeStats.end()) {
    return nullptr;
  }
  return trace_it->second.bestRoute();
}

const ESPEasy_now_traceroute_struct* NodesHandler::getDiscoveryRoute(uint8_t unit) const
{
  auto trace_it = _nodeStats.find(unit);
  if (trace_it == _nodeStats.end()) {
    return nullptr;
  }
  return &(trace_it->second.discoveryRoute());
}

void NodesHandler::setTraceRoute(const MAC_address& mac, const ESPEasy_now_traceroute_struct& traceRoute)
{
  if (traceRoute.computeSuccessRate() == 0) {
    // No need to store traceroute with low success rate.
    return;
  }
  NodeStruct* node = getNodeByMac(mac);
  if (node != nullptr) {
    auto trace_it = _nodeStats.find(node->unit);
    if (trace_it != _nodeStats.end()) {
      _lastTimeValidDistance = millis();
      trace_it->second.addRoute(node->unit, traceRoute);
    }
  }
}

#endif


void NodesHandler::updateThisNode() {
  NodeStruct thisNode;

  // Set local data
  {
    MAC_address mac = NetworkMacAddress();
    mac.get(thisNode.sta_mac);
  }
  WiFi.softAPmacAddress(thisNode.ap_mac);
  {
    const bool addIP = NetworkConnected();
    #ifdef USES_ESPEASY_NOW
    if (use_EspEasy_now) {
      thisNode.useAP_ESPEasyNow = 1;
    }
    #endif
    if (addIP) {
      const IPAddress localIP = NetworkLocalIP();

      for (uint8_t i = 0; i < 4; ++i) {
        thisNode.ip[i] = localIP[i];
      }
    }
  }
  #ifdef USES_ESPEASY_NOW
  thisNode.channel = getESPEasyNOW_channel();
  #else
  thisNode.channel = WiFiEventData.usedChannel;
  #endif
  if (thisNode.channel == 0) {
    thisNode.channel = WiFi.channel();
  }

  thisNode.unit  = Settings.Unit;
  thisNode.build = Settings.Build;
  memcpy(thisNode.nodeName, Settings.getName().c_str(), 25);
  thisNode.nodeType = NODE_TYPE_ID;

  thisNode.webgui_portnumber = Settings.WebserverPort;
  const int load_int = getCPUload() * 2.55;

  if (load_int > 255) {
    thisNode.load = 255;
  } else {
    thisNode.load = load_int;
  }
  thisNode.timeSource = static_cast<uint8_t>(node_time.getTimeSource());

  switch (node_time.getTimeSource()) {
    case timeSource_t::No_time_source:
      thisNode.lastUpdated = (1 << 30);
      break;
    default:
    {
      thisNode.lastUpdated = timePassedSince(node_time.lastSyncTime_ms);
      break;
    }
  }
  if (node_time.systemTimePresent()) {
    // NodeStruct is a packed struct, so we cannot directly use its members as a reference.
    uint32_t unix_time_frac = 0;
    thisNode.unix_time_sec = node_time.getUnixTime(unix_time_frac);
    thisNode.unix_time_frac = unix_time_frac;
  }
  #ifdef USES_ESPEASY_NOW
  if (Settings.UseESPEasyNow()) {
    thisNode.ESPEasyNowPeer = 1;
  }
  #endif

  const uint8_t lastDistance = _distance;
  #ifdef USES_ESPEASY_NOW
  ESPEasy_now_traceroute_struct thisTraceRoute;
  #endif
  if (isEndpoint()) {
    _distance = 0;
    _lastTimeValidDistance = millis();
    if (lastDistance != _distance) {
      _recentlyBecameDistanceZero = true;
    }
    #ifdef USES_ESPEASY_NOW
    thisNode.distance = _distance;
    thisNode.setRSSI(WiFi.RSSI());
    thisTraceRoute.addUnit(thisNode.unit);
    #endif
  } else {
    _distance = 255;
    #ifdef USES_ESPEASY_NOW
    const NodeStruct *preferred = getPreferredNode_notMatching(thisNode.sta_mac);

    if (preferred != nullptr) {
      if (!preferred->isExpired()) {
        // Only take the distance of another node if it is running a build which does not send out traceroute
        // If it is a build sending traceroute, only consider having a distance if you know how to reach the gateway node
        // This does impose an issue when a gateway node is running an older version, as the next hops never will have a traceroute too.
        // Therefore the reported build for those units will be faked to be an older version.
        if (preferred->build < 20113) {
          if (preferred->distance != 255) {
            _distance = preferred->distance + 1;
            thisNode.build = 20112;
          }
        } else {
          const ESPEasy_now_traceroute_struct* tracert_ptr = getTraceRoute(preferred->unit);
          if (tracert_ptr != nullptr && tracert_ptr->getDistance() < 255) {
            // Make a copy of the traceroute
            thisTraceRoute = *tracert_ptr;
            thisTraceRoute.addUnit(thisNode.unit);
            if (preferred->distance != 255) {
              // Traceroute is only updated when a node is connected.
              // Thus the traceroute may be outdated, while the node info will already indicate if a node has lost its route to the gateway node.
              // So we only must set the distance of this node if the preferred node has a distance.
              _distance = thisTraceRoute.getDistance();  // This node is already included in the traceroute.
            }
          }
        }
      }
    }
    #endif
  }
  thisNode.distance = _distance;

  #if FEATURE_USE_IPV6
  thisNode.hasIPv4 = thisNode.IP() != INADDR_NONE;
  thisNode.hasIPv6_mac_based_link_local = is_IPv6_link_local_from_MAC(thisNode.sta_mac);
  thisNode.hasIPv6_mac_based_link_global = is_IPv6_global_from_MAC(thisNode.sta_mac);
  #endif

  #ifdef USES_ESPEASY_NOW
  addNode(thisNode, thisTraceRoute);
  if (thisNode.distance == 0) {
    // Since we're the end node, claim highest success rate
    updateSuccessRate(thisNode.unit, 255);
  }
  #else
  addNode(thisNode);
  #endif
}

const NodeStruct * NodesHandler::getThisNode() {
//  node_time.now();
  updateThisNode();
  MAC_address this_mac = NetworkMacAddress();
  return getNodeByMac(this_mac.mac);
}

uint8_t NodesHandler::getDistance() const {
  // Perform extra check since _distance is only updated once every 30 seconds.
  // And we don't want to tell other nodes we have distance 0 when we haven't.
  if (isEndpoint()) return 0;
  if (_distance == 0) {
    // Outdated info, so return "we don't know"
    return 255;
  }
  return _distance;
}


NodesMap::const_iterator NodesHandler::begin() const {
  return _nodes.begin();
}

NodesMap::const_iterator NodesHandler::end() const {
  return _nodes.end();
}

NodesMap::const_iterator NodesHandler::find(uint8_t unit_nr) const
{
  return _nodes.find(unit_nr);
}

bool NodesHandler::refreshNodeList(unsigned long max_age_allowed, unsigned long& max_age)
{
  max_age = 0;
  bool nodeRemoved = false;

  for (auto it = _nodes.begin(); it != _nodes.end();) {
    unsigned long age = it->second.getAge();
    if (age > max_age_allowed) {
      bool mustErase = true;
      #ifdef USES_ESPEASY_NOW
      auto route_it = _nodeStats.find(it->second.unit);
      if (route_it != _nodeStats.end()) {
        if (route_it->second.getAge() > max_age_allowed) {
          _nodeStats_mutex.lock();
          _nodeStats.erase(route_it);
          _nodeStats_mutex.unlock();
        } else {
          mustErase = false;
        }
      }
      #endif
      if (mustErase) {
        if (Settings.UseRules && it->second.unit != 0)
        {
          // Add event about removing node from nodeslist.
          eventQueue.addMove(strformat(F("p2pNode#Disconnected=%d"), it->second.unit));
        }
        {
          _nodes_mutex.lock();
          it          = _nodes.erase(it);
          _nodes_mutex.unlock();
        }
        nodeRemoved = true;
      }
    } else {
      ++it;

      if (age > max_age) {
        max_age = age;
      }
    }
  }
  return nodeRemoved;
}

// FIXME TD-er: should be a check per controller to see if it will accept messages
bool NodesHandler::isEndpoint() const
{
  // FIXME TD-er: Must check controller to see if it needs wifi (e.g. LoRa or cache controller do not need it)
  #if FEATURE_MQTT
  controllerIndex_t enabledMqttController = firstEnabledMQTT_ControllerIndex();
  if (validControllerIndex(enabledMqttController)) {
    // FIXME TD-er: Must call updateMQTTclient_connected() and see what effect
    // the MQTTclient_connected state has when using ESPEasy-NOW.
    return MQTTclient_connected;
  }
  #endif

  if (!NetworkConnected()) return false;

  return false;
}

#ifdef USES_ESPEASY_NOW
uint8_t NodesHandler::getESPEasyNOW_channel() const
{
  if (active_network_medium == NetworkMedium_t::WIFI && NetworkConnected()) {
    return WiFi.channel();
  }
  if (Settings.ForceESPEasyNOWchannel > 0) {
    return Settings.ForceESPEasyNOWchannel;
  }
  if (isEndpoint()) {
    if (active_network_medium == NetworkMedium_t::WIFI) {
      return WiFi.channel();
    }
  }
  const NodeStruct *preferred = getPreferredNode();
  if (preferred != nullptr) {
    if (preferred->distance < 255) {
      return preferred->channel;
    }
  }
  return WiFiEventData.usedChannel;
}
#endif

bool NodesHandler::recentlyBecameDistanceZero() {
  if (!_recentlyBecameDistanceZero) {
    return false;
  }
  _recentlyBecameDistanceZero = false;
  return true;
}

void NodesHandler::setRSSI(const MAC_address& mac, int rssi)
{
  setRSSI(getNodeByMac(mac), rssi);
}

void NodesHandler::setRSSI(uint8_t unit, int rssi)
{
  setRSSI(getNode(unit), rssi);
}

void NodesHandler::setRSSI(NodeStruct * node, int rssi)
{
  if (node != nullptr) {
    node->setRSSI(rssi);
  }
}

bool NodesHandler::lastTimeValidDistanceExpired() const
{
//  if (_lastTimeValidDistance == 0) return false;
  return timePassedSince(_lastTimeValidDistance) > 120000; // 2 minutes
}

#ifdef USES_ESPEASY_NOW
void NodesHandler::updateSuccessRate(uint8_t unit, bool success)
{
  auto it = _nodeStats.find(unit);
  if (it != _nodeStats.end()) {
    it->second.updateSuccessRate(unit, success);
  }
}

void NodesHandler::updateSuccessRate(const MAC_address& mac, bool success)
{
  const NodeStruct * node = getNodeByMac(mac);
  if (node == nullptr) {
    return;
  }
  updateSuccessRate(node->unit, success);
}

int NodesHandler::getRouteSuccessRate(uint8_t unit, uint8_t& distance) const
{
  distance = 255;
  auto it = _nodeStats.find(unit);
  if (it != _nodeStats.end()) {
    const ESPEasy_now_traceroute_struct* route = it->second.bestRoute();
    if (route != nullptr) {
      distance = route->getDistance();
      return route->computeSuccessRate();
    }
  }
  return 0;
}

uint8_t NodesHandler::getSuccessRate(uint8_t unit) const
{
  auto it = _nodeStats.find(unit);
  if (it != _nodeStats.end()) {
    return it->second.getNodeSuccessRate();
  }
  return 127;
}

ESPEasy_Now_MQTT_QueueCheckState::Enum NodesHandler::getMQTTQueueState(uint8_t unit) const
{
  auto it = _nodeStats.find(unit);
  if (it != _nodeStats.end()) {
    return it->second.getMQTTQueueState();
  }
  return ESPEasy_Now_MQTT_QueueCheckState::Enum::Unset;

}

void NodesHandler::setMQTTQueueState(uint8_t unit, ESPEasy_Now_MQTT_QueueCheckState::Enum state)
{
  auto it = _nodeStats.find(unit);
  if (it != _nodeStats.end()) {
    it->second.setMQTTQueueState(state);
  }
}

void NodesHandler::setMQTTQueueState(const MAC_address& mac, ESPEasy_Now_MQTT_QueueCheckState::Enum state)
{
  const NodeStruct * node = getNodeByMac(mac);
  if (node != nullptr) {
    setMQTTQueueState(node->unit, state);
  }
}

#endif

#endif
#include "../DataStructs/WiFi_AP_Candidate.h"

#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/SecuritySettings.h"
#include "../Globals/Statistics.h"
#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/Misc.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringGenerator_WiFi.h"
#include "../../ESPEasy_common.h"

#if defined(ESP8266)
  # include <ESP8266WiFi.h>
#endif // if defined(ESP8266)
#if defined(ESP32)
  # include <WiFi.h>
#endif // if defined(ESP32)

#define WIFI_AP_CANDIDATE_MAX_AGE   300000  // 5 minutes in msec


WiFi_AP_Candidate::WiFi_AP_Candidate() :
#ifdef ESP32
# if ESP_IDF_VERSION_MAJOR >= 5
country({
    .cc = "01",
    .schan = 1,
    .nchan = 14,
    .policy = WIFI_COUNTRY_POLICY_AUTO,
}),
#endif
#endif
  last_seen(0), rssi(0), channel(0), index(0), enc_type(0)
{
  _allBits = 0u;
}

WiFi_AP_Candidate::WiFi_AP_Candidate(const WiFi_AP_Candidate& other)
: ssid(other.ssid),
  last_seen(other.last_seen), 
  bssid(other.bssid),
  rssi(other.rssi), 
  channel(other.channel), 
  index(other.index), 
  enc_type(other.enc_type)
{
  _allBits = other._allBits;
  #ifdef ESP32
  # if ESP_IDF_VERSION_MAJOR >= 5
  memcpy(&this->country, &other.country, sizeof(wifi_country_t));
  #endif
  #endif
}

WiFi_AP_Candidate::WiFi_AP_Candidate(uint8_t index_c, const String& ssid_c) :
#ifdef ESP32
# if ESP_IDF_VERSION_MAJOR >= 5
country({
    .cc = "01",
    .schan = 1,
    .nchan = 14,
    .policy = WIFI_COUNTRY_POLICY_AUTO,
}),
#endif
#endif
  last_seen(0), rssi(0), channel(0), index(index_c), enc_type(0)
{
  _allBits = 0u;

  const size_t ssid_length = ssid_c.length();

  if ((ssid_length == 0) || equals(ssid_c, F("ssid"))) {
    return;
  }

  if (ssid_length > 32) { return; }

  ssid = ssid_c;
}

WiFi_AP_Candidate::WiFi_AP_Candidate(uint8_t networkItem) : index(0) {
  // Need to make sure the phy isn't known as we can't get this information from the AP
  // See: https://github.com/letscontrolit/ESPEasy/issues/4996
  // Not sure why this makes any difference as the flags should already have been set to 0.
  _allBits = 0u;

  ssid     = WiFi.SSID(networkItem);
  rssi     = WiFi.RSSI(networkItem);
  channel  = WiFi.channel(networkItem);
  bssid    = WiFi.BSSID(networkItem);
  enc_type = WiFi.encryptionType(networkItem);
  #ifdef ESP8266
  bits.isHidden = WiFi.isHidden(networkItem);
  # ifdef CORE_POST_3_0_0
  const bss_info *it = reinterpret_cast<const bss_info *>(WiFi.getScanInfoByIndex(networkItem));

  if (it) {
    bits.phy_11b = it->phy_11b;
    bits.phy_11g = it->phy_11g;
    bits.phy_11n = it->phy_11n;
    bits.wps     = it->wps;
  }
  # endif // ifdef CORE_POST_3_0_0
  #endif // ifdef ESP8266
  #ifdef ESP32
  bits.isHidden = ssid.isEmpty();
  wifi_ap_record_t *it = reinterpret_cast<wifi_ap_record_t *>(WiFi.getScanInfoByIndex(networkItem));

  if (it) {
    bits.phy_11b = it->phy_11b;
    bits.phy_11g = it->phy_11g;
    bits.phy_11n = it->phy_11n;
    bits.phy_lr  = it->phy_lr;
# if ESP_IDF_VERSION_MAJOR >= 5
    bits.phy_11ax      = it->phy_11ax;
    bits.ftm_initiator = it->ftm_initiator;
    bits.ftm_responder = it->ftm_responder;
# endif // if ESP_IDF_VERSION_MAJOR >= 5
    bits.wps = it->wps;

    // FIXME TD-er: Maybe also add other info like 2nd channel, ftm and phy_lr support?
# if ESP_IDF_VERSION_MAJOR >= 5
    memcpy(&country, &(it->country), sizeof(wifi_country_t));
#endif
  }
  #endif // ifdef ESP32
  last_seen = millis();
}

#ifdef ESP8266
# if FEATURE_ESP8266_DIRECT_WIFI_SCAN
WiFi_AP_Candidate::WiFi_AP_Candidate(const bss_info& ap) :
  rssi(ap.rssi), channel(ap.channel), bssid(ap.bssid),
  index(0), enc_type(0), isHidden(ap.is_hidden),
  phy_11b(ap.phy_11b), phy_11g(ap.phy_11g), phy_11n(ap.phy_11n),
  wps(ap.wps)
{
  _allBits = 0u;
  last_seen = millis();

  switch (ap.authmode) {
    case AUTH_OPEN: enc_type         = ENC_TYPE_NONE; break;
    case AUTH_WEP:  enc_type         = ENC_TYPE_WEP; break;
    case AUTH_WPA_PSK: enc_type      =  ENC_TYPE_TKIP; break;
    case AUTH_WPA2_PSK: enc_type     =  ENC_TYPE_CCMP; break;
    case AUTH_WPA_WPA2_PSK: enc_type =  ENC_TYPE_AUTO; break;
    case AUTH_MAX: break;
  }

  char tmp[33]; // ssid can be up to 32chars, => plus null term
  const size_t ssid_len = std::min(static_cast<size_t>(ap.ssid_len), sizeof(ap.ssid));

  memcpy(tmp, ap.ssid, ssid_len);
  tmp[ssid_len] = 0; // nullterm marking end of string

  ssid = String(reinterpret_cast<const char *>(tmp));
}

# endif // if FEATURE_ESP8266_DIRECT_WIFI_SCAN
#endif // ifdef ESP8266


bool WiFi_AP_Candidate::operator<(const WiFi_AP_Candidate& other) const {
  if (bits.isEmergencyFallback != other.bits.isEmergencyFallback) {
    return bits.isEmergencyFallback;
  }

  if (bits.lowPriority != other.bits.lowPriority) {
    return !bits.lowPriority;
  }

  // Prefer non hidden over hidden.
  if (bits.isHidden != other.bits.isHidden) {
    return !bits.isHidden;
  }

  // RSSI values >= 0 are invalid
  if (rssi >= 0) { return false; }

  if (other.rssi >= 0) { return true; }

  // RSSI values are negative, so the larger value is the better one.
  return rssi > other.rssi;
}

WiFi_AP_Candidate& WiFi_AP_Candidate::operator=(const WiFi_AP_Candidate& other)
{
  ssid = other.ssid;
  last_seen = other.last_seen;
  bssid = other.bssid;
  rssi = other.rssi;
  channel = other.channel;
  index = other.index;
  enc_type = other.enc_type;
  _allBits = other._allBits;
  #ifdef ESP32
  # if ESP_IDF_VERSION_MAJOR >= 5
  memcpy(&this->country, &other.country, sizeof(wifi_country_t));
  #endif
  #endif

  return *this;
}

bool WiFi_AP_Candidate::usable() const {
  // Allow for empty pass
  // if (key.isEmpty()) return false;
  if (bits.isEmergencyFallback) {
    int allowedUptimeMinutes = 10;
    #ifdef CUSTOM_EMERGENCY_FALLBACK_ALLOW_MINUTES_UPTIME
    allowedUptimeMinutes = CUSTOM_EMERGENCY_FALLBACK_ALLOW_MINUTES_UPTIME;
    #endif // ifdef CUSTOM_EMERGENCY_FALLBACK_ALLOW_MINUTES_UPTIME

    if ((getUptimeMinutes() > allowedUptimeMinutes) ||
        !SecuritySettings.hasWiFiCredentials() ||
        WiFiEventData.performedClearWiFiCredentials ||
        (lastBootCause != BOOT_CAUSE_COLD_BOOT)) {
      return false;
    }
  }

  if (!bits.isHidden && (ssid.isEmpty())) { return false; }
  return !expired();
}

bool WiFi_AP_Candidate::expired() const {
  if (last_seen == 0) {
    // Not set, so cannot expire
    return false;
  }
  return timePassedSince(last_seen) > WIFI_AP_CANDIDATE_MAX_AGE;
}

String WiFi_AP_Candidate::toString(const String& separator) const {
  String result = ssid;

  htmlEscape(result);

  if (bits.isHidden) {
    result += F("#Hidden#");
  }
  result += strformat(
    F("%s%s%sCh:%u"),
    separator.c_str(),
    bssid.toString().c_str(),
    separator.c_str(),
    channel);

  if (rssi == -1) {
    result += F(" (RTC) ");
  } else {
    result += strformat(F(" (%ddBm) "), rssi);
  }

  result += encryption_type();

#ifdef ESP32
# if ESP_IDF_VERSION_MAJOR >= 5
  // Country code string
  if (country.cc[0] != '\0' && country.cc[1] != '\0') {
    result += strformat(F(" '%c%c'"), country.cc[0], country.cc[1]);
    switch (country.cc[2]) {
      case 'O': // Outdoor
      case 'I': // Indoor
      case 'X': // "non-country"
        result += strformat(F("(%c)"), country.cc[2]);
        break;
    }
  }
  if (country.nchan > 0) {
    result += strformat(F(" ch: %d..%d"), country.schan, country.schan + country.nchan - 1);
  }
#endif
#endif

  if (phy_known()) {
    String phy_str;

    if (bits.phy_11b) { phy_str += 'b'; }

    if (bits.phy_11g) { phy_str += 'g'; }

    if (bits.phy_11n) { phy_str += 'n'; }
#ifdef ESP32

    if (bits.phy_11ax) { phy_str += F("/ax"); }

    if (bits.phy_lr) { phy_str += F("/lr"); }

    if (bits.ftm_initiator) { phy_str += F("/FTM_i"); }

    if (bits.ftm_responder) { phy_str += F("/FTM_r"); }
#endif // ifdef ESP32

    if (phy_str.length()) {
      result += strformat(F(" (%s)"), phy_str.c_str());
    }
  }
  return result;
}

String WiFi_AP_Candidate::encryption_type() const {
  return WiFi_encryptionType(enc_type);
}

#include "../DataStructs/ESPEasy_packed_raw_data.h"

#if FEATURE_PACKED_RAW_DATA

uint8_t getPackedDataTypeSize(PackedData_enum dtype, float& factor, float& offset) {
  offset = 0;
  factor = 1;
  if (dtype > 0x1000 && dtype < 0x12FF) {
    const uint32_t exponent = dtype & 0xF;
    switch(exponent) {
      case 0: factor = 1; break;
      case 1: factor = 1e1; break;
      case 2: factor = 1e2; break;
      case 3: factor = 1e3; break;
      case 4: factor = 1e4; break;
      case 5: factor = 1e5; break;
      case 6: factor = 1e6; break;
    }
    const uint8_t size = (dtype >> 4) & 0xF;
    return size;
  }
  switch (dtype) {
    case PackedData_pluginid:    factor = 1;         return 1;
    case PackedData_latLng:      factor = 46600;     return 3; // 2^23 / 180
    case PackedData_hdop:        factor = 10;        return 1;
    case PackedData_altitude:    factor = 4;     offset = 1000; return 2; // -1000 .. 15383.75 meter
    case PackedData_vcc:         factor = 41.83; offset = 1;    return 1; // -1 .. 5.12V
    case PackedData_pct_8:       factor = 2.56;                 return 1; // 0 .. 100%
    default:
      break;
  }

  // Unknown type
  factor = 1;
  return 0;
}

void LoRa_uintToBytes(uint64_t value, uint8_t byteSize, uint8_t *data, uint8_t& cursor) {
  // Clip values to upper limit
  const uint64_t upperlimit = (1ull << (8*byteSize)) - 1;
  if (value > upperlimit) { value = upperlimit; }
  for (uint8_t x = 0; x < byteSize; x++) {
    uint8_t next = 0;
    if (sizeof(value) > x) {
      next = static_cast<uint8_t>((value >> (x * 8)) & 0xFF);
    }
    data[cursor] = next;
    ++cursor;
  }
}

void LoRa_intToBytes(int64_t value, uint8_t byteSize, uint8_t *data, uint8_t& cursor) {
  // Clip values to lower limit
  const int64_t lowerlimit = (1ull << ((8*byteSize) - 1)) * -1;
  if (value < lowerlimit) { value = lowerlimit; }
  if (value < 0) {
    value += (1ull << (8*byteSize));
  }
  LoRa_uintToBytes(value, byteSize, data, cursor);
}

String LoRa_base16Encode(uint8_t *data, size_t size) {
  String output;
  output.reserve(size * 2);
  char buffer[3];
  for (unsigned i=0; i<size; i++)
  {
    sprintf_P(buffer, PSTR("%02X"), data[i]);
    output += buffer[0];
    output += buffer[1];
  }
  return output;
}

String LoRa_addInt(uint64_t value, PackedData_enum datatype) {
  float factor, offset;
  uint8_t byteSize = getPackedDataTypeSize(datatype, factor, offset);
  uint8_t data[4] = {0};
  uint8_t cursor = 0;
  LoRa_uintToBytes((value + offset) * factor, byteSize, &data[0], cursor);
  return LoRa_base16Encode(data, cursor);
}


String LoRa_addFloat(float value, PackedData_enum datatype) {
  float factor, offset;
  uint8_t byteSize = getPackedDataTypeSize(datatype, factor, offset);
  uint8_t data[4] = {0};
  uint8_t cursor = 0;
  LoRa_intToBytes((value + offset) * factor, byteSize, &data[0], cursor);
  return LoRa_base16Encode(data, cursor);
}

#endif
#include "../DataStructs/TimeChangeRule.h"



TimeChangeRule::TimeChangeRule() :  week(0), dow(1), month(1), hour(0), offset(0) {}

TimeChangeRule::TimeChangeRule(uint8_t weeknr, uint8_t downr, uint8_t m, uint8_t h, int16_t minutesoffset) :
  week(weeknr), dow(downr), month(m), hour(h), offset(minutesoffset) {}

// Construct time change rule from stored values optimized for minimum space.
TimeChangeRule::TimeChangeRule(uint16_t flash_stored_value, int16_t minutesoffset) : offset(minutesoffset) {
  hour  = flash_stored_value & 0x001f;
  month = (flash_stored_value >> 5) & 0x000f;
  dow   = (flash_stored_value >> 9) & 0x0007;
  week  = (flash_stored_value >> 12) & 0x0007;
}

uint16_t TimeChangeRule::toFlashStoredValue() const {
  uint16_t value = hour;

  value = value | (month << 5);
  value = value | (dow << 9);
  value = value | (week << 12);
  return value;
}

bool TimeChangeRule::isValid() const {
  return (week <= 4) && (dow != 0) && (dow <= 7) &&
          (month != 0) && (month <= 12) && (hour <= 23) &&
          (offset > -720) && (offset < 900); // UTC-12h ... UTC+14h + 1h DSToffset
}

#include "../DataStructs/EventQueue.h"

#include "../../ESPEasy_common.h"

#include "../Globals/Settings.h"
#include "../Helpers/Misc.h"
#include "../Helpers/StringConverter.h"



void EventQueueStruct::add(const String& event, bool deduplicate)
{
  if (!deduplicate || !isDuplicate(event)) {
#if defined(USE_SECOND_HEAP) || defined(ESP32)
    String tmp;
    reserve_special(tmp, event.length());
    tmp = event;

#ifdef USE_SECOND_HEAP
    // Do not add to the list while on 2nd heap
    HeapSelectDram ephemeral;
#endif


    _eventQueue.emplace_back(std::move(tmp));
#else
    _eventQueue.push_back(event);
#endif
  }
}

void EventQueueStruct::add(const __FlashStringHelper *event, bool deduplicate)
{
  String str;
  move_special(str, String(event));
  add(str, deduplicate);
}

void EventQueueStruct::addMove(String&& event, bool deduplicate)
{
  if (!event.length()) { return; }

  if (!deduplicate || !isDuplicate(event)) {
    #if defined(USE_SECOND_HEAP) || defined(ESP32)
    String tmp;
    move_special(tmp, std::move(event));

    #ifdef USE_SECOND_HEAP
    // Do not add to the list while on 2nd heap
    HeapSelectDram ephemeral;
    #endif
    _eventQueue.emplace_back(std::move(tmp));
    #else
    _eventQueue.emplace_back(std::move(event));
    #endif // ifdef USE_SECOND_HEAP
  }
}

void EventQueueStruct::add(taskIndex_t TaskIndex, const String& varName, const String& eventValue)
{
  if (Settings.UseRules) {
    if (eventValue.isEmpty()) {
      addMove(strformat(
        F("%s#%s"), 
        getTaskDeviceName(TaskIndex).c_str(), 
        varName.c_str()));
    } else {
      addMove(strformat(
        F("%s#%s=%s"), 
        getTaskDeviceName(TaskIndex).c_str(), 
        varName.c_str(), 
        eventValue.c_str()));
    }
  }
}

void EventQueueStruct::add(taskIndex_t TaskIndex, const String& varName, int eventValue)
{
  if (Settings.UseRules) {
    add(TaskIndex, varName, String(eventValue));
  }
}

void EventQueueStruct::add(taskIndex_t TaskIndex, const __FlashStringHelper *varName, const String& eventValue)
{
  if (Settings.UseRules) {
    add(TaskIndex, String(varName), eventValue);
  }
}

void EventQueueStruct::add(taskIndex_t TaskIndex, const __FlashStringHelper *varName, int eventValue)
{
  if (Settings.UseRules) {
    add(TaskIndex, String(varName), String(eventValue));
  }
}

bool EventQueueStruct::getNext(String& event)
{
  if (_eventQueue.empty()) {
    return false;
  }
  event = std::move(_eventQueue.front());
  _eventQueue.pop_front();
  return true;
}

void EventQueueStruct::clear()
{
  _eventQueue.clear();
}

bool EventQueueStruct::isEmpty() const
{
  return _eventQueue.empty();
}

bool EventQueueStruct::isDuplicate(const String& event) {
  return std::find(_eventQueue.begin(), _eventQueue.end(), event) != _eventQueue.end();
}

#include "../DataStructs/SecurityStruct.h"

#include "../../ESPEasy_common.h"
#include "../CustomBuild/ESPEasyLimits.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../Globals/CPlugins.h"

SecurityStruct::SecurityStruct() {
  ZERO_FILL(WifiSSID);
  ZERO_FILL(WifiKey);
  ZERO_FILL(WifiSSID2);
  ZERO_FILL(WifiKey2);
  ZERO_FILL(WifiAPKey);

  for (controllerIndex_t i = 0; i < CONTROLLER_MAX; ++i) {
    ZERO_FILL(ControllerUser[i]);
    ZERO_FILL(ControllerPassword[i]);
  }
  ZERO_FILL(Password);
}

ChecksumType SecurityStruct::computeChecksum() const {
  constexpr size_t len_upto_md5 = offsetof(SecurityStruct, md5);
  return ChecksumType(
    reinterpret_cast<const uint8_t *>(this), 
    sizeof(SecurityStruct),
    len_upto_md5);
}

bool SecurityStruct::checksumMatch() const {
  return computeChecksum().matchChecksum(md5);
}

bool SecurityStruct::updateChecksum() {
  const ChecksumType checksum = computeChecksum();
  if (checksum.matchChecksum(md5)) {
    return false;
  }
  checksum.getChecksum(md5);
  return true;
}

void SecurityStruct::validate() {
  ZERO_TERMINATE(WifiSSID);
  ZERO_TERMINATE(WifiKey);
  ZERO_TERMINATE(WifiSSID2);
  ZERO_TERMINATE(WifiKey2);
  ZERO_TERMINATE(WifiAPKey);

  for (controllerIndex_t i = 0; i < CONTROLLER_MAX; ++i) {
    ZERO_TERMINATE(ControllerUser[i]);
    ZERO_TERMINATE(ControllerPassword[i]);
  }
  ZERO_TERMINATE(Password);
}

void SecurityStruct::forceSave() {
  memset(md5, 0, 16);
}

void SecurityStruct::clearWiFiCredentials() {
  ZERO_FILL(WifiSSID);
  ZERO_FILL(WifiKey);
  ZERO_FILL(WifiSSID2);
  ZERO_FILL(WifiKey2);
  #ifndef BUILD_MINIMAL_OTA
  addLog(LOG_LEVEL_INFO, F("WiFi : Clear WiFi credentials from settings"));
  #endif
}

void SecurityStruct::clearWiFiCredentials(SecurityStruct::WiFiCredentialsSlot slot) {
  if (slot == SecurityStruct::WiFiCredentialsSlot::first) {
    ZERO_FILL(WifiSSID);
    ZERO_FILL(WifiKey);
  } else if (slot == SecurityStruct::WiFiCredentialsSlot::second) {
    ZERO_FILL(WifiSSID2);
    ZERO_FILL(WifiKey2);
  }
}

bool SecurityStruct::hasWiFiCredentials() const {
  return hasWiFiCredentials(SecurityStruct::WiFiCredentialsSlot::first) ||
         hasWiFiCredentials(SecurityStruct::WiFiCredentialsSlot::second);
}

bool SecurityStruct::hasWiFiCredentials(SecurityStruct::WiFiCredentialsSlot slot) const {
  if (slot == SecurityStruct::WiFiCredentialsSlot::first)
      return (WifiSSID[0] != 0 && !String(WifiSSID).equalsIgnoreCase(F("ssid")));
  if (slot == SecurityStruct::WiFiCredentialsSlot::second)
      return (WifiSSID2[0] != 0 && !String(WifiSSID2).equalsIgnoreCase(F("ssid")));

  return false;
}

String SecurityStruct::getPassword() const {
  String res;
  const size_t passLength = strnlen(Password, sizeof(Password));
  res.reserve(passLength);
  for (size_t i = 0; i < passLength; ++i) {
    res += Password[i];
  }
  return res;
}
#include "../DataStructs/C013_p2p_SensorInfoStruct.h"

#ifdef USES_C013

# include "../DataStructs/NodeStruct.h"
# include "../Globals/ExtraTaskSettings.h"
# include "../Globals/Nodes.h"
# include "../Globals/Plugins.h"
# include "../Globals/Settings.h"

# include "../CustomBuild/CompiletimeDefines.h"

# include "../Helpers/ESPEasy_Storage.h"
# include "../Helpers/StringConverter.h"

bool C013_SensorInfoStruct::prepareForSend(size_t& sizeToSend)
{
  if (!(validTaskIndex(sourceTaskIndex) &&
        validTaskIndex(destTaskIndex) &&
        validPluginID(deviceNumber))) {
    return false;
  }

  sizeToSend = sizeof(C013_SensorInfoStruct);

  sourceNodeBuild = get_build_nr();
  checksum.clear();

  ZERO_FILL(taskName);
  safe_strncpy(taskName, getTaskDeviceName(sourceTaskIndex), sizeof(taskName));

  for (uint8_t x = 0; x < VARS_PER_TASK; x++) {
    ZERO_FILL(ValueNames[x]);
    safe_strncpy(ValueNames[x], getTaskValueName(sourceTaskIndex, x), sizeof(ValueNames[x]));
  }


  if (sourceNodeBuild >= 20871) {
    LoadTaskSettings(sourceTaskIndex);

    ExtraTaskSettings_version = ExtraTaskSettings.version;

    for (uint8_t x = 0; x < VARS_PER_TASK; x++) {
      TaskDeviceValueDecimals[x] = ExtraTaskSettings.TaskDeviceValueDecimals[x];
      TaskDeviceMinValue[x]      = ExtraTaskSettings.TaskDeviceMinValue[x];
      TaskDeviceMaxValue[x]      = ExtraTaskSettings.TaskDeviceMaxValue[x];
      TaskDeviceErrorValue[x]    = ExtraTaskSettings.TaskDeviceErrorValue[x];
      VariousBits[x]             = ExtraTaskSettings.VariousBits[x];

/*
      ZERO_FILL(TaskDeviceFormula[x]);

      if (ExtraTaskSettings.TaskDeviceFormula[x][0] != 0) {
        safe_strncpy(TaskDeviceFormula[x], ExtraTaskSettings.TaskDeviceFormula[x], sizeof(TaskDeviceFormula[x]));
      }
*/
    }

    for (uint8_t x = 0; x < PLUGIN_CONFIGVAR_MAX; ++x) {
      TaskDevicePluginConfig[x] = Settings.TaskDevicePluginConfig[sourceTaskIndex][x];
    }
  }

  // Check to see if last bytes are all zero, so we can simply not send them
  bool doneShrinking                          = false;
  constexpr unsigned len_upto_sourceNodeBuild = offsetof(C013_SensorInfoStruct, sourceNodeBuild);

  const uint8_t *data = reinterpret_cast<const uint8_t *>(this);

  while (!doneShrinking) {
    if (sizeToSend < len_upto_sourceNodeBuild) {
      doneShrinking = true;
    }
    else {
      if (data[sizeToSend - 1] == 0) {
        --sizeToSend;
      } else {
        doneShrinking = true;
      }
    }
  }

  if (sourceNodeBuild >= 20871) {
    // Make sure to add checksum as last step
    constexpr unsigned len_upto_checksum = offsetof(C013_SensorInfoStruct, checksum);
    const ShortChecksumType tmpChecksum(
      reinterpret_cast<const uint8_t *>(this),
      sizeToSend,
      len_upto_checksum);

    checksum = tmpChecksum;
  }

  return true;
}

bool C013_SensorInfoStruct::setData(const uint8_t *data, size_t size)
{
  // First clear entire struct
  memset(this, 0, sizeof(C013_SensorInfoStruct));

  if (size < 6) {
    return false;
  }

  if ((data[0] != 255) || // header
      (data[1] != 3)) {   // ID
    return false;
  }

  // Before copying the data, compute the checksum of the entire packet
  constexpr unsigned len_upto_checksum = offsetof(C013_SensorInfoStruct, checksum);
  const ShortChecksumType tmpChecksum(
    data,
    size,
    len_upto_checksum);

  // Need to keep track of different possible versions of data which still need to be supported.
  if (size > sizeof(C013_SensorInfoStruct)) {
    size = sizeof(C013_SensorInfoStruct);
  }

  if (size <= 138) {
    deviceNumber = INVALID_PLUGIN_ID;
    sensorType   = Sensor_VType::SENSOR_TYPE_NONE;

    NodeStruct *sourceNode = Nodes.getNode(data[2]); // sourceUnit

    if (sourceNode != nullptr) {
      sourceNodeBuild = sourceNode->build;
    }
  }

  memcpy(this, data, size);

  if (checksum.isSet()) {
    if (!(tmpChecksum == checksum)) {
      return false;
    }
  }

  return validTaskIndex(sourceTaskIndex) &&
         validTaskIndex(destTaskIndex) &&
         validPluginID(deviceNumber);
}

#endif // ifdef USES_C013

#include "../DataStructs/ExtendedControllerCredentialsStruct.h"

#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/StringConverter.h"

#define EXT_CONTR_CRED_USER_OFFSET 0
#define EXT_CONTR_CRED_PASS_OFFSET 1


ChecksumType last_ExtendedControllerCredentialsStruct_md5;


ExtendedControllerCredentialsStruct::ExtendedControllerCredentialsStruct() {}

bool ExtendedControllerCredentialsStruct::validateChecksum() const
{
  const ChecksumType tmp_checksum(_strings, CONTROLLER_MAX * 2);
  if (tmp_checksum == last_ExtendedControllerCredentialsStruct_md5) {
    return true;
  }
  // Data has changed, copy computed checksum
  last_ExtendedControllerCredentialsStruct_md5 = tmp_checksum;
  return false;
}

void ExtendedControllerCredentialsStruct::clear() {
  for (size_t i = 0; i < CONTROLLER_MAX * 2; ++i) {
    free_string(_strings[i]);
  }
}

String ExtendedControllerCredentialsStruct::load()
{
  const String res =
    LoadStringArray(SettingsType::Enum::ExtdControllerCredentials_Type,
                    0,
                    _strings, CONTROLLER_MAX * 2, 0);

  for (int i = 0; i < CONTROLLER_MAX * 2; ++i) {
    _strings[i].trim();
  }

  // Update the checksum after loading.
  validateChecksum();

  return res;
}

String ExtendedControllerCredentialsStruct::save() const
{
  if (validateChecksum()) {
    return EMPTY_STRING;
  }
  return SaveStringArray(SettingsType::Enum::ExtdControllerCredentials_Type,
                         0,
                         _strings, CONTROLLER_MAX * 2, 0);
}

String ExtendedControllerCredentialsStruct::getControllerUser(controllerIndex_t controller_idx) const
{
  if (validControllerIndex(controller_idx)) {
    return _strings[controller_idx * 2 + EXT_CONTR_CRED_USER_OFFSET];
  }
  return EMPTY_STRING;
}

String ExtendedControllerCredentialsStruct::getControllerPass(controllerIndex_t controller_idx) const
{
  if (validControllerIndex(controller_idx)) {
    return _strings[controller_idx * 2 + EXT_CONTR_CRED_PASS_OFFSET];
  }
  return EMPTY_STRING;
}

void ExtendedControllerCredentialsStruct::setControllerUser(controllerIndex_t controller_idx, const String& user)
{
  if (validControllerIndex(controller_idx)) {
    _strings[controller_idx * 2 + EXT_CONTR_CRED_USER_OFFSET] = user;
  }
}

void ExtendedControllerCredentialsStruct::setControllerPass(controllerIndex_t controller_idx, const String& pass)
{
  if (validControllerIndex(controller_idx)) {
    _strings[controller_idx * 2 + EXT_CONTR_CRED_PASS_OFFSET] = pass;
  }
}

#include "../DataStructs/ChartJS_dataset_config.h"

#if FEATURE_CHART_JS

ChartJS_dataset_config::ChartJS_dataset_config(
  const __FlashStringHelper * set_label,
  const __FlashStringHelper * set_color)
  : label(set_label), color(set_color) {}

ChartJS_dataset_config::ChartJS_dataset_config(
  const String& set_label,
  const String& set_color)
  : label(set_label), color(set_color) {}

#endif
#include "../DataStructs/UnitMessageCount.h"

bool UnitLastMessageCount_map::isNew(const UnitMessageCount_t *count) const {
  if (count == nullptr) { return true; }
  auto it = _map.find(count->unit);

  if (it != _map.end()) {
    return it->second != count->count;
  }
  return true;
}

void UnitLastMessageCount_map::add(const UnitMessageCount_t *count) {
  if (count == nullptr) { return; }

  if ((count->unit != 0) && (count->unit != 255)) {
    _map[count->unit] = count->count;
  }
}

#include "../DataStructs/MAC_address.h"

#include "../../ESPEasy_common.h"


MAC_address::MAC_address(const uint8_t new_mac[6])
{
  memcpy(mac, new_mac, 6);
}

MAC_address::MAC_address(const MAC_address& other)
{
  for (int i = 0; i < 6; ++i) {
    mac[i] = other.mac[i];
  }
}

MAC_address& MAC_address::operator=(const MAC_address& other)
{
  for (int i = 0; i < 6; ++i) {
    mac[i] = other.mac[i];
  }
  return *this;
}

bool MAC_address::set(const char *string)
{
  unsigned u[6];
  int c = sscanf(string, "%x:%x:%x:%x:%x:%x", u, u + 1, u + 2, u + 3, u + 4, u + 5);

  if (c != 6) {
    return false;
  }

  for (int i = 0; i < 6; ++i) {
    mac[i] = static_cast<uint8_t>(u[i]);
  }
  return true;
}

void MAC_address::set(const uint8_t other[6])
{
  memcpy(mac, other, 6);
}

void MAC_address::get(uint8_t mac_out[6]) const
{
  memcpy(mac_out, mac, 6);
}


bool MAC_address::all_zero() const
{
  for (int i = 0; i < 6; ++i) {
    if (mac[i] != 0) {
      return false;
    }
  }
  return true;
}

bool MAC_address::all_one() const
{
  for (int i = 0; i < 6; ++i) {
    if (mac[i] != 0xFF) {
      return false;
    }
  }
  return true;
}

String MAC_address::toString() const
{
  char str[18] = { 0 };
  sprintf_P(str, PSTR("%02X:%02X:%02X:%02X:%02X:%02X"), mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
  return String(str);
}

bool MAC_address::mac_addr_cmp(const uint8_t other[6]) const
{
  for (int i = 0; i < 6; ++i) {
    if (mac[i] != other[i]) {
      return false;
    }
  }
  return true;
}
#include "../DataStructs/timer_id_couple.h"

#include "../Helpers/ESPEasy_time_calc.h"

bool timer_id_couple::operator<(const timer_id_couple& other) const {
  const unsigned long now(millis());

  // timediff > 0, means timer has already passed
  return timeDiff(_timer, now) > timeDiff(other._timer, now);
}

#include "../DataStructs/NodeStruct.h"

#if FEATURE_ESPEASY_P2P
#include "../../ESPEasy-Globals.h"
#include "../DataTypes/NodeTypeID.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../Globals/SecuritySettings.h"
#include "../Globals/Settings.h"
#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/StringConverter.h"


#define NODE_STRUCT_AGE_TIMEOUT 300000  // 5 minutes

NodeStruct::NodeStruct() : 
  ESPEasyNowPeer(0), 
  useAP_ESPEasyNow(0), 
  scaled_rssi(0)
#if FEATURE_USE_IPV6
   ,hasIPv4(0)
   ,hasIPv6_mac_based_link_local(0)
   ,hasIPv6_mac_based_link_global(0)
   ,unused(0)
#endif
{}

bool NodeStruct::valid() const {
  // FIXME TD-er: Must make some sanity checks to see if it is a valid message
  return true;
}

bool NodeStruct::validate(const IPAddress& remoteIP) {
  if (build < 20107) {
    // webserverPort introduced in 20107
    webgui_portnumber = 80;
    for (uint8_t i = 0; i < 6; ++i) {
      ap_mac[i] = 0;
    }
    load              = 0;
    distance          = 255;
    timeSource        = static_cast<uint8_t>(timeSource_t::No_time_source);
    channel           = 0;
    ESPEasyNowPeer    = 0;
    useAP_ESPEasyNow  = 0;
    setRSSI(0);
    lastUpdated = 0;
  }
  if (build < 20253) {
    version = 0;
#if FEATURE_USE_IPV6
    hasIPv4                       = 0;
    hasIPv6_mac_based_link_local  = 0;
    hasIPv6_mac_based_link_global = 0;

    unused = 0;
#else
    unused = 0;
#endif

    unix_time_frac = 0;
    unix_time_sec = 0;
  }

#if FEATURE_USE_IPV6
  // Check if we're in the same global subnet
  if (Settings.EnableIPv6() &&
      hasIPv6_mac_based_link_global && 
      remoteIP.type() == IPv6) {
    const IPAddress this_global = NetworkGlobalIP6();
    // Check first 64 bit to see if we're in the same global scope
    for (int i = 0; i < 8 && hasIPv6_mac_based_link_global; ++i) {
      if (this_global[i] != remoteIP[i])
        hasIPv6_mac_based_link_global = false;
    }
  }
#endif

  // FIXME TD-er: Must make some sanity checks to see if it is a valid message
  return valid();
}

bool NodeStruct::operator<(const NodeStruct &other) const {
  const bool thisExpired = isExpired();
  if (thisExpired != other.isExpired()) {
    return !thisExpired;
  }

  const bool markedAsPriority = markedAsPriorityPeer();
  if (markedAsPriority != other.markedAsPriorityPeer()) {
    return markedAsPriority;
  }

  if (ESPEasyNowPeer != other.ESPEasyNowPeer) {
    // One is confirmed, so prefer that one.
    return ESPEasyNowPeer;
  }

  const int8_t thisRssi = getRSSI();
  const int8_t otherRssi = other.getRSSI();

  int score_this = getLoad();
  int score_other = other.getLoad();

  if (distance != other.distance) {
    if (!isExpired() && !other.isExpired()) {
      // Distance is not the same, so take distance into account.
      return distance < other.distance;
/*
      int distance_penalty = distance - other.distance;
      distance_penalty = distance_penalty * distance_penalty * 10;
      if (distance > other.distance) {
        score_this += distance_penalty;
      } else {
        score_other += distance_penalty;
      }
*/
    }
  }

  if (thisRssi >= 0 || otherRssi >= 0) {
    // One or both have no RSSI, so cannot use RSSI in computing score
  } else {
    // RSSI value is negative, so subtract the value
    // RSSI range from -38 ... 99
    // Shift RSSI and add a weighing factor to make sure
    // A load of 100% with RSSI of -40 is preferred over a load of 20% with an RSSI of -80.
    score_this -= (thisRssi + 38) * 2;
    score_other -= (otherRssi + 38) * 2;
  }
  return score_this < score_other;
}


const __FlashStringHelper * NodeStruct::getNodeTypeDisplayString() const {
  return toNodeTypeDisplayString(nodeType);
}

String NodeStruct::getNodeName() const {
  String res;
  size_t length = strnlen(reinterpret_cast<const char *>(nodeName), sizeof(nodeName));

  if (reserve_special(res, length)) {
    for (size_t i = 0; i < length; ++i) {
      res += static_cast<char>(nodeName[i]);
    }
  }
  return res;
}

IPAddress NodeStruct::IP() const {
  return IPAddress(ip[0], ip[1], ip[2], ip[3]);
}

#if FEATURE_USE_IPV6
IPAddress NodeStruct::IPv6_link_local(bool stripZone) const
{
  if (Settings.EnableIPv6() && hasIPv6_mac_based_link_local) {
    // Base IPv6 on MAC address
    IPAddress ipv6;
    if (IPv6_link_local_from_MAC(sta_mac, ipv6)) {
      if (stripZone) {
        return IPAddress(IPv6, &ipv6[0], 0);
      }
      return ipv6;
    }
  }
  return IN6ADDR_ANY;
}

IPAddress NodeStruct::IPv6_global() const
{
  if (Settings.EnableIPv6() && hasIPv6_mac_based_link_global) {
    // Base IPv6 on MAC address
    IPAddress ipv6;
    if (IPv6_global_from_MAC(sta_mac, ipv6)) {
      return ipv6;
    }
  }
  return IN6ADDR_ANY;
}

bool NodeStruct::hasIPv6() const {
  if (!Settings.EnableIPv6()) return false;
  return hasIPv6_mac_based_link_local ||
         hasIPv6_mac_based_link_global;
}
#endif


MAC_address NodeStruct::STA_MAC() const {
  return MAC_address(sta_mac);
}

MAC_address NodeStruct::ESPEasy_Now_MAC() const {
  if (ESPEasyNowPeer == 0) return MAC_address();
  if (useAP_ESPEasyNow) {
    return MAC_address(ap_mac);
  }
  return MAC_address(sta_mac);
}

unsigned long NodeStruct::getAge() const {
  return timePassedSince(lastUpdated);
}

bool  NodeStruct::isExpired() const {
  return getAge() > NODE_STRUCT_AGE_TIMEOUT;
}

float NodeStruct::getLoad() const {
  return load / 2.55;
}

String NodeStruct::getSummary() const {
  String res;
  if (reserve_special(res, 48)) {
    res  = F("Unit: ");
    res += unit;
    res += F(" \"");
    res += getNodeName();
    res += '"';
    res += F(" load: ");
    res += String(getLoad(), 1);
    res += F(" RSSI: ");
    res += getRSSI();
    res += F(" ch: ");
    res += channel;
    res += F(" dst: ");
    res += distance;
  }
  return res;
}

bool NodeStruct::setESPEasyNow_mac(const MAC_address& received_mac)
{
  if (received_mac.all_zero()) return false;
  if (received_mac == sta_mac) {
    ESPEasyNowPeer   = 1;
    useAP_ESPEasyNow = 0;
    return true;
  }

  if (received_mac == ap_mac) {
    ESPEasyNowPeer   = 1;
    useAP_ESPEasyNow = 1;
    return true;
  }
  return false;
}

int8_t NodeStruct::getRSSI() const
{
  if (scaled_rssi == 0) {
    return 0; // Not set
  }

  if (scaled_rssi == 0x3F) {
    return 31; // Error state
  }

  // scaled_rssi = 1 ... 62
  // output = -38 ... -99
  int8_t rssi = scaled_rssi + 37;
  return rssi * -1;
}

void NodeStruct::setRSSI(int8_t rssi)
{
  if (rssi == 0) {
    // Not set
    scaled_rssi = 0;
    return;
  }

  if (rssi > 0) {
    // Error state
    scaled_rssi = 0x3F;
    return;
  }
  rssi *= -1;
  rssi -= 37;

  if (rssi < 1) {
    scaled_rssi = 1;
    return;
  }

  if (rssi >= 0x3F) {
    scaled_rssi = 0x3F - 1;
    return;
  }
  scaled_rssi = rssi;
}

bool NodeStruct::markedAsPriorityPeer() const
{
#ifdef USES_ESPEASY_NOW
  for (int i = 0; i < ESPEASY_NOW_PEER_MAX; ++i) {
    if (SecuritySettings.peerMacSet(i)) {
      if (match(SecuritySettings.EspEasyNowPeerMAC[i])) {
        return true;
      }
    }
  }
#endif
  return false;
}

bool NodeStruct::match(const MAC_address& mac) const
{
  return (mac == sta_mac || mac == ap_mac);
}

bool NodeStruct::isThisNode() const
{
    // Check to see if we process a node we've sent ourselves.
    if (WifiSoftAPmacAddress() == ap_mac) return true;
    if (WifiSTAmacAddress() == sta_mac) return true;

    return false;
}

void NodeStruct::setAP_MAC(const MAC_address& mac)
{
  mac.get(ap_mac);
}

#endif

#include "../DataStructs/LogEntry.h"

#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/StringConverter.h"

#define LOG_STRUCT_MESSAGE_SIZE 128

#ifdef ESP32
  #define LOG_BUFFER_EXPIRE         30000  // Time after which a buffered log item is considered expired.
#else
  #define LOG_BUFFER_EXPIRE         5000  // Time after which a buffered log item is considered expired.
#endif


bool LogEntry_t::add(const uint8_t loglevel, const String& line)
{
  if (line.length() == 0) {
    return false;
  }

  {
    if (line.length() > LOG_STRUCT_MESSAGE_SIZE - 1) {
      #ifdef USE_SECOND_HEAP

      // Need to make a substring or a copy, which is a new allocation, on the 2nd heap
      HeapSelectIram ephemeral;
      #endif // ifdef USE_SECOND_HEAP
      move_special(_message, line.substring(0, LOG_STRUCT_MESSAGE_SIZE - 1));
    } else {
      reserve_special(_message, line.length());
      _message = line;
    }
  }
  _loglevel  = loglevel;
  _timestamp = millis();
  return true;
}

bool LogEntry_t::add(const uint8_t loglevel, String&& line)
{
  if (line.length() == 0) {
    return false;
  }

  if (line.length() > LOG_STRUCT_MESSAGE_SIZE - 1) {
      #ifdef USE_SECOND_HEAP

    // Need to make a substring, which is a new allocation, on the 2nd heap
    HeapSelectIram ephemeral;
      #endif // ifdef USE_SECOND_HEAP
    move_special(_message, line.substring(0, LOG_STRUCT_MESSAGE_SIZE - 1));
  } else {
    move_special(_message, std::move(line));
  }
  _loglevel  = loglevel;
  _timestamp = millis();
  return true;
}

void LogEntry_t::clear()
{
  free_string(_message);
  _timestamp = 0;
  _loglevel  = 0;
}

bool LogEntry_t::isExpired() const
{
    return timePassedSince(_timestamp) >= LOG_BUFFER_EXPIRE;

}
#include "../DataStructs/mBusPacket.h"

#include "../Helpers/CRC_functions.h"
#include "../Helpers/StringConverter.h"

#define FRAME_FORMAT_A_FIRST_BLOCK_LENGTH 10
#define FRAME_FORMAT_A_OTHER_BLOCK_LENGTH 16


mBusPacket_header_t::mBusPacket_header_t()
{
  _manufacturer = mBus_packet_wildcard_manufacturer;
  _meterType    = mBus_packet_wildcard_metertype;
  _serialNr     = mBus_packet_wildcard_serial;
  _length       = 0u;
}

String mBusPacket_header_t::decodeManufacturerID(int id)
{
  String res;
  int    shift = 15;

  for (int i = 0; i < 3; ++i) {
    shift -= 5;
    res   += static_cast<char>(((id >> shift) & 0x1f) + 64);
  }
  return res;
}

int mBusPacket_header_t::encodeManufacturerID(const String& id_str)
{
  int res     = 0;
  int nrChars = id_str.length();

  if (nrChars > 3) { nrChars = 3; }

  int i = 0;

  while (i < nrChars) {
    res <<= 5;
    const int c = static_cast<int>(toUpperCase(id_str[i])) - 64;

    if (c >= 0) {
      res += c & 0x1f;
    }
    ++i;
  }
  return res;
}

String mBusPacket_header_t::getManufacturerId() const
{
  return decodeManufacturerID(_manufacturer);
}

String mBusPacket_header_t::toString() const
{
  String res = decodeManufacturerID(_manufacturer);

  res += '.';
  res += formatToHex_no_prefix(_meterType, 2);
  res += '.';
  res += formatToHex_no_prefix(_serialNr, 8);
  return res;
}

uint64_t mBusPacket_header_t::encode_toUInt64() const
{
  if (!isValid()) { return 0ull; }
  mBusPacket_header_t tmp(*this);

  tmp._length = 0;
  return tmp._encodedValue;
}

void mBusPacket_header_t::decode_fromUint64(uint64_t encodedValue)
{
  _encodedValue = encodedValue;
  _length       = 1; // To pass isValid() check
}

bool mBusPacket_header_t::isValid() const
{
  return
    _manufacturer != mBus_packet_wildcard_manufacturer &&
    _meterType != mBus_packet_wildcard_metertype &&
    _serialNr != mBus_packet_wildcard_serial &&
    _length > 0;
}

void mBusPacket_header_t::clear()
{
  _manufacturer = mBus_packet_wildcard_manufacturer;
  _meterType    = mBus_packet_wildcard_metertype;
  _serialNr     = mBus_packet_wildcard_serial;
  _length       = 0;
}

bool mBusPacket_header_t::matchSerial(uint32_t serialNr) const
{
  return isValid() && (_serialNr == serialNr);
}

const mBusPacket_header_t * mBusPacket_t::getDeviceHeader() const
{
  // FIXME TD-er: Which deviceID is the device and which the wrapper?
  if (_deviceId1.isValid()) { return &_deviceId1; }

  if (_deviceId2.isValid()) { return &_deviceId2; }

  return nullptr;
}

uint32_t mBusPacket_t::getDeviceSerial() const
{
  const mBusPacket_header_t *header = getDeviceHeader();

  if (header == nullptr) { return 0u; }
  return header->_serialNr;
}

uint32_t mBusPacket_t::deviceID_to_map_key() const
{
  return deviceID_to_map_key(_deviceId1._encodedValue, _deviceId2._encodedValue);
}

uint32_t mBusPacket_t::deviceID_to_map_key_no_length() const {
  return deviceID_to_map_key(_deviceId1.encode_toUInt64(), _deviceId2.encode_toUInt64());
}

uint32_t mBusPacket_t::deviceID_to_map_key(uint64_t id1, uint64_t id2)
{
  uint32_t res = 0;

  if (id1 != 0ull) {
    res ^= calc_CRC32((const uint8_t *)(&id1), sizeof(uint64_t));
  }

  if (id2 != 0ull) {
    // There is a forwarding device.
    // To prevent issues when the forwarding device is the same as the forwarded device, alter the already existing checksum.
    res ^= calc_CRC32((const uint8_t *)(&res), sizeof(res));
    res ^= calc_CRC32((const uint8_t *)(&id2), sizeof(uint64_t));
  }

  return res;
}

bool mBusPacket_t::parse(const String& payload)
{
  if (payload[0] != 'b') { return false; }

  _checksum = 0;
  mBusPacket_data payloadWithoutChecksums;

  if (payload[1] == 'Y') {
    // Start with "bY"
    payloadWithoutChecksums = removeChecksumsFrameB(payload, _checksum);
  } else {
    payloadWithoutChecksums = removeChecksumsFrameA(payload, _checksum);
  }

  if (payloadWithoutChecksums.size() < 10) { return false; }

  int pos_semicolon = payload.indexOf(';');

  if (pos_semicolon == -1) { pos_semicolon = payload.length(); }

  _lqi_rssi = hexToUL(payload, pos_semicolon - 4, 4);
  return parseHeaders(payloadWithoutChecksums);
}

int16_t mBusPacket_t::decode_LQI_RSSI(uint16_t lqi_rssi, uint8_t& LQI)
{
  LQI = (lqi_rssi >> 8) & 0x7f; // Bit 7 = CRC OK Bit

  int rssi = lqi_rssi & 0xFF;

  if (rssi >= 128) {
    rssi -= 256; // 2-complement
  }
  return (rssi / 2) - 74;
}

bool mBusPacket_t::matchSerial(uint32_t serialNr) const
{
  return _deviceId1.matchSerial(serialNr) || _deviceId2.matchSerial(serialNr);
}

bool mBusPacket_t::parseHeaders(const mBusPacket_data& payloadWithoutChecksums)
{
  const int payloadSize = payloadWithoutChecksums.size();

  _deviceId1.clear();
  _deviceId2.clear();

  if (payloadSize < 10) { return false; }
  int offset = 0;

  // 1st block is a static DataLinkLayer of 10 bytes
  {
    _deviceId1._manufacturer = makeWord(payloadWithoutChecksums[offset + 3], payloadWithoutChecksums[offset + 2]);

    // Type (offset + 9; convert to hex)
    _deviceId1._meterType = payloadWithoutChecksums[offset + 9];

    // Serial (offset + 4; 4 Bytes; least significant first; converted to hex)
    _deviceId1._serialNr = 0;

    for (int i = 0; i < 4; ++i) {
      const uint32_t val = payloadWithoutChecksums[offset + 4 + i];
      _deviceId1._serialNr += val << (i * 8);
    }
    offset            += 10;
    _deviceId1._length = payloadWithoutChecksums[0];
  }

  // next blocks can be anything. we skip known blocks of no interest, parse known blocks if interest and stop on onknown blocks
  while (offset < payloadSize) {
    switch (static_cast<int>(payloadWithoutChecksums[offset])) {
      case 0x8C:                                            // ELL short
        offset            += 3;                             // fixed length
        _deviceId1._length = payloadSize - offset;
        break;
      case 0x90:                                            // AFL
        offset++;
        offset += (payloadWithoutChecksums[offset] & 0xff); // dynamic length with length in 1st byte
        offset++;                                           // length byte
        _deviceId1._length = payloadSize - offset;
        break;
      case 0x72:                                            // TPL_RESPONSE_MBUS_LONG_HEADER
        _deviceId2 = _deviceId1;

        // note that serial/manufacturer are swapped !!

        _deviceId1._manufacturer = makeWord(payloadWithoutChecksums[offset + 6], payloadWithoutChecksums[offset + 5]);

        // Type (offset + 9; convert to hex)
        _deviceId1._meterType = payloadWithoutChecksums[offset + 8];

        // Serial (offset + 4; 4 Bytes; least significant first; converted to hex)
        _deviceId1._serialNr = 0;


        for (int i = 0; i < 4; ++i) {
          const uint32_t val = payloadWithoutChecksums[offset + 1 + i];
          _deviceId1._serialNr += val << (i * 8);
        }

        // We're done
        offset = payloadSize;
        break;
      default:
        // We're done
        //        addLog(LOG_LEVEL_ERROR, concat(F("CUL : offset "), offset) + F(" Data: ") + formatToHex(payloadWithoutChecksums[offset]));
        offset = payloadSize;
        break;
    }
  }

  if (_deviceId1.isValid() && _deviceId2.isValid() && _deviceId1.toString().startsWith(F("ITW.30."))) {
    // ITW does not follow the spec and puts the redio converter behind the actual meter. Need to swap both
    std::swap(_deviceId1, _deviceId2);
  }

  return _deviceId1.isValid();
}

String mBusPacket_t::toString() const
{
  static size_t expectedSize = 96;
  String res;

  if (res.reserve(expectedSize)) {
    if (_deviceId1.isValid()) {
      res += F(" deviceId1: ");
      res += _deviceId1.toString();
      res += '(';
      res += static_cast<uint8_t>(_deviceId1._length);
      res += ')';
    }

    if (_deviceId2.isValid()) {
      res += F(" deviceId2: ");
      res += _deviceId2.toString();
      res += '(';
      res += static_cast<uint8_t>(_deviceId2._length);
      res += ')';
    }
    res += F(" chksum: ");
    res += formatToHex(_checksum, 8);

    uint8_t LQI        = 0;
    const int16_t rssi = decode_LQI_RSSI(_lqi_rssi, LQI);
    res += F(" LQI: ");
    res += LQI;
    res += F(" RSSI: ");
    res += rssi;
  }

  if (res.length() > expectedSize) { expectedSize = res.length(); }

  return res;
}

uint8_t mBusPacket_t::hexToByte(const String& str, size_t index)
{
  // Need to have at least 2 HEX nibbles
  if ((index + 1) >= str.length()) { return 0; }
  return hexToUL(str, index, 2);
}

/**
 * Format:
 * [10 bytes message] + [2 bytes CRC]
 * [16 bytes message] + [2 bytes CRC]
 * [16 bytes message] + [2 bytes CRC]
 * ...
 * (last block can be < 16 bytes)
 */
mBusPacket_data mBusPacket_t::removeChecksumsFrameA(const String& payload, uint32_t& checksum)
{
  mBusPacket_data result;
  const int payloadLength = payload.length();

  if (payloadLength < 4) { return result; }

  int sourceIndex = 1; // Starts with "b"
  int targetIndex = 0;

  // 1st byte contains length of data (excuding 1st byte and excluding CRC)
  const int expectedMessageSize = hexToByte(payload, sourceIndex) + 1;

  if (payloadLength < (2 * expectedMessageSize)) {
    // Not an exact check, but close enough to fail early on packets which are seriously too short.
    return result;
  }

  result.reserve(expectedMessageSize);

  while (targetIndex < expectedMessageSize) {
    // end index is start index + block size + 2 byte checksums
    int blockSize = (sourceIndex == 1) ? FRAME_FORMAT_A_FIRST_BLOCK_LENGTH : FRAME_FORMAT_A_OTHER_BLOCK_LENGTH;

    if ((targetIndex + blockSize) > expectedMessageSize) { // last block
      blockSize = expectedMessageSize - targetIndex;
    }

    // FIXME: handle truncated source messages
    for (int i = 0; i < blockSize; ++i) {
      result.push_back(hexToByte(payload, sourceIndex));
      sourceIndex += 2; // 2 hex chars
    }

    // [2 bytes CRC]
    checksum   <<= 8;
    checksum    ^= hexToUL(payload, sourceIndex, 4);
    sourceIndex += 4; // Skip 2 bytes CRC => 4 hex chars
    targetIndex += blockSize;
  }
  return result;
}

/**
 * Format:
 * [126 bytes message] + [2 bytes CRC]
 * [125 bytes message] + [2 bytes CRC]
 * (if message length <=126 bytes, only the 1st block exists)
 * (last block can be < 125 bytes)
 */
mBusPacket_data mBusPacket_t::removeChecksumsFrameB(const String& payload, uint32_t& checksum)
{
  mBusPacket_data result;
  const int payloadLength = payload.length();

  if (payloadLength < 4) { return result; }

  int sourceIndex = 2; // Starts with "bY"

  // 1st byte contains length of data (excuding 1st byte BUT INCLUDING CRC)
  int expectedMessageSize = hexToByte(payload, sourceIndex) + 1;

  if (payloadLength < (2 * expectedMessageSize)) {
    return result;
  }

  expectedMessageSize -= 2;   // CRC of 1st block

  if (expectedMessageSize > 128) {
    expectedMessageSize -= 2; // CRC of 2nd block
  }

  result.reserve(expectedMessageSize);

  // FIXME: handle truncated source messages

  const int block1Size = expectedMessageSize < 126 ? expectedMessageSize : 126;

  for (int i = 0; i < block1Size; ++i) {
    result.push_back(hexToByte(payload, sourceIndex));
    sourceIndex += 2; // 2 hex chars
  }

  // [2 bytes CRC]
  checksum   <<= 8;
  checksum    ^= hexToUL(payload, sourceIndex, 4);
  sourceIndex += 4; // Skip 2 bytes CRC => 4 hex chars

  if (expectedMessageSize > 126) {
    int block2Size = expectedMessageSize - 127;

    if (block2Size > 124) { block2Size = 124; }

    for (int i = 0; i < block2Size; ++i) {
      result.push_back(hexToByte(payload, sourceIndex));
      sourceIndex += 2; // 2 hex chars
    }

    // [2 bytes CRC]
    checksum <<= 8;
    checksum  ^= hexToUL(payload, sourceIndex, 4);
  }

  // remove the checksums and the 1st byte from the actual message length, so that the meaning of this byte is the same as in Frame A
  result[0] = static_cast<uint8_t>((expectedMessageSize - 1) & 0xff);

  return result;
}

#include "../DataStructs/Scheduler_TaskDeviceTimerID.h"

#include "../Helpers/Misc.h"
#include "../Helpers/StringConverter.h"

TaskDeviceTimerID::TaskDeviceTimerID(taskIndex_t taskIndex) :
  SchedulerTimerID(SchedulerTimerType_e::TaskDeviceTimer)
{
  setId(static_cast<uint32_t>(taskIndex));
}

#ifndef BUILD_NO_DEBUG
String TaskDeviceTimerID::decode() const
{
  const taskIndex_t taskIndex = getTaskIndex();

  return concat(F("Task "), validTaskIndex(taskIndex)
        ? getTaskDeviceName(taskIndex)
        : String(getId()));
}

#endif // ifndef BUILD_NO_DEBUG

#include "../DataStructs/ChecksumType.h"

#include "../Helpers/StringConverter.h"

#include <MD5Builder.h>

ChecksumType::ChecksumType(const ChecksumType& rhs)
{
  memcpy(_checksum, rhs._checksum, 16);
}

ChecksumType::ChecksumType(uint8_t checksum[16])
{
  memcpy(_checksum, checksum, 16);
}

ChecksumType::ChecksumType(const uint8_t *data,
                           size_t   data_length)
{
  computeChecksum(_checksum, data, data_length, data_length, true);
}

ChecksumType::ChecksumType(const uint8_t *data,
                           size_t   data_length,
                           size_t   len_upto_md5)
{
  computeChecksum(_checksum, data, data_length, len_upto_md5, true);
}

ChecksumType::ChecksumType(const String strings[], size_t nrStrings)
{
  MD5Builder md5;

  md5.begin();

  for (size_t i = 0; i < nrStrings; ++i) {
    md5.add(strings[i].c_str());
  }
  md5.calculate();
  md5.getBytes(_checksum);
}

bool ChecksumType::computeChecksum(
  uint8_t  checksum[16],
  const uint8_t *data,
  size_t   data_length,
  size_t   len_upto_md5,
  bool     updateChecksum)
{
  if (len_upto_md5 > data_length) { len_upto_md5 = data_length; }
  MD5Builder md5;

  md5.begin();

  if (len_upto_md5 > 0) {
    // MD5Builder::add has non-const argument
    md5.add(const_cast<uint8_t *>(data), len_upto_md5);
  }

  if ((len_upto_md5 + 16) < data_length) {
    data += len_upto_md5 + 16;
    const int len_after_md5 = data_length - 16 - len_upto_md5;

    if (len_after_md5 > 0) {
      // MD5Builder::add has non-const argument
      md5.add(const_cast<uint8_t *>(data), len_after_md5);
    }
  }
  md5.calculate();
  uint8_t tmp_md5[16] = { 0 };

  md5.getBytes(tmp_md5);

  if (memcmp(tmp_md5, checksum, 16) != 0) {
    // Data has changed, copy computed checksum
    if (updateChecksum) {
      memcpy(checksum, tmp_md5, 16);
    }
    return false;
  }
  return true;
}

void ChecksumType::getChecksum(uint8_t checksum[16]) const {
  memcpy(checksum, _checksum, 16);
}

void ChecksumType::setChecksum(const uint8_t checksum[16]) {
  memcpy(_checksum, checksum, 16);
}

bool ChecksumType::matchChecksum(const uint8_t checksum[16]) const {
  return memcmp(_checksum, checksum, 16) == 0;
}

bool ChecksumType::operator==(const ChecksumType& rhs) const {
  return memcmp(_checksum, rhs._checksum, 16) == 0;
}

ChecksumType& ChecksumType::operator=(const ChecksumType& rhs) {
  memcpy(_checksum, rhs._checksum, 16);
  return *this;
}

String ChecksumType::toString() const {
  return formatToHex_array(_checksum, 16);
}
#include "../DataStructs/GpioFactorySettingsStruct.h"

#include "../CustomBuild/ESPEasyDefaults.h"

GpioFactorySettingsStruct::GpioFactorySettingsStruct(DeviceModel model)
  :
  status_led(DEFAULT_PIN_STATUS_LED),
  i2c_sda(DEFAULT_PIN_I2C_SDA),
  i2c_scl(DEFAULT_PIN_I2C_SCL),
  eth_phyaddr(DEFAULT_ETH_PHY_ADDR),
  eth_phytype(DEFAULT_ETH_PHY_TYPE),
  eth_mdc(DEFAULT_ETH_PIN_MDC),
  eth_mdio(DEFAULT_ETH_PIN_MDIO),
  eth_power(DEFAULT_ETH_PIN_POWER),
  eth_clock_mode(DEFAULT_ETH_CLOCK_MODE),
  network_medium(DEFAULT_NETWORK_MEDIUM)

{
  for (int i = 0; i < 4; ++i) {
    button[i] = -1;
    relais[i] = -1;
  }

#ifndef LIMIT_BUILD_SIZE

  switch (model) {
#if defined(ESP8266)
    case DeviceModel::DeviceModel_Sonoff_Basic:
    case DeviceModel::DeviceModel_Sonoff_TH1x:
    case DeviceModel::DeviceModel_Sonoff_S2x:
    case DeviceModel::DeviceModel_Sonoff_TouchT1:
    case DeviceModel::DeviceModel_Sonoff_POWr2:
      button[0]  = 0;  // Single Button
      relais[0]  = 12; // Red Led and Relay (0 = Off, 1 = On)
      status_led = 13; // Green/Blue Led (0 = On, 1 = Off)
      i2c_sda    = -1;
      i2c_scl    = -1;
      break;
    case DeviceModel::DeviceModel_Sonoff_POW:
      button[0]  = 0;  // Single Button
      relais[0]  = 12; // Red Led and Relay (0 = Off, 1 = On)
      status_led = 15; // Blue Led (0 = On, 1 = Off)
      i2c_sda    = -1;
      i2c_scl    = -1; // GPIO5 conflicts with HLW8012 Sel output
      break;
    case DeviceModel::DeviceModel_Sonoff_TouchT2:
      button[0]  = 0;  // Button 1
      button[1]  = 9;  // Button 2
      relais[0]  = 12; // Led and Relay1 (0 = Off, 1 = On)
      relais[1]  = 4;  // Led and Relay2 (0 = Off, 1 = On)
      status_led = 13; // Blue Led (0 = On, 1 = Off)
      i2c_sda    = -1; // GPIO4 conflicts with GPIO_REL3
      i2c_scl    = -1; // GPIO5 conflicts with GPIO_REL2
      break;
    case DeviceModel::DeviceModel_Sonoff_TouchT3:
      button[0]  = 0;  // Button 1
      button[1]  = 10; // Button 2
      button[2]  = 9;  // Button 3
      relais[0]  = 12; // Led and Relay1 (0 = Off, 1 = On)
      relais[1]  = 5;  // Led and Relay2 (0 = Off, 1 = On)
      relais[2]  = 4;  // Led and Relay3 (0 = Off, 1 = On)
      status_led = 13; // Blue Led (0 = On, 1 = Off)
      i2c_sda    = -1; // GPIO4 conflicts with GPIO_REL3
      i2c_scl    = -1; // GPIO5 conflicts with GPIO_REL2
      break;

    case DeviceModel::DeviceModel_Sonoff_4ch:
      button[0]  = 0;             // Button 1
      button[1]  = 9;             // Button 2
      button[2]  = 10;            // Button 3
      button[3]  = 14;            // Button 4
      relais[0]  = 12;            // Red Led and Relay1 (0 = Off, 1 = On)
      relais[1]  = 5;             // Red Led and Relay2 (0 = Off, 1 = On)
      relais[2]  = 4;             // Red Led and Relay3 (0 = Off, 1 = On)
      relais[3]  = 15;            // Red Led and Relay4 (0 = Off, 1 = On)
      status_led = 13;            // Blue Led (0 = On, 1 = Off)
      i2c_sda    = -1;            // GPIO4 conflicts with GPIO_REL3
      i2c_scl    = -1;            // GPIO5 conflicts with GPIO_REL2
      break;
    case DeviceModel::DeviceModel_Shelly1:
      button[0]  = 5;             // Single Button
      relais[0]  = 4;             // Red Led and Relay (0 = Off, 1 = On)
      status_led = 15;            // Blue Led (0 = On, 1 = Off)
      i2c_sda    = -1;            // GPIO4 conflicts with relay control.
      i2c_scl    = -1;            // GPIO5 conflicts with SW input
      break;
    case DeviceModel::DeviceModel_ShellyPLUG_S:
      button[0]  = 13;            // Single Button
      relais[0]  = 15;            // Red Led and Relay (0 = Off, 1 = On)
      status_led = 2;             // Blue Led (0 = On, 1 = Off)
      i2c_sda    = -1;            // GPIO4 conflicts with relay control.
      i2c_scl    = -1;            // GPIO5 conflicts with SW input
      break;
#else
    case DeviceModel::DeviceModel_Sonoff_Basic:
    case DeviceModel::DeviceModel_Sonoff_TH1x:
    case DeviceModel::DeviceModel_Sonoff_S2x:
    case DeviceModel::DeviceModel_Sonoff_TouchT1:
    case DeviceModel::DeviceModel_Sonoff_POWr2:
    case DeviceModel::DeviceModel_Sonoff_POW:
    case DeviceModel::DeviceModel_Sonoff_TouchT2:
    case DeviceModel::DeviceModel_Sonoff_TouchT3:
    case DeviceModel::DeviceModel_Sonoff_4ch:
    case DeviceModel::DeviceModel_Shelly1:
    case DeviceModel::DeviceModel_ShellyPLUG_S:
      break;
#endif


# if CONFIG_ETH_USE_ESP32_EMAC
    case DeviceModel::DeviceModel_Olimex_ESP32_PoE:
      button[0]             = 34; // BUT1 Button
      relais[0]             = -1; // No LED's or relays on board
      status_led            = -1;
      i2c_sda               = 13;
      i2c_scl               = 16;
      eth_phyaddr           = 0;
      eth_phytype           = EthPhyType_t::LAN8720;
      eth_mdc               = 23;
      eth_mdio              = 18;
      eth_power             = 12;
      eth_clock_mode        = EthClockMode_t::Int_50MHz_GPIO_17_inv;
      network_medium = NetworkMedium_t::Ethernet;
      break;
    case DeviceModel::DeviceModel_Olimex_ESP32_EVB:
      button[0] = 34; // BUT1 Button
      relais[0] = 32; // LED1 + Relay1 (0 = Off, 1 = On)
      relais[1] = 33; // LED2 + Relay2 (0 = Off, 1 = On)

      status_led            = -1;
      i2c_sda               = 13;
      i2c_scl               = 16;
      eth_phyaddr           = 0;
      eth_phytype           = EthPhyType_t::LAN8720;
      eth_mdc               = 23;
      eth_mdio              = 18;
      eth_power             = -1; // No Ethernet power pin
      eth_clock_mode        = EthClockMode_t::Ext_crystal_osc;
      network_medium = NetworkMedium_t::Ethernet;
      break;

    case DeviceModel::DeviceModel_Olimex_ESP32_GATEWAY:
      button[0]             = 34; // BUT1 Button
      relais[0]             = -1; // No LED's or relays on board
      status_led            = 33;
      i2c_sda               = -1;
      i2c_scl               = -1;
      eth_phyaddr           = 0;
      eth_phytype           = EthPhyType_t::LAN8720;
      eth_mdc               = 23;
      eth_mdio              = 18;
      eth_power             = 5;
      eth_clock_mode        = EthClockMode_t::Int_50MHz_GPIO_17_inv;
      network_medium = NetworkMedium_t::Ethernet;
      // Rev A to E:
      // GPIO 5, 17 can be used only if Ethernet functionality is not used
      // GPIO 6, 7, 8, 9, 10, 11 used for internal flash and SD card
      // GPIO 33 - Status LED
      // GPIO 34 - User button
      // GPIO 16, 32, 35, 36, 39 free to use

      // Rev F and up:
      // GPIO 5, 17 can be used only if Ethernet functionality is not used
      // GPIO 2, 14, 15 are used for SD card, they are free to use if SD is not used.
      // GPIO 33 - Status LED
      // GPIO 34 - User button
      // GPIO 4, 12, 13, 32, 35, 36, 39 free to use

      // ESPEasy default setting:
      // No GPIO pins selected in profile for I2C.
      // Since there are none free to use in all revisions capable of input/output.
      // N.B. GPIO 35 and up are input only.

      break;

    case DeviceModel::DeviceModel_wESP32:
      status_led            = -1;
      i2c_sda               = 15;
      i2c_scl               = 4;
      eth_phyaddr           = 0;
      eth_phytype           = EthPhyType_t::LAN8720;
      eth_mdc               = 16;
      eth_mdio              = 17;
      eth_power             = -1;
      eth_clock_mode        = EthClockMode_t::Ext_crystal_osc;
      network_medium = NetworkMedium_t::Ethernet;
      break;

    case DeviceModel::DeviceModel_WT32_ETH01:
      status_led            = -1;
      i2c_sda               = 21;
      i2c_scl               = 22;
      eth_phyaddr           = 1;
      eth_phytype           = EthPhyType_t::LAN8720;
      eth_mdc               = 23;
      eth_mdio              = 18;
      eth_power             = 12;  // TODO TD-er: Better to use GPIO-16? as shown here: https://letscontrolit.com/forum/viewtopic.php?p=50133#p50133
      eth_clock_mode        = EthClockMode_t::Ext_crystal_osc;
      network_medium = NetworkMedium_t::Ethernet;
      break;

  #endif

    case DeviceModel::DeviceModel_default:
    case DeviceModel::DeviceModel_MAX:
      break;

      // Do not use default: as this allows the compiler to detect any missing cases.
      // default: break;
  }
  #endif
}

#include "../DataStructs/NTP_packet.h"

#include "../Helpers/ESPEasy_time_calc.h"

#include "../Helpers/StringConverter.h"

NTP_packet::NTP_packet()
{
  // li, vn, and mode:
  // - li.   2 bits. Leap indicator.
  //    0 = no warning
  //    1 = last minute of the day has 61 seconds
  //    2 = last minute of the day has 59 seconds
  //    3 = unknown (clock unsynchronized)
  // - vn.   3 bits. Version number of the protocol. (0b100 = v4)
  // - mode. 3 bits. Client will pick mode 3 for client.
  //    0 = reserved
  //    1 = symmetric active
  //    2 = symmetric passive
  //    3 = client
  //    4 = server
  //    5 = broadcast
  //    6 = NTP control message
  //    7 = reserved for private use
  data[0] = 0b11100011; // Unsynchronized, V4, client mode

  // Stratum level of the local clock.
  //    0      = unspecified or invalid
  //    1      = primary server (e.g., equipped with a GPS receiver)
  //    2-15   = secondary server (via NTP)
  //    16     = unsynchronized
  //    17-255 = reserved
  data[1] = 0u;

  // Poll: 8-bit signed integer representing the maximum interval between
  //       successive messages, in log2 seconds.  Suggested default limits for
  //       minimum and maximum poll intervals are 6 and 10, respectively.
  data[2] = 6u;

  // Precision: 8-bit signed integer representing the precision of the
  //    system clock, in log2 seconds.  For instance, a value of -18
  //    corresponds to a precision of about one microsecond.  The precision
  //    can be determined when the service first starts up as the minimum
  //    time of several iterations to read the system clock.
  data[3] = 0xEC; // -20 -> 2^-20 sec -> microsec precision.

  //constexpr int8_t precision = 0xEC;

  // Reference clock identifier. ASCII: "1N14"
  data[12] = 0x31;
  data[13] = 0x4E;
  data[14] = 0x31;
  data[15] = 0x34;
}

uint32_t NTP_packet::readWord(uint8_t startIndex) const
{
  uint32_t res{};

  res  = (uint32_t)data[startIndex] << 24;
  res |= (uint32_t)data[startIndex + 1] << 16;
  res |= (uint32_t)data[startIndex + 2] << 8;
  res |= (uint32_t)data[startIndex + 3];
  return res;
}

uint64_t NTP_packet::ntp_timestamp_to_Unix_time(uint8_t startIndex) const {
  // Apply offset from 1900/01/01 to 1970/01/01
  constexpr uint64_t offset_since_1900 = 2208988800ULL * 1000000ull;

  const uint32_t Tm_s      = readWord(startIndex);
  const uint32_t Tm_f      = readWord(startIndex + 4);
  uint64_t usec_since_1900 = sec_time_frac_to_Micros(Tm_s, Tm_f);

  if (usec_since_1900 < offset_since_1900) {
    // Fix overflow which will occur in 2036
    usec_since_1900 += (4294967296ull * 1000000ull);
  }
  return usec_since_1900 - offset_since_1900;
}

void NTP_packet::writeWord(uint32_t value, uint8_t startIndex)
{
  data[startIndex]     = (value >> 24) & 0xFF;
  data[startIndex + 1] = (value >> 16) & 0xFF;
  data[startIndex + 2] = (value >> 8) & 0xFF;
  data[startIndex + 3] = (value) & 0xFF;
}

bool NTP_packet::isUnsynchronized() const
{
  return (data[0] /*li_vn_mode*/ & 0b11000000) == 0b11000000;
}

uint64_t NTP_packet::getReferenceTimestamp_usec() const
{
  return ntp_timestamp_to_Unix_time(16);
}

uint64_t NTP_packet::getOriginTimestamp_usec() const
{
  return ntp_timestamp_to_Unix_time(24);
}

uint64_t NTP_packet::getReceiveTimestamp_usec() const
{
  return ntp_timestamp_to_Unix_time(32);
}

uint64_t NTP_packet::getTransmitTimestamp_usec() const
{
  return ntp_timestamp_to_Unix_time(40);
}

void NTP_packet::setTxTimestamp(uint64_t micros)
{
  constexpr uint64_t offset_since_1900 = 2208988800ULL * 1000000ull;

  micros += offset_since_1900;
  uint32_t tmp_origTm_f{};

  writeWord(micros_to_sec_time_frac(micros, tmp_origTm_f), 40);
  writeWord(tmp_origTm_f,                                  44);
}

bool NTP_packet::compute_usec(
  uint64_t localTXTimestamp_usec,
  uint64_t localRxTimestamp_usec,
  int64_t& offset_usec,
  int64_t& roundtripDelay_usec) const
{
  int64_t t1 = getOriginTimestamp_usec();

  if (t1 == 0) {
    t1 = localTXTimestamp_usec;
  }

  const int64_t t2 = getReceiveTimestamp_usec();
  const int64_t t3 = getTransmitTimestamp_usec();

  if ((t3 == 0) || (t3 < t2)) {
    // No time stamp received
    return false;
  }
  const int64_t t4 = localRxTimestamp_usec;

  offset_usec  = (t2 - t1) + (t3 - t4);
  offset_usec /= 2;

  roundtripDelay_usec = (t4 - t1) - (t3 - t2);
  return true;
}

String NTP_packet::getRefID_str(bool& isError) const
{
  String refID;

  const uint8_t stratum = data[1];

  isError = false;

  if ((stratum == 0) || (stratum == 1)) {
    refID = strformat(F("%c%c%c%c"),
                      static_cast<char>(data[12] & 0x7F),
                      static_cast<char>(data[13] & 0x7F),
                      static_cast<char>(data[14] & 0x7F),
                      static_cast<char>(data[15] & 0x7F));

    if (stratum == 0) {
      if (refID.equals(F("DENY")) ||
          refID.equals(F("RSTR"))) {
        // For kiss codes DENY and RSTR, the client MUST
        // demobilize any associations to that server and
        // stop sending packets to that server;
        // DENY = Access denied by remote server.
        // RSTR = Access denied due to local policy.
        isError = true;
      } else if (refID.equals(F("RATE"))) {
        // For kiss code RATE, the client MUST immediately reduce its
        // polling interval to that server and continue to reduce it each
        // time it receives a RATE kiss code.
      }
    }
  } else {
    const IPAddress addrv4(readWord(12));
    refID = addrv4.toString();
  }
  return refID;
}

#ifndef BUILD_NO_DEBUG
String NTP_packet::toDebugString() const
{
  const uint8_t li   = (data[0] >> 6) & 0x3; // Leap Indicator
  const uint8_t ver  = (data[0] >> 3) & 0x7; // Version
  const uint8_t mode = data[0] & 0x7;        // Mode

  bool isError{};

  return strformat(
    F("    li: %u ver: %u mode: %u\n"
      "    strat: %u poll: %d prec: %d\n"
      "    del:   %u disp: %u refID: '%s'\n"
      "    refTm_s : %u refTm_f : %u\n"
      "    origTm_s: %u origTm_f: %u\n"
      "    rxTm_s  : %u rxTm_f  : %u\n"
      "    txTm_s  : %u txTm_f  : %u\n"),
    li, ver, mode,                                        // li_vn_mode
    data[1],                                              // stratum,
    (int8_t)(data[2]),                                    // poll in log2 seconds
    (int8_t)(data[3]),                                    // precision in log2 seconds
    readWord(4), readWord(8),                             // rootDelay, rootDispersion,
    getRefID_str(isError).c_str(),
    readWord(16), unix_time_frac_to_micros(readWord(20)), // refTm_s, unix_time_frac_to_micros(refTm_f),
    readWord(24), unix_time_frac_to_micros(readWord(28)), // origTm_s, unix_time_frac_to_micros(origTm_f),
    readWord(32), unix_time_frac_to_micros(readWord(36)), // rxTm_s, unix_time_frac_to_micros(rxTm_f),
    readWord(40), unix_time_frac_to_micros(readWord(44))  // txTm_s, unix_time_frac_to_micros(txTm_f)

    );
}

#endif // ifndef BUILD_NO_DEBUG

#include "../DataStructs/I2CTypes.h"

const __FlashStringHelper* toString(I2C_bus_state state) {
  switch (state) {
    case I2C_bus_state::BusCleared:            return F("Bus Cleared");
    case I2C_bus_state::OK:                    return F("OK");
    case I2C_bus_state::NotConfigured:         return F("Not Configured");
    case I2C_bus_state::ClearingProcessActive: return F("Clearing Process Active");
    case I2C_bus_state::SCL_Low:               return F("SCL Low");
    case I2C_bus_state::SDA_Low_over_2_sec:    return F("SCL Low by I2C device clock stretch > 2 sec");
    case I2C_bus_state::SDA_Low_20_clocks:     return F("SDA Low");
  }
  return F("");
}

#include "../DataStructs/CRCStruct.h"

#include <string.h>

bool CRCStruct::checkPassed() const
{
  return memcmp(compileTimeMD5, runTimeMD5, 16) == 0;
}

#include "../DataTypes/TaskIndex.h"

#include "../../ESPEasy_common.h"

taskIndex_t    const INVALID_TASK_INDEX    = TASKS_MAX;
userVarIndex_t const INVALID_USERVAR_INDEX = USERVAR_MAX_INDEX;
taskVarIndex_t const INVALID_TASKVAR_INDEX = VARS_PER_TASK;
#include "../DataTypes/SchedulerIntervalTimer.h"

#include "../Helpers/StringConverter.h"

#ifndef BUILD_NO_DEBUG
const __FlashStringHelper * toString_f(SchedulerIntervalTimer_e timer) {
  switch (timer) {
    case SchedulerIntervalTimer_e::TIMER_20MSEC:           return F("TIMER_20MSEC");
    case SchedulerIntervalTimer_e::TIMER_100MSEC:          return F("TIMER_100MSEC");
    case SchedulerIntervalTimer_e::TIMER_1SEC:             return F("TIMER_1SEC");
    case SchedulerIntervalTimer_e::TIMER_30SEC:            return F("TIMER_30SEC");
    case SchedulerIntervalTimer_e::TIMER_MQTT:             return F("TIMER_MQTT");
    case SchedulerIntervalTimer_e::TIMER_STATISTICS:       return F("TIMER_STATISTICS");
    case SchedulerIntervalTimer_e::TIMER_GRATUITOUS_ARP:   return F("TIMER_GRATUITOUS_ARP");
    case SchedulerIntervalTimer_e::TIMER_MQTT_DELAY_QUEUE: return F("TIMER_MQTT_DELAY_QUEUE");
    default: 
      break;
  }
  return F("unknown");
}
#endif

String toString(SchedulerIntervalTimer_e timer) {
#ifdef BUILD_NO_DEBUG
  return String(static_cast<int>(timer));
#else // ifdef BUILD_NO_DEBUG

  if (timer >= SchedulerIntervalTimer_e::TIMER_C001_DELAY_QUEUE && 
      timer <= SchedulerIntervalTimer_e::TIMER_C025_DELAY_QUEUE) 
  {
    const int id = static_cast<int>(timer) - static_cast<int>(SchedulerIntervalTimer_e::TIMER_C001_DELAY_QUEUE) + 1;
    String res;
    res.reserve(24);
    res = F("TIMER_");
    res += get_formatted_Controller_number(id);
    res += F("_DELAY_QUEUE");
    return res;
  }
  return toString_f(timer);
#endif // ifdef BUILD_NO_DEBUG
}
#include "../DataTypes/PluginID.h"

#include "../Helpers/StringConverter.h"


String pluginID_t::toDisplayString() const {
  if (value == 0) { return F("P---"); }
  return strformat(F("P%03d"), value);
}

const pluginID_t INVALID_PLUGIN_ID;

#include "../DataTypes/TLS_types.h"

#if FEATURE_TLS
const __FlashStringHelper* toString(TLS_types tls_type)
{
  switch (tls_type) {
    case TLS_types::NoTLS:        break;
//    case TLS_types::TLS_PSK:      return F("TLS PreSharedKey");
    case TLS_types::TLS_CA_CERT:  return F("TLS CA Cert");
    case TLS_types::TLS_insecure: return F("TLS No Checks (insecure)");
    case TLS_types::TLS_FINGERPRINT: return F("TLS Certficate Fingerprint");
  }
  return F("No TLS");
}
#endif
#include "../DataTypes/TaskValues_Data.h"

#include "../DataStructs/TimingStats.h"
#include "../Helpers/Numerical.h"
#include "../Helpers/StringConverter_Numerical.h"

TaskValues_Data_t::TaskValues_Data_t() {
  ZERO_FILL(binary);
}

TaskValues_Data_t::TaskValues_Data_t(const TaskValues_Data_t& other)
{
  memcpy(binary, other.binary, sizeof(binary));
}

TaskValues_Data_t& TaskValues_Data_t::operator=(const TaskValues_Data_t& other)
{
  memcpy(binary, other.binary, sizeof(binary));
  return *this;
}

void TaskValues_Data_t::clear() {
  ZERO_FILL(binary);
}

void TaskValues_Data_t::copyValue(const TaskValues_Data_t& other, uint8_t varNr, Sensor_VType sensorType)
{
  if (sensorType != Sensor_VType::SENSOR_TYPE_STRING) {
    if (is32bitOutputDataType(sensorType)) {
      if (varNr < VARS_PER_TASK) {
        constexpr unsigned int size_32bit = sizeof(float);
        memcpy(binary + (varNr * size_32bit), other.binary + (varNr * size_32bit), size_32bit);
      }
    }
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    else {
      if ((varNr < (VARS_PER_TASK / 2))) {
        constexpr unsigned int size_64bit = sizeof(uint64_t);
        memcpy(binary + (varNr * size_64bit), other.binary + (varNr * size_64bit), size_64bit);
      }
    }
#endif
  }
}

unsigned long TaskValues_Data_t::getSensorTypeLong() const
{
  const uint16_t low   = getFloat(0);
  const uint16_t high  = getFloat(1);
  unsigned long  value = high;

  value <<= 16;
  value  |= low;
  return value;
}

void TaskValues_Data_t::setSensorTypeLong(unsigned long value)
{
  setFloat(0, value & 0xFFFF);
  setFloat(1, (value >> 16) & 0xFFFF);
}

#if FEATURE_EXTENDED_TASK_VALUE_TYPES

int32_t TaskValues_Data_t::getInt32(uint8_t varNr) const
{
  if (varNr < VARS_PER_TASK) {
    int32_t res{};
    constexpr unsigned int size_32bit = sizeof(float);
    memcpy(&res, binary + (varNr * size_32bit), size_32bit);
    return res;
  }
  return 0;
}

void TaskValues_Data_t::setInt32(uint8_t varNr, int32_t value)
{
  if (varNr < VARS_PER_TASK) {
    constexpr unsigned int size_32bit = sizeof(float);
    memcpy(binary + (varNr * size_32bit), &value, size_32bit);
  }
}
#endif

uint32_t TaskValues_Data_t::getUint32(uint8_t varNr) const
{
  if (varNr < VARS_PER_TASK) {
    uint32_t res{};
    constexpr unsigned int size_32bit = sizeof(float);
    memcpy(&res, binary + (varNr * size_32bit), size_32bit);
    return res;
  }
  return 0u;
}

void TaskValues_Data_t::setUint32(uint8_t varNr, uint32_t value)
{
  if (varNr < VARS_PER_TASK) {
    constexpr unsigned int size_32bit = sizeof(float);
    memcpy(binary + (varNr * size_32bit), &value, size_32bit);
  }
}

#if FEATURE_EXTENDED_TASK_VALUE_TYPES

int64_t TaskValues_Data_t::getInt64(uint8_t varNr) const
{
  if ((varNr < (VARS_PER_TASK / 2))) {
    int64_t res{};
    constexpr unsigned int size_64bit = sizeof(uint64_t);
    memcpy(&res, binary + (varNr * size_64bit), size_64bit);
    return res;
  }
  return 0;
}

void TaskValues_Data_t::setInt64(uint8_t varNr, int64_t value)
{
  if ((varNr < (VARS_PER_TASK / 2))) {
    constexpr unsigned int size_64bit = sizeof(uint64_t);
    memcpy(binary + (varNr * size_64bit), &value, size_64bit);
  }
}

uint64_t TaskValues_Data_t::getUint64(uint8_t varNr) const
{
  if ((varNr < (VARS_PER_TASK / 2))) {
    uint64_t res{};
    constexpr unsigned int size_64bit = sizeof(uint64_t);
    memcpy(&res, binary + (varNr * size_64bit), size_64bit);
    return res;
  }
  return 0u;
}

void TaskValues_Data_t::setUint64(uint8_t varNr, uint64_t value)
{
  if ((varNr < (VARS_PER_TASK / 2))) {
    constexpr unsigned int size_64bit = sizeof(uint64_t);
    memcpy(binary + (varNr * size_64bit), &value, size_64bit);
  }
}
#endif

float TaskValues_Data_t::getFloat(uint8_t varNr) const
{
  if (varNr < VARS_PER_TASK) {
    float res{};
    constexpr unsigned int size_32bit = sizeof(float);
    memcpy(&res, binary + (varNr * size_32bit), size_32bit);
    return res;
  }
  return 0.0f;
}

void TaskValues_Data_t::setFloat(uint8_t varNr, float  value)
{
  if (varNr < VARS_PER_TASK) {
    constexpr unsigned int size_32bit = sizeof(float);
    memcpy(binary + (varNr * size_32bit), &value, size_32bit);
  }
}

#if FEATURE_EXTENDED_TASK_VALUE_TYPES
#if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
double TaskValues_Data_t::getDouble(uint8_t varNr) const
{
  if ((varNr < (VARS_PER_TASK / 2))) {
    double res{};
    constexpr unsigned int size_64bit = sizeof(uint64_t);
    memcpy(&res, binary + (varNr * size_64bit), size_64bit);
    return res;
  }
  return 0.0;
}

void TaskValues_Data_t::setDouble(uint8_t varNr, double  value)
{
  if ((varNr < (VARS_PER_TASK / 2))) {
    constexpr unsigned int size_64bit = sizeof(uint64_t);
    memcpy(binary + (varNr * size_64bit), &value, size_64bit);
  }
}
#endif
#endif

ESPEASY_RULES_FLOAT_TYPE TaskValues_Data_t::getAsDouble(uint8_t varNr, Sensor_VType sensorType) const
{
  if (sensorType == Sensor_VType::SENSOR_TYPE_ULONG) {
    return getSensorTypeLong();
  } else if (isFloatOutputDataType(sensorType)) {
    return getFloat(varNr);
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
  } else if (isUInt32OutputDataType(sensorType)) {
    return getUint32(varNr);
  } else if (isInt32OutputDataType(sensorType)) {
    return getInt32(varNr);
  } else if (isUInt64OutputDataType(sensorType)) {
    return getUint64(varNr);
  } else if (isInt64OutputDataType(sensorType)) {
    return getInt64(varNr);
#if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
  } else if (isDoubleOutputDataType(sensorType)) {
    return getDouble(varNr);
#endif    
#endif
  }
  return 0.0;
}

void TaskValues_Data_t::set(uint8_t varNr, const ESPEASY_RULES_FLOAT_TYPE& value, Sensor_VType sensorType)
{
  if (sensorType == Sensor_VType::SENSOR_TYPE_ULONG) {
    // Legacy formatting the old "SENSOR_TYPE_ULONG" type
    setSensorTypeLong(value);
  } else if (isFloatOutputDataType(sensorType)) {
    setFloat(varNr, value);
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
  } else if (isUInt32OutputDataType(sensorType)) {
    setUint32(varNr, value);
  } else if (isInt32OutputDataType(sensorType)) {
    setInt32(varNr, value);
  } else if (isUInt64OutputDataType(sensorType)) {
    setUint64(varNr, value);
  } else if (isInt64OutputDataType(sensorType)) {
    setInt64(varNr, value);
#if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
  } else if (isDoubleOutputDataType(sensorType)) {
    setDouble(varNr, value);
#endif
#endif
  }
}

bool TaskValues_Data_t::isValid(uint8_t varNr, Sensor_VType  sensorType) const
{
  if (sensorType == Sensor_VType::SENSOR_TYPE_NONE) {
    return false;
  } else if (isFloatOutputDataType(sensorType)) {
    return isValidFloat(getFloat(varNr));
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
#if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
  } else if (isDoubleOutputDataType(sensorType)) {
    return isValidDouble(getDouble(varNr));
#endif
#endif
  }
  return true;
}

String TaskValues_Data_t::getAsString(uint8_t varNr, Sensor_VType  sensorType, uint8_t nrDecimals) const
{
  String result;

  if (isFloatOutputDataType(sensorType)) {
    const float value = getFloat(varNr);
    if (nrDecimals == 254) {  // FIXME TD-er: Must use defines for these special situations
      nrDecimals = maxNrDecimals_fpType(value);
    }
    result = toString(value, nrDecimals);
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
#if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
  } else if (isDoubleOutputDataType(sensorType)) {
    const double value = getDouble(varNr);
    if (nrDecimals == 254) {  // FIXME TD-er: Must use defines for these special situations
      nrDecimals = maxNrDecimals_fpType(value);
    }
    result = doubleToString(value, nrDecimals);
#endif
#endif
  } else if (sensorType == Sensor_VType::SENSOR_TYPE_ULONG) {
    return String(getSensorTypeLong());
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
  } else if (isUInt32OutputDataType(sensorType)) {
    return String(getUint32(varNr));
  } else if (isInt32OutputDataType(sensorType)) {
    return String(getInt32(varNr));
  } else if (isUInt64OutputDataType(sensorType)) {
    return ull2String(getUint64(varNr));
  } else if (isInt64OutputDataType(sensorType)) {
    return ll2String(getInt64(varNr));
#endif
  }
  result.trim();
  return result;
}

#include "../DataTypes/FormSelectorOptions.h"

#include "../WebServer/Markup.h"
#include "../WebServer/Markup_Forms.h"
#include "../WebServer/HTML_wrappers.h"

FormSelectorOptions::FormSelectorOptions()
  : classname(F("wide")), _onlySelectorHead(true)
{}


FormSelectorOptions::FormSelectorOptions(int optionCount)
  :   classname(F("wide")), _optionCount(optionCount)
{}

FormSelectorOptions::FormSelectorOptions(
  int          optionCount,
  const int    indices[],
  const String attr[]) :
  classname(F("wide")),
  _optionCount(optionCount),
  _indices(indices),
  _attr_str(attr)
{}

FormSelectorOptions::FormSelectorOptions(
  int          optionCount,
  const String options[],
  const int    indices[],
  const String attr[]) :
  classname(F("wide")),
  _optionCount(optionCount),
  _names_str(options),
  _indices(indices),
  _attr_str(attr)
{}

FormSelectorOptions::FormSelectorOptions(
  int                        optionCount,
  const __FlashStringHelper *options[],
  const int                  indices[],
  const String               attr[]) :
  classname(F("wide")),
  _optionCount(optionCount),
  _names_f(options),
  _indices(indices),
  _attr_str(attr)
{}

FormSelectorOptions::~FormSelectorOptions() {}

String FormSelectorOptions::getOptionString(int index) const
{
  if (index >= _optionCount) {
    return EMPTY_STRING;
  }

  if (_names_f != nullptr) {
    return String(_names_f[index]);
  }

  if (_names_str != nullptr) {
    return String(_names_str[index]);
  }

  if (_indices != nullptr) {
    return String(_indices[index]);
  }
  return String(index);
}

int FormSelectorOptions::getIndexValue(int index) const
{
  if (index >= _optionCount) {
    return -1;
  }

  if (_indices != nullptr) {
    return _indices[index];
  }
  return index;
}

bool FormSelectorOptions::isDone(int index) const
{
  return index >= _optionCount;
}

void FormSelectorOptions::clearClassName()
{
  classname = F("");
}

void FormSelectorOptions::addFormSelector(
  const __FlashStringHelper *label,
  const __FlashStringHelper *id,
  int                        selectedIndex) const
{
  addFormSelector(String(label), String(id), selectedIndex);
}

void FormSelectorOptions::addFormSelector(
  const String             & label,
  const __FlashStringHelper *id,
  int                        selectedIndex) const
{
  addFormSelector(label, String(id), selectedIndex);
}

void FormSelectorOptions::addFormSelector(
  const __FlashStringHelper *label,
  const String             & id,
  int                        selectedIndex) const
{
  addFormSelector(String(label), id, selectedIndex);
}

void FormSelectorOptions::addFormSelector(
  const String& label,
  const String& id,
  int           selectedIndex) const
{
  addRowLabel_tr_id(label, id);
  addSelector(id, selectedIndex);
}

void FormSelectorOptions::addSelector(const __FlashStringHelper *id,
                                      int                        selectedIndex) const
{
  addSelector(String(id), selectedIndex);
}

void FormSelectorOptions::addSelector(const String& id,
                                      int           selectedIndex) const
{
  // FIXME TD-er Change bool 'enabled' to disabled
  if (reloadonchange)
  {
    addSelector_Head_reloadOnChange(id, classname, !enabled
                                    #if FEATURE_TOOLTIPS
                                    , tooltip
                                    #endif // if FEATURE_TOOLTIPS
                                    );
  } else {
    do_addSelector_Head(id, classname, onChangeCall, !enabled
                        #if FEATURE_TOOLTIPS
                        , tooltip
                        #endif // if FEATURE_TOOLTIPS
                        );
  }

  if (_onlySelectorHead) { return; }

  for (int i = 0; !isDone(i); ++i)
  {
    const int index = getIndexValue(i);
    String optionString = getOptionString(i);
    if (index == default_index) {
      optionString += F(" (default)");
    }
    addSelector_Item(
      optionString,
      index,
      selectedIndex == index,
      false,
      _attr_str ? _attr_str[i] : EMPTY_STRING);

    if ((i & 0x07) == 0) { delay(0); }
  }

  addSelectorFoot();
}

void FormSelectorOptions::addSelectorFoot() const
{
  addSelector_Foot(reloadonchange);
}

#include "../DataTypes/NPluginID.h"

#include "../Helpers/StringConverter.h"

String npluginID_t::toDisplayString() const {
  if (value == 0) { return F("N---"); }
  return strformat(F("N%03d"), value);
}

const npluginID_t INVALID_N_PLUGIN_ID;

#include "../DataTypes/NodeTypeID.h"


const __FlashStringHelper* toNodeTypeDisplayString(uint8_t nodeType) {
  switch (nodeType)
  {
    case NODE_TYPE_ID_ESP_EASY_STD:      return F("ESP Easy");
    case NODE_TYPE_ID_ESP_EASYM_STD:     return F("ESP Easy Mega");
    case NODE_TYPE_ID_ESP_EASY32_STD:    return F("ESP Easy 32");
    case NODE_TYPE_ID_ESP_EASY32S2_STD:  return F("ESP Easy 32-S2");
    case NODE_TYPE_ID_ESP_EASY32S3_STD:  return F("ESP Easy 32-S3");
    case NODE_TYPE_ID_ESP_EASY32C2_STD:  return F("ESP Easy 32-C2");
    case NODE_TYPE_ID_ESP_EASY32C3_STD:  return F("ESP Easy 32-C3");
    case NODE_TYPE_ID_ESP_EASY32C5_STD:  return F("ESP Easy 32-C5");
    case NODE_TYPE_ID_ESP_EASY32C6_STD:  return F("ESP Easy 32-C6");
    case NODE_TYPE_ID_ESP_EASY32C61_STD: return F("ESP Easy 32-C61");
    case NODE_TYPE_ID_ESP_EASY32H2_STD:  return F("ESP Easy 32-H2");
    case NODE_TYPE_ID_ESP_EASY32P4_STD:  return F("ESP Easy 32-P4");
    case NODE_TYPE_ID_RPI_EASY_STD:      return F("RPI Easy");
    case NODE_TYPE_ID_ARDUINO_EASY_STD:  return F("Arduino Easy");
    case NODE_TYPE_ID_NANO_EASY_STD:     return F("Nano Easy");
  }
  return F("");
}
#include "../DataTypes/ControllerIndex.h"

#include "../CustomBuild/ESPEasyLimits.h"

controllerIndex_t INVALID_CONTROLLER_INDEX = CONTROLLER_MAX;
#include "../DataTypes/IntendedRebootReason.h"

const __FlashStringHelper * toString_f(IntendedRebootReason_e reason) {
  switch (reason) {
    case IntendedRebootReason_e::DeepSleep:              return F("DeepSleep");
    case IntendedRebootReason_e::DelayedReboot:          return F("DelayedReboot");
    case IntendedRebootReason_e::ResetFactory:           return F("ResetFactory");
    case IntendedRebootReason_e::ResetFactoryPinActive:  return F("ResetFactoryPinActive");
    case IntendedRebootReason_e::ResetFactoryCommand:    return F("ResetFactoryCommand");
    case IntendedRebootReason_e::CommandReboot:          return F("CommandReboot");
    case IntendedRebootReason_e::RestoreSettings:        return F("RestoreSettings");
    case IntendedRebootReason_e::OTA_error:              return F("OTA_error");
    case IntendedRebootReason_e::ConnectionFailuresThreshold: return F("ConnectionFailuresThreshold");
  }
  return F("");
}

String toString(IntendedRebootReason_e reason) {
  String res = toString_f(reason);
  if (res.isEmpty()) return String(static_cast<int>(reason));
  return res;
}

#include "../DataTypes/SchedulerPluginPtrType.h"

const __FlashStringHelper* toString(SchedulerPluginPtrType_e pluginType) {
  switch (pluginType) {
    case SchedulerPluginPtrType_e::TaskPlugin:         return F("Plugin");
    case SchedulerPluginPtrType_e::ControllerPlugin:   return F("Controller");
#if FEATURE_NOTIFIER
    case SchedulerPluginPtrType_e::NotificationPlugin: return F("Notification");
#endif // if FEATURE_NOTIFIER
  }
  return F("Unknown");
}

#include "../DataTypes/DeviceIndex.h"


deviceIndex_t INVALID_DEVICE_INDEX = deviceIndex_t::toDeviceIndex(DEVICE_INDEX_MAX);

#include "../DataTypes/ESPEasyFileType.h"

#include "../../ESPEasy_common.h"

#include "../CustomBuild/StorageLayout.h"

#include "../Globals/ResetFactoryDefaultPref.h"

#include "../Helpers/StringConverter.h"

bool matchFileType(const String& filename, FileType::Enum filetype)
{
  if (filename.startsWith(F("/"))) {
    return matchFileType(filename.substring(1), filetype);
  }
  return filename.equalsIgnoreCase(getFileName(filetype));
}

bool isProtectedFileType(const String& filename)
{
  #if FEATURE_EXTENDED_CUSTOM_SETTINGS
  bool isTaskSpecificConfig = false;
  const String fname        = filename.substring(filename.startsWith(F("/")) ? 1 : 0);
  const String mask         = F(DAT_TASKS_CUSTOM_EXTENSION_FILEMASK);
  const int8_t mPerc        = mask.indexOf('%');

  if ((mPerc > -1) && fname.startsWith(mask.substring(0, mPerc))) {
    for (uint8_t n = 0; n < TASKS_MAX && !isTaskSpecificConfig; ++n) {
      isTaskSpecificConfig |= (fname.equalsIgnoreCase(strformat(mask, n + 1)));
    }
  }
  #endif // if FEATURE_EXTENDED_CUSTOM_SETTINGS

  return
    #if FEATURE_EXTENDED_CUSTOM_SETTINGS
    isTaskSpecificConfig || // Support for extcfgNN.dat
    #endif // if FEATURE_EXTENDED_CUSTOM_SETTINGS
    matchFileType(filename, FileType::CONFIG_DAT) ||
    matchFileType(filename, FileType::SECURITY_DAT) ||
    matchFileType(filename, FileType::NOTIFICATION_DAT) ||
    matchFileType(filename, FileType::PROVISIONING_DAT);
}

const __FlashStringHelper* getFileName(FileType::Enum filetype) {
  switch (filetype)
  {
    case FileType::CONFIG_DAT:       return F("config.dat");
    case FileType::NOTIFICATION_DAT: return F("notification.dat");
    case FileType::SECURITY_DAT:     return F("security.dat");
    case FileType::PROVISIONING_DAT: return F("provisioning.dat");
    case FileType::RULES_TXT:
      // Use getRulesFileName
      break;

    case FileType::MAX_FILETYPE:
      break;
  }
  return F("");
}

String getFileName(FileType::Enum filetype, unsigned int filenr) {
  if (filetype == FileType::RULES_TXT) {
    return getRulesFileName(filenr);
  }
  return getFileName(filetype);
}

// filenr = 0...3 for files rules1.txt ... rules4.txt
String getRulesFileName(unsigned int filenr) {
  String result;

  if (filenr < RULESETS_MAX) {
    result += F("rules");
    result += filenr + 1;
    result += F(".txt");
  }
  return result;
}

bool getDownloadFiletypeChecked(FileType::Enum filetype, unsigned int filenr) {
  switch (filetype) {
    case FileType::CONFIG_DAT:       return ResetFactoryDefaultPreference.fetchConfigDat();
    case FileType::SECURITY_DAT:     return ResetFactoryDefaultPreference.fetchSecurityDat();
    case FileType::NOTIFICATION_DAT: return ResetFactoryDefaultPreference.fetchNotificationDat();
    case FileType::RULES_TXT:        return ResetFactoryDefaultPreference.fetchRulesTXT(filenr);
    case FileType::PROVISIONING_DAT: return ResetFactoryDefaultPreference.fetchProvisioningDat();
      break;

    case FileType::MAX_FILETYPE:
      break;
  }
  return false;
}

#include "../DataTypes/WiFiConnectionProtocol.h"

const __FlashStringHelper * toString(WiFiConnectionProtocol proto) {
  switch (proto) {
    case WiFiConnectionProtocol::WiFi_Protocol_11b:
      return F("802.11b");
    case WiFiConnectionProtocol::WiFi_Protocol_11g:
      return F("802.11g");
#ifdef ESP8266
    case WiFiConnectionProtocol::WiFi_Protocol_11n:
      return F("802.11n");
#endif
#ifdef ESP32
    case WiFiConnectionProtocol::WiFi_Protocol_HT20:
      return F("802.11n (HT20)");
    case WiFiConnectionProtocol::WiFi_Protocol_HT40:
      return F("802.11n (HT40)");
    case WiFiConnectionProtocol::WiFi_Protocol_HE20:
      return F("802.11ax (HE20)");
#if ESP_IDF_VERSION > ESP_IDF_VERSION_VAL(5, 2, 0)
    case WiFiConnectionProtocol::WiFi_Protocol_11a:
      return F("802.11a");
    case WiFiConnectionProtocol::WiFi_Protocol_VHT20:
      return F("802.11ac (VHT20)");
#endif
    case WiFiConnectionProtocol::WiFi_Protocol_LR:
      return F("802.11lr");

#endif
    case WiFiConnectionProtocol::Unknown:
      break;;
  }
  return F("-");
}
#include "../DataTypes/SPI_options.h"


#ifdef ESP32

// ESP32 VSPI:
//  SCK  = 18
//  MISO = 19
//  MOSI = 23
// ESP32 HSPI:
//  SCK  = 14
//  MISO = 12
//  MOSI = 13


// ESP32-S2 FSPI:
//  SCK  = 36
//  MISO = 37
//  MOSI = 35

// ESP32-S3 FSPI:
//  SCK  = 36
//  MISO = 37
//  MOSI = 35

// ESP32-C2 SPI:
//  SCK  = 4
//  MISO = 5
//  MOSI = 6

// ESP32-C3 SPI:
//  SCK  = 4
//  MISO = 5
//  MOSI = 6


#  if CONFIG_IDF_TARGET_ESP32S3   // ESP32-S3
#define VSPI_FSPI_SHORT_STRING "FSPI"
#  elif CONFIG_IDF_TARGET_ESP32S2   // ESP32-S2
#define VSPI_FSPI_SHORT_STRING "FSPI"
#  elif CONFIG_IDF_TARGET_ESP32C2 // ESP32-C2
#define VSPI_FSPI_SHORT_STRING "FSPI"
#  elif CONFIG_IDF_TARGET_ESP32C3 // ESP32-C3
#define VSPI_FSPI_SHORT_STRING "SPI"
#  elif CONFIG_IDF_TARGET_ESP32C6 // ESP32-C6
#define VSPI_FSPI_SHORT_STRING "FSPI"
#  elif CONFIG_IDF_TARGET_ESP32   // ESP32/PICO-D4
#define VSPI_FSPI_SHORT_STRING "VSPI"

#  else // if CONFIG_IDF_TARGET_ESP32S2
#   error Target CONFIG_IDF_TARGET is not supported
#  endif // if CONFIG_IDF_TARGET_ESP32S2


const __FlashStringHelper* getSPI_optionToString(SPI_Options_e option) {
  switch (option) {
    case SPI_Options_e::None:
      return F("Disabled");
    case SPI_Options_e::Vspi_Fspi:
      return F(
        VSPI_FSPI_SHORT_STRING 
        ": CLK=GPIO-" STRINGIFY(VSPI_FSPI_SCK) 
        ", MISO=GPIO-" STRINGIFY(VSPI_FSPI_MISO) 
        ", MOSI=GPIO-" STRINGIFY(VSPI_FSPI_MOSI) );
#ifdef ESP32_CLASSIC
    case SPI_Options_e::Hspi:
      return F(
        "HSPI"
        ": CLK=GPIO-" STRINGIFY(HSPI_SCLK) 
        ", MISO=GPIO-" STRINGIFY(HSPI_MISO) 
        ", MOSI=GPIO-" STRINGIFY(HSPI_MOSI) );

      
#endif
    case SPI_Options_e::UserDefined:
      return F("User-defined: CLK, MISO, MOSI GPIO-pins");
  }
  return F("Unknown");
}

const __FlashStringHelper* getSPI_optionToShortString(SPI_Options_e option) {
  switch (option) {
    case SPI_Options_e::None:
      return F("Disabled");
    case SPI_Options_e::Vspi_Fspi:
      return F(VSPI_FSPI_SHORT_STRING);
#ifdef ESP32_CLASSIC
    case SPI_Options_e::Hspi:
      return F("HSPI");
#endif
    case SPI_Options_e::UserDefined:
      return F("User-defined SPI");
  }
  return F("Unknown");
}

#endif // ifdef ESP32

#include "../DataTypes/NotifierIndex.h"

#include "../CustomBuild/ESPEasyLimits.h"

notifierIndex_t INVALID_NOTIFIER_INDEX = NOTIFICATION_MAX;

#include "../DataTypes/ProtocolIndex.h"

#include "../CustomBuild/ESPEasyLimits.h"

protocolIndex_t   INVALID_PROTOCOL_INDEX   = CPLUGIN_MAX;
#include "../DataTypes/CPluginID.h"

const cpluginID_t INVALID_C_PLUGIN_ID = 0;


#include "../DataTypes/NetworkMedium.h"

bool isValid(NetworkMedium_t medium) {
  switch (medium) {
    case NetworkMedium_t::WIFI:
    case NetworkMedium_t::Ethernet:
#ifdef USES_ESPEASY_NOW
    case NetworkMedium_t::ESPEasyNOW_only:
#endif
      return true;

    case NetworkMedium_t::NotSet:
      return false;

      // Do not use default: as this allows the compiler to detect any missing cases.
  }
  return false;
}

const __FlashStringHelper * toString(NetworkMedium_t medium) {
  switch (medium) {
    case NetworkMedium_t::WIFI:     return F("WiFi");
    case NetworkMedium_t::Ethernet: return F("Ethernet");
#ifdef USES_ESPEASY_NOW
    case NetworkMedium_t::ESPEasyNOW_only:  return F(ESPEASY_NOW_NAME " only");
#endif
    case NetworkMedium_t::NotSet:   return F("Not Set");

      // Do not use default: as this allows the compiler to detect any missing cases.
  }
  return F("Unknown");
}
#include "../DataTypes/TimeSource.h"

const __FlashStringHelper* toString(ExtTimeSource_e timeSource)
{
  switch (timeSource) {
    case ExtTimeSource_e::None: break;
    case ExtTimeSource_e::DS1307:  return F("DS1307");
    case ExtTimeSource_e::DS3231:  return F("DS3231");
    case ExtTimeSource_e::PCF8523: return F("PCF8523");
    case ExtTimeSource_e::PCF8563: return F("PCF8563");
  }
  return F("-");
}

#include "../DataTypes/SensorVType.h"
#include "../Helpers/StringConverter.h"


/*********************************************************************************************\
   Get value count from sensor type

   Only use this function to determine nr of output values when changing output type of a task
   To get the actual output values for a task, use getValueCountForTask
\*********************************************************************************************/
uint8_t getValueCountFromSensorType(Sensor_VType sensorType) {
  return getValueCountFromSensorType(sensorType, true);
}
uint8_t getValueCountFromSensorType(Sensor_VType sensorType, bool log)
{
  switch (sensorType)
  {
    case Sensor_VType::SENSOR_TYPE_NONE:
      return 0;
    case Sensor_VType::SENSOR_TYPE_SINGLE:        // single value sensor, used for Dallas, BH1750, etc
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_INT32_SINGLE:  // 1x int32_t
    case Sensor_VType::SENSOR_TYPE_UINT64_SINGLE: // 1x uint64_t
    case Sensor_VType::SENSOR_TYPE_INT64_SINGLE:  // 1x int64_t
    case Sensor_VType::SENSOR_TYPE_DOUBLE_SINGLE: // 1x ESPEASY_RULES_FLOAT_TYPE
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_SWITCH:
    case Sensor_VType::SENSOR_TYPE_DIMMER:
    case Sensor_VType::SENSOR_TYPE_ULONG:  // single unsigned LONG value, stored in two floats (rfid tags)
    case Sensor_VType::SENSOR_TYPE_STRING: // String type data stored in the event->String2
      return 1;
    case Sensor_VType::SENSOR_TYPE_TEMP_HUM:
    case Sensor_VType::SENSOR_TYPE_TEMP_BARO:
    case Sensor_VType::SENSOR_TYPE_DUAL:        // 2x float
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_UINT32_DUAL: // 2x uint32_t
    case Sensor_VType::SENSOR_TYPE_INT32_DUAL:  // 2x int32_t
    case Sensor_VType::SENSOR_TYPE_UINT64_DUAL: // 2x uint64_t
    case Sensor_VType::SENSOR_TYPE_INT64_DUAL:  // 2x int64_t
    case Sensor_VType::SENSOR_TYPE_DOUBLE_DUAL: // 2x ESPEASY_RULES_FLOAT_TYPE
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
      return 2;
    case Sensor_VType::SENSOR_TYPE_TEMP_HUM_BARO:
    case Sensor_VType::SENSOR_TYPE_TEMP_EMPTY_BARO: // Values 1 and 3 will contain data.
    case Sensor_VType::SENSOR_TYPE_TRIPLE:          // 3x float
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_UINT32_TRIPLE:   // 3x uint32_t
    case Sensor_VType::SENSOR_TYPE_INT32_TRIPLE:    // 3x int32_t
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_WIND:
      return 3;
    case Sensor_VType::SENSOR_TYPE_QUAD:        // 4x float
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_UINT32_QUAD: // 4x uint32_t
    case Sensor_VType::SENSOR_TYPE_INT32_QUAD:  // 4x int32_t
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
      return 4;
    case Sensor_VType::SENSOR_TYPE_NOT_SET:  break;

    // FIXME To be ignored?
    case Sensor_VType::SENSOR_TYPE_ANALOG_ONLY:
    case Sensor_VType::SENSOR_TYPE_TEMP_ONLY:
    case Sensor_VType::SENSOR_TYPE_HUM_ONLY:
    case Sensor_VType::SENSOR_TYPE_LUX_ONLY:
    case Sensor_VType::SENSOR_TYPE_DISTANCE_ONLY:
    case Sensor_VType::SENSOR_TYPE_DIRECTION_ONLY:
    case Sensor_VType::SENSOR_TYPE_DUSTPM2_5_ONLY:
    case Sensor_VType::SENSOR_TYPE_DUSTPM1_0_ONLY:
    case Sensor_VType::SENSOR_TYPE_DUSTPM10_ONLY:
    case Sensor_VType::SENSOR_TYPE_MOISTURE_ONLY:
    case Sensor_VType::SENSOR_TYPE_CO2_ONLY:
    case Sensor_VType::SENSOR_TYPE_GPS_ONLY:
    case Sensor_VType::SENSOR_TYPE_UV_ONLY:
    case Sensor_VType::SENSOR_TYPE_UV_INDEX_ONLY:
    case Sensor_VType::SENSOR_TYPE_IR_ONLY:
    case Sensor_VType::SENSOR_TYPE_WEIGHT_ONLY:
    case Sensor_VType::SENSOR_TYPE_VOLTAGE_ONLY:
    case Sensor_VType::SENSOR_TYPE_CURRENT_ONLY:
    case Sensor_VType::SENSOR_TYPE_POWER_USG_ONLY:
    case Sensor_VType::SENSOR_TYPE_POWER_FACT_ONLY:
    case Sensor_VType::SENSOR_TYPE_APPRNT_POWER_USG_ONLY:
    case Sensor_VType::SENSOR_TYPE_TVOC_ONLY:
    case Sensor_VType::SENSOR_TYPE_BARO_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_RED_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_GREEN_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_BLUE_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_TEMP_ONLY:
    case Sensor_VType::SENSOR_TYPE_REACTIVE_POWER_ONLY:
    case Sensor_VType::SENSOR_TYPE_AQI_ONLY:
    case Sensor_VType::SENSOR_TYPE_NOX_ONLY:
    case Sensor_VType::SENSOR_TYPE_SWITCH_INVERTED:
    case Sensor_VType::SENSOR_TYPE_WIND_SPEED:
    case Sensor_VType::SENSOR_TYPE_DURATION:
    case Sensor_VType::SENSOR_TYPE_DATE:
    case Sensor_VType::SENSOR_TYPE_TIMESTAMP:
    case Sensor_VType::SENSOR_TYPE_DATA_RATE:
    case Sensor_VType::SENSOR_TYPE_DATA_SIZE:
    case Sensor_VType::SENSOR_TYPE_SOUND_PRESSURE:
    case Sensor_VType::SENSOR_TYPE_SIGNAL_STRENGTH:
    case Sensor_VType::SENSOR_TYPE_REACTIVE_ENERGY:
    case Sensor_VType::SENSOR_TYPE_FREQUENCY:
    case Sensor_VType::SENSOR_TYPE_ENERGY:
    case Sensor_VType::SENSOR_TYPE_ENERGY_STORAGE:
    case Sensor_VType::SENSOR_TYPE_ABS_HUMIDITY:
    case Sensor_VType::SENSOR_TYPE_ATMOS_PRESSURE:
    case Sensor_VType::SENSOR_TYPE_BLOOD_GLUCOSE_C:
    case Sensor_VType::SENSOR_TYPE_CO_ONLY:
    case Sensor_VType::SENSOR_TYPE_ENERGY_DISTANCE:
    case Sensor_VType::SENSOR_TYPE_GAS_ONLY:
    case Sensor_VType::SENSOR_TYPE_NITROUS_OXIDE:
    case Sensor_VType::SENSOR_TYPE_OZONE_ONLY:
    case Sensor_VType::SENSOR_TYPE_PRECIPITATION:
    case Sensor_VType::SENSOR_TYPE_PRECIPITATION_INTEN:
    case Sensor_VType::SENSOR_TYPE_SULPHUR_DIOXIDE:
    case Sensor_VType::SENSOR_TYPE_VOC_PARTS:
    case Sensor_VType::SENSOR_TYPE_VOLUME:
    case Sensor_VType::SENSOR_TYPE_VOLUME_FLOW_RATE:
    case Sensor_VType::SENSOR_TYPE_VOLUME_STORAGE:
    case Sensor_VType::SENSOR_TYPE_WATER:
      return 1;
  }
  #ifndef BUILD_NO_DEBUG
  if (log) {
    addLog(LOG_LEVEL_ERROR, F("getValueCountFromSensorType: Unknown sensortype"));
  }
  #endif // ifndef BUILD_NO_DEBUG
  return 0;
}

const __FlashStringHelper* getSensorTypeLabel(Sensor_VType sensorType) {
  switch (sensorType) {
    case Sensor_VType::SENSOR_TYPE_SWITCH:           return F("Switch");
    case Sensor_VType::SENSOR_TYPE_DIMMER:           return F("Dimmer");
    case Sensor_VType::SENSOR_TYPE_TEMP_HUM:         return F("Temp / Hum");
    case Sensor_VType::SENSOR_TYPE_TEMP_BARO:        return F("Temp / Baro");
    case Sensor_VType::SENSOR_TYPE_TEMP_EMPTY_BARO:  return F("Temp / - / Baro");
    case Sensor_VType::SENSOR_TYPE_TEMP_HUM_BARO:    return F("Temp / Hum / Baro");
    case Sensor_VType::SENSOR_TYPE_SINGLE:           return F("Single");
    case Sensor_VType::SENSOR_TYPE_DUAL:             return F("Dual");
    case Sensor_VType::SENSOR_TYPE_TRIPLE:           return F("Triple");
    case Sensor_VType::SENSOR_TYPE_QUAD:             return F("Quad");
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_INT32_SINGLE:     return F("Int32 (1x)");
    case Sensor_VType::SENSOR_TYPE_INT32_DUAL:       return F("Int32 (2x)");
    case Sensor_VType::SENSOR_TYPE_INT32_TRIPLE:     return F("Int32 (3x)");
    case Sensor_VType::SENSOR_TYPE_INT32_QUAD:       return F("Int32 (4x)");
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_ULONG:            return F("UInt32 (1x)");
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_UINT32_DUAL:      return F("UInt32 (2x)");
    case Sensor_VType::SENSOR_TYPE_UINT32_TRIPLE:    return F("UInt32 (3x)");
    case Sensor_VType::SENSOR_TYPE_UINT32_QUAD:      return F("UInt32 (4x)");
    case Sensor_VType::SENSOR_TYPE_UINT64_SINGLE:    return F("UInt64 (1x)");
    case Sensor_VType::SENSOR_TYPE_UINT64_DUAL:      return F("UInt64 (2x)");
    case Sensor_VType::SENSOR_TYPE_INT64_SINGLE:     return F("Int64 (1x)");
    case Sensor_VType::SENSOR_TYPE_INT64_DUAL:       return F("Int64 (2x)");
    case Sensor_VType::SENSOR_TYPE_DOUBLE_SINGLE:    return F("Double (1x)");
    case Sensor_VType::SENSOR_TYPE_DOUBLE_DUAL:      return F("Double (2x)");
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_WIND:             return F("Wind");
    case Sensor_VType::SENSOR_TYPE_STRING:           return F("String");
    case Sensor_VType::SENSOR_TYPE_NONE:             return F("None");
    case Sensor_VType::SENSOR_TYPE_NOT_SET:  break;

      // FIXME To be ignored?
    #if FEATURE_MQTT_DISCOVER || FEATURE_CUSTOM_TASKVAR_VTYPE
    case Sensor_VType::SENSOR_TYPE_ANALOG_ONLY:      return F("Analog");
    case Sensor_VType::SENSOR_TYPE_TEMP_ONLY:        return F("Temp");
    case Sensor_VType::SENSOR_TYPE_HUM_ONLY:         return F("Hum");
    case Sensor_VType::SENSOR_TYPE_LUX_ONLY:         return F("Lux");
    case Sensor_VType::SENSOR_TYPE_DISTANCE_ONLY:    return F("Distance");
    case Sensor_VType::SENSOR_TYPE_DIRECTION_ONLY:   return F("Direction");
    case Sensor_VType::SENSOR_TYPE_DUSTPM2_5_ONLY:   return F("Dust PM2.5");
    case Sensor_VType::SENSOR_TYPE_DUSTPM1_0_ONLY:   return F("Dust PM1.0");
    case Sensor_VType::SENSOR_TYPE_DUSTPM10_ONLY:    return F("Dust PM10");
    case Sensor_VType::SENSOR_TYPE_MOISTURE_ONLY:    return F("Moisture");
    case Sensor_VType::SENSOR_TYPE_CO2_ONLY:         return F("(e)CO2");
    case Sensor_VType::SENSOR_TYPE_GPS_ONLY:         return F("GPS");
    case Sensor_VType::SENSOR_TYPE_UV_ONLY:          return F("UV");
    case Sensor_VType::SENSOR_TYPE_UV_INDEX_ONLY:    return F("UV Index");
    case Sensor_VType::SENSOR_TYPE_IR_ONLY:          return F("IR");
    case Sensor_VType::SENSOR_TYPE_WEIGHT_ONLY:      return F("Weight");
    case Sensor_VType::SENSOR_TYPE_VOLTAGE_ONLY:     return F("Voltage");
    case Sensor_VType::SENSOR_TYPE_CURRENT_ONLY:     return F("Current");
    case Sensor_VType::SENSOR_TYPE_POWER_USG_ONLY:   return F("Power Usage");
    case Sensor_VType::SENSOR_TYPE_POWER_FACT_ONLY:  return F("Power Factor");
    case Sensor_VType::SENSOR_TYPE_APPRNT_POWER_USG_ONLY: return F("Apparent Power Usage");
    case Sensor_VType::SENSOR_TYPE_TVOC_ONLY:        return F("TVOC");
    case Sensor_VType::SENSOR_TYPE_BARO_ONLY:        return F("Baro");
    case Sensor_VType::SENSOR_TYPE_COLOR_RED_ONLY:   return F("Red");
    case Sensor_VType::SENSOR_TYPE_COLOR_GREEN_ONLY: return F("Green");
    case Sensor_VType::SENSOR_TYPE_COLOR_BLUE_ONLY:  return F("Blue");
    case Sensor_VType::SENSOR_TYPE_COLOR_TEMP_ONLY:  return F("Color temperature");
    case Sensor_VType::SENSOR_TYPE_REACTIVE_POWER_ONLY: return F("Reactive Power");
    case Sensor_VType::SENSOR_TYPE_AQI_ONLY:         return F("AQI");
    case Sensor_VType::SENSOR_TYPE_NOX_ONLY:         return F("NOx");
    case Sensor_VType::SENSOR_TYPE_SWITCH_INVERTED:  return F("Switch (inv.)");
    case Sensor_VType::SENSOR_TYPE_WIND_SPEED:       return F("Wind speed");
    case Sensor_VType::SENSOR_TYPE_DURATION:         return F("Duration");
    case Sensor_VType::SENSOR_TYPE_DATE:             return F("Date");
    case Sensor_VType::SENSOR_TYPE_TIMESTAMP:        return F("Timestamp");
    case Sensor_VType::SENSOR_TYPE_DATA_RATE:        return F("Data rate");
    case Sensor_VType::SENSOR_TYPE_DATA_SIZE:        return F("Data size");
    case Sensor_VType::SENSOR_TYPE_SOUND_PRESSURE:   return F("Sound pressure");
    case Sensor_VType::SENSOR_TYPE_SIGNAL_STRENGTH:  return F("Signal strength");
    case Sensor_VType::SENSOR_TYPE_REACTIVE_ENERGY:  return F("Reactive Energy");
    case Sensor_VType::SENSOR_TYPE_FREQUENCY:        return F("Frequency");
    case Sensor_VType::SENSOR_TYPE_ENERGY:           return F("Energy");
    case Sensor_VType::SENSOR_TYPE_ENERGY_STORAGE:   return F("Energy storage");
    case Sensor_VType::SENSOR_TYPE_ABS_HUMIDITY:     return F("Absolute humidity");
    case Sensor_VType::SENSOR_TYPE_ATMOS_PRESSURE:   return F("Atmospheric pressure");
    case Sensor_VType::SENSOR_TYPE_BLOOD_GLUCOSE_C:  return F("Blood glucose conc.");
    case Sensor_VType::SENSOR_TYPE_CO_ONLY:          return F("CO");
    case Sensor_VType::SENSOR_TYPE_ENERGY_DISTANCE:  return F("Energy distance");
    case Sensor_VType::SENSOR_TYPE_GAS_ONLY:         return F("Gas");
    case Sensor_VType::SENSOR_TYPE_NITROUS_OXIDE:    return F("N2O");
    case Sensor_VType::SENSOR_TYPE_OZONE_ONLY:       return F("Ozone");
    case Sensor_VType::SENSOR_TYPE_PRECIPITATION:    return F("Precipitation");
    case Sensor_VType::SENSOR_TYPE_PRECIPITATION_INTEN: return F("Precipitation intensity");
    case Sensor_VType::SENSOR_TYPE_SULPHUR_DIOXIDE:  return F("SO2");
    case Sensor_VType::SENSOR_TYPE_VOC_PARTS:        return F("VOC parts");
    case Sensor_VType::SENSOR_TYPE_VOLUME:           return F("Volume");
    case Sensor_VType::SENSOR_TYPE_VOLUME_FLOW_RATE: return F("Volume flow rate");
    case Sensor_VType::SENSOR_TYPE_VOLUME_STORAGE:   return F("Volume storage");
    case Sensor_VType::SENSOR_TYPE_WATER:            return F("Water cons.");
    #else // if FEATURE_MQTT_DISCOVER || FEATURE_CUSTOM_TASKVAR_VTYPE
    case Sensor_VType::SENSOR_TYPE_ANALOG_ONLY:
    case Sensor_VType::SENSOR_TYPE_TEMP_ONLY:
    case Sensor_VType::SENSOR_TYPE_HUM_ONLY:
    case Sensor_VType::SENSOR_TYPE_LUX_ONLY:
    case Sensor_VType::SENSOR_TYPE_DISTANCE_ONLY:
    case Sensor_VType::SENSOR_TYPE_DIRECTION_ONLY:
    case Sensor_VType::SENSOR_TYPE_DUSTPM2_5_ONLY:
    case Sensor_VType::SENSOR_TYPE_DUSTPM1_0_ONLY:
    case Sensor_VType::SENSOR_TYPE_DUSTPM10_ONLY:
    case Sensor_VType::SENSOR_TYPE_MOISTURE_ONLY:
    case Sensor_VType::SENSOR_TYPE_CO2_ONLY:
    case Sensor_VType::SENSOR_TYPE_GPS_ONLY:
    case Sensor_VType::SENSOR_TYPE_UV_ONLY:
    case Sensor_VType::SENSOR_TYPE_UV_INDEX_ONLY:
    case Sensor_VType::SENSOR_TYPE_IR_ONLY:
    case Sensor_VType::SENSOR_TYPE_WEIGHT_ONLY:
    case Sensor_VType::SENSOR_TYPE_VOLTAGE_ONLY:
    case Sensor_VType::SENSOR_TYPE_CURRENT_ONLY:
    case Sensor_VType::SENSOR_TYPE_POWER_USG_ONLY:
    case Sensor_VType::SENSOR_TYPE_POWER_FACT_ONLY:
    case Sensor_VType::SENSOR_TYPE_APPRNT_POWER_USG_ONLY:
    case Sensor_VType::SENSOR_TYPE_TVOC_ONLY:
    case Sensor_VType::SENSOR_TYPE_BARO_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_RED_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_GREEN_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_BLUE_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_TEMP_ONLY:
    case Sensor_VType::SENSOR_TYPE_REACTIVE_POWER_ONLY:
    case Sensor_VType::SENSOR_TYPE_AQI_ONLY:
    case Sensor_VType::SENSOR_TYPE_NOX_ONLY:
    case Sensor_VType::SENSOR_TYPE_SWITCH_INVERTED:
    case Sensor_VType::SENSOR_TYPE_WIND_SPEED:
    case Sensor_VType::SENSOR_TYPE_DURATION:
    case Sensor_VType::SENSOR_TYPE_DATE:
    case Sensor_VType::SENSOR_TYPE_TIMESTAMP:
    case Sensor_VType::SENSOR_TYPE_DATA_RATE:
    case Sensor_VType::SENSOR_TYPE_DATA_SIZE:
    case Sensor_VType::SENSOR_TYPE_SOUND_PRESSURE:
    case Sensor_VType::SENSOR_TYPE_SIGNAL_STRENGTH:
    case Sensor_VType::SENSOR_TYPE_REACTIVE_ENERGY:
    case Sensor_VType::SENSOR_TYPE_FREQUENCY:
    case Sensor_VType::SENSOR_TYPE_ENERGY:
    case Sensor_VType::SENSOR_TYPE_ENERGY_STORAGE:
    case Sensor_VType::SENSOR_TYPE_ABS_HUMIDITY:
    case Sensor_VType::SENSOR_TYPE_ATMOS_PRESSURE:
    case Sensor_VType::SENSOR_TYPE_BLOOD_GLUCOSE_C:
    case Sensor_VType::SENSOR_TYPE_CO_ONLY:
    case Sensor_VType::SENSOR_TYPE_ENERGY_DISTANCE:
    case Sensor_VType::SENSOR_TYPE_GAS_ONLY:
    case Sensor_VType::SENSOR_TYPE_NITROUS_OXIDE:
    case Sensor_VType::SENSOR_TYPE_OZONE_ONLY:
    case Sensor_VType::SENSOR_TYPE_PRECIPITATION:
    case Sensor_VType::SENSOR_TYPE_PRECIPITATION_INTEN:
    case Sensor_VType::SENSOR_TYPE_SULPHUR_DIOXIDE:
    case Sensor_VType::SENSOR_TYPE_VOC_PARTS:
    case Sensor_VType::SENSOR_TYPE_VOLUME:
    case Sensor_VType::SENSOR_TYPE_VOLUME_FLOW_RATE:
    case Sensor_VType::SENSOR_TYPE_VOLUME_STORAGE:
    case Sensor_VType::SENSOR_TYPE_WATER:
      break;
    #endif // if FEATURE_MQTT_DISCOVER || FEATURE_CUSTOM_TASKVAR_VTYPE
  }
  return F("");
}

#if FEATURE_CUSTOM_TASKVAR_VTYPE
bool isMQTTDiscoverySensorType(Sensor_VType sensorType)
{
  switch (sensorType)
  {
    case Sensor_VType::SENSOR_TYPE_NONE:
    case Sensor_VType::SENSOR_TYPE_SINGLE:        // single value sensor, used for Dallas, BH1750, etc
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_INT32_SINGLE:  // 1x int32_t
    case Sensor_VType::SENSOR_TYPE_UINT64_SINGLE: // 1x uint64_t
    case Sensor_VType::SENSOR_TYPE_INT64_SINGLE:  // 1x int64_t
    case Sensor_VType::SENSOR_TYPE_DOUBLE_SINGLE: // 1x ESPEASY_RULES_FLOAT_TYPE
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_ULONG:  // single unsigned LONG value, stored in two floats (rfid tags)
    case Sensor_VType::SENSOR_TYPE_STRING: // String type data stored in the event->String2
    case Sensor_VType::SENSOR_TYPE_DUAL:        // 2x float
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_UINT32_DUAL: // 2x uint32_t
    case Sensor_VType::SENSOR_TYPE_INT32_DUAL:  // 2x int32_t
    case Sensor_VType::SENSOR_TYPE_UINT64_DUAL: // 2x uint64_t
    case Sensor_VType::SENSOR_TYPE_INT64_DUAL:  // 2x int64_t
    case Sensor_VType::SENSOR_TYPE_DOUBLE_DUAL: // 2x ESPEASY_RULES_FLOAT_TYPE
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_TRIPLE:          // 3x float
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_UINT32_TRIPLE:   // 3x uint32_t
    case Sensor_VType::SENSOR_TYPE_INT32_TRIPLE:    // 3x int32_t
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_QUAD:        // 4x float
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_UINT32_QUAD: // 4x uint32_t
    case Sensor_VType::SENSOR_TYPE_INT32_QUAD:  // 4x int32_t
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
    case Sensor_VType::SENSOR_TYPE_NOT_SET:
    case Sensor_VType::SENSOR_TYPE_ANALOG_ONLY: // TODO Implement discovery
    case Sensor_VType::SENSOR_TYPE_DIMMER:      // TODO Implement discovery
    case Sensor_VType::SENSOR_TYPE_GPS_ONLY:    // TODO Implement discovery
      break;

    // All value types that are used in MQTT AutoDiscovery
    case Sensor_VType::SENSOR_TYPE_SWITCH:
    case Sensor_VType::SENSOR_TYPE_TEMP_HUM:
    case Sensor_VType::SENSOR_TYPE_TEMP_BARO:
    case Sensor_VType::SENSOR_TYPE_TEMP_HUM_BARO:
    case Sensor_VType::SENSOR_TYPE_TEMP_EMPTY_BARO: // Values 1 and 3 will contain data.
    case Sensor_VType::SENSOR_TYPE_WIND:
    case Sensor_VType::SENSOR_TYPE_TEMP_ONLY:
    case Sensor_VType::SENSOR_TYPE_HUM_ONLY:
    case Sensor_VType::SENSOR_TYPE_LUX_ONLY:
    case Sensor_VType::SENSOR_TYPE_DISTANCE_ONLY:
    case Sensor_VType::SENSOR_TYPE_DIRECTION_ONLY:
    case Sensor_VType::SENSOR_TYPE_DUSTPM2_5_ONLY:
    case Sensor_VType::SENSOR_TYPE_DUSTPM1_0_ONLY:
    case Sensor_VType::SENSOR_TYPE_DUSTPM10_ONLY:
    case Sensor_VType::SENSOR_TYPE_MOISTURE_ONLY:
    case Sensor_VType::SENSOR_TYPE_CO2_ONLY:
    case Sensor_VType::SENSOR_TYPE_UV_ONLY:
    case Sensor_VType::SENSOR_TYPE_UV_INDEX_ONLY:
    case Sensor_VType::SENSOR_TYPE_IR_ONLY:
    case Sensor_VType::SENSOR_TYPE_WEIGHT_ONLY:
    case Sensor_VType::SENSOR_TYPE_VOLTAGE_ONLY:
    case Sensor_VType::SENSOR_TYPE_CURRENT_ONLY:
    case Sensor_VType::SENSOR_TYPE_POWER_USG_ONLY:
    case Sensor_VType::SENSOR_TYPE_POWER_FACT_ONLY:
    case Sensor_VType::SENSOR_TYPE_APPRNT_POWER_USG_ONLY:
    case Sensor_VType::SENSOR_TYPE_TVOC_ONLY:
    case Sensor_VType::SENSOR_TYPE_BARO_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_RED_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_GREEN_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_BLUE_ONLY:
    case Sensor_VType::SENSOR_TYPE_COLOR_TEMP_ONLY:
    case Sensor_VType::SENSOR_TYPE_REACTIVE_POWER_ONLY:
    case Sensor_VType::SENSOR_TYPE_AQI_ONLY:
    case Sensor_VType::SENSOR_TYPE_NOX_ONLY:
    case Sensor_VType::SENSOR_TYPE_SWITCH_INVERTED:
    case Sensor_VType::SENSOR_TYPE_WIND_SPEED:
    case Sensor_VType::SENSOR_TYPE_DURATION:
    case Sensor_VType::SENSOR_TYPE_DATE:
    case Sensor_VType::SENSOR_TYPE_TIMESTAMP:
    case Sensor_VType::SENSOR_TYPE_DATA_RATE:
    case Sensor_VType::SENSOR_TYPE_DATA_SIZE:
    case Sensor_VType::SENSOR_TYPE_SOUND_PRESSURE:
    case Sensor_VType::SENSOR_TYPE_SIGNAL_STRENGTH:
    case Sensor_VType::SENSOR_TYPE_REACTIVE_ENERGY:
    case Sensor_VType::SENSOR_TYPE_FREQUENCY:
    case Sensor_VType::SENSOR_TYPE_ENERGY:
    case Sensor_VType::SENSOR_TYPE_ENERGY_STORAGE:
    case Sensor_VType::SENSOR_TYPE_ABS_HUMIDITY:
    case Sensor_VType::SENSOR_TYPE_ATMOS_PRESSURE:
    case Sensor_VType::SENSOR_TYPE_BLOOD_GLUCOSE_C:
    case Sensor_VType::SENSOR_TYPE_CO_ONLY:
    case Sensor_VType::SENSOR_TYPE_ENERGY_DISTANCE:
    case Sensor_VType::SENSOR_TYPE_GAS_ONLY:
    case Sensor_VType::SENSOR_TYPE_NITROUS_OXIDE:
    case Sensor_VType::SENSOR_TYPE_OZONE_ONLY:
    case Sensor_VType::SENSOR_TYPE_PRECIPITATION:
    case Sensor_VType::SENSOR_TYPE_PRECIPITATION_INTEN:
    case Sensor_VType::SENSOR_TYPE_SULPHUR_DIOXIDE:
    case Sensor_VType::SENSOR_TYPE_VOC_PARTS:
    case Sensor_VType::SENSOR_TYPE_VOLUME:
    case Sensor_VType::SENSOR_TYPE_VOLUME_FLOW_RATE:
    case Sensor_VType::SENSOR_TYPE_VOLUME_STORAGE:
    case Sensor_VType::SENSOR_TYPE_WATER:
      return true;
  }
  return false;
}
#endif // FEATURE_CUSTOM_TASKVAR_VTYPE

bool isSimpleOutputDataType(Sensor_VType sensorType)
{
  return sensorType == Sensor_VType::SENSOR_TYPE_SINGLE ||
         sensorType == Sensor_VType::SENSOR_TYPE_DUAL   ||
         sensorType == Sensor_VType::SENSOR_TYPE_TRIPLE ||
         sensorType == Sensor_VType::SENSOR_TYPE_QUAD;
}

bool isUInt32OutputDataType(Sensor_VType sensorType)
{
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
  return sensorType == Sensor_VType::SENSOR_TYPE_ULONG         ||
         sensorType == Sensor_VType::SENSOR_TYPE_UINT32_DUAL   ||
         sensorType == Sensor_VType::SENSOR_TYPE_UINT32_TRIPLE ||
         sensorType == Sensor_VType::SENSOR_TYPE_UINT32_QUAD;
#else // if FEATURE_EXTENDED_TASK_VALUE_TYPES
  return sensorType == Sensor_VType::SENSOR_TYPE_ULONG;
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
}

#if FEATURE_EXTENDED_TASK_VALUE_TYPES
bool isInt32OutputDataType(Sensor_VType sensorType)
{
  return sensorType == Sensor_VType::SENSOR_TYPE_INT32_SINGLE ||
         sensorType == Sensor_VType::SENSOR_TYPE_INT32_DUAL   ||
         sensorType == Sensor_VType::SENSOR_TYPE_INT32_TRIPLE ||
         sensorType == Sensor_VType::SENSOR_TYPE_INT32_QUAD;
}

bool isUInt64OutputDataType(Sensor_VType sensorType)
{
  return sensorType == Sensor_VType::SENSOR_TYPE_UINT64_SINGLE ||
         sensorType == Sensor_VType::SENSOR_TYPE_UINT64_DUAL;
}

bool isInt64OutputDataType(Sensor_VType sensorType)
{
  return sensorType == Sensor_VType::SENSOR_TYPE_INT64_SINGLE ||
         sensorType == Sensor_VType::SENSOR_TYPE_INT64_DUAL;
}

#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES

bool isFloatOutputDataType(Sensor_VType sensorType)
{
  return sensorType != Sensor_VType::SENSOR_TYPE_NONE &&
         sensorType != Sensor_VType::SENSOR_TYPE_ULONG &&
         sensorType < Sensor_VType::SENSOR_TYPE_STRING;
}

#if FEATURE_EXTENDED_TASK_VALUE_TYPES
bool isDoubleOutputDataType(Sensor_VType sensorType)
{
  return sensorType == Sensor_VType::SENSOR_TYPE_DOUBLE_SINGLE ||
         sensorType == Sensor_VType::SENSOR_TYPE_DOUBLE_DUAL;
}

#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES

bool isIntegerOutputDataType(Sensor_VType sensorType)
{
#if FEATURE_EXTENDED_TASK_VALUE_TYPES
  return isUInt32OutputDataType(sensorType)  ||
         isInt32OutputDataType(sensorType)   ||
         isUInt64OutputDataType(sensorType) ||
         isInt64OutputDataType(sensorType);
#else // if FEATURE_EXTENDED_TASK_VALUE_TYPES
  return isUInt32OutputDataType(sensorType);
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
}

bool is32bitOutputDataType(Sensor_VType sensorType)
{
#if FEATURE_EXTENDED_TASK_VALUE_TYPES

  if (isUInt64OutputDataType(sensorType) ||
      isInt64OutputDataType(sensorType) ||
      isDoubleOutputDataType(sensorType) ||
      (sensorType == Sensor_VType::SENSOR_TYPE_STRING)) {
    return false;
  }
#endif // if FEATURE_EXTENDED_TASK_VALUE_TYPES
  return true;
}

# if FEATURE_MQTT && FEATURE_MQTT_DISCOVER
const char mqtt_valueType_ha_deviceclass_names[] PROGMEM = // !! Offset, starting from Sensor_VType::SENSOR_TYPE_ANALOG_ONLY !!
  "|temperature|humidity|illuminance|distance|wind_direction|" // ANALOG_ONLY .. DIRECTION_ONLY
  "pm25|pm1|pm10|mdi:cup-water|carbon_dioxide||" // DUSTPM2_5_ONLY .. GPS_ONLY
  "irradiance|irradiance|irradiance|mdi:scale|" // UV_ONLY .. WEIGHT_ONLY
  "voltage|current|power|power_factor|power|" // VOLTAGE_ONLY .. APPRNT_POWER_USG_ONLY
  "volatile_organic_compounds|pressure|mdi:palette|mdi:palette|mdi:palette|" // TVOC_ONLY .. COLOR_BLUE_ONLY
  "mdi:temperature-kelvin|reactive_power|aqi|nitrogen_dioxide|" // COLOR_TEMP_ONLY .. NOX_ONLY
  "|wind_speed|duration|date|" // SWITCH_INVERTED .. DATE
  "timestamp|data_rate|data_size|sound_pressure|signal_strength|reactive_energy|" // TIMESTAMP .. REACTIVE_ENERGY
  "frequency|energy|energy_storage|" // FREQUENCY .. ENERGY_STORAGE
  "absolute_humidity|atmospheric_pressure|blood_glucose_concentration|" // ABS_HUMIDITY .. ATMOSPH_PRESSURE
  "carbon_monoxide|energy_distance|gas|nitrous_oxide|ozone|" // CO_ONLY .. OZONE_ONLY
  "precipitation|precipitation_intensity|sulphur_dioxide|volatile_organic_compounds_parts|" // PRECIPITATION .. VOC_PARTS
  "volume|volume_flow_rate|volume_storage|water|" // VOLUME .. WATER
  ;

/**
 * getValueType2HADeviceClass: Convert the ValueType to a HA Device Class
 */
String getValueType2HADeviceClass(Sensor_VType sensorType) {
  char tmp[33]{};                                                   // length: volatile_organic_compounds_parts + \0
  int devClassIndex = static_cast<int>(sensorType);
  if (sensorType >= Sensor_VType::SENSOR_TYPE_ANALOG_ONLY) {
    devClassIndex -= static_cast<int>(Sensor_VType::SENSOR_TYPE_ANALOG_ONLY); // Subtract offset
  // } else if (sensorType == Sensor_VType::SENSOR_TYPE_SWITCH) {
  //   return EMPTY_STRING;
  } else if (sensorType == Sensor_VType::SENSOR_TYPE_WIND) {
    return F("wind_speed");
  // } else if (sensorType == Sensor_VType::SENSOR_TYPE_DIMMER) {
  //   return EMPTY_STRING;
  } else {
    return EMPTY_STRING;
  }

  String result(GetTextIndexed(tmp, sizeof(tmp), devClassIndex, mqtt_valueType_ha_deviceclass_names));

  return result;
}

const char mqtt_valueType_default_ha_uom_names[] PROGMEM = // !! Offset, starting from Sensor_VType::SENSOR_TYPE_ANALOG_ONLY !!
  "|°C|%|lx|cm|°|" // ANALOG_ONLY .. DIRECTION_ONLY
  "µg/m³|µg/m³|µg/m³|%|ppm||" // DUSTPM2_5_ONLY .. GPS_ONLY
  "W/m²|UV Index|W/m²|kg|" // UV_ONLY .. WEIGHT_ONLY
  "V|A|W|%|VA|" // VOLTAGE_ONLY .. APPRNT_POWER_USG_ONLY
  "ppd|hPa|lx|lx|lx|" // TVOC_ONLY .. COLOR_BLUE_ONLY
  "K|var||µg/m³|" // COLOR_TEMP_ONLY .. NOX_ONLY
  "|m/s|min||" // SWITCH_INVERTED .. DATE
  "|bit/s|B|dB|dBm|kvar|" // TIMESTAMP .. REACTIVE_ENERGY
  "Hz|kWh|kWh|" // FREQUENCY .. ENERGY_STORAGE
  "g/m³|mbar|mmol/L|" // ABS_HUMIDITY .. BLOOD_GLUCOSE_C
  "ppm|kWh/100km|L|µg/m³|µg/m³|" // CO_ONLY .. OZONE_ONLY
  "mm|mm/h|µg/m³|ppm|" // PRICIPIATION .. VOC_PARTS
  "L|m³/h|L|L|" // VOLUME .. WATER
  ;

/**
 * getValueType2DefaultHAUoM: Convert the Value Type to a default UoM for HA AutoDiscovery
 */
String getValueType2DefaultHAUoM(Sensor_VType sensorType) {
  char tmp[9]{};                                                   // length: UV Index + \0
  int devClassIndex = static_cast<int>(sensorType);
  if (sensorType >= Sensor_VType::SENSOR_TYPE_ANALOG_ONLY) {
    devClassIndex -= static_cast<int>(Sensor_VType::SENSOR_TYPE_ANALOG_ONLY); // Subtract offset
  // } else if (sensorType == Sensor_VType::SENSOR_TYPE_SWITCH) {
  //   return EMPTY_STRING;
  } else if (sensorType == Sensor_VType::SENSOR_TYPE_WIND) {
    return F("m/s");
  // } else if (sensorType == Sensor_VType::SENSOR_TYPE_DIMMER) {
  //   return EMPTY_STRING;
  } else {
    return EMPTY_STRING;
  }

  String result(GetTextIndexed(tmp, sizeof(tmp), devClassIndex, mqtt_valueType_default_ha_uom_names));

  return result;
}
# endif // if FEATURE_MQTT && FEATURE_MQTT_DISCOVER

#include "../DataTypes/EthernetParameters.h"

bool isValid(EthClockMode_t clockMode) {
  switch (clockMode) {
    case  EthClockMode_t::Ext_crystal_osc:
    case  EthClockMode_t::Int_50MHz_GPIO_0:
    case  EthClockMode_t::Int_50MHz_GPIO_16:
    case  EthClockMode_t::Int_50MHz_GPIO_17_inv:
      return true;

      // Do not use default: as this allows the compiler to detect any missing cases.
  }
  return false;
}

bool isGpioUsedInETHClockMode(EthClockMode_t clockMode,
                              int8_t         gpio) {
  if (((clockMode == EthClockMode_t::Int_50MHz_GPIO_0)      && (gpio == 0)) ||
      ((clockMode == EthClockMode_t::Int_50MHz_GPIO_16)     && (gpio == 16)) ||
      ((clockMode == EthClockMode_t::Int_50MHz_GPIO_17_inv) && (gpio == 17))) {
    return true;
  }
  return false;
}

const __FlashStringHelper* toString(EthClockMode_t clockMode) {
  switch (clockMode) {
    case  EthClockMode_t::Ext_crystal_osc:       return F("External crystal oscillator");
    case  EthClockMode_t::Int_50MHz_GPIO_0:      return F("50MHz APLL Output on GPIO0");
    case  EthClockMode_t::Int_50MHz_GPIO_16:     return F("50MHz APLL Output on GPIO16");
    case  EthClockMode_t::Int_50MHz_GPIO_17_inv: return F("50MHz APLL Inverted Output on GPIO17");

      // Do not use default: as this allows the compiler to detect any missing cases.
  }
  return F("Unknown");
}

bool isValid(EthPhyType_t phyType) {
  switch (phyType) {
#if CONFIG_ETH_USE_ESP32_EMAC
    case EthPhyType_t::LAN8720:
    case EthPhyType_t::TLK110:
# if ESP_IDF_VERSION_MAJOR > 3
    case EthPhyType_t::RTL8201:
#if ETH_TYPE_JL1101_SUPPORTED
    case EthPhyType_t::JL1101:
#endif
    case EthPhyType_t::DP83848:
    case EthPhyType_t::KSZ8041:
    case EthPhyType_t::KSZ8081:
# endif // if ESP_IDF_VERSION_MAJOR > 3
      return true;
#endif // if CONFIG_ETH_USE_ESP32_EMAC

#if ESP_IDF_VERSION_MAJOR >= 5
# if CONFIG_ETH_SPI_ETHERNET_DM9051
    case EthPhyType_t::DM9051: return true;
# endif // if CONFIG_ETH_SPI_ETHERNET_DM9051
# if CONFIG_ETH_SPI_ETHERNET_W5500
    case EthPhyType_t::W5500:  return true;
# endif // if CONFIG_ETH_SPI_ETHERNET_W5500
# if CONFIG_ETH_SPI_ETHERNET_KSZ8851SNL
    case EthPhyType_t::KSZ8851: return true;
# endif // if CONFIG_ETH_SPI_ETHERNET_KSZ8851SNL
#endif // if ESP_IDF_VERSION_MAJOR >= 5
    case EthPhyType_t::notSet:
      break;
  }
  return false;
}

#if FEATURE_ETHERNET
bool isSPI_EthernetType(EthPhyType_t phyType) {
# if ESP_IDF_VERSION_MAJOR >= 5
  return
#  if CONFIG_ETH_SPI_ETHERNET_DM9051
    phyType ==  EthPhyType_t::DM9051 ||
#  endif // if CONFIG_ETH_SPI_ETHERNET_DM9051
#  if CONFIG_ETH_SPI_ETHERNET_W5500
    phyType ==  EthPhyType_t::W5500 ||
#  endif // if CONFIG_ETH_SPI_ETHERNET_W5500
#  if CONFIG_ETH_SPI_ETHERNET_KSZ8851SNL
    phyType ==  EthPhyType_t::KSZ8851 ||
#  endif // if CONFIG_ETH_SPI_ETHERNET_KSZ8851SNL
    false;
# else // if ESP_IDF_VERSION_MAJOR >= 5
  return false;
# endif // if ESP_IDF_VERSION_MAJOR >= 5
}

eth_phy_type_t to_ESP_phy_type(EthPhyType_t phyType)
{
  switch (phyType) {
# if CONFIG_ETH_USE_ESP32_EMAC
    case EthPhyType_t::LAN8720:  return ETH_PHY_LAN8720;
    case EthPhyType_t::TLK110:   return ETH_PHY_TLK110;
#  if ESP_IDF_VERSION_MAJOR > 3
    case EthPhyType_t::RTL8201:  return ETH_PHY_RTL8201;
#   if ETH_TYPE_JL1101_SUPPORTED
    case EthPhyType_t::JL1101:   return ETH_PHY_JL1101;
#   endif
    case EthPhyType_t::DP83848:  return ETH_PHY_DP83848;
    case EthPhyType_t::KSZ8041:  return ETH_PHY_KSZ8041;
    case EthPhyType_t::KSZ8081:  return ETH_PHY_KSZ8081;
#  endif // if ESP_IDF_VERSION_MAJOR > 3
# endif // if CONFIG_ETH_USE_ESP32_EMAC

# if ESP_IDF_VERSION_MAJOR >= 5
#  if CONFIG_ETH_SPI_ETHERNET_DM9051
    case EthPhyType_t::DM9051:   return ETH_PHY_DM9051;
#  endif // if CONFIG_ETH_SPI_ETHERNET_DM9051
#  if CONFIG_ETH_SPI_ETHERNET_W5500
    case EthPhyType_t::W5500:   return ETH_PHY_W5500;
#  endif // if CONFIG_ETH_SPI_ETHERNET_W5500
#  if CONFIG_ETH_SPI_ETHERNET_KSZ8851SNL
    case EthPhyType_t::KSZ8851:   return ETH_PHY_KSZ8851;
#  endif // if CONFIG_ETH_SPI_ETHERNET_KSZ8851SNL
# endif // if ESP_IDF_VERSION_MAJOR >= 5
    case EthPhyType_t::notSet:
      break;
  }
  return ETH_PHY_MAX;
}

#endif // if FEATURE_ETHERNET


const __FlashStringHelper* toString(EthPhyType_t phyType) {
  switch (phyType) {
#if CONFIG_ETH_USE_ESP32_EMAC
    case EthPhyType_t::LAN8720:  return F("LAN8710/LAN8720");
    case EthPhyType_t::TLK110:   return F("TLK110");
# if ESP_IDF_VERSION_MAJOR > 3
    case EthPhyType_t::RTL8201:  return F("RTL8201");
#if ETH_TYPE_JL1101_SUPPORTED
    case EthPhyType_t::JL1101:   return F("JL1101");
#endif
    case EthPhyType_t::DP83848:  return F("DP83848");
    case EthPhyType_t::KSZ8041:  return F("KSZ8041");
    case EthPhyType_t::KSZ8081:  return F("KSZ8081");
# endif // if ESP_IDF_VERSION_MAJOR > 3
#endif // if CONFIG_ETH_USE_ESP32_EMAC

#if ESP_IDF_VERSION_MAJOR >= 5
# if CONFIG_ETH_SPI_ETHERNET_DM9051
    case EthPhyType_t::DM9051:   return F("DM9051(SPI)");
# endif // if CONFIG_ETH_SPI_ETHERNET_DM9051
# if CONFIG_ETH_SPI_ETHERNET_W5500
    case EthPhyType_t::W5500:   return F("W5500(SPI)");
# endif // if CONFIG_ETH_SPI_ETHERNET_W5500
# if CONFIG_ETH_SPI_ETHERNET_KSZ8851SNL
    case EthPhyType_t::KSZ8851:   return F("KSZ8851(SPI)");
# endif // if CONFIG_ETH_SPI_ETHERNET_KSZ8851SNL
#endif // if ESP_IDF_VERSION_MAJOR >= 5
    case EthPhyType_t::notSet:
      break;

      // Do not use default: as this allows the compiler to detect any missing cases.
  }
  return F("- None -");
}

#include "../DataTypes/TaskEnabledState.h"

TaskEnabledState::TaskEnabledState() : value(0) {}

TaskEnabledState & TaskEnabledState::operator=(const bool& enabledState)
{
  enabled = enabledState;
  return *this;
}

void TaskEnabledState::clearTempDisableFlags()
{
  const bool enabledSet = enabled;

  value   = 0;
  enabled = enabledSet;
}

void TaskEnabledState::setRetryInit()
{
  // If intended state is enabled, set to retry.
  retryInit = enabled;
}

#include "../DataTypes/SettingsType.h"

#include "../CustomBuild/StorageLayout.h"
#include "../DataStructs/ControllerSettingsStruct.h"
#include "../DataStructs/ExtraTaskSettingsStruct.h"
#include "../DataStructs/NotificationSettingsStruct.h"
#include "../DataStructs/SecurityStruct.h"
#include "../DataTypes/ESPEasyFileType.h"
#include "../Globals/Settings.h"
#include "../Helpers/StringConverter.h"

const __FlashStringHelper * SettingsType::getSettingsTypeString(Enum settingsType) {
  switch (settingsType) {
    case Enum::BasicSettings_Type:             return F("Settings");
    case Enum::TaskSettings_Type:              return F("TaskSettings");
    case Enum::CustomTaskSettings_Type:        return F("CustomTaskSettings");
    case Enum::ControllerSettings_Type:        return F("ControllerSettings");
    case Enum::CustomControllerSettings_Type:  return F("CustomControllerSettings");
    case Enum::NotificationSettings_Type:      
    #if FEATURE_NOTIFIER
        return F("NotificationSettings");
    #else
        break;
    #endif
    case Enum::SecuritySettings_Type:          return F("SecuritySettings");
    case Enum::ExtdControllerCredentials_Type: return F("ExtendedControllerCredentials");
    #if FEATURE_ALTERNATIVE_CDN_URL
    case Enum::CdnSettings_Type:               return F("CDN_url");
    #endif

    case Enum::SettingsType_MAX: break;
  }
  return F("");
}

/********************************************************************************************\
   Offsets in settings files
 \*********************************************************************************************/
bool SettingsType::getSettingsParameters(Enum settingsType, int index, int& max_index, int& offset, int& max_size, int& struct_size) {
  // The defined offsets should be used with () just in case they are the result of a formula in the defines.
  struct_size = 0;
  max_index = -1;
  offset    = -1;

  switch (settingsType) {
    case Enum::BasicSettings_Type:
    {
      max_index   = 1;
      offset      = 0;
      max_size    = (DAT_BASIC_SETTINGS_SIZE);
      struct_size = sizeof(SettingsStruct);
      break;
    }
    case Enum::TaskSettings_Type:
    {
      max_index   = TASKS_MAX;
      offset      = (DAT_OFFSET_TASKS) + (index * (DAT_TASKS_DISTANCE));
      max_size    = DAT_TASKS_SIZE;
      struct_size = sizeof(ExtraTaskSettingsStruct);
      break;
    }
    case Enum::CustomTaskSettings_Type:
    {
      if (!getSettingsParameters(Enum::TaskSettings_Type, index, max_index, offset, max_size, struct_size))
        return false;
      offset  += (DAT_TASKS_CUSTOM_OFFSET);
      max_size = (DAT_TASKS_CUSTOM_SIZE + DAT_TASKS_CUSTOM_EXTENSION_SIZE);

      // struct_size may differ.
      struct_size = 0;
      break;
    }
    case Enum::ControllerSettings_Type:
    {
      max_index   = CONTROLLER_MAX;
      offset      = (DAT_OFFSET_CONTROLLER) + (index * (DAT_CONTROLLER_SIZE));
      max_size    = DAT_CONTROLLER_SIZE;
      struct_size = sizeof(ControllerSettingsStruct);
      break;
    }
    case Enum::CustomControllerSettings_Type:
    {
      max_index = CONTROLLER_MAX;
      offset    = (DAT_OFFSET_CUSTOM_CONTROLLER) + (index * (DAT_CUSTOM_CONTROLLER_SIZE));
      max_size  = DAT_CUSTOM_CONTROLLER_SIZE;

      // struct_size may differ.
      struct_size = 0;
      break;
    }
    case Enum::NotificationSettings_Type:
    {
#if FEATURE_NOTIFIER
      max_index   = NOTIFICATION_MAX;
      offset      = index * (DAT_NOTIFICATION_SIZE);
      max_size    = DAT_NOTIFICATION_SIZE;
      struct_size = sizeof(NotificationSettingsStruct);
      break;
#else
      return false;
#endif
    }
    case Enum::SecuritySettings_Type:
    {
      max_index   = 1;
      offset      = 0;
      max_size    = DAT_SECURITYSETTINGS_SIZE;
      struct_size = sizeof(SecurityStruct);
      break;
    }
    case Enum::ExtdControllerCredentials_Type:
    {
      max_index = 1;
      offset    = DAT_EXTDCONTR_CRED_OFFSET;
      max_size  = DAT_EXTDCONTR_CRED_SIZE;

      // struct_size may differ.
      struct_size = 0;
      break;
    }
#if FEATURE_ALTERNATIVE_CDN_URL
    case Enum::CdnSettings_Type:
    {
      max_index   = 1;
      offset      = DAT_OFFSET_CDN;
      max_size    = DAT_CDN_SIZE;

      // struct_size may differ.
      struct_size = 0;
    }
    break;
#endif

    case Enum::SettingsType_MAX:
    {
      max_index = -1;
      offset    = -1;
      return false;
    }
  }
  return index >= 0 && index < max_index;
}

bool SettingsType::getSettingsParameters(Enum settingsType, int index, int& offset, int& max_size) {
  int max_index = -1;
  int struct_size;

  if (!getSettingsParameters(settingsType, index, max_index, offset, max_size, struct_size)) {
    return false;
  }

  if ((index >= 0) && (index < max_index)) { return true; }
  offset = -1;
  return false;
}

int SettingsType::getMaxFilePos(Enum settingsType) {
  int max_index, offset, max_size{};
  int struct_size = 0;

  if (getSettingsParameters(settingsType, 0,             max_index, offset, max_size, struct_size) &&
      getSettingsParameters(settingsType, max_index - 1, offset,    max_size))
    return offset + max_size - 1;
  return -1;
}

int SettingsType::getFileSize(Enum settingsType) {
  SettingsType::SettingsFileEnum file_type = SettingsType::getSettingsFile(settingsType);
  int max_file_pos                         = 0;

  for (int st = 0; st < static_cast<int>(Enum::SettingsType_MAX); ++st) {
    if (SettingsType::getSettingsFile(static_cast<Enum>(st)) == file_type) {
      const int filePos = SettingsType::getMaxFilePos(static_cast<Enum>(st));

      if (filePos > max_file_pos) {
        max_file_pos = filePos;
      }
    }
  }
  return max_file_pos;
}

#ifndef BUILD_MINIMAL_OTA
unsigned int SettingsType::getSVGcolor(Enum settingsType) {
  switch (settingsType) {
    case Enum::BasicSettings_Type:
      return 0x5F0A87;
    case Enum::TaskSettings_Type:
      return 0xEE6352;
    case Enum::CustomTaskSettings_Type:
      return 0x59CD90;
    case Enum::ControllerSettings_Type:
      return 0x3FA7D6;
    case Enum::CustomControllerSettings_Type:
      return 0xFAC05E;
    case Enum::NotificationSettings_Type:
      return 0xF79D84;

    case Enum::SecuritySettings_Type:
      return 0xff00a2;
    case Enum::ExtdControllerCredentials_Type:
      return 0xc300ff;
#if FEATURE_ALTERNATIVE_CDN_URL
    case Enum::CdnSettings_Type:
      return 0xff6600;
#endif
    case Enum::SettingsType_MAX:
      break;
  }
  return 0;
}

#endif // ifndef BUILD_MINIMAL_OTA

SettingsType::SettingsFileEnum SettingsType::getSettingsFile(Enum settingsType)
{
  switch (settingsType) {
    case Enum::BasicSettings_Type:
    case Enum::TaskSettings_Type:
    case Enum::CustomTaskSettings_Type:
    case Enum::ControllerSettings_Type:
    case Enum::CustomControllerSettings_Type:
#if FEATURE_ALTERNATIVE_CDN_URL
    case Enum::CdnSettings_Type:
#endif
      return SettingsFileEnum::FILE_CONFIG_type;
    case Enum::NotificationSettings_Type:
      return SettingsFileEnum::FILE_NOTIFICATION_type;
    case Enum::SecuritySettings_Type:
    case Enum::ExtdControllerCredentials_Type:
      return SettingsFileEnum::FILE_SECURITY_type;

    case Enum::SettingsType_MAX:
      break;
  }
  return SettingsFileEnum::FILE_UNKNOWN_type;
}

String SettingsType::getSettingsFileName(Enum settingsType, int index) {
  #if FEATURE_EXTENDED_CUSTOM_SETTINGS
  if ((Enum::CustomTaskSettings_Type == settingsType) && validTaskIndex(index)) {
    return strformat(F(DAT_TASKS_CUSTOM_EXTENSION_FILEMASK), index + 1); // Add 0/1 offset to match displayed task ID
  }
  #endif // if FEATURE_EXTENDED_CUSTOM_SETTINGS
  return getSettingsFileName(getSettingsFile(settingsType));
}

const __FlashStringHelper * SettingsType::getSettingsFileName(SettingsType::SettingsFileEnum file_type) {
  switch (file_type) {
    case SettingsFileEnum::FILE_CONFIG_type:        return getFileName(FileType::CONFIG_DAT);
    case SettingsFileEnum::FILE_NOTIFICATION_type:  return getFileName(FileType::NOTIFICATION_DAT);
    case SettingsFileEnum::FILE_SECURITY_type:      return getFileName(FileType::SECURITY_DAT);
    case SettingsFileEnum::FILE_UNKNOWN_type:       break;
  }
  return F("");
}

size_t SettingsType::getInitFileSize(SettingsType::SettingsFileEnum file_type) {
  switch (file_type) {
    case SettingsFileEnum::FILE_CONFIG_type:        return CONFIG_FILE_SIZE;
    case SettingsFileEnum::FILE_NOTIFICATION_type:  return 4096;
    case SettingsFileEnum::FILE_SECURITY_type:      return 4096;
    case SettingsFileEnum::FILE_UNKNOWN_type:       break;
  }
  return 0;
}
#include "../DataTypes/SchedulerTimerType.h"



const __FlashStringHelper * toString(SchedulerTimerType_e timerType) {
  switch (timerType) {
    case SchedulerTimerType_e::SystemEventQueue:        return F("SystemEventQueue");
    case SchedulerTimerType_e::ConstIntervalTimer:      return F("Const Interval");
    case SchedulerTimerType_e::PLUGIN_TASKTIMER_IN_e:   return F("PLUGIN_TASKTIMER_IN");
    case SchedulerTimerType_e::TaskDeviceTimer:         return F("PLUGIN_READ");
    case SchedulerTimerType_e::GPIO_timer:              return F("GPIO_timer");
    case SchedulerTimerType_e::PLUGIN_DEVICETIMER_IN_e: return F("PLUGIN_DEVICETIMER_IN");
    case SchedulerTimerType_e::RulesTimer:              return F("Rules#Timer");
    case SchedulerTimerType_e::IntendedReboot:          return F("Intended Reboot");
  }
  return F("Unknown");
}

#include "../DataTypes/ESPEasyTimeSource.h"

#include "../../ESPEasy_common.h"

const __FlashStringHelper* toString(timeSource_t timeSource)
{
  switch (timeSource) {
    case timeSource_t::GPS_PPS_time_source:      return F("GPS PPS");
    case timeSource_t::GPS_time_source:          return F("GPS");
    case timeSource_t::GPS_time_source_no_fix:   return F("GPS no Fix");
    case timeSource_t::NTP_time_source:          return F("NTP");
    case timeSource_t::Manual_set:               return F("Manual");
    case timeSource_t::ESP_now_peer:             return F(ESPEASY_NOW_NAME " peer");
    case timeSource_t::ESPEASY_p2p_UDP:          return F("ESPEasy p2p");
    case timeSource_t::External_RTC_time_source: return F("Ext. RTC at boot");
    case timeSource_t::Restore_RTC_time_source:  return F("RTC at boot");
    case timeSource_t::No_time_source:           return F("No time set");
  }
  return F("Unknown");
}

bool isExternalTimeSource(timeSource_t timeSource)
{
  // timeSource_t::ESP_now_peer or timeSource_t::ESPEASY_p2p_UDP
  // should NOT be considered "external"
  // It may be an unreliable source if no other source is present in the network.

  switch (timeSource) {
    case timeSource_t::GPS_PPS_time_source:
    case timeSource_t::GPS_time_source:
    case timeSource_t::GPS_time_source_no_fix:
    case timeSource_t::NTP_time_source:
    case timeSource_t::External_RTC_time_source:
    case timeSource_t::Manual_set:
      return true;
    default:
      return false;
  }
}

// Typical time wander for ESP nodes should be less than 10 ppm
// Meaning per hour, the time should wander less than 36 msec.
#define TIME_WANDER_FACTOR  100000

uint32_t updateExpectedWander(
  int32_t  current_wander,
  uint32_t timePassedSinceLastTimeSync)
{
  if (current_wander < 0) {
    return current_wander;
  }
  return current_wander + (timePassedSinceLastTimeSync / TIME_WANDER_FACTOR);
}

uint32_t computeExpectedWander(timeSource_t timeSource,
                               uint32_t     timePassedSinceLastTimeSync)
{
  uint32_t expectedWander_ms = timePassedSinceLastTimeSync / TIME_WANDER_FACTOR;

  switch (timeSource) {
    case timeSource_t::GPS_PPS_time_source:
    {
      expectedWander_ms += 1;
      break;
    }
    case timeSource_t::GPS_time_source:
    {
      // Not sure about the wander here, as GPS does not have a drift.
      // But the moment a message is received from a second's start may differ.
      expectedWander_ms += 10;
      break;
    }
    case timeSource_t::GPS_time_source_no_fix:
    {
      // When the GPS has no fix, the reported time may differ quite a bit
      expectedWander_ms += 2000;
      break;
    }
    case timeSource_t::NTP_time_source:
    {
      // Typical time needed to perform a NTP request to an online NTP server
      expectedWander_ms += 30;
      break;
    }

    case timeSource_t::ESP_now_peer:
    case timeSource_t::ESPEASY_p2p_UDP:
    {
      // expected wander is 36 per hour.
      // Using a 'penalty' of 1000 makes it only preferrable over NTP after +/- 28 hour.
      expectedWander_ms += 1000;
      break;
    }

    case timeSource_t::External_RTC_time_source:
    {
      // Will be off by +/- 500 msec
      expectedWander_ms += 500;
      break;
    }
    case timeSource_t::Restore_RTC_time_source:
    {
      // May be off by the time needed to reboot + some time since the last update of the RTC
      // If a reboot was due to a watchdog reset, then it will be an additional 2 - 6 seconds.
      expectedWander_ms += 5000;
      break;
    }
    case timeSource_t::Manual_set:
    {
      expectedWander_ms += 10000;
      break;
    }
    case timeSource_t::No_time_source:
    {
      // Cannot sync from it.
      return 1 << 30;
    }
  }
  return expectedWander_ms;
}

#include "../ESPEasyCore/ESPEasy_loop.h"


#include "../../ESPEasy-Globals.h"
#include "../Commands/ExecuteCommand.h"
#include "../DataStructs/TimingStats.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/ESPEasyWifi_ProcessEvent.h"
#include "../ESPEasyCore/ESPEasy_backgroundtasks.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../Globals/ESPEasy_Scheduler.h"
#include "../Globals/EventQueue.h"
#include "../Globals/RTC.h"
#include "../Globals/Settings.h"
#include "../Globals/Statistics.h"
#include "../Helpers/DeepSleep.h"
#include "../Helpers/ESPEasyRTC.h"
#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/I2C_access.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Networking.h"
#include "../Helpers/PeriodicalActions.h"
#include "../Helpers/StringConverter.h"


#include "../Commands/InternalCommands_decoder.h"

void updateLoopStats() {
  ++loopCounter;
  ++loopCounter_full;

  if (lastLoopStart == 0) {
    lastLoopStart = micros();
    return;
  }
  const int32_t usecSince = usecPassedSince_fast(lastLoopStart);

  #if FEATURE_TIMING_STATS
  ADD_TIMER_STAT(LOOP_STATS, usecSince);
  #endif // if FEATURE_TIMING_STATS

  loop_usec_duration_total += usecSince;
  lastLoopStart             = micros();

  if ((usecSince <= 0) || (usecSince > 10000000)) {
    return; // No loop should take > 10 sec.
  }

  if (shortestLoop > static_cast<unsigned long>(usecSince)) {
    shortestLoop   = usecSince;
    loopCounterMax = 30 * 1000000 / usecSince;
  }

  if (longestLoop < static_cast<unsigned long>(usecSince)) {
    longestLoop = usecSince;
  }
}

/*********************************************************************************************\
* MAIN LOOP
\*********************************************************************************************/
void ESPEasy_loop()
{
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif
  /*
     //FIXME TD-er: No idea what this does.
     if(MainLoopCall_ptr)
      MainLoopCall_ptr();
   */

  updateLoopStats();

  handle_unprocessedNetworkEvents();

  bool firstLoopConnectionsEstablished = NetworkConnected() && firstLoop;

  if (firstLoopConnectionsEstablished) {
    #ifndef BUILD_MINIMAL_OTA
    addLog(LOG_LEVEL_INFO, F("firstLoopConnectionsEstablished"));
    #endif
    firstLoop               = false;
    timerAwakeFromDeepSleep = millis(); // Allow to run for "awake" number of seconds, now we have wifi.

    // schedule_all_task_device_timers(); // Disabled for now, since we are now using queues for controllers.
    if (Settings.UseRules && isDeepSleepEnabled())
    {
      String event = F("System#NoSleep=");
      event += Settings.deepSleep_wakeTime;
      eventQueue.addMove(std::move(event));
    }

#ifndef BUILD_NO_DEBUG
    checkAll_internalCommands();
#endif


    RTC.bootFailedCount = 0;
    saveToRTC();
    #if FEATURE_ESPEASY_P2P
    sendSysInfoUDP(1);
    #endif
  }
#if FEATURE_CLEAR_I2C_STUCK
  if (Settings.EnableClearHangingI2Cbus())
  {
    // Check I2C bus to see if it needs to be cleared.
    // See: http://www.forward.com.au/pfod/ArduinoProgramming/I2C_ClearBus/index.html
    const I2C_bus_state I2C_state_prev = I2C_state;  
    I2C_state = I2C_check_bus(Settings.Pin_i2c_scl, Settings.Pin_i2c_sda);
    switch (I2C_state) {
      case I2C_bus_state::BusCleared:
        // Log I2C bus cleared, update stats
        ++I2C_bus_cleared_count;
        addLog(LOG_LEVEL_ERROR, F("I2C  : Cleared I2C bus error state"));
        I2C_state = I2C_bus_state::OK;
        initI2C();
        break;
      case I2C_bus_state::SCL_Low:
        addLog(LOG_LEVEL_ERROR, F("I2C  : I2C bus error, SCL clock line held low"));
        break;
      case I2C_bus_state::SDA_Low_over_2_sec:
        addLog(LOG_LEVEL_ERROR, F("I2C  : I2C bus error, SCL clock line held low by slave clock stretch for >2 sec"));
        break;
      case I2C_bus_state::SDA_Low_20_clocks:
        addLog(LOG_LEVEL_ERROR, F("I2C  : I2C bus error, SDA data line held low"));
        break;
      case I2C_bus_state::ClearingProcessActive:
        if (I2C_state_prev != I2C_state) {
          addLog(LOG_LEVEL_ERROR, F("I2C  : I2C bus error, start clearing process"));
        }
        break;
      case I2C_bus_state::NotConfigured:
      case I2C_bus_state::OK:
        break;
    }
  }
#endif


  // Work around for nodes that do not have WiFi connection for a long time and may reboot after N unsuccessful connect attempts
  static bool bootFailedCountReset = false;
  if (getUptimeMinutes() > 2 && !bootFailedCountReset) {
    // Apparently the uptime is already a few minutes. Let's consider it a successful boot.
    RTC.bootFailedCount = 0;
    saveToRTC();
    bootFailedCountReset = true;
  }

  // Deep sleep mode, just run all tasks one (more) time and go back to sleep as fast as possible
  if ((firstLoopConnectionsEstablished || readyForSleep()) && isDeepSleepEnabled())
  {
#if FEATURE_MQTT
    runPeriodicalMQTT();
#endif // if FEATURE_MQTT
    // Now run all frequent tasks
    run50TimesPerSecond();
    run10TimesPerSecond();
    runEach30Seconds();
    runOncePerSecond();
  }

  // normal mode, run each task when its time
  else
  {
    if (!UseRTOSMultitasking) {
      // On ESP32, when using RTOS multitasking, the schedule is executed in a separate RTOS task
      Scheduler.handle_schedule();
    }
  }

  // Calls above may have received/generated commands for the command queue, thus need to process them.
  processExecuteCommandQueue();
  backgroundtasks();

  if (readyForSleep()) {
    prepare_deepSleep(Settings.Delay);

    // deepsleep will never return, its a special kind of reboot
  }
}

#include "../ESPEasyCore/ESPEasy_Console.h"


#include "../Commands/ExecuteCommand.h"

#include "../DataStructs/TimingStats.h"

#include "../DataTypes/ESPEasy_plugin_functions.h"

#include "../Globals/Cache.h"
#include "../Globals/Logging.h"
#include "../Globals/Plugins.h"
#include "../Globals/Settings.h"

#include "../Helpers/Memory.h"

#include <ESPEasySerialPort.h>


EspEasy_Console_t::EspEasy_Console_t()
{
#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  const ESPEasySerialPort port = static_cast<ESPEasySerialPort>(_console_serial_port);

# if USES_USBCDC

  /*
     if (port == ESPEasySerialPort::usb_cdc_0 ||
        port == ESPEasySerialPort::usb_cdc_1)
     {
      USB.manufacturerName(F("ESPEasy"));
      USB.productName()
     }
   */
# endif // if USES_USBCDC

# ifdef ESP8266
  constexpr size_t buffSize = 256;
# endif // ifdef ESP8266
# ifdef ESP32

  // Ideal buffer size is a trade-off between bootspeed
  // and not missing data when the ESP is busy processing stuff.
  // Since we do have a separate buffer in the console,
  // it may just take less time in the background tasks to dump
  // any logs as larger chunks can be transferred at once.
  constexpr size_t buffSize = 512;
# endif // ifdef ESP32

  ESPEasySerialConfig config;
  config.port          = port;
  config.baud          = DEFAULT_SERIAL_BAUD;
  config.receivePin    = _console_serial_rxpin;
  config.transmitPin   = _console_serial_txpin;
  config.inverse_logic = false;
  config.rxBuffSize    = 256;
  config.txBuffSize    = buffSize;

  {
    # ifdef USE_SECOND_HEAP
    HeapSelectDram ephemeral;
    # endif // ifdef USE_SECOND_HEAP

    _mainSerial._serial = new (std::nothrow) ESPeasySerial(config);
  }

# if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  if (Settings.console_serial0_fallback && (port != ESPEasySerialPort::serial0)) {
    config.port        = ESPEasySerialPort::serial0;
    config.receivePin  = SOC_RX0;
    config.transmitPin = SOC_TX0;

    _fallbackSerial._serial = new (std::nothrow) ESPeasySerial(config);
  }
# endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT


#endif  // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
}

void EspEasy_Console_t::reInit()
{
  updateActiveTaskUseSerial0();
  bool somethingChanged = false;
#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  const ESPEasySerialPort port = static_cast<ESPEasySerialPort>(Settings.console_serial_port);

  const bool consoleUseSerial0 = (
# ifdef ESP8266
    (port == ESPEasySerialPort::serial0_swap) ||
# endif // ifdef ESP8266
    port == ESPEasySerialPort::serial0);

  const bool canUseSerial0 = !activeTaskUseSerial0() && !log_to_serial_disabled;


  bool mustHaveSerial = Settings.UseSerial && (!consoleUseSerial0 || canUseSerial0);


# if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  bool mustHaveFallback = false;

  if (Settings.UseSerial) {
    if (consoleUseSerial0 && canUseSerial0 && (_fallbackSerial._serial != nullptr)) {
      // Should not destruct an already running fallback serial port
      mustHaveFallback = true;
      mustHaveSerial   = false;
    } else {
      if (Settings.console_serial0_fallback && !consoleUseSerial0 && canUseSerial0)
      {
        mustHaveFallback = true;
      }
    }
  }

  if (!mustHaveFallback) {
    if (_fallbackSerial._serial != nullptr) {
      _fallbackSerial._serial->flush();
      _fallbackSerial._serial->end();
      delete _fallbackSerial._serial;
      _fallbackSerial._serial = nullptr;
      somethingChanged = true;
    }
  }
# endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT


  if ((_console_serial_port != Settings.console_serial_port) ||
      (_console_serial_rxpin != Settings.console_serial_rxpin) ||
      (_console_serial_txpin != Settings.console_serial_txpin) ||
      !mustHaveSerial) {
    if (_mainSerial._serial != nullptr) {
      _mainSerial._serial->flush();
      _mainSerial._serial->end();
      delete _mainSerial._serial;
      _mainSerial._serial = nullptr;
      somethingChanged = true;
    }

    _console_serial_port  = Settings.console_serial_port;
    _console_serial_rxpin = Settings.console_serial_rxpin;
    _console_serial_txpin = Settings.console_serial_txpin;
  }

  if ((_mainSerial._serial == nullptr) && mustHaveSerial) {
    # ifdef USE_SECOND_HEAP
    HeapSelectDram ephemeral;
    # endif // ifdef USE_SECOND_HEAP

    unsigned int buffsize = 128;

    const ESPEasySerialPort mainSerialPort = static_cast<ESPEasySerialPort>(_console_serial_port);

#if USES_HWCDC
    if (mainSerialPort == ESPEasySerialPort::usb_hw_cdc) {
      buffsize = 2048;
    }
#endif // if USES_HWCDC

    _mainSerial._serial = new (std::nothrow) ESPeasySerial(
      mainSerialPort,
      _console_serial_rxpin,
      _console_serial_txpin, 
      false,
      buffsize);
    somethingChanged = true;
  }
# if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  if ((_fallbackSerial._serial == nullptr) && mustHaveFallback) {
    _fallbackSerial._serial = new (std::nothrow) ESPeasySerial(
      ESPEasySerialPort::serial0,
      SOC_RX0,
      SOC_TX0);
    somethingChanged = true;
  }
# endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT


# if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  if (_fallbackSerial._serial == nullptr) {
    _fallbackSerial._serialWriteBuffer.clear();
  }

# endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  if (_mainSerial._serial == nullptr) {
    _mainSerial._serialWriteBuffer.clear();
  }
#endif // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  if (somethingChanged) {
    begin(Settings.BaudRate);
  }
}

void EspEasy_Console_t::begin(uint32_t baudrate)
{
  updateActiveTaskUseSerial0();
  _baudrate = baudrate;

  if (_mainSerial._serial != nullptr) {
#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
    _mainSerial._serial->begin(baudrate);
    addLog(LOG_LEVEL_INFO, F("ESPEasy console using ESPEasySerial"));
#else // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
# ifdef ESP8266
    _mainSerial._serial->begin(baudrate);
    #ifndef BUILD_MINIMAL_OTA
    addLog(LOG_LEVEL_INFO, F("ESPEasy console using HW Serial"));
    #endif
# endif // ifdef ESP8266
# ifdef ESP32

    // Allow to flush data from the serial buffers
    // When not opening the USB serial port, the ESP may hang at boot.
    delay(10);
    _mainSerial._serial->end();
    delay(10);
    _mainSerial._serial->begin(baudrate);
    _mainSerial._serial->flush();
# endif // ifdef ESP32
#endif  // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  }
#if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  if (_fallbackSerial._serial != nullptr) {
    _fallbackSerial._serial->begin(baudrate);
    addLog(LOG_LEVEL_INFO, F("ESPEasy console fallback enabled"));
  }
#endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT
}

void EspEasy_Console_t::init() {
#if FEATURE_IMPROV
  _mainSerial._improv.init();
# if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
#  if USES_ESPEASY_CONSOLE_FALLBACK_PORT
  _fallbackSerial._improv.init();
#  endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT
# endif // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
#endif // if FEATURE_IMPROV
  updateActiveTaskUseSerial0();

  if (!Settings.UseSerial) {
    return;
  }

#if !FEATURE_DEFINE_SERIAL_CONSOLE_PORT

  if (activeTaskUseSerial0() || log_to_serial_disabled) {
    return;
  }
#endif // if !FEATURE_DEFINE_SERIAL_CONSOLE_PORT

  begin(Settings.BaudRate);
}

void EspEasy_Console_t::loop()
{
  if (!Settings.UseSerial) return;

  START_TIMER;

#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  const bool consoleUsesSerial0 =
    (static_cast<ESPEasySerialPort>(_console_serial_port) == ESPEasySerialPort::serial0
# ifdef ESP8266
     || static_cast<ESPEasySerialPort>(_console_serial_port) == ESPEasySerialPort::serial0_swap
# endif // ifdef ESP8266
    );

  if (handledByPluginSerialIn())
  {
    // Any serial0 data is already dealt with
    if (!consoleUsesSerial0 && (_mainSerial._serial != nullptr)) {
      readInput(_mainSerial);
    }
    return;
  }
#else // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT

  if (handledByPluginSerialIn()) {
    return;
  }
#endif // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT

  readInput(_mainSerial);
#if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  readInput(_fallbackSerial);
#endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  STOP_TIMER(CONSOLE_LOOP);
}

void EspEasy_Console_t::addToSerialBuffer(const __FlashStringHelper *line)
{
  addToSerialBuffer(String(line));
}

void EspEasy_Console_t::addToSerialBuffer(char c)
{
  _mainSerial.addToSerialBuffer(c);
#if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  _fallbackSerial.addToSerialBuffer(c);
#endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT
  process_serialWriteBuffer();
}

void EspEasy_Console_t::addToSerialBuffer(const String& line) {
  process_serialWriteBuffer();

  _mainSerial.addToSerialBuffer(line);
#if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  _fallbackSerial.addToSerialBuffer(line);
#endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT
  process_serialWriteBuffer();
}

void EspEasy_Console_t::addNewlineToSerialBuffer() {
  _mainSerial.addNewlineToSerialBuffer();
#if USES_ESPEASY_CONSOLE_FALLBACK_PORT
  _fallbackSerial.addNewlineToSerialBuffer();
#endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT
  process_serialWriteBuffer();
}

bool EspEasy_Console_t::process_serialWriteBuffer() {
  START_TIMER;
  bool res = false;

  if (_mainSerial.process_serialWriteBuffer()) {
    res = true;
  }

#if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  if (_fallbackSerial.process_serialWriteBuffer()) {
    res = true;
  }
#endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  STOP_TIMER(CONSOLE_WRITE_SERIAL);
  return res;
}

void EspEasy_Console_t::setDebugOutput(bool enable)
{
  auto port = getPort();

  if (port != nullptr) {
    port->setDebugOutput(enable);
  }
}

String EspEasy_Console_t::getPortDescription() const
{
  return _mainSerial.getPortDescription();
}

#if USES_ESPEASY_CONSOLE_FALLBACK_PORT
String EspEasy_Console_t::getFallbackPortDescription() const
{
  return _fallbackSerial.getPortDescription();
}

#endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT

bool EspEasy_Console_t::handledByPluginSerialIn()
{
#if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  if ((_fallbackSerial._serial != nullptr) &&
      _fallbackSerial._serial->available())
  {
    String dummy;

    return PluginCall(PLUGIN_SERIAL_IN, 0, dummy);
  }
#endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT
#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT

  if ((_mainSerial._serial != nullptr) && _mainSerial._serial->available() &&
      (static_cast<ESPEasySerialPort>(_console_serial_port) == ESPEasySerialPort::serial0
# ifdef ESP8266
       || static_cast<ESPEasySerialPort>(_console_serial_port) == ESPEasySerialPort::serial0_swap
# endif // ifdef ESP8266
      ))
#endif  // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  {
    String dummy;

    return PluginCall(PLUGIN_SERIAL_IN, 0, dummy);
  }
  return false;
}

void EspEasy_Console_t::readInput(EspEasy_Console_Port& port)
{
  size_t bytesToRead = port.available();

  while (bytesToRead > 0)
  {
    --bytesToRead;
    delay(0);
    const int SerialInByte = port.read();

    if (SerialInByte >= 0) {
      if (port.process_consoleInput(SerialInByte)) {
        // Processed a full line
        return;
      }
    }
  }
}

#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
ESPeasySerial * EspEasy_Console_t::getPort()
{
  if (_mainSerial._serial != nullptr) {
    return _mainSerial._serial;
  }
# if USES_ESPEASY_CONSOLE_FALLBACK_PORT

  if (_fallbackSerial._serial != nullptr) {
    return _fallbackSerial._serial;
  }
# endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT
  return nullptr;
}

#else // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
HardwareSerial * EspEasy_Console_t::getPort()
{
  if (_mainSerial._serial != nullptr) {
    return _mainSerial._serial;
  }
  return nullptr;
}

#endif // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT


void EspEasy_Console_t::endPort()
{
  _mainSerial.endPort();
#if USES_ESPEASY_CONSOLE_FALLBACK_PORT
  _fallbackSerial.endPort();
#endif // if USES_ESPEASY_CONSOLE_FALLBACK_PORT
  delay(10);
}

int EspEasy_Console_t::availableForWrite()
{
  auto serial = getPort();

  if (serial != nullptr) {
    return serial->availableForWrite();
  }
  return 0;
}

#include "../ESPEasyCore/ESPEasy_Console_Port.h"

#include "../Commands/ExecuteCommand.h"

#include "../DataStructs/TimingStats.h"

#include "../DataTypes/ESPEasy_plugin_functions.h"

#include "../Globals/Cache.h"
#include "../Globals/Logging.h"
#include "../Globals/Plugins.h"
#include "../Globals/Settings.h"

#include "../Helpers/Memory.h"

#include <ESPEasySerialPort.h>


/*
 #if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
 # include "../Helpers/_Plugin_Helper_serial.h"
 #endif // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
 */

EspEasy_Console_Port::~EspEasy_Console_Port()
{
#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  if (_serial != nullptr) {
    delete _serial;
    _serial = nullptr;
  }
#endif
}

EspEasy_Console_Port::operator bool() const
{
  if (_serial != nullptr) {
    return true;

    //    return _serial->operator bool();
  }
  return false;
}

int EspEasy_Console_Port::read()
{
#if FEATURE_IMPROV
  uint8_t b;

  if (_improv.getFromBuffer(b)) {
    return b;
  }

#endif // if FEATURE_IMPROV
  int res = -1;

  if (_serial != nullptr)
  {
    res = _serial->read();
#if FEATURE_IMPROV

    if (res >= 0) {
      if (_improv.handle(res, _serial)) {
        // Looks like it might be an IMPROV command, so like we didn't have any data
        return -1;
      }

      if (_improv.getFromBuffer(b)) {
        return b;
      }
    }

#endif // if FEATURE_IMPROV
  }
  return res;
}

size_t EspEasy_Console_Port::available() const
{
  size_t res = 0u;

#if FEATURE_IMPROV
  res += _improv.available();
#endif // if FEATURE_IMPROV

  if (_serial != nullptr) {
    res += _serial->available();
  }
  return res;
}

void EspEasy_Console_Port::endPort()
{
  if (_serial != nullptr) {
    _serial->end();
  }
}

void EspEasy_Console_Port::addToSerialBuffer(char c)
{
  if (_serial != nullptr) {
    _serialWriteBuffer.add(c);
  }
}

void EspEasy_Console_Port::addToSerialBuffer(const String& line)
{
  if (_serial != nullptr) {
    _serialWriteBuffer.add(line);
  }
}

void EspEasy_Console_Port::addNewlineToSerialBuffer()
{
  if (_serial != nullptr) {
    _serialWriteBuffer.addNewline();
  }
}

bool EspEasy_Console_Port::process_serialWriteBuffer()
{
  if (_serial != nullptr) {
    const int snip = _serial->availableForWrite();
    
    if (snip > 0) {
      return _serialWriteBuffer.write(*_serial, snip) != 0;
    }
  }
  return false;
}

bool EspEasy_Console_Port::process_consoleInput(uint8_t SerialInByte)
{
  if (isprint(SerialInByte))
  {
    if (SerialInByteCounter < CONSOLE_INPUT_BUFFER_SIZE) { // add char to string if it still fits
      InputBuffer_Serial[SerialInByteCounter++] = SerialInByte;
    }
  }

  if ((SerialInByte == '\b') && (SerialInByteCounter > 0)) // Correct a typo using BackSpace
  {
    --SerialInByteCounter;
  } else
  if ((SerialInByte == '\r') || (SerialInByte == '\n'))
  {
    // Ignore empty command
    if (SerialInByteCounter != 0) {
      InputBuffer_Serial[SerialInByteCounter] = 0; // serial data completed
      addToSerialBuffer('>');
      String cmd(InputBuffer_Serial);
      addToSerialBuffer(cmd);
      addToSerialBuffer('\n');
      ExecuteCommand_all({EventValueSource::Enum::VALUE_SOURCE_SERIAL, std::move(cmd)}, true);
      SerialInByteCounter   = 0;
      InputBuffer_Serial[0] = 0; // serial data processed, clear buffer
      return true;
    }
  }
  return false;
}

String EspEasy_Console_Port::getPortDescription() const
{
  if (_serial != nullptr) {
  #if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
    return _serial->getPortDescription();
  #else // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
    String res = F("HW Serial0 @ ");
    res += _serial->baudRate();
    return res;
  #endif // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  }

  return F("-");
}

#include "../ESPEasyCore/ESPEasyNetwork.h"

#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/ESPEasyEth.h"
#include "../ESPEasyCore/ESPEasyWifi.h"
#include "../Globals/ESPEasy_time.h"
#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/NetworkState.h"
#include "../Globals/Settings.h"

#include "../Helpers/Network.h"
#include "../Helpers/Networking.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/MDNS_Helper.h"

#if FEATURE_ETHERNET
#include "../Globals/ESPEasyEthEvent.h"
#include <ETH.h>
#endif


#if FEATURE_USE_IPV6
#include <esp_netif.h>

// -----------------------------------------------------------------------------------------------------------------------
// ---------------------------------------------------- Private functions ------------------------------------------------
// -----------------------------------------------------------------------------------------------------------------------

esp_netif_t* get_esp_interface_netif(esp_interface_t interface);
#endif


void setNetworkMedium(NetworkMedium_t new_medium) {
#if !(FEATURE_ETHERNET)
  if (new_medium == NetworkMedium_t::Ethernet) {
    new_medium = NetworkMedium_t::WIFI;
  }
#endif
  if (active_network_medium == new_medium) {
    return;
  }
  switch (active_network_medium) {
    case NetworkMedium_t::Ethernet:
      #if FEATURE_ETHERNET
      // FIXME TD-er: How to 'end' ETH?
//      ETH.end();
      if (new_medium == NetworkMedium_t::WIFI) {
        WiFiEventData.clearAll();
#if ESP_IDF_VERSION_MAJOR >= 5
        WiFi.STA.setDefault();
#endif
      }
      #endif
      break;
    case NetworkMedium_t::WIFI:
      WiFiEventData.timerAPoff.setMillisFromNow(WIFI_AP_OFF_TIMER_DURATION);
      WiFiEventData.timerAPstart.clear();
      if (new_medium == NetworkMedium_t::Ethernet) {
#if ESP_IDF_VERSION_MAJOR >= 5
#if FEATURE_ETHERNET
        ETH.setDefault();
#endif
#endif
        WifiDisconnect();
      }
      break;
    case NetworkMedium_t::NotSet:
      break;
  }
  statusLED(true);
  active_network_medium = new_medium;
  #ifndef BUILD_MINIMAL_OTA
  addLog(LOG_LEVEL_INFO, concat(F("Set Network mode: "), toString(active_network_medium)));
  #endif
}


/*********************************************************************************************\
   Ethernet or Wifi Support for ESP32 Build flag FEATURE_ETHERNET
\*********************************************************************************************/
void NetworkConnectRelaxed() {
  if (NetworkConnected()) return;
#if FEATURE_ETHERNET
  if(active_network_medium == NetworkMedium_t::Ethernet) {
    if (ETHConnectRelaxed()) {
      return;
    }
    // Failed to start the Ethernet network, probably not present of wrong parameters.
    // So set the runtime active medium to WiFi to try connecting to WiFi or at least start the AP.
    setNetworkMedium(NetworkMedium_t::WIFI);
  }
#endif
  // Failed to start the Ethernet network, probably not present of wrong parameters.
  // So set the runtime active medium to WiFi to try connecting to WiFi or at least start the AP.
  WiFiConnectRelaxed();
}

bool NetworkConnected() {
  #if FEATURE_ETHERNET
  if(active_network_medium == NetworkMedium_t::Ethernet) {
    return ETHConnected();
  }
  #endif
  return WiFiConnected();
}

IPAddress NetworkLocalIP() {
  #if FEATURE_ETHERNET
  if(active_network_medium == NetworkMedium_t::Ethernet) {
    if(EthEventData.ethInitSuccess) {
      return ETH.localIP();
    } else {
      addLog(LOG_LEVEL_ERROR, F("Call NetworkLocalIP() only on connected Ethernet!"));
      return IPAddress();
    }
  }
  #endif
  return WiFi.localIP();
}

IPAddress NetworkSubnetMask() {
  #if FEATURE_ETHERNET
  if(active_network_medium == NetworkMedium_t::Ethernet) {
    if(EthEventData.ethInitSuccess) {
      return ETH.subnetMask();
    } else {
      addLog(LOG_LEVEL_ERROR, F("Call NetworkSubnetMask() only on connected Ethernet!"));
      return IPAddress();
    }
  }
  #endif
  return WiFi.subnetMask();
}

IPAddress NetworkGatewayIP() {
  #if FEATURE_ETHERNET
  if(active_network_medium == NetworkMedium_t::Ethernet) {
    if(EthEventData.ethInitSuccess) {
      return ETH.gatewayIP();
    } else {
      addLog(LOG_LEVEL_ERROR, F("Call NetworkGatewayIP() only on connected Ethernet!"));
      return IPAddress();
    }
  }
  #endif
  return WiFi.gatewayIP();
}

IPAddress NetworkDnsIP(uint8_t dns_no) {
  scrubDNS();
  #if FEATURE_ETHERNET
  if(active_network_medium == NetworkMedium_t::Ethernet) {
    if(EthEventData.ethInitSuccess) {
      return ETH.dnsIP(dns_no);
    } else {
      addLog(LOG_LEVEL_ERROR, F("Call NetworkDnsIP(uint8_t dns_no) only on connected Ethernet!"));
      return IPAddress();
    }
  }
  #endif
  return WiFi.dnsIP(dns_no);
}

#if FEATURE_USE_IPV6
esp_netif_t * getActiveNetworkMediumInterface() {
  esp_interface_t iface = ESP_IF_MAX;
  #if FEATURE_ETHERNET
  if(active_network_medium == NetworkMedium_t::Ethernet) {
    if(EthEventData.ethInitSuccess) {
      esp_netif_t *res = ETH.netif();
      if (res == nullptr) {
        res = get_esp_interface_netif(ESP_IF_ETH);
      }
      if (res != nullptr)
        return res;
    }
  } else
  #endif
  {
    if (WifiIsSTA(WiFi.getMode())) {
      iface = ESP_IF_WIFI_STA;
    }
  }
  if (ESP_IF_MAX == iface) 
    return nullptr;
  return get_esp_interface_netif(iface);
}

IPAddress NetworkLocalIP6() {
  esp_netif_t * iface = getActiveNetworkMediumInterface();
  esp_ip6_addr_t addr;
  if (nullptr == iface ||
     esp_netif_get_ip6_linklocal(iface, &addr)) 
  {
    return IN6ADDR_ANY;
  }

  IPAddress res(IPv6, (const uint8_t*)addr.addr, addr.zone);
  return res;
}

IPAddress NetworkGlobalIP6() {
  esp_netif_t * iface = getActiveNetworkMediumInterface();
  esp_ip6_addr_t addr;
  if (nullptr == iface ||
     esp_netif_get_ip6_global(iface, &addr)) 
  {
    return IN6ADDR_ANY;
  }

  IPAddress res(IPv6, (const uint8_t*)addr.addr, addr.zone);
  return res;
}

IP6Addresses_t NetworkAllIPv6() {
  IP6Addresses_t addresses;
  esp_netif_t * iface = getActiveNetworkMediumInterface();
  if (nullptr != iface) {
    esp_ip6_addr_t esp_ip6_addr[LWIP_IPV6_NUM_ADDRESSES]{};

    int count = esp_netif_get_all_ip6(iface, esp_ip6_addr);
    for (int i = 0; i < count; ++i) {
      addresses.emplace_back(IPv6, (const uint8_t*)esp_ip6_addr[i].addr, esp_ip6_addr[i].zone);
    }
  }

  return addresses;
}

bool IPv6_from_MAC(const MAC_address& mac, IPAddress& ipv6)
{
  if (ipv6 == IN6ADDR_ANY) { return false; }
  int index_offset = 8;

  for (int i = 0; i < 6; ++i, ++index_offset) {
    ipv6[index_offset] = mac.mac[i];

    if (i == 0) {
      // invert bit 2
      bitToggle(ipv6[index_offset], 1);
    }

    if (i == 2) {
      ipv6[++index_offset] = 0xFF;
      ipv6[++index_offset] = 0xFE;
    }
  }
/*
  addLog(LOG_LEVEL_INFO, strformat(
     F("IPv6_from_MAC: Mac %s IP %s"),
     mac.toString().c_str(),
     ipv6.toString(true).c_str()
     ));
*/
  return true;
}

bool is_IPv6_based_on_MAC(const MAC_address& mac, const IPAddress& ipv6)
{
  IPAddress tmp = ipv6;

  if (IPv6_from_MAC(mac, tmp)) {
    return ipv6 == tmp;
  }
  return false;
}

bool IPv6_link_local_from_MAC(const MAC_address& mac, IPAddress& ipv6)
{
  ipv6 = NetworkLocalIP6();
  return IPv6_from_MAC(mac, ipv6);
}

bool is_IPv6_link_local_from_MAC(const MAC_address& mac)
{
  return is_IPv6_based_on_MAC(mac, NetworkLocalIP6());
}

// Assume we're in the same subnet, thus use our own IPv6 global address
bool IPv6_global_from_MAC(const MAC_address& mac, IPAddress& ipv6)
{
  ipv6 = NetworkGlobalIP6();
  return IPv6_from_MAC(mac, ipv6);
}

bool is_IPv6_global_from_MAC(const MAC_address& mac)
{
  return is_IPv6_based_on_MAC(mac, NetworkGlobalIP6());
}

#endif // if FEATURE_USE_IPV6



MAC_address NetworkMacAddress() {
  #if FEATURE_ETHERNET
  if(active_network_medium == NetworkMedium_t::Ethernet) {
    return ETHMacAddress();
  }
  #endif
  MAC_address mac;
  WiFi.macAddress(mac.mac);
  return mac;
}

String NetworkGetHostname() {
    #ifdef ESP32
      #if FEATURE_ETHERNET 
      if(Settings.NetworkMedium == NetworkMedium_t::Ethernet && EthEventData.ethInitSuccess) {
        return String(ETH.getHostname());
      }
      #endif
      return String(WiFi.getHostname());
    #else
      return String(WiFi.hostname());
    #endif
}

// ********************************************************************************
// Determine Wifi AP name to set. (also used for mDNS)
// ********************************************************************************
String NetworkGetHostNameFromSettings(bool force_add_unitnr)
{
  if (force_add_unitnr) return Settings.getHostname(true);
  return Settings.getHostname();
}

String NetworkCreateRFCCompliantHostname(bool force_add_unitnr) {
  // Create hostname with - instead of spaces
  return makeRFCCompliantName(NetworkGetHostNameFromSettings(force_add_unitnr));
}

String makeRFCCompliantName(const String& name, const char replaceChar, const char allowedChar, const size_t maxlength) {
  String hostname(name);
  // See RFC952. (when using the default arguments: '-', '-', 24)
  // Allowed chars:
  // * letters (a-z, A-Z)
  // * numerals (0-9)
  // * Hyphen (-)
  // * Max length 24
  replaceUnicodeByChar(hostname, replaceChar);
  for (size_t i = 0; i < hostname.length(); ++i) {
    const char c = hostname[i];
    if (!isAlphaNumeric(c) && (c != allowedChar)) {
      hostname[i] = replaceChar;
    }
  }

  // May not start or end with a hyphen
  const String dash(replaceChar);
  while (hostname.startsWith(dash)) {
    hostname = hostname.substring(1);
  }
  while (hostname.endsWith(dash)) {
    hostname = hostname.substring(0, hostname.length() - 1);
  }

  // May not contain only numerals
  bool onlyNumerals = true;
  for (size_t i = 0; onlyNumerals && i < hostname.length(); ++i) {
    const char c = hostname[i];
    if (!isdigit(c)) {
      onlyNumerals = false;
    }
  }
  if (onlyNumerals) {
    hostname = strformat(F("ESPEasy%c%s"), replaceChar, hostname.c_str());
  }

  if ((maxlength > 0) && (hostname.length() > maxlength)) {
    hostname = hostname.substring(0, maxlength);
  }

  return hostname;
}

MAC_address WifiSoftAPmacAddress() {
  MAC_address mac;
  WiFi.softAPmacAddress(mac.mac);
  return mac;
}

MAC_address WifiSTAmacAddress() {
  MAC_address mac;
  WiFi.macAddress(mac.mac);
  return mac;
}

void CheckRunningServices() {
  // First try to get the time, since that may be used in logs
  if (Settings.UseNTP() && node_time.getTimeSource() > timeSource_t::NTP_time_source) {
    node_time.lastNTPSyncTime_ms = 0;
    node_time.initTime();
  }
#if FEATURE_SET_WIFI_TX_PWR
  if (active_network_medium == NetworkMedium_t::WIFI) 
  {
    SetWiFiTXpower();
  }
#endif
  set_mDNS();
}

#if FEATURE_ETHERNET
bool EthFullDuplex()
{
  if (EthEventData.ethInitSuccess)
    return ETH.fullDuplex();
  return false;
}

bool EthLinkUp()
{
  if (EthEventData.ethInitSuccess) {
    #if ESP_IDF_VERSION_MAJOR < 5
    // FIXME TD-er: See: https://github.com/espressif/arduino-esp32/issues/6105
    return EthEventData.EthConnected();
    #else
    return ETH.linkUp();
    #endif
  }
  return false;
}

uint8_t EthLinkSpeed()
{
  if (EthEventData.ethInitSuccess) {
    return ETH.linkSpeed();
  }
  return 0;
}
#endif

#include "../ESPEasyCore/ESPEasyEth_ProcessEvent.h"

#if FEATURE_ETHERNET

# include "../../ESPEasy-Globals.h"

# include "../ESPEasyCore/ESPEasyEth.h"
# include "../ESPEasyCore/ESPEasyNetwork.h"
# include "../ESPEasyCore/ESPEasyWifi.h" // LogConnectionStatus

# include "../Globals/ESPEasyEthEvent.h"
# include "../Globals/ESPEasyWiFiEvent.h"
# include "../Globals/ESPEasy_Scheduler.h"
# include "../Globals/ESPEasy_time.h"
# include "../Globals/EventQueue.h"
# include "../Globals/MQTT.h"
# include "../Globals/NetworkState.h"
# include "../Globals/Settings.h"

# include "../Helpers/LongTermTimer.h"
# include "../Helpers/Network.h"
# include "../Helpers/Networking.h"
# include "../Helpers/PeriodicalActions.h"
# include "../Helpers/StringConverter.h"

# include <ETH.h>

void handle_unprocessedEthEvents() {
  if (EthEventData.unprocessedEthEvents()) {
    // Process disconnect events before connect events.
#if FEATURE_USE_IPV6
    if (!EthEventData.processedGotIP6) {
      processEthernetGotIPv6();
    }
#endif

    if (!EthEventData.processedDisconnect) {
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_DEBUG, F("Eth  : Entering processDisconnect()"));
      # endif // ifndef BUILD_NO_DEBUG
      processEthernetDisconnected();
    }

    // Must process the Ethernet Connected event regardless the active network medium.
    // It may happen by plugging in the cable while WiFi was active.
    if (!EthEventData.processedConnect) {
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_DEBUG, F("Eth  : Entering processConnect()"));
      # endif // ifndef BUILD_NO_DEBUG
      processEthernetConnected();
    }
  }

  if (active_network_medium == NetworkMedium_t::Ethernet) {
    const bool EthServices_was_initialized = EthEventData.EthServicesInitialized();

    if (!EthEventData.EthServicesInitialized() || EthEventData.unprocessedEthEvents())
    {
      if (!EthEventData.unprocessedEthEvents() && EthEventData.EthConnectAllowed()) {
        NetworkConnectRelaxed();
      }

      if (!EthEventData.processedGotIP) {
        # ifndef BUILD_NO_DEBUG
        addLog(LOG_LEVEL_DEBUG, F("Eth  : Entering processGotIP()"));
        # endif // ifndef BUILD_NO_DEBUG
        processEthernetGotIP();
      }

      if (!EthEventData.processedDHCPTimeout) {
        # ifndef BUILD_NO_DEBUG
        addLog(LOG_LEVEL_DEBUG, F("Eth  : DHCP timeout, Calling disconnect()"));
        # endif // ifndef BUILD_NO_DEBUG
        EthEventData.processedDHCPTimeout = true;

        // WifiDisconnect();
      }
    }
    EthEventData.setEthServicesInitialized();

    if (!EthServices_was_initialized && EthEventData.setEthServicesInitialized()) {
      registerEthEventHandler();
    }
  }
}

void check_Eth_DNS_valid() {
  // Check if DNS is still valid, as this may have been reset by the WiFi module turned off.
  // DNS records are shared between WiFi and Ethernet IP stack.
  // Turning off WiFi will clead DNS records for Ethernet
  if (EthEventData.EthServicesInitialized() &&
      (active_network_medium == NetworkMedium_t::Ethernet) &&
      EthEventData.ethInitSuccess &&
      !ethUseStaticIP()) {
    const bool has_cache = 
      valid_DNS_address(EthEventData.dns0_cache) || 
      valid_DNS_address(EthEventData.dns1_cache);

    if (has_cache) {
      const IPAddress dns0 = ETH.dnsIP(0);
      const IPAddress dns1 = ETH.dnsIP(1);

      if (!valid_DNS_address(dns0) && !valid_DNS_address(dns1)) {
        static uint32_t lastLog = 0;
        if (timePassedSince(lastLog) > 1000) {
          addLogMove(LOG_LEVEL_ERROR, concat(
            F("ETH  : DNS server was cleared, use cached DNS IP: "), 
            formatIP(EthEventData.dns0_cache)));
          lastLog = millis();
        }
        setDNS(0, EthEventData.dns0_cache);
        setDNS(1, EthEventData.dns1_cache);
      }
    }
  }
}

void processEthernetConnected() {
  if (EthEventData.processedConnect) { return; }

  // FIXME TD-er: Must differentiate among reconnects for WiFi and Ethernet.
  ++WiFiEventData.wifi_reconnects;
  addLog(LOG_LEVEL_INFO, F("processEthernetConnected()"));
  EthEventData.setEthConnected();
  EthEventData.processedConnect = true;

  if (Settings.UseRules)
  {
    eventQueue.add(F("Ethernet#LinkUp"));
  }
  setNetworkMedium(Settings.NetworkMedium);

  if (ethUseStaticIP()) { EthEventData.processedGotIP = false; }
}

void processEthernetDisconnected() {
  if (EthEventData.processedDisconnect) { return; }
  EthEventData.setEthDisconnected();
  EthEventData.processedDisconnect     = true;
  EthEventData.ethConnectAttemptNeeded = true;

  if (Settings.UseRules)
  {
    eventQueue.add(F("Ethernet#Disconnected"));
  }
}

void processEthernetGotIP() {
  if (EthEventData.processedGotIP || !EthEventData.ethInitSuccess) {
    return;
  }

  if (ethUseStaticIP()) {
    ethSetupStaticIPconfig();
  }
  const IPAddress ip = NetworkLocalIP();

  if (!ip) {
    return;
  }

  const IPAddress gw     = NetworkGatewayIP();
  const IPAddress subnet = NetworkSubnetMask();

  IPAddress dns0                              = ETH.dnsIP(0);
  IPAddress dns1                              = ETH.dnsIP(1);
  const LongTermTimer::Duration dhcp_duration = EthEventData.lastConnectMoment.timeDiff(EthEventData.lastGetIPmoment);
  #if ESP_IDF_VERSION_MAJOR >= 5
  const bool mustRequestDHCP = (!dns0 && !dns1) && !ethUseStaticIP();
  #endif

  if (!ethUseStaticIP()) {
    if (!dns0 && !dns1) {
      addLog(LOG_LEVEL_ERROR, F("ETH  : No DNS server received via DHCP, use cached DNS IP"));
      if (EthEventData.dns0_cache) setDNS(0, EthEventData.dns0_cache);
      if (EthEventData.dns1_cache) setDNS(1, EthEventData.dns1_cache);
    } else {
      EthEventData.dns0_cache = dns0;
      EthEventData.dns1_cache = dns1;
    }
  }

  if (loglevelActiveFor(LOG_LEVEL_INFO))
  {
    String log;

    if (log.reserve(160)) {
      log  = F("ETH MAC: ");
      log += NetworkMacAddress().toString();
      log += ' ';

      if (ethUseStaticIP()) {
        log += F("Static");
      } else {
        log += F("DHCP");
      }
      log += F(" IP: ");
      log += formatIP(ip);
      log += ' ';
      log += wrap_braces(NetworkGetHostname());
      log += F(" GW: ");
      log += formatIP(gw);
      log += F(" SN: ");
      log += formatIP(subnet);
      log += F(" DNS: ");
      log += formatIP(dns0);
      log += '/';
      log += formatIP(dns1);

      if (EthLinkUp()) {
        if (EthFullDuplex()) {
          log += F(" FULL_DUPLEX");
        }
        log += ' ';
        log += EthLinkSpeed();
        log += F("Mbps");
      } else {
        log += F(" Link Down");
      }

      if ((dhcp_duration > 0ll) && (dhcp_duration < 30000000ll)) {
        // Just log times when they make sense.
        log += F("   duration: ");
        log += static_cast<int32_t>(dhcp_duration / 1000);
        log += F(" ms");
      }

      addLogMove(LOG_LEVEL_INFO, log);
    }
  }

  // First try to get the time, since that may be used in logs
  if (node_time.systemTimePresent()) {
    node_time.initTime();
  }
# if FEATURE_MQTT
  mqtt_reconnect_count        = 0;
  MQTTclient_should_reconnect = true;
  timermqtt_interval          = 100;
  Scheduler.setIntervalTimer(SchedulerIntervalTimer_e::TIMER_MQTT);
  scheduleNextMQTTdelayQueue();
# endif // if FEATURE_MQTT
  Scheduler.sendGratuitousARP_now();

  if (Settings.UseRules)
  {
    eventQueue.add(F("Ethernet#Connected"));
  }
  statusLED(true);
  logConnectionStatus();

  EthEventData.processedGotIP = true;
#if ESP_IDF_VERSION_MAJOR >= 5
  if (mustRequestDHCP /*&& EthEventData.lastConnectMoment.millisPassedSince() < 10000*/) {
    // FIXME TD-er: Must add some check other than fixed timeout here to not constantly make DHCP requests.
    // Force new DHCP request.
    ETH.config();
  }
#endif

  EthEventData.setEthGotIP();
  CheckRunningServices();
}

#if FEATURE_USE_IPV6
void processEthernetGotIPv6() {
  if (!EthEventData.processedGotIP6) {
    if (loglevelActiveFor(LOG_LEVEL_INFO))
      addLog(LOG_LEVEL_INFO, String(F("ETH event: Got IP6 ")) + EthEventData.unprocessed_IP6.toString(true));
    EthEventData.processedGotIP6 = true;
#if FEATURE_ESPEASY_P2P
//    updateUDPport(true);
#endif

  }
}
#endif

#endif // if FEATURE_ETHERNET

#include "../ESPEasyCore/ESPEasyEthEvent.h"

#ifdef ESP32
# if FEATURE_ETHERNET
#  include <ETH.h>

#include "../ESPEasyCore/ESPEasyEth.h"
#include "../Globals/ESPEasyEthEvent.h"


// ********************************************************************************
// Functions called on events.  ** WARNING **
// Make sure not to call anything in these functions that result in delay() or yield()
// ********************************************************************************

#  include <WiFi.h>


#  if ESP_IDF_VERSION_MAJOR > 3
void EthEvent(WiFiEvent_t event, arduino_event_info_t info) {
  switch (event) {
    case ARDUINO_EVENT_ETH_START:

      if (ethPrepare()) {
//      addLog(LOG_LEVEL_INFO, F("ETH event: Started"));
//    } else {
//      addLog(LOG_LEVEL_ERROR, F("ETH event: Could not prepare ETH!"));
      }
      break;
    case ARDUINO_EVENT_ETH_CONNECTED:
//      addLog(LOG_LEVEL_INFO, F("ETH event: Connected"));
      EthEventData.markConnected();
      break;
    case ARDUINO_EVENT_ETH_GOT_IP:
      EthEventData.markGotIP();
//      addLog(LOG_LEVEL_INFO,  F("ETH event: Got IP"));
      break;
    case ARDUINO_EVENT_ETH_DISCONNECTED:
//      addLog(LOG_LEVEL_ERROR, F("ETH event: Disconnected"));
      EthEventData.markDisconnect();
      break;
    case ARDUINO_EVENT_ETH_STOP:
//      addLog(LOG_LEVEL_INFO, F("ETH event: Stopped"));
      break;
    #   if ESP_IDF_VERSION_MAJOR > 3
    case ARDUINO_EVENT_ETH_GOT_IP6:
    #   else // if ESP_IDF_VERSION_MAJOR > 3
    case ARDUINO_EVENT_GOT_IP6:
    #   endif // if ESP_IDF_VERSION_MAJOR > 3
    #if FEATURE_USE_IPV6
    {
      ip_event_got_ip6_t * event = static_cast<ip_event_got_ip6_t*>(&info.got_ip6);
      IPAddress ip(IPv6, (const uint8_t*)event->ip6_info.ip.addr, event->ip6_info.ip.zone);
      EthEventData.markGotIPv6(ip);
//      addLog(LOG_LEVEL_INFO, String(F("ETH event: Got IP6 ")) + ip.toString(true));
    }
    #else
//    addLog(LOG_LEVEL_INFO, F("ETH event: Got IP6"));
    #endif
      break;
    default:
    {
      break;
    }
  }
}

#  endif // if ESP_IDF_VERSION_MAJOR > 3

# endif // if FEATURE_ETHERNET

#endif // ifdef ESP32

#include "../ESPEasyCore/Serial.h"


#include "../Commands/ExecuteCommand.h"

#include "../Globals/Cache.h"
#include "../Globals/ESPEasy_Console.h"
//#include "../Globals/Logging.h" //  For serialWriteBuffer
#include "../Globals/Settings.h"

#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/Memory.h"



void initSerial()
{
  ESPEasy_Console.init();
}

void serial()
{
  ESPEasy_Console.loop();
}

bool process_serialWriteBuffer()
{
  return ESPEasy_Console.process_serialWriteBuffer();
}

// For now, only send it to the serial buffer and try to process it.
// Later we may want to wrap it into a log.
void serialPrint(const __FlashStringHelper *text) {
  ESPEasy_Console.addToSerialBuffer(text);
}

void serialPrint(const String& text) {
  ESPEasy_Console.addToSerialBuffer(text);
}

void serialPrintln(const __FlashStringHelper *text) {
  ESPEasy_Console.addToSerialBuffer(text);
  serialPrintln();
}

void serialPrintln(const String& text) {
  ESPEasy_Console.addToSerialBuffer(text);
  serialPrintln();
}

void serialPrintln() {
  ESPEasy_Console.addNewlineToSerialBuffer();
  ESPEasy_Console.process_serialWriteBuffer();
}

// Do not add helper functions for other types, since those types can only be
// explicit matched at a constructor, not a function declaration.

/*
   void serialPrint(char c) {
   serialPrint(String(c));
   }


   void serialPrint(unsigned long value) {
   serialPrint(String(value));
   }

   void serialPrint(long value) {
   serialPrint(String(value));
   }

   void serialPrintln(unsigned long value) {
   serialPrintln(String(value));
   }
 */

#include "../ESPEasyCore/ESPEasy_backgroundtasks.h"

#include "../../ESPEasy_common.h"

#include "../../ESPEasy-Globals.h"
#include "../DataStructs/TimingStats.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/Serial.h"
#include "../Globals/NetworkState.h"
#include "../Globals/Services.h"
#include "../Globals/Settings.h"
#if FEATURE_RTTTL && FEATURE_ANYRTTTL_LIB && FEATURE_ANYRTTTL_ASYNC
#include "../Helpers/Audio.h"
#endif // if FEATURE_RTTTL && FEATURE_ANYRTTTL_LIB && FEATURE_ANYRTTTL_ASYNC
#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/Network.h"
#include "../Helpers/Networking.h"


#if FEATURE_ARDUINO_OTA
#include "../Helpers/OTA.h"
#endif



/*********************************************************************************************\
* run background tasks
\*********************************************************************************************/
bool runningBackgroundTasks = false;
void backgroundtasks()
{
  // checkRAM(F("backgroundtasks"));
  // always start with a yield
  delay(0);

  /*
     // Remove this watchdog feed for now.
     // See https://github.com/letscontrolit/ESPEasy/issues/1722#issuecomment-419659193

   #ifdef ESP32
     // Have to find a similar function to call ESP32's esp_task_wdt_feed();
   #else
     ESP.wdtFeed();
   #endif
   */

  // prevent recursion!
  if (runningBackgroundTasks)
  {
    return;
  }

  // Rate limit calls to run backgroundtasks
  static uint32_t lastRunBackgroundTasks = 0;
  if (timePassedSince(lastRunBackgroundTasks) < 10) return;
  lastRunBackgroundTasks = millis();

  START_TIMER
  #if FEATURE_MDNS || FEATURE_ESPEASY_P2P
  const bool networkConnected = NetworkConnected();
  #else
  NetworkConnected();
  #endif

  runningBackgroundTasks = true;

  /*
     // Not needed anymore, see: https://arduino-esp8266.readthedocs.io/en/latest/faq/readme.html#how-to-clear-tcp-pcbs-in-time-wait-state
     if (networkConnected) {
   #if defined(ESP8266)
      tcpCleanup();
   #endif
     }
   */

  process_serialWriteBuffer();

  if (!UseRTOSMultitasking) {
    serial();

//    if (webserverRunning) {
/*
    {
      START_TIMER
      web_server.handleClient();
      STOP_TIMER(WEBSERVER_HANDLE_CLIENT);
    }
*/
    #if FEATURE_ESPEASY_P2P
    if (networkConnected) {
      checkUDP();
    }
    #endif
  }

  #if FEATURE_DNS_SERVER

  // process DNS, only used if the ESP has no valid WiFi config
  if (dnsServerActive) {
    dnsServer.processNextRequest();
  }
  #endif // if FEATURE_DNS_SERVER

  #if FEATURE_ARDUINO_OTA

  if (Settings.ArduinoOTAEnable) {
    ArduinoOTA_handle();
  }

  // once OTA is triggered, only handle that and dont do other stuff. (otherwise it fails)
  while (ArduinoOTAtriggered)
  {
    delay(0);

    ArduinoOTA_handle();
  }

  #endif // if FEATURE_ARDUINO_OTA

  #if FEATURE_MDNS

  // Allow MDNS processing
  if (networkConnected) {
    # ifdef ESP8266

    // ESP32 does not have an update() function
    MDNS.update();
    # endif // ifdef ESP8266
  }
  #endif // if FEATURE_MDNS

  delay(0);

  #if FEATURE_RTTTL && FEATURE_ANYRTTTL_LIB && FEATURE_ANYRTTTL_ASYNC
  update_rtttl();
  #endif // if FEATURE_RTTTL && FEATURE_ANYRTTTL_LIB && FEATURE_ANYRTTTL_ASYNC

  statusLED(false);

  runningBackgroundTasks = false;
  STOP_TIMER(BACKGROUND_TASKS);
}

#include "../ESPEasyCore/ESPEasyWifi_ProcessEvent.h"

#include "../../ESPEasy-Globals.h"

#if FEATURE_ETHERNET
#include "../ESPEasyCore/ESPEasyEth_ProcessEvent.h"
#endif
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/ESPEasyWifi.h"

#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/ESPEasy_Scheduler.h"
#include "../Globals/ESPEasy_time.h"
#include "../Globals/EventQueue.h"
#include "../Globals/MQTT.h"
#include "../Globals/NetworkState.h"
#include "../Globals/RTC.h"
#include "../Globals/SecuritySettings.h"
#include "../Globals/Services.h"
#include "../Globals/Settings.h"
#include "../Globals/WiFi_AP_Candidates.h"

#include "../Helpers/Convert.h"
#include "../Helpers/ESPEasyRTC.h"
#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/Network.h"
#include "../Helpers/Networking.h"
#include "../Helpers/PeriodicalActions.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringGenerator_WiFi.h"
#include "../Helpers/StringProvider.h"

// #include "../ESPEasyCore/ESPEasyEth.h"
// #include "../ESPEasyCore/ESPEasyWiFiEvent.h"
// #include "../ESPEasyCore/ESPEasy_Log.h"
// #include "../Helpers/ESPEasy_time_calc.h"
// #include "../Helpers/Misc.h"
// #include "../Helpers/Scheduler.h"

#include "../WebServer/ESPEasy_WebServer.h"


// ********************************************************************************
// Called from the loop() to make sure events are processed as soon as possible.
// These functions are called from Setup() or Loop() and thus may call delay() or yield()
// ********************************************************************************
void handle_unprocessedNetworkEvents()
{
#if FEATURE_ETHERNET
  handle_unprocessedEthEvents();
#endif

  if (active_network_medium == NetworkMedium_t::WIFI) {
    const bool should_be_initialized = (WiFiEventData.WiFiGotIP() && WiFiEventData.WiFiConnected()) || NetworkConnected();
    if (WiFiEventData.WiFiServicesInitialized() != should_be_initialized)
    {
      if (!WiFiEventData.WiFiServicesInitialized()) {
        WiFiEventData.processedDHCPTimeout  = true;  // FIXME TD-er:  Find out when this happens  (happens on ESP32 sometimes)
        if (WiFiConnected()) {
          if (!WiFiEventData.WiFiGotIP()) {
            # ifndef BUILD_NO_DEBUG
            addLog(LOG_LEVEL_DEBUG, F("WiFi : Missed gotIP event"));
            #endif
            WiFiEventData.processedGotIP = false;
            processGotIP();
          }
          if (!WiFiEventData.WiFiConnected()) {
            # ifndef BUILD_NO_DEBUG
            addLog(LOG_LEVEL_DEBUG, F("WiFi : Missed connected event"));
            #endif
            WiFiEventData.processedConnect = false;
            processConnect();
          }
          // Apparently we are connected, so no need to process any late disconnect event
          WiFiEventData.processedDisconnect = true;
        }        
        WiFiEventData.setWiFiServicesInitialized();
//#ifdef ESP32
        setWebserverRunning(false);
        delay(1);
        setWebserverRunning(true);
        delay(1);
/*        
#else
        CheckRunningServices();
#endif
*/
      }
    }
  }

  if (WiFiEventData.unprocessedWifiEvents()) {
    // Process disconnect events before connect events.
    if (!WiFiEventData.processedDisconnect) {
      #ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_DEBUG, F("WIFI : Entering processDisconnect()"));
      #endif // ifndef BUILD_NO_DEBUG
      processDisconnect();
    }
  }

  if (active_network_medium == NetworkMedium_t::WIFI) {
    if ((!WiFiEventData.WiFiServicesInitialized()) || WiFiEventData.unprocessedWifiEvents() || WiFiEventData.wifiConnectAttemptNeeded) {
      // WiFi connection is not yet available, so introduce some extra delays to
      // help the background tasks managing wifi connections
      delay(0);
      
      if (!WiFiEventData.processedConnect) {
        #ifndef BUILD_NO_DEBUG
        addLog(LOG_LEVEL_DEBUG, F("WIFI : Entering processConnect()"));
        #endif // ifndef BUILD_NO_DEBUG
        processConnect();
      }

      if (!WiFiEventData.processedGotIP) {
        #ifndef BUILD_NO_DEBUG
        addLog(LOG_LEVEL_DEBUG, F("WIFI : Entering processGotIP()"));
        #endif // ifndef BUILD_NO_DEBUG
        processGotIP();
      }

      if (!WiFiEventData.processedDHCPTimeout) {
        #ifndef BUILD_NO_DEBUG
        addLog(LOG_LEVEL_INFO, F("WIFI : DHCP timeout, Calling disconnect()"));
        #endif // ifndef BUILD_NO_DEBUG
        WiFiEventData.processedDHCPTimeout = true;
        WifiDisconnect();
      }

      if (WiFi.status() == WL_DISCONNECTED && WiFiEventData.wifiConnectInProgress) {
        if (WiFiEventData.last_wifi_connect_attempt_moment.isSet() && 
            WiFiEventData.last_wifi_connect_attempt_moment.timeoutReached(DEFAULT_WIFI_CONNECTION_TIMEOUT)) {
          logConnectionStatus();
          resetWiFi();
        }
        if (!WiFiEventData.last_wifi_connect_attempt_moment.isSet()) {
          WiFiEventData.wifiConnectInProgress = false;
        }
        delay(10);
      }

      if (!WiFiEventData.wifiConnectInProgress) {
        WiFiEventData.wifiConnectAttemptNeeded = true;
        NetworkConnectRelaxed();
      }
    }


    if (WiFiEventData.WiFiDisconnected()) {
      #ifndef BUILD_NO_DEBUG

      if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
        static LongTermTimer lastDisconnectMoment_log;
        static uint8_t lastWiFiStatus_log = 0;
        uint8_t cur_wifi_status = WiFi.status();
        if (WiFiEventData.lastDisconnectMoment.get() != lastDisconnectMoment_log.get() || 
            lastWiFiStatus_log != cur_wifi_status) {
          lastDisconnectMoment_log.set(WiFiEventData.lastDisconnectMoment.get());
          lastWiFiStatus_log = cur_wifi_status;
          String wifilog = F("WIFI : Disconnected: WiFi.status() = ");
          wifilog += WiFiEventData.ESPeasyWifiStatusToString();
          wifilog += F(" RSSI: ");
          wifilog += String(WiFi.RSSI());
          wifilog += F(" status: ");
          #ifdef ESP8266
          station_status_t status = wifi_station_get_connect_status();
          wifilog += SDKwifiStatusToString(status);
          #endif
          #ifdef ESP32
          wifilog += ArduinoWifiStatusToString(WiFi.status());
          #endif
          addLogMove(LOG_LEVEL_DEBUG, wifilog);
        }
      }
      #endif // ifndef BUILD_NO_DEBUG

      // While connecting to WiFi make sure the device has ample time to do so
      delay(10);
    }

    if (!WiFiEventData.processedDisconnectAPmode) { processDisconnectAPmode(); }

    if (!WiFiEventData.processedConnectAPmode) { processConnectAPmode(); }

    if (WiFiEventData.timerAPoff.isSet()) { processDisableAPmode(); }

    if (!WiFiEventData.processedScanDone) { processScanDone(); }

    if (WiFiEventData.wifi_connect_attempt > 0) {
      // We only want to clear this counter if the connection is currently stable.
      if (WiFiEventData.WiFiServicesInitialized()) {
        if (WiFiEventData.lastConnectMoment.isSet() && WiFiEventData.lastConnectMoment.timeoutReached(WIFI_CONNECTION_CONSIDERED_STABLE)) {
          // Connection considered stable
          WiFiEventData.wifi_connect_attempt = 0;
          WiFiEventData.wifi_considered_stable = true;
          WiFi_AP_Candidates.markCurrentConnectionStable();

          if (WiFi.getAutoReconnect() != Settings.SDK_WiFi_autoreconnect()) {
            WiFi.setAutoReconnect(Settings.SDK_WiFi_autoreconnect());
            delay(1);
          }
        } else {
          if (WiFi.getAutoReconnect()) {
            WiFi.setAutoReconnect(false);
            delay(1);
          }
        }
      }
    }
  }
#if FEATURE_ETHERNET
  check_Eth_DNS_valid();
#endif // if FEATURE_ETHERNET

#if FEATURE_ESPEASY_P2P
  updateUDPport(false);
#endif
}

// ********************************************************************************
// Functions to process the data gathered from the events.
// These functions are called from Setup() or Loop() and thus may call delay() or yield()
// ********************************************************************************
void processDisconnect() {
  if (WiFiEventData.processedDisconnect) { return; }

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log = strformat(
      F("WIFI : Disconnected! Reason: '%s'"), 
      getLastDisconnectReason().c_str());

    if (WiFiEventData.lastConnectedDuration_us > 0) {
      log += concat(
        F(" Connected for "), 
        format_msec_duration(WiFiEventData.lastConnectedDuration_us / 1000ll));
    }
    addLogMove(LOG_LEVEL_INFO, log);
  }
  logConnectionStatus();

  if (WiFiEventData.processingDisconnect.isSet()) {
    if (WiFiEventData.processingDisconnect.millisPassedSince() > 5000 || WiFiEventData.processedDisconnect) {
      WiFiEventData.processedDisconnect = true;
      WiFiEventData.processingDisconnect.clear();
    }
  }


  if (WiFiEventData.processedDisconnect || 
      WiFiEventData.processingDisconnect.isSet()) { return; }
  WiFiEventData.processingDisconnect.setNow();
  WiFiEventData.setWiFiDisconnected();
  WiFiEventData.wifiConnectAttemptNeeded = true;
  delay(100); // FIXME TD-er: See https://github.com/letscontrolit/ESPEasy/issues/1987#issuecomment-451644424

  if (Settings.UseRules) {
    eventQueue.add(F("WiFi#Disconnected"));
  }


  // FIXME TD-er: With AutoReconnect enabled, WiFi must be reset or else we completely loose track of the actual WiFi state
  bool mustRestartWiFi = Settings.WiFiRestart_connection_lost() || WiFi.getAutoReconnect();
  if (WiFiEventData.lastConnectedDuration_us > 0 && (WiFiEventData.lastConnectedDuration_us / 1000) < 5000) {
    if (!WiFi_AP_Candidates.getBestCandidate().usable())
//      addLog(LOG_LEVEL_INFO, F("WIFI : !getBestCandidate().usable()  => mustRestartWiFi = true"));

      mustRestartWiFi = true;
  }
  
  if (WiFi.status() == WL_IDLE_STATUS) {
//    addLog(LOG_LEVEL_INFO, F("WIFI : WiFi.status() == WL_IDLE_STATUS  => mustRestartWiFi = true"));
    mustRestartWiFi = true;
  }


  #ifdef USES_ESPEASY_NOW
  if (use_EspEasy_now) {
//    mustRestartWiFi = true;
  }
  #endif
  //WifiDisconnect(); // Needed or else node may not reconnect reliably.

  if (mustRestartWiFi) {
    WiFiEventData.processedDisconnect = true;
    resetWiFi();
//    WifiScan(false);
//    delay(100);
//    setWifiMode(WIFI_OFF);
//    initWiFi();
//    delay(100);
  }
//  delay(500);
  logConnectionStatus();
  WiFiEventData.processedDisconnect = true;
  WiFiEventData.processingDisconnect.clear();
}

void processConnect() {
  if (WiFiEventData.processedConnect) { return; }
  //delay(100); // FIXME TD-er: See https://github.com/letscontrolit/ESPEasy/issues/1987#issuecomment-451644424
  if (checkAndResetWiFi()) {
    return;
  }
  WiFiEventData.processedConnect = true;
  if (WiFi.status() == WL_DISCONNECTED) {
    // Apparently not really connected
    return;
  }

  WiFiEventData.setWiFiConnected();
  ++WiFiEventData.wifi_reconnects;

  if (WiFi_AP_Candidates.getCurrent().bits.isEmergencyFallback) {
    #ifdef CUSTOM_EMERGENCY_FALLBACK_RESET_CREDENTIALS
    const bool mustResetCredentials = CUSTOM_EMERGENCY_FALLBACK_RESET_CREDENTIALS;
    #else
    const bool mustResetCredentials = false;
    #endif
    #ifdef CUSTOM_EMERGENCY_FALLBACK_START_AP
    const bool mustStartAP = CUSTOM_EMERGENCY_FALLBACK_START_AP;
    #else
    const bool mustStartAP = false;
    #endif
    if (mustStartAP) {
      int allowedUptimeMinutes = 10;
      #ifdef CUSTOM_EMERGENCY_FALLBACK_ALLOW_MINUTES_UPTIME
      allowedUptimeMinutes = CUSTOM_EMERGENCY_FALLBACK_ALLOW_MINUTES_UPTIME;
      #endif
      if (getUptimeMinutes() < allowedUptimeMinutes) {
        WiFiEventData.timerAPstart.setNow();
      }
    }
    if (mustResetCredentials && !WiFiEventData.performedClearWiFiCredentials) {
      WiFiEventData.performedClearWiFiCredentials = true;
      SecuritySettings.clearWiFiCredentials();
      SaveSecuritySettings();
      WiFiEventData.markDisconnect(WIFI_DISCONNECT_REASON_AUTH_EXPIRE);
      WiFi_AP_Candidates.force_reload();
    }
  }

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    const LongTermTimer::Duration connect_duration = WiFiEventData.last_wifi_connect_attempt_moment.timeDiff(WiFiEventData.lastConnectMoment);
    String log = strformat(
      F("WIFI : Connected! AP: %s (%s) Ch: %d"),
      WiFi.SSID().c_str(),
      WiFi.BSSIDstr().c_str(),
      RTC.lastWiFiChannel);

    if ((connect_duration > 0ll) && (connect_duration < 30000000ll)) {
      // Just log times when they make sense.
      log += strformat(
        F(" Duration: %d ms"),
        static_cast<int32_t>(connect_duration / 1000));
    }
    addLogMove(LOG_LEVEL_INFO, log);
  }

//  WiFiEventData.last_wifi_connect_attempt_moment.clear();

  if (Settings.UseRules) {
    if (WiFiEventData.bssid_changed) {
      eventQueue.add(F("WiFi#ChangedAccesspoint"));
    }

    if (WiFiEventData.channel_changed) {
      eventQueue.add(F("WiFi#ChangedWiFichannel"));
    }
  } 

  if (useStaticIP()) {
    WiFiEventData.markGotIP(); // in static IP config the got IP event is never fired.
  }
  saveToRTC();

  logConnectionStatus();
}

void processGotIP() {
  if (WiFiEventData.processedGotIP) {
    return;
  }
  if (checkAndResetWiFi()) {
    return;
  }

  IPAddress ip = NetworkLocalIP();

  if (!useStaticIP()) {
    #ifdef ESP8266
    if (!ip.isSet()) {
    #else
    if (ip[0] == 0 && ip[1] == 0 && ip[2] == 0 && ip[3] == 0) {
    #endif
      return;
    }
  }
  const IPAddress gw       = WiFi.gatewayIP();
  const IPAddress subnet   = WiFi.subnetMask();
  const LongTermTimer::Duration dhcp_duration = WiFiEventData.lastConnectMoment.timeDiff(WiFiEventData.lastGetIPmoment);
  WiFiEventData.dns0_cache = WiFi.dnsIP(0);
  WiFiEventData.dns1_cache = WiFi.dnsIP(1);

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log = strformat(
      F("WIFI : %s (%s) GW: %s SN: %s DNS: %s"),
      concat(useStaticIP() ? F("Static IP: ") : F("DHCP IP: "), formatIP(ip)).c_str(),
      NetworkGetHostname().c_str(),
      formatIP(gw).c_str(),
      formatIP(subnet).c_str(),
      getValue(LabelType::DNS).c_str());

    if ((dhcp_duration > 0ll) && (dhcp_duration < 30000000ll)) {
      // Just log times when they make sense.
      log += strformat(F("   duration: %d ms"), static_cast<int32_t>(dhcp_duration / 1000));
    }
    addLogMove(LOG_LEVEL_INFO, log);
  }

  // Might not work in core 2.5.0
  // See https://github.com/esp8266/Arduino/issues/5839
  if ((Settings.IP_Octet != 0) && (Settings.IP_Octet != 255))
  {
    ip[3] = Settings.IP_Octet;

    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLogMove(LOG_LEVEL_INFO, concat(F("IP   : Fixed IP octet:"), formatIP(ip)));
    }
    WiFi.config(ip, gw, subnet, WiFiEventData.dns0_cache, WiFiEventData.dns1_cache);
  }

#if FEATURE_MQTT
  mqtt_reconnect_count        = 0;
  MQTTclient_should_reconnect = true;
  timermqtt_interval          = 100;
  Scheduler.setIntervalTimer(SchedulerIntervalTimer_e::TIMER_MQTT);
  scheduleNextMQTTdelayQueue();
#endif // if FEATURE_MQTT
  Scheduler.sendGratuitousARP_now();

  if (Settings.UseRules)
  {
    eventQueue.add(F("WiFi#Connected"));
  }
  statusLED(true);

  //  WiFi.scanDelete();

  if (WiFiEventData.wifiSetup) {
    // Wifi setup was active, Apparently these settings work.
    WiFiEventData.wifiSetup = false;
    SaveSecuritySettings();
  }

  if ((WiFiEventData.WiFiConnected() || WiFi.isConnected()) && hasIPaddr()) {
    WiFiEventData.setWiFiGotIP();
  }
  #if FEATURE_ESPEASY_P2P
  refreshNodeList();
  #endif
  logConnectionStatus();
}

#if FEATURE_USE_IPV6
void processGotIPv6() {
  if (!WiFiEventData.processedGotIP6) {
    WiFiEventData.processedGotIP6 = true;
    if (loglevelActiveFor(LOG_LEVEL_INFO))
      addLog(LOG_LEVEL_INFO, String(F("WIFI : STA got IP6 ")) + WiFiEventData.unprocessed_IP6.toString(true));
#if FEATURE_ESPEASY_P2P
//    updateUDPport(true);
#endif
  }
}
#endif

// A client disconnected from the AP on this node.
void processDisconnectAPmode() {
  if (WiFiEventData.processedDisconnectAPmode) { return; }
  WiFiEventData.processedDisconnectAPmode = true;

#ifndef BUILD_NO_DEBUG
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    const int nrStationsConnected = WiFi.softAPgetStationNum();
    String    log                 = F("AP Mode: Client disconnected: ");
    log += WiFiEventData.lastMacDisconnectedAPmode.toString();
    log += F(" Connected devices: ");
    log += nrStationsConnected;
    addLogMove(LOG_LEVEL_INFO, log);
  }
#endif  
}

// Client connects to AP on this node
void processConnectAPmode() {
  if (WiFiEventData.processedConnectAPmode) { return; }
  WiFiEventData.processedConnectAPmode = true;
  // Extend timer to switch off AP.
  WiFiEventData.timerAPoff.setMillisFromNow(WIFI_AP_OFF_TIMER_DURATION);
#ifndef BUILD_NO_DEBUG
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log = F("AP Mode: Client connected: ");
    log += WiFiEventData.lastMacConnectedAPmode.toString();
    log += F(" Connected devices: ");
    log += WiFi.softAPgetStationNum();
    addLogMove(LOG_LEVEL_INFO, log);
  }
#endif

  #if FEATURE_DNS_SERVER
  // Start DNS, only used if the ESP has no valid WiFi config
  // It will reply with it's own address on all DNS requests
  // (captive portal concept)
  if (!dnsServerActive) {
    dnsServerActive = true;
    dnsServer.start(DNS_PORT, "*", apIP);
  }
  #endif // if FEATURE_DNS_SERVER
}

// Switch of AP mode when timeout reached and no client connected anymore.
void processDisableAPmode() {
  if (!WiFiEventData.timerAPoff.isSet()) { return; }

  if (!WifiIsAP(WiFi.getMode())) {
    return;
  }
  // disable AP after timeout and no clients connected.
  if (WiFiEventData.timerAPoff.timeReached() && (WiFi.softAPgetStationNum() == 0)) {
    setAP(false);
  }

  if (!WifiIsAP(WiFi.getMode())) {
    WiFiEventData.timerAPoff.clear();
    if (WiFiEventData.wifiConnectAttemptNeeded) {
      // Force a reconnect cycle
      WifiDisconnect();
    }
  }
}

void processScanDone() {
  WiFi_AP_Candidates.load_knownCredentials();
  if (WiFiEventData.processedScanDone) { return; }



  // Better act on the scan done event, as it may get triggered for normal wifi begin calls.
  int8_t scanCompleteStatus = WiFi.scanComplete();
  switch (scanCompleteStatus) {
    case 0: // Nothing (yet) found
      if (WiFiEventData.lastGetScanMoment.timeoutReached(5000)) {
        WiFi.scanDelete();
        WiFiEventData.processedScanDone = true;
      }
      return;
    case -1: // WIFI_SCAN_RUNNING
      // FIXME TD-er: Set timeout...
      if (WiFiEventData.lastGetScanMoment.timeoutReached(5000)) {
        # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, F("WiFi : Scan Running Timeout"));
      #endif
        WiFi.scanDelete();
        WiFiEventData.processedScanDone = true;
      }
      return;
    case -2: // WIFI_SCAN_FAILED
      addLog(LOG_LEVEL_ERROR, F("WiFi : Scan failed"));
      WiFi.scanDelete();
      WiFiEventData.processedScanDone = true;
      return;
  }

  WiFiEventData.lastGetScanMoment.setNow();
  WiFiEventData.processedScanDone = true;
# ifndef BUILD_NO_DEBUG
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLogMove(LOG_LEVEL_INFO, concat(F("WiFi : Scan finished, found: "), scanCompleteStatus));
  }
#endif

#if !FEATURE_ESP8266_DIRECT_WIFI_SCAN
  WiFi_AP_Candidates.process_WiFiscan(scanCompleteStatus);
#endif
  WiFi_AP_Candidates.load_knownCredentials();

  if (WiFi_AP_Candidates.addedKnownCandidate() && !NetworkConnected()) {
    if (!WiFiEventData.wifiConnectInProgress) {
      WiFiEventData.wifiConnectAttemptNeeded = true;
      # ifndef BUILD_NO_DEBUG
      if (WiFi_AP_Candidates.addedKnownCandidate()) {
        addLog(LOG_LEVEL_INFO, F("WiFi : Added known candidate, try to connect"));
      }
      #endif
#ifdef ESP32
//      setSTA(false);
#endif
      NetworkConnectRelaxed();
#ifdef USES_ESPEASY_NOW
      temp_disable_EspEasy_now_timer = millis() + 20000;
#endif
    }
  } else if (!WiFiEventData.wifiConnectInProgress && !NetworkConnected()) {
    WiFiEventData.timerAPstart.setNow();
  }

}





#include "../ESPEasyCore/ESPEasyRules.h"

#include "../../_Plugin_Helper.h"

#include "../Commands/ExecuteCommand.h"
#include "../DataStructs/TimingStats.h"
#include "../DataTypes/EventValueSource.h"
#include "../ESPEasyCore/ESPEasy_backgroundtasks.h"
#include "../ESPEasyCore/Serial.h"
#include "../Globals/Cache.h"
#include "../Globals/Device.h"
#include "../Globals/EventQueue.h"
#include "../Globals/Plugins.h"
#include "../Globals/Plugins_other.h"
#include "../Globals/RulesCalculate.h"
#include "../Globals/Settings.h"
#include "../Helpers/CRC_functions.h"
#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/FS_Helper.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Numerical.h"
#include "../Helpers/RulesHelper.h"
#include "../Helpers/RulesMatcher.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringParser.h"


#include <math.h>
#include <vector>

#ifdef WEBSERVER_NEW_RULES
String EventToFileName(const String& eventName) {
  int size  = eventName.length();
  int index = eventName.indexOf('=');

  if (index > -1) {
    size = index;
  }
#if defined(ESP8266)
  String fileName = F("rules/");
#endif // if defined(ESP8266)
#if defined(ESP32)
  String fileName = F("/rules/");
#endif // if defined(ESP32)
  fileName += eventName.substring(0, size);
  fileName.replace('#', RULE_FILE_SEPARAROR);
  fileName.toLowerCase();
  return fileName;
}

String FileNameToEvent(const String& fileName) {
#if defined(ESP8266)
  String eventName = fileName.substring(6);
#endif // if defined(ESP8266)
#if defined(ESP32)
  String eventName = fileName.substring(7);
#endif // if defined(ESP32)
  eventName.replace(RULE_FILE_SEPARAROR, '#');
  return eventName;
}
#endif

void checkRuleSets() {
  Cache.rulesHelper.closeAllFiles();
}

/********************************************************************************************\
   Process next event from event queue
 \*********************************************************************************************/
bool processNextEvent() {
  if (Settings.UseRules)
  {
    String nextEvent;

    if (eventQueue.getNext(nextEvent)) {
      rulesProcessing(nextEvent);
      return true;
    }
  }

  // Just make sure any (accidentally) added or remaining events are not kept.
  eventQueue.clear();
  return false;
}

/********************************************************************************************\
   Rules processing
 \*********************************************************************************************/
void rulesProcessing(const String& event) {
  if (!Settings.UseRules) {
    return;
  }
  START_TIMER
  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("rulesProcessing"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
#ifndef BUILD_NO_DEBUG
  const unsigned long timer = millis();
#endif // ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLogMove(LOG_LEVEL_INFO, concat(F("EVENT: "), event));
  }

  if (Settings.OldRulesEngine()) {
    bool eventHandled = false;

    if (Settings.EnableRulesCaching()) {
      String filename;
      size_t pos = 0;
      if (Cache.rulesHelper.findMatchingRule(event, filename, pos)) {
        const bool startOnMatched = true; // We already matched the event
        eventHandled = rulesProcessingFile(filename, event, pos, startOnMatched);
      }
    } else {
      for (uint8_t x = 0; x < RULESETS_MAX && !eventHandled; x++) {
        eventHandled = rulesProcessingFile(getRulesFileName(x), event);
      }
    }
  } else {
    #ifdef WEBSERVER_NEW_RULES
    String fileName = EventToFileName(event);

    // if exists processed the rule file
    if (fileExists(fileName)) {
      rulesProcessingFile(fileName, event);
    }
    # ifndef BUILD_NO_DEBUG
    else {
      addLog(LOG_LEVEL_DEBUG, strformat(F("EVENT: %s is ingnored. File %s not found."),
             event.c_str(), fileName.c_str()));
    }
    # endif    // ifndef BUILD_NO_DEBUG
    #endif // WEBSERVER_NEW_RULES
  }

#ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    addLogMove(LOG_LEVEL_DEBUG, strformat(F("EVENT: %s Processing: %d ms"), event.c_str(), timePassedSince(timer)));
  }
#endif // ifndef BUILD_NO_DEBUG
  STOP_TIMER(RULES_PROCESSING);
  backgroundtasks();
}

/********************************************************************************************\
   Rules processing
 \*********************************************************************************************/
bool rulesProcessingFile(const String& fileName, 
                         const String& event, 
                         size_t pos,
                         bool   startOnMatched) {
  if (!Settings.UseRules || !fileExists(fileName)) {
    return false;
  }
  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("rulesProcessingFile"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
#ifndef BUILD_NO_DEBUG

  if (Settings.SerialLogLevel == LOG_LEVEL_DEBUG_DEV) {
    serialPrint(F("RuleDebug Processing:"));
    serialPrintln(fileName);
    serialPrintln(F("     flags CMI  parse output:"));
  }
#endif // ifndef BUILD_NO_DEBUG

  static uint8_t nestingLevel = 0;

  nestingLevel++;

  if (nestingLevel > RULES_MAX_NESTING_LEVEL) {
    addLog(LOG_LEVEL_ERROR, F("EVENT: Error: Nesting level exceeded!"));
    nestingLevel--;
    return false;
  }


  bool match     = false;
  bool codeBlock = false;
  bool isCommand = false;
  bool condition[RULES_IF_MAX_NESTING_LEVEL];
  bool ifBranche[RULES_IF_MAX_NESTING_LEVEL];
  uint8_t ifBlock     = 0;
  uint8_t fakeIfBlock = 0;


  bool moreAvailable = true;
  bool eventHandled = false;
  while (moreAvailable && !eventHandled) {
    const bool searchNextOnBlock = !codeBlock && !match;
    String line = Cache.rulesHelper.readLn(fileName, pos, moreAvailable, searchNextOnBlock);

    // Parse the line and extract the action (if there is any)
    String action;
    {
      START_TIMER
      const bool matched_before_parse = match;
      bool isOneLiner = false;
      parseCompleteNonCommentLine(line, event, action, match, codeBlock,
                                  isCommand, isOneLiner, condition, ifBranche, ifBlock,
                                  fakeIfBlock, startOnMatched);
      if ((matched_before_parse && !match) || isOneLiner) {
        // We were processing a matching event and now crossed the "endon"
        // Or just dealing with a oneliner.
        // So we're done processing
        eventHandled = true;
        backgroundtasks();
      }
      STOP_TIMER(RULES_PARSE_LINE);
    }

    if (match) // rule matched for one action or a block of actions
    {
      START_TIMER
      processMatchedRule(action, event,
                         isCommand, condition,
                         ifBranche, ifBlock, fakeIfBlock);
      STOP_TIMER(RULES_PROCESS_MATCHED);
    }
  }

/*
  if (f) {
    f.close();
  }
*/

  nestingLevel--;
  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("rulesProcessingFile2"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
  backgroundtasks();
  return eventHandled; // && nestingLevel == 0;
}


/********************************************************************************************\
   Parse string commands
 \*********************************************************************************************/
bool get_next_inner_bracket(const String& line, unsigned int& startIndex, int& closingIndex, char closingBracket)
{
  if (line.length() <= 1) {
    // Not possible to have opening and closing bracket on a line this short.
    return false;
  }
  char openingBracket = closingBracket;

  switch (closingBracket) {
    case ']': openingBracket = '['; break;
    case '}': openingBracket = '{'; break;
    case ')': openingBracket = '('; break;
    default:
      // unknown bracket type
      return false;
  }
  // Closing bracket should not be found on the first position.
  closingIndex = line.indexOf(closingBracket, startIndex + 1);

  if (closingIndex == -1) { 
    // not found
    return false; 
  }

  for (int i = (closingIndex - 1); (i >= static_cast<int>(startIndex)) && (i >= 0); --i) {
    if (line[i] == openingBracket) {
      startIndex = i;
      return true;
    }
  }
  return false;
}

bool get_next_argument(const String& fullCommand, int& index, String& argument, char separator)
{
  if (index == -1) {
    return false;
  }
  int newIndex = fullCommand.indexOf(separator, index);

  if (newIndex == -1) {
    argument = fullCommand.substring(index);
  } else {
    argument = fullCommand.substring(index, newIndex);
  }

  if (argument.startsWith(String(separator))) {
    argument = argument.substring(1);
  }

  //  addLog(LOG_LEVEL_INFO, String("get_next_argument: ") + String(index) + " " + fullCommand + " " + argument);
  index = newIndex;

  if (index != -1) {
    ++index;
  }
  return argument.length() > 0;
}

const char bitwise_functions[] PROGMEM = "bitread|bitset|bitclear|bitwrite|xor|and|or";
enum class bitwise_functions_e {
  bitread,
  bitset,
  bitclear,
  bitwrite,
  xor_e,  // protected keywords, thus appended _e
  and_e,
  or_e
};


bool parse_bitwise_functions(const String& cmd_s_lower, const String& arg1, const String& arg2, const String& arg3, int64_t& result) {
  #ifndef BUILD_NO_DEBUG
  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    String log = F("Bitwise: {");
    log += wrapIfContains(cmd_s_lower, ':', '\"');
    log += ':';
    log += wrapIfContains(arg1, ':', '\"');

    if (arg2.length() > 0) {
      log += ':';
      log += wrapIfContains(arg2, ':', '\"');

      if (arg3.length() > 0) {
        log += ':';
        log += wrapIfContains(arg3, ':', '\"');
      }
    }
    log += '}';
    addLogMove(LOG_LEVEL_DEBUG, log);
  }
  #endif

  if (cmd_s_lower.length() < 2) {
    return false;
  }

  int command_i = GetCommandCode(cmd_s_lower.c_str(), bitwise_functions);
  if (command_i == -1) {
    // No matching function found
    return false;
  }
  
  if (cmd_s_lower.startsWith(F("bit"))) {
    uint32_t bitnr = 0;
    uint64_t iarg2 = 0;

    if (!validUIntFromString(arg1, bitnr) || !validUInt64FromString(arg2, iarg2)) {
      return false;
    }

    switch(static_cast<bitwise_functions_e>(command_i)) {
      case bitwise_functions_e::bitread:
        // Syntax like {bitread:0:123} to get a single decimal '1'
        result = bitRead(iarg2, bitnr);
        break;
      case bitwise_functions_e::bitset:
        // Syntax like {bitset:0:122} to set least significant bit of the given nr '122' to '1' => '123'
        result = iarg2;
        bitSetULL(result, bitnr);
        break;
      case bitwise_functions_e::bitclear:
        // Syntax like {bitclear:0:123} to set least significant bit of the given nr '123' to '0' => '122'
        result = iarg2;
        bitClearULL(result, bitnr);
        break;
      case bitwise_functions_e::bitwrite:
      {
        uint32_t iarg3 = 0;
        // Syntax like {bitwrite:0:122:1} to set least significant bit of the given nr '122' to '1' => '123'
        if (validUIntFromString(arg3, iarg3)) {
          const int bitvalue = (iarg3 & 1); // Only use the last bit of the given parameter
          result = iarg2;
          bitWriteULL(result, bitnr, bitvalue);
        } else {
          // Need 3 parameters, but 3rd one is not a valid uint
          return false;
        }
        break;
      }
      default: 
        return false;
    }

    // all functions starting with "bit" are checked
    return true;
  }

  uint64_t iarg1, iarg2 = 0;

  if (!validUInt64FromString(arg1, iarg1) || !validUInt64FromString(arg2, iarg2)) {
    return false;
  }

  switch(static_cast<bitwise_functions_e>(command_i)) {
    case bitwise_functions_e::xor_e:
      // Syntax like {xor:127:15} to XOR the binary values 1111111 and 1111 => 1110000
      result = iarg1 ^ iarg2;
      break;
    case bitwise_functions_e::and_e:
      // Syntax like {and:254:15} to AND the binary values 11111110 and 1111 => 1110
      result = iarg1 & iarg2;
      break;
    case bitwise_functions_e::or_e:
      // Syntax like {or:254:15} to OR the binary values 11111110 and 1111 => 11111111
      result = iarg1 | iarg2;
      break;
    default: 
      return false;

  }
  return true;
}

bool parse_math_functions(const String& cmd_s_lower, const String& arg1, const String& arg2, const String& arg3, ESPEASY_RULES_FLOAT_TYPE& result) {
  ESPEASY_RULES_FLOAT_TYPE farg1;
  float  farg2, farg3 = 0.0f;

  if (!cmd_s_lower.startsWith("crc") && !validDoubleFromString(arg1, farg1)) {
    return false;
  }

  if (equals(cmd_s_lower, F("constrain"))) {
    // Contrain a value X to be within range of A to B
    // Syntax like {constrain:x:a:b} to constrain x in range a...b
    if (validFloatFromString(arg2, farg2) && validFloatFromString(arg3, farg3)) {
      if (farg2 > farg3) {
        const float tmp = farg2;
        farg2 = farg3;
        farg3 = tmp;
      }
      result = constrain(farg1, farg2, farg3);
    } else {
      return false;
    }
  } else if (cmd_s_lower.startsWith("crc")) {
    std::vector<uint8_t> argument = parseHexTextData(arg1, 1);
    const String crctype          = cmd_s_lower.substring(3);

    if (argument.size() > 0) {
      if (equals(crctype, F("8"))) {
        result = calc_CRC8(&argument[0], argument.size());
      // } else if (equals(crctype, F("16"))) { // FIXME crc16 not supported until needed/used/tested
      //   result = calc_CRC16((const char *)argument.data(), argument.size());
      } else if (equals(crctype, F("32"))) {
        result = calc_CRC32(&argument[0], argument.size());
      } else {
        return false;
      }

      if (!arg2.isEmpty() && validDoubleFromString(arg2, farg1)) { // Optional expected crc value
        result = essentiallyEqual(result, farg1) ? 1.0 : 0.0; // Return 1 if the calculated crc == expected crc
      }
    } else {
      return false;
    }
  } else {
    // No matching function found
    return false;
  }
  return true;
}

const char string_commands[] PROGMEM = "substring|indexof|indexof_ci|equals|equals_ci|timetomin|timetosec|strtol|tobin|tohex|ord|urlencode"
  #if FEATURE_STRING_VARIABLES
  "|lookup"
  #endif // if FEATURE_STRING_VARIABLES
  ;
enum class string_commands_e {
  substring,
  indexof,
  indexof_ci,
  equals,
  equals_ci,
  timetomin,
  timetosec,
  strtol,
  tobin,
  tohex,
  ord,
  urlencode,
  #if FEATURE_STRING_VARIABLES
  lookup,
  #endif // if FEATURE_STRING_VARIABLES
};


void parse_string_commands(String& line) {
  unsigned int startIndex = 0;
  int closingIndex;

  bool mustReplaceMaskedChars = false;
  bool mustReplaceEscapedBracket = false;
  bool mustReplaceEscapedCurlyBracket = false;
  String MaskEscapedBracket;

  if (hasEscapedCharacter(line,'(') || hasEscapedCharacter(line,')')) {
    // replace the \( and \) with other characters to mask the escaped brackets so we can continue parsing.
    // We have to unmask then after we're finished.
    MaskEscapedBracket = static_cast<char>(0x11); // ASCII 0x11 = Device control 1
    line.replace(F("\\("), MaskEscapedBracket);
    MaskEscapedBracket = static_cast<char>(0x12); // ASCII 0x12 = Device control 2
    line.replace(F("\\)"), MaskEscapedBracket);
    mustReplaceEscapedBracket = true;
  }
  if (hasEscapedCharacter(line,'{') || hasEscapedCharacter(line,'}')) {
    // replace the \{ and \} with other characters to mask the escaped curly brackets so we can continue parsing.
    // We have to unmask then after we're finished.
    MaskEscapedBracket = static_cast<char>(0x13); // ASCII 0x13 = Device control 3
    line.replace(F("\\{"), MaskEscapedBracket);
    MaskEscapedBracket = static_cast<char>(0x14); // ASCII 0x14 = Device control 4
    line.replace(F("\\}"), MaskEscapedBracket);
    mustReplaceEscapedCurlyBracket = true;
  }

  while (get_next_inner_bracket(line, startIndex, closingIndex, '}')) {
    // Command without opening and closing brackets.
    const String fullCommand = line.substring(startIndex + 1, closingIndex);
    const String cmd_s_lower = parseString(fullCommand, 1, ':');
    const String arg1        = parseStringKeepCaseNoTrim(fullCommand, 2, ':');
    const String arg2        = parseStringKeepCaseNoTrim(fullCommand, 3, ':');
    const String arg3        = parseStringKeepCaseNoTrim(fullCommand, 4, ':');

    if (cmd_s_lower.length() > 0) {
      String replacement; // maybe just replace with empty to avoid looping?
      uint64_t iarg1, iarg2 = 0;
      ESPEASY_RULES_FLOAT_TYPE fresult{};
      int64_t  iresult = 0;
      int32_t startpos, endpos = -1;
      const bool arg1valid = validIntFromString(arg1, startpos);
      const bool arg2valid = validIntFromString(arg2, endpos);

      if (parse_math_functions(cmd_s_lower, arg1, arg2, arg3, fresult)) {
        const bool trimTrailingZeros = true;
        #if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
        replacement = doubleToString(fresult, maxNrDecimals_fpType(fresult), trimTrailingZeros);
        #else
        replacement = floatToString(fresult, maxNrDecimals_fpType(fresult), trimTrailingZeros);
        #endif
      } else if (parse_bitwise_functions(cmd_s_lower, arg1, arg2, arg3, iresult)) {
        replacement = ull2String(iresult);
      } else {

        int command_i = GetCommandCode(cmd_s_lower.c_str(), string_commands);
        if (command_i != -1) {
          const string_commands_e command = static_cast<string_commands_e>(command_i);

              //  addLog(LOG_LEVEL_INFO, strformat(F("parse_string_commands cmd: %s %s %s %s"), 
              // cmd_s_lower.c_str(), arg1.c_str(), arg2.c_str(), arg3.c_str()));

          switch (command) {
            case string_commands_e::substring:
              // substring arduino style (first char included, last char excluded)
              // Syntax like 12345{substring:8:12:ANOTHER HELLO WORLD}67890

              if (arg1valid) {
                if (arg2valid){
                  replacement = arg3.substring(startpos, endpos);
                } else {
                  replacement = arg3.substring(startpos);
                }
              }
              break;
            #if FEATURE_STRING_VARIABLES
            case string_commands_e::lookup:
              if (arg1valid && arg2valid && startpos > -1 && endpos > -1) {
                replacement = arg3.substring(startpos * endpos, (startpos + 1) * endpos);
              }
              break;
            #endif // if FEATURE_STRING_VARIABLES
            case string_commands_e::indexof:
            case string_commands_e::indexof_ci:
              // indexOf arduino style (0-based position of first char returned, -1 if not found, case sensitive), 3rd argument is search-offset
              // indexOf_ci : case-insensitive
              // Syntax like {indexof:HELLO:"ANOTHER HELLO WORLD"} => 8, {indexof:hello:"ANOTHER HELLO WORLD"} => -1, {indexof_ci:Hello:"ANOTHER HELLO WORLD"} => 8
              // or like {indexof_ci:hello:"ANOTHER HELLO WORLD":10} => -1

              if (!arg1.isEmpty()
                  && !arg2.isEmpty()) {
                uint32_t offset = 0;
                validUIntFromString(arg3, offset);
                if (command == string_commands_e::indexof_ci) {
                  String arg1copy(arg1);
                  String arg2copy(arg2);
                  arg1copy.toLowerCase();
                  arg2copy.toLowerCase();
                  replacement = arg2copy.indexOf(arg1copy, offset);
                } else {
                  replacement = arg2.indexOf(arg1, offset);
                }
              }
              break;
            case string_commands_e::equals:
            case string_commands_e::equals_ci:
              // equals: compare strings 1 = equal, 0 = unequal (case sensitive)
              // equals_ci: case-insensitive compare
              // Syntax like {equals:HELLO:HELLO} => 1, {equals:hello:HELLO} => 0, {equals_ci:hello:HELLO} => 1, {equals_ci:hello:BLA} => 0

              if (!arg1.isEmpty()
                  && !arg2.isEmpty()) {
                if (command == string_commands_e::equals_ci) {
                  replacement = arg2.equalsIgnoreCase(arg1);
                } else {
                  replacement = arg2.equals(arg1);
                }
              }
              break;
            case string_commands_e::timetomin:
            case string_commands_e::timetosec:
              // time to minutes, transform a substring hh:mm to minutes
              // time to seconds, transform a substring hh:mm:ss to seconds
              // syntax similar to substring

              if (arg1valid
                  && arg2valid) {
                int timeSeconds = 0;
                String timeString;
                if(timeStringToSeconds(arg3.substring(startpos, endpos), timeSeconds, timeString)) {
                  if (command == string_commands_e::timetosec) {
                    replacement = timeSeconds;
                  } else { // timetomin
                    replacement = timeSeconds / 60;
                  }
                }
              }
              break;
            case string_commands_e::strtol:
              // string to long integer (from cstdlib)
              // Syntax like 1234{strtol:16:38}7890
              if (validUInt64FromString(arg1, iarg1)
                  && validUInt64FromString(arg2, iarg2)) {
                replacement = String(strtoul(arg2.c_str(), nullptr, iarg1));
              }
              break;
            case string_commands_e::tobin:
              // Convert to binary string
              // Syntax like 1234{tobin:15}7890
              if (validUInt64FromString(arg1, iarg1)) {
                replacement = ull2String(iarg1, BIN);
              }
              break;
            case string_commands_e::tohex:
              // Convert to HEX string
              // Syntax like 1234{tohex:15[,minHexDigits]}7890
              if (validUInt64FromString(arg1, iarg1)) {
                if (!validUInt64FromString(arg2, iarg2)) {
                  iarg2 = 0;
                }
                replacement = formatToHex_no_prefix(iarg1, iarg2);
              }
              break;
            case string_commands_e::ord:
              {
                // Give the ordinal/integer value of the first character of a string
                // Syntax like let 1,{ord:B}
                uint8_t uval = arg1.c_str()[0];
                replacement = String(uval);
              }
              break;
            case string_commands_e::urlencode:
              // Convert to url-encoded string
              // Syntax like {urlencode:"string to/encode"}
              if (!arg1.isEmpty()) {
                replacement = URLEncode(arg1);
              }
              break;
          }
        }
      }

      if (replacement.isEmpty()) {
        // part in braces is not a supported command.
        // replace the {} with other characters to mask the braces so we can continue parsing.
        // We have to unmask then after we're finished.
        // See: https://github.com/letscontrolit/ESPEasy/issues/2932#issuecomment-596139096
        replacement = line.substring(startIndex, closingIndex + 1);
        replacement.replace('{', static_cast<char>(0x02));
        replacement.replace('}', static_cast<char>(0x03));
        mustReplaceMaskedChars = true;
      }

      // Replace the full command including opening and closing brackets.
      line.replace(line.substring(startIndex, closingIndex + 1), replacement);

      /*
         if (replacement.length() > 0) {
         addLog(LOG_LEVEL_INFO, strformat(F("parse_string_commands cmd: %s -> %s"), fullCommand.c_str(), replacement.c_str());
         }
       */
    }
  }

  if (mustReplaceMaskedChars) {
    // We now have to check if we did mask some parts and unmask them.
    // Let's hope we don't mess up any Unicode here.
    line.replace(static_cast<char>(0x02), '{');
    line.replace(static_cast<char>(0x03), '}');
  }

  if (mustReplaceEscapedBracket) {
    // We now have to check if we did mask some escaped bracket and unmask them.
    // Let's hope we don't mess up any Unicode here.
    MaskEscapedBracket = static_cast<char>(0x11); // ASCII 0x11 = Device control 1
    line.replace(MaskEscapedBracket, F("\\("));
    MaskEscapedBracket = static_cast<char>(0x12); // ASCII 0x12 = Device control 2
    line.replace(MaskEscapedBracket, F("\\)"));
  }

  if (mustReplaceEscapedCurlyBracket) {
    // We now have to check if we did mask some escaped curly bracket and unmask them.
    // Let's hope we don't mess up any Unicode here.
    MaskEscapedBracket = static_cast<char>(0x13); // ASCII 0x13 = Device control 3
    line.replace(MaskEscapedBracket, F("\\{"));
    MaskEscapedBracket = static_cast<char>(0x14); // ASCII 0x14 = Device control 4
    line.replace(MaskEscapedBracket, F("\\}"));
  }
}

void substitute_eventvalue(String& line, const String& event) {
  if (substitute_eventvalue_CallBack_ptr != nullptr) {
    substitute_eventvalue_CallBack_ptr(line, event);
  }

  if (line.indexOf(F("%event")) != -1) {
    const int equalsPos = event.indexOf('=');

    if (event.charAt(0) == '!') {
      line.replace(F("%eventvalue%"), event); // substitute %eventvalue% with
                                              // literal event string if
                                              // starting with '!'
    } else {

      String argString;

      if (equalsPos > 0) {
        argString = event.substring(equalsPos + 1);
      }

      // Replace %eventvalueX% with the actual value of the event.
      // For compatibility reasons also replace %eventvalue%  (argc = 0)
      line.replace(F("%eventvalue%"), F("%eventvalue1%"));
      int eventvalue_pos = line.indexOf(F("%eventvalue"));

      while (eventvalue_pos != -1) {
        const int percent_pos = line.indexOf('%', eventvalue_pos + 1);

        if (percent_pos == -1) {
          // Found "%eventvalue" without closing %
          if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
            String log = F("Rules : Syntax error, missing '%' in '%eventvalue%': '");
            log += line;
            log += F("' %-pos: ");
            log += percent_pos;
            addLog(LOG_LEVEL_ERROR, log);
          }
          line.replace(F("%eventvalue"), F(""));
        } else {
          // Find the optional part for a default value when the asked for eventvalue does not exist.
          // Syntax: %eventvalueX|Y%
          // With: X = event value nr, Y = default value when eventvalue does not exist.
          String defaultValue('0');
          int or_else_pos = line.indexOf('|', eventvalue_pos);
          if ((or_else_pos == -1) || (or_else_pos > percent_pos)) {
            or_else_pos = percent_pos;
          } else {
            defaultValue = line.substring(or_else_pos + 1, percent_pos);
          }
          const String nr         = line.substring(eventvalue_pos + 11, or_else_pos);
          const String eventvalue = line.substring(eventvalue_pos, percent_pos + 1);
          int argc                = -1;

          if (equals(nr, '0')) {
            // Replace %eventvalue0% with the entire list of arguments.
            line.replace(eventvalue, argString);
          } else {
            argc = nr.toInt();
            
            // argc will be 0 on invalid int (0 was already handled)
            if (argc > 0) {
              String tmpParam;

              if (!GetArgv(argString.c_str(), tmpParam, argc)) {
                // Replace with default value for non existing event values
                tmpParam = parseTemplate(defaultValue);
              }
              line.replace(eventvalue, tmpParam);
            } else {
              // Just remove the invalid eventvalue variable
              if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
                addLog(LOG_LEVEL_ERROR, concat(F("Rules : Syntax error, invalid variable: "), eventvalue));
              }
              line.replace(eventvalue, EMPTY_STRING);
            }
          }
        }
        eventvalue_pos = line.indexOf(F("%eventvalue"));
      }
    }

    if ((line.indexOf(F("%eventname%")) != -1) ||
        (line.indexOf(F("%eventpar%")) != -1)) {
      const String eventName = equalsPos == -1 ? event : event.substring(0, equalsPos);

      // Replace %eventname% with the literal event
      line.replace(F("%eventname%"), eventName);

      // Part of %eventname% after the # char
      const int hash_pos = eventName.indexOf('#');
      line.replace(F("%eventpar%"), hash_pos == -1 ? EMPTY_STRING : eventName.substring(hash_pos + 1));
    }
  }
}

void parseCompleteNonCommentLine(String& line, const String& event,
                                 String& action, bool& match,
                                 bool& codeBlock, bool& isCommand, bool& isOneLiner,
                                 bool condition[], bool ifBranche[],
                                 uint8_t& ifBlock, uint8_t& fakeIfBlock,
                                 bool   startOnMatched) {
  if (line.length() == 0) {
    return;
  }
  const bool lineStartsWith_on = line.substring(0, 3).equalsIgnoreCase(F("on "));

  if (!codeBlock && !match) {
    // We're looking for a new code block.
    // Continue with next line if none was found on current line.
    if (!lineStartsWith_on) {
      return;
    }
  }

  if (line.equalsIgnoreCase(F("endon"))) // Check if action block has ended, then we will
                                           // wait for a new "on" rule
  {
    isCommand   = false;
    codeBlock   = false;
    match       = false;
    ifBlock     = 0;
    fakeIfBlock = 0;
    return;
  }

  const bool lineStartsWith_pct_event = line.startsWith(F("%event"));;

  isCommand = true;

  if (match || !codeBlock) {
    // only parse [xxx#yyy] if we have a matching ruleblock or need to eval the
    // "on" (no codeBlock)
    // This to avoid wasting CPU time...
    if (match && !fakeIfBlock) {
      // substitution of %eventvalue% is made here so it can be used on if
      // statement too
      substitute_eventvalue(line, event);
    }

    if (match || lineStartsWith_on) {
      // Only parseTemplate when we are actually doing something with the line.
      // When still looking for the "on ... do" part, do not change it before we found the block.
      line = parseTemplate(line);
    }
  }

  if (!codeBlock) // do not check "on" rules if a block of actions is to be
                  // processed
  {
    action.clear();
    if (lineStartsWith_on) {
      ifBlock     = 0;
      fakeIfBlock = 0;

      String ruleEvent;
      if (getEventFromRulesLine(line, ruleEvent, action)) {
        START_TIMER
        match = startOnMatched || ruleMatch(event, ruleEvent);
        STOP_TIMER(RULES_MATCH);
      } else {
        match = false;
      }

      if (action.length() > 0) // single on/do/action line, no block
      {
        isCommand  = true;
        isOneLiner = true;
        codeBlock  = false;
      } else {
        isCommand = false;
        codeBlock = true;
      }
    }
  } else {
    #ifndef BUILD_NO_DEBUG
    if (loglevelActiveFor(LOG_LEVEL_DEBUG_DEV)) {
      // keep the line for the log
      action = line;
    } else {
      action = std::move(line);  
    }
    #else
    action = std::move(line);
    #endif
  }

  if (isCommand && lineStartsWith_pct_event) {
    action = concat(F("restrict,"), action);
    if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
      addLogMove(LOG_LEVEL_ERROR, 
        concat(F("Rules : Prefix command with 'restrict': "), action));
    }
  }

#ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_DEBUG_DEV)) {
    String log = F("RuleDebug: ");
    log += codeBlock ? 0 : 1;
    log += match ? 0 : 1;
    log += isCommand ? 0 : 1;
    log += F(": ");
    log += line;
    addLogMove(LOG_LEVEL_DEBUG_DEV, log);
  }
#endif // ifndef BUILD_NO_DEBUG
}

void processMatchedRule(String& action, const String& event,
                        bool& isCommand, bool condition[], bool ifBranche[],
                        uint8_t& ifBlock, uint8_t& fakeIfBlock) {
  String lcAction = action;

  lcAction.toLowerCase();
  lcAction.trim();

  if (fakeIfBlock) {
    isCommand = false;
  }
  else if (ifBlock) {
    if (condition[ifBlock - 1] != ifBranche[ifBlock - 1]) {
      isCommand = false;
    }
  }
  int split =
    lcAction.startsWith(F("elseif ")) ? 0 : -1; // check for optional "elseif" condition

  if (split != -1) {
    // Found "elseif" condition
    isCommand = false;

    if (ifBlock && !fakeIfBlock) {
      if (ifBranche[ifBlock - 1]) {
        if (condition[ifBlock - 1]) {
          ifBranche[ifBlock - 1] = false;
        }
        else {
          String check = lcAction.substring(split + 7);
          check.trim();
          condition[ifBlock - 1] = conditionMatchExtended(check);
#ifndef BUILD_NO_DEBUG

          if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
            String log  = F("Lev.");
            log += String(ifBlock);
            log += F(": [elseif ");
            log += check;
            log += F("]=");
            log += boolToString(condition[ifBlock - 1]);
            addLogMove(LOG_LEVEL_DEBUG, log);
          }
#endif // ifndef BUILD_NO_DEBUG
        }
      }
    }
  } else {
    // check for optional "if" condition
    split = lcAction.startsWith(F("if ")) ? 0 : -1;

    if (split != -1) {
      if (ifBlock < RULES_IF_MAX_NESTING_LEVEL) {
        if (isCommand) {
          ifBlock++;
          String check = lcAction.substring(split + 3);
          check.trim();
          condition[ifBlock - 1] = conditionMatchExtended(check);
          ifBranche[ifBlock - 1] = true;
#ifndef BUILD_NO_DEBUG

          if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
            String log  = F("Lev.");
            log += String(ifBlock);
            log += F(": [if ");
            log += check;
            log += F("]=");
            log += boolToString(condition[ifBlock - 1]);
            addLogMove(LOG_LEVEL_DEBUG, log);
          }
#endif // ifndef BUILD_NO_DEBUG
        } else {
          fakeIfBlock++;
        }
      } else {
        fakeIfBlock++;

        if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
          String log  = F("Lev.");
          log += String(ifBlock);
          log += F(": Error: IF Nesting level exceeded!");
          addLogMove(LOG_LEVEL_ERROR, log);
        }
      }
      isCommand = false;
    }
  }

  if ((equals(lcAction, F("else"))) && !fakeIfBlock) // in case of an "else" block of
                                               // actions, set ifBranche to
                                               // false
  {
    ifBranche[ifBlock - 1] = false;
    isCommand              = false;
#ifndef BUILD_NO_DEBUG

    if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
      String log  = F("Lev.");
      log += String(ifBlock);
      log += F(": [else]=");
      log += boolToString(condition[ifBlock - 1] == ifBranche[ifBlock - 1]);
      addLogMove(LOG_LEVEL_DEBUG, log);
    }
#endif // ifndef BUILD_NO_DEBUG
  }

  if (equals(lcAction, F("endif"))) // conditional block ends here
  {
    if (fakeIfBlock) {
      fakeIfBlock--;
    }
    else if (ifBlock) {
      ifBlock--;
    }
    isCommand = false;
  }

  // process the action if it's a command and unconditional, or conditional and
  // the condition matches the if or else block.
  if (isCommand) {
    substitute_eventvalue(action, event);

    const bool executeRestricted = equals(parseString(action, 1), F("restrict"));

    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      String actionlog = executeRestricted ? F("ACT  : (restricted) ") : F("ACT  : ");
      actionlog += action;
      addLogMove(LOG_LEVEL_INFO, actionlog);
    }

    if (executeRestricted) {
      ExecuteCommand_all({EventValueSource::Enum::VALUE_SOURCE_RULES_RESTRICTED, parseStringToEndKeepCase(action, 2)});
    } else {
      // Use action.c_str() here as we need to preserve the action string.
      ExecuteCommand_all({EventValueSource::Enum::VALUE_SOURCE_RULES, action.c_str()});
    }
    delay(0);
  }
}

/********************************************************************************************\
   Check if an event matches to a given rule
 \*********************************************************************************************/


/********************************************************************************************\
   Check expression
 \*********************************************************************************************/
bool conditionMatchExtended(String& check) {
  int  condAnd   = -1;
  int  condOr    = -1;
  bool rightcond = false;
  bool leftcond  = conditionMatch(check); // initial check

  #ifndef BUILD_NO_DEBUG
  String debugstr;

  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    debugstr += boolToString(leftcond);
  }
  #endif // ifndef BUILD_NO_DEBUG

  do {
    #ifndef BUILD_NO_DEBUG

    if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
      String log = F("conditionMatchExtended: ");
      log += debugstr;
      log += ' ';
      log += wrap_String(check, '"');
      addLogMove(LOG_LEVEL_DEBUG, log);
    }
    #endif // ifndef BUILD_NO_DEBUG
    condAnd = check.indexOf(F(" and "));
    condOr  = check.indexOf(F(" or "));

    if ((condAnd > 0) || (condOr > 0)) {                             // we got AND/OR
      if ((condAnd > 0) && (((condOr < 0) /*&& (condOr < condAnd)*/) ||
                            ((condOr > 0) && (condOr > condAnd)))) { // AND is first
        check     = check.substring(condAnd + 5);
        rightcond = conditionMatch(check);
        leftcond  = (leftcond && rightcond);

        #ifndef BUILD_NO_DEBUG

        if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
          debugstr += F(" && ");
        }
        #endif // ifndef BUILD_NO_DEBUG
      } else { // OR is first
        check     = check.substring(condOr + 4);
        rightcond = conditionMatch(check);
        leftcond  = (leftcond || rightcond);

        #ifndef BUILD_NO_DEBUG

        if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
          debugstr += F(" || ");
        }
        #endif // ifndef BUILD_NO_DEBUG
      }

      #ifndef BUILD_NO_DEBUG

      if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
        debugstr += boolToString(rightcond);
      }
      #endif // ifndef BUILD_NO_DEBUG
    }
  } while (condAnd > 0 || condOr > 0);

  #ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    check = debugstr;
  }
  #endif // ifndef BUILD_NO_DEBUG
  return leftcond;
}


void logtimeStringToSeconds(const String& tBuf, int hours, int minutes, int seconds, bool valid)
{
  #ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    String log;
    log  = F("timeStringToSeconds: ");
    log += wrap_String(tBuf, '"');
    log += F(" --> ");
    if (valid) {
      log += formatIntLeadingZeroes(hours, 2);
      log += ':';
      log += formatIntLeadingZeroes(minutes, 2);
      log += ':';
      log += formatIntLeadingZeroes(seconds, 2);
    } else {
      log += F("invalid");
    }
    addLogMove(LOG_LEVEL_DEBUG, log);
  }

  #endif // ifndef BUILD_NO_DEBUG
}

// convert old and new time string to nr of seconds
// return whether it should be considered a time string.
bool timeStringToSeconds(const String& tBuf, int& time_seconds, String& timeString) {
  {
    // Make sure we only check for expected characters
    // e.g. if 10:11:12 > 7:07 and 10:11:12 < 20:09:04
    // Should only try to match "7:07", not "7:07 and 10:11:12" 
    // Or else it will find "7:07:11"
    bool done = false;
    for (uint8_t pos = 0; !done && timeString.length() < 8 && pos < tBuf.length(); ++pos) {
      char c = tBuf[pos];
      if (isdigit(c) || c == ':') {
        timeString += c;
      } else {
        done = true;
      }
    }
  }

  time_seconds = -1;
  int32_t hours   = 0;
  int32_t minutes = 0;
  int32_t seconds = 0;

  int tmpIndex = 0;
  String hours_str, minutes_str, seconds_str;
  bool   validTime = false;

  if (get_next_argument(timeString, tmpIndex, hours_str, ':')) {
    if (validIntFromString(hours_str, hours)) {
      validTime = true;

      if ((hours < 0) || (hours > 24)) {
        validTime = false;
      } else {
        time_seconds = hours * 60 * 60;
      }

      if (validTime && get_next_argument(timeString, tmpIndex, minutes_str, ':')) {
        if (validIntFromString(minutes_str, minutes)) {
          if ((minutes < 0) || (minutes > 59)) {
            validTime = false;
          } else {
            time_seconds += minutes * 60;
          }

          if (validTime && get_next_argument(timeString, tmpIndex, seconds_str, ':')) {
            // New format, only HH:MM:SS
            if (validIntFromString(seconds_str, seconds)) {
              if ((seconds < 0) || (seconds > 59)) {
                validTime = false;
              } else {
                time_seconds += seconds;
              }
            }
          } else {
            // Old format, only HH:MM
          }
        }
      } else {
        // It is a valid time string, but could also be just a numerical.
        // We mark it here as invalid, meaning the 'other' time to compare it to must contain more than just the hour.
        validTime = false;
      }
    }
  }
  logtimeStringToSeconds(timeString, hours, minutes, seconds, validTime);
  return validTime;
}

// Balance the count of parentheses (aka round braces) by adding the missing left or right parentheses, if any
// Returns the number of added parentheses, < 0 is left parentheses added, > 0 is right parentheses added
int balanceParentheses(String& string) {
  int left = 0;
  int right = 0;
  for (unsigned int i = 0; i < string.length(); i++) {
    if (string[i] == '(') {
      left++;
    } else if (string[i] == ')') {
      right++;
    }
  }
  if (left != right) {
    string.reserve(string.length() + abs(right - left)); // Re-allocate max. once
  }
  if (left > right) {
    for (int i = 0; i < left - right; i++) {
      string += ')';
    }
  } else if (right > left) {
    for (int i = 0; i < right - left; i++) {
      string = String('(') + string; // This is quite 'expensive'
    }
  }
  return left - right;
}

bool conditionMatch(const String& check) {
  int  posStart, posEnd;
  char compare;

  if (!findCompareCondition(check, compare, posStart, posEnd)) {
    return false;
  }

  String tmpCheck1 = check.substring(0, posStart);
  String tmpCheck2 = check.substring(posEnd);

  tmpCheck1.trim();
  tmpCheck2.trim();
  ESPEASY_RULES_FLOAT_TYPE Value1{};
  ESPEASY_RULES_FLOAT_TYPE Value2{};

  int  timeInSec1 = 0;
  int  timeInSec2 = 0;
  String timeString1, timeString2;
  bool validTime1 = timeStringToSeconds(tmpCheck1, timeInSec1, timeString1);
  bool validTime2 = timeStringToSeconds(tmpCheck2, timeInSec2, timeString2);
  bool result     = false;

  bool compareTimes = false;

  if ((validTime1 || validTime2) && (timeInSec1 != -1) && (timeInSec2 != -1))
  {
    // At least one is a time containing ':' separator
    // AND both can be considered a time, so use it as a time and compare seconds.
    compareTimes = true;
    result       = compareIntValues(compare, timeInSec1, timeInSec2);
    tmpCheck1    = timeString1;
    tmpCheck2    = timeString2;
  } else {
    int condAnd = tmpCheck2.indexOf(F(" and "));
    int condOr  = tmpCheck2.indexOf(F(" or "));
    if (condAnd > -1 || condOr > -1) {            // Only parse first condition, rest will be parsed 'later'
      if (condAnd > -1 && (condOr == -1 || condAnd < condOr)) {
        tmpCheck2 = tmpCheck2.substring(0, condAnd);
      } else if (condOr > -1) {
        tmpCheck2 = tmpCheck2.substring(0, condOr);
      }
      tmpCheck2.trim();
    }
    balanceParentheses(tmpCheck1);
    balanceParentheses(tmpCheck2);
    if (isError(Calculate(tmpCheck1, Value1
                          #if FEATURE_STRING_VARIABLES
                          , false // suppress logging specific errors when parsing strings
                          #endif // if FEATURE_STRING_VARIABLES
                         )) ||
        isError(Calculate(tmpCheck2, Value2
                          #if FEATURE_STRING_VARIABLES
                          , false // suppress logging specific errors when parsing strings
                          #endif // if FEATURE_STRING_VARIABLES
                         )))
    {
      #if FEATURE_STRING_VARIABLES
      result = compareStringValues(compare, tmpCheck1, tmpCheck2);
      #else // if FEATURE_STRING_VARIABLES
      return false;
      #endif // if FEATURE_STRING_VARIABLES
    }
    else {
      result = compareDoubleValues(compare, Value1, Value2);
    }
  }

  #ifndef BUILD_NO_DEBUG

  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    String log = F("conditionMatch: ");
    log += wrap_String(check, '"');
    log += F(" --> ");

    log += wrap_String(tmpCheck1, '"');
    log += wrap_String(check.substring(posStart, posEnd), ' '); // Compare
    log += wrap_String(tmpCheck2, '"');

    log += F(" --> ");
    log += boolToString(result);
    log += ' ';

    log += '(';
    const bool trimTrailingZeros = true;
    log += compareTimes ? String(timeInSec1) : 
#if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
    doubleToString(Value1, 6, trimTrailingZeros);
#else
    floatToString(Value1, 6, trimTrailingZeros);
#endif
    log += wrap_String(check.substring(posStart, posEnd), ' '); // Compare
    log += compareTimes ? String(timeInSec2) : 
#if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
    doubleToString(Value2, 6, trimTrailingZeros);
#else
    floatToString(Value2, 6, trimTrailingZeros);
#endif    
    log += ')';
    addLogMove(LOG_LEVEL_DEBUG, log);
  }
  #else // ifndef BUILD_NO_DEBUG
  (void)compareTimes; // To avoid compiler warning
  #endif // ifndef BUILD_NO_DEBUG
  return result;
}

/********************************************************************************************\
   Generate rule events based on task refresh
 \*********************************************************************************************/
void createRuleEvents(struct EventStruct *event) {
  if (!Settings.UseRules) {
    return;
  }
  const deviceIndex_t DeviceIndex = getDeviceIndex_from_TaskIndex(event->TaskIndex);

  if (!validDeviceIndex(DeviceIndex)) { return; }

  const uint8_t valueCount = getValueCountForTask(event->TaskIndex);
  String taskName = getTaskDeviceName(event->TaskIndex);
  #if FEATURE_STRING_VARIABLES
  String postfix;
  const String search = getDerivedValueSearchAndPostfix(taskName, postfix);
  #endif // if FEATURE_STRING_VARIABLES

  // Small optimization as sensor type string may result in large strings
  // These also only yield a single value, so no need to check for combining task values.
  if (event->getSensorType() == Sensor_VType::SENSOR_TYPE_STRING) {
    size_t expectedSize = 2 + getTaskDeviceName(event->TaskIndex).length();
    expectedSize += Cache.getTaskDeviceValueName(event->TaskIndex, 0).length();
   
    bool appendCompleteStringvalue = false;

    String eventString;
    if (reserve_special(eventString, expectedSize + event->String2.length())) {
      appendCompleteStringvalue = true;
    } else if (!reserve_special(eventString, expectedSize + 24)) {
      // No need to continue as we can't even allocate the event, we probably also cannot process it
      addLog(LOG_LEVEL_ERROR, F("Not enough memory for event"));
      return;
    }
    eventString += taskName;
    eventString += '#';
    eventString += Cache.getTaskDeviceValueName(event->TaskIndex, 0);
    eventString += '=';
    eventString += '`';
    if (appendCompleteStringvalue) {
      eventString += event->String2;
    } else {
      eventString += event->String2.substring(0, 10);
      eventString += F("...");
      eventString += event->String2.substring(event->String2.length() - 10);
    }
    eventString += '`';
    eventQueue.addMove(std::move(eventString));    
  } else if (Settings.CombineTaskValues_SingleEvent(event->TaskIndex)) {
    String eventvalues;
    reserve_special(eventvalues, 32); // Enough for most use cases, prevent lots of memory allocations.

    uint8_t varNr = 0;
    for (; varNr < valueCount; ++varNr) {
      if (varNr != 0) {
        eventvalues += ',';
      }
      eventvalues += formatUserVarNoCheck(event, varNr);
    }
    #if FEATURE_STRING_VARIABLES
    if (Settings.EventAndLogDerivedTaskValues(event->TaskIndex)) {

      auto it = customStringVar.begin();
      while (it != customStringVar.end()) {
        if (it->first.startsWith(search) && it->first.endsWith(postfix)) {
          if (!it->second.isEmpty()) {
            String value(it->second);
            value = parseTemplateAndCalculate(value);
            if (varNr != 0) {
              eventvalues += ',';
            }
            eventvalues += value;
            ++varNr;
          }
        }
        else if (it->first.substring(0, search.length()).compareTo(search) > 0) {
          break;
        }
        ++it;
      }
    }
    #endif // if FEATURE_STRING_VARIABLES
    eventQueue.add(event->TaskIndex, F("All"), eventvalues);
  } else {
    for (uint8_t varNr = 0; varNr < valueCount; varNr++) {
      eventQueue.add(event->TaskIndex, Cache.getTaskDeviceValueName(event->TaskIndex, varNr), formatUserVarNoCheck(event, varNr));
    }
    #if FEATURE_STRING_VARIABLES
    if (Settings.EventAndLogDerivedTaskValues(event->TaskIndex)) {
      taskName.toLowerCase();

      auto it = customStringVar.begin();
      while (it != customStringVar.end()) {
        if (it->first.startsWith(search) && it->first.endsWith(postfix)) {
          String valueName = it->first.substring(search.length(), it->first.indexOf('-'));
          const String vname2 = getDerivedValueName(taskName, valueName);
          if (!vname2.isEmpty()) {
            valueName = vname2;
          }
          if (!it->second.isEmpty()) {
            String value(it->second);
            value = parseTemplateAndCalculate(value);
            eventQueue.add(event->TaskIndex, valueName, value);
          }
        }
        else if (it->first.substring(0, search.length()).compareTo(search) > 0) {
          break;
        }
        ++it;
      }
    }
    #endif // if FEATURE_STRING_VARIABLES
  }
}

#include "../ESPEasyCore/ESPEasyWifi.h"

#include "../../ESPEasy-Globals.h"
#include "../DataStructs/TimingStats.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/ESPEasyWiFiEvent.h"
#include "../ESPEasyCore/ESPEasyWifi_ProcessEvent.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/Serial.h"
#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/EventQueue.h"
#include "../Globals/NetworkState.h"
#include "../Globals/Nodes.h"
#include "../Globals/RTC.h"
#include "../Globals/SecuritySettings.h"
#include "../Globals/Services.h"
#include "../Globals/Settings.h"
#include "../Globals/WiFi_AP_Candidates.h"
#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/Hardware_defines.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Networking.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringGenerator_WiFi.h"
#include "../Helpers/StringProvider.h"

#ifdef ESP32
#include <WiFiGeneric.h>
#include <esp_wifi.h> // Needed to call ESP-IDF functions like esp_wifi_....

#include <esp_phy_init.h>
#endif

// FIXME TD-er: Cleanup of WiFi code
#ifdef ESPEASY_WIFI_CLEANUP_WORK_IN_PROGRESS
bool ESPEasyWiFi_t::begin() {
  return true;
}

void ESPEasyWiFi_t::end() {


}


void ESPEasyWiFi_t::loop() {
  switch (_state) {
    case WiFiState_e::OFF:
    break;
    case WiFiState_e::AP_only:
    break;
    case WiFiState_e::ErrorRecovery:
    // Wait for timeout to expire
    // Start again from scratch
    break;
    case WiFiState_e::STA_Scanning:
    case WiFiState_e::STA_AP_Scanning:
    // Check if scanning is finished
    // When scanning per channel, call for scanning next channel
    break;
    case WiFiState_e::STA_Connecting:
    case WiFiState_e::STA_Reconnecting:
    // Check if (re)connecting has finished
    break;
    case WiFiState_e::STA_Connected:
    // Check if still connected
    // Reconnect if not.
    // Else mark last timestamp seen as connected
    break;
  }


  {
    // Check if we need to start AP
    // Flag captive portal in webserver and/or whether we might be in setup mode
  }

#ifdef USE_IMPROV
  {
    // Check for Improv mode.
  }
#endif


}


IPAddress  ESPEasyWiFi_t::getIP() const {

  IPAddress res;


  return res;
}

void  ESPEasyWiFi_t::disconnect() {

}


void ESPEasyWiFi_t::checkConnectProgress() {

}

void ESPEasyWiFi_t::startScanning() {
  _state = WiFiState_e::STA_Scanning;
  WifiScan(true);
  _last_state_change.setNow();
}


bool ESPEasyWiFi_t::connectSTA() {
  if (!WiFi_AP_Candidates.hasCandidateCredentials()) {
    if (!WiFiEventData.warnedNoValidWiFiSettings) {
      addLog(LOG_LEVEL_ERROR, F("WIFI : No valid wifi settings"));
      WiFiEventData.warnedNoValidWiFiSettings = true;
    }
    WiFiEventData.last_wifi_connect_attempt_moment.clear();
    WiFiEventData.wifi_connect_attempt     = 1;
    WiFiEventData.wifiConnectAttemptNeeded = false;

    // No need to wait longer to start AP mode.
    if (!Settings.DoNotStartAP()) {
      setAP(true);
    }
    return false;
  }
  WiFiEventData.warnedNoValidWiFiSettings = false;
  setSTA(true);
  #if defined(ESP8266)
  wifi_station_set_hostname(NetworkCreateRFCCompliantHostname().c_str());

  #endif // if defined(ESP8266)
  #if defined(ESP32)
  WiFi.config(INADDR_NONE, INADDR_NONE, INADDR_NONE);
  #endif // if defined(ESP32)
  setConnectionSpeed();
  setupStaticIPconfig();



    // Start the process of connecting or starting AP
    if (WiFi_AP_Candidates.getNext(true)) {
      // Try to connect to AP

    } else {
      // No (known) AP, start scanning
      startScanning();
    }


  return true;
}

#endif // ESPEASY_WIFI_CLEANUP_WORK_IN_PROGRESS


// ********************************************************************************
// WiFi state
// ********************************************************************************

/*
   WiFi STA states:
   1 STA off                 => ESPEASY_WIFI_DISCONNECTED
   2 STA connecting
   3 STA connected           => ESPEASY_WIFI_CONNECTED
   4 STA got IP              => ESPEASY_WIFI_GOT_IP
   5 STA connected && got IP => ESPEASY_WIFI_SERVICES_INITIALIZED

   N.B. the states are flags, meaning both "connected" and "got IP" must be set
        to be considered ESPEASY_WIFI_SERVICES_INITIALIZED

   The flag wifiConnectAttemptNeeded indicates whether a new connect attempt is needed.
   This is set to true when:
   - Security settings have been saved with AP mode enabled. FIXME TD-er, this may not be the best check.
   - WiFi connect timeout reached  &  No client is connected to the AP mode of the node.
   - Wifi is reset
   - WiFi setup page has been loaded with SSID/pass values.


   WiFi AP mode states:
   1 AP on                        => reset AP disable timer
   2 AP client connect/disconnect => reset AP disable timer
   3 AP off                       => AP disable timer = 0;

   AP mode will be disabled when both apply:
   - AP disable timer (timerAPoff) expired
   - No client is connected to the AP.

   AP mode will be enabled when at least one applies:
   - No valid WiFi settings
   - Start AP timer (timerAPstart) expired

   Start AP timer is set or cleared at:
   - Set timerAPstart when "valid WiFi connection" state is observed.
   - Disable timerAPstart when ESPEASY_WIFI_SERVICES_INITIALIZED wifi state is reached.

   For the first attempt to connect after a cold boot (RTC values are 0), a WiFi scan will be 
   performed to find the strongest known SSID.
   This will set RTC.lastBSSID and RTC.lastWiFiChannel
   
   Quick reconnect (using BSSID/channel of last connection) when both apply:
   - If wifi_connect_attempt < 3
   - RTC.lastBSSID is known
   - RTC.lastWiFiChannel != 0

   Change of wifi settings when both apply:
   - "other" settings valid
   - (wifi_connect_attempt % 2) == 0

   Reset of wifi_connect_attempt to 0 when both apply:
   - connection successful
   - Connection stable (connected for > 5 minutes)

 */


// ********************************************************************************
// Check WiFi connected status
// This is basically the state machine to switch between states:
// - Initiate WiFi reconnect
// - Start/stop of AP mode
// ********************************************************************************
bool WiFiConnected() {
  START_TIMER;

  static bool recursiveCall = false;

  static uint32_t lastCheckedTime = 0;
  static bool lastState = false;

#if FEATURE_USE_IPV6
  if (!WiFiEventData.processedGotIP6) {
    processGotIPv6();
  }
#endif

  if (!WifiIsSTA(WiFi.getMode())) {
    lastState = false;
    return lastState;
  }


  const int32_t timePassed = timePassedSince(lastCheckedTime);
  if (lastCheckedTime != 0) {
    if (timePassed < 100) {
      if (WiFiEventData.lastDisconnectMoment.isSet() &&
          WiFiEventData.lastDisconnectMoment.millisPassedSince() > timePassed)
      {
        // Try to rate-limit the nr of calls to this function or else it will be called 1000's of times a second.
        return lastState;
      }
    }
    if (timePassed < 10) {
      // Rate limit time spent in WiFiConnected() to max. 100x per sec to process the rest of this function
      return lastState;
    }
  }



  if (WiFiEventData.unprocessedWifiEvents()) { return false; }

  bool wifi_isconnected = WiFi.isConnected();
  #ifdef ESP8266
  // Perform check on SDK function, see: https://github.com/esp8266/Arduino/issues/7432
  station_status_t status = wifi_station_get_connect_status();
  switch(status) {
    case STATION_GOT_IP:
      wifi_isconnected = true;
      break;
    case STATION_NO_AP_FOUND:
    case STATION_CONNECT_FAIL:
    case STATION_WRONG_PASSWORD:
      wifi_isconnected = false;
      break;
    case STATION_IDLE:
    case STATION_CONNECTING:
      break;

    default:
      wifi_isconnected = false;
      break;
  }
  #endif

  if (recursiveCall) return wifi_isconnected;
  recursiveCall = true;


  // For ESP82xx, do not rely on WiFi.status() with event based wifi.
  const int32_t wifi_rssi = WiFi.RSSI();
  bool validWiFi = (wifi_rssi < 0) && wifi_isconnected && hasIPaddr();
  /*
  if (validWiFi && WiFi.channel() != WiFiEventData.usedChannel) {
    validWiFi = false;
  }
  */
  if (validWiFi != WiFiEventData.WiFiServicesInitialized()) {
    // else wifiStatus is no longer in sync.
    if (checkAndResetWiFi()) {
      // Wifi has been reset, so no longer valid WiFi
      validWiFi = false;
    }
  }

  if (validWiFi) {
    // Connected, thus disable any timer to start AP mode. (except when in WiFi setup mode)
    if (!WiFiEventData.wifiSetupConnect) {
      WiFiEventData.timerAPstart.clear();
    }
    STOP_TIMER(WIFI_ISCONNECTED_STATS);
    recursiveCall = false;
    // Only return true after some time since it got connected.
#if FEATURE_SET_WIFI_TX_PWR
    SetWiFiTXpower();
#endif
    lastState = WiFiEventData.wifi_considered_stable || WiFiEventData.lastConnectMoment.timeoutReached(100);
    lastCheckedTime = millis();
    return lastState;
  }

  if ((WiFiEventData.timerAPstart.isSet()) && WiFiEventData.timerAPstart.timeReached()) {
    if (WiFiEventData.timerAPoff.isSet() && !WiFiEventData.timerAPoff.timeReached()) {
      if (!Settings.DoNotStartAP()) {
        // Timer reached, so enable AP mode.
        if (!WifiIsAP(WiFi.getMode())) {
          if (!WiFiEventData.wifiConnectAttemptNeeded) {
            #ifndef BUILD_MINIMAL_OTA
            addLog(LOG_LEVEL_INFO, F("WiFi : WiFiConnected(), start AP"));
            #endif
            WifiScan(false);
            setSTA(false); // Force reset WiFi + reduce power consumption
            setAP(true);
          }
        }
      }
    } else {
      WiFiEventData.timerAPstart.clear();
      WiFiEventData.timerAPoff.clear();
    }
  }


  // When made this far in the code, we apparently do not have valid WiFi connection.
  if (!WiFiEventData.timerAPstart.isSet() && !WifiIsAP(WiFi.getMode())) {
    // First run we do not have WiFi connection any more, set timer to start AP mode
    // Only allow the automatic AP mode in the first N minutes after boot.
    if (getUptimeMinutes() < WIFI_ALLOW_AP_AFTERBOOT_PERIOD) {
      WiFiEventData.timerAPstart.setMillisFromNow(WIFI_RECONNECT_WAIT);
      // Fixme TD-er: Make this more elegant as it now needs to know about the extra time needed for the AP start timer.
      WiFiEventData.timerAPoff.setMillisFromNow(WIFI_RECONNECT_WAIT + WIFI_AP_OFF_TIMER_DURATION);
    }
  }

  const bool timeoutReached = WiFiEventData.last_wifi_connect_attempt_moment.isSet() && 
                              WiFiEventData.last_wifi_connect_attempt_moment.timeoutReached(2 * DEFAULT_WIFI_CONNECTION_TIMEOUT);

  if (timeoutReached && !WiFiEventData.wifiSetup) {
    // It took too long to make a connection, set flag we need to try again
    //if (!wifiAPmodeActivelyUsed()) {
      WiFiEventData.wifiConnectAttemptNeeded = true;
    //}
    WiFiEventData.wifiConnectInProgress = false;
    if (!WiFiEventData.WiFiDisconnected()) {
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_INFO, F("WiFi : wifiConnectTimeoutReached"));
      #endif
      WifiDisconnect();
    }
  }
  delay(0);
  STOP_TIMER(WIFI_NOTCONNECTED_STATS);
  recursiveCall = false;
  return false;
}

void WiFiConnectRelaxed() {
  if (!WiFiEventData.processedDisconnect) {
    processDisconnect();
  }
  if (!WiFiEventData.WiFiConnectAllowed() || WiFiEventData.wifiConnectInProgress) {
    if (WiFiEventData.wifiConnectInProgress) {
      if (WiFiEventData.last_wifi_connect_attempt_moment.isSet()) { 
        if (WiFiEventData.last_wifi_connect_attempt_moment.timeoutReached(WIFI_PROCESS_EVENTS_TIMEOUT)) {
          WiFiEventData.wifiConnectInProgress = false;
        }
      }
    }

    if (WiFiEventData.wifiConnectInProgress) {
      return; // already connected or connect attempt in progress need to disconnect first
    }
  }
  if (!WiFiEventData.processedScanDone) {
    // Scan is still active, so do not yet connect.
    return;
  }

  if (WiFiEventData.unprocessedWifiEvents()) {
    # ifndef BUILD_NO_DEBUG
    if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
      String log = F("WiFi : Connecting not possible, unprocessed WiFi events: ");
      if (!WiFiEventData.processedConnect) {
        log += F(" conn");
      }
      if (!WiFiEventData.processedDisconnect) {
        log += F(" disconn");
      }
      if (!WiFiEventData.processedGotIP) {
        log += F(" gotIP");
      }
#if FEATURE_USE_IPV6
      if (!WiFiEventData.processedGotIP6) {
        log += F(" gotIP6");
      }
#endif

      if (!WiFiEventData.processedDHCPTimeout) {
        log += F(" DHCP_t/o");
      }
      
      addLogMove(LOG_LEVEL_ERROR, log);
      logConnectionStatus();
    }
    #endif
    return;
  }

  if (!WiFiEventData.wifiSetupConnect && wifiAPmodeActivelyUsed()) {
    return;
  }


  // FIXME TD-er: Should not try to prepare when a scan is still busy.
  // This is a logic error which may lead to strange issues if some kind of timeout happens and/or RF calibration was not OK.
  // Split this function into separate parts, with the last part being the actual connect attempt either after a scan is complete or quick connect is possible.

  AttemptWiFiConnect();
}

void AttemptWiFiConnect() {
  if (!WiFiEventData.wifiConnectAttemptNeeded) {
    return;
  }

  if (WiFiEventData.wifiConnectInProgress) {
    return;
  }

  setNetworkMedium(NetworkMedium_t::WIFI);
  if (active_network_medium != NetworkMedium_t::WIFI) 
  {
    return;
  }


  if (WiFiEventData.wifiSetupConnect) {
    // wifiSetupConnect is when run from the setup page.
    RTC.clearLastWiFi(); // Force slow connect
    WiFiEventData.wifi_connect_attempt = 0;
    WiFiEventData.wifiSetupConnect     = false;
    if (WiFiEventData.timerAPoff.isSet()) {
      WiFiEventData.timerAPoff.setMillisFromNow(WIFI_RECONNECT_WAIT + WIFI_AP_OFF_TIMER_DURATION);
    }
  }

  if (WiFiEventData.last_wifi_connect_attempt_moment.isSet()) {
    if (!WiFiEventData.last_wifi_connect_attempt_moment.timeoutReached(DEFAULT_WIFI_CONNECTION_TIMEOUT)) {
      return;
    }
  }

  if (WiFiEventData.unprocessedWifiEvents()) {
    return;
  }
  setSTA(false);

  setSTA(true);

  if (WiFi_AP_Candidates.getNext(WiFiScanAllowed())) {
    const WiFi_AP_Candidate candidate = WiFi_AP_Candidates.getCurrent();

    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLogMove(LOG_LEVEL_INFO, strformat(
        F("WIFI : Connecting %s attempt #%u"),
        candidate.toString().c_str(),
        WiFiEventData.wifi_connect_attempt));
    }
    WiFiEventData.markWiFiBegin();
    if (prepareWiFi()) {
      setNetworkMedium(NetworkMedium_t::WIFI);
      RTC.clearLastWiFi();
      RTC.lastWiFiSettingsIndex = candidate.index;
      
      float tx_pwr = 0; // Will be set higher based on RSSI when needed.
      // FIXME TD-er: Must check WiFiEventData.wifi_connect_attempt to increase TX power
#if FEATURE_SET_WIFI_TX_PWR
      if (Settings.UseMaxTXpowerForSending()) {
        tx_pwr = Settings.getWiFi_TX_power();
      }
      SetWiFiTXpower(tx_pwr, candidate.rssi);
#endif
      // Start connect attempt now, so no longer needed to attempt new connection.
      WiFiEventData.wifiConnectAttemptNeeded = false;
      WiFiEventData.wifiConnectInProgress = true;
      const String key = WiFi_AP_CandidatesList::get_key(candidate.index);

#if FEATURE_USE_IPV6
      if (Settings.EnableIPv6()) {
        WiFi.enableIPv6(true);
      }
#endif

#ifdef ESP32
      if (Settings.IncludeHiddenSSID()) {
        wifi_country_t config = {
          .cc = "01",
          .schan = 1,
          .nchan = 14,
          .policy = WIFI_COUNTRY_POLICY_MANUAL,
        };
        esp_wifi_set_country(&config);
      }
#endif


      if ((Settings.HiddenSSID_SlowConnectPerBSSID() || !candidate.bits.isHidden)
           && candidate.allowQuickConnect()) {
        WiFi.begin(candidate.ssid.c_str(), key.c_str(), candidate.channel, candidate.bssid.mac);
      } else {
        WiFi.begin(candidate.ssid.c_str(), key.c_str());
      }
#ifdef ESP32
  // Always wait for a second on ESP32
      WiFi.waitForConnectResult(1000);  // https://github.com/arendst/Tasmota/issues/14985
#else
      if (Settings.WaitWiFiConnect() || candidate.bits.isHidden) {
//        WiFi.waitForConnectResult(candidate.isHidden ? 3000 : 1000);  // https://github.com/arendst/Tasmota/issues/14985
        WiFi.waitForConnectResult(1000);  // https://github.com/arendst/Tasmota/issues/14985
      }
#endif
      delay(1);
    } else {
      WiFiEventData.wifiConnectInProgress = false;
    }
  } else {
    if (!wifiAPmodeActivelyUsed() || WiFiEventData.wifiSetupConnect) {
      if (!prepareWiFi()) {
        //return;
      }

      if (WiFiScanAllowed()) {
        // Maybe not scan async to give the ESP some slack in power consumption?
        const bool async = false;
        WifiScan(async);
      }
      // Limit nr of attempts as we don't have any AP candidates.
      WiFiEventData.last_wifi_connect_attempt_moment.setMillisFromNow(60000);
      WiFiEventData.timerAPstart.setNow();
    }
  }

  logConnectionStatus();
}

// ********************************************************************************
// Set Wifi config
// ********************************************************************************
bool prepareWiFi() {
  #if defined(ESP32)
  registerWiFiEventHandler();
  #endif

  if (!WiFi_AP_Candidates.hasCandidateCredentials()) {
    if (!WiFiEventData.warnedNoValidWiFiSettings) {
      addLog(LOG_LEVEL_ERROR, F("WIFI : No valid wifi settings"));
      WiFiEventData.warnedNoValidWiFiSettings = true;
    }
//    WiFiEventData.last_wifi_connect_attempt_moment.clear();
    WiFiEventData.wifi_connect_attempt     = 1;
    WiFiEventData.wifiConnectAttemptNeeded = false;

    // No need to wait longer to start AP mode.
    if (!Settings.DoNotStartAP()) {
      WifiScan(false);
//      setAP(true);
    }
    return false;
  }
  WiFiEventData.warnedNoValidWiFiSettings = false;
  setSTA(true);

  #if defined(ESP8266)
  wifi_station_set_hostname(NetworkCreateRFCCompliantHostname().c_str());

  #endif // if defined(ESP8266)
  #if defined(ESP32)
  WiFi.config(INADDR_NONE, INADDR_NONE, INADDR_NONE);
  #endif // if defined(ESP32)
  setConnectionSpeed();
  setupStaticIPconfig();
  WiFiEventData.wifiConnectAttemptNeeded = true;

  return true;
}

bool checkAndResetWiFi() {
  #ifdef ESP8266
  station_status_t status = wifi_station_get_connect_status();

  switch(status) {
    case STATION_GOT_IP:
      if (WiFi.RSSI() < 0 && WiFi.localIP().isSet()) {
        //if (WiFi.channel() == WiFiEventData.usedChannel || WiFiEventData.usedChannel == 0) {
          // This is a valid status, no need to reset
          return false;
        //}
      }
      break;
    case STATION_NO_AP_FOUND:
    case STATION_CONNECT_FAIL:
    case STATION_WRONG_PASSWORD:
      // Reason to reset WiFi
      break;
    case STATION_IDLE:
    case STATION_CONNECTING:
      if (WiFiEventData.last_wifi_connect_attempt_moment.isSet() && !WiFiEventData.last_wifi_connect_attempt_moment.timeoutReached(DEFAULT_WIFI_CONNECTION_TIMEOUT)) {
        return false;
      }
      break;
  }
  #endif
  #ifdef ESP32
  if (WiFi.isConnected()) {
    //if (WiFi.channel() == WiFiEventData.usedChannel || WiFiEventData.usedChannel == 0) {
      return false;
    //}
  }
  if (WiFiEventData.last_wifi_connect_attempt_moment.isSet() && !WiFiEventData.last_wifi_connect_attempt_moment.timeoutReached(DEFAULT_WIFI_CONNECTION_TIMEOUT)) {
    return false;
  }
  #endif
  # ifndef BUILD_NO_DEBUG
  String log = F("WiFi : WiFiConnected() out of sync: ");
  log += WiFiEventData.ESPeasyWifiStatusToString();
  log += F(" RSSI: ");
  log += String(WiFi.RSSI());
  #ifdef ESP8266
  log += F(" status: ");
  log += SDKwifiStatusToString(status);
  #endif
  #endif

  // Call for reset first, to make sure a syslog call will not try to send.
  resetWiFi();
  # ifndef BUILD_NO_DEBUG
  addLogMove(LOG_LEVEL_INFO, log);
  #endif
  return true;
}


void resetWiFi() {
  //if (wifiAPmodeActivelyUsed()) return;
  if (WiFiEventData.lastWiFiResetMoment.isSet() && !WiFiEventData.lastWiFiResetMoment.timeoutReached(1000)) {
    // Don't reset WiFi too often
    return;
  }
  FeedSW_watchdog();
  WiFiEventData.clearAll();
  WifiDisconnect();

  // Send this log only after WifiDisconnect() or else sending to syslog may cause issues
  #ifndef BUILD_MINIMAL_OTA
  addLog(LOG_LEVEL_INFO, F("Reset WiFi."));
  #endif

  //  setWifiMode(WIFI_OFF);

  initWiFi();
}

#ifdef ESP32
void removeWiFiEventHandler()
{
  WiFi.removeEvent(WiFiEventData.wm_event_id);
  WiFiEventData.wm_event_id = 0;
}

void registerWiFiEventHandler()
{
  if (WiFiEventData.wm_event_id != 0) {
    removeWiFiEventHandler();
  }
  WiFiEventData.wm_event_id = WiFi.onEvent(WiFiEvent);
}
#endif


void initWiFi()
{
#ifdef ESP8266

  // See https://github.com/esp8266/Arduino/issues/5527#issuecomment-460537616
  // FIXME TD-er: Do not destruct WiFi object, it may cause crashes with queued UDP traffic.
//  WiFi.~ESP8266WiFiClass();
//  WiFi = ESP8266WiFiClass();
#endif // ifdef ESP8266
#ifdef ESP32
  removeWiFiEventHandler();
#endif


  WiFi.persistent(false); // Do not use SDK storage of SSID/WPA parameters
  // The WiFi.disconnect() ensures that the WiFi is working correctly. If this is not done before receiving WiFi connections,
  // those WiFi connections will take a long time to make or sometimes will not work at all.
  WiFi.disconnect(false);
  delay(1);
  if (active_network_medium != NetworkMedium_t::NotSet) {
    setSTA(true);
    WifiScan(false);
  }
  setWifiMode(WIFI_OFF);

#if defined(ESP32)
  registerWiFiEventHandler();
#endif
#ifdef ESP8266
  // WiFi event handlers
  static bool handlers_initialized = false;
  if (!handlers_initialized) {
    stationConnectedHandler = WiFi.onStationModeConnected(onConnected);
    stationDisconnectedHandler = WiFi.onStationModeDisconnected(onDisconnect);
    stationGotIpHandler = WiFi.onStationModeGotIP(onGotIP);
    stationModeDHCPTimeoutHandler = WiFi.onStationModeDHCPTimeout(onDHCPTimeout);
    stationModeAuthModeChangeHandler = WiFi.onStationModeAuthModeChanged(onStationModeAuthModeChanged);
    APModeStationConnectedHandler = WiFi.onSoftAPModeStationConnected(onConnectedAPmode);
    APModeStationDisconnectedHandler = WiFi.onSoftAPModeStationDisconnected(onDisconnectedAPmode);
    handlers_initialized = true;
  }
#endif
  delay(100);
}

// ********************************************************************************
// Configure WiFi TX power
// ********************************************************************************
#if FEATURE_SET_WIFI_TX_PWR
void SetWiFiTXpower() {
  SetWiFiTXpower(0); // Just some minimal value, will be adjusted in SetWiFiTXpower
}

void SetWiFiTXpower(float dBm) { 
  SetWiFiTXpower(dBm, WiFi.RSSI());
}

void SetWiFiTXpower(float dBm, float rssi) {
  const WiFiMode_t cur_mode = WiFi.getMode();
  if (cur_mode == WIFI_OFF) {
    return;
  }

  if (Settings.UseMaxTXpowerForSending()) {
    dBm = 30; // Just some max, will be limited later
  }

  // Range ESP32  : -1dBm - 20dBm
  // Range ESP8266: 0dBm - 20.5dBm
  float maxTXpwr;
  float threshold = GetRSSIthreshold(maxTXpwr);
  #ifdef ESP8266
  float minTXpwr{};
  #endif
  #ifdef ESP32
  float minTXpwr = -1.0f;
  #endif

  threshold += Settings.WiFi_sensitivity_margin; // Margin in dBm on top of threshold

  // Assume AP sends with max set by ETSI standard.
  // 2.4 GHz: 100 mWatt (20 dBm)
  // US and some other countries allow 1000 mW (30 dBm)
  // We cannot send with over 20 dBm, thus it makes no sense to force higher TX power all the time.
  const float newrssi = rssi - 20;
  if (newrssi < threshold) {
    minTXpwr = threshold - newrssi;
  }
  if (minTXpwr > maxTXpwr) {
    minTXpwr = maxTXpwr;
  }
  if (dBm > maxTXpwr) {
    dBm = maxTXpwr;
  } else if (dBm < minTXpwr) {
    dBm = minTXpwr;
  }

  #ifdef ESP32
  int8_t power = dBm * 4;
  if (esp_wifi_set_max_tx_power(power) == ESP_OK)  {
    if (esp_wifi_get_max_tx_power(&power) == ESP_OK)  {
      dBm = static_cast<float>(power) / 4.0f;
    }
  }
  #endif

  #ifdef ESP8266
  WiFi.setOutputPower(dBm);
  #endif

  if (WiFiEventData.wifi_TX_pwr < dBm) {
    // Will increase the TX power, give power supply of the unit some rest
    delay(1);
  }

  WiFiEventData.wifi_TX_pwr = dBm;

  delay(0);
  #ifndef BUILD_NO_DEBUG
  if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
    const int TX_pwr_int = WiFiEventData.wifi_TX_pwr * 4;
    const int maxTXpwr_int = maxTXpwr * 4;
    if (TX_pwr_int != maxTXpwr_int) {
      static int last_log = -1;
      if (TX_pwr_int != last_log) {
        last_log = TX_pwr_int;
        String log = strformat(
          F("WiFi : Set TX power to %ddBm sensitivity: %ddBm"),
          static_cast<int>(dBm),
          static_cast<int>(threshold));
        if (rssi < 0) {
          log += strformat(F(" RSSI: %ddBm"), static_cast<int>(rssi));
        }
        addLogMove(LOG_LEVEL_DEBUG, log);
      }
    }
  }
  #endif
}
#endif




float GetRSSIthreshold(float& maxTXpwr) {
  maxTXpwr = Settings.getWiFi_TX_power();
  float threshold = WIFI_SENSITIVITY_n;
  switch (getConnectionProtocol()) {
    case WiFiConnectionProtocol::WiFi_Protocol_11b:
      threshold = WIFI_SENSITIVITY_11b;
      if (maxTXpwr > MAX_TX_PWR_DBM_11b) maxTXpwr = MAX_TX_PWR_DBM_11b;
      break;
    case WiFiConnectionProtocol::WiFi_Protocol_11g:
      threshold = WIFI_SENSITIVITY_54g;
      if (maxTXpwr > MAX_TX_PWR_DBM_54g) maxTXpwr = MAX_TX_PWR_DBM_54g;
      break;
#ifdef ESP8266
    case WiFiConnectionProtocol::WiFi_Protocol_11n:
#else
    case WiFiConnectionProtocol::WiFi_Protocol_HT20:
    case WiFiConnectionProtocol::WiFi_Protocol_HT40:
    case WiFiConnectionProtocol::WiFi_Protocol_HE20:
#endif

      threshold = WIFI_SENSITIVITY_n;
      if (maxTXpwr > MAX_TX_PWR_DBM_n) maxTXpwr = MAX_TX_PWR_DBM_n;
      break;
#ifdef ESP32
#if ESP_IDF_VERSION > ESP_IDF_VERSION_VAL(5, 2, 0)
    case WiFiConnectionProtocol::WiFi_Protocol_11a:
    case WiFiConnectionProtocol::WiFi_Protocol_VHT20:
      // FIXME TD-er: Must determine max. TX power for these 5 GHz modi
#endif
    case WiFiConnectionProtocol::WiFi_Protocol_LR:
#endif
    case WiFiConnectionProtocol::Unknown:
      break;
  }
  return threshold;
}

int GetRSSI_quality() {
  long rssi = WiFi.RSSI();

  if (-50 < rssi) { return 10; }

  if (rssi <= -98) { return 0;  }
  rssi = rssi + 97; // Range 0..47 => 1..9
  return (rssi / 5) + 1;
}

WiFiConnectionProtocol getConnectionProtocol() {
  if (WiFi.RSSI() < 0) {
    #ifdef ESP8266
    switch (wifi_get_phy_mode()) {
      case PHY_MODE_11B:
        return WiFiConnectionProtocol::WiFi_Protocol_11b;
      case PHY_MODE_11G:
        return WiFiConnectionProtocol::WiFi_Protocol_11g;
      case PHY_MODE_11N:
        return WiFiConnectionProtocol::WiFi_Protocol_11n;
    }
    #endif
    #ifdef ESP32

    wifi_phy_mode_t phymode;
    esp_wifi_sta_get_negotiated_phymode(&phymode);
    switch (phymode) {
      case WIFI_PHY_MODE_11B:   return WiFiConnectionProtocol::WiFi_Protocol_11b;
      case WIFI_PHY_MODE_11G:   return WiFiConnectionProtocol::WiFi_Protocol_11g;
      case WIFI_PHY_MODE_HT20:  return WiFiConnectionProtocol::WiFi_Protocol_HT20;
      case WIFI_PHY_MODE_HT40:  return WiFiConnectionProtocol::WiFi_Protocol_HT40;
      case WIFI_PHY_MODE_HE20:  return WiFiConnectionProtocol::WiFi_Protocol_HE20;
      case WIFI_PHY_MODE_LR:    return WiFiConnectionProtocol::WiFi_Protocol_LR;
#if ESP_IDF_VERSION >= ESP_IDF_VERSION_VAL(5, 3, 0)
      // 5 GHz
      case WIFI_PHY_MODE_11A:   return WiFiConnectionProtocol::WiFi_Protocol_11a;
      case WIFI_PHY_MODE_VHT20: return WiFiConnectionProtocol::WiFi_Protocol_VHT20;
#endif
    }
    #endif
  }
  return WiFiConnectionProtocol::Unknown;
}

#ifdef ESP32
int64_t WiFi_get_TSF_time()
{
  return esp_wifi_get_tsf_time(WIFI_IF_STA);
}
#endif


// ********************************************************************************
// Disconnect from Wifi AP
// ********************************************************************************
void WifiDisconnect()
{
  if (!WiFiEventData.processedDisconnect || 
       WiFiEventData.processingDisconnect.isSet()) {
    return;
  }
  if (WiFi.status() == WL_DISCONNECTED) {
    return;
  }
  // Prevent recursion
  static LongTermTimer processingDisconnectTimer;
  if (processingDisconnectTimer.isSet() && 
     !processingDisconnectTimer.timeoutReached(200)) return;
  processingDisconnectTimer.setNow();
  # ifndef BUILD_NO_DEBUG
  addLog(LOG_LEVEL_INFO, F("WiFi : WifiDisconnect()"));
  #endif
  #ifdef ESP32
  removeWiFiEventHandler();
  WiFi.disconnect();
  delay(100);
  {
    const IPAddress ip;
    const IPAddress gw;
    const IPAddress subnet;
    const IPAddress dns;
    WiFi.config(ip, gw, subnet, dns);
  }
  #endif
  #ifdef ESP8266
  // Only call disconnect when STA is active
  if (WifiIsSTA(WiFi.getMode())) {
    wifi_station_disconnect();
  }
  station_config conf{};
  memset(&conf, 0, sizeof(conf));
  ETS_UART_INTR_DISABLE();
  wifi_station_set_config_current(&conf);
  ETS_UART_INTR_ENABLE();
  #endif
  WiFiEventData.setWiFiDisconnected();
  WiFiEventData.markDisconnect(WIFI_DISCONNECT_REASON_UNSPECIFIED);
  /*
  if (!Settings.UseLastWiFiFromRTC()) {
    RTC.clearLastWiFi();
  }
  */
  delay(100);
  WiFiEventData.processingDisconnect.clear();
  WiFiEventData.processedDisconnect = false;
  processDisconnect();
  processingDisconnectTimer.clear();
}

// ********************************************************************************
// Scan WiFi network
// ********************************************************************************
bool WiFiScanAllowed() {
  if (WiFi_AP_Candidates.scanComplete() == WIFI_SCAN_RUNNING) {
    return false;
  }
  if (!WiFiEventData.processedScanDone) { 
    processScanDone(); 
  }
  if (!WiFiEventData.processedDisconnect) {
    processDisconnect();
  }

  if (WiFiEventData.wifiConnectInProgress) {
    return false;
  }

  if (WiFiEventData.intent_to_reboot) {
    return false;
  }

  if (WiFiEventData.unprocessedWifiEvents()) {
    # ifndef BUILD_NO_DEBUG
    if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
      String log = F("WiFi : Scan not allowed, unprocessed WiFi events: ");
      if (!WiFiEventData.processedConnect) {
        log += F(" conn");
      }
      if (!WiFiEventData.processedDisconnect) {
        log += F(" disconn");
      }
      if (!WiFiEventData.processedGotIP) {
        log += F(" gotIP");
      }
      if (!WiFiEventData.processedDHCPTimeout) {
        log += F(" DHCP_t/o");
      }
      
      addLogMove(LOG_LEVEL_ERROR, log);
      logConnectionStatus();
    }
    #endif
    return false;
  }
  /*
  if (!wifiAPmodeActivelyUsed() && !NetworkConnected()) {
    return true;
  }
  */
  WiFi_AP_Candidates.purge_expired();
  if (WiFiEventData.wifiConnectInProgress) {
    return false;
  }
  if (WiFiEventData.lastScanMoment.isSet()) {
    if (NetworkConnected() && WiFi_AP_Candidates.getBestCandidate().usable()) {
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, F("WiFi : Scan not needed, good candidate present"));
      #endif
      return false;
    }
  }

  if (WiFiEventData.lastDisconnectMoment.isSet() && WiFiEventData.lastDisconnectMoment.millisPassedSince() < WIFI_RECONNECT_WAIT) {
    if (!NetworkConnected()) {
      return WiFiEventData.processedConnect;
    }
  }
  if (WiFiEventData.lastScanMoment.isSet()) {
    const LongTermTimer::Duration scanInterval = wifiAPmodeActivelyUsed() ? WIFI_SCAN_INTERVAL_AP_USED : WIFI_SCAN_INTERVAL_MINIMAL;
    if (WiFiEventData.lastScanMoment.millisPassedSince() < scanInterval) {
      return false;
    }
  }
  return WiFiEventData.processedConnect;
}


void WifiScan(bool async, uint8_t channel) {
  setSTA(true);
  if (!WiFiScanAllowed()) {
    return;
  }
#ifdef ESP32
  // TD-er: Don't run async scan on ESP32.
  // Since IDF 4.4 it seems like the active channel may be messed up when running async scan
  // Perform a disconnect after scanning.
  // See: https://github.com/letscontrolit/ESPEasy/pull/3579#issuecomment-967021347
  async = false;

  if (Settings.IncludeHiddenSSID()) {
    wifi_country_t config = {
      .cc = "01",
      .schan = 1,
      .nchan = 14,
      .policy = WIFI_COUNTRY_POLICY_MANUAL,
    };
    esp_wifi_set_country(&config);
  }


#endif

  START_TIMER;
  WiFiEventData.lastScanMoment.setNow();
  # ifndef BUILD_NO_DEBUG
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    if (channel == 0) {
      addLog(LOG_LEVEL_INFO, F("WiFi : Start network scan all channels"));
    } else {
      addLogMove(LOG_LEVEL_INFO, strformat(F("WiFi : Start network scan ch: %d "), channel));
    }
  }
  #endif
  bool show_hidden         = true;
  WiFiEventData.processedScanDone = false;
  WiFiEventData.lastGetScanMoment.setNow();
  WiFiEventData.lastScanChannel = channel;

  unsigned int nrScans = 1 + (async ? 0 : Settings.NumberExtraWiFiScans);
  while (nrScans > 0) {
    if (!async) {
      WiFi_AP_Candidates.begin_sync_scan();
      FeedSW_watchdog();
    }
    --nrScans;
#ifdef ESP8266
#if FEATURE_ESP8266_DIRECT_WIFI_SCAN
    {
      static bool FIRST_SCAN = true;

      struct scan_config config;
      memset(&config, 0, sizeof(config));
      config.ssid = nullptr;
      config.bssid = nullptr;
      config.channel = channel;
      config.show_hidden = show_hidden ? 1 : 0;;
      config.scan_type = WIFI_SCAN_TYPE_ACTIVE;
      if (FIRST_SCAN) {
        config.scan_time.active.min = 100;
        config.scan_time.active.max = 200;
      } else {
        config.scan_time.active.min = 400;
        config.scan_time.active.max = 500;
      }
      FIRST_SCAN = false;
      wifi_station_scan(&config, &onWiFiScanDone);
      if (!async) {
        // will resume when SYSTEM_EVENT_SCAN_DONE event is fired
        do {
          delay(0);
        } while (!WiFiEventData.processedScanDone);
      }
 
    }
#else
    WiFi.scanNetworks(async, show_hidden, channel);
#endif
#endif
#ifdef ESP32
    const bool passive = Settings.PassiveWiFiScan();
    const uint32_t max_ms_per_chan = 120;
    WiFi.setScanTimeout(14 * max_ms_per_chan * 2);
    WiFi.scanNetworks(async, show_hidden, passive, max_ms_per_chan /*, channel */);
#endif
    if (!async) {
      FeedSW_watchdog();
      processScanDone();
    }
  }
#if FEATURE_TIMING_STATS
  if (async) {
    STOP_TIMER(WIFI_SCAN_ASYNC);
  } else {
    STOP_TIMER(WIFI_SCAN_SYNC);
  }
#endif

#ifdef ESP32
#if ESP_IDF_VERSION_MAJOR<5
  RTC.clearLastWiFi();
  if (WiFiConnected()) {
    # ifndef BUILD_NO_DEBUG
    addLog(LOG_LEVEL_INFO, F("WiFi : Disconnect after scan"));
    #endif

    const bool needReconnect = WiFiEventData.wifiConnectAttemptNeeded;
    WifiDisconnect();
    WiFiEventData.wifiConnectAttemptNeeded = needReconnect;
  }
#endif
#endif

}

// ********************************************************************************
// Scan all Wifi Access Points
// ********************************************************************************
void WiFiScan_log_to_serial()
{
  // Direct Serial is allowed here, since this function will only be called from serial input.
  serialPrintln(F("WIFI : SSID Scan start"));
  if (WiFi_AP_Candidates.scanComplete() <= 0) {
    WiFiMode_t cur_wifimode = WiFi.getMode();
    WifiScan(false);
    setWifiMode(cur_wifimode);
  }

  const int8_t scanCompleteStatus = WiFi_AP_Candidates.scanComplete();
  if (scanCompleteStatus <= 0) {
    serialPrintln(F("WIFI : No networks found"));
  }
  else
  {
    serialPrint(F("WIFI : "));
    serialPrint(String(scanCompleteStatus));
    serialPrintln(F(" networks found"));

    int i = 0;

    for (auto it = WiFi_AP_Candidates.scanned_begin(); it != WiFi_AP_Candidates.scanned_end(); ++it)
    {
      ++i;
      // Print SSID and RSSI for each network found
      serialPrint(F("WIFI : "));
      serialPrint(String(i));
      serialPrint(": ");
      serialPrintln(it->toString());
      delay(10);
    }
  }
  serialPrintln("");
}

// ********************************************************************************
// Manage Wifi Modes
// ********************************************************************************
void setSTA(bool enable) {
  switch (WiFi.getMode()) {
    case WIFI_OFF:

      if (enable) { setWifiMode(WIFI_STA); }
      break;
    case WIFI_STA:

      if (!enable) { setWifiMode(WIFI_OFF); }
      break;
    case WIFI_AP:

      if (enable) { setWifiMode(WIFI_AP_STA); }
      break;
    case WIFI_AP_STA:

      if (!enable) { setWifiMode(WIFI_AP); }
      break;
    default:
      break;
  }
}

void setAP(bool enable) {
  WiFiMode_t wifimode = WiFi.getMode();

  switch (wifimode) {
    case WIFI_OFF:

      if (enable) { 
        setWifiMode(WIFI_AP); 
      }
      break;
    case WIFI_STA:

      if (enable) { setWifiMode(WIFI_AP_STA); }
      break;
    case WIFI_AP:

      if (!enable) { setWifiMode(WIFI_OFF); }
      break;
    case WIFI_AP_STA:

      if (!enable) { setWifiMode(WIFI_STA); }
      break;
    default:
      break;
  }
}

// Only internal scope
void setAPinternal(bool enable)
{
  if (enable) {
    // create and store unique AP SSID/PW to prevent ESP from starting AP mode with default SSID and No password!
    // setup ssid for AP Mode when needed
    String softAPSSID = NetworkCreateRFCCompliantHostname();
    String pwd        = SecuritySettings.WifiAPKey;
    IPAddress subnet(DEFAULT_AP_SUBNET);

    if (!WiFi.softAPConfig(apIP, apIP, subnet)) {
      addLog(LOG_LEVEL_ERROR, strformat(
        ("WIFI : [AP] softAPConfig failed! IP: %s, GW: %s, SN: %s"),
        apIP.toString().c_str(), 
        apIP.toString().c_str(), 
        subnet.toString().c_str()
      )
      );
    }

    int channel = 1;
    if (WifiIsSTA(WiFi.getMode()) && WiFiConnected()) {
      channel = WiFi.channel();
    }

    if (WiFi.softAP(softAPSSID.c_str(), pwd.c_str(), channel)) {
      eventQueue.add(F("WiFi#APmodeEnabled"));
      if (loglevelActiveFor(LOG_LEVEL_INFO)) {
        addLogMove(LOG_LEVEL_INFO, strformat(
          F("WIFI : AP Mode enabled. SSID: %s IP: %s ch: %d"),
          softAPSSID.c_str(),
          formatIP(WiFi.softAPIP()).c_str(),
          channel));
      }
    } else {
      if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
        addLogMove(LOG_LEVEL_ERROR, strformat(
          F("WIFI : Error while starting AP Mode with SSID: %s IP: %s"),
          softAPSSID.c_str(),
          formatIP(apIP).c_str()));
      }
    }
    #ifdef ESP32

    #else // ifdef ESP32

    if (wifi_softap_dhcps_status() != DHCP_STARTED) {
      if (!wifi_softap_dhcps_start()) {
        addLog(LOG_LEVEL_ERROR, F("WIFI : [AP] wifi_softap_dhcps_start failed!"));
      }
    }
    #endif // ifdef ESP32
    WiFiEventData.timerAPoff.setMillisFromNow(WIFI_AP_OFF_TIMER_DURATION);
  } else {
    #if FEATURE_DNS_SERVER
    if (dnsServerActive) {
      dnsServerActive = false;
      dnsServer.stop();
    }
    #endif // if FEATURE_DNS_SERVER
  }
}

const __FlashStringHelper * getWifiModeString(WiFiMode_t wifimode)
{
  switch (wifimode) {
    case WIFI_OFF:   return F("OFF");
    case WIFI_STA:   return F("STA");
    case WIFI_AP:    return F("AP");
    case WIFI_AP_STA: return F("AP+STA");
    default:
      break;
  }
  return F("Unknown");
}

void setWifiMode(WiFiMode_t new_mode) {
  const WiFiMode_t cur_mode = WiFi.getMode();
  static WiFiMode_t processing_wifi_mode = cur_mode;
  if (cur_mode == new_mode) {
    return;
  }
  if (processing_wifi_mode == new_mode) {
    // Prevent loops
    return;
  }
  processing_wifi_mode = new_mode;

  if (cur_mode == WIFI_OFF) {
    #if defined(ESP32)
    // Needs to be set while WiFi is off
    WiFi.hostname(NetworkCreateRFCCompliantHostname());
    #endif
    WiFiEventData.markWiFiTurnOn();
  }
  if (new_mode != WIFI_OFF) {
    #ifdef ESP8266
    // See: https://github.com/esp8266/Arduino/issues/6172#issuecomment-500457407
    WiFi.forceSleepWake(); // Make sure WiFi is really active.
    #endif
    delay(100);
  } else {
    WifiDisconnect();
//    delay(100);
    processDisconnect();
    WiFiEventData.clear_processed_flags();
  }

  addLog(LOG_LEVEL_INFO, concat(F("WIFI : Set WiFi to "), getWifiModeString(new_mode)));

  int retry = 2;
  while (!WiFi.mode(new_mode) && retry > 0) {
    #ifndef BUILD_MINIMAL_OTA
    addLog(LOG_LEVEL_INFO, F("WIFI : Cannot set mode!!!!!"));
    #endif
    delay(100);
    --retry;
  }
  retry = 2;
  while (WiFi.getMode() != new_mode && retry > 0) {
    #ifndef BUILD_MINIMAL_OTA
    addLog(LOG_LEVEL_INFO, F("WIFI : mode not yet set"));
    #endif
    delay(100);
    --retry;
  }


  if (new_mode == WIFI_OFF) {
    WiFiEventData.markWiFiTurnOn();
    #if defined(ESP32)
    // Needs to be set while WiFi is off
    WiFi.hostname(NetworkCreateRFCCompliantHostname());
    #endif
    delay(100);
    #if defined(ESP32)
    esp_wifi_set_ps(WIFI_PS_NONE); 
//    esp_wifi_set_ps(WIFI_PS_MAX_MODEM);
    #endif
    #ifdef ESP8266
    WiFi.forceSleepBegin();
    #endif // ifdef ESP8266
    delay(1);
  } else {
    #ifdef ESP32
    if (cur_mode == WIFI_OFF) {
      registerWiFiEventHandler();
    }
    #endif
    // Only set power mode when AP is not enabled
    // When AP is enabled, the sleep mode is already set to WIFI_NONE_SLEEP
    if (!WifiIsAP(new_mode)) {
      if (Settings.WifiNoneSleep()) {
        #ifdef ESP8266
        WiFi.setSleepMode(WIFI_NONE_SLEEP);
        #endif
        #ifdef ESP32
        WiFi.setSleep(WIFI_PS_NONE);
        #endif
      } else if (Settings.EcoPowerMode()) {
        // Allow light sleep during idle times
        #ifdef ESP8266
        WiFi.setSleepMode(WIFI_LIGHT_SLEEP);
        #endif
        #ifdef ESP32
        // Maximum modem power saving. 
        // In this mode, interval to receive beacons is determined by the listen_interval parameter in wifi_sta_config_t
        // FIXME TD-er: Must test if this is desired behavior in ESP32.
        WiFi.setSleep(WIFI_PS_MAX_MODEM);
        #endif
      } else {
        // Default
        #ifdef ESP8266
        WiFi.setSleepMode(WIFI_MODEM_SLEEP);
        #endif
        #ifdef ESP32
        // Minimum modem power saving. 
        // In this mode, station wakes up to receive beacon every DTIM period
        WiFi.setSleep(WIFI_PS_MIN_MODEM);
        #endif
      }
    }
#if FEATURE_SET_WIFI_TX_PWR
    SetWiFiTXpower();
#endif
    if (WifiIsSTA(new_mode)) {
//      WiFi.setAutoConnect(Settings.SDK_WiFi_autoreconnect());
      WiFi.setAutoReconnect(Settings.SDK_WiFi_autoreconnect());
    }
    delay(100); // Must allow for some time to init.
  }
  const bool new_mode_AP_enabled = WifiIsAP(new_mode);

  if (WifiIsAP(cur_mode) && !new_mode_AP_enabled) {
    eventQueue.add(F("WiFi#APmodeDisabled"));
  }

  if (WifiIsAP(cur_mode) != new_mode_AP_enabled) {
    // Mode has changed
    setAPinternal(new_mode_AP_enabled);
  }
  #if FEATURE_MDNS
  #ifdef ESP8266
  // notifyAPChange() is not present in the ESP32 MDNSResponder
  MDNS.notifyAPChange();
  #endif
  #endif
}

bool WifiIsAP(WiFiMode_t wifimode)
{
  #if defined(ESP32)
  return (wifimode == WIFI_MODE_AP) || (wifimode == WIFI_MODE_APSTA);
  #else // if defined(ESP32)
  return (wifimode == WIFI_AP) || (wifimode == WIFI_AP_STA);
  #endif // if defined(ESP32)
}

bool WifiIsSTA(WiFiMode_t wifimode)
{
  #if defined(ESP32)
  return (wifimode & WIFI_MODE_STA) != 0;
  #else // if defined(ESP32)
  return (wifimode & WIFI_STA) != 0;
  #endif // if defined(ESP32)
}

bool WiFiUseStaticIP() {
  return Settings.IP[0] != 0 && Settings.IP[0] != 255;
}

bool wifiAPmodeActivelyUsed()
{
  if (!WifiIsAP(WiFi.getMode()) || (!WiFiEventData.timerAPoff.isSet())) {
    // AP not active or soon to be disabled in processDisableAPmode()
    return false;
  }
  return WiFi.softAPgetStationNum() != 0;

  // FIXME TD-er: is effectively checking for AP active enough or must really check for connected clients to prevent automatic wifi
  // reconnect?
}

void setConnectionSpeed() {
  #ifdef ESP8266
  // ESP8266 only supports 802.11g mode when running in STA+AP
  const bool forcedByAPmode = WifiIsAP(WiFi.getMode());
  WiFiPhyMode_t phyMode = (Settings.ForceWiFi_bg_mode() || forcedByAPmode) ? WIFI_PHY_MODE_11G : WIFI_PHY_MODE_11N;
  if (!forcedByAPmode) {
    const WiFi_AP_Candidate candidate = WiFi_AP_Candidates.getCurrent();
    if (candidate.phy_known() && (candidate.bits.phy_11g != candidate.bits.phy_11n)) {
      if ((WIFI_PHY_MODE_11G == phyMode) && !candidate.bits.phy_11g) {
        phyMode = WIFI_PHY_MODE_11N;
        #ifndef BUILD_MINIMAL_OTA
        addLog(LOG_LEVEL_INFO, F("WIFI : AP is set to 802.11n only"));
        #endif
      } else if ((WIFI_PHY_MODE_11N == phyMode) && !candidate.bits.phy_11n) {
        phyMode = WIFI_PHY_MODE_11G;
        #ifndef BUILD_MINIMAL_OTA
        addLog(LOG_LEVEL_INFO, F("WIFI : AP is set to 802.11g only"));
        #endif
      }      
    } else {
      bool useAlternate = WiFiEventData.connectionFailures > 10;
      if (useAlternate) {
        phyMode = (WIFI_PHY_MODE_11G == phyMode) ? WIFI_PHY_MODE_11N : WIFI_PHY_MODE_11G;
      }
    }
  } else {
    // No need to perform a next attempt.
    WiFi_AP_Candidates.markAttempt();
  }

  if (WiFi.getPhyMode() == phyMode) {
    return;
  }
  #ifndef BUILD_NO_DEBUG
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log = concat(F("WIFI : Set to 802.11"), (WIFI_PHY_MODE_11G == phyMode) ? 'g' : 'n');
    if (forcedByAPmode) {
      log += (F(" (AP+STA mode)"));
    }
    if (Settings.ForceWiFi_bg_mode()) {
      log += F(" Force B/G mode");
    }
    addLogMove(LOG_LEVEL_INFO, log);
  }
  #endif
  WiFi.setPhyMode(phyMode);
  #endif // ifdef ESP8266

  // Does not (yet) work, so commented out.
  #ifdef ESP32

  // HT20 = 20 MHz channel width.
  // HT40 = 40 MHz channel width.
  // In theory, HT40 can offer upto 150 Mbps connection speed.
  // However since HT40 is using nearly all channels on 2.4 GHz WiFi,
  // Thus you are more likely to experience disturbances.
  // The response speed and stability is better at HT20 for ESP units.
  esp_wifi_set_bandwidth(WIFI_IF_STA, WIFI_BW_HT20);

  uint8_t protocol = WIFI_PROTOCOL_11B | WIFI_PROTOCOL_11G; // Default to BG

  if (!Settings.ForceWiFi_bg_mode() || (WiFiEventData.connectionFailures > 10)) {
    // Set to use BGN
    protocol |= WIFI_PROTOCOL_11N;
    #ifdef ESP32C6
    protocol |= WIFI_PROTOCOL_11AX;
    #endif
  }

  const WiFi_AP_Candidate candidate = WiFi_AP_Candidates.getCurrent();
  if (candidate.phy_known()) {
    // Check to see if the access point is set to "N-only"
    if ((protocol & WIFI_PROTOCOL_11N) == 0) {
      if (!candidate.bits.phy_11b && !candidate.bits.phy_11g && candidate.bits.phy_11n) {
        if (candidate.bits.phy_11n) {
          // Set to use BGN
          protocol |= WIFI_PROTOCOL_11N;
          addLog(LOG_LEVEL_INFO, F("WIFI : AP is set to 802.11n only"));
        }
#ifdef ESP32C6
        if (candidate.bits.phy_11ax) {
          // Set to use WiFi6
          protocol |= WIFI_PROTOCOL_11AX;
          addLog(LOG_LEVEL_INFO, F("WIFI : AP is set to 802.11ax"));
        }
#endif
      }
    }
  }


  if (WifiIsSTA(WiFi.getMode())) {
    // Set to use "Long GI" making it more resilliant to reflections
    // See: https://www.tp-link.com/us/configuration-guides/q_a_basic_wireless_concepts/?configurationId=2958#_idTextAnchor038
    esp_wifi_config_80211_tx_rate(WIFI_IF_STA, WIFI_PHY_RATE_MCS3_LGI);
    esp_wifi_set_protocol(WIFI_IF_STA, protocol);
  }

  if (WifiIsAP(WiFi.getMode())) {
    esp_wifi_set_protocol(WIFI_IF_AP, protocol);
  }
  #endif // ifdef ESP32
}

void setupStaticIPconfig() {
  setUseStaticIP(WiFiUseStaticIP());

  if (!WiFiUseStaticIP()) { return; }
  const IPAddress ip     (Settings.IP);
  const IPAddress gw     (Settings.Gateway);
  const IPAddress subnet (Settings.Subnet);
  const IPAddress dns    (Settings.DNS);

  WiFiEventData.dns0_cache = dns;

  WiFi.config(ip, gw, subnet, dns);

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLogMove(LOG_LEVEL_INFO, strformat(
      F("IP   : Static IP : %s GW: %s SN: %s DNS: %s"),
      formatIP(ip).c_str(),
      formatIP(gw).c_str(),
      formatIP(subnet).c_str(),
      getValue(LabelType::DNS).c_str()));
  }
}

// ********************************************************************************
// Formatting WiFi related strings
// ********************************************************************************
String formatScanResult(int i, const String& separator) {
  int32_t rssi = 0;

  return formatScanResult(i, separator, rssi);
}

String formatScanResult(int i, const String& separator, int32_t& rssi) {
  WiFi_AP_Candidate tmp(i);
  rssi = tmp.rssi;
  return tmp.toString(separator);
}


void logConnectionStatus() {
  static unsigned long lastLog = 0;
  if (lastLog != 0 && timePassedSince(lastLog) < 1000) {
    return;
  }
  lastLog = millis();
#ifndef BUILD_NO_DEBUG
  #ifdef ESP8266
  const uint8_t arduino_corelib_wifistatus = WiFi.status();
  const uint8_t sdk_wifistatus             = wifi_station_get_connect_status();

  if ((arduino_corelib_wifistatus == WL_CONNECTED) != (sdk_wifistatus == STATION_GOT_IP)) {
    if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
      String log = F("WiFi : SDK station status differs from Arduino status. SDK-status: ");
      log += SDKwifiStatusToString(sdk_wifistatus);
      log += F(" Arduino status: ");
      log += ArduinoWifiStatusToString(arduino_corelib_wifistatus);
      addLogMove(LOG_LEVEL_ERROR, log);
    }
  }
  #endif

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLogMove(LOG_LEVEL_INFO, strformat(
      F("WIFI : Arduino wifi status: %s ESPeasy internal wifi status: %s"),
      ArduinoWifiStatusToString(WiFi.status()).c_str(),
      WiFiEventData.ESPeasyWifiStatusToString().c_str()));
  }
/*
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log;

    switch (WiFi.status()) {
      case WL_NO_SSID_AVAIL: {
        log = F("WIFI : No SSID found matching: ");
        break;
      }
      case WL_CONNECT_FAILED: {
        log = F("WIFI : Connection failed to: ");
        break;
      }
      case WL_DISCONNECTED: {
        log = F("WIFI : WiFi.status() = WL_DISCONNECTED  SSID: ");
        break;
      }
      case WL_IDLE_STATUS: {
        log = F("WIFI : Connection in IDLE state: ");
        break;
      }
      case WL_CONNECTED: {
        break;
      }
      default:
        break;
    }

    if (log.length() > 0) {
      const char *ssid = getLastWiFiSettingsSSID();
      log += ssid;
      addLog(LOG_LEVEL_INFO, log);
    }
  }
  */
#endif // ifndef BUILD_NO_DEBUG
}

#include "../ESPEasyCore/ESPEasy_setup.h"

#include "../../ESPEasy_fdwdecl.h" // Needed for PluginInit() and CPluginInit()

#include "../../ESPEasy-Globals.h"
#include "../../_Plugin_Helper.h"
#include "../Commands/InternalCommands_decoder.h"
#include "../CustomBuild/CompiletimeDefines.h"
#include "../ESPEasyCore/ESPEasyGPIO.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/ESPEasyRules.h"
#include "../ESPEasyCore/ESPEasyWifi.h"
#include "../ESPEasyCore/ESPEasyWifi_ProcessEvent.h"
#include "../ESPEasyCore/Serial.h"
#include "../Globals/Cache.h"
#include "../Globals/ESPEasy_Console.h"
#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/ESPEasy_time.h"
#include "../Globals/NetworkState.h"
#include "../Globals/RTC.h"
#include "../Globals/Statistics.h"
#include "../Globals/WiFi_AP_Candidates.h"
#include "../Helpers/_CPlugin_init.h"
#include "../Helpers/_NPlugin_init.h"
#include "../Helpers/_Plugin_init.h"
#include "../Helpers/DeepSleep.h"
#include "../Helpers/ESPEasyRTC.h"
#include "../Helpers/ESPEasy_FactoryDefault.h"
#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/ESPEasy_checks.h"
#include "../Helpers/Hardware_device_info.h"
#include "../Helpers/Memory.h"
#include "../Helpers/Misc.h"
#include "../Helpers/StringGenerator_System.h"
#include "../WebServer/ESPEasy_WebServer.h"


#ifdef USE_RTOS_MULTITASKING
# include "../Helpers/Networking.h"
# include "../Helpers/PeriodicalActions.h"
#endif // ifdef USE_RTOS_MULTITASKING

#if FEATURE_ARDUINO_OTA
# include "../Helpers/OTA.h"
#endif // if FEATURE_ARDUINO_OTA

#ifdef ESP32

# if ESP_IDF_VERSION_MAJOR < 5
#  include <esp_pm.h>
#  include <soc/efuse_reg.h>
#  include <soc/boot_mode.h>
#  include <soc/gpio_reg.h>
# else // if ESP_IDF_VERSION_MAJOR < 5
#  include <hal/efuse_hal.h>
#  include <rom/gpio.h>
#  include <esp_pm.h>
# endif // if ESP_IDF_VERSION_MAJOR < 5

# if CONFIG_IDF_TARGET_ESP32
#  if ESP_IDF_VERSION_MAJOR < 5
#   include "hal/efuse_ll.h"
#   include "hal/efuse_hal.h"
#  else // if ESP_IDF_VERSION_MAJOR < 5
#   include <soc/efuse_defs.h>
#   include <bootloader_common.h>

#   if ESP_IDF_VERSION > ESP_IDF_VERSION_VAL(5, 2, 0)

// IDF5.3 fix esp_gpio_reserve used in init PSRAM.
#    include "esp_private/esp_gpio_reserve.h"
#   endif // if ESP_IDF_VERSION > ESP_IDF_VERSION_VAL(5, 2, 0)
#  endif // if ESP_IDF_VERSION_MAJOR < 5
# endif // if CONFIG_IDF_TARGET_ESP32

#endif // ifdef ESP32


#ifdef USE_RTOS_MULTITASKING

void RTOS_TaskServers(void *parameter)
{
  while (true) {
    delay(100);
    web_server.handleClient();
    # if FEATURE_ESPEASY_P2P
    checkUDP();
    # endif // if FEATURE_ESPEASY_P2P
  }
}

void RTOS_TaskSerial(void *parameter)
{
  while (true) {
    delay(100);
    serial();
  }
}

void RTOS_Task10ps(void *parameter)
{
  while (true) {
    delay(100);
    run10TimesPerSecond();
  }
}

void RTOS_HandleSchedule(void *parameter)
{
  while (true) {
    Scheduler.handle_schedule();
  }
}

#endif // ifdef USE_RTOS_MULTITASKING

/*********************************************************************************************\
* ISR call back function for handling the watchdog.
\*********************************************************************************************/
void sw_watchdog_callback(void *arg)
{
  yield(); // feed the WD
  ++sw_watchdog_callback_count;
}

/*********************************************************************************************\
* SETUP
\*********************************************************************************************/
void ESPEasy_setup()
{
#if defined(ESP8266_DISABLE_EXTRA4K) || defined(USE_SECOND_HEAP)

  //  disable_extra4k_at_link_time();
#endif // if defined(ESP8266_DISABLE_EXTRA4K) || defined(USE_SECOND_HEAP)
#ifdef PHASE_LOCKED_WAVEFORM
  enablePhaseLockedWaveform();
#endif // ifdef PHASE_LOCKED_WAVEFORM
#ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
#endif // ifdef USE_SECOND_HEAP
#ifdef ESP32
# ifdef DISABLE_ESP32_BROWNOUT
  DisableBrownout(); // Workaround possible weak LDO resulting in brownout detection during Wifi connection
# endif  // DISABLE_ESP32_BROWNOUT

# ifdef BOARD_HAS_PSRAM
  psramInit();
# endif // ifdef BOARD_HAS_PSRAM

# if CONFIG_IDF_TARGET_ESP32

  // restore GPIO16/17 if no PSRAM is found
  if (!FoundPSRAM()) {
    // test if the CPU is not pico
    #  if ESP_IDF_VERSION_MAJOR < 5
    uint32_t chip_ver    = REG_GET_FIELD(EFUSE_BLK0_RDATA3_REG, EFUSE_RD_CHIP_VER_PKG);
    uint32_t pkg_version = chip_ver & 0x7;
    #  else // if ESP_IDF_VERSION_MAJOR < 5
    uint32_t pkg_version = bootloader_common_get_chip_ver_pkg();
    uint32_t chip_ver    = REG_GET_FIELD(EFUSE_BLK0_RDATA3_REG, EFUSE_RD_CHIP_PACKAGE);
    #  endif // if ESP_IDF_VERSION_MAJOR < 5

    if (pkg_version <= 7) { // D0WD, S0WD, D2WD
#ifdef CORE32SOLO1
      gpio_num_t PSRAM_CLK = GPIO_NUM_NC; //GPIO_NUM_17;
      gpio_num_t PSRAM_CS  = GPIO_NUM_NC; //GPIO_NUM_16;
#else
      gpio_num_t PSRAM_CLK = static_cast<gpio_num_t>(CONFIG_D0WD_PSRAM_CLK_IO);
      gpio_num_t PSRAM_CS  = static_cast<gpio_num_t>(CONFIG_D0WD_PSRAM_CS_IO);

      switch (pkg_version)
      {
        case EFUSE_RD_CHIP_VER_PKG_ESP32U4WDH:
          // Very strange model as the same model exists as:
          // - Single core @160 MHz
          // - Dual   core @240 MHz
          // See: https://www.letscontrolit.com/forum/viewtopic.php?t=10735
          PSRAM_CLK = GPIO_NUM_NC; //GPIO_NUM_17;
          PSRAM_CS  = GPIO_NUM_NC; //GPIO_NUM_16;
          break;
        case EFUSE_RD_CHIP_VER_PKG_ESP32D2WDQ5:
          PSRAM_CLK = static_cast<gpio_num_t>(CONFIG_D2WD_PSRAM_CLK_IO);
          PSRAM_CS  = static_cast<gpio_num_t>(CONFIG_D2WD_PSRAM_CS_IO);
          break;
        case EFUSE_RD_CHIP_VER_PKG_ESP32PICOD4:
        case EFUSE_RD_CHIP_VER_PKG_ESP32PICOV302:
          PSRAM_CLK = static_cast<gpio_num_t>(GPIO_NUM_NC);
          PSRAM_CS  = static_cast<gpio_num_t>(CONFIG_PICO_PSRAM_CS_IO);
          break;
      }
#endif
#  if ESP_IDF_VERSION > ESP_IDF_VERSION_VAL(5, 2, 0)

      // Thanks Theo Arends from Tasmota
      if (PSRAM_CLK != GPIO_NUM_NC) {
        esp_gpio_revoke(BIT64(PSRAM_CLK) | BIT64(PSRAM_CS));
      } else {
        if (PSRAM_CS != GPIO_NUM_NC) {
          esp_gpio_revoke(BIT64(PSRAM_CS));
        }
      }
#  endif // if ESP_IDF_VERSION > ESP_IDF_VERSION_VAL(5, 2, 0)

      if (PSRAM_CLK != GPIO_NUM_NC) {
        gpio_reset_pin(PSRAM_CLK);
      }
      if (PSRAM_CS != GPIO_NUM_NC) {
        gpio_reset_pin(PSRAM_CS);
      }
    }
  }
# endif  // if CONFIG_IDF_TARGET_ESP32
  initADC();
#endif  // ESP32
#ifndef BUILD_NO_RAM_TRACKER
  lowestFreeStack = getFreeStackWatermark();
  lowestRAM       = FreeMem();
#endif // ifndef BUILD_NO_RAM_TRACKER

  /*
   #ifdef ESP32
     {
     ESPEasy_NVS_Helper preferences;
     ResetFactoryDefaultPreference.init(preferences);
     }
   #endif
   */
#ifndef BUILD_NO_DEBUG

  //  checkAll_internalCommands();
#endif // ifndef BUILD_NO_DEBUG

  PluginSetup();
  CPluginSetup();

  initWiFi();
  WiFiEventData.clearAll();

#ifndef BUILD_MINIMAL_OTA
  run_compiletime_checks();
#endif // ifndef BUILD_MINIMAL_OTA
#ifdef ESP8266

  //  ets_isr_attach(8, sw_watchdog_callback, nullptr);  // Set a callback for feeding the watchdog.
#endif // ifdef ESP8266


  // Read ADC at boot, before WiFi tries to connect.
  // see https://github.com/letscontrolit/ESPEasy/issues/2646
#if FEATURE_ADC_VCC
  vcc = ESP.getVcc() / 1000.0f;
#endif // if FEATURE_ADC_VCC
#ifdef ESP8266
  espeasy_analogRead(A0);
#endif // ifdef ESP8266

  initAnalogWrite();

  resetPluginTaskData();

  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("setup"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
  ESPEasy_Console.begin(115200);

  // serialPrint("\n\n\nBOOOTTT\n\n\n");

  initLog();
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("initLog()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
  #ifdef BOARD_HAS_PSRAM

  if (FoundPSRAM()) {
    if (UsePSRAM()) {
      addLog(LOG_LEVEL_INFO, F("Using PSRAM"));
    } else {
      addLog(LOG_LEVEL_ERROR, F("PSRAM found, unable to use"));
    }
  }
  #endif // ifdef BOARD_HAS_PSRAM

  if (SpiffsSectors() < 32)
  {
    serialPrintln(F("\nNo (or too small) FS area..\nSystem Halted\nPlease reflash with 128k FS minimum!"));

    while (true) {
      delay(1);
    }
  }

  emergencyReset();

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log = F("\n\n\rINIT : Booting version: ");
    log += getValue(LabelType::BINARY_FILENAME);
    log += F(", (");
    log += get_build_origin();
    log += F(") ");
    log += getValue(LabelType::GIT_BUILD);
    log += F(" (");
    log += getSystemLibraryString();
    log += ')';
    addLogMove(LOG_LEVEL_INFO, log);
    log  = F("INIT : Free RAM:");
    log += FreeMem();
    addLogMove(LOG_LEVEL_INFO, log);
  }

  readBootCause();

  {
    String log;

    if (readFromRTC())
    {
      RTC.bootFailedCount++;
      RTC.bootCounter++;
      lastMixedSchedulerId_beforereboot.mixed_id = RTC.lastMixedSchedulerId;
      readUserVarFromRTC();
      log = concat(F("INIT : "), getLastBootCauseString()) + 
            concat(F(" #"), RTC.bootCounter);

      #ifndef BUILD_NO_DEBUG
      log += F(" Last Action before Reboot: ");
      log += ESPEasy_Scheduler::decodeSchedulerId(lastMixedSchedulerId_beforereboot);
      log += F(" Last systime: ");
      log += RTC.lastSysTime;
      #endif // ifndef BUILD_NO_DEBUG
    }

    // cold boot (RTC memory empty)
    else
    {
      initRTC();

      // cold boot situation
      if (lastBootCause == BOOT_CAUSE_MANUAL_REBOOT) { // only set this if not set earlier during boot stage.
        lastBootCause = BOOT_CAUSE_COLD_BOOT;
      }
      log = F("INIT : Cold Boot");
    }

    log += concat(F(" - Restart Reason: "), getResetReasonString());

    RTC.deepSleepState = 0;
    saveToRTC();

    addLogMove(LOG_LEVEL_INFO, log);
  }
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("RTC init"));
  #endif // ifndef BUILD_NO_RAM_TRACKER

  fileSystemCheck();
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("fileSystemCheck()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER

  //  progMemMD5check();
  LoadSettings();
#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  ESPEasy_Console.reInit();
#endif // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT

  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("LoadSettings()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER

  addLog(LOG_LEVEL_INFO, concat(F("CPU Frequency: "), ESP.getCpuFreqMHz()));
  

#ifdef ESP32
#ifndef CORE32SOLO1

  // Configure dynamic frequency scaling:
  // maximum and minimum frequencies are set in sdkconfig,
  // automatic light sleep is enabled if tickless idle support is enabled.
  ESP_PM_CONFIG_T pm_config = {
    .max_freq_mhz = getCPU_MaxFreqMHz(),
    .min_freq_mhz = Settings.EcoPowerMode() ? getCPU_MinFreqMHz() : getCPU_MaxFreqMHz(),
# if CONFIG_FREERTOS_USE_TICKLESS_IDLE
    .light_sleep_enable = Settings.EcoPowerMode()
# else
    .light_sleep_enable = false
# endif // if CONFIG_FREERTOS_USE_TICKLESS_IDLE
  };
  esp_pm_configure(&pm_config);
#endif
#endif // ifdef ESP32


  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("hardwareInit"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
  hardwareInit();
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("hardwareInit()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER

  node_time.restoreFromRTC();

  Settings.UseRTOSMultitasking = false; // For now, disable it, we experience heap corruption.

  if ((RTC.bootFailedCount > 10) && (RTC.bootCounter > 10)) {
    uint8_t toDisable = RTC.bootFailedCount - 10;
    toDisable = disablePlugin(toDisable);

    if (toDisable != 0) {
      toDisable = disableController(toDisable);
    }
    #if FEATURE_NOTIFIER

    if (toDisable != 0) {
      toDisable = disableNotification(toDisable);
    }
    #endif // if FEATURE_NOTIFIER

    if (toDisable != 0) {
      toDisable = disableRules(toDisable);
    }

    if (toDisable != 0) {
      toDisable = disableAllPlugins(toDisable);
    }

    if (toDisable != 0) {
      toDisable = disableAllControllers(toDisable);
    }
#if FEATURE_NOTIFIER

    if (toDisable != 0) {
      toDisable = disableAllNotifications(toDisable);
    }
#endif // if FEATURE_NOTIFIER
  }
  #if FEATURE_ETHERNET

  // This ensures, that changing WIFI OR ETHERNET MODE happens properly only after reboot. Changing without reboot would not be a good idea.
  // This only works after LoadSettings();
  // Do not call setNetworkMedium here as that may try to clean up settings.
  active_network_medium = Settings.NetworkMedium;
  #else // if FEATURE_ETHERNET

  if (Settings.NetworkMedium == NetworkMedium_t::Ethernet) {
    Settings.NetworkMedium = NetworkMedium_t::WIFI;
  }
  #endif // if FEATURE_ETHERNET

  setNetworkMedium(Settings.NetworkMedium);

  bool initWiFi = active_network_medium == NetworkMedium_t::WIFI;

  // FIXME TD-er: Must add another check for 'delayed start WiFi' for poorly designed ESP8266 nodes.


  if (initWiFi) {
#ifdef ESP32
    // FIXME TD-er: Disabled for now, as this may not return and thus block the ESP forever.
    // See: https://github.com/espressif/esp-idf/issues/15862
    //check_and_update_WiFi_Calibration();
#endif

    WiFi_AP_Candidates.clearCache();
    WiFi_AP_Candidates.load_knownCredentials();
    setSTA(true);

    if (!WiFi_AP_Candidates.hasCandidates()) {
      WiFiEventData.wifiSetup = true;
      RTC.clearLastWiFi(); // Must scan all channels
      // Wait until scan has finished to make sure as many as possible are found
      // We're still in the setup phase, so nothing else is taking resources of the ESP.
      WifiScan(false);
      WiFiEventData.lastScanMoment.clear();
    }

    // Always perform WiFi scan
    // It appears reconnecting from RTC may take just as long to be able to send first packet as performing a scan first and then connect.
    // Perhaps the WiFi radio needs some time to stabilize first?
    if (!WiFi_AP_Candidates.hasCandidates()) {
      WifiScan(false, RTC.lastWiFiChannel);
    }
    WiFi_AP_Candidates.clearCache();
    processScanDone();
    WiFi_AP_Candidates.load_knownCredentials();

    if (!WiFi_AP_Candidates.hasCandidates()) {
      #ifndef BUILD_MINIMAL_OTA
      addLog(LOG_LEVEL_INFO, F("Setup: Scan all channels"));
      #endif
      WifiScan(false);
    }

    //    setWifiMode(WIFI_OFF);
  }
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("WifiScan()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER


  //  setWifiMode(WIFI_STA);
  checkRuleSets();
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("checkRuleSets()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER


  // if different version, eeprom settings structure has changed. Full Reset needed
  // on a fresh ESP module eeprom values are set to 255. Version results into -1 (signed int)
  if ((Settings.Version != VERSION) || (Settings.PID != ESP_PROJECT_PID))
  {
    // Direct Serial is allowed here, since this is only an emergency task.
    serialPrint(F("\nPID:"));
    serialPrintln(String(Settings.PID));
    serialPrint(F("Version:"));
    serialPrintln(String(Settings.Version));
    serialPrintln(F("INIT : Incorrect PID or version!"));
    delay(1000);
    ResetFactory();
  }

  initSerial();
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("initSerial()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    addLogMove(LOG_LEVEL_INFO, concat(F("INIT : Free RAM:"), FreeMem()));
  }

#ifndef BUILD_NO_DEBUG

  if (Settings.UseSerial && (Settings.SerialLogLevel >= LOG_LEVEL_DEBUG_MORE)) {
    ESPEasy_Console.setDebugOutput(true);
  }
#endif // ifndef BUILD_NO_DEBUG

  timermqtt_interval      = 100; // Interval for checking MQTT
  timerAwakeFromDeepSleep = millis();
  CPluginInit();
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("CPluginInit()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
  #if FEATURE_NOTIFIER
  NPluginInit();
  # ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("NPluginInit()"));
  # endif
  #endif // if FEATURE_NOTIFIER

  PluginInit();

  initSerial(); // Plugins may have altered serial, so re-init serial

  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("PluginInit()"));
  #endif

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log;
    if (reserve_special(log, 80)) {
      log += concat(F("INFO : Plugins: "), getDeviceCount() + 1);
      log += ' ';
      log += getPluginDescriptionString();
      log += F(" (");
      log += getSystemLibraryString();
      log += ')';
      addLogMove(LOG_LEVEL_INFO, log);
    }
  }

  /*
     if ((getDeviceCount() + 1) >= PLUGIN_MAX) {
      addLog(LOG_LEVEL_ERROR, concat(F("Programming error! - Increase PLUGIN_MAX ("), getDeviceCount()) + ')');
     }
   */

  clearAllCaches();
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("clearAllCaches()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER

  if (Settings.UseRules && isDeepSleepEnabled())
  {
    String event = F("System#NoSleep=");
    event += Settings.deepSleep_wakeTime;
    rulesProcessing(event); // TD-er: Process events in the setup() now.
  }

  if (Settings.UseRules)
  {
    String event = F("System#Wake");
    rulesProcessing(event); // TD-er: Process events in the setup() now.
  }
  #ifdef ESP32

  if (Settings.UseRules)
  {
    const uint32_t gpio_strap =   GPIO_REG_READ(GPIO_STRAP_REG);

    //    BOOT_MODE_GET();

    // Event values:
    // ESP32   :  GPIO-5, GPIO-15, GPIO-4, GPIO-2, GPIO-0, GPIO-12
    // ESP32-C3:  bit 0: GPIO2, bit 2: GPIO8, bit 3: GPIO9
    // ESP32-S2: Unclear what bits represent which strapping state.
    // ESP32-S3: bit5 ~ bit2 correspond to strapping pins GPIO3, GPIO45, GPIO0, and GPIO46 respectively.
    String event = F("System#BootMode=");
    event += bitRead(gpio_strap, 0);
    event += ',';
    event += bitRead(gpio_strap, 1);
    event += ',';
    event += bitRead(gpio_strap, 2);
    event += ',';
    event += bitRead(gpio_strap, 3);
    event += ',';
    event += bitRead(gpio_strap, 4);
    event += ',';
    event += bitRead(gpio_strap, 5);
    rulesProcessing(event);
  }
  #endif // ifdef ESP32

  #if FEATURE_ETHERNET

  if (Settings.ETH_Pin_power_rst != -1) {
    GPIO_Write(PLUGIN_GPIO, Settings.ETH_Pin_power_rst, 1);
  }

  #endif // if FEATURE_ETHERNET

  NetworkConnectRelaxed();
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("NetworkConnectRelaxed()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER

  setWebserverRunning(true);
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("setWebserverRunning()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER


  #if FEATURE_REPORTING
  ReportStatus();
  #endif // if FEATURE_REPORTING

  #if FEATURE_ARDUINO_OTA
  ArduinoOTAInit();
  # ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("ArduinoOTAInit()"));
  # endif
  #endif // if FEATURE_ARDUINO_OTA

  if (node_time.systemTimePresent()) {
    node_time.initTime();
    #ifndef BUILD_NO_RAM_TRACKER
    logMemUsageAfter(F("node_time.initTime()"));
    #endif // ifndef BUILD_NO_RAM_TRACKER
  }

  if (Settings.UseRules)
  {
    String event = F("System#Boot");
    rulesProcessing(event); // TD-er: Process events in the setup() now.
    #ifndef BUILD_NO_RAM_TRACKER
    logMemUsageAfter(F("rulesProcessing(System#Boot)"));
    #endif // ifndef BUILD_NO_RAM_TRACKER
  }

  writeDefaultCSS();
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("writeDefaultCSS()"));
  #endif // ifndef BUILD_NO_RAM_TRACKER


  #ifdef USE_RTOS_MULTITASKING
  UseRTOSMultitasking = Settings.UseRTOSMultitasking;

  if (UseRTOSMultitasking) {
    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLog(LOG_LEVEL_INFO, F("RTOS : Launching tasks"));
    }
    xTaskCreatePinnedToCore(RTOS_TaskServers, "RTOS_TaskServers", 16384, nullptr, 1, nullptr, 1);
    xTaskCreatePinnedToCore(RTOS_TaskSerial,  "RTOS_TaskSerial",  8192,  nullptr, 1, nullptr, 1);
    xTaskCreatePinnedToCore(RTOS_Task10ps,    "RTOS_Task10ps",    8192,  nullptr, 1, nullptr, 1);
    xTaskCreatePinnedToCore(
      RTOS_HandleSchedule,   /* Function to implement the task */
      "RTOS_HandleSchedule", /* Name of the task */
      16384,                 /* Stack size in words */
      nullptr,               /* Task input parameter */
      1,                     /* Priority of the task */
      nullptr,               /* Task handle. */
      1);                    /* Core where the task should run */
  }
  #endif // ifdef USE_RTOS_MULTITASKING

  // Start the interval timers at N msec from now.
  // Make sure to start them at some time after eachother,
  // since they will keep running at the same interval.
  Scheduler.setIntervalTimerOverride(SchedulerIntervalTimer_e::TIMER_20MSEC,     5);    // timer for periodic actions 50 x per/sec
  Scheduler.setIntervalTimerOverride(SchedulerIntervalTimer_e::TIMER_100MSEC,    66);   // timer for periodic actions 10 x per/sec
  Scheduler.setIntervalTimerOverride(SchedulerIntervalTimer_e::TIMER_1SEC,       777);  // timer for periodic actions once per/sec
  Scheduler.setIntervalTimerOverride(SchedulerIntervalTimer_e::TIMER_30SEC,      1333); // timer for watchdog once per 30 sec
  Scheduler.setIntervalTimerOverride(SchedulerIntervalTimer_e::TIMER_MQTT,       88);   // timer for interaction with MQTT
  Scheduler.setIntervalTimerOverride(SchedulerIntervalTimer_e::TIMER_STATISTICS, 2222);
  #ifndef BUILD_NO_RAM_TRACKER
  logMemUsageAfter(F("Scheduler.setIntervalTimerOverride"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
}


#include "../ESPEasyCore/ESPEasyWiFiEvent.h"

#if FEATURE_ETHERNET
#include <ETH.h>
#endif

#include "../DataStructs/RTCStruct.h"

#include "../DataTypes/ESPEasyTimeSource.h"

#include "../ESPEasyCore/ESPEasyEth.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/ESPEasyWifi.h"
#include "../ESPEasyCore/ESPEasyWifi_ProcessEvent.h"

#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/NetworkState.h"
#include "../Globals/RTC.h"
#include "../Globals/WiFi_AP_Candidates.h"

#include "../Helpers/ESPEasy_time_calc.h"


#if FEATURE_ETHERNET
#include "../Globals/ESPEasyEthEvent.h"
#endif


#ifdef ESP32
void setUseStaticIP(bool enabled) {
}

#endif // ifdef ESP32
#ifdef ESP8266
void WiFi_Access_Static_IP::set_use_static_ip(bool enabled) {
  _useStaticIp = enabled;
}
void setUseStaticIP(bool enabled) {
  WiFi_Access_Static_IP tmp_wifi;

  tmp_wifi.set_use_static_ip(enabled);
}

#endif // ifdef ESP8266




// ********************************************************************************
// Functions called on events.
// Make sure not to call anything in these functions that result in delay() or yield()
// ********************************************************************************
#ifdef ESP32
#include <WiFi.h>

static bool ignoreDisconnectEvent = false;

#if ESP_IDF_VERSION_MAJOR > 3
void WiFiEvent(WiFiEvent_t event, arduino_event_info_t info) {
  switch (event) {
    case ARDUINO_EVENT_WIFI_READY:
      // ESP32 WiFi ready
      break;
    case ARDUINO_EVENT_WIFI_STA_START:
    # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event STA Started"));
    #endif
      break;
    case ARDUINO_EVENT_WIFI_STA_STOP:
    # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event STA Stopped"));
    #endif
      break;
    case ARDUINO_EVENT_WIFI_AP_START:
    # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event AP Started"));
    #endif
      break;
    case ARDUINO_EVENT_WIFI_AP_STOP:
    # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event AP Stopped"));
    #endif
      break;
    case ARDUINO_EVENT_WIFI_STA_LOST_IP:
      // ESP32 station lost IP and the IP is reset to 0
      #if FEATURE_ETHERNET
      if (active_network_medium == NetworkMedium_t::Ethernet) {
        // DNS records are shared among WiFi and Ethernet (very bad design!)
        // So we must restore the DNS records for Ethernet in case we started with WiFi and then plugged in Ethernet.
        // As soon as WiFi is turned off, the DNS entry for Ethernet is cleared.
        EthEventData.markLostIP();
      }
      #endif // if FEATURE_ETHERNET
      WiFiEventData.markLostIP();
      # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, 
      /*
        active_network_medium == NetworkMedium_t::Ethernet ?
        F("ETH : Event Lost IP") :
      */
//         F("WiFi : Event Lost IP"));
      #endif
      break;

    case ARDUINO_EVENT_WIFI_AP_PROBEREQRECVED:
      // Receive probe request packet in soft-AP interface
      // TODO TD-er: Must implement like onProbeRequestAPmode for ESP8266
      # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event AP got probed"));
      #endif
      break;

    case ARDUINO_EVENT_WIFI_STA_AUTHMODE_CHANGE:
      #if ESP_IDF_VERSION_MAJOR > 3
      WiFiEventData.setAuthMode(info.wifi_sta_authmode_change.new_mode);
      #else
      WiFiEventData.setAuthMode(info.auth_change.new_mode);
      #endif
      break;

    case ARDUINO_EVENT_WIFI_STA_CONNECTED:
    {
      char ssid_copy[33]; // Ensure space for maximum len SSID (32) plus trailing 0
      #if ESP_IDF_VERSION_MAJOR > 3
      memcpy(ssid_copy, info.wifi_sta_connected.ssid, info.wifi_sta_connected.ssid_len);
      ssid_copy[32] = 0; // Potentially add 0-termination if none present earlier
      WiFiEventData.markConnected((const char*) ssid_copy, info.wifi_sta_connected.bssid, info.wifi_sta_connected.channel);
      WiFiEventData.setAuthMode(info.wifi_sta_connected.authmode);
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event WIFI_STA_CONNECTED"));
      #else
      memcpy(ssid_copy, info.connected.ssid, info.connected.ssid_len);
      ssid_copy[32] = 0; // Potentially add 0-termination if none present earlier
      WiFiEventData.markConnected((const char*) ssid_copy, info.connected.bssid, info.connected.channel);
      #endif
      break;
    }
    case ARDUINO_EVENT_WIFI_STA_DISCONNECTED:
      if (!ignoreDisconnectEvent) {
        ignoreDisconnectEvent = true;
        #if ESP_IDF_VERSION_MAJOR > 3
        WiFiEventData.markDisconnect(static_cast<WiFiDisconnectReason>(info.wifi_sta_disconnected.reason));
        if (info.wifi_sta_disconnected.reason == WIFI_REASON_AUTH_EXPIRE) {
          // See: https://github.com/espressif/arduino-esp32/issues/8877#issuecomment-1807677897
          #if ESP_IDF_VERSION_MAJOR >= 5
          // FIXME TD-er: Should no longer be needed.
          WiFi.STA._setStatus(WL_CONNECTION_LOST);
          #else
          WiFiSTAClass::_setStatus(WL_CONNECTION_LOST);
          #endif
        }
        #else
        WiFiEventData.markDisconnect(static_cast<WiFiDisconnectReason>(info.disconnected.reason));
        if (info.disconnected.reason == WIFI_REASON_AUTH_EXPIRE) {
          // See: https://github.com/espressif/arduino-esp32/issues/8877#issuecomment-1807677897
          WiFiSTAClass::_setStatus(WL_CONNECTION_LOST);
        }
        #endif
        WiFi.persistent(false);
        WiFi.disconnect(true);
      }
      break;
    case ARDUINO_EVENT_WIFI_STA_GOT_IP:
      ignoreDisconnectEvent = false;
      WiFiEventData.markGotIP();
      break;
    #if FEATURE_USE_IPV6
    case ARDUINO_EVENT_WIFI_STA_GOT_IP6:
    {
      ip_event_got_ip6_t * event = static_cast<ip_event_got_ip6_t*>(&info.got_ip6);
      const IPAddress ip(IPv6, (const uint8_t*)event->ip6_info.ip.addr, event->ip6_info.ip.zone);
      WiFiEventData.markGotIPv6(ip);
      break;
    }
    case ARDUINO_EVENT_WIFI_AP_GOT_IP6:
      addLog(LOG_LEVEL_INFO, F("WIFI : AP got IP6"));
      break;
    #endif
    case ARDUINO_EVENT_WIFI_AP_STACONNECTED:
      #if ESP_IDF_VERSION_MAJOR > 3
      WiFiEventData.markConnectedAPmode(info.wifi_ap_staconnected.mac);
      #else
      WiFiEventData.markConnectedAPmode(info.sta_connected.mac);
      #endif
      break;
    case ARDUINO_EVENT_WIFI_AP_STADISCONNECTED:
      #if ESP_IDF_VERSION_MAJOR > 3
      WiFiEventData.markDisconnectedAPmode(info.wifi_ap_stadisconnected.mac);
      #else
      WiFiEventData.markDisconnectedAPmode(info.sta_disconnected.mac);
      #endif
      break;
    case ARDUINO_EVENT_WIFI_SCAN_DONE:
      WiFiEventData.processedScanDone = false;
      break;
#if FEATURE_ETHERNET
    case ARDUINO_EVENT_ETH_START:
    case ARDUINO_EVENT_ETH_CONNECTED:
    case ARDUINO_EVENT_ETH_GOT_IP:
    case ARDUINO_EVENT_ETH_DISCONNECTED:
    case ARDUINO_EVENT_ETH_STOP:
    #if ESP_IDF_VERSION_MAJOR > 3
    case ARDUINO_EVENT_ETH_GOT_IP6:
    #else
    case ARDUINO_EVENT_GOT_IP6:
    #endif
      // Handled in EthEvent
      break;
#endif //FEATURE_ETHERNET
    default:
      {

        // addLogMove(LOG_LEVEL_ERROR, concat(F("UNKNOWN WIFI/ETH EVENT: "),  event));
      }
      break;
  }
}
#else
void WiFiEvent(system_event_id_t event, system_event_info_t info) {
  switch (event) {
    case SYSTEM_EVENT_WIFI_READY:
      // ESP32 WiFi ready
      break;
    case SYSTEM_EVENT_STA_START:
    # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event STA Started"));
    #endif
      break;
    case SYSTEM_EVENT_STA_STOP:
    # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event STA Stopped"));
    #endif
      break;
    case SYSTEM_EVENT_AP_START:
    # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event AP Started"));
    #endif
      break;
    case SYSTEM_EVENT_AP_STOP:
    # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event AP Stopped"));
    #endif
      break;
    case SYSTEM_EVENT_STA_LOST_IP:
      // ESP32 station lost IP and the IP is reset to 0
      #if FEATURE_ETHERNET
      if (active_network_medium == NetworkMedium_t::Ethernet) {
        EthEventData.markLostIP();
      }
      else
      #endif // if FEATURE_ETHERNET
      WiFiEventData.markLostIP();
      # ifndef BUILD_NO_DEBUG
      /*
      addLog(LOG_LEVEL_INFO, 
        active_network_medium == NetworkMedium_t::Ethernet ?
        F("ETH : Event Lost IP") : F("WiFi : Event Lost IP"));
      */
      #endif
      break;

    case SYSTEM_EVENT_AP_PROBEREQRECVED:
      // Receive probe request packet in soft-AP interface
      // TODO TD-er: Must implement like onProbeRequestAPmode for ESP8266
      # ifndef BUILD_NO_DEBUG
      //addLog(LOG_LEVEL_INFO, F("WiFi : Event AP got probed"));
      #endif
      break;

    case SYSTEM_EVENT_STA_AUTHMODE_CHANGE:
      #if ESP_IDF_VERSION_MAJOR > 3
      WiFiEventData.setAuthMode(info.wifi_sta_authmode_change.new_mode);
      #else
      WiFiEventData.setAuthMode(info.auth_change.new_mode);
      #endif
      break;

    case SYSTEM_EVENT_STA_CONNECTED:
    {
      char ssid_copy[33] = { 0 }; // Ensure space for maximum len SSID (32) plus trailing 0
      #if ESP_IDF_VERSION_MAJOR > 3
      memcpy(ssid_copy, info.wifi_sta_connected.ssid, info.wifi_sta_connected.ssid_len);
      ssid_copy[32] = 0; // Potentially add 0-termination if none present earlier
      WiFiEventData.markConnected((const char*) ssid_copy, info.wifi_sta_connected.bssid, info.wifi_sta_connected.channel);
      #else
      memcpy(ssid_copy, info.connected.ssid, info.connected.ssid_len);
      ssid_copy[32] = 0; // Potentially add 0-termination if none present earlier
      WiFiEventData.markConnected((const char*) ssid_copy, info.connected.bssid, info.connected.channel);
      #endif
      break;
    }
    case SYSTEM_EVENT_STA_DISCONNECTED:
      if (!ignoreDisconnectEvent) {
        ignoreDisconnectEvent = true;
        #if ESP_IDF_VERSION_MAJOR > 3
        WiFiEventData.markDisconnect(static_cast<WiFiDisconnectReason>(info.wifi_sta_disconnected.reason));
        #else
        WiFiEventData.markDisconnect(static_cast<WiFiDisconnectReason>(info.disconnected.reason));
        #endif
        WiFi.persistent(false);
        WiFi.disconnect(true);
      }
      break;
    case SYSTEM_EVENT_STA_GOT_IP:
      ignoreDisconnectEvent = false;
      WiFiEventData.markGotIP();
      break;
    case SYSTEM_EVENT_AP_STACONNECTED:
      #if ESP_IDF_VERSION_MAJOR > 3
      WiFiEventData.markConnectedAPmode(info.wifi_ap_staconnected.mac);
      #else
      WiFiEventData.markConnectedAPmode(info.sta_connected.mac);
      #endif
      break;
    case SYSTEM_EVENT_AP_STADISCONNECTED:
      #if ESP_IDF_VERSION_MAJOR > 3
      WiFiEventData.markDisconnectedAPmode(info.wifi_ap_stadisconnected.mac);
      #else
      WiFiEventData.markDisconnectedAPmode(info.sta_disconnected.mac);
      #endif
      break;
    case SYSTEM_EVENT_SCAN_DONE:
      WiFiEventData.processedScanDone = false;
      break;
#if FEATURE_ETHERNET
    case SYSTEM_EVENT_ETH_START:
      if (ethPrepare()) {
        //addLog(LOG_LEVEL_INFO, F("ETH event: Started"));
      } else {
        //addLog(LOG_LEVEL_ERROR, F("ETH event: Could not prepare ETH!"));
      }
      break;
    case SYSTEM_EVENT_ETH_CONNECTED:
      //addLog(LOG_LEVEL_INFO, F("ETH event: Connected"));
      EthEventData.markConnected();
      break;
    case SYSTEM_EVENT_ETH_GOT_IP:
      EthEventData.markGotIP();
      //addLog(LOG_LEVEL_INFO, F("ETH event: Got IP"));
      break;
    case SYSTEM_EVENT_ETH_DISCONNECTED:
      //addLog(LOG_LEVEL_ERROR, F("ETH event: Disconnected"));
      EthEventData.markDisconnect();
      break;
    case SYSTEM_EVENT_ETH_STOP:
      //addLog(LOG_LEVEL_INFO, F("ETH event: Stopped"));
      break;
    case SYSTEM_EVENT_GOT_IP6:
      //addLog(LOG_LEVEL_INFO, F("ETH event: Got IP6"));
      break;
#endif //FEATURE_ETHERNET
    default:
      {
        //addLogMove(LOG_LEVEL_ERROR, concat(F("UNKNOWN WIFI/ETH EVENT: "), event));
      }
      break;
  }
}

#endif

#endif // ifdef ESP32

#ifdef ESP8266

void onConnected(const WiFiEventStationModeConnected& event) {
  WiFiEventData.markConnected(event.ssid, event.bssid, event.channel);
}

void onDisconnect(const WiFiEventStationModeDisconnected& event) {
  WiFiEventData.markDisconnect(event.reason);
  if (WiFi.status() == WL_CONNECTED) {
    // See https://github.com/esp8266/Arduino/issues/5912
    WiFi.persistent(false);
    WiFi.disconnect(false);
    delay(0);
  }
}

void onGotIP(const WiFiEventStationModeGotIP& event) {
  WiFiEventData.markGotIP();
}

void onDHCPTimeout() {
  WiFiEventData.processedDHCPTimeout = false;
}

void onConnectedAPmode(const WiFiEventSoftAPModeStationConnected& event) {
  WiFiEventData.markConnectedAPmode(event.mac);
}

void onDisconnectedAPmode(const WiFiEventSoftAPModeStationDisconnected& event) {
  WiFiEventData.markDisconnectedAPmode(event.mac);
}

void onStationModeAuthModeChanged(const WiFiEventStationModeAuthModeChanged& event) {
  WiFiEventData.setAuthMode(event.newMode);
}

#if FEATURE_ESP8266_DIRECT_WIFI_SCAN
void onWiFiScanDone(void *arg, STATUS status) {
  if (status == OK) {
    auto *head = reinterpret_cast<bss_info *>(arg);
    int scanCount = 0;
    for (bss_info *it = head; it != nullptr; it = STAILQ_NEXT(it, next)) {
      WiFi_AP_Candidates.process_WiFiscan(*it);
      ++scanCount;
    }
    WiFi_AP_Candidates.after_process_WiFiscan();
    WiFiEventData.lastGetScanMoment.setNow();
//    WiFiEventData.processedScanDone = true;
# ifndef BUILD_NO_DEBUG
/*
    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLogMove(LOG_LEVEL_INFO, concat(F("WiFi : Scan finished (ESP8266), found: "), scanCount));
    }
*/
#endif
    WiFi_AP_Candidates.load_knownCredentials();
    if (WiFi_AP_Candidates.addedKnownCandidate() || !NetworkConnected()) {
      WiFiEventData.wifiConnectAttemptNeeded = true;
      # ifndef BUILD_NO_DEBUG
      if (WiFi_AP_Candidates.addedKnownCandidate()) {
        //addLog(LOG_LEVEL_INFO, F("WiFi : Added known candidate, try to connect"));
      }
      #endif
      NetworkConnectRelaxed();
    }

  }

  WiFiMode_t mode = WiFi.getMode();
  setWifiMode(WIFI_OFF);
  delay(1);
  setWifiMode(mode);
  delay(1);
}
#endif

#endif // ifdef ESP8266

#include "../ESPEasyCore/ESPEasyEth.h"

#if FEATURE_ETHERNET

#include "../CustomBuild/ESPEasyLimits.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"
#include "../ESPEasyCore/ESPEasyWifi.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/ESPEasyGPIO.h"
#include "../ESPEasyCore/ESPEasyEthEvent.h"
#include "../Globals/ESPEasyEthEvent.h"
#include "../Globals/NetworkState.h"
#include "../Globals/Settings.h"
#include "../Helpers/Hardware_GPIO.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/Networking.h"

#include <ETH.h>
#include <lwip/dns.h>
#if ESP_IDF_VERSION_MAJOR > 3
 #include <esp_eth_phy.h>
#else
 #include <eth_phy/phy.h>
#endif

#include <WiFi.h>

bool ethUseStaticIP() {
  return Settings.ETH_IP[0] != 0 && Settings.ETH_IP[0] != 255;
}

void ethSetupStaticIPconfig() {
  const IPAddress IP_zero(0, 0, 0, 0); 
  if (!ethUseStaticIP()) { 
    if (!ETH.config(IP_zero, IP_zero, IP_zero, IP_zero)) {
      addLog(LOG_LEVEL_ERROR, F("ETH  : Cannot set IP config"));
    }
    return; 
  }
  const IPAddress ip     = Settings.ETH_IP;
  const IPAddress gw     = Settings.ETH_Gateway;
  const IPAddress subnet = Settings.ETH_Subnet;
  const IPAddress dns    = Settings.ETH_DNS;

  EthEventData.dns0_cache = dns;
  EthEventData.dns1_cache = IP_zero;


  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log = F("ETH IP   : Static IP : ");
    log += formatIP(ip);
    log += F(" GW: ");
    log += formatIP(gw);
    log += F(" SN: ");
    log += formatIP(subnet);
    log += F(" DNS: ");
    log += formatIP(dns);
    addLogMove(LOG_LEVEL_INFO, log);
  }
  ETH.config(ip, gw, subnet, dns);
  setDNS(0, EthEventData.dns0_cache);
  setDNS(1, EthEventData.dns1_cache);
}

bool ethCheckSettings() {
  return isValid(Settings.ETH_Phy_Type) 
#if CONFIG_ETH_USE_ESP32_EMAC
      && (isValid(Settings.ETH_Clock_Mode)/* || isSPI_EthernetType(Settings.ETH_Phy_Type)*/)
#endif
      && isValid(Settings.NetworkMedium)
      && validGpio(Settings.ETH_Pin_mdc_cs)
      && (isSPI_EthernetType(Settings.ETH_Phy_Type) ||
          ( validGpio(Settings.ETH_Pin_mdio_irq) && 
            (validGpio(Settings.ETH_Pin_power_rst) || (Settings.ETH_Pin_power_rst == -1))
          )
        ); // Some boards have fixed power
}

bool ethPrepare() {
  char hostname[40];
  safe_strncpy(hostname, NetworkCreateRFCCompliantHostname().c_str(), sizeof(hostname));
  ETH.setHostname(hostname);
  ethSetupStaticIPconfig();
  return true;
}

void ethPrintSettings() {
  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log;
    if (log.reserve(115)) {
//    log += F("ETH/Wifi mode: ");
//    log += toString(active_network_medium);
      log += F("ETH PHY Type: ");
      log += toString(Settings.ETH_Phy_Type);
      log += F(" PHY Addr: ");
      log += Settings.ETH_Phy_Addr;

      if (!isSPI_EthernetType(Settings.ETH_Phy_Type)) {
        log += F(" Eth Clock mode: ");
        log += toString(Settings.ETH_Clock_Mode);
      }
      log += strformat(isSPI_EthernetType(Settings.ETH_Phy_Type) 
        ? F(" CS: %d IRQ: %d RST: %d") : F(" MDC: %d MIO: %d PWR: %d"),
        Settings.ETH_Pin_mdc_cs,
        Settings.ETH_Pin_mdio_irq,
        Settings.ETH_Pin_power_rst);
      addLogMove(LOG_LEVEL_INFO, log);
    }
  }
}

MAC_address ETHMacAddress() {
  MAC_address mac;
  if(!EthEventData.ethInitSuccess) {
    addLog(LOG_LEVEL_ERROR, F("Call NetworkMacAddress() only on connected Ethernet!"));
  } else {
    #if ESP_IDF_VERSION_MAJOR > 3
    ETH.macAddress(mac.mac);
    #else
    esp_eth_get_mac(mac.mac);
    #endif
  }
  return mac;
}

void removeEthEventHandler()
{
  WiFi.removeEvent(EthEventData.wm_event_id);
  EthEventData.wm_event_id = 0;
}

void registerEthEventHandler()
{
  if (EthEventData.wm_event_id != 0) {
    removeEthEventHandler();
  }
  EthEventData.wm_event_id = WiFi.onEvent(EthEvent);
}


bool ETHConnectRelaxed() {
  if (EthEventData.ethInitSuccess) {
    return EthLinkUp();
  }
  ethPrintSettings();
  if (!ethCheckSettings())
  {
    addLog(LOG_LEVEL_ERROR, F("ETH: Settings not correct!!!"));
    EthEventData.ethInitSuccess = false;
    return false;
  }
  // Re-register event listener
  removeEthEventHandler();

  ethPower(true);
  EthEventData.markEthBegin();

  // Re-register event listener
  registerEthEventHandler();

  if (!EthEventData.ethInitSuccess) {
#if ESP_IDF_VERSION_MAJOR < 5
    EthEventData.ethInitSuccess = ETH.begin( 
      Settings.ETH_Phy_Addr,
      Settings.ETH_Pin_power_rst,
      Settings.ETH_Pin_mdc_cs,
      Settings.ETH_Pin_mdio_irq,
      (eth_phy_type_t)Settings.ETH_Phy_Type,
      (eth_clock_mode_t)Settings.ETH_Clock_Mode);
#else
#if FEATURE_USE_IPV6
    if (Settings.EnableIPv6()) {
      ETH.enableIPv6(true);
    }
#endif

    if (isSPI_EthernetType(Settings.ETH_Phy_Type)) {
      spi_host_device_t SPI_host = Settings.getSPI_host();
      if (SPI_host == spi_host_device_t::SPI_HOST_MAX) {
        addLog(LOG_LEVEL_ERROR, F("SPI not enabled"));
        #ifdef ESP32C3
        // FIXME TD-er: Fallback for ETH01-EVO board
        SPI_host = spi_host_device_t::SPI2_HOST;
        Settings.InitSPI = static_cast<int>(SPI_Options_e::UserDefined);
        Settings.SPI_SCLK_pin = 7;
        Settings.SPI_MISO_pin = 3;
        Settings.SPI_MOSI_pin = 10;
        #endif
      }
      // else 
      {
#if ETH_SPI_SUPPORTS_CUSTOM
        EthEventData.ethInitSuccess = ETH.begin( 
          to_ESP_phy_type(Settings.ETH_Phy_Type),
          Settings.ETH_Phy_Addr,
          Settings.ETH_Pin_mdc_cs,
          Settings.ETH_Pin_mdio_irq,
          Settings.ETH_Pin_power_rst,
          SPI);
#else
        EthEventData.ethInitSuccess = ETH.begin( 
          to_ESP_phy_type(Settings.ETH_Phy_Type),
          Settings.ETH_Phy_Addr,
          Settings.ETH_Pin_mdc_cs,
          Settings.ETH_Pin_mdio_irq,
          Settings.ETH_Pin_power_rst,
          SPI_host,
          static_cast<int>(Settings.SPI_SCLK_pin),
          static_cast<int>(Settings.SPI_MISO_pin),
          static_cast<int>(Settings.SPI_MOSI_pin));
#endif
      }
    } else {
# if CONFIG_ETH_USE_ESP32_EMAC
    ethResetGPIOpins();
    EthEventData.ethInitSuccess = ETH.begin( 
      to_ESP_phy_type(Settings.ETH_Phy_Type),
      Settings.ETH_Phy_Addr,
      Settings.ETH_Pin_mdc_cs,
      Settings.ETH_Pin_mdio_irq,
      Settings.ETH_Pin_power_rst,
      (eth_clock_mode_t)Settings.ETH_Clock_Mode);
#endif
    }

#endif
  }
  if (EthEventData.ethInitSuccess) {
    // FIXME TD-er: Not sure if this is correctly set to false
    //EthEventData.ethConnectAttemptNeeded = false;

    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
#if ESP_IDF_VERSION_MAJOR < 5
      addLog(LOG_LEVEL_INFO, strformat(
        F("ETH  : MAC: %s speed: %dM %s Link: %s"),
        ETH.macAddress().c_str(),
        ETH.linkSpeed(),
        String(ETH.fullDuplex() ? F("Full Duplex") : F("Half Duplex")).c_str(),
        String(ETH.linkUp() ? F("Up") : F("Down")).c_str()));
#else
      addLog(LOG_LEVEL_INFO, strformat(
        F("ETH  : MAC: %s phy addr: %d speed: %dM %s Link: %s"),
        ETH.macAddress().c_str(),
        ETH.phyAddr(),
        ETH.linkSpeed(),
        concat(
          ETH.fullDuplex() ? F("Full Duplex") : F("Half Duplex"),
          ETH.autoNegotiation() ? F("(auto)") : F("")).c_str(),
        String(ETH.linkUp() ? F("Up") : F("Down")).c_str()));
#endif
    }
    
    if (EthLinkUp()) {
      // We might miss the connected event, since we are already connected.
      EthEventData.markConnected();
    }
  } else {
    addLog(LOG_LEVEL_ERROR, F("ETH  : Failed to initialize ETH"));
  }
  return EthEventData.ethInitSuccess;
}

void ethPower(bool enable) {
  if (isSPI_EthernetType(Settings.ETH_Phy_Type)) 
    return;
  if (Settings.ETH_Pin_power_rst != -1) {
    if (GPIO_Internal_Read(Settings.ETH_Pin_power_rst) == enable) {
      // Already the desired state
      return;
    }
    addLog(LOG_LEVEL_INFO, enable ? F("ETH power ON") : F("ETH power OFF"));
    if (!enable) {
      EthEventData.ethInitSuccess = false;
      EthEventData.clearAll();
      #ifdef ESP_IDF_VERSION_MAJOR
      // FIXME TD-er: See: https://github.com/espressif/arduino-esp32/issues/6105
      // Need to store the last link state, as it will be cleared after destructing the object.
      EthEventData.setEthDisconnected();
      if (ETH.linkUp()) {
        EthEventData.setEthConnected();
      }
      #endif
//      ETH = ETHClass();
    }
    if (enable) {
//      ethResetGPIOpins();
    }
//    gpio_reset_pin((gpio_num_t)Settings.ETH_Pin_power_rst);

    GPIO_Write(PLUGIN_GPIO, Settings.ETH_Pin_power_rst, enable ? 1 : 0);
    if (!enable) {
      if (Settings.ETH_Clock_Mode == EthClockMode_t::Ext_crystal_osc) {
        delay(600); // Give some time to discharge any capacitors
        // Delay is needed to make sure no clock signal remains present which may cause the ESP to boot into flash mode.
      }
    } else {
      delay(400); // LAN chip needs to initialize before calling Eth.begin()
    }
  }
}

void ethResetGPIOpins() {
  if (isSPI_EthernetType(Settings.ETH_Phy_Type)) 
    return;

  // fix an disconnection issue after rebooting Olimex POE - this forces a clean state for all GPIO involved in RMII
  // Thanks to @s-hadinger and @Jason2866
  // Resetting state of power pin is done in ethPower()
  addLog(LOG_LEVEL_INFO, F("ethResetGPIOpins()"));
  gpio_reset_pin((gpio_num_t)Settings.ETH_Pin_mdc_cs);
  gpio_reset_pin((gpio_num_t)Settings.ETH_Pin_mdio_irq);
# if CONFIG_ETH_USE_ESP32_EMAC
  gpio_reset_pin(GPIO_NUM_19);    // EMAC_TXD0 - hardcoded
  gpio_reset_pin(GPIO_NUM_21);    // EMAC_TX_EN - hardcoded
  gpio_reset_pin(GPIO_NUM_22);    // EMAC_TXD1 - hardcoded
  gpio_reset_pin(GPIO_NUM_25);    // EMAC_RXD0 - hardcoded
  gpio_reset_pin(GPIO_NUM_26);    // EMAC_RXD1 - hardcoded
  gpio_reset_pin(GPIO_NUM_27);    // EMAC_RX_CRS_DV - hardcoded
#endif
  /*
  switch (Settings.ETH_Clock_Mode) {
    case EthClockMode_t::Ext_crystal_osc:       // ETH_CLOCK_GPIO0_IN
    case EthClockMode_t::Int_50MHz_GPIO_0:      // ETH_CLOCK_GPIO0_OUT
      gpio_reset_pin(GPIO_NUM_0);
      break;
    case EthClockMode_t::Int_50MHz_GPIO_16:     // ETH_CLOCK_GPIO16_OUT
      gpio_reset_pin(GPIO_NUM_16);
      break;
    case EthClockMode_t::Int_50MHz_GPIO_17_inv: // ETH_CLOCK_GPIO17_OUT
      gpio_reset_pin(GPIO_NUM_17);
      break;
  }
  */
  delay(1);
}

bool ETHConnected() {
  if (EthEventData.EthServicesInitialized()) {
    if (EthLinkUp()) {
      return true;
    }
    // Apparently we missed an event
    EthEventData.processedDisconnect = false;
  } else if (EthEventData.ethInitSuccess) {
    if (EthLinkUp()) {
      EthEventData.setEthConnected();
      if (NetworkLocalIP() != IPAddress(0, 0, 0, 0) && 
          !EthEventData.EthGotIP()) {
        EthEventData.processedGotIP = false;
      }
      if (EthEventData.lastConnectMoment.isSet()) {
        if (!EthEventData.EthServicesInitialized()) {
          if (EthEventData.lastConnectMoment.millisPassedSince() > 10000 &&
              EthEventData.lastGetIPmoment.isSet()) {
            EthEventData.processedGotIP = false;
            EthEventData.markLostIP();
          }
        }
      }
      return EthEventData.EthServicesInitialized();
    } else {
      if (EthEventData.last_eth_connect_attempt_moment.isSet() && 
          EthEventData.last_eth_connect_attempt_moment.millisPassedSince() < 5000) {
        return false;
      }
      setNetworkMedium(NetworkMedium_t::WIFI);
    }
  }
  return false;
}

#endif // if FEATURE_ETHERNET
#include "../ESPEasyCore/ESPEasy_Log.h"

#include "../DataStructs/LogStruct.h"
#include "../ESPEasyCore/Serial.h"
#include "../Globals/Cache.h"
#include "../Globals/ESPEasy_Console.h"
#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/Logging.h"
#include "../Globals/Settings.h"
#include "../Helpers/Networking.h"
#include "../Helpers/StringConverter.h"

#include <FS.h>

#if FEATURE_SD
#include <SD.h>
#include "../Helpers/ESPEasy_Storage.h"
#endif

#define UPDATE_LOGLEVEL_ACTIVE_CACHE_INTERVAL 5000

/********************************************************************************************\
  Init critical variables for logging (important during initial factory reset stuff )
  \*********************************************************************************************/
void initLog()
{
  //make sure addLog doesnt do any stuff before initalisation of Settings is complete.
  Settings.UseSerial=true;
  Settings.SyslogFacility=0;
  setLogLevelFor(LOG_TO_SYSLOG, 0);
  setLogLevelFor(LOG_TO_SERIAL, 2); //logging during initialisation
  setLogLevelFor(LOG_TO_WEBLOG, 2);
  setLogLevelFor(LOG_TO_SDCARD, 0);
}


/********************************************************************************************\
  Logging
  \*********************************************************************************************/
const __FlashStringHelper * getLogLevelDisplayString(int logLevel) {
  switch (logLevel) {
    case LOG_LEVEL_NONE:       return F("None");
    case LOG_LEVEL_ERROR:      return F("Error");
    case LOG_LEVEL_INFO:       return F("Info");
# ifndef BUILD_NO_DEBUG
    case LOG_LEVEL_DEBUG:      return F("Debug");
    case LOG_LEVEL_DEBUG_MORE: return F("Debug More");
    case LOG_LEVEL_DEBUG_DEV:  return F("Debug dev");
#endif

    default:
    break;
  }
  return F("");
}

const __FlashStringHelper * getLogLevelDisplayStringFromIndex(uint8_t index, int& logLevel) {
  switch (index) {
    case 0: logLevel = LOG_LEVEL_ERROR;      break;
    case 1: logLevel = LOG_LEVEL_INFO;       break;
# ifndef BUILD_NO_DEBUG
    case 2: logLevel = LOG_LEVEL_DEBUG;      break;
    case 3: logLevel = LOG_LEVEL_DEBUG_MORE; break;
    case 4: logLevel = LOG_LEVEL_DEBUG_DEV;  break;
#endif

    default: logLevel = -1; return F("");
  }
  return getLogLevelDisplayString(logLevel);
}

void disableSerialLog() {
  log_to_serial_disabled = true;
  setLogLevelFor(LOG_TO_SERIAL, 0);
}

void setLogLevelFor(uint8_t destination, uint8_t logLevel) {
  switch (destination) {
    case LOG_TO_SERIAL:
      if (!log_to_serial_disabled || logLevel == 0) {
        Settings.SerialLogLevel = logLevel; 
      }
      break;
    case LOG_TO_SYSLOG: Settings.SyslogLevel = logLevel;    break;
    case LOG_TO_WEBLOG: Settings.WebLogLevel = logLevel;    break;
    case LOG_TO_SDCARD: Settings.SDLogLevel = logLevel;     break;
    default:
      break;
  }
  updateLogLevelCache();
}

void updateLogLevelCache() {
  uint8_t max_lvl = 0;
  // FIXME TD-er: Must add check whether SW serial may be using the same pins as Serial0
  const bool useSerial = Settings.UseSerial && !activeTaskUseSerial0();
  if (log_to_serial_disabled) {
    if (useSerial) {
      ESPEasy_Console.setDebugOutput(false);
    }
  } else {
    max_lvl = _max(max_lvl, Settings.SerialLogLevel);
#ifndef BUILD_NO_DEBUG
    if (useSerial && Settings.SerialLogLevel >= LOG_LEVEL_DEBUG_MORE) {
      ESPEasy_Console.setDebugOutput(true);
    }
#endif
  }
  max_lvl = _max(max_lvl, Settings.SyslogLevel);
  if (Logging.logActiveRead()) {
    max_lvl = _max(max_lvl, Settings.WebLogLevel);
  }
#if FEATURE_SD
  max_lvl = _max(max_lvl, Settings.SDLogLevel);
#endif
  highest_active_log_level = max_lvl;
}

bool loglevelActiveFor(uint8_t logLevel) {
  #ifdef ESP32
  if (xPortInIsrContext()) {
    // When called from an ISR, you should not send out logs.
    // Allocating memory from within an ISR is a big no-no.
    // Also long-time blocking like sending logs (especially to a syslog server) 
    // is also really not a good idea from an ISR call.
    return false;
  }
  #endif
  static uint32_t lastUpdateLogLevelCache = 0;
  if (lastUpdateLogLevelCache == 0 || 
      timePassedSince(lastUpdateLogLevelCache) > UPDATE_LOGLEVEL_ACTIVE_CACHE_INTERVAL)
  {
    lastUpdateLogLevelCache = millis();
    updateLogLevelCache();
  }
  return logLevel <= highest_active_log_level;
}

uint8_t getSerialLogLevel() {
#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
  // FIXME TD-er: Must add check whether SW serial may be using the same pins as Serial0
  if (log_to_serial_disabled || !Settings.UseSerial) return 0;
#else
  if (log_to_serial_disabled || !Settings.UseSerial || activeTaskUseSerial0()) return 0;
#endif
  if (!(WiFiEventData.WiFiServicesInitialized())){
    if (Settings.SerialLogLevel < LOG_LEVEL_INFO) {
      return LOG_LEVEL_INFO;
    }
  }
  return Settings.SerialLogLevel;
}

uint8_t getWebLogLevel() {
  if (Logging.logActiveRead()) {
    return Settings.WebLogLevel;
  } 
  if (Settings.WebLogLevel != LOG_LEVEL_NONE) {
    updateLogLevelCache();
  }
  return LOG_LEVEL_NONE;
}

bool loglevelActiveFor(uint8_t destination, uint8_t logLevel) {
  #ifdef ESP32
  if (xPortInIsrContext()) {
    // When called from an ISR, you should not send out logs.
    // Allocating memory from within an ISR is a big no-no.
    // Also long-time blocking like sending logs (especially to a syslog server) 
    // is also really not a good idea from an ISR call.
    return false;
  }
  #endif

  uint8_t logLevelSettings = 0;
  switch (destination) {
    case LOG_TO_SERIAL: {
      logLevelSettings = getSerialLogLevel();
      break;
    }
    case LOG_TO_SYSLOG: {
      logLevelSettings = Settings.SyslogLevel;
      break;
    }
    case LOG_TO_WEBLOG: {
      logLevelSettings = getWebLogLevel();
      break;
    }
    case LOG_TO_SDCARD: {
      #if FEATURE_SD
      logLevelSettings = Settings.SDLogLevel;
      #endif
      break;
    }
    default:
      return false;
  }
  return logLevel <= logLevelSettings;
}

void addLog(uint8_t logLevel, const __FlashStringHelper *str)
{
  if (loglevelActiveFor(logLevel)) {
    String copy;
    if (!reserve_special(copy, strlen_P((PGM_P)str))) {
      return;
    }
    copy = str;
    addToLogMove(logLevel, std::move(copy));
  }
}

void addLog(uint8_t logLevel, const char *line)
{
  // Please note all functions called from here handling line must be PROGMEM aware.
  if (loglevelActiveFor(logLevel)) {

    String copy;
    #ifdef USE_SECOND_HEAP
    {
      // Allow to store the logs in 2nd heap if present.
      if (mmu_is_iram(line)) {
        size_t length = 0;
        const char* cur_char = line;
        bool copying = false;
        bool done = false;
        while (!done) {
          const uint8_t ch = mmu_get_uint8(cur_char++);
          if (ch == 0) {
            if (copying) {
              done = true;
            } else {
              if (!reserve_special(copy, length)) {
                return;
              }
              copying = true;
              cur_char = line;
            }
          } else {
            if (copying) {
              copy +=  (char)ch;
            } else {
              ++length;
            }
          }
        }
      } else {
        if (!reserve_special(copy, strlen_P((PGM_P)line))) {
          return;
        }
        copy = line;
      }
    }
    #else 
    if (!reserve_special(copy, strlen_P((PGM_P)line))) {
      return;
    }
    copy = line;
    #endif
    addToLogMove(logLevel, std::move(copy));
  }
}

void addLog(uint8_t logLevel, String&& string)
{
  addToLogMove(logLevel, std::move(string));
}


#ifndef LIMIT_BUILD_SIZE
#include "../Helpers/Memory.h"
#endif

void addToSerialLog(uint8_t logLevel, const String& string)
{
  if (loglevelActiveFor(LOG_TO_SERIAL, logLevel)) {
    ESPEasy_Console.addToSerialBuffer(format_msec_duration(millis()));
    #ifndef LIMIT_BUILD_SIZE
    ESPEasy_Console.addToSerialBuffer(strformat(F(" : (%d) "), FreeMem()));
    #endif
    {
      String loglevelDisplayString = getLogLevelDisplayString(logLevel);
      while (loglevelDisplayString.length() < 6) {
        loglevelDisplayString += ' ';
      }
      ESPEasy_Console.addToSerialBuffer(loglevelDisplayString);
    }
    ESPEasy_Console.addToSerialBuffer(F(" : "));
    ESPEasy_Console.addToSerialBuffer(string);
    ESPEasy_Console.addNewlineToSerialBuffer();
  }
}

void addToSysLog(uint8_t logLevel, const String& string)
{
  if (loglevelActiveFor(LOG_TO_SYSLOG, logLevel)) {
    sendSyslog(logLevel, string);
  }
}

void addToSDLog(uint8_t logLevel, const String& string)
{
#if FEATURE_SD
  if (!string.isEmpty() && loglevelActiveFor(LOG_TO_SDCARD, logLevel)) {
    String   logName = patch_fname(F("log.txt"));
    fs::File logFile = SD.open(logName, "a+");
    if (logFile) {
      const size_t stringLength = string.length();
      for (size_t i = 0; i < stringLength; ++i) {
        logFile.print(string[i]);
      }
      logFile.println();
    }
    logFile.close();
  }
#endif
}


void addLog(uint8_t logLevel, const String& string)
{
  #ifdef ESP32
  if (xPortInIsrContext()) {
    // When called from an ISR, you should not send out logs.
    // Allocating memory from within an ISR is a big no-no.
    // Also long-time blocking like sending logs (especially to a syslog server) 
    // is also really not a good idea from an ISR call.
    return;
  }
  #endif

  if (string.isEmpty()) return;
  addToSerialLog(logLevel, string);
  addToSysLog(logLevel, string);
  addToSDLog(logLevel, string);
  if (loglevelActiveFor(LOG_TO_WEBLOG, logLevel)) {
    Logging.add(logLevel, string);
  }
}

void addToLogMove(uint8_t logLevel, String&& string)
{
  #ifdef ESP32
  if (xPortInIsrContext()) {
    // When called from an ISR, you should not send out logs.
    // Allocating memory from within an ISR is a big no-no.
    // Also long-time blocking like sending logs (especially to a syslog server) 
    // is also really not a good idea from an ISR call.
    return;
  }
  #endif

  if (string.isEmpty()) return;
  String tmp;
  move_special(tmp, std::move(string));
  addToSerialLog(logLevel, tmp);
  addToSysLog(logLevel, tmp);
  addToSDLog(logLevel, tmp);

  // May clear the string, so call as last one.
  if (loglevelActiveFor(LOG_TO_WEBLOG, logLevel)) {
    Logging.add(logLevel, std::move(tmp));
  }
}


#include "../ESPEasyCore/ESPEasyGPIO.h"

/****************************************************************************/

// Central functions for GPIO handling
// **************************************************************************/
#include "../../_Plugin_Helper.h"
#include "../ESPEasyCore/ESPEasyRules.h"
#include "../Globals/GlobalMapPortStatus.h"
#include "../Helpers/PortStatus.h"


#include <GPIO_Direct_Access.h>


// ********************************************************************************
// Internal GPIO write
// ********************************************************************************
void GPIO_Internal_Write(int pin, uint8_t value)
{
  if (checkValidPortRange(PLUGIN_GPIO, pin)) {
    const uint32_t key = createKey(PLUGIN_GPIO, pin);
    auto it            = globalMapPortStatus.find(key);

    if (it != globalMapPortStatus.end()) {
      if (it->second.mode == PIN_MODE_PWM) {
        set_Gpio_PWM(pin, value);
      } else {
        // gpio_reset_pin(static_cast<gpio_num_t>(pin));
        pinMode(pin, OUTPUT);
        digitalWrite(pin, value);

        // DIRECT_PINMODE_OUTPUT(pin);
        // DIRECT_pinWrite(pin, value);
      }
    }
  }
}

// ********************************************************************************
// Internal GPIO read
// ********************************************************************************
bool GPIO_Internal_Read(int pin)
{
  if (checkValidPortRange(PLUGIN_GPIO, pin)) {
    return DIRECT_pinRead(pin) != 0u;
  }
  return false;
}

bool GPIO_Read_Switch_State(struct EventStruct *event) {
  const int pin = CONFIG_PIN1;

  if (checkValidPortRange(PLUGIN_GPIO, pin)) {
    const uint32_t key = createKey(PLUGIN_GPIO, pin);

    auto it = globalMapPortStatus.find(key);

    if (it != globalMapPortStatus.end()) {
      return GPIO_Read_Switch_State(pin, it->second.mode);
    }
  }
  return false;
}

bool GPIO_Read_Switch_State(int pin, uint8_t pinMode) {
  bool canRead = false;

  if (checkValidPortRange(PLUGIN_GPIO, pin)) {
    switch (pinMode)
    {
      case PIN_MODE_UNDEFINED:
      case PIN_MODE_INPUT:
      case PIN_MODE_INPUT_PULLUP:
      case PIN_MODE_OUTPUT:
        canRead = true;
        break;

      /*
            case PIN_MODE_PWM:
              break;
            case PIN_MODE_SERVO:
              break;
            case PIN_MODE_OFFLINE:
              break;
       */
      default:
        break;
    }
  }

  if (!canRead) { return false; }

  // Do not read from the pin while mode is set to PWM or servo.
  // See https://github.com/letscontrolit/ESPEasy/issues/2117#issuecomment-443516794
  return DIRECT_pinRead(pin) != 0u;
}

#ifdef USES_P009

// ********************************************************************************
// MCP23017 PIN read
// ********************************************************************************
int8_t GPIO_MCP_Read(int Par1)
{
  int8_t pinState = -1;

  if (checkValidPortRange(PLUGIN_MCP, Par1)) {
    uint8_t unit           = (Par1 - 1) / 16;
    uint8_t port           = Par1 - (unit * 16) - 1;
    uint8_t address        = 0x20 + unit;
    uint8_t IOBankValueReg = (port < 8) ? MCP23017_GPIOA : MCP23017_GPIOB;
    port = port % 8;

    uint8_t retValue;

    if (GPIO_MCP_ReadRegister(address, IOBankValueReg, &retValue)) {
      retValue = (retValue & (1 << port)) >> port;
      pinState = (retValue == 0) ? 0 : 1;

      /*
         // get the current pin status
         Wire.beginTransmission(address);
         Wire.write(IOBankValueReg); // IO data register
         Wire.endTransmission();
         Wire.requestFrom(address, (uint8_t)0x1);
         if (Wire.available())
         {
         state = (Wire.read() & (1 << port)) >> port;
         }
       */
    }
  }
  return pinState;
}

// ********************************************************************************
// MCP23017 read register
// ********************************************************************************

bool GPIO_MCP_ReadRegister(uint8_t mcpAddr, uint8_t regAddr, uint8_t *retValue) {
  bool success = false;

  // Read the register
  const uint8_t value = I2C_read8_reg(mcpAddr, regAddr, &success);

  if (success) {
    *retValue = value;
  }
  return success;
}

// ********************************************************************************
// MCP23017 write register
// ********************************************************************************

void GPIO_MCP_WriteRegister(uint8_t mcpAddr, uint8_t regAddr, uint8_t regValue) {
  // Write the register
  I2C_write8_reg(mcpAddr, regAddr, regValue);
}

// ********************************************************************************
// MCP23017 write pin
// ********************************************************************************
bool GPIO_MCP_Write(int Par1, uint8_t Par2)
{
  if (!checkValidPortRange(PLUGIN_MCP, Par1)) {
    return false;
  }
  bool success            = false;
  uint8_t unit            = (Par1 - 1) / 16;
  uint8_t port            = Par1 - (unit * 16) - 1;
  uint8_t address         = int((Par1 - 1) / 16) + 0x20;
  uint8_t IOBankConfigReg = (port < 8) ? MCP23017_IODIRA : MCP23017_IODIRB;
  uint8_t IOBankValueReg  = (port < 8) ? MCP23017_GPIOA : MCP23017_GPIOB;
  port = port % 8;

  uint8_t retValue;

  // turn this port into output, first read current config
  if (GPIO_MCP_ReadRegister(address, IOBankConfigReg, &retValue)) {
    retValue &= ~(1 << port); // change pin to output (0)

    // write new IO config
    GPIO_MCP_WriteRegister(address, IOBankConfigReg, retValue);

    if (GPIO_MCP_ReadRegister(address, IOBankValueReg, &retValue)) {
      (Par2 == 1) ? retValue |= (1 << port) : retValue &= ~(1 << port);

      // write new IO config
      GPIO_MCP_WriteRegister(address, IOBankValueReg, retValue);
      success = true;
    }
  }
  return success;

  /*
     uint8_t portvalue = 0;
     // turn this port into output, first read current config
     Wire.beginTransmission(address);
     Wire.write(IOBankConfigReg); // IO config register
     Wire.endTransmission();
     Wire.requestFrom(address, (uint8_t)0x1);
     if (Wire.available())
     {
      portvalue = Wire.read();
      portvalue &= ~(1 << port ); // change pin from (default) input to output

      // write new IO config
      Wire.beginTransmission(address);
      Wire.write(IOBankConfigReg); // IO config register
      Wire.write(portvalue);
      Wire.endTransmission();
     }
     // get the current pin status
     Wire.beginTransmission(address);
     Wire.write(IOBankValueReg); // IO data register
     Wire.endTransmission();
     Wire.requestFrom(address, (uint8_t)0x1);
     if (Wire.available())
     {
      portvalue = Wire.read();
      if (Par2 == 1)
        portvalue |= (1 << port);
      else
        portvalue &= ~(1 << port);

      // write back new data
      Wire.beginTransmission(address);
      Wire.write(IOBankValueReg);
      Wire.write(portvalue);
      Wire.endTransmission();
      success = true;
     }
     return(success);
   */
}

// ********************************************************************************
// MCP23017 config
// Par2: 0: Pullup disabled
// Par2: 1: Pullup enabled
// ********************************************************************************
bool setMCPInputAndPullupMode(uint8_t Par1, bool enablePullUp)
{
  if (!checkValidPortRange(PLUGIN_MCP, Par1)) {
    return false;
  }

  bool success            = false;
  uint8_t unit            = (Par1 - 1) / 16;
  uint8_t port            = Par1 - (unit * 16) - 1;
  uint8_t address         = 0x20 + unit;
  uint8_t IOBankPullUpReg = (port < 8) ? MCP23017_GPPUA : MCP23017_GPPUB;
  uint8_t IOBankIODirReg  = (port < 8) ? MCP23017_IODIRA : MCP23017_IODIRB;
  port = port % 8;

  uint8_t retValue;

  // set this port mode to INPUT (bit=1)
  if (GPIO_MCP_ReadRegister(address, IOBankIODirReg, &retValue)) {
    retValue |= (1 << port);
    GPIO_MCP_WriteRegister(address, IOBankIODirReg, retValue);

    // turn this port pullup on or off
    if (GPIO_MCP_ReadRegister(address, IOBankPullUpReg, &retValue)) {
      enablePullUp ? retValue |= (1 << port) : retValue &= ~(1 << port);
      GPIO_MCP_WriteRegister(address, IOBankPullUpReg, retValue);

      success = true;
    }
  }
  return success;
}

bool setMCPOutputMode(uint8_t Par1)
{
  if (!checkValidPortRange(PLUGIN_MCP, Par1)) {
    return false;
  }

  bool success = false;
  uint8_t retValue;
  uint8_t unit           = (Par1 - 1) / 16;
  uint8_t port           = Par1 - (unit * 16) - 1;
  uint8_t address        = 0x20 + unit;
  uint8_t IOBankIODirReg = (port < 8) ? MCP23017_IODIRA : MCP23017_IODIRB;
  port = port % 8;

  // set this port mode to OUTPUT (bit=0)
  if (GPIO_MCP_ReadRegister(address, IOBankIODirReg, &retValue)) {
    retValue &= ~(1 << port);
    GPIO_MCP_WriteRegister(address, IOBankIODirReg, retValue);
    success = true;
  }
  return success;
}

#endif // ifdef USES_P009

#ifdef USES_P019

// ********************************************************************************
// PCF8574 read pin
// ********************************************************************************
// @giig1967g-20181023: changed to int8_t
int8_t GPIO_PCF_Read(int Par1)
{
  int8_t state = -1;

  if (checkValidPortRange(PLUGIN_PCF, Par1)) {
    const uint8_t unit = (Par1 - 1) / 8;
    const uint8_t port = Par1 - (unit * 8) - 1;
    uint8_t address    = 0x20 + unit;

    if (unit > 7) { address += 0x10; }

    // get the current pin status
    bool is_ok          = false;
    const uint8_t value = I2C_read8(address, &is_ok);

    if (is_ok)
    {
      state = ((value & _BV(port)) >> (port));
    }
  }
  return state;
}

bool GPIO_PCF_ReadAllPins(uint8_t address, uint8_t *retValue)
{
  bool success = false;

  const uint8_t value = I2C_read8(address, &success);

  if (success) {
    *retValue = value;
  }
  return success;
}

// ********************************************************************************
// PCF8574 write pin
// *******************************************************************************
void GPIO_PCF_WriteAllPins(uint8_t address, uint8_t value)
{
  I2C_write8(address, value);
}

bool GPIO_PCF_Write(int Par1, uint8_t Par2)
{
  if (!checkValidPortRange(PLUGIN_PCF, Par1)) {
    return false;
  }
  uint8_t unit    = (Par1 - 1) / 8;
  uint8_t port    = Par1 - (unit * 8) - 1;
  uint8_t address = 0x20 + unit;

  if (unit > 7) { address += 0x10; }

  // generate bitmask
  int i            = 0;
  uint8_t portmask = 255;
  unit = unit * 8 + 1; // calculate first pin

  uint32_t key;

  // REMEMBER: all input pins must be set to 1 when writing to the unit
  for (i = 0; i < 8; i++) {
    key = createKey(PLUGIN_PCF, unit + i);

    auto it = globalMapPortStatus.find(key);

    if (it != globalMapPortStatus.end()) {
      if ((it->second.mode == PIN_MODE_OUTPUT) && (it->second.state == 0)) {
        portmask &= ~(1 << i); // set port i = 0
      }
    }
  }

  if (Par2 == 1) {
    portmask |= (1 << port);
  }
  else {
    portmask &= ~(1 << port);
  }

  GPIO_PCF_WriteAllPins(address, portmask);
  return true;
}

bool setPCFInputMode(uint8_t pin)
{
  if (!checkValidPortRange(PLUGIN_PCF, pin)) {
    return false;
  }
  uint8_t unit       = (pin - 1) / 8;
  const uint8_t port = pin - (unit * 8) - 1;
  uint8_t address    = 0x20 + unit;

  if (unit > 7) { address += 0x10; }

  // generate bitmask
  int i = 0;
  uint8_t portmask;

  if (GPIO_PCF_ReadAllPins(address, &portmask)) {
    unit = unit * 8 + 1; // calculate first pin
    uint32_t key;

    // REMEMBER: all input pins must be set to 1 when writing to the unit
    for (i = 0; i < 8; i++) {
      key = createKey(PLUGIN_PCF, unit + i);

      if (!existPortStatus(key) ||
          (existPortStatus(key) &&
           ((globalMapPortStatus[key].mode == PIN_MODE_INPUT_PULLUP) || (globalMapPortStatus[key].mode == PIN_MODE_INPUT))) ||
          (port == i)) {      // set to 1 the PIN to be set as INPUT
        portmask |= (1 << i); // set port i = 1
      }
    }

    GPIO_PCF_WriteAllPins(address, portmask);

    return true;
  } else {
    return false;
  }
}

#endif // ifdef USES_P019


// *********************************************************
// GPIO_Monitor10xSec:
// What it does:
// a) It updates the pinstate structure
// b) Returns an EVENT in case monitor is enabled
// c) Does not update the pinstate state if the port is defined in a task, as it will be picked up by thePLUGIN_TEN_PER_SECOND event
// called from: run10TimesPerSecond in ESPEasy.ino
// *********************************************************

void GPIO_Monitor10xSec()
{
  for (auto it = globalMapPortStatus.begin(); it != globalMapPortStatus.end(); ++it) {
    // only call monitor function if there is the need to
    if (it->second.monitor || it->second.command || it->second.init) {
      const uint16_t   gpioPort              = getPortFromKey(it->first);
      const pluginID_t pluginID              = getPluginFromKey(it->first);
      int8_t currentState                    = -1;
      const __FlashStringHelper *eventString = F("");
      bool caseFound                         = true;

      switch (pluginID.value)
      {
        case PLUGIN_GPIO_INT:
          currentState = GPIO_Read_Switch_State(gpioPort, it->second.mode);
          eventString  = F("GPIO");
          break;
#ifdef USES_P009
        case PLUGIN_MCP_INT:
          currentState = GPIO_MCP_Read(gpioPort);
          eventString  = F("MCP");
          break;
#endif // ifdef USES_P009
#ifdef USES_P019
        case PLUGIN_PCF_INT:
          currentState = GPIO_PCF_Read(gpioPort);
          eventString  = F("PCF");
          break;
#endif // ifdef USES_P019
        default:
          caseFound = false;
          break;
      }

      if (caseFound) {
        if (currentState == -1) {
          it->second.mode = PIN_MODE_OFFLINE;
        } else if (it->second.mode == PIN_MODE_OFFLINE) {
          it->second.mode = PIN_MODE_UNDEFINED;
        }

        if (((it->second.state != currentState) || (it->second.forceMonitor && it->second.monitor))) {
          // update state
          if (!it->second.task) {
            it->second.state = currentState; // update state ONLY if task flag=false otherwise it will not be picked up by 10xSEC function

            // send event if not task, otherwise is sent in the task PLUGIN_TEN_PER_SECOND
            if (it->second.monitor) {
              sendMonitorEvent(eventString, gpioPort, currentState);
            }
          }
        }
      }
      it->second.forceMonitor = 0; // reset flag
    }
  }
}

// prefix should be either "GPIO", "PCF", "MCP"
void sendMonitorEvent(const __FlashStringHelper *prefix, int port, int8_t state)
{
  String eventString = prefix;

  eventString += '#';
  eventString += port;
  eventString += '=';
  eventString += state;
  rulesProcessing(eventString);
}

bool checkValidPortRange(pluginID_t pluginID, int port)
{
  switch (pluginID.value)
  {
    case PLUGIN_GPIO_INT:
      return validGpio(port);

    case PLUGIN_MCP_INT:
    case PLUGIN_PCF_INT:
      return port >= 1 && port <= 128;
  }
  return false;
}

void setInternalGPIOPullupMode(uint8_t port)
{
  if (checkValidPortRange(PLUGIN_GPIO, port)) {
  #if defined(ESP8266)

    if (port == 16) {
      pinMode(port, INPUT_PULLDOWN_16);
    }
    else {
      pinMode(port, INPUT_PULLUP);
    }
  #else // if defined(ESP8266)
    pinMode(port, INPUT_PULLUP);
  #endif // if defined(ESP8266)
  }
}

bool GPIO_Write(pluginID_t pluginID, int port, uint8_t value, uint8_t pinMode)
{
  bool success = true;

  switch (pluginID.value)
  {
    case PLUGIN_GPIO_INT:

      if (pinMode == PIN_MODE_OUTPUT) {
        GPIO_Internal_Write(port, value);
      }
      else {
        success = false;
      }
      break;
    case PLUGIN_MCP_INT:
#ifdef USES_P009
      GPIO_MCP_Write(port, value);
#endif // ifdef USES_P009
      break;
    case PLUGIN_PCF_INT:
#ifdef USES_P019
      GPIO_PCF_Write(port, value);
#endif // ifdef USES_P019
      break;
    default:
      success = false;
  }

  if (success) {
    Scheduler.clearGPIOTimer(pluginID, port);
  }
  return success;
}

bool GPIO_Read(pluginID_t pluginID, int port, int8_t& value)
{
  bool success = true;

  switch (pluginID.value)
  {
    case PLUGIN_GPIO_INT:
      value = GPIO_Internal_Read(port);
      break;
    case PLUGIN_MCP_INT:
#ifdef USES_P009
      value = GPIO_MCP_Read(port);
#endif // ifdef USES_P009
      break;
    case PLUGIN_PCF_INT:
#ifdef USES_P019
      value = GPIO_PCF_Read(port);
#endif // ifdef USES_P019
      break;
    default:
      success = false;
  }
  return success;
}

#include "../ESPEasyCore/Controller.h"

#include "../../ESPEasy_common.h"
#include "../../ESPEasy-Globals.h"

#include "../../_Plugin_Helper.h"

#include "../ControllerQueue/MQTT_queue_element.h"

#include "../CustomBuild/Certificate_CA.h"

#include "../DataStructs/ControllerSettingsStruct.h"
#include "../DataStructs/ESPEasy_EventStruct.h"

#include "../DataTypes/ESPEasy_plugin_functions.h"
#include "../DataTypes/SPI_options.h"

#include "../ESPEasyCore/ESPEasyRules.h"
#include "../ESPEasyCore/Serial.h"

#include "../Globals/CPlugins.h"
#include "../Globals/Device.h"
#include "../Globals/ESPEasyWiFiEvent.h"
#include "../Globals/ESPEasy_Scheduler.h"
#include "../Globals/MQTT.h"
#include "../Globals/Plugins.h"
#include "../Globals/RulesCalculate.h"

#include "../Helpers/_CPlugin_Helper.h"

// #include "../Helpers/Memory.h"
#include "../Helpers/Misc.h"
#include "../Helpers/Network.h"
#include "../Helpers/PeriodicalActions.h"
#include "../Helpers/PortStatus.h"


constexpr pluginID_t PLUGIN_ID_MQTT_IMPORT(37);

// ********************************************************************************
// Interface for Sending to Controllers
// ********************************************************************************
void sendData(struct EventStruct *event, bool sendEvents)
{
  START_TIMER;
  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("sendData"));
  #endif // ifndef BUILD_NO_RAM_TRACKER
  //  LoadTaskSettings(event->TaskIndex);

  if (Settings.UseRules && sendEvents) {
    createRuleEvents(event);
  }

  if (Settings.UseValueLogger && (Settings.InitSPI > static_cast<int>(SPI_Options_e::None)) && (Settings.Pin_sd_cs >= 0)) {
    SendValueLogger(event->TaskIndex);
  }

  //  LoadTaskSettings(event->TaskIndex); // could have changed during background tasks.

  for (controllerIndex_t x = 0; x < CONTROLLER_MAX; x++)
  {
    if (Settings.ControllerEnabled[x] &&
        Settings.TaskDeviceSendData[x][event->TaskIndex] &&
        Settings.Protocol[x])
    {
      event->ControllerIndex = x;
      const protocolIndex_t ProtocolIndex = getProtocolIndex_from_ControllerIndex(event->ControllerIndex);
      event->idx = Settings.TaskDeviceID[x][event->TaskIndex];

      if (validUserVar(event)) {
        String dummy;
        CPluginCall(ProtocolIndex, CPlugin::Function::CPLUGIN_PROTOCOL_SEND, event, dummy);
      }
#ifndef BUILD_NO_DEBUG
      else {
        if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
          String log = F("Invalid value detected for controller ");
          log += getCPluginNameFromProtocolIndex(ProtocolIndex);
          addLogMove(LOG_LEVEL_DEBUG, log);
        }
      }
#endif // ifndef BUILD_NO_DEBUG
    }
  }

  lastSend = millis();
  STOP_TIMER(SEND_DATA_STATS);
}

bool validUserVar(struct EventStruct *event) {
  if (!validTaskIndex(event->TaskIndex)) { return false; }
  const Sensor_VType vtype = event->getSensorType();

  if (isIntegerOutputDataType(vtype) ||
      (vtype == Sensor_VType::SENSOR_TYPE_STRING)) // FIXME TD-er: Must look at length of event->String2 ?
  {
    return true;
  }
  const uint8_t valueCount = getValueCountForTask(event->TaskIndex);

  for (int i = 0; i < valueCount; ++i) {
    if (!UserVar.isValid(event->TaskIndex, i, vtype)) {
      return false;
    }
  }
  return true;
}

#if FEATURE_MQTT

/*********************************************************************************************\
* Handle incoming MQTT messages
\*********************************************************************************************/

// handle MQTT messages
void incoming_mqtt_callback(char *c_topic, uint8_t *b_payload, unsigned int length) {
  statusLED(true);
  controllerIndex_t enabledMqttController = firstEnabledMQTT_ControllerIndex();

  if (!validControllerIndex(enabledMqttController)) {
    addLog(LOG_LEVEL_ERROR, F("MQTT : No enabled MQTT controller"));
    return;
  }

  if (length > MQTT_MAX_PACKET_SIZE)
  {
    addLog(LOG_LEVEL_ERROR, F("MQTT : Ignored too big message"));
    return;
  }

  // TD-er: This one cannot set the TaskIndex, but that may seem to work out.... hopefully.
  protocolIndex_t ProtocolIndex = getProtocolIndex_from_ControllerIndex(enabledMqttController);

  Scheduler.schedule_mqtt_controller_event_timer(
    ProtocolIndex,
    CPlugin::Function::CPLUGIN_PROTOCOL_RECV,
    c_topic, b_payload, length);

  deviceIndex_t DeviceIndex = getDeviceIndex(PLUGIN_ID_MQTT_IMPORT); // Check if P037_MQTTimport is present in the build

  if (validDeviceIndex(DeviceIndex)) {
    //  Here we loop over all tasks and call each 037 plugin with function PLUGIN_MQTT_IMPORT
    for (taskIndex_t taskIndex = 0; taskIndex < TASKS_MAX; taskIndex++)
    {
      if (Settings.TaskDeviceEnabled[taskIndex] && (Settings.getPluginID_for_task(taskIndex) == PLUGIN_ID_MQTT_IMPORT))
      {
        Scheduler.schedule_mqtt_plugin_import_event_timer(
          DeviceIndex, taskIndex, PLUGIN_MQTT_IMPORT,
          c_topic, b_payload, length);
      }
    }
  }
}

/*********************************************************************************************\
* Disconnect from MQTT message broker
\*********************************************************************************************/
void MQTTDisconnect()
{
  if (MQTTclient.connected()) {
    #if FEATURE_MQTT_CONNECT_BACKGROUND
    if (MQTT_task_data.taskHandle) {
      vTaskDelete(MQTT_task_data.taskHandle);
      MQTT_task_data.taskHandle = NULL;
    }
    MQTT_task_data.status = MQTT_connect_status_e::Disconnected;
    #endif // if FEATURE_MQTT_CONNECT_BACKGROUND
    MQTTclient.disconnect();
    addLog(LOG_LEVEL_INFO, F("MQTT : Disconnected from broker"));
  }
  updateMQTTclient_connected();
}

bool MQTTConnect_prepareClient(controllerIndex_t controller_idx) {
  MakeControllerSettings(ControllerSettings); // -V522

  if (!AllocatedControllerSettings()) {
    # ifndef BUILD_NO_DEBUG
    addLog(LOG_LEVEL_ERROR, F("MQTT : Cannot connect, out of RAM"));
    # endif // ifndef BUILD_NO_DEBUG
    return false;
  }
  LoadControllerSettings(controller_idx, *ControllerSettings);

  if (!ControllerSettings->checkHostReachable(true)) {
    return false;
  }

  if (MQTTclient.connected()) {
    #if FEATURE_MQTT_CONNECT_BACKGROUND
    if (MQTT_task_data.status != MQTT_connect_status_e::Connecting) {
      MQTT_task_data.status = MQTT_connect_status_e::Disconnected;
    }
    #endif // if FEATURE_MQTT_CONNECT_BACKGROUND
    MQTTclient.disconnect();
    # if FEATURE_MQTT_TLS

    if (mqtt_tls != nullptr) {
      delete mqtt_tls;
      mqtt_tls = nullptr;
    }
    mqtt_rootCA.clear();
    # endif // if FEATURE_MQTT_TLS
  }

  updateMQTTclient_connected();

  //  mqtt = WiFiClient(); // workaround see: https://github.com/esp8266/Arduino/issues/4497#issuecomment-373023864
  delay(0);
# if FEATURE_MQTT_TLS

  uint16_t mqttPort = ControllerSettings->Port;

  const TLS_types TLS_type = ControllerSettings->TLStype();

  if ((TLS_type != TLS_types::NoTLS) && (nullptr == mqtt_tls)) {
#  ifdef ESP32
  #   if MQTT_MAX_PACKET_SIZE > 2000
    mqtt_tls = new BearSSL::WiFiClientSecure_light(4096, 4096);
  #   else // if MQTT_MAX_PACKET_SIZE > 2000
    mqtt_tls = new BearSSL::WiFiClientSecure_light(2048, 2048);
  #   endif // if MQTT_MAX_PACKET_SIZE > 2000
#  else // ESP32 - ESP8266
    mqtt_tls = new BearSSL::WiFiClientSecure_light(1024, 1024);
#  endif // ifdef ESP32
    mqtt_rootCA.clear();

    if (mqtt_tls == nullptr) {
      mqtt_tls_last_errorstr = F("MQTT : Could not create TLS client, out of memory");
      addLog(LOG_LEVEL_ERROR, mqtt_tls_last_errorstr);
      return false;
    } else {
      mqtt_tls->setUtcTime_fcn(getUnixTime);
      mqtt_tls->setCfgTime_fcn(get_build_unixtime);
    }
    mqtt_tls_last_errorstr.clear();
    mqtt_tls_last_error = 0;
  }

  switch (TLS_type) {
    case TLS_types::NoTLS:
    {
      // Ignoring the ACK from the server is probably set for a reason.
      // For example because the server does not give an acknowledgement.
      // This way, we always need the set amount of timeout to handle the request.
      // Thus we should not make the timeout dynamic here if set to ignore ack.
      const uint32_t timeout = ControllerSettings->MustCheckReply
      ? WiFiEventData.getSuggestedTimeout(Settings.Protocol[controller_idx], ControllerSettings->ClientTimeout)
      : ControllerSettings->ClientTimeout;

  #  ifdef MUSTFIX_CLIENT_TIMEOUT_IN_SECONDS

      // See: https://github.com/espressif/arduino-esp32/pull/6676
      mqtt.setTimeout((timeout + 500) / 1000); // in seconds!!!!
      Client *pClient = &mqtt;
      pClient->setTimeout(timeout);
  #  else // ifdef MUSTFIX_CLIENT_TIMEOUT_IN_SECONDS
      mqtt.setTimeout(timeout); // in msec as it should be!
  #  endif // ifdef MUSTFIX_CLIENT_TIMEOUT_IN_SECONDS
      MQTTclient.setClient(mqtt);
      MQTTclient.setKeepAlive(ControllerSettings->KeepAliveTime ? ControllerSettings->KeepAliveTime : CONTROLLER_KEEP_ALIVE_TIME_DFLT);
      MQTTclient.setSocketTimeout(timeout);
      break;
    }

    /*
       case TLS_types::TLS_PSK:
       {
       // if (mqtt_tls != nullptr)
       //  mqtt_tls->setPreSharedKey(const char *pskIdent, const char *psKey); // psKey in Hex
       break;
       }
     */
    case TLS_types::TLS_CA_CERT:
    {
      mqtt_rootCA.clear();

      /*
         // FIXME TD-er: Must convert rootCA from file to format accepted by bearSSL

         if (mqtt_rootCA.isEmpty() && (mqtt_tls != nullptr)) {
         LoadCertificate(ControllerSettings->getCertificateFilename(), mqtt_rootCA);

         if (mqtt_rootCA.isEmpty()) {
         // Fingerprint must be of some minimal length to continue.
         mqtt_tls_last_errorstr = F("MQTT : No TLS root CA");
         addLog(LOG_LEVEL_ERROR, mqtt_tls_last_errorstr);
         return false;
         }



         //mqtt_X509List.append(mqtt_rootCA.c_str());
         //        mqtt_tls->setTrustAnchors(&mqtt_X509List);
         }
       */
      if (mqtt_tls != nullptr) {
        mqtt_tls->setTrustAnchor(Tasmota_TA, Tasmota_TA_size);
      }
      break;
    }

    /*
       case TLS_types::TLS_CA_CLI_CERT:
       {
       //if (mqtt_tls != nullptr)
       //  mqtt_tls->setCertificate(const char *client_ca);
       break;
       }
     */
    case TLS_types::TLS_FINGERPRINT:
    {
      // Fingerprint is checked when making the connection.
      mqtt_rootCA.clear();
      mqtt_fingerprint.clear();
      LoadCertificate(ControllerSettings->getCertificateFilename(), mqtt_fingerprint, false);

      if (mqtt_fingerprint.length() < 32) {
        // Fingerprint must be of some minimal length to continue.
        mqtt_tls_last_errorstr = F("MQTT : Stored TLS fingerprint too small");
        addLog(LOG_LEVEL_ERROR, mqtt_tls_last_errorstr);
        return false;
      }

      if (mqtt_tls != nullptr) {
        mqtt_tls->setInsecure();
      }
      break;
    }
    case TLS_types::TLS_insecure:
    {
      mqtt_rootCA.clear();

      if (mqtt_tls != nullptr) {
        mqtt_tls->setTrustAnchor(Tasmota_TA, Tasmota_TA_size);
        mqtt_tls->setInsecure();
      }
      break;
    }
  }

  if ((TLS_type != TLS_types::NoTLS) && (mqtt_tls != nullptr)) {
    // Certificate expiry not enabled in Mbed TLS.
    //    mqtt_tls->setX509Time(node_time.getUnixTime());
    // Ignoring the ACK from the server is probably set for a reason.
    // For example because the server does not give an acknowledgement.
    // This way, we always need the set amount of timeout to handle the request.
    // Thus we should not make the timeout dynamic here if set to ignore ack.
    const uint32_t timeout = ControllerSettings->MustCheckReply
      ? WiFiEventData.getSuggestedTimeout(Settings.Protocol[controller_idx], ControllerSettings->ClientTimeout)
      : ControllerSettings->ClientTimeout;

    if (mqtt_tls_last_error == 296) {
      // in this special case of cipher mismatch, we force enable ECDSA
      // this would be the case for newer letsencrypt certificates now defaulting
      // to EC certificates requiring ECDSA instead of RSA
      mqtt_tls->setECDSA(true);
      addLog(LOG_LEVEL_INFO, F("MQTT : TLS now enabling ECDSA"));
    }


#  ifdef MUSTFIX_CLIENT_TIMEOUT_IN_SECONDS

    // See: https://github.com/espressif/arduino-esp32/pull/6676
    mqtt_tls->setTimeout((timeout + 500) / 1000); // in seconds!!!!
    Client *pClient = mqtt_tls;
    pClient->setTimeout(timeout);
#  else // ifdef MUSTFIX_CLIENT_TIMEOUT_IN_SECONDS
    mqtt_tls->setTimeout(timeout); // in msec as it should be!
#  endif // ifdef MUSTFIX_CLIENT_TIMEOUT_IN_SECONDS

#  ifdef ESP8266
    mqtt_tls->setBufferSizes(1024, 1024);
#  endif // ifdef ESP8266
    MQTTclient.setClient(*mqtt_tls);
    MQTTclient.setKeepAlive(ControllerSettings->KeepAliveTime ? ControllerSettings->KeepAliveTime : CONTROLLER_KEEP_ALIVE_TIME_DFLT);
    MQTTclient.setSocketTimeout(timeout);


    if (mqttPort == 1883) {
      mqttPort = 8883;
    }
  } else {
    if (mqttPort == 8883) {
      mqttPort = 1883;
    }
  }

# else // if FEATURE_MQTT_TLS

  // Ignoring the ACK from the server is probably set for a reason.
  // For example because the server does not give an acknowledgement.
  // This way, we always need the set amount of timeout to handle the request.
  // Thus we should not make the timeout dynamic here if set to ignore ack.
  const uint32_t timeout = ControllerSettings->MustCheckReply
    ? WiFiEventData.getSuggestedTimeout(Settings.Protocol[controller_idx], ControllerSettings->ClientTimeout)
    : ControllerSettings->ClientTimeout;

#  ifdef MUSTFIX_CLIENT_TIMEOUT_IN_SECONDS

  // See: https://github.com/espressif/arduino-esp32/pull/6676
  mqtt.setTimeout((timeout + 500) / 1000); // in seconds!!!!
  Client *pClient = &mqtt;
  pClient->setTimeout(timeout);
#  else // ifdef MUSTFIX_CLIENT_TIMEOUT_IN_SECONDS
  mqtt.setTimeout(timeout); // in msec as it should be!
#  endif // ifdef MUSTFIX_CLIENT_TIMEOUT_IN_SECONDS

  MQTTclient.setClient(mqtt);
  MQTTclient.setKeepAlive(ControllerSettings->KeepAliveTime ? ControllerSettings->KeepAliveTime : CONTROLLER_KEEP_ALIVE_TIME_DFLT);
  MQTTclient.setSocketTimeout(timeout);
# endif // if FEATURE_MQTT_TLS

  if (ControllerSettings->UseDNS) {
    #if !defined(BUILD_NO_DEBUG) && FEATURE_MQTT_CONNECT_BACKGROUND
    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLog(LOG_LEVEL_INFO, strformat(F("MQTT : Connecting to: %s:%u"), ControllerSettings->getHost().c_str(), ControllerSettings->Port));
    }
    #endif // if !defined(BUILD_NO_DEBUG) && FEATURE_MQTT_CONNECT_BACKGROUND
    MQTTclient.setServer(ControllerSettings->getHost().c_str(), ControllerSettings->Port);
  } else {
    #if !defined(BUILD_NO_DEBUG) && FEATURE_MQTT_CONNECT_BACKGROUND
    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLog(LOG_LEVEL_INFO, strformat(F("MQTT : Connecting to: %s:%u"), ControllerSettings->getIP().toString().c_str(), ControllerSettings->Port));
    }
    #endif // if !defined(BUILD_NO_DEBUG) && FEATURE_MQTT_CONNECT_BACKGROUND
    MQTTclient.setServer(ControllerSettings->getIP(), ControllerSettings->Port);
  }
  MQTTclient.setCallback(incoming_mqtt_callback);

  if (MQTTclient_should_reconnect) {
    addLog(LOG_LEVEL_ERROR, F("MQTT : Intentional reconnect"));
  }

  return true;
}

bool MQTTConnect_clientConnect(controllerIndex_t controller_idx) {
  MakeControllerSettings(ControllerSettings); // -V522

  if (!AllocatedControllerSettings()) {
    #ifndef BUILD_MINIMAL_OTA
    addLog(LOG_LEVEL_ERROR, F("MQTT : Cannot connect, out of RAM"));
    #endif
    return false;
  }
  LoadControllerSettings(controller_idx, *ControllerSettings);

  bool MQTTresult                   = false;

  // MQTT needs a unique clientname to subscribe to broker
  const String clientid = getMQTTclientID(*ControllerSettings);

  const String LWTTopic             = getLWT_topic(*ControllerSettings);
  const String LWTMessageDisconnect = getLWT_messageDisconnect(*ControllerSettings);
  const uint8_t willQos             = 0;
  const bool    willRetain          = ControllerSettings->mqtt_willRetain() && ControllerSettings->mqtt_sendLWT();
  // As suggested here: https://github.com/knolleary/pubsubclient/issues/458#issuecomment-493875150
  const bool    cleanSession        = ControllerSettings->mqtt_cleanSession();
  const bool    hasCredentials      = hasControllerCredentialsSet(controller_idx, *ControllerSettings);

  const uint64_t statisticsTimerStart(getMicros64());

  MQTTresult =
      MQTTclient.connect(clientid.c_str(),
                        hasCredentials ? getControllerUser(controller_idx, *ControllerSettings).c_str() : nullptr,
                        hasCredentials ? getControllerPass(controller_idx, *ControllerSettings).c_str() : nullptr,
                        ControllerSettings->mqtt_sendLWT() ? LWTTopic.c_str() : nullptr,
                        willQos,
                        willRetain,
                        ControllerSettings->mqtt_sendLWT() ? LWTMessageDisconnect.c_str() : nullptr,
                        cleanSession);
  delay(0);

  count_connection_results(
    MQTTresult,
    F("MQTT : Broker "),
    Settings.Protocol[controller_idx],
    statisticsTimerStart);

  return MQTTresult;
}

bool MQTTConnect_wrapUpConnect(controllerIndex_t controller_idx, bool MQTTresult) {
  MakeControllerSettings(ControllerSettings); // -V522

  if (!AllocatedControllerSettings()) {
    #ifndef BUILD_MINIMAL_OTA
    addLog(LOG_LEVEL_ERROR, F("MQTT : Cannot connect, out of RAM"));
    #endif
    return false;
  }
  LoadControllerSettings(controller_idx, *ControllerSettings);

  const String clientid   = getMQTTclientID(*ControllerSettings);
  const String LWTTopic   = getLWT_topic(*ControllerSettings);
  const bool   willRetain = ControllerSettings->mqtt_willRetain() && ControllerSettings->mqtt_sendLWT();

  # if FEATURE_MQTT_TLS

  const TLS_types TLS_type = ControllerSettings->TLStype();

  if (mqtt_tls != nullptr)
  {
    #  ifdef ESP32
    mqtt_tls_last_error = mqtt_tls->getLastError();
    mqtt_tls->clearLastError();
    mqtt_tls_last_cipher_suite = mqtt_tls->getLastCipherSuite();
    #  endif // ifdef ESP32

    // mqtt_tls_last_errorstr = buf;
    if (mqtt_tls_last_error == ERR_OOM) { mqtt_tls_last_errorstr = F("OutOfMemory"); }

    if (mqtt_tls_last_error == ERR_CANT_RESOLVE_IP) { mqtt_tls_last_errorstr = F("Can't resolve IP"); }

    if (mqtt_tls_last_error == ERR_TCP_CONNECT) { mqtt_tls_last_errorstr = F("TCP Connect error"); }

    //    if (mqtt_tls_last_error == ERR_MISSING_CA) mqtt_tls_last_errorstr = F("Missing CA");
    if (mqtt_tls_last_error == ERR_TLS_TIMEOUT) { mqtt_tls_last_errorstr = F("TLS Timeout"); }

#  ifndef BUILD_NO_DEBUG

    if (BR_TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256 == mqtt_tls_last_cipher_suite) {
      addLog(LOG_LEVEL_DEBUG, F("TLS cipher suite: ECDHE_RSA_AES_128_GCM_SHA256"));
    } else if (BR_TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 == mqtt_tls_last_cipher_suite) {
      addLog(LOG_LEVEL_DEBUG, F("TLS cipher suite: ECDHE_ECDSA_AES_128_GCM_SHA256"));
    } else if (0 != mqtt_tls_last_cipher_suite) {
      addLog(LOG_LEVEL_DEBUG, strformat(F("TLS cipher suite: 0x%04X"), mqtt_tls_last_cipher_suite));
    }
#  endif // ifndef BUILD_NO_DEBUG
  }
  #  ifdef ESP32

  // FIXME TD-er: There seems to be no verify function in BearSSL used on ESP8266
  if (TLS_type == TLS_types::TLS_FINGERPRINT)
  {
    // Check fingerprint
    if (MQTTresult) {
      const int newlinepos = mqtt_fingerprint.indexOf('\n');
      String    fp;
      String    dn;

      if (ControllerSettings->UseDNS) { dn = ControllerSettings->getHost(); }

      if (newlinepos == -1) {
        fp = mqtt_fingerprint;
      } else {
        fp = mqtt_fingerprint.substring(0, newlinepos);
        const int newlinepos2 = mqtt_fingerprint.indexOf('\n', newlinepos);

        if (newlinepos2 == -1) {
          dn = mqtt_fingerprint.substring(newlinepos + 1);
        }
        else {
          dn = mqtt_fingerprint.substring(newlinepos + 1, newlinepos2);
        }
        dn.trim();
      }

      // FIXME TD-er: Must implement fingerprint verification

      /*
         if (mqtt_tls != nullptr) {
         if (!mqtt_tls->verify(
         fp.c_str(),
         dn.isEmpty() ? nullptr : dn.c_str()))
         {
         mqtt_tls_last_errorstr += F("TLS Fingerprint does not match");
         addLog(LOG_LEVEL_INFO, mqtt_fingerprint);
         MQTTresult = false;
         }
         }
       */
    }
  }
  #  endif // ifdef ESP32

  # endif  // if FEATURE_MQTT_TLS

  if (!MQTTresult) {
    # if FEATURE_MQTT_TLS

    if ((mqtt_tls_last_error != 0) && loglevelActiveFor(LOG_LEVEL_ERROR)) {
      String log = F("MQTT : TLS error code: ");
      log += mqtt_tls_last_error;
      log += ' ';
      log += mqtt_tls_last_errorstr;
      addLog(LOG_LEVEL_ERROR, log);
    }
    # endif // if FEATURE_MQTT_TLS

    #if FEATURE_MQTT_CONNECT_BACKGROUND
    if (MQTT_task_data.status != MQTT_connect_status_e::Connecting) {
      MQTT_task_data.status = MQTT_connect_status_e::Disconnected;
    }
    #endif // if FEATURE_MQTT_CONNECT_BACKGROUND
    MQTTclient.disconnect();
    # if FEATURE_MQTT_TLS

    if (mqtt_tls != nullptr) {
      mqtt_tls->stop();
    }
    # endif // if FEATURE_MQTT_TLS

    updateMQTTclient_connected();

    return false;
  }

  if (loglevelActiveFor(LOG_LEVEL_INFO))
  {
    addLogMove(LOG_LEVEL_INFO, concat(F("MQTT : Connected to broker with client ID: "), clientid));
  }

  # if FEATURE_MQTT_TLS
  #  ifdef ESP32

  // FIXME TD-er: Must get certificate info

  /*

     if ((mqtt_tls != nullptr) && loglevelActiveFor(LOG_LEVEL_INFO))
     {
     String log = F("MQTT : Peer certificate info: ");
     log += ControllerSettings->getHost();
     log += ' ';
     log += mqtt_tls->getPeerCertificateInfo();
     addLogMove(LOG_LEVEL_INFO, log);
     }
   */
  #  endif // ifdef ESP32
  # endif  // if FEATURE_MQTT_TLS

  MQTTparseSystemVariablesAndSubscribe(String(ControllerSettings->Subscribe));

  # if FEATURE_MQTT_DISCOVER

  if (ControllerSettings->mqtt_autoDiscovery()) {
    MQTTparseSystemVariablesAndSubscribe(String(ControllerSettings->MqttAutoDiscoveryTrigger));
  }
  # endif // if FEATURE_MQTT_DISCOVER

  updateMQTTclient_connected();
  statusLED(true);
  mqtt_reconnect_count = 0;

  // call all installed controller to publish autodiscover data
  if (MQTTclient_should_reconnect) { CPluginCall(CPlugin::Function::CPLUGIN_GOT_CONNECTED, 0); }
  MQTTclient_should_reconnect = false;

  if (ControllerSettings->mqtt_sendLWT()) {
    String LWTMessageConnect = getLWT_messageConnect(*ControllerSettings);

    if (!MQTTclient.publish(LWTTopic.c_str(), LWTMessageConnect.c_str(), willRetain)) {
      MQTTclient_must_send_LWT_connected = true;
    }
  }

  return true;
}

/*********************************************************************************************\
* Connect to MQTT message broker
\*********************************************************************************************/
bool MQTTConnect(controllerIndex_t controller_idx)
{
  if (MQTTclient_next_connect_attempt.isSet() && !MQTTclient_next_connect_attempt.timeoutReached(timermqtt_interval)) {
    return false;
  }
  MQTTclient_next_connect_attempt.setNow();
  ++mqtt_reconnect_count;

  if (!MQTTConnect_prepareClient(controller_idx)) {
    return false;
  }
  
  bool MQTTresult = MQTTConnect_clientConnect(controller_idx);

  return MQTTConnect_wrapUpConnect(controller_idx, MQTTresult);
}

void MQTTparseSystemVariablesAndSubscribe(String subscribeTo) {
  if (subscribeTo.isEmpty()) { return; }
  parseSystemVariables(subscribeTo, false);
  subscribeTo.trim();

  if (!subscribeTo.isEmpty()) {
    MQTTclient.subscribe(subscribeTo.c_str());
# ifndef BUILD_NO_DEBUG

    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLogMove(LOG_LEVEL_INFO, concat(F("MQTT : Subscribed to: "),  subscribeTo));
    }
# endif // ifndef BUILD_NO_DEBUG
  }
}

String getMQTTclientID(const ControllerSettingsStruct& ControllerSettings) {
  String clientid = ControllerSettings.ClientID;

  if (clientid.isEmpty()) {
    // Try to generate some default
    clientid = F(CONTROLLER_DEFAULT_CLIENTID);
  }
  parseSystemVariables(clientid, false);
  clientid.replace(' ', '_'); // Make sure no spaces are present in the client ID

  if ((WiFiEventData.wifi_reconnects >= 1) && ControllerSettings.mqtt_uniqueMQTTclientIdReconnect()) {
    // Work-around for 'lost connections' to the MQTT broker.
    // If the broker thinks the connection is still alive, a reconnect from the
    // client will be refused.
    // To overcome this issue, append the number of reconnects to the client ID to
    // make it different from the previous one.
    clientid += '_';
    clientid += WiFiEventData.wifi_reconnects;
  }
  return clientid;
}

#if FEATURE_MQTT_CONNECT_BACKGROUND
void MQTT_execute_connect_task(void *parameter)
{
  MQTT_connect_request*MQTT_task_data = static_cast<MQTT_connect_request *>(parameter);

  uint32_t timeout = MQTT_task_data->timeout;
  uint8_t  incr    = 5; // Increment timeout between connection attempts 5 times by 100 msec

  while (!MQTTConnect_clientConnect(MQTT_task_data->ControllerIndex) && !MQTTclient.connected()) {
    const TickType_t xDelay = timeout / portTICK_PERIOD_MS;
    vTaskDelay(xDelay); // Use regular controller timeout also for delay between connection attempts (range 10..4000)
    if (incr > 0) {
      timeout += 100; // Increment next few (5) times with 100 msec
      incr--;
    }
    if (timePassedSince(MQTT_task_data->startTime) > 120000) { // Quit after 120 seconds
      break;
    }
  }
  MQTT_task_data->result     = MQTTclient.connected();
  MQTT_task_data->status     = MQTT_task_data->result
                               ? MQTT_connect_status_e::Connected
                               : MQTT_connect_status_e::Failure;
  MQTT_task_data->endTime    = millis();
  MQTT_task_data->taskHandle = NULL;
  vTaskDelete(MQTT_task_data->taskHandle);
}

bool MQTTConnectInBackground(controllerIndex_t controller_idx, bool reportOnly) {
  if (!Settings.MQTTConnectInBackground()) { return false; }

  if ((MQTT_task_data.status == MQTT_connect_status_e::Connected) || (MQTT_task_data.status == MQTT_connect_status_e::Failure)) {
    MQTT_task_data.status = MQTT_connect_status_e::Ready; // Set status first to avoid re-entry

    const bool result = MQTTConnect_wrapUpConnect(MQTT_task_data.ControllerIndex, MQTT_task_data.result);

    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLog(LOG_LEVEL_INFO, strformat(F("MQTT : Background Connect request %s, took %d msec"),
                                       FsP(MQTT_task_data.result ? F("success") : F("FAILED")),
                                       MQTT_task_data.endTime - MQTT_task_data.startTime
                                      ));
    }

    return result && MQTT_task_data.result;
  }
  
  if ((MQTT_task_data.status == MQTT_connect_status_e::Ready) && MQTTclient.connected()) {
    if (!MQTT_task_data.logged && loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLog(LOG_LEVEL_INFO, F("MQTT : Background Connect request Ready"));
    }
    MQTT_task_data.logged = true;
    return MQTT_task_data.result;
  }

  if (MQTT_task_data.status == MQTT_connect_status_e::Connecting) {
    if (timePassedSince(MQTT_task_data.loopTime) > 1000) {
      if (loglevelActiveFor(LOG_LEVEL_INFO)) {
        addLog(LOG_LEVEL_INFO, strformat(F("MQTT : Background waiting to Connect, %d sec"),
                                         timePassedSince(MQTT_task_data.startTime) / 1000
                                        ));
      }
      MQTT_task_data.loopTime = millis();
    }
    return false; // Not ready yet
  }

  if ((MQTT_task_data.status == MQTT_connect_status_e::Ready) &&
      !MQTTclient.connected() &&
      NetworkConnected(10)) { // Unexpected network disconnect and reconnect?
    MQTT_task_data.status = MQTT_connect_status_e::Disconnected;
    reportOnly            = false; // Reconnect ASAP
    if (CONTROLLER_MAX == controller_idx) {
      controller_idx = firstEnabledMQTT_ControllerIndex();
    }
  }

  if (!reportOnly && (MQTT_task_data.status == MQTT_connect_status_e::Disconnected) && (controller_idx < CONTROLLER_MAX)) {
    if (MQTTclient_next_connect_attempt.isSet() && !MQTTclient_next_connect_attempt.timeoutReached(timermqtt_interval)) {
      return false;
    }
    MQTTclient_next_connect_attempt.setNow();
    ++mqtt_reconnect_count;

    if (loglevelActiveFor(LOG_LEVEL_INFO)) {
      addLog(LOG_LEVEL_INFO, strformat(F("MQTT : Start background connect for controller %d"), controller_idx + 1));
    }
    MQTT_task_data.result          = false;

    if (MQTTConnect_prepareClient(controller_idx)) {
      MakeControllerSettings(ControllerSettings); // -V522

      if (!AllocatedControllerSettings()) {
        #ifndef BUILD_MINIMAL_OTA
        addLog(LOG_LEVEL_ERROR, F("MQTT : Cannot load Controller settings, out of RAM"));
        #endif
        return false;
      }
      LoadControllerSettings(controller_idx, *ControllerSettings);

      const uint32_t timeout = ControllerSettings->MustCheckReply
                               ? WiFiEventData.getSuggestedTimeout(Settings.Protocol[controller_idx], ControllerSettings->ClientTimeout)
                               : ControllerSettings->ClientTimeout;

      MQTT_task_data.status          = MQTT_connect_status_e::Connecting;
      MQTT_task_data.logged          = false;
      MQTT_task_data.ControllerIndex = controller_idx;
      MQTT_task_data.startTime       = millis();
      MQTT_task_data.loopTime        = millis();
      MQTT_task_data.timeout         = timeout;

      xTaskCreatePinnedToCore(
        MQTT_execute_connect_task,   // Function that should be called
        "MQTTClient.connect()",      // Name of the task (for debugging)
        8192,                        // Stack size (bytes)
        &MQTT_task_data,             // Parameter to pass
        1,                           // Task priority
        &MQTT_task_data.taskHandle,  // Task handle
        xPortGetCoreID()             // Core you want to run the task on (0 or 1)
        );
    } else {
      MQTT_task_data.status = MQTT_connect_status_e::Failure;
    }
  }
  return false;
}
#endif // if FEATURE_MQTT_CONNECT_BACKGROUND

/*********************************************************************************************\
* Check connection MQTT message broker
\*********************************************************************************************/
bool MQTTCheck(controllerIndex_t controller_idx)
{
  if (!NetworkConnected(10)) {
    return false;
  }
  protocolIndex_t ProtocolIndex = getProtocolIndex_from_ControllerIndex(controller_idx);

  if (!validProtocolIndex(ProtocolIndex)) {
    return false;
  }

  if (getProtocolStruct(ProtocolIndex).usesMQTT)
  {
    bool   mqtt_sendLWT = false;
    String LWTTopic, LWTMessageConnect;
    bool   willRetain = false;
    {
      MakeControllerSettings(ControllerSettings); // -V522

      if (!AllocatedControllerSettings()) {
        addLog(LOG_LEVEL_ERROR, F("MQTT : Cannot check, out of RAM"));
        return false;
      }

      LoadControllerSettings(controller_idx, *ControllerSettings);

      // FIXME TD-er: Is this still needed?

      /*
       #ifdef USES_ESPEASY_NOW
         if (!MQTTclient.connected()) {
         if (ControllerSettings->enableESPEasyNowFallback()) {
         return true;
         }
         }
       #endif
       */

      if (!ControllerSettings->isSet()) {
        return true;
      }

      if (ControllerSettings->mqtt_sendLWT()) {
        mqtt_sendLWT      = true;
        LWTTopic          = getLWT_topic(*ControllerSettings);
        LWTMessageConnect = getLWT_messageConnect(*ControllerSettings);
        willRetain        = ControllerSettings->mqtt_willRetain();
      }
    }

    if (MQTTclient_should_reconnect || !MQTTclient.connected())
    {
      #if FEATURE_MQTT_CONNECT_BACKGROUND
      if (Settings.MQTTConnectInBackground()) {
        return MQTTConnectInBackground(controller_idx, false);
      }
      #endif // ifdef ESP32
      return MQTTConnect(controller_idx);
    }

    if (MQTTclient_must_send_LWT_connected) {
      if (mqtt_sendLWT) {
        if (MQTTclient.publish(LWTTopic.c_str(), LWTMessageConnect.c_str(), willRetain)) {
          MQTTclient_must_send_LWT_connected = false;
        }
      } else {
        MQTTclient_must_send_LWT_connected = false;
      }
    }
  }

  // When no MQTT protocol is enabled, all is fine.
  return true;
}

String getLWT_topic(const ControllerSettingsStruct& ControllerSettings) {
  String LWTTopic;

  if (ControllerSettings.mqtt_sendLWT()) {
    LWTTopic = ControllerSettings.MQTTLwtTopic;

    if (LWTTopic.isEmpty())
    {
      LWTTopic  = ControllerSettings.Subscribe;
      LWTTopic += F("/LWT");
    }
    LWTTopic.replace(F("/#"), F("/status"));
    parseSystemVariables(LWTTopic, false);
  }
  return LWTTopic;
}

String getLWT_messageConnect(const ControllerSettingsStruct& ControllerSettings) {
  String LWTMessageConnect;

  if (ControllerSettings.mqtt_sendLWT()) {
    LWTMessageConnect = ControllerSettings.LWTMessageConnect;

    if (LWTMessageConnect.isEmpty()) {
      LWTMessageConnect = F(DEFAULT_MQTT_LWT_CONNECT_MESSAGE);
    }
    parseSystemVariables(LWTMessageConnect, false);
  }
  return LWTMessageConnect;
}

String getLWT_messageDisconnect(const ControllerSettingsStruct& ControllerSettings) {
  String LWTMessageDisconnect;

  if (ControllerSettings.mqtt_sendLWT()) {
    LWTMessageDisconnect = ControllerSettings.LWTMessageDisconnect;

    if (LWTMessageDisconnect.isEmpty()) {
      LWTMessageDisconnect = F(DEFAULT_MQTT_LWT_DISCONNECT_MESSAGE);
    }
    parseSystemVariables(LWTMessageDisconnect, false);
  }
  return LWTMessageDisconnect;
}

#endif // if FEATURE_MQTT

/*********************************************************************************************\
* Send status info to request source
\*********************************************************************************************/
void SendStatusOnlyIfNeeded(struct EventStruct *event, bool param1, uint32_t key, const String& param2, int16_t param3) {
  if (SourceNeedsStatusUpdate(event->Source)) {
    SendStatus(event, getPinStateJSON(param1, key, param2, param3));
    printToWeb = false; // SP: 2020-06-12: to avoid to add more info to a JSON structure
  }
}

bool SourceNeedsStatusUpdate(EventValueSource::Enum eventSource)
{
  switch (eventSource) {
    case EventValueSource::Enum::VALUE_SOURCE_HTTP:
    case EventValueSource::Enum::VALUE_SOURCE_SERIAL:
    case EventValueSource::Enum::VALUE_SOURCE_MQTT:
    case EventValueSource::Enum::VALUE_SOURCE_WEB_FRONTEND:
      return true;

    default:
      break;
  }
  return false;
}

void SendStatus(struct EventStruct *event, const __FlashStringHelper *status)
{
  SendStatus(event, String(status));
}

void SendStatus(struct EventStruct *event, const String& status)
{
  if (status.isEmpty()) { return; }

  switch (event->Source)
  {
    case EventValueSource::Enum::VALUE_SOURCE_HTTP:
    case EventValueSource::Enum::VALUE_SOURCE_WEB_FRONTEND:

      if (printToWeb) {
        printWebString += status;
      }
      break;
#if FEATURE_MQTT
    case EventValueSource::Enum::VALUE_SOURCE_MQTT:
      MQTTStatus(event, status);
      break;
#endif // if FEATURE_MQTT
    case EventValueSource::Enum::VALUE_SOURCE_SERIAL:
      serialPrintln(status);
      break;

    default:
      break;
  }
}

#if FEATURE_MQTT

controllerIndex_t firstEnabledMQTT_ControllerIndex() {
  for (controllerIndex_t i = 0; i < CONTROLLER_MAX; ++i) {
    protocolIndex_t ProtocolIndex = getProtocolIndex_from_ControllerIndex(i);

    if (validProtocolIndex(ProtocolIndex)) {
      if (getProtocolStruct(ProtocolIndex).usesMQTT && Settings.ControllerEnabled[i]) {
        return i;
      }
    }
  }
  return INVALID_CONTROLLER_INDEX;
}

bool MQTT_queueFull(controllerIndex_t controller_idx) {
  if (MQTTDelayHandler == nullptr) {
    return true;
  }

  if (MQTTDelayHandler->queueFull(controller_idx)) {
    // The queue is full, try to make some room first.
    processMQTTdelayQueue();
    return MQTTDelayHandler->queueFull(controller_idx);
  }
  return false;
}

bool MQTTpublish(controllerIndex_t controller_idx,
                 taskIndex_t       taskIndex,
                 const char       *topic,
                 const char       *payload,
                 bool              retained,
                 bool              callbackTask)
{
  if (MQTTDelayHandler == nullptr) {
    return false;
  }

  if (MQTT_queueFull(controller_idx)) {
    return false;
  }
  String topic_str;
  String payload_str;

  if (!reserve_special(topic_str, strlen_P(topic)) ||
      !reserve_special(payload_str, strlen_P(payload))) {
    return false;
  }
  topic_str   = topic;
  payload_str = payload;

  bool success = false;

  constexpr unsigned size = sizeof(MQTT_queue_element);
  void *ptr               = special_calloc(1, size);

  if (ptr != nullptr) {
    success =
      MQTTDelayHandler->addToQueue(
        std::unique_ptr<MQTT_queue_element>(
          new (ptr) MQTT_queue_element(
            controller_idx, taskIndex,
            std::move(topic_str),
            std::move(payload_str), retained,
            callbackTask)));
  }

  scheduleNextMQTTdelayQueue();
  return success;
}

bool MQTTpublish(controllerIndex_t controller_idx,
                 taskIndex_t       taskIndex,
                 String         && topic,
                 String         && payload,
                 bool              retained,
                 bool              callbackTask) {
  if (MQTTDelayHandler == nullptr) {
    return false;
  }

  if (MQTT_queueFull(controller_idx)) {
    return false;
  }

  bool success = false;

  constexpr unsigned size = sizeof(MQTT_queue_element);
  void *ptr               = special_calloc(1, size);

  if (ptr != nullptr) {
    success =
      MQTTDelayHandler->addToQueue(
        std::unique_ptr<MQTT_queue_element>(
          new (ptr) MQTT_queue_element(
            controller_idx, taskIndex,
            std::move(topic),
            std::move(payload), retained,
            callbackTask)));
  }
  scheduleNextMQTTdelayQueue();
  return success;
}

/*********************************************************************************************\
* Send status info back to channel where request came from
\*********************************************************************************************/
void MQTTStatus(struct EventStruct *event, const String& status)
{
  controllerIndex_t enabledMqttController = firstEnabledMQTT_ControllerIndex();

  if (validControllerIndex(enabledMqttController)) {
    controllerIndex_t DomoticzMQTT_controllerIndex = findFirstEnabledControllerWithId(2);

    if (DomoticzMQTT_controllerIndex == enabledMqttController) {
      // Do not send MQTT status updates to Domoticz
      return;
    }
    String pubname;
    bool   mqtt_retainFlag;
    {
      // Place the ControllerSettings in a scope to free the memory as soon as we got all relevant information.
      MakeControllerSettings(ControllerSettings); // -V522

      if (!AllocatedControllerSettings()) {
        addLog(LOG_LEVEL_ERROR, F("MQTT : Cannot send status, out of RAM"));
        return;
      }

      LoadControllerSettings(enabledMqttController, *ControllerSettings);
      pubname         = ControllerSettings->Publish;
      mqtt_retainFlag = ControllerSettings->mqtt_retainFlag();
    }

    // FIXME TD-er: Why check for "/#" suffix on a publish topic?
    // It makes no sense to have a subscribe wildcard on a publish topic.
    pubname.replace(F("/#"), F("/status"));

    parseSingleControllerVariable(pubname, event, 0, false);
    parseControllerVariables(pubname, event, false);


    if (!pubname.endsWith(F("/status"))) {
      pubname += F("/status");
    }

    MQTTpublish(enabledMqttController, event->TaskIndex, pubname.c_str(), status.c_str(), mqtt_retainFlag);
  }
}

# if FEATURE_MQTT_TLS

bool GetTLSfingerprint(String& fp)
{
  #  ifdef ESP32

  if (MQTTclient_connected && (mqtt_tls != nullptr)) {
    const uint8_t *recv_fingerprint = mqtt_tls->getRecvPubKeyFingerprint();

    if (recv_fingerprint != nullptr) {
      fp.reserve(64);

      for (size_t i = 0; i < 21; ++i) {
        const String tmp(recv_fingerprint[i], HEX);

        switch (tmp.length()) {
          case 0:
            fp += '0';

          // fall through
          case 1:
            fp += '0';
            break;
        }
        fp += tmp;
      }
      fp.toLowerCase();
      return true;
    }
  }
  #  endif // ifdef ESP32
  return false;
}

bool GetTLS_Certificate(String& cert, bool caRoot)
{
  #  ifdef ESP32

  // FIXME TD-er: Implement retrieval of certificate

  /*

     if (MQTTclient_connected && (mqtt_tls != nullptr)) {
     String subject;

     if (mqtt_tls->getPeerCertificate(cert, subject, caRoot) == 0) {
     return true;
     }
     }
   */
  #  endif // ifdef ESP32
  return false;
}

# endif // if FEATURE_MQTT_TLS

#endif  // if FEATURE_MQTT

/*********************************************************************************************\
* send specific sensor task data, effectively calling PluginCall(PLUGIN_READ...)
\*********************************************************************************************/
void SensorSendTask(struct EventStruct *event, unsigned long timestampUnixTime) {
  SensorSendTask(event, timestampUnixTime, millis());
}

void SensorSendTask(struct EventStruct *event, unsigned long timestampUnixTime, unsigned long lasttimer)
{
  if (!validTaskIndex(event->TaskIndex)) {
    return;
  }

  // FIXME TD-er: Should a 'disabled' task be rescheduled?
  // If not, then it should be rescheduled after the check to see if it is enabled.
  Scheduler.reschedule_task_device_timer(event->TaskIndex, lasttimer);

  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("SensorSendTask"));
  #endif // ifndef BUILD_NO_RAM_TRACKER

  if (Settings.TaskDeviceEnabled[event->TaskIndex])
  {
    START_TIMER;
    const deviceIndex_t DeviceIndex = getDeviceIndex_from_TaskIndex(event->TaskIndex);

    if (!validDeviceIndex(DeviceIndex)) { return; }

    struct EventStruct TempEvent(event->TaskIndex);
    TempEvent.Source        = event->Source;
    TempEvent.timestamp_sec = timestampUnixTime;
    checkDeviceVTypeForTask(&TempEvent);

    String dummy;

    if (PluginCall(PLUGIN_READ, &TempEvent, dummy)) {
      sendData(&TempEvent);
    }
    STOP_TIMER(SENSOR_SEND_TASK);
  }
}

#include "../Globals/Device.h"

// TODO TD-er: Move often used functions like determine valueCount to here.
#ifdef ESP8266
int deviceCount = -1;
#else
DeviceCount_t deviceCount;
#endif

int getDeviceCount()
{
#ifdef ESP8266
    return deviceCount;
#else
    return deviceCount.value;
#endif
}

DeviceVector Device;
#include "../Globals/ESPEasy_time.h"

ESPEasy_time node_time;

uint32_t getUnixTime() {
    return node_time.getUnixTime();
}
#include "../Globals/CRCValues.h"

CRCStruct CRCValues;
#include "../Globals/NPlugins.h"

#if FEATURE_NOTIFIER

#include "../DataStructs/ESPEasy_EventStruct.h"
#include "../DataStructs/NotificationStruct.h"
#include "../Globals/Settings.h"


nprotocolIndex_t INVALID_NPROTOCOL_INDEX = NPLUGIN_MAX;


bool (*NPlugin_ptr[NPLUGIN_MAX])(NPlugin::Function,
                                    struct EventStruct *,
                                    String&);
npluginID_t NPlugin_id[NPLUGIN_MAX];

NotificationStruct Notification[NPLUGIN_MAX];

int notificationCount = -1;


bool NPluginCall(NPlugin::Function Function, struct EventStruct *event)
{
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif
  int x;
  struct EventStruct TempEvent;

  if (event == 0) {
    event = &TempEvent;
  }

  switch (Function)
  {
    // Unconditional calls to all plugins
    case NPlugin::Function::NPLUGIN_PROTOCOL_ADD:

      for (x = 0; x < NPLUGIN_MAX; x++) {
        if (validNPluginID(NPlugin_id[x])) {
          String dummy;
          NPlugin_ptr[x](Function, event, dummy);
        }
      }
      return true;
      break;

    case NPlugin::Function::NPLUGIN_GET_DEVICENAME:
    case NPlugin::Function::NPLUGIN_WEBFORM_SAVE:
    case NPlugin::Function::NPLUGIN_WEBFORM_LOAD:
    case NPlugin::Function::NPLUGIN_WRITE:
    case NPlugin::Function::NPLUGIN_NOTIFY:
      break;
  }

  return false;
}


bool validNProtocolIndex(nprotocolIndex_t index) {
  return index != INVALID_NPROTOCOL_INDEX;
}

bool validNotifierIndex(notifierIndex_t index)
{
  return index < NOTIFICATION_MAX;
}

bool validNPluginID(npluginID_t npluginID)
{
  if (npluginID == INVALID_N_PLUGIN_ID) {
    return false;
  }

  // FIXME TD-er: Must search to all included plugins
  return true;
}

String getNPluginNameFromNotifierIndex(notifierIndex_t NotifierIndex) {
  String name;

  if (validNPluginID(NPlugin_id[NotifierIndex])) {
    NPlugin_ptr[NotifierIndex](NPlugin::Function::NPLUGIN_GET_DEVICENAME, nullptr, name);
  }
  return name;
}

/********************************************************************************************\
   Get notificatoin protocol index (plugin index), by NPlugin_id
 \*********************************************************************************************/
nprotocolIndex_t getNProtocolIndex(npluginID_t Number)
{
  for (uint8_t x = 0; x <= notificationCount; x++) {
    if (Notification[x].Number == Number.value) {
      return x;
    }
  }
  return INVALID_NPROTOCOL_INDEX;
}

nprotocolIndex_t getNProtocolIndex_from_NotifierIndex(notifierIndex_t index) {
  if (validNotifierIndex(index)) {
    return getNProtocolIndex(npluginID_t::toPluginID(Settings.Notification[index]));
  }
  return INVALID_NPROTOCOL_INDEX;
}

bool addNPlugin(/*npluginID_t*/ uint8_t npluginID, nprotocolIndex_t x) {
  if (x < NPLUGIN_MAX) { 
    // FIXME TD-er: Must add lookup for notification plugins too
//    ProtocolIndex_to_NPlugin_id[x] = npluginID; 
//    NPlugin_id_to_ProtocolIndex[npluginID] = x;

    NPlugin_id[x] = npluginID_t::toPluginID(npluginID);
    return true;
  }
  /*
  {
    String log = F("System: Error - Too many N-Plugins. NPLUGIN_MAX = ");
    log += NPLUGIN_MAX;
    addLog(LOG_LEVEL_ERROR, log);
  }
  */
  return false;
}

#endif
#include "../Globals/Services.h"

#if FEATURE_ARDUINO_OTA
  bool ArduinoOTAtriggered = false;
#endif


#ifdef ESP8266  
  ESP8266WebServer web_server(80);
  #ifndef NO_HTTP_UPDATER
  ESP8266HTTPUpdateServer httpUpdater(true);
  #endif
#endif

#ifdef ESP32
  WebServer web_server(80);
  #ifndef NO_HTTP_UPDATER
  ESP32HTTPUpdateServer httpUpdater(true);
  #endif
#endif


#if FEATURE_DNS_SERVER
  DNSServer  dnsServer;
  bool dnsServerActive = false;
#endif // if FEATURE_DNS_SERVER

#include "../Globals/TXBuffer.h"

#include "../DataStructs/Web_StreamingBuffer.h"


Web_StreamingBuffer TXBuffer;
#include "../Globals/MQTT.h"

#include "../../ESPEasy_common.h"

#if FEATURE_MQTT


// MQTT client
WiFiClient mqtt;
# if FEATURE_MQTT_TLS
String  mqtt_tls_last_errorstr;
int32_t mqtt_tls_last_error = 0;

#  ifdef ESP32
BearSSL::WiFiClientSecure_light*mqtt_tls{};
#  endif // ifdef ESP32
#  ifdef ESP8266
BearSSL::WiFiClientSecure*mqtt_tls;
BearSSL::X509List mqtt_X509List;
#  endif // ifdef ESP8266
int32_t mqtt_tls_last_cipher_suite{};
String mqtt_rootCA;
String mqtt_fingerprint;
# endif  // if FEATURE_MQTT_TLS

PubSubClient MQTTclient(mqtt);
bool MQTTclient_should_reconnect        = true;
bool MQTTclient_must_send_LWT_connected = false;
bool MQTTclient_connected               = false;
int  mqtt_reconnect_count               = 0;
LongTermTimer MQTTclient_next_connect_attempt;

# if FEATURE_MQTT_CONNECT_BACKGROUND
MQTT_connect_request MQTT_task_data;
# endif // if FEATURE_MQTT_CONNECT_BACKGROUND

# if FEATURE_MQTT_DISCOVER

controllerIndex_t mqttDiscoveryController = INVALID_CONTROLLER_INDEX;
taskIndex_t mqttDiscoverOnlyTask          = INVALID_TASK_INDEX;
uint32_t    mqttDiscoveryTimeout          = 0; // Decremented in 10 per second, random timeout before discovery is sent, for broker load
                                               // distribution
# endif // if FEATURE_MQTT_DISCOVER
#endif // if FEATURE_MQTT

#ifdef USES_P037

// mqtt import status
bool P037_MQTTImport_connected = false;
#endif // ifdef USES_P037

#include "../Globals/ESPEasy_Console.h"

EspEasy_Console_t ESPEasy_Console;


#include "../Globals/I2Cdev.h"


I2Cdev i2cdev;

#include "../Globals/Plugins_other.h"


 
 void (*parseTemplate_CallBack_ptr)(String& tmpString, bool useURLencode) = nullptr;
 void (*substitute_eventvalue_CallBack_ptr)(String& line, const String& event) = nullptr;
#include "../Globals/Logging.h"

#include "../DataStructs/LogStruct.h"


LogStruct Logging;

uint8_t highest_active_log_level = 0;
bool log_to_serial_disabled = false;


#include "../Globals/ExtraTaskSettings.h"


// TODO TD-er: Move fuctions to manage the ExtraTaskSettings cache to this file.

ExtraTaskSettingsStruct ExtraTaskSettings;
#include "../Globals/C016_ControllerCache.h"

#ifdef USES_C016


# include "../ControllerQueue/C016_queue_element.h"

ControllerCache_struct ControllerCache;

void C016_flush() {
  ControllerCache.flush();
}

bool C016_CacheInitialized() {
  return ControllerCache.isInitialized();
}

String C016_getCacheFileName(int& fileNr, bool& islast) {
  return ControllerCache.getNextCacheFileName(fileNr, islast);
}

bool C016_deleteOldestCacheBlock() {
  return ControllerCache.deleteOldestCacheBlock();
}

bool C016_deleteAllCacheBlocks() {
  return ControllerCache.deleteAllCacheBlocks();
}

bool C016_getTaskSample(C016_binary_element& element) {
  return ControllerCache.peek((uint8_t *)&element, sizeof(element));
}

struct EventStruct C016_getTaskSample(
  unsigned long& timestamp,
  uint8_t      & valueCount,
  float        & val1,
  float        & val2,
  float        & val3,
  float        & val4)
{
  C016_binary_element element;

  if (!ControllerCache.peek((uint8_t *)&element, sizeof(element))) {
    return EventStruct();
  }

  timestamp  = element.unixTime;
  valueCount = element.valueCount;
  val1       = element.values.getFloat(0);
  val2       = element.values.getFloat(1);
  val3       = element.values.getFloat(2);
  val4       = element.values.getFloat(3);

  EventStruct event(element.TaskIndex);

  // FIXME TD-er: Is this needed?
  //  event.ControllerIndex = element._controller_idx;
  event.sensorType = element.sensorType;

  return event;
}

#endif // ifdef USES_C016

#include "../Globals/NetworkState.h"

#include "../../ESPEasy_common.h"


// Ethernet Connection status
NetworkMedium_t active_network_medium = NetworkMedium_t::NotSet;

bool webserverRunning(false);
bool webserver_init(false);

#if FEATURE_MDNS
bool mDNS_init(false);
#endif


// NTP status
bool statusNTPInitialized = false;


// Setup DNS, only used if the ESP has no valid WiFi config
const uint8_t DNS_PORT = 53;
IPAddress  apIP(DEFAULT_AP_IP);



// udp protocol stuff (syslog, global sync, node info list, ntp time)
WiFiUDP portUDP;

#include "../Globals/ESPEasyEthEvent.h"

#include "../../ESPEasy_common.h"

#if FEATURE_ETHERNET
EthernetEventData_t EthEventData;
#endif


#include "../Globals/MainLoopCommand.h"

uint8_t cmd_within_mainloop = 0;
#include "../Globals/ESPEasyWiFi.h"

#ifdef ESPEASY_WIFI_CLEANUP_WORK_IN_PROGRESS
ESPEasyWiFi_t ESPEasyWiFi;
#endif // ifdef ESPEASY_WIFI_CLEANUP_WORK_IN_PROGRESS

#include "../Globals/Settings.h"


SettingsStruct Settings;
#include "../Globals/CPlugins.h"

#include "../../_Plugin_Helper.h"
#include "../DataStructs/ESPEasy_EventStruct.h"
#include "../DataStructs/TimingStats.h"
#include "../DataTypes/ESPEasy_plugin_functions.h"
#include "../ESPEasyCore/ESPEasy_Log.h"
#if FEATURE_MQTT_DISCOVER
# include "../Globals/MQTT.h"
#endif // if FEATURE_MQTT_DISCOVER
#include "../Globals/Settings.h"
#include "../Helpers/_CPlugin_init.h"


/********************************************************************************************\
   Call CPlugin functions
 \*********************************************************************************************/
bool CPluginCall(CPlugin::Function Function, struct EventStruct *event) {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif // ifdef USE_SECOND_HEAP

  String dummy;

  return CPluginCall(Function, event, dummy);
}

bool CPluginCall(CPlugin::Function Function, struct EventStruct *event, String& str)
{
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif // ifdef USE_SECOND_HEAP

  struct EventStruct TempEvent;

  if (event == 0) {
    event = &TempEvent;
  }

  switch (Function)
  {
    case CPlugin::Function::CPLUGIN_PROTOCOL_ADD:
      // only called from CPluginSetup() directly using protocolIndex
      break;

    // calls to all active controllers
    case CPlugin::Function::CPLUGIN_INIT_ALL:
    case CPlugin::Function::CPLUGIN_UDP_IN:
    case CPlugin::Function::CPLUGIN_INTERVAL:      // calls to send stats information
    case CPlugin::Function::CPLUGIN_GOT_CONNECTED: // calls to send autodetect information
    case CPlugin::Function::CPLUGIN_GOT_INVALID:   // calls to mark unit as invalid
    case CPlugin::Function::CPLUGIN_FLUSH:
    case CPlugin::Function::CPLUGIN_TEN_PER_SECOND:
    case CPlugin::Function::CPLUGIN_FIFTY_PER_SECOND:
    case CPlugin::Function::CPLUGIN_WRITE:
    {
      const bool success = Function != CPlugin::Function::CPLUGIN_WRITE;

      if (Function == CPlugin::Function::CPLUGIN_INIT_ALL) {
        Function = CPlugin::Function::CPLUGIN_INIT;
      }

      for (controllerIndex_t x = 0; x < CONTROLLER_MAX; x++) {
        if ((Settings.Protocol[x] != 0) && Settings.ControllerEnabled[x]) {
          event->ControllerIndex = x;
          String command;

          if (Function == CPlugin::Function::CPLUGIN_WRITE) {
            command = str;
          }

          if (CPluginCall(
                getProtocolIndex_from_ControllerIndex(x),
                Function,
                event,
                command)) {
            if (Function == CPlugin::Function::CPLUGIN_WRITE) {
              // Need to stop when write call was handled
              return true;
            }
          }
        }
      }
      return success;
    }

    // calls to specific controller
    case CPlugin::Function::CPLUGIN_INIT:
    case CPlugin::Function::CPLUGIN_EXIT:
    case CPlugin::Function::CPLUGIN_PROTOCOL_TEMPLATE:
    case CPlugin::Function::CPLUGIN_PROTOCOL_SEND:
    case CPlugin::Function::CPLUGIN_PROTOCOL_RECV:
    case CPlugin::Function::CPLUGIN_GET_DEVICENAME:
    case CPlugin::Function::CPLUGIN_WEBFORM_LOAD:
    case CPlugin::Function::CPLUGIN_WEBFORM_SAVE:
    case CPlugin::Function::CPLUGIN_GET_PROTOCOL_DISPLAY_NAME:
    case CPlugin::Function::CPLUGIN_TASK_CHANGE_NOTIFICATION:
    case CPlugin::Function::CPLUGIN_WEBFORM_SHOW_HOST_CONFIG:
    {
      const controllerIndex_t controllerindex = event->ControllerIndex;
      bool success                            = false;

      if (validControllerIndex(controllerindex)) {
        if (Settings.ControllerEnabled[controllerindex] && supportedCPluginID(Settings.Protocol[controllerindex]))
        {
          if (Function == CPlugin::Function::CPLUGIN_PROTOCOL_SEND) {
            checkDeviceVTypeForTask(event);
          }
          success = CPluginCall(
            getProtocolIndex_from_ControllerIndex(controllerindex),
            Function,
            event,
            str);
        }
        #ifdef ESP32

        if (Function == CPlugin::Function::CPLUGIN_EXIT) {
          Cache.clearControllerSettings(controllerindex);
        }
        #endif // ifdef ESP32
        #if FEATURE_MQTT_DISCOVER

        if (Function == CPlugin::Function::CPLUGIN_EXIT) {
          if (mqttDiscoveryController == controllerindex) {
            mqttDiscoveryController = INVALID_CONTROLLER_INDEX; // Reset
            mqttDiscoverOnlyTask    = INVALID_TASK_INDEX;
          }
        }
        #endif // if FEATURE_MQTT_DISCOVER
      }
      return success;
    }

    case CPlugin::Function::CPLUGIN_ACKNOWLEDGE: // calls to send acknowledge back to controller

      for (controllerIndex_t x = 0; x < CONTROLLER_MAX; x++) {
        if (Settings.ControllerEnabled[x] && supportedCPluginID(Settings.Protocol[x])) {
          CPluginCall(
            getProtocolIndex_from_ControllerIndex(x),
            Function,
            event,
            str);
        }
      }
      return true;
    case CPlugin::Function::CPLUGIN_CONNECT_SUCCESS:
    case CPlugin::Function::CPLUGIN_CONNECT_FAIL:
      break;
  }

  return false;
}

// Check if there is any controller enabled.
bool anyControllerEnabled() {
  for (controllerIndex_t x = 0; x < CONTROLLER_MAX; x++) {
    if (Settings.ControllerEnabled[x] && supportedCPluginID(Settings.Protocol[x])) {
      return true;
    }
  }
  return false;
}

// Find first enabled controller index with this protocol
controllerIndex_t findFirstEnabledControllerWithId(cpluginID_t cpluginid) {
  if (supportedCPluginID(cpluginid)) {
    for (controllerIndex_t i = 0; i < CONTROLLER_MAX; i++) {
      if ((Settings.Protocol[i] == cpluginid) && Settings.ControllerEnabled[i]) {
        return i;
      }
    }
  }
  return INVALID_CONTROLLER_INDEX;
}

bool validProtocolIndex(protocolIndex_t index)
{
  return validProtocolIndex_init(index);
}

/*
   bool validControllerIndex(controllerIndex_t index)
   {
   return index < CONTROLLER_MAX;
   }
 */
bool validCPluginID(cpluginID_t cpluginID)
{
  return getProtocolIndex_from_CPluginID_(cpluginID) != INVALID_PROTOCOL_INDEX;
}

bool supportedCPluginID(cpluginID_t cpluginID)
{
  return validProtocolIndex(getProtocolIndex_from_CPluginID_(cpluginID));
}

protocolIndex_t getProtocolIndex_from_ControllerIndex(controllerIndex_t index) {
  if (validControllerIndex(index)) {
    return getProtocolIndex_from_CPluginID_(Settings.Protocol[index]);
  }
  return INVALID_PROTOCOL_INDEX;
}

protocolIndex_t getProtocolIndex_from_CPluginID(cpluginID_t cpluginID) {
  return getProtocolIndex_from_CPluginID_(cpluginID);
}

cpluginID_t getCPluginID_from_ProtocolIndex(protocolIndex_t index) {
  return getCPluginID_from_ProtocolIndex_(index);
}

cpluginID_t getCPluginID_from_ControllerIndex(controllerIndex_t index) {
  const protocolIndex_t protocolIndex = getProtocolIndex_from_ControllerIndex(index);

  return getCPluginID_from_ProtocolIndex(protocolIndex);
}

String getCPluginNameFromProtocolIndex(protocolIndex_t ProtocolIndex) {
  String controllerName;

  if (validProtocolIndex(ProtocolIndex)) {
    CPluginCall(ProtocolIndex, CPlugin::Function::CPLUGIN_GET_DEVICENAME, nullptr, controllerName);
  }
  return controllerName;
}

String getCPluginNameFromCPluginID(cpluginID_t cpluginID) {
  protocolIndex_t protocolIndex = getProtocolIndex_from_CPluginID_(cpluginID);

  if (!validProtocolIndex(protocolIndex)) {
    return strformat(F("CPlugin %d not included in build"), cpluginID);
  }
  return getCPluginNameFromProtocolIndex(protocolIndex);
}

#include "../Globals/ESPEasy_Scheduler.h"

ESPEasy_Scheduler Scheduler;
#include "../Globals/SecuritySettings.h"


SecurityStruct SecuritySettings;
ExtendedControllerCredentialsStruct ExtendedControllerCredentials;
#include "../Globals/ResetFactoryDefaultPref.h"
#include "../DataStructs/FactoryDefaultPref.h"

ResetFactoryDefaultPreference_struct ResetFactoryDefaultPreference;
#include "../Globals/RamTracker.h"

#ifndef BUILD_NO_RAM_TRACKER

#if FEATURE_TIMING_STATS
#include "../DataStructs/TimingStats.h"
#endif

#include "../ESPEasyCore/ESPEasy_Log.h"

#include "../Globals/Settings.h"
#include "../Globals/Statistics.h"

#include "../Helpers/Memory.h"
#include "../Helpers/Misc.h"
#include "../Helpers/StringConverter.h"

RamTracker myRamTracker;

/********************************************************************************************\
   Global convenience functions calling RamTracker
 \*********************************************************************************************/

void checkRAMtoLog(void){
  myRamTracker.getTraceBuffer();
}

void checkRAM(const String &flashString, int a ) {
  checkRAM(flashString, String(a));
}

void checkRAM(const __FlashStringHelper * flashString, int a) {
  checkRAM(String(flashString), String(a));
}


void checkRAM(const __FlashStringHelper * flashString, const String& a) {
  checkRAM(String(flashString), a);
}

void checkRAM(const __FlashStringHelper * flashString, const __FlashStringHelper * a) {
  checkRAM_values values;
  if (!values.mustContinue()) return;
  String s = flashString;
  s += F(" (");
  s += a;
  s += ')';
  checkRAM(values, s);
}

void checkRAM(const String& flashString, const String &a ) {
  checkRAM_values values;
  if (!values.mustContinue()) return;
  String s = flashString;
  s += F(" (");
  s += a;
  s += ')';
  checkRAM(values, s);
}

void checkRAM(const __FlashStringHelper * descr ) {
  checkRAM_values values;
  if (values.mustContinue()) 
    checkRAM(values, String(descr));
}

void checkRAM_PluginCall_task(uint8_t taskIndex, uint8_t Function) {
  checkRAM_values values;
  if (!values.mustContinue()) return;
  String s = concat(F("PluginCall_task_"), taskIndex + 1);

  s += F(" (");
  #if FEATURE_TIMING_STATS
  s += getPluginFunctionName(Function);
  #else // if FEATURE_TIMING_STATS
  s += String(Function);
  #endif // if FEATURE_TIMING_STATS
  s += ')';
  checkRAM(values, s);
}

void checkRAM(const String& descr ) {
  checkRAM_values values;
  checkRAM(values, descr);
}

checkRAM_values::checkRAM_values() {
  freeStack = getFreeStackWatermark();
#ifdef ESP32
  freeRAM = ESP.getMinFreeHeap();
#else
  freeRAM = FreeMem();
#endif
}

bool checkRAM_values::mustContinue() const {
  // Here we simply check to see if it is desired to continue creating a description string.
  // When no description string is created, it would still be nice to get some idea of the lowest stack/ram while we're here.
  if (Settings.EnableRAMTracking()) {
    return freeStack <= lowestFreeStack ||
           freeRAM <= lowestRAM;
  }

  if (freeStack <= lowestFreeStack) {
    lowestFreeStack = freeStack;
  }

  if (freeRAM <= lowestRAM)
  {
    lowestRAM = freeRAM;
  }
  return false;
}

void checkRAM(const checkRAM_values & values, const String& descr)
{
  if (Settings.EnableRAMTracking())
    myRamTracker.registerRamState(descr);

  if (values.freeStack <= lowestFreeStack) {
    lowestFreeStack = values.freeStack;
    lowestFreeStackfunction = descr;
  }

  if (values.freeRAM <= lowestRAM)
  {
    lowestRAM = values.freeRAM;
    lowestRAMfunction = std::move(descr);
  }
}

/********************************************************************************************\
   RamTracker class
 \*********************************************************************************************/



// find highest the trace with the largest minimum memory (gets replaced by worse one)
unsigned int RamTracker::bestCaseTrace(void) {
  unsigned int lowestMemoryInTrace      = 0;
  unsigned int lowestMemoryInTraceIndex = 0;

  for (int i = 0; i < TRACES; i++) {
    if (tracesMemory[i] > lowestMemoryInTrace) {
      lowestMemoryInTrace      = tracesMemory[i];
      lowestMemoryInTraceIndex = i;
    }
  }

  // serialPrintln(lowestMemoryInTraceIndex);
  return lowestMemoryInTraceIndex;
}

RamTracker::RamTracker(void) {
  readPtr  = 0;
  writePtr = 0;

  for (int i = 0; i < TRACES; i++) {
    free_string(traces[i]);
    tracesMemory[i] = 0xffffffff; // init with best case memory values, so they get replaced if memory goes lower
  }

  for (int i = 0; i < TRACEENTRIES; i++) {
    nextAction[i]            = "startup";
    nextActionStartMemory[i] = ESP.getFreeHeap(); // init with best case memory values, so they get replaced if memory goes lower
  }
}

void RamTracker::registerRamState(const String& s) {   // store function
  nextAction[writePtr]            = s;                 // name and mem
  nextActionStartMemory[writePtr] = ESP.getFreeHeap(); // in cyclic buffer.
  int bestCase = bestCaseTrace();                      // find best case memory trace

  if (ESP.getFreeHeap() < tracesMemory[bestCase]) {    // compare to current memory value
    free_string(traces[bestCase]);
    readPtr          = writePtr + 1;                   // read out buffer, oldest value first

    if (readPtr >= TRACEENTRIES) { 
      readPtr = 0;        // read pointer wrap around
    }
    tracesMemory[bestCase] = ESP.getFreeHeap();        // store new lowest value of that trace

    for (int i = 0; i < TRACEENTRIES; i++) {           // tranfer cyclic buffer strings and mem values to this trace
      traces[bestCase] += nextAction[readPtr];
      traces[bestCase] += F("-> ");
      traces[bestCase] += String(nextActionStartMemory[readPtr]);
      traces[bestCase] += ' ';
      readPtr++;

      if (readPtr >= TRACEENTRIES) { readPtr = 0; // wrap around read pointer
      }
    }
  }
  writePtr++;

  if (writePtr >= TRACEENTRIES) { writePtr = 0; // inc write pointer and wrap around too.
  }
}

 // return giant strings, one line per trace. Add stremToWeb method to avoid large strings.
void RamTracker::getTraceBuffer() {
#ifndef BUILD_NO_DEBUG
  if (Settings.EnableRAMTracking() && loglevelActiveFor(LOG_LEVEL_DEBUG_DEV)) {
    String retval = F("Memtrace\n");

    for (int i = 0; i < TRACES; i++) {
      retval += String(i);
      retval += F(": lowest: ");
      retval += String(tracesMemory[i]);
      retval += ' ';
      retval += traces[i];
      addLogMove(LOG_LEVEL_DEBUG_DEV, retval);
      retval.clear();
    }
  }
#endif // ifndef BUILD_NO_DEBUG
}

#else // BUILD_NO_RAM_TRACKER
/*

void checkRAMtoLog(void) {}

void checkRAM(const String& flashString,
              int           a) {}

void checkRAM(const String& flashString,
              const String& a) {}

void checkRAM(const String& descr) {}
*/

#endif // BUILD_NO_RAM_TRACKER
#include "../Globals/ESPEasyWiFiEvent.h"

#include "../../ESPEasy_common.h"


#ifdef ESP8266
WiFiEventHandler stationConnectedHandler;
WiFiEventHandler stationDisconnectedHandler;
WiFiEventHandler stationGotIpHandler;
WiFiEventHandler stationModeDHCPTimeoutHandler;
WiFiEventHandler stationModeAuthModeChangeHandler;
WiFiEventHandler APModeStationConnectedHandler;
WiFiEventHandler APModeStationDisconnectedHandler;
#endif // ifdef ESP8266

WiFiEventData_t WiFiEventData;



#include "../Globals/GlobalMapPortStatus.h"

MapPortStatus globalMapPortStatus;
#include "../Globals/Cache.h"

#include "../DataStructs/Caches.h"

void clearAllCaches()
{
  Cache.clearAllCaches();
}

void clearAllButTaskCaches()
{
  Cache.clearAllButTaskCaches();
}

void clearTaskCache(taskIndex_t TaskIndex)
{
  Cache.clearTaskCache(TaskIndex);
}

void clearFileCaches()
{
  Cache.clearFileCaches();
}

void updateActiveTaskUseSerial0()
{
  Cache.updateActiveTaskUseSerial0();
}

bool activeTaskUseSerial0()
{
  return Cache.activeTaskUseSerial0;
}

Caches Cache;

#include "../Globals/Statistics.h"


#ifndef BUILD_NO_RAM_TRACKER
uint32_t lowestRAM = 0;
String   lowestRAMfunction;
uint32_t lowestFreeStack = 0;
String   lowestFreeStackfunction;
#endif

uint8_t lastBootCause                           = BOOT_CAUSE_MANUAL_REBOOT;
SchedulerTimerID lastMixedSchedulerId_beforereboot(0);

unsigned long idle_msec_per_sec = 0;
unsigned long elapsed10ps       = 0;
unsigned long elapsed10psU      = 0;
unsigned long elapsed50ps       = 0;
unsigned long loopCounter       = 0;
unsigned long loopCounterLast   = 0;
unsigned long loopCounterMax    = 1;
uint32_t      lastLoopStart     = 0;
unsigned long shortestLoop      = 10000000;
unsigned long longestLoop       = 0;
unsigned long loopCounter_full  = 1;
float loop_usec_duration_total  = 0.0f;


unsigned long dailyResetCounter                   = 0;

ESPEASY_VOLATILE(unsigned long) sw_watchdog_callback_count{};


I2C_bus_state I2C_state = I2C_bus_state::OK;
unsigned long I2C_bus_cleared_count = 0;

#include "../Globals/RulesCalculate.h"

#include "../DataStructs/TimingStats.h"
#include "../Helpers/Numerical.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringConverter_Numerical.h"

RulesCalculate_t RulesCalculate{};

/*******************************************************************************************
* Helper functions to actually interact with the rules calculation functions.
* *****************************************************************************************/
int64_t CalculateParam(const String& TmpStr, int64_t errorValue) {
  int64_t returnValue = errorValue;

  if (TmpStr.length() == 0) {
    return returnValue;
  }

  // Minimize calls to the Calulate function.
  // Only if TmpStr starts with '=' then call Calculate(). Otherwise do not call it
  if (TmpStr[0] != '=') {
    if (!validInt64FromString(TmpStr, returnValue)) {
      return errorValue;
    }
  } else {
    ESPEASY_RULES_FLOAT_TYPE param{};

    // Starts with an '=', so Calculate starting at next position
    CalculateReturnCode returnCode = Calculate(TmpStr.substring(1), param);

    if (!isError(returnCode)) {
#ifndef BUILD_NO_DEBUG

      if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
        addLogMove(LOG_LEVEL_DEBUG,
                   strformat(F("CALCULATE PARAM: %s = %.6g"), TmpStr.c_str(), roundf(param)));
      }
#endif // ifndef BUILD_NO_DEBUG
      // return integer only as it's valid only for variable indices and device/task id
      #if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
      return round(param);
      #else
      return roundf(param);
      #endif
    }
  }
  return returnValue;
}

CalculateReturnCode Calculate_preProcessed(const String            & preprocessd_input,
                                           ESPEASY_RULES_FLOAT_TYPE& result)
{
  START_TIMER;
  CalculateReturnCode returnCode = RulesCalculate.doCalculate(
    preprocessd_input.c_str(),
    &result);

  STOP_TIMER(COMPUTE_STATS);
  return returnCode;
}

CalculateReturnCode Calculate(const String& input,
                              ESPEASY_RULES_FLOAT_TYPE& result
                              #if           FEATURE_STRING_VARIABLES
                              , const bool  logStringErrors
                              #endif // if FEATURE_STRING_VARIABLES
                              )
{
  CalculateReturnCode returnCode = Calculate_preProcessed(
    RulesCalculate_t::preProces(input),
    result);

  #ifndef LIMIT_BUILD_SIZE
  # if FEATURE_STRING_VARIABLES
  bool skipError = false;
  # endif // if FEATURE_STRING_VARIABLES

  if (isError(returnCode)) {
    if (loglevelActiveFor(LOG_LEVEL_ERROR)) {
      String log = F("Calculate: ");

      switch (returnCode) {
        case CalculateReturnCode::ERROR_STACK_OVERFLOW:
          log += F("Stack Overflow");
          break;
        case CalculateReturnCode::ERROR_BAD_OPERATOR:
          log += F("Bad Operator");
          break;
        case CalculateReturnCode::ERROR_PARENTHESES_MISMATCHED:
          log += F("Parenthesis mismatch");
          break;
        case CalculateReturnCode::ERROR_UNKNOWN_TOKEN:
          log += F("Unknown token");
          # if FEATURE_STRING_VARIABLES
          skipError = !logStringErrors;
          # endif // if FEATURE_STRING_VARIABLES
          break;
        case CalculateReturnCode::ERROR_TOKEN_LENGTH_EXCEEDED:
          log += strformat(F("Exceeded token length (%d)"), TOKEN_LENGTH);
          # if FEATURE_STRING_VARIABLES
          skipError = !logStringErrors;
          # endif // if FEATURE_STRING_VARIABLES
          break;
        case CalculateReturnCode::OK:
          // Already handled, but need to have all cases here so the compiler can warn if we're missing one.
          break;
      }

      # if FEATURE_STRING_VARIABLES

      if (!skipError)
      # endif // if FEATURE_STRING_VARIABLES
      {
        # ifndef BUILD_NO_DEBUG
        log += F(" input: ");
        log += input;
        log += F(" = ");

        const bool trimTrailingZeros = true;
        #  if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
        log += doubleToString(result, 6, trimTrailingZeros);
        #  else // if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
        log += floatToString(result, 6, trimTrailingZeros);
        #  endif // if FEATURE_USE_DOUBLE_AS_ESPEASY_RULES_FLOAT_TYPE
        # endif // ifndef BUILD_NO_DEBUG

        addLogMove(LOG_LEVEL_ERROR, log);
      }
    }
  }
  #endif // ifndef LIMIT_BUILD_SIZE
  return returnCode;
}

#include "../Globals/EventQueue.h"


EventQueueStruct eventQueue;
#include "../Globals/Nodes.h"

#if FEATURE_ESPEASY_P2P

NodesHandler Nodes;

#endif
#include "../Globals/RTC.h"

#include "../DataStructs/RTCStruct.h"


RTCStruct RTC;


#include "../Globals/RuntimeData.h"


std::map<String, ESPEASY_RULES_FLOAT_TYPE> customFloatVar;

#if FEATURE_STRING_VARIABLES
std::map<String, String> customStringVar;
#endif // if FEATURE_STRING_VARIABLES
//float UserVar[VARS_PER_TASK * TASKS_MAX];

UserVarStruct UserVar;


ESPEASY_RULES_FLOAT_TYPE getCustomFloatVar(String indexName, ESPEASY_RULES_FLOAT_TYPE defaultValue) {
  indexName.toLowerCase();
  auto it = customFloatVar.find(indexName);

  if (it != customFloatVar.end()) {
    return it->second;
  }
  return defaultValue;
}

void setCustomFloatVar(String indexName, const ESPEASY_RULES_FLOAT_TYPE& value) {
  indexName.toLowerCase();
  // std::map doesn't handle 2nd heap well, so make sure we keep using the default heap.
  # ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  # endif // ifdef USE_SECOND_HEAP

  customFloatVar[indexName] = value;
}

bool getNextCustomFloatVar(String& indexName, ESPEASY_RULES_FLOAT_TYPE& value) {
  String valueName(indexName);
  valueName.toLowerCase();
  auto it = customFloatVar.find(valueName);

  if (it == customFloatVar.end()) { return false; }
  ++it;

  if (it == customFloatVar.end()) { return false; }
  indexName = it->first;
  value = it->second;
  return true;
}

#if FEATURE_STRING_VARIABLES
String getCustomStringVar(String indexName, String defaultValue) {
  indexName.toLowerCase();
  auto it = customStringVar.find(indexName);

  if (it != customStringVar.end()) {
    return it->second;
  }
  return defaultValue;
}

void setCustomStringVar(String indexName, const String& value) {
  indexName.toLowerCase();
  // std::map doesn't handle 2nd heap well, so make sure we keep using the default heap.
  # ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  # endif // ifdef USE_SECOND_HEAP

  customStringVar[indexName] = value;
}

void clearCustomStringVar(String indexName) {
  indexName.toLowerCase();
  // std::map doesn't handle 2nd heap well, so make sure we keep using the default heap.
  # ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  # endif // ifdef USE_SECOND_HEAP

  auto it = customStringVar.find(indexName);

  if (it != customStringVar.end()) {
    customStringVar.erase(it);
  }
}

bool hasCustomStringVar(String indexName) {
  indexName.toLowerCase();
  auto it = customStringVar.find(indexName);

  return it != customStringVar.end();
}

bool getNextCustomStringVar(String& indexName, String& value) {
  String valueName(indexName);
  valueName.toLowerCase();
  auto it = customStringVar.find(valueName);

  if (it == customStringVar.end()) { return false; }
  ++it;

  if (it == customStringVar.end()) { return false; }
  indexName = it->first;
  value = it->second;
  return true;
}
#endif // if FEATURE_STRING_VARIABLES

#include "../Globals/TimeZone.h"

ESPEasy_time_zone time_zone;
#include "../Globals/Plugins.h"

#include "../CustomBuild/ESPEasyLimits.h"

#include "../../_Plugin_Helper.h"

#include "../DataStructs/ESPEasy_EventStruct.h"
#include "../DataStructs/TimingStats.h"

#include "../DataTypes/ESPEasy_plugin_functions.h"

#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/Serial.h"

#include "../Globals/Cache.h"
#include "../Globals/Device.h"
#include "../Globals/ESPEasy_Scheduler.h"
#include "../Globals/ExtraTaskSettings.h"
#include "../Globals/EventQueue.h"
#include "../Globals/GlobalMapPortStatus.h"
#include "../Globals/NetworkState.h"
#include "../Globals/Settings.h"
#include "../Globals/Statistics.h"

#if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
# include "../Helpers/_Plugin_Helper_serial.h"
#endif // if FEATURE_DEFINE_SERIAL_CONSOLE_PORT

#include "../Helpers/ESPEasyRTC.h"
#include "../Helpers/ESPEasy_Storage.h"
#include "../Helpers/Hardware_I2C.h"
#include "../Helpers/Misc.h"
#include "../Helpers/_Plugin_init.h"
#include "../Helpers/PortStatus.h"
#include "../Helpers/StringConverter.h"
#include "../Helpers/StringParser.h"

#include <vector>


bool validDeviceIndex(deviceIndex_t index) {
  return validDeviceIndex_init(index);
}

/*
   bool validTaskIndex(taskIndex_t index) {
   return index < TASKS_MAX;
   }

   bool validPluginID(pluginID_t pluginID) {
   return (pluginID != INVALID_PLUGIN_ID);
   }
 */
bool validPluginID_fullcheck(pluginID_t pluginID) {
  return getDeviceIndex_from_PluginID(pluginID) != INVALID_DEVICE_INDEX;
}

/*
   bool validUserVarIndex(userVarIndex_t index) {
   return index < USERVAR_MAX_INDEX;
   }

   bool validTaskVarIndex(taskVarIndex_t index) {
   return index < VARS_PER_TASK;
   }
 */
bool supportedPluginID(pluginID_t pluginID) {
  return validDeviceIndex(getDeviceIndex(pluginID));
}

deviceIndex_t getDeviceIndex_from_TaskIndex(taskIndex_t taskIndex) {
  if (validTaskIndex(taskIndex)) {
    return getDeviceIndex(Settings.getPluginID_for_task(taskIndex));
  }
  return INVALID_DEVICE_INDEX;
}

/*********************************************************************************************
 * get the taskPluginID with required checks, INVALID_PLUGIN_ID when invalid
 ********************************************************************************************/
pluginID_t getPluginID_from_TaskIndex(taskIndex_t taskIndex) {
  if (validTaskIndex(taskIndex)) {
    const pluginID_t pluginID = Settings.getPluginID_for_task(taskIndex);

    if (supportedPluginID(pluginID)) {
      return pluginID;
    }
  }
  return INVALID_PLUGIN_ID;
}

#if FEATURE_PLUGIN_PRIORITY
bool isPluginI2CPowerManager_from_TaskIndex(taskIndex_t taskIndex, uint8_t i2cBus) {
  if (validTaskIndex(taskIndex)) {
    # if FEATURE_I2C_MULTIPLE

    if (Settings.getI2CInterface(taskIndex) != i2cBus) {
      return false;
    }
    # endif // if FEATURE_I2C_MULTIPLE
    deviceIndex_t deviceIndex = getDeviceIndex_from_TaskIndex(taskIndex);

    if (validDeviceIndex(deviceIndex)) {
      return (Device[deviceIndex].Type == DEVICE_TYPE_I2C) &&
             Device[deviceIndex].PowerManager &&
             Settings.isPowerManagerTask(taskIndex);
    }
  }
  return false;
}

#endif // if FEATURE_PLUGIN_PRIORITY

deviceIndex_t getDeviceIndex(pluginID_t pluginID)
{
  return getDeviceIndex_from_PluginID(pluginID);
}

/********************************************************************************************\
   Find name of plugin given the plugin device index..
 \*********************************************************************************************/
String getPluginNameFromDeviceIndex(deviceIndex_t deviceIndex) {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif // ifdef USE_SECOND_HEAP

  String deviceName;

  if (validDeviceIndex(deviceIndex)) {
    PluginCall(deviceIndex, PLUGIN_GET_DEVICENAME, nullptr, deviceName);
  }
  return deviceName;
}

String getPluginNameFromPluginID(pluginID_t pluginID) {
  deviceIndex_t deviceIndex = getDeviceIndex(pluginID);

  if (!validDeviceIndex(deviceIndex)) {
    return strformat(F("Plugin %d not included in build"), pluginID.value);
  }
  return getPluginNameFromDeviceIndex(deviceIndex);
}

#if FEATURE_I2C_DEVICE_SCAN
bool checkPluginI2CAddressFromDeviceIndex(deviceIndex_t deviceIndex, uint8_t i2cAddress) {
  bool hasI2CAddress = false;

  if (validDeviceIndex(deviceIndex)) {
    String dummy;
    struct EventStruct TempEvent;
    TempEvent.Par1 = i2cAddress;
    hasI2CAddress  = PluginCall(deviceIndex, PLUGIN_I2C_HAS_ADDRESS, &TempEvent, dummy);
  }
  return hasI2CAddress;
}

#endif // if FEATURE_I2C_DEVICE_SCAN

bool getPluginDisplayParametersFromTaskIndex(taskIndex_t taskIndex, uint16_t& x, uint16_t& y, uint16_t& r, uint16_t& colorDepth) {
  if (!validTaskIndex(taskIndex)) { return false; }
  const deviceIndex_t deviceIndex = getDeviceIndex_from_TaskIndex(taskIndex);

  if (validDeviceIndex(deviceIndex)) {
    const pluginID_t pluginID = getPluginID_from_DeviceIndex(deviceIndex);

    if (validPluginID(pluginID)) {
      String dummy;
      struct EventStruct TempEvent;
      TempEvent.setTaskIndex(taskIndex);

      if (PluginCall(deviceIndex, PLUGIN_GET_DISPLAY_PARAMETERS, &TempEvent, dummy)) {
        x          = TempEvent.Par1;
        y          = TempEvent.Par2;
        r          = TempEvent.Par3;
        colorDepth = TempEvent.Par4;
        return true;
      }
    }
  }
  return false;
}

#if FEATURE_I2C_GET_ADDRESS
uint8_t getTaskI2CAddress(taskIndex_t taskIndex) {
  uint8_t getI2CAddress           = 0;
  const deviceIndex_t deviceIndex = getDeviceIndex_from_TaskIndex(taskIndex);

  if (validTaskIndex(taskIndex) && validDeviceIndex(deviceIndex)) {
    String dummy;
    struct EventStruct TempEvent;
    TempEvent.setTaskIndex(taskIndex);
    TempEvent.Par1 = 0;

    if (PluginCall(deviceIndex, PLUGIN_I2C_GET_ADDRESS, &TempEvent, dummy)) {
      getI2CAddress = TempEvent.Par1;
    }
  }
  return getI2CAddress;
}

#endif // if FEATURE_I2C_GET_ADDRESS


// ********************************************************************************
// Functions to assist changing I2C multiplexer port or clock speed
// when addressing a task
// ********************************************************************************

bool prepare_I2C_by_taskIndex(taskIndex_t taskIndex, deviceIndex_t DeviceIndex) {
  if (!validTaskIndex(taskIndex) || !validDeviceIndex(DeviceIndex)) {
    return false;
  }

  if (Device[DeviceIndex].Type != DEVICE_TYPE_I2C) {
    return true; // No I2C task, so consider all-OK
  }

  if (!Settings.isI2CEnabled(Settings.getI2CInterface(taskIndex))) {
    return false; // Plugin-selected I2C bus is not configured, fail
  }

  if (I2C_state != I2C_bus_state::OK) {
    return false; // Bus state is not OK, so do not consider task runnable
  }

  #if FEATURE_I2C_MULTIPLE
  const uint8_t i2cBus = Settings.getI2CInterface(taskIndex);
  #else // if FEATURE_I2C_MULTIPLE
  const uint8_t i2cBus = 0;
  #endif // if FEATURE_I2C_MULTIPLE

  if (bitRead(Settings.I2C_Flags[taskIndex], I2C_FLAGS_SLOW_SPEED)) {
    I2CSelectLowClockSpeed(i2cBus);  // Set to slow, also switch the bus
  } else {
    I2CSelectHighClockSpeed(i2cBus); // Set to normal, also switch the bus
  }

  #if FEATURE_I2CMULTIPLEXER
  I2CMultiplexerSelectByTaskIndex(taskIndex);

  // Output is selected after this write, so now we must make sure the
  // frequency is set before anything else is sent.
  #endif // if FEATURE_I2CMULTIPLEXER

  return true;
}

void post_I2C_by_taskIndex(taskIndex_t taskIndex, deviceIndex_t DeviceIndex) {
  if (!validTaskIndex(taskIndex) || !validDeviceIndex(DeviceIndex)) {
    return;
  }

  if (Device[DeviceIndex].Type != DEVICE_TYPE_I2C) {
    return;
  }
  #if FEATURE_I2C_MULTIPLE
  const uint8_t i2cBus = Settings.getI2CInterface(taskIndex);
  #else // if FEATURE_I2C_MULTIPLE
  const uint8_t i2cBus = 0;
  #endif // ifdef ESP32
  #if FEATURE_I2CMULTIPLEXER
  I2CMultiplexerOff(i2cBus);
  #endif // if FEATURE_I2CMULTIPLEXER

  I2CSelectHighClockSpeed(i2cBus); // Reset, stay on current bus
}

// Add an event to the event queue.
// event value 1 = taskIndex (first task = 1)
// event value 2 = return value of the plugin function
// Example:  TaskInit#bme=1,0    (taskindex = 0, return value = 0)
void queueTaskEvent(const String& eventName, taskIndex_t taskIndex, const String& value_str) {
  if (Settings.UseRules) {
    String event = strformat(
      F("%s#%s=%d"),
      eventName.c_str(),
      getTaskDeviceName(taskIndex).c_str(),
      taskIndex + 1);

    if (value_str.length() > 0) {
      event += ',';
      event += wrapWithQuotesIfContainsParameterSeparatorChar(value_str);
    }
    eventQueue.addMove(std::move(event));
  }
}

void queueTaskEvent(const String& eventName, taskIndex_t taskIndex, const int& value1) {
  queueTaskEvent(eventName, taskIndex, String(value1));
}

void queueTaskEvent(const __FlashStringHelper *eventName, taskIndex_t taskIndex, const String& value1) {
  queueTaskEvent(String(eventName), taskIndex, value1);
}

void queueTaskEvent(const __FlashStringHelper *eventName, taskIndex_t taskIndex, const int& value1) {
  queueTaskEvent(String(eventName), taskIndex, String(value1));
}

void loadDefaultTaskValueNames_ifEmpty(taskIndex_t TaskIndex) {
  String  oldNames[VARS_PER_TASK];
  uint8_t oldNrDec[VARS_PER_TASK];

  LoadTaskSettings(TaskIndex);

  for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
    oldNames[i] = ExtraTaskSettings.TaskDeviceValueNames[i];
    oldNrDec[i] = ExtraTaskSettings.TaskDeviceValueDecimals[i];
    oldNames[i].trim();
  }

  struct EventStruct TempEvent(TaskIndex);
  String dummy;

  // the plugin call should populate ExtraTaskSettings with its default values.
  PluginCall(PLUGIN_GET_DEVICEVALUENAMES, &TempEvent, dummy);

  // Restore the settings that were already set by the user
  for (uint8_t i = 0; i < VARS_PER_TASK; ++i) {
    const bool isDefault = oldNames[i].isEmpty();
    ExtraTaskSettings.isDefaultTaskVarName(i, isDefault);

    if (!isDefault) {
      ExtraTaskSettings.setTaskDeviceValueName(i, oldNames[i]);
      ExtraTaskSettings.TaskDeviceValueDecimals[i] = oldNrDec[i];
    }
  }
}

/**
 * Call the plugin of 1 task for 1 function, with standard EventStruct and optional command string
 */
bool PluginCallForTask(taskIndex_t taskIndex, uint8_t Function, EventStruct *TempEvent, String& command, EventStruct *event = nullptr) {
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif // ifdef USE_SECOND_HEAP

  bool retval                    = false;
  const bool considerTaskEnabled = Settings.TaskDeviceEnabled[taskIndex];

  // || (Settings.TaskDeviceEnabled[taskIndex].enabled && Function == PLUGIN_INIT);

  if (considerTaskEnabled) {
    if (validPluginID_fullcheck(Settings.getPluginID_for_task(taskIndex)))
    {
      const deviceIndex_t DeviceIndex = getDeviceIndex_from_TaskIndex(taskIndex);

      if (validDeviceIndex(DeviceIndex)) {
        if (Function == PLUGIN_INIT) {
          UserVar.clear_computed(taskIndex);
          LoadTaskSettings(taskIndex);
        }

        if (Settings.TaskDeviceDataFeed[taskIndex] == 0) // these calls only to tasks with local feed
        {
          TempEvent->setTaskIndex(taskIndex);

          // Need to 'clear' the sensorType first, before calling getSensorType()
          TempEvent->sensorType = Sensor_VType::SENSOR_TYPE_NOT_SET;
          TempEvent->getSensorType();

          if (event != nullptr) {
            TempEvent->OriginTaskIndex = event->TaskIndex;
          }

          if (!prepare_I2C_by_taskIndex(taskIndex, DeviceIndex)) {
            return false;
          }
          #ifndef BUILD_NO_RAM_TRACKER

          if (Settings.EnableRAMTracking()) {
            switch (Function) {
              case PLUGIN_WRITE:         // First set
              case PLUGIN_REQUEST:
              case PLUGIN_ONCE_A_SECOND: // Second set
              case PLUGIN_TEN_PER_SECOND:
              case PLUGIN_FIFTY_PER_SECOND:
              case PLUGIN_INIT:          // Second set, instead of PLUGIN_INIT_ALL
              case PLUGIN_CLOCK_IN:
              case PLUGIN_TIME_CHANGE:
            # if FEATURE_PLUGIN_PRIORITY
              case PLUGIN_PRIORITY_INIT:
            # endif // if FEATURE_PLUGIN_PRIORITY
              {
                checkRAM(F("PluginCall_s"), taskIndex);
                break;
              }
            }
          }
          #endif // ifndef BUILD_NO_RAM_TRACKER
          #if FEATURE_I2C_DEVICE_CHECK
          bool i2cStatusOk = true;

          if ((Function == PLUGIN_INIT) && (Device[DeviceIndex].Type == DEVICE_TYPE_I2C) && !Device[DeviceIndex].I2CNoDeviceCheck) {
            const uint8_t i2cAddr = getTaskI2CAddress(taskIndex);

            if (i2cAddr > 0) {
              START_TIMER;
              i2cStatusOk = I2C_deviceCheck(i2cAddr);
              STOP_TIMER_TASK(DeviceIndex, PLUGIN_I2C_GET_ADDRESS);
            }
          }

          if (i2cStatusOk) {
          #endif // if FEATURE_I2C_DEVICE_CHECK
            #ifndef BUILD_NO_RAM_TRACKER

          if (Settings.EnableRAMTracking()) {
            switch (Function) {
              case PLUGIN_WRITE:         // First set
              case PLUGIN_REQUEST:
              case PLUGIN_ONCE_A_SECOND: // Second set
              case PLUGIN_TEN_PER_SECOND:
              case PLUGIN_FIFTY_PER_SECOND:
              case PLUGIN_INIT:          // Second set, instead of PLUGIN_INIT_ALL
              case PLUGIN_CLOCK_IN:
              case PLUGIN_TIME_CHANGE:
              {
                checkRAM(F("PluginCall_s"), taskIndex);
                break;
              }
            }
          }
            #endif // ifndef BUILD_NO_RAM_TRACKER

          if (Function == PLUGIN_INIT) {
            // Schedule the plugin to be read.
            // Do this before actual init, to allow the plugin to schedule a specific first read.
            Scheduler.schedule_task_device_timer_at_init(TempEvent->TaskIndex);
          }

          START_TIMER;
          retval = (PluginCall(DeviceIndex, Function, TempEvent, command));

          STOP_TIMER_TASK(DeviceIndex, Function);

          if (Function == PLUGIN_INIT) {
              #if FEATURE_PLUGIN_STATS

            if (Device[DeviceIndex].PluginStats) {
              PluginTaskData_base *taskData = getPluginTaskData(taskIndex);

              if (taskData == nullptr) {
                // Plugin apparently does not have PluginTaskData.
                // Create Plugin Task data if it has "Stats" checked.
                LoadTaskSettings(taskIndex);

                if (ExtraTaskSettings.anyEnabledPluginStats()) {
                  special_initPluginTaskData(taskIndex, _StatsOnly_data_struct);
                }
              }
            }
              #endif // if FEATURE_PLUGIN_STATS
            queueTaskEvent(F("TaskInit"), taskIndex, retval);
          }
          #if FEATURE_I2C_DEVICE_CHECK
        }
          #endif // if FEATURE_I2C_DEVICE_CHECK

          post_I2C_by_taskIndex(taskIndex, DeviceIndex);
          delay(0); // SMY: call delay(0) unconditionally
        } else {
          #if FEATURE_PLUGIN_STATS

          if ((Function == PLUGIN_INIT) && Device[DeviceIndex].PluginStats) {
            PluginTaskData_base *taskData = getPluginTaskData(taskIndex);

            if (taskData == nullptr) {
              // Plugin apparently does not have PluginTaskData.
              // Create Plugin Task data if it has "Stats" checked.
              LoadTaskSettings(taskIndex);

              if (ExtraTaskSettings.anyEnabledPluginStats()) {
                // Try to allocate in PSRAM or 2nd heap if possible
                special_initPluginTaskData(taskIndex, _StatsOnly_data_struct);
              }
            }
          }
          #endif // if FEATURE_PLUGIN_STATS
        }
      }
    }
  }
  return retval;
}

/*********************************************************************************************\
* Function call to all or specific plugins
\*********************************************************************************************/
bool PluginCall(uint8_t Function, struct EventStruct *event, String& str)
{
  #ifdef USE_SECOND_HEAP
  HeapSelectDram ephemeral;
  #endif // ifdef USE_SECOND_HEAP

  struct EventStruct TempEvent;

  if (event == nullptr) {
    event = &TempEvent;
  }
  else {
    TempEvent.deep_copy(*event);
  }

  #ifndef BUILD_NO_RAM_TRACKER
  checkRAM(F("PluginCall"), Function);
  #endif // ifndef BUILD_NO_RAM_TRACKER

  switch (Function)
  {
    // Unconditional calls to all plugins
    // FIXME TD-er: PLUGIN_UNCONDITIONAL_POLL is not being used at the moment

    /*
       case PLUGIN_UNCONDITIONAL_POLL:
       {

       const unsigned maxDeviceIndex = getNrBuiltInDeviceIndex();

       for (deviceIndex_t x; x < maxDeviceIndex; ++x) {
          START_TIMER;
          PluginCall(x, Function, event, str);
          STOP_TIMER_TASK(x, Function);
          delay(0); // SMY: call delay(0) unconditionally
       }
       return true;
       }
     */

    case PLUGIN_MONITOR:

      for (auto it = globalMapPortStatus.begin(); it != globalMapPortStatus.end(); ++it) {
        // only call monitor function if there the need to
        if (it->second.monitor || it->second.command || it->second.init) {
          TempEvent.Par1 = getPortFromKey(it->first);

          // initialize the "x" variable to synch with the pluginNumber if second.x == -1
          if (!validDeviceIndex(it->second.x)) { it->second.x = getDeviceIndex(getPluginFromKey(it->first)); }

          const deviceIndex_t DeviceIndex = it->second.x;

          if (validDeviceIndex(DeviceIndex))  {
            START_TIMER;
            #if FEATURE_I2C_MULTIPLE
            const uint8_t i2cBus = Settings.getI2CInterfacePCFMCP();
            I2CSelectHighClockSpeed(i2cBus); // Switch to requested bus, no need to switch back,
                                             // next I2C plugin call will switch to desired bus
            #endif // if FEATURE_I2C_MULTIPLE
            PluginCall(DeviceIndex, Function, &TempEvent, str);
            STOP_TIMER_TASK(DeviceIndex, Function);
          }
        }
      }
      return true;


    // Call to all plugins. Return at first match
    case PLUGIN_WRITE:
      //    case PLUGIN_REQUEST: @giig1967g: replaced by new function getGPIOPluginValues()
    {
      taskIndex_t firstTask = 0;
      taskIndex_t lastTask  = TASKS_MAX;
      String command        = String(str);           // Local copy to avoid warning in ExecuteCommand
      int dotPos            = command.indexOf('.');  // Find first period

      if ((Function == PLUGIN_WRITE)                 // Only applicable on PLUGIN_WRITE function
          && (dotPos > -1)) {                        // First precondition is just a quick check for a period (fail-fast strategy)
        const String arg0 = parseString(command, 1); // Get first argument
        dotPos = arg0.indexOf('.');

        if (dotPos > -1) {
          String thisTaskName = parseString(arg0, 1, '.'); // Extract taskname prefix
          removeChar(thisTaskName, '[');                   // Remove the optional square brackets
          removeChar(thisTaskName, ']');

          if (thisTaskName.length() > 0) {                 // Second precondition
            taskIndex_t thisTask = findTaskIndexByName(thisTaskName);

            if (!validTaskIndex(thisTask)) {               // Taskname not found or invalid, check for a task number?
              thisTask = static_cast<taskIndex_t>(atoi(thisTaskName.c_str()));

              if ((thisTask == 0) || (thisTask > TASKS_MAX)) {
                thisTask = INVALID_TASK_INDEX;
              } else {
                thisTask--; // 0-based
              }
            }

            if (validTaskIndex(thisTask)) {            // Known taskindex?
#ifdef USES_P022                                       // Exclude P022 as it has rather explicit differences in commands when used with the
                                                       // [<TaskName>]. prefix
              const pluginID_t pluginID = Settings.getPluginID_for_task(thisTask);

              if (Settings.TaskDeviceEnabled[thisTask] // and internally needs to know wether it was called with the taskname prefixed
                  && validPluginID_fullcheck(pluginID)
                  && (Settings.TaskDeviceDataFeed[thisTask] == 0)) {
                const deviceIndex_t DeviceIndex = getDeviceIndex_from_TaskIndex(thisTask);
                constexpr pluginID_t P022_PCA9685_PLUGIN_ID(22);

                if (validDeviceIndex(DeviceIndex) &&
                    (pluginID == P022_PCA9685_PLUGIN_ID) /* PLUGIN_ID_022 define no longer available, 'assume' 22 for now */) {
                  thisTask = INVALID_TASK_INDEX;
                }
              }

              if (validTaskIndex(thisTask)) {
#endif // ifdef USES_P022
              firstTask = thisTask;
              lastTask  = thisTask + 1;                  // Add 1 to satisfy the for condition
              command   = command.substring(dotPos + 1); // Remove [<TaskName>]. prefix
#ifdef USES_P022
            }
#endif // ifdef USES_P022
            }
          }
        }
      }

      // String info = F("PLUGIN_WRITE first: "); // To remove
      // info += firstTask;
      // info += F(" last: ");
      // info += lastTask;
      // addLog(LOG_LEVEL_INFO, info);

      for (taskIndex_t task = firstTask; task < lastTask; task++)
      {
        bool retval = PluginCallForTask(task, Function, &TempEvent, command);

        if (!retval) {
          if (1 == (lastTask - firstTask)) {
            // These plugin task data commands are generic, so only apply them on a specific task.
            // Don't try to match them on the first task that may have such data.
            PluginTaskData_base *taskData = getPluginTaskDataBaseClassOnly(task);

            if (nullptr != taskData) {
              if (taskData->plugin_write_base(event, command)) {
                retval = true;
              }
            }
          }
        }

        if (retval) {
          EventStruct CPlugin_ack_event;
          CPlugin_ack_event.deep_copy(TempEvent);
          CPlugin_ack_event.setTaskIndex(task);
          CPluginCall(CPlugin::Function::CPLUGIN_ACKNOWLEDGE, &CPlugin_ack_event, command);
          return true;
        }
      }

      /*
            if (Function == PLUGIN_REQUEST) {
              // @FIXME TD-er: work-around as long as gpio command is still performed in P001_switch.
              for (deviceIndex_t deviceIndex = 0; validDeviceIndex(deviceIndex); deviceIndex++) {
                if (PluginCall(deviceIndex, Function, event, str)) {
                  delay(0); // SMY: call delay(0) unconditionally
                  CPluginCall(CPlugin::Function::CPLUGIN_ACKNOWLEDGE, event, str);
                  return true;
                }
              }
            }
       */
      break;
    }

    // Call to all plugins used in a task. Return at first match
    case PLUGIN_SERIAL_IN:
    case PLUGIN_UDP_IN:
    {
      for (taskIndex_t taskIndex = 0; taskIndex < TASKS_MAX; taskIndex++)
      {
        if (Settings.TaskDeviceEnabled[taskIndex]) {
          if (PluginCallForTask(taskIndex, Function, &TempEvent, str)) {
            #ifndef BUILD_NO_RAM_TRACKER
            checkRAM(F("PluginCallUDP"), taskIndex);
            #endif // ifndef BUILD_NO_RAM_TRACKER
            return true;
          }
        }
      }
      return false;
    }

    // Call to all plugins that are used in a task
    case PLUGIN_ONCE_A_SECOND:
    case PLUGIN_TEN_PER_SECOND:
    case PLUGIN_FIFTY_PER_SECOND:
    case PLUGIN_INIT_ALL:
    case PLUGIN_CLOCK_IN:
    case PLUGIN_TIME_CHANGE:
    {
      if (Function == PLUGIN_INIT_ALL) {
        Function = PLUGIN_INIT;
      }
      bool result = true;

      for (taskIndex_t taskIndex = 0; taskIndex < TASKS_MAX; taskIndex++)
      {
        #ifndef BUILD_NO_DEBUG
        int freemem_begin{};

        if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
          freemem_begin = ESP.getFreeHeap();
        }
        #endif // ifndef BUILD_NO_DEBUG

        bool retval = PluginCallForTask(taskIndex, Function, &TempEvent, str, event);

        if (Function == PLUGIN_INIT) {
          UserVar.clear_computed(taskIndex);

          if (!retval && (Settings.TaskDeviceDataFeed[taskIndex] == 0)) {
            // Disable temporarily as PLUGIN_INIT failed
            // FIXME TD-er: Should reschedule call to PLUGIN_INIT????
            // What interval?  (see: PR #4793)
            // Settings.TaskDeviceEnabled[taskIndex].setRetryInit();
            // Scheduler.setPluginTaskTimer(10000, taskIndex, PLUGIN_INIT);
            Settings.TaskDeviceEnabled[taskIndex] = false;
            result                                = false;
          }
          #ifndef BUILD_NO_DEBUG

          if (loglevelActiveFor(LOG_LEVEL_DEBUG)) {
            // See also logMemUsageAfter()
            const int freemem_end = ESP.getFreeHeap();
            String log;

            if (reserve_special(log, 128)) {
              log  = F("After PLUGIN_INIT ");
              log += F(" task: ");

              if (taskIndex < 9) { log += ' '; }
              log += taskIndex + 1;

              while (log.length() < 30) { log += ' '; }
              log += F("Free mem after: ");
              log += freemem_end;

              while (log.length() < 53) { log += ' '; }
              log += F("plugin: ");
              log += freemem_begin - freemem_end;

              while (log.length() < 67) { log += ' '; }

              log += Settings.TaskDeviceEnabled[taskIndex] ? F("[ena]") : F("[dis]");

              while (log.length() < 73) { log += ' '; }
              log += getPluginNameFromDeviceIndex(getDeviceIndex_from_TaskIndex(taskIndex));

              addLogMove(LOG_LEVEL_DEBUG, log);
            }
          }
          #endif // ifndef BUILD_NO_DEBUG
        }
      }

      return result;
    }

    #if FEATURE_PLUGIN_PRIORITY
    case PLUGIN_PRIORITY_INIT_ALL:
    {
      if (Function == PLUGIN_PRIORITY_INIT_ALL) {
        addLogMove(LOG_LEVEL_INFO, F("INIT : Check for Priority tasks"));
        PluginInit(true); // Priority only, load plugins but don't initialize them yet
        Function = PLUGIN_PRIORITY_INIT;
      }

      for (taskIndex_t taskIndex = 0; taskIndex < TASKS_MAX; taskIndex++) {
        bool isPriority = PluginCallForTask(taskIndex, Function, &TempEvent, str, event);

        if ((Function == PLUGIN_PRIORITY_INIT) && isPriority) { // If this is a priority task, then initialize it, next PLUGIN_INIT call
                                                                // must be self-ignored by plugin!
          clearPluginTaskData(taskIndex);                       // Make sure any task data is actually cleared.

          if (PluginCallForTask(taskIndex, PLUGIN_INIT, &TempEvent, str, event) &&
              loglevelActiveFor(LOG_LEVEL_INFO)) {
            addLog(LOG_LEVEL_INFO, strformat(F("INIT : Started Priority task %d, [%s] %s"),
                                             taskIndex + 1,
                                             getTaskDeviceName(taskIndex).c_str(),
                                             getPluginNameFromDeviceIndex(getDeviceIndex_from_TaskIndex(taskIndex)).c_str()));
          }
        }
      }

      return true;
    }
    #endif // if FEATURE_PLUGIN_PRIORITY

    // Call to specific task which may interact with the hardware
    case PLUGIN_INIT:
    case PLUGIN_EXIT:
    case PLUGIN_WEBFORM_LOAD:
    case PLUGIN_WEBFORM_LOAD_ALWAYS:
    case PLUGIN_WEBFORM_LOAD_OUTPUT_SELECTOR:
    case PLUGIN_READ:
    case PLUGIN_GET_PACKED_RAW_DATA:
    case PLUGIN_TASKTIMER_IN:
    case PLUGIN_PROCESS_CONTROLLER_DATA:
    {
      // FIXME TD-er: Code duplication with PluginCallForTask
      if (!validTaskIndex(event->TaskIndex)) {
        return false;
      }

      if ((Function == PLUGIN_READ) || (Function == PLUGIN_INIT) || (Function == PLUGIN_PROCESS_CONTROLLER_DATA)) {
        if (!Settings.TaskDeviceEnabled[event->TaskIndex]) {
          return false;
        }

        if (Function == PLUGIN_INIT) {
          clearTaskCache(event->TaskIndex);
          UserVar.clear_computed(event->TaskIndex);
        }
      }
      const deviceIndex_t DeviceIndex = getDeviceIndex_from_TaskIndex(event->TaskIndex);

      if (validDeviceIndex(DeviceIndex)) {
        if (ExtraTaskSettings.TaskIndex != event->TaskIndex) {
          if ((Function == PLUGIN_READ) && Device[DeviceIndex].ErrorStateValues) {
            // PLUGIN_READ should not need to access ExtraTaskSettings except for what's already being cached.
            // Only exception is when ErrorStateValues is needed.
            // Therefore only need to call LoadTaskSettings for those tasks with ErrorStateValues
            LoadTaskSettings(event->TaskIndex);
          } else if ((Function == PLUGIN_INIT) || (Function == PLUGIN_WEBFORM_LOAD) || (Function == PLUGIN_WEBFORM_LOAD_ALWAYS)) {
            // LoadTaskSettings may call PLUGIN_GET_DEVICEVALUENAMES.
            LoadTaskSettings(event->TaskIndex);
          }
        }
        event->BaseVarIndex = event->TaskIndex * VARS_PER_TASK;

        #ifndef BUILD_NO_RAM_TRACKER
        checkRAM_PluginCall_task(event->TaskIndex, Function);
        #endif // ifndef BUILD_NO_RAM_TRACKER

        if (!prepare_I2C_by_taskIndex(event->TaskIndex, DeviceIndex)) {
          return false;
        }
        bool retval                  = false;
        const bool performPluginCall =
          (Function != PLUGIN_READ && Function != PLUGIN_INIT) ||
          (Settings.TaskDeviceDataFeed[event->TaskIndex] == 0);
        #if FEATURE_I2C_DEVICE_CHECK
        bool i2cStatusOk = true;

        if (Settings.TaskDeviceDataFeed[event->TaskIndex] == 0) {
          // Only for locally connected sensors, not virtual ones via p2p.
          if (((Function == PLUGIN_INIT) || (Function == PLUGIN_READ))
              && (Device[DeviceIndex].Type == DEVICE_TYPE_I2C) && !Device[DeviceIndex].I2CNoDeviceCheck) {
            const uint8_t i2cAddr = getTaskI2CAddress(event->TaskIndex);

            if (i2cAddr > 0) {
              START_TIMER;

              // Disable task when device is unreachable for 10 PLUGIN_READs or 1 PLUGIN_INIT
              i2cStatusOk = I2C_deviceCheck(i2cAddr, event->TaskIndex, Function == PLUGIN_INIT ? 1 : 10);
              STOP_TIMER_TASK(DeviceIndex, PLUGIN_I2C_GET_ADDRESS);
            }
          }
        }

        if (i2cStatusOk) {
        #endif // if FEATURE_I2C_DEVICE_CHECK
        START_TIMER;

        if (((Function == PLUGIN_INIT) ||
             (Function == PLUGIN_WEBFORM_LOAD)) &&
            Device[DeviceIndex].ErrorStateValues) { // Only when we support ErrorStateValues
          // FIXME TD-er: Not sure if this should be called here.
          // It may be better if ranges are set in the call for default values and error values set via PLUGIN_INIT.
          // Also these may be plugin specific so perhaps create a helper function to load/save these values and call these helpers from the
          // plugin code.
          PluginCall(DeviceIndex, PLUGIN_INIT_VALUE_RANGES, event, str); // Initialize value range(s)
        }

        if ((Function == PLUGIN_INIT)
              #if FEATURE_PLUGIN_PRIORITY
            && !Settings.isPriorityTask(event->TaskIndex) // Don't clear already initialized PriorityTask data
              #endif // if FEATURE_PLUGIN_PRIORITY
            ) {
          // Make sure any task data is actually cleared.
          clearPluginTaskData(event->TaskIndex);

          /*
           #if FEATURE_DEFINE_SERIAL_CONSOLE_PORT
             if (Device[DeviceIndex].isSerial()) {
              checkSerialConflict(
                serialHelper_getSerialType(event),
                serialHelper_getRxPin(event),
                serialHelper_getTxPin(event));
             }
           #endif
           */
        }

        if (performPluginCall) {
          retval = PluginCall(DeviceIndex, Function, event, str);
        } else {
          retval = event->Source == EventValueSource::Enum::VALUE_SOURCE_UDP;
        }

        if (Function == PLUGIN_READ) {
          if (!retval) {
            String errorStr;

            if (PluginCall(DeviceIndex, PLUGIN_READ_ERROR_OCCURED, event, errorStr))
            {
              // Apparently the last read call resulted in an error
              // Send event indicating the error.
              queueTaskEvent(F("TaskError"), event->TaskIndex, errorStr);
            }
          } else {
            // Must be done as soon as there are new values, so we can keep a copy of the previous value
            // This previous value may be needed in formulas using %pvalue%
            UserVar.markPluginRead(event->TaskIndex);
              #if FEATURE_PLUGIN_STATS
            PluginTaskData_base *taskData = getPluginTaskDataBaseClassOnly(event->TaskIndex);

            if (taskData != nullptr) {
              // FIXME TD-er: Must make this flag configurable
              const bool onlyUpdateTimestampWhenSame = true;
              const bool trackpeaks                  =
                Settings.TaskDeviceDataFeed[event->TaskIndex] != 0 || // Receive data from remote node
                !Device[DeviceIndex].TaskLogsOwnPeaks;

              taskData->pushPluginStatsValues(
                event,
                trackpeaks,
                onlyUpdateTimestampWhenSame);
            }
              #endif // if FEATURE_PLUGIN_STATS
            saveUserVarToRTC();
          }
        }

        if (Function == PLUGIN_INIT) {
          if (!retval && (Settings.TaskDeviceDataFeed[event->TaskIndex] == 0)) {
            // Disable temporarily as PLUGIN_INIT failed
            // FIXME TD-er: Should reschedule call to PLUGIN_INIT????
            Settings.TaskDeviceEnabled[event->TaskIndex] = false;
          } else {
              #if FEATURE_PLUGIN_STATS

            if (Device[DeviceIndex].PluginStats) {
              PluginTaskData_base *taskData = getPluginTaskData(event->TaskIndex);

              if (taskData == nullptr) {
                // Plugin apparently does not have PluginTaskData.
                // Create Plugin Task data if it has "Stats" checked.
                LoadTaskSettings(event->TaskIndex);

                if (ExtraTaskSettings.anyEnabledPluginStats()) {
                  special_initPluginTaskData(event->TaskIndex, _StatsOnly_data_struct);
                }
              }
            }
              #endif // if FEATURE_PLUGIN_STATS
            // Schedule the plugin to be read.
            Scheduler.schedule_task_device_timer_at_init(TempEvent.TaskIndex);
            queueTaskEvent(F("TaskInit"), event->TaskIndex, retval);
          }
        }

        if (Function == PLUGIN_EXIT) {
          UserVar.clear_computed(event->TaskIndex);
          clearPluginTaskData(event->TaskIndex);

          //          clearTaskCache(event->TaskIndex);

          //            initSerial();
          queueTaskEvent(F("TaskExit"), event->TaskIndex, retval);
          updateActiveTaskUseSerial0();
        }
        STOP_TIMER_TASK(DeviceIndex, Function);
        #if FEATURE_I2C_DEVICE_CHECK
      }
        #endif // if FEATURE_I2C_DEVICE_CHECK
        post_I2C_by_taskIndex(event->TaskIndex, DeviceIndex);
        delay(0); // SMY: call delay(0) unconditionally

        return retval;
      }
      return false;
    }

    // Call to specific task not interacting with hardware
    case PLUGIN_GET_CONFIG_VALUE:
    case PLUGIN_GET_DEVICEVALUENAMES:
    case PLUGIN_GET_DEVICEGPIONAMES:
    case PLUGIN_WEBFORM_SAVE:
    case PLUGIN_WEBFORM_SHOW_VALUES:
    case PLUGIN_WEBFORM_SHOW_CONFIG:
    case PLUGIN_WEBFORM_SHOW_I2C_PARAMS:
    case PLUGIN_WEBFORM_PRE_SERIAL_PARAMS:
    case PLUGIN_WEBFORM_SHOW_SERIAL_PARAMS:
    case PLUGIN_WEBFORM_SHOW_GPIO_DESCR:
    #if FEATURE_PLUGIN_STATS
    case PLUGIN_WEBFORM_LOAD_SHOW_STATS:
    #endif // if FEATURE_PLUGIN_STATS
    case PLUGIN_SET_CONFIG:
    case PLUGIN_SET_DEFAULTS:
    case PLUGIN_I2C_HAS_ADDRESS:
    case PLUGIN_WEBFORM_SHOW_ERRORSTATE_OPT:
    case PLUGIN_INIT_VALUE_RANGES:
    #ifdef USES_ESPEASY_NOW
    case PLUGIN_FILTEROUT_CONTROLLER_DATA:
    #endif // ifdef USES_ESPEASY_NOW
    #if FEATURE_MQTT_DISCOVER || FEATURE_CUSTOM_TASKVAR_VTYPE
    case PLUGIN_GET_DISCOVERY_VTYPES:
    #endif // if FEATURE_MQTT_DISCOVER || FEATURE_CUSTOM_TASKVAR_VTYPE
    #if FEATURE_TASKVALUE_UNIT_OF_MEASURE
    case PLUGIN_GET_UOM_GROUPS:
    #endif // if FEATURE_TASKVALUE_UNIT_OF_MEASURE

    #if FEATURE_MQTT
    case PLUGIN_MQTT_CONNECTION_STATE:
    case PLUGIN_MQTT_IMPORT:
    #endif // if FEATURE_MQTT
    {
      START_TIMER;
      const deviceIndex_t DeviceIndex = getDeviceIndex_from_TaskIndex(event->TaskIndex);

      if (validDeviceIndex(DeviceIndex)) {
        if ((Function == PLUGIN_GET_DEVICEVALUENAMES) ||
            (Function == PLUGIN_WEBFORM_SAVE) ||
            (Function == PLUGIN_WEBFORM_LOAD) ||
            (Function == PLUGIN_WEBFORM_LOAD_ALWAYS) ||
            (Function == PLUGIN_SET_DEFAULTS) ||
            (Function == PLUGIN_INIT_VALUE_RANGES) ||
            (Function == PLUGIN_WEBFORM_SHOW_SERIAL_PARAMS) ||
            (Function == PLUGIN_WEBFORM_PRE_SERIAL_PARAMS)
            #if FEATURE_MQTT_DISCOVER || FEATURE_CUSTOM_TASKVAR_VTYPE
            || (Function == PLUGIN_GET_DISCOVERY_VTYPES)
            #endif // if FEATURE_MQTT_DISCOVER || FEATURE_CUSTOM_TASKVAR_VTYPE
            #if FEATURE_TASKVALUE_UNIT_OF_MEASURE
            || (Function == PLUGIN_GET_UOM_GROUPS)
            #endif // if FEATURE_TASKVALUE_UNIT_OF_MEASURE
            ) {
          LoadTaskSettings(event->TaskIndex);
        }
        event->BaseVarIndex = event->TaskIndex * VARS_PER_TASK;

        #ifndef BUILD_NO_RAM_TRACKER
        checkRAM_PluginCall_task(event->TaskIndex, Function);
        #endif // ifndef BUILD_NO_RAM_TRACKER

        if (Function == PLUGIN_SET_DEFAULTS) {
          for (int i = 0; i < VARS_PER_TASK; ++i) {
            UserVar.setFloat(event->TaskIndex, i, 0.0f);
          }
        }

        bool retval =  PluginCall(DeviceIndex, Function, event, str);

        // Calls may have updated ExtraTaskSettings, so validate them.
        ExtraTaskSettings.validate();

        if ((Function == PLUGIN_GET_DEVICEVALUENAMES) ||
            (Function == PLUGIN_WEBFORM_SAVE) ||
            (Function == PLUGIN_SET_DEFAULTS) ||
            (Function == PLUGIN_INIT_VALUE_RANGES) ||
            ((Function == PLUGIN_SET_CONFIG) && retval)) {
          // Each of these may update ExtraTaskSettings, but it may not have been saved yet.
          // Thus update the cache just in case something from it is requested from the cache.
          Cache.updateExtraTaskSettingsCache();
        }

        if (Function == PLUGIN_SET_DEFAULTS) {
          saveUserVarToRTC();
        }

        if ((Function == PLUGIN_GET_CONFIG_VALUE) && !retval) {
          // Try to match a statistical property of a task value.
          // e.g.: [taskname#valuename.avg]
          PluginTaskData_base *taskData = getPluginTaskDataBaseClassOnly(event->TaskIndex);

          if (nullptr != taskData) {
            if (taskData->plugin_get_config_value_base(event, str)) {
              retval = true;
            }
          }
        }


        STOP_TIMER_TASK(DeviceIndex, Function);
        delay(0); // SMY: call delay(0) unconditionally
        return retval;
      }
      return false;
    }

    // Frequently made call to specific task not interacting with hardware
    case PLUGIN_GET_DEVICEVALUECOUNT:
    case PLUGIN_GET_DEVICEVTYPE:
    case PLUGIN_FORMAT_USERVAR:
    {
      START_TIMER;
      const deviceIndex_t DeviceIndex = getDeviceIndex_from_TaskIndex(event->TaskIndex);

      if (validDeviceIndex(DeviceIndex)) {
        event->BaseVarIndex = event->TaskIndex * VARS_PER_TASK;

        #ifndef BUILD_NO_RAM_TRACKER

        //        checkRAM_PluginCall_task(event->TaskIndex, Function);
        #endif // ifndef BUILD_NO_RAM_TRACKER

        if (Function == PLUGIN_GET_DEVICEVALUECOUNT) {
          event->Par1 = Device[DeviceIndex].ValueCount;
        }

        if (Function == PLUGIN_GET_DEVICEVTYPE) {
          event->sensorType = Device[DeviceIndex].VType;
        }
        bool retval =  PluginCall(DeviceIndex, Function, event, str);

        if (Function == PLUGIN_GET_DEVICEVALUECOUNT) {
          // Check if we have a valid value count.
          if (Output_Data_type_t::Simple == Device[DeviceIndex].OutputDataType) {
            if ((event->Par1 < 1) || (event->Par1 > VARS_PER_TASK)) {
              // Output_Data_type_t::Simple only allows for 1 .. 4 output types.
              // Apparently the value is not correct, so use the default.
              event->Par1 = Device[DeviceIndex].ValueCount;
            }
          }
        }
        STOP_TIMER_TASK(DeviceIndex, Function);
        delay(0); // SMY: call delay(0) unconditionally
        return retval;
      }
      return false;
    }
  } // case
  return false;
}

#include "../Globals/WiFi_AP_Candidates.h"

WiFi_AP_CandidatesList WiFi_AP_Candidates;

#include "../Helpers/BusCmd_Handler_I2C.h"

/**
 * Constructor BusCmd_Handler_I2C
 */
BusCmd_Handler_I2C::BusCmd_Handler_I2C(uint8_t  i2cAddress,
                                       TwoWire *wire)
  : _i2cAddress(i2cAddress), _wire(wire)
{
  //
}

bool BusCmd_Handler_I2C::init() {
  return nullptr != _wire && _i2cAddress > 0 && _i2cAddress < 128;
}

uint8_t BusCmd_Handler_I2C::read8u() {
  return I2C_read8(_i2cAddress, &_ok);
}

uint16_t BusCmd_Handler_I2C::read16u() {
  return I2C_read16(_i2cAddress, &_ok);
}

uint32_t BusCmd_Handler_I2C::read24u() {
  return I2C_read24(_i2cAddress, &_ok);
}

uint32_t BusCmd_Handler_I2C::read32u() {
  return I2C_read32(_i2cAddress, &_ok);
}

uint8_t BusCmd_Handler_I2C::read8uREG(uint16_t reg,
                                      bool     wideReg) {
  return I2C_read8_reg(_i2cAddress, reg, &_ok);
}

uint16_t BusCmd_Handler_I2C::read16uREG(uint16_t reg,
                                        bool     wideReg) {
  return I2C_read16_reg(_i2cAddress, reg, &_ok);
}

uint32_t BusCmd_Handler_I2C::read24uREG(uint16_t reg,
                                        bool     wideReg) {
  return I2C_read24_reg(_i2cAddress, reg, &_ok);
}

uint32_t BusCmd_Handler_I2C::read32uREG(uint16_t reg,
                                        bool     wideReg) {
  return I2C_read32_reg(_i2cAddress, reg, &_ok);
}

std::vector<uint8_t>BusCmd_Handler_I2C::read8uB(uint32_t size) {
  std::vector<uint8_t> data_b;

  #ifdef ESP32

  if (size > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(size)) { // 0 return on error
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), size));
      # endif // ifndef BUILD_NO_DEBUG
      return data_b;
    }
  }
  #endif // ifdef ESP32
  data_b.reserve(size);

  _wire->requestFrom(_i2cAddress, size);

  for (uint8_t i = 0; i < size; ++i) {
    data_b.push_back(_wire->read());
  }
  #ifdef ESP32

  if (size > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  #endif // ifdef ESP32
  return data_b;
}

std::vector<uint16_t>BusCmd_Handler_I2C::read16uW(uint32_t size) {
  std::vector<uint16_t> data_w;

  #ifdef ESP32

  if ((size * 2) > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(size * 2)) { // 0 return on error
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), size * 2));
      # endif // ifndef BUILD_NO_DEBUG
      return data_w;
    }
  }
  #endif // ifdef ESP32
  data_w.reserve(size);

  _wire->requestFrom(_i2cAddress, size * 2);

  for (uint8_t i = 0; i < size; ++i) {
    data_w.push_back((_wire->read() << 8) | _wire->read());
  }
  #ifdef ESP32

  if ((size * 2) > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  #endif // ifdef ESP32
  return data_w;
}

std::vector<uint8_t>BusCmd_Handler_I2C::read8uBREG(uint16_t reg,
                                                   uint32_t size,
                                                   bool     wideReg) {
  std::vector<uint8_t> data_b;

  #ifdef ESP32

  if (size > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(size)) { // 0 return on error
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), size));
      # endif // ifndef BUILD_NO_DEBUG
      return data_b;
    }
  }
  #endif // ifdef ESP32
  data_b.reserve(size);

  if (I2C_write8(_i2cAddress, reg) && (_wire->requestFrom(_i2cAddress, size) == size)) {
    for (uint8_t i = 0; i < size; ++i) {
      data_b.push_back(_wire->read());
    }
  }
  #ifdef ESP32

  if (size > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  #endif // ifdef ESP32
  return data_b;
}

std::vector<uint16_t>BusCmd_Handler_I2C::read16uWREG(uint16_t reg,
                                                     uint32_t size,
                                                     bool     wideReg) {
  std::vector<uint16_t> data_w;

  #ifdef ESP32

  if ((size * 2) > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(size * 2)) { // 0 return on error
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), size * 2));
      # endif // ifndef BUILD_NO_DEBUG
      return data_w;
    }
  }
  #endif // ifdef ESP32
  data_w.reserve(size);

  if (I2C_write8(_i2cAddress, reg) && (_wire->requestFrom(_i2cAddress, size * 2) == size * 2)) {
    for (uint8_t i = 0; i < size; ++i) {
      data_w.push_back((_wire->read() << 8) | _wire->read());
    }
  }
  #ifdef ESP32

  if ((size * 2) > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  #endif // ifdef ESP32
  return data_w;
}

#if FEATURE_BUSCMD_STRING
String BusCmd_Handler_I2C::readString(uint32_t len) {
  String result;

  # ifdef ESP32

  if (len > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(len)) { // 0 return on error
      #  ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), len));
      #  endif // ifndef BUILD_NO_DEBUG
      return result;
    }
  }
  # endif // ifdef ESP32
  result.reserve(len);

  # ifdef ESP32
  const size_t maxBytesPerChunk = len;
  # endif // ifdef ESP32
  # ifdef ESP8266
  constexpr size_t maxBytesPerChunk = 128; // Chuncked in 32 byte blocks
  # endif // ifdef ESP8266
  size_t index = 0;

  while (index < len) {
    const size_t bytesRemaining = len - index;
    const size_t bytesThisChunk = min(bytesRemaining, maxBytesPerChunk);
    const size_t endIndex       = index + bytesThisChunk;
    const bool   isLastChunk    = (bytesRemaining <= maxBytesPerChunk);
    _wire->requestFrom(_i2cAddress, bytesThisChunk, isLastChunk);

    for (; index < endIndex; ++index) {
      # ifndef BUILD_NO_DEBUG

      if (_wire->available() < 1) {
        addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error reading data, got just: %u from %u requested bytes"), index, bytesThisChunk));
        break;
      }
      # endif // ifndef BUILD_NO_DEBUG
      const uint8_t c = _wire->read();

      if (c != 0) {
        result += static_cast<char>(c);
      }
    }
  }

  # ifdef ESP32

  if (len > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  # endif // ifdef ESP32
  return result;
}

String BusCmd_Handler_I2C::readStringREG(uint16_t reg,
                                         uint32_t len,
                                         bool     wideReg) {
  String result;

  # ifdef ESP32

  if (len > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(len)) { // 0 return on error
      #  ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), len));
      #  endif // ifndef BUILD_NO_DEBUG
      return result;
    }
  }
  # endif // ifdef ESP32

  result.reserve(len);
  _wire->requestFrom(_i2cAddress, len);

  if (I2C_write8(_i2cAddress, reg) && (_wire->requestFrom(_i2cAddress, len) == len)) {
    for (uint8_t i = 0; i < len; ++i) {
      result += static_cast<char>(_wire->read());
    }
  }
  # ifdef ESP32

  if (len > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  # endif // ifdef ESP32
  return result;
}

#endif // if FEATURE_BUSCMD_STRING

bool BusCmd_Handler_I2C::write8u(uint8_t data) {
  return I2C_write8(_i2cAddress, data);
}

bool BusCmd_Handler_I2C::write16u(uint16_t data) {
  return I2C_write16(_i2cAddress, data);
}

bool BusCmd_Handler_I2C::write24u(uint32_t data) {
  return I2C_write24(_i2cAddress, data);
}

bool BusCmd_Handler_I2C::write32u(uint32_t data) {
  return I2C_write32(_i2cAddress, data);
}

bool BusCmd_Handler_I2C::write8uREG(uint16_t reg,
                                    uint8_t  data,
                                    bool     wideReg) {
  return I2C_write8_reg(_i2cAddress, reg, data);
}

bool BusCmd_Handler_I2C::write16uREG(uint16_t reg,
                                     uint16_t data,
                                     bool     wideReg) {
  return I2C_write16_reg(_i2cAddress, reg, data);
}

bool BusCmd_Handler_I2C::write24uREG(uint16_t reg,
                                     uint32_t data,
                                     bool     wideReg) {
  return I2C_write24_reg(_i2cAddress, reg, data);
}

bool BusCmd_Handler_I2C::write32uREG(uint16_t reg,
                                     uint32_t data,
                                     bool     wideReg) {
  return I2C_write32_reg(_i2cAddress, reg, data);
}

uint32_t BusCmd_Handler_I2C::write8uB(std::vector<uint8_t>data) {
  const size_t size = data.size();

  #ifdef ESP32

  if (size > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(size)) { // 0 return on error
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), size));
      # endif // ifndef BUILD_NO_DEBUG
      return 0;
    }
  }
  #endif // ifdef ESP32
  _wire->beginTransmission(_i2cAddress);

  for (size_t itb = 0; itb < size; ++itb) {
    _wire->write(data[itb]);
  }
  #ifdef ESP32

  if (size > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  #endif // ifdef ESP32
  return _wire->endTransmission() == 0;
}

uint32_t BusCmd_Handler_I2C::write16uW(std::vector<uint16_t>data) {
  const size_t size = data.size();

  #ifdef ESP32

  if ((size * 2) > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(size * 2)) { // 0 return on error
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), size * 2));
      # endif // ifndef BUILD_NO_DEBUG
      return 0;
    }
  }
  #endif // ifdef ESP32
  _wire->beginTransmission(_i2cAddress);

  for (size_t itw = 0; itw < size; ++itw) {
    _wire->write((uint8_t)(data[itw] << 8));
    _wire->write((uint8_t)(data[itw] & 0xFF));
  }
  #ifdef ESP32

  if ((size * 2) > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  #endif // ifdef ESP32
  return _wire->endTransmission() == 0;
}

uint32_t BusCmd_Handler_I2C::write8uBREG(uint16_t            reg,
                                         std::vector<uint8_t>data,
                                         bool                wideReg) {
  const size_t size = data.size();

  #ifdef ESP32

  if (size > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(size)) { // 0 return on error
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), size));
      # endif // ifndef BUILD_NO_DEBUG
      return 0;
    }
  }
  #endif // ifdef ESP32
  _wire->beginTransmission(_i2cAddress);
  _wire->write((uint8_t)reg);

  for (size_t itb = 0; itb < size; ++itb) {
    _wire->write(data[itb]);
  }
  #ifdef ESP32

  if (size > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  #endif // ifdef ESP32
  return _wire->endTransmission() == 0;
}

uint32_t BusCmd_Handler_I2C::write16uWREG(uint16_t             reg,
                                          std::vector<uint16_t>data,
                                          bool                 wideReg) {
  const size_t size = data.size();

  #ifdef ESP32

  if ((size * 2) > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(size * 2)) { // 0 return on error
      # ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), size * 2));
      # endif // ifndef BUILD_NO_DEBUG
      return 0;
    }
  }
  #endif // ifdef ESP32
  _wire->beginTransmission(_i2cAddress);
  _wire->write((uint8_t)reg);

  for (size_t itw = 0; itw < size; ++itw) {
    _wire->write((uint8_t)(data[itw] << 8));
    _wire->write((uint8_t)(data[itw] & 0xFF));
  }
  #ifdef ESP32

  if ((size * 2) > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  #endif // ifdef ESP32
  return _wire->endTransmission() == 0;
}

#if FEATURE_BUSCMD_STRING
uint32_t BusCmd_Handler_I2C::writeString(const String& data) {
  const size_t len = data.length();

  # ifdef ESP32

  if (len > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(len)) { // 0 return on error
      #  ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), len));
      #  endif // ifndef BUILD_NO_DEBUG
      return 0;
    }
  }
  # endif // ifdef ESP32
  _wire->beginTransmission(_i2cAddress);

  for (size_t its = 0; its < len; ++its) {
    _wire->write(data[its]);
  }
  # ifdef ESP32

  if (len > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  # endif // ifdef ESP32
  return _wire->endTransmission() == 0;
}

uint32_t BusCmd_Handler_I2C::writeStringReg(uint16_t      reg,
                                            const String& data,
                                            bool          wideReg) {
  const size_t len = data.length();

  # ifdef ESP32

  if (len > I2C_BUFFER_LENGTH) {
    if (!_wire->setBufferSize(len)) { // 0 return on error
      #  ifndef BUILD_NO_DEBUG
      addLog(LOG_LEVEL_ERROR, strformat(F("I2C  : Error resizing buffer to size: %u"), len));
      #  endif // ifndef BUILD_NO_DEBUG
      return 0;
    }
  }
  # endif // ifdef ESP32
  _wire->beginTransmission(_i2cAddress);
  _wire->write((uint8_t)reg);

  for (size_t its = 0; its < len; ++its) {
    _wire->write(data[its]);
  }
  # ifdef ESP32

  if (len > I2C_BUFFER_LENGTH) {
    _wire->setBufferSize(I2C_BUFFER_LENGTH); // Restore default
  }
  # endif // ifdef ESP32
  return _wire->endTransmission() == 0;
}

#endif // if FEATURE_BUSCMD_STRING

#include "../Helpers/CUL_interval_filter.h"


#ifdef USES_P094

# include "../ESPEasyCore/ESPEasy_Log.h"
# include "../Globals/ESPEasy_time.h"
# include "../Globals/TimeZone.h"
# include "../Helpers/ESPEasy_time_calc.h"
# include "../Helpers/StringConverter.h"


CUL_time_filter_struct::CUL_time_filter_struct(uint32_t checksum, unsigned long UnixTimeExpiration)
  : _checksum(checksum), _UnixTimeExpiration(UnixTimeExpiration) {}

String CUL_interval_filter_getExpiration_log_str(const P094_filter& filter)
{
  const unsigned long expiration = filter.computeUnixTimeExpiration();

  if ((expiration != 0) && (expiration != 0xFFFFFFFF)) {
    struct tm exp_tm;
    breakTime(time_zone.toLocal(expiration), exp_tm);

    return concat(F(" Expiration: "), formatDateTimeString(exp_tm));
  }
  return EMPTY_STRING;
}

bool CUL_interval_filter::filter(const mBusPacket_t& packet, const P094_filter& filter)
{
  if (!enabled) {
    return true;
  }

  if (filter.getFilterWindow() == P094_Filter_Window::None) {
    // Will always be rejected, so no need to keep track of the message
    return false;
  }

  if (filter.getFilterWindow() == P094_Filter_Window::All) {
    // Will always be allowed, so no need to keep track of the message
    return true;
  }

  const uint32_t key = packet.deviceID_to_map_key();
  auto it            = _mBusFilterMap.find(key);

  if (it != _mBusFilterMap.end()) {
    // Already present
    if (node_time.getUnixTime() < it->second._UnixTimeExpiration) {
      if (loglevelActiveFor(LOG_LEVEL_INFO)) {
        String log = concat(F("CUL   : Interval filtered: "), packet.toString());
        log += CUL_interval_filter_getExpiration_log_str(filter);
        addLogMove(LOG_LEVEL_INFO, log);
      }
      return false;
    }

    if (packet._checksum == it->second._checksum) {
      if (loglevelActiveFor(LOG_LEVEL_INFO)) {
        addLogMove(LOG_LEVEL_INFO, concat(F("CUL   : Interval Same Checksum: "), packet.toString()));
      }
      return false;
    }

    // Has expired, so remove from filter map
    _mBusFilterMap.erase(it);
  }

  const unsigned long expiration = filter.computeUnixTimeExpiration();

  CUL_time_filter_struct item(packet._checksum, expiration);

  _mBusFilterMap[key] = item;

  if (loglevelActiveFor(LOG_LEVEL_INFO)) {
    String log = concat(F("CUL   : Add to IntervalFilter: "), packet.toString());
    log += CUL_interval_filter_getExpiration_log_str(filter);

    addLogMove(LOG_LEVEL_INFO, log);
  }

  return true;
}

void CUL_interval_filter::purgeExpired()
{
  auto it = _mBusFilterMap.begin();

  const unsigned long currentTime = node_time.getUnixTime();

  for (; it != _mBusFilterMap.end();) {
    if (currentTime > it->second._UnixTimeExpiration) {
      it = _mBusFilterMap.erase(it);
    } else {
      ++it;
    }
  }
}

#endif // ifdef USES_P094

#include "../Helpers/Hardware_device_info.h"

#ifdef ESP32C2

# include <soc/soc.h>
# include <soc/efuse_reg.h>
# include <soc/spi_reg.h>
# include <soc/spi_pins.h>
# include <soc/rtc.h>
# include <esp_chip_info.h>
# include <bootloader_common.h>


bool isFlashInterfacePin_ESPEasy(int gpio) {
  // GPIO-11: Flash voltage selector
  // For chip variants with a SiP flash built in, GPIO11~ GPIO17 are dedicated to connecting SiP flash, not for other uses
  //  return (gpio) >= 12 && (gpio) <= 17;
  switch (gpio) {
#if ESP_IDF_VERSION < ESP_IDF_VERSION_VAL(5, 4, 0)
    case SPI_IOMUX_PIN_NUM_HD:
    case SPI_IOMUX_PIN_NUM_CS:
    case SPI_IOMUX_PIN_NUM_MOSI:
    case SPI_IOMUX_PIN_NUM_CLK:
    case SPI_IOMUX_PIN_NUM_MISO:
    case SPI_IOMUX_PIN_NUM_WP:
#else
    case MSPI_IOMUX_PIN_NUM_HD:
    case MSPI_IOMUX_PIN_NUM_WP:
    case MSPI_IOMUX_PIN_NUM_CS0:
    case MSPI_IOMUX_PIN_NUM_CLK:
    case MSPI_IOMUX_PIN_NUM_MOSI:
    case MSPI_IOMUX_PIN_NUM_MISO:
#endif
      return true;
  }
  return false;
}

bool flashVddPinCanBeUsedAsGPIO()
{
  return false;
}

int32_t getEmbeddedFlashSize()
{
  // ESP32-C2 doesn't have eFuse field FLASH_CAP.
  // Can't get info about the flash chip.
  return 0;
}

int32_t getEmbeddedPSRAMSize()
{
  // Doesn't have PSRAM
  return 0;
}

# ifndef isPSRAMInterfacePin
bool isPSRAMInterfacePin(int gpio) {
  return false;
}

# endif // ifndef isPSRAMInterfacePin


const __FlashStringHelper* getChipModel(uint32_t chip_model, uint32_t chip_revision, uint32_t pkg_version, bool single_core)
{
  switch (pkg_version) {
    case 0: return F("ESP32-C2");
    case 1: return F("ESP32-C2");
  }
  return F("ESP32-C2");
}

#endif // ifdef ESP32C2

#include "../Helpers/StringGenerator_System.h"

#include "../Helpers/ESPEasy_time_calc.h"
#include "../Helpers/Hardware_device_info.h"
#include "../Helpers/StringConverter.h"

/*********************************************************************************************\
   ESPEasy specific strings
\*********************************************************************************************/

#include "../CustomBuild/CompiletimeDefines.h"

#if defined(ESP32)
  #ifdef ESP32S2
    #include <esp32s2/rom/rtc.h>
  #elif defined(ESP32S3)
    #include <esp32s3/rom/rtc.h>
  #elif defined(ESP32C2)
    #include <esp32c2/rom/rtc.h>
  #elif defined(ESP32C3)
    #include <esp32c3/rom/rtc.h>
  #elif defined(ESP32C6)
    #include <esp32c6/rom/rtc.h>
  # elif defined(ESP32_CLASSIC)
    #if ESP_IDF_VERSION_MAJOR > 3
      #include <esp32/rom/rtc.h>
    #else
      #include <rom/rtc.h>
    #endif
  # else

    static_assert(false, "Implement processor architecture");

  #endif
#endif

#if FEATURE_MQTT

//#include <PubSubClient.h>
#include "../Globals/MQTT.h"

const __FlashStringHelper * getMQTT_state() {
  switch (MQTTclient.state()) {
    case MQTT_CONNECTION_TIMEOUT: return F("Connection timeout");
    case MQTT_CONNECTION_LOST: return F("Connection lost");
    case MQTT_CONNECT_FAILED: return F("Connect failed");
    case MQTT_DISCONNECTED: return F("Disconnected");
    case MQTT_CONNECTED: return F("Connected");
    case MQTT_CONNECT_BAD_PROTOCOL: return F("Connect bad protocol");
    case MQTT_CONNECT_BAD_CLIENT_ID: return F("Connect bad client_id");
    case MQTT_CONNECT_UNAVAILABLE: return F("Connect unavailable");
    case MQTT_CONNECT_BAD_CREDENTIALS: return F("Connect bad credentials");
    case MQTT_CONNECT_UNAUTHORIZED: return F("Connect unauthorized");
    default: break;
  }
  return F("");
}

#endif // if FEATURE_MQTT

/********************************************************************************************\
   Get system information
 \*********************************************************************************************/
const __FlashStringHelper * getLastBootCauseString() {
  switch (lastBootCause)
  {
    case BOOT_CAUSE_MANUAL_REBOOT: return F("Manual Reboot");
    case BOOT_CAUSE_DEEP_SLEEP:    return F("Deep Sleep");
    case BOOT_CAUSE_COLD_BOOT:     return F("Cold Boot");
    case BOOT_CAUSE_EXT_WD:        return F("External Watchdog");
    case BOOT_CAUSE_SOFT_RESTART:  return F("Soft Reboot");
    case BOOT_CAUSE_SW_WATCHDOG:   return F("SW Watchdog");
    case BOOT_CAUSE_EXCEPTION:     return F("Exception");
    case BOOT_CAUSE_POWER_UNSTABLE: return F("PWR Unstable"); // ESP32 only
  }
  return F("Unknown");
}

#ifdef ESP32

// See https://github.com/espressif/esp-idf/blob/master/components/esp32/include/rom/rtc.h
const __FlashStringHelper * getResetReasonString_f(uint8_t icore, bool& isDEEPSLEEP_RESET) {
  isDEEPSLEEP_RESET = false;

  #ifdef ESP32S2

  // See tools\sdk\esp32\include\esp_rom\include\esp32s2\rom\rtc.h
  switch (rtc_get_reset_reason(icore)) {
    case NO_MEAN                : break;
    case POWERON_RESET          : return F("Vbat power on reset");
    case RTC_SW_SYS_RESET       : return F("Software reset digital core");
    case DEEPSLEEP_RESET        : isDEEPSLEEP_RESET = true; break;
    case TG0WDT_SYS_RESET       : return F("Timer Group0 Watch dog reset digital core");
    case TG1WDT_SYS_RESET       : return F("Timer Group1 Watch dog reset digital core");
    case RTCWDT_SYS_RESET       : return F("RTC Watch dog reset digital core");
    case INTRUSION_RESET        : return F("Instrusion tested to reset CPU");
    case TG0WDT_CPU_RESET       : return F("Time Group0 reset CPU");
    case RTC_SW_CPU_RESET       : return F("Software reset CPU");
    case RTCWDT_CPU_RESET       : return F("RTC Watch dog reset CPU");
    case RTCWDT_BROWN_OUT_RESET : return F("Reset when the vdd voltage is not stable");
    case RTCWDT_RTC_RESET       : return F("RTC Watch dog reset digital core and rtc module");
    case TG1WDT_CPU_RESET       : return F("Time Group1 reset CPU");
    case SUPER_WDT_RESET        : return F("super watchdog reset digital core and rtc module");
    case GLITCH_RTC_RESET       : return F("glitch reset digital core and rtc module");
    case EFUSE_RESET            : return F("efuse reset digital core");
  }

  #elif defined(ESP32S3)

  // See tools\sdk\esp32\include\esp_rom\include\esp32s3\rom\rtc.h
  switch (rtc_get_reset_reason(icore)) {
    case NO_MEAN                : break;
    case POWERON_RESET          : return F("Vbat power on reset");
    case RTC_SW_SYS_RESET       : return F("Software reset digital core");
    case DEEPSLEEP_RESET        : isDEEPSLEEP_RESET = true; break;
    case TG0WDT_SYS_RESET       : return F("Timer Group0 Watch dog reset digital core");
    case TG1WDT_SYS_RESET       : return F("Timer Group1 Watch dog reset digital core");
    case RTCWDT_SYS_RESET       : return F("RTC Watch dog reset digital core");
    case INTRUSION_RESET        : return F("Instrusion tested to reset CPU");
    case TG0WDT_CPU_RESET       : return F("Time Group0 reset CPU");
    case RTC_SW_CPU_RESET       : return F("Software reset CPU");
    case RTCWDT_CPU_RESET       : return F("RTC Watch dog reset CPU");
    case RTCWDT_BROWN_OUT_RESET : return F("Reset when the vdd voltage is not stable");
    case RTCWDT_RTC_RESET       : return F("RTC Watch dog reset digital core and rtc module");
    case TG1WDT_CPU_RESET       : return F("Time Group1 reset CPU");
    case SUPER_WDT_RESET        : return F("super watchdog reset digital core and rtc module");
    case GLITCH_RTC_RESET       : return F("glitch reset digital core and rtc module");
    case EFUSE_RESET            : return F("efuse reset digital core");
    case USB_UART_CHIP_RESET    : return F("usb uart reset digital core ");
    case USB_JTAG_CHIP_RESET    : return F("usb jtag reset digital core ");
    case POWER_GLITCH_RESET     : return F("power glitch reset digital core and rtc module");
  }

  #elif defined(ESP32C2)

  // See tools\sdk\esp32\include\esp_rom\include\esp32c2\rom\rtc.h
  switch (rtc_get_reset_reason(icore)) {
    case NO_MEAN                : break;
    case POWERON_RESET          : return F("Vbat power on reset");
    case RTC_SW_SYS_RESET       : return F("Software reset digital core");
    case DEEPSLEEP_RESET        : isDEEPSLEEP_RESET = true; break;
    // case DEEPSLEEP_RESET        : return F("Deep Sleep reset digital core");
    case TG0WDT_SYS_RESET       : return F("Timer Group0 Watch dog reset digital core");
    case RTCWDT_SYS_RESET       : return F("RTC Watch dog Reset digital core");
    case INTRUSION_RESET        : return F("Instrusion tested to reset CPU");
    case TG0WDT_CPU_RESET       : return F("Time Group0 reset CPU");
    case RTC_SW_CPU_RESET       : return F("Software reset CPU");
    case RTCWDT_CPU_RESET       : return F("RTC Watch dog Reset CPU");
    case RTCWDT_BROWN_OUT_RESET : return F("Reset when the vdd voltage is not stable");
    case RTCWDT_RTC_RESET       : return F("RTC Watch dog reset digital core and rtc module");
    case SUPER_WDT_RESET        : return F("super watchdog reset digital core and rtc module");
    case GLITCH_RTC_RESET       : return F("glitch reset digital core and rtc module");
    case EFUSE_RESET            : return F("efuse reset digital core");
    case JTAG_RESET             : return F("jtag reset CPU");
  }


  #elif defined(ESP32C3)

  // See tools\sdk\esp32\include\esp_rom\include\esp32c3\rom\rtc.h
  switch (rtc_get_reset_reason(icore)) {
    case NO_MEAN                : break;
    case POWERON_RESET          : return F("Vbat power on reset");
    case RTC_SW_SYS_RESET       : return F("Software reset digital core");
    case DEEPSLEEP_RESET        : isDEEPSLEEP_RESET = true; break;
    case TG0WDT_SYS_RESET       : return F("Timer Group0 Watch dog reset digital core");
    case TG1WDT_SYS_RESET       : return F("Timer Group1 Watch dog reset digital core");
    case RTCWDT_SYS_RESET       : return F("RTC Watch dog reset digital core");
    case INTRUSION_RESET        : return F("Instrusion tested to reset CPU");
    case TG0WDT_CPU_RESET       : return F("Time Group0 reset CPU");
    case RTC_SW_CPU_RESET       : return F("Software reset CPU");
    case RTCWDT_CPU_RESET       : return F("RTC Watch dog reset CPU");
    case RTCWDT_BROWN_OUT_RESET : return F("Reset when the vdd voltage is not stable");
    case RTCWDT_RTC_RESET       : return F("RTC Watch dog reset digital core and rtc module");
    case TG1WDT_CPU_RESET       : return F("Time Group1 reset CPU");
    case SUPER_WDT_RESET        : return F("super watchdog reset digital core and rtc module");
    case GLITCH_RTC_RESET       : return F("glitch reset digital core and rtc module");
    case EFUSE_RESET            : return F("efuse reset digital core");
    case USB_UART_CHIP_RESET    : return F("usb uart reset digital core ");
    case USB_JTAG_CHIP_RESET    : return F("usb jtag reset digital core ");
    case POWER_GLITCH_RESET     : return F("power glitch reset digital core and rtc module");
  }

  #elif defined(ESP32C6)

  // See tools\sdk\esp32\include\esp_rom\include\esp32c6\rom\rtc.h
  switch (rtc_get_reset_reason(icore)) {
    case NO_MEAN                : break;
    case POWERON_RESET          : return F("Vbat power on reset");
    case RTC_SW_SYS_RESET       : return F("Software reset digital core (hp system)");
    case DEEPSLEEP_RESET        : isDEEPSLEEP_RESET = true; break;
    //case DEEPSLEEP_RESET        : return F("Deep Sleep reset digital core (hp system)");
    case SDIO_RESET             : return F("Reset by SLC module, reset digital core (hp system)");
    case TG0WDT_SYS_RESET       : return F("Timer Group0 Watch dog reset digital core (hp system)");
    case TG1WDT_SYS_RESET       : return F("Timer Group1 Watch dog reset digital core (hp system)");
    case RTCWDT_SYS_RESET       : return F("RTC Watch dog Reset digital core (hp system)");
    case TG0WDT_CPU_RESET       : return F("Time Group0 reset CPU");
    case RTC_SW_CPU_RESET       : return F("Software reset CPU");
    case RTCWDT_CPU_RESET       : return F("RTC Watch dog Reset CPU");
    case RTCWDT_BROWN_OUT_RESET : return F("Reset when the vdd voltage is not stable");
    case RTCWDT_RTC_RESET       : return F("RTC Watch dog reset digital core and rtc module");
    case TG1WDT_CPU_RESET       : return F("Time Group1 reset CPU");
    case SUPER_WDT_RESET        : return F("super watchdog reset digital core and rtc module");
    case EFUSE_RESET            : return F("efuse reset digital core (hp system)");
    case USB_UART_CHIP_RESET    : return F("usb uart reset digital core (hp system)");
    case USB_JTAG_CHIP_RESET    : return F("usb jtag reset digital core (hp system)");
    case JTAG_RESET             : return F("jtag reset CPU");
  }


  # elif defined(ESP32_CLASSIC)

  // See https://github.com/espressif/esp-idf/blob/master/components/esp32/include/rom/rtc.h
  switch (rtc_get_reset_reason((RESET_REASON)icore)) {
    case NO_MEAN                : break;
    case POWERON_RESET          : return F("Vbat power on reset");
    case SW_RESET               : return F("Software reset digital core");
    case OWDT_RESET             : return F("Legacy watch dog reset digital core");
    case DEEPSLEEP_RESET        : isDEEPSLEEP_RESET = true; break;
    case SDIO_RESET             : return F("Reset by SLC module, reset digital core");
    case TG0WDT_SYS_RESET       : return F("Timer Group0 Watch dog reset digital core");
    case TG1WDT_SYS_RESET       : return F("Timer Group1 Watch dog reset digital core");
    case RTCWDT_SYS_RESET       : return F("RTC Watch dog reset digital core");
    case INTRUSION_RESET        : return F("Instrusion tested to reset CPU");
    case TGWDT_CPU_RESET        : return F("Time Group reset CPU");
    case SW_CPU_RESET           : return F("Software reset CPU");
    case RTCWDT_CPU_RESET       : return F("RTC Watch dog reset CPU");
    case EXT_CPU_RESET          : return F("for APP CPU, reset by PRO CPU");
    case RTCWDT_BROWN_OUT_RESET : return F("Reset when the vdd voltage is not stable");
    case RTCWDT_RTC_RESET       : return F("RTC Watch dog reset digital core and rtc module");
    default: break;
  }

  # else
    static_assert(false, "Implement processor architecture");

  #endif
  return F("");
}


String getResetReasonString(uint8_t icore) {
  bool isDEEPSLEEP_RESET(false);
  const String res = getResetReasonString_f(icore, isDEEPSLEEP_RESET);
  if (!res.isEmpty()) return res;

  if (isDEEPSLEEP_RESET) {
    String reason = F("Deep Sleep, Wakeup reason (");
    reason += rtc_get_wakeup_cause();
    reason += ')';

/*
  switch (reason) {
  #if CONFIG_IDF_TARGET_ESP32S2
    case POWERON_RESET:
    case RTC_SW_CPU_RESET:
    case DEEPSLEEP_RESET:
    case RTC_SW_SYS_RESET:
  #elif CONFIG_IDF_TARGET_ESP32
    case POWERON_RESET:
    case SW_CPU_RESET:
    case DEEPSLEEP_RESET:
    case SW_RESET:
  #endif
  }
*/
    return reason;
  }

  return F("Unknown");
}

#endif // ifdef ESP32

String getResetReasonString() {
  #ifdef ESP32
  String reason = F("CPU0: ");
  reason += getResetReasonString(0);
  if (getChipCores() > 1) { // Only report if we really have more than 1 core
    reason += F(" CPU1: ");
    reason += getResetReasonString(1);
  }
  return reason;
  #else // ifdef ESP32
  return ESP.getResetReason();
  #endif // ifdef ESP32
}

String getSystemBuildString() {
  return formatSystemBuildNr(get_build_nr());
}

String formatSystemBuildNr(uint16_t buildNr) {
  if (buildNr < 20200) return String(buildNr);

  // Build NR is used as a "revision" nr for settings
  // As of 2022-08-18, it is the nr of days since 2022-08-18 + 20200
  const uint32_t seconds_since_start = (buildNr - 20200) * 86400;
  const uint32_t unix_time_start = 1660780800; // Thu Aug 18 2022 00:00:00 GMT+0000
  struct tm build_time;
  breakTime(unix_time_start + seconds_since_start, build_time);

  String res = formatDateString(build_time, '\0');
  return res;
}

String getPluginDescriptionString() {
  String result = F("["
  #ifdef PLUGIN_BUILD_NORMAL
    "\"Normal\""
  #endif // ifdef PLUGIN_BUILD_NORMAL
  #ifdef PLUGIN_BUILD_COLLECTION
    "\"Collection\""
  #endif // ifdef PLUGIN_BUILD_COLLECTION
  #ifdef PLUGIN_BUILD_DEV
    "\"Development\""
  #endif // ifdef PLUGIN_BUILD_DEV
  #ifdef PLUGIN_DESCR
    "\"" PLUGIN_DESCR "\""
  #endif // ifdef PLUGIN_DESCR
  #ifdef BUILD_NO_DEBUG
    "\"No Debug Log\""
  #endif
  #if FEATURE_NON_STANDARD_24_TASKS && defined(ESP8266)
    "\"24tasks\""
  #endif // if FEATURE_NON_STANDARD_24_TASKS && defined(ESP8266)
  "]");
  result.replace("\"\"", "\",\"");
  return result;
}

String getSystemLibraryString() {
  String result;

  #if defined(ESP32)
  result += F("ESP32 SDK ");
  result += ESP.getSdkVersion();
  #else // if defined(ESP32)
  result += F("ESP82xx Core ");
  result += ESP.getCoreVersion();
  result += F(", NONOS SDK ");
  result += system_get_sdk_version();
  result += F(", LWIP: ");
  result += getLWIPversion();
  #endif // if defined(ESP32)

  if (puyaSupport()) {
    result += F(" PUYA support");
  }
  return result;
}

#ifdef ESP8266
String getLWIPversion() {
  String result;

  result += LWIP_VERSION_MAJOR;
  result += '.';
  result += LWIP_VERSION_MINOR;
  result += '.';
  result += LWIP_VERSION_REVISION;

  if (LWIP_VERSION_IS_RC) {
    result += F("-RC");
    result += LWIP_VERSION_RC;
  } else if (LWIP_VERSION_IS_DEVELOPMENT) {
    result += F("-dev");
  }
  return result;
}

#endif // ifdef ESP8266


#include "../Helpers/ESPEasy_time.h"

#include "../../ESPEasy_common.h"

#include "../../_Plugin_Helper.h"

#include "../CustomBuild/CompiletimeDefines.h"

#include "../DataStructs/NTP_packet.h"
#include "../DataStructs/TimingStats.h"

#include "../DataTypes/TimeSource.h"

#include "../ESPEasyCore/ESPEasy_Log.h"
#include "../ESPEasyCore/ESPEasyNetwork.h"

#include "../Globals/EventQueue.h"
#include "../Globals/NetworkState.h"
#include "../Globals/Nodes.h"
#include "../Globals/RTC.h"
#include "../Globals