/**
 * The MIT License (MIT)
 *
 * Copyright (c) 2016 by Daniel Eichhorn
 * Copyright (c) 2016 by Fabrice Weinberg
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Credits for parts of this code go to Mike Rankin. Thank you so much for sharing!
 */

#ifndef SH1106Spi_h
#define SH1106Spi_h

#include "OLEDDisplay.h"
#include <SPI.h>

class SH1106Spi : public OLEDDisplay {
  private:
      uint8_t             _rst;
      uint8_t             _dc;

  public:

    SH1106Spi(uint8_t _rst, uint8_t _dc) {
      this->_rst = _rst;
      this->_dc  = _dc;
    }

    bool connect(){
      pinMode(_dc, OUTPUT);
      pinMode(_rst, OUTPUT);

      SPI.begin ();
      SPI.setClockDivider (SPI_CLOCK_DIV2);

      // Pulse Reset low for 10ms
      digitalWrite(_rst, HIGH);
      delay(1);
      digitalWrite(_rst, LOW);
      delay(10);
      digitalWrite(_rst, HIGH);
      return true;
    }

    void display(void) {
    #ifdef OLEDDISPLAY_DOUBLE_BUFFER
       uint8_t minBoundX, minBoundY, maxBoundX, maxBoundY;
       if (!getChangedBoundingBox(minBoundX, minBoundY, maxBoundX, maxBoundY))
         return;

       // Calculate the colum offset
       uint8_t minBoundXp2H = (minBoundX + 2) & 0x0F;
       uint8_t minBoundXp2L = 0x10 | ((minBoundX + 2) >> 4 );

       for (uint8_t y = minBoundY; y <= maxBoundY; y++) {
         sendCommand(0xB0 + y);
         sendCommand(minBoundXp2H);
         sendCommand(minBoundXp2L);
         digitalWrite(_dc, HIGH);   // data mode
         for (uint8_t x = minBoundX; x <= maxBoundX; x++) {
           SPI.transfer(buffer[x + y * DISPLAY_WIDTH]);
         }
         yield();
       }
     #else
      for (uint8_t y=0; y<DISPLAY_HEIGHT/8; y++) {
        sendCommand(0xB0 + y);
        sendCommand(0x02);
        sendCommand(0x10);
        digitalWrite(_dc, HIGH);   // data mode
        for( uint8_t x=0; x < DISPLAY_WIDTH; x++) {
          SPI.transfer(buffer[x + y * DISPLAY_WIDTH]);
        }
        yield();
      }
     #endif
    }

  private:
    inline void sendCommand(uint8_t com) __attribute__((always_inline)){
      digitalWrite(_dc, LOW);
      SPI.transfer(com);
    }
};

#endif
